#!/usr/bin/env python
# Copyright (c) 2014 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import CliSave
import IntfCliSave
import IpUtils
import Tac
from RoutingIntfUtils import allRoutingProtocolIntfNames

AddressFamily = Tac.Type( "Arnet::AddressFamily" )

def _sortedByPrefix( coll ):
   # pylint: disable-msg=W0108
   sPrefix = sorted( coll, cmp=IpUtils.compareIpPrefixStr,
                     key=( lambda x: str(x) ) )
   return sPrefix

IntfCliSave.IntfConfigMode.addCommandSequence( \
      'McastBoundary.intf', after=[ 'Ira.ipIntf' ] )

def mcastRoutingSupported( sysdbRoot, routingHardwareStatus=None ):
   if routingHardwareStatus is None:
      routingHardwareStatus = sysdbRoot[ 'routing' ][ 'hardware' ][ 'status' ]
   return routingHardwareStatus.multicastRoutingSupported

def McastBoundaryConfig( mcastBoundaryConfig, root, sysdbRoot, options,
                         requireMounts, af ):
   #Save defaults only if platform supports multicast routing
   saveAll = options.saveAll
   saveAllDetail = options.saveAllDetail

   if af == AddressFamily.ipv4:
      hwStatus = requireMounts[ 'routing/hardware/status' ]
   else:
      hwStatus = requireMounts[ 'routing6/hardware/status' ]

   if not mcastRoutingSupported(
         sysdbRoot,
         hwStatus ):
      saveAll = False
      saveAllDetail = False

   if saveAllDetail:
      cfgIntfNames = allRoutingProtocolIntfNames( sysdbRoot, includeEligible=True,
                                                  requireMounts=requireMounts )
   elif saveAll:
      # Routing configuration is allowed on switchports as well.
      # Save configuration on all routing protocol interfaces and switchports
      # with non-default config.
      cfgIntfNames = set(
            allRoutingProtocolIntfNames( sysdbRoot, requireMounts=requireMounts ) +
            mcastBoundaryConfig.intfConfig.keys() )
   else:
      cfgIntfNames = mcastBoundaryConfig.intfConfig

   for intfName in cfgIntfNames:
      intfConfig = mcastBoundaryConfig.intfConfig.get( intfName )

      if not intfConfig:
         if saveAll:
            intfConfig = Tac.newInstance( "Routing::McastBoundary::IntfConfig",
                  intfName )
         else:
            continue

      saveIntfConfigConverted( intfConfig, root, sysdbRoot, saveAll,
                               saveAllDetail, af )

@CliSave.saver( "Routing::McastBoundary::Config", 'routing/mcastboundary/config',
                requireMounts=( 'routing/hardware/status',
                                'interface/config/all', 'interface/status/all' ) )
def McastBoundaryIpv4Config( mcastBoundaryConfig, root, sysdbRoot, options,
                             requireMounts ):
   McastBoundaryConfig( mcastBoundaryConfig, root, sysdbRoot, options,
                        requireMounts, AddressFamily.ipv4 )

@CliSave.saver( "Routing::McastBoundary::Config", 'routing6/mcastboundary/config',
                requireMounts=( 'routing6/hardware/status',
                                'interface/config/all', 'interface/status/all' ) )
def McastBoundaryIpv6Config( mcastBoundaryConfig, root, sysdbRoot, options,
                             requireMounts ):
   McastBoundaryConfig( mcastBoundaryConfig, root, sysdbRoot, options,
                        requireMounts, AddressFamily.ipv6 )

def saveIntfConfigConverted( mcastBoundaryIntfConfig, root, sysdbRoot, saveAll,
                             saveAllDetail, af ):
   cmdPrefix = 'multicast %s boundary' % af

   intf = mcastBoundaryIntfConfig
   mode = root[ IntfCliSave.IntfConfigMode ].getOrCreateModeInstance( intf.intfId )
   cmds = mode[ 'McastBoundary.intf' ]

   for k in intf.boundary:
      v = intf.boundary[ k ]
      prefix = k.v4Prefix.stringValue if af==AddressFamily.ipv4 \
                                      else k.v6Prefix.stringValue
      out = " out" if v.out else ""
      cmds.addCommand( '%s %s%s' % \
                       ( cmdPrefix, prefix, out ) )

   if intf.boundaryAcl.acl != '':
      out = " out" if intf.boundaryAcl.out else ""
      cmds.addCommand( '%s %s%s' % ( cmdPrefix, intf.boundaryAcl.acl, out ) )


