#!/usr/bin/env python
# Copyright (c) 2014 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tac, re
import CliSave
from CliMode.BgpGroup import BgpGroupMode
from CliMode.BgpMaintenanceProfile import BgpMaintenanceProfileMode
from CliMode.MaintenanceModeCliSaveLib import MaintenanceConfigMode
from CliSavePlugin.IraVrfCliSave import VrfDefinitionCliSaveMode
from CliMode.BgpMaintenanceMode import BgpMaintenanceMode

defaultProfileName = \
   Tac.Type( 'Maintenance::Profile::DefaultProfile' ).systemDefaultProfileName
bgpGroupType = Tac.Type( "Group::GroupType" ).groupTypeBgp
unitType = Tac.Type( "Maintenance::Unit::UnitType" )

#-----------------------------------------------------------------     
# CliSave Plugin for Bgp Submode under Maintenance Config Mode   
#-----------------------------------------------------------------   
class BgpMaintenanceCliSaveMode( BgpMaintenanceMode, CliSave.Mode ):

   def __init__( self, param ):
      BgpMaintenanceMode.__init__( self, param )                 
      CliSave.Mode.__init__( self, param ) 
                                             
CliSave.GlobalConfigMode.addChildMode( MaintenanceConfigMode )
MaintenanceConfigMode.addChildMode( BgpMaintenanceCliSaveMode )
BgpMaintenanceCliSaveMode.addCommandSequence( 'Maintenance.bgpMode' )         

@CliSave.saver( 'Maintenance::Unit::ConfigDir', 'maintenance/unit/config',
                requireMounts = ( 'maintenance/unit/input/cli', ) )
def saveMaintenanceBgpConfig( entity, root, sysdbRoot, options, requireMounts ):
   unitInputCli = requireMounts[ 'maintenance/unit/input/cli' ]
   for unit in entity.config.values():
      if unit.unitType == unitType.dynamic:
         isBgpUnit = False
         for group in unit.group:
            if group.type == bgpGroupType:
               isBgpUnit = True
         if isBgpUnit:
            bgpVrfRe = "<Dynamic.+><.+><vrf-(.+)>"
            if re.search( bgpVrfRe, unit.unitName ):
               mode = root[ MaintenanceConfigMode ].getOrCreateModeInstance( 
                  'maintenance' )
               bgpMode = mode[ BgpMaintenanceCliSaveMode ].getOrCreateModeInstance( 
                  unit.unitName )
            cmds = bgpMode[ 'Maintenance.bgpMode' ]
            if unit.unitName in unitInputCli.unit:
               cmds.addCommand( 'quiesce' )

#-----------------------------------------------------------------
# CliSave Plugin for Bgp Maintenance Groups
#-----------------------------------------------------------------
class BgpGroupConfigMode( BgpGroupMode, CliSave.Mode ):

   def __init__( self, param ):
      BgpGroupMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

CliSave.GlobalConfigMode.addChildMode( BgpGroupConfigMode,
                                       after=[ VrfDefinitionCliSaveMode ] )
BgpGroupConfigMode.addCommandSequence( 'BgpGroup.configDir' )

@CliSave.saver( 'BgpGroup::ConfigDir', 'group/config/bgp',
                requireMounts = ( 'maintenance/group/config', ) )
def saveBgpGroupConfig( entity, root, sysdbRoot, options, requireMounts ):
   maintenanceGroupDir = requireMounts[ 'maintenance/group/config' ]
   groupOrigin = Tac.Type( "Group::GroupOrigin" )
   for groupConfig in entity.config.values():
      if groupConfig.origin == groupOrigin.dynamic:
         continue
      group = groupConfig.groupName
      mode = root[ BgpGroupConfigMode ].getOrCreateModeInstance( group )
      cmds = mode[ 'BgpGroup.configDir' ]
      member = entity.config[ group ]
      if groupConfig.origin == groupOrigin.userConfigured:
         if member.vrfName != 'default':
            cmds.addCommand( 'vrf %s' %( member.vrfName ) )
         for neighbor in member.neighbor.keys() :
            cmds.addCommand( 'neighbor %s' % ( neighbor.stringValue ) )
      maintenanceGroup = maintenanceGroupDir.maintenanceGroup.get( member.key )
      if maintenanceGroup:
         for profile in maintenanceGroup.profile.keys():
            cmds.addCommand( 'maintenance profile bgp %s' %( profile.name ) )

      #XXX Temporary hack to avoid 'maintenance' config mode following this from 
      # cribbing for IncompleteCommandError
      cmds.addCommand( 'exit' )

#-----------------------------------------------------------------
# CliSave Plugin for Bgp Maintenance Profiles
#-----------------------------------------------------------------
class BgpMaintenanceProfileConfigMode( BgpMaintenanceProfileMode, CliSave.Mode ):
   
   def __init__( self, param ):
      BgpMaintenanceProfileMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

MaintenanceConfigMode.addChildMode( BgpMaintenanceProfileConfigMode )
BgpMaintenanceProfileConfigMode.addCommandSequence( 'BgpProfile.configDir' )
MaintenanceConfigMode.addCommandSequence( 'DefaultProfile' )

@CliSave.saver( 'BgpMaintenanceProfile::ConfigDir',
                'maintenance/profile/config/bgp' )
def saveBgpProfileConfig( entity, root, sysdb, options ):
   for profile in entity.config:
      if profile != defaultProfileName:
         mode = root[ MaintenanceConfigMode ].getOrCreateModeInstance(
            'maintenance' )
         profileMode = mode[ 
            BgpMaintenanceProfileConfigMode ].getOrCreateModeInstance( profile )
         cmds = profileMode[ 'BgpProfile.configDir' ]
         member = entity.config[ profile ]
         if member.routeMapInOut:
            cmds.addCommand( 'initiator route-map %s inout' % member.routeMapInOut )
         if member.routeMapIn:
            cmds.addCommand( 'initiator route-map %s in' % member.routeMapIn )
         if member.routeMapOut:
            cmds.addCommand( 'initiator route-map %s out' % member.routeMapOut )

@CliSave.saver( 'Maintenance::Profile::DefaultProfile',
                'maintenance/profile/config/default/bgp' )
def saveDefaultBgpProfile( entity, root, sysdb, options ):
   if entity.profileName != defaultProfileName:
      mode = root[ MaintenanceConfigMode ].getOrCreateModeInstance(
            'maintenance' )
      cmds = mode[ 'DefaultProfile' ]
      cmds.addCommand( 'profile bgp %s default' % entity.profileName )
