# Copyright (c) 2015 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import CliSave
import Tac
from CliMode.Macsec import MacsecProfileBaseMode, MacsecBaseMode
from CliMode.Macsec import MacsecProfileStaticSakMode
from IntfCliSave import IntfConfigMode
import Toggles.MacsecToggleLib as macsecToggle
from MacsecCommon import tacCipherSuiteToCli

IntfConfigMode.addCommandSequence( 'Macsec.intf' )

class MacsecConfigMode( MacsecBaseMode, CliSave.Mode ):
   def __init__( self, param ):
      MacsecBaseMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

CliSave.GlobalConfigMode.addChildMode( MacsecConfigMode )
MacsecConfigMode.addCommandSequence( 'Macsec.config' )

class MacsecProfileConfigMode( MacsecProfileBaseMode, CliSave.Mode ):
   def __init__( self, param ):
      MacsecProfileBaseMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

MacsecConfigMode.addChildMode( MacsecProfileConfigMode )
MacsecProfileConfigMode.addCommandSequence( 'Macsec.config.profile' )

class MacsecStaticSakConfigMode( MacsecProfileStaticSakMode, CliSave.Mode ):
   def __init__( self, param ):
      MacsecProfileStaticSakMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

   def skipIfEmpty( self ):
      return True

MacsecProfileConfigMode.addChildMode( MacsecStaticSakConfigMode )
MacsecStaticSakConfigMode.addCommandSequence( 'Macsec.config.profile.sc' )

def saveProfileConfig( profile, root, config, sysdbRoot, options ):
   secMode = root[ MacsecConfigMode ].getOrCreateModeInstance( None )
   mode = secMode[ MacsecProfileConfigMode ].getOrCreateModeInstance( profile.name )
   cmds = mode[ 'Macsec.config.profile' ]

   emptyKey = Tac.Value( "Macsec::Cak" )
   if profile.cipherSuite != profile.cipherSuiteDefault:
      cmds.addCommand('cipher %s ' % tacCipherSuiteToCli[ profile.cipherSuite ] )
     
   if profile.key != emptyKey:
      cmd = 'key %s 7 %s' % ( profile.key.ckn,
            CliSave.sanitizedOutput( options, "%s" % profile.key.cak ) )
      cmds.addCommand( cmd )

   if profile.defaultKey != emptyKey:
      cmd = 'key %s 7 %s fallback' % ( profile.defaultKey.ckn,
            CliSave.sanitizedOutput( options, "%s" % profile.defaultKey.cak ) )
      cmds.addCommand( cmd )
  
   if profile.keyServerPriority != profile.keyServerPriorityDefault:
      cmds.addCommand( 'mka key-server priority %d' % profile.keyServerPriority )
   elif options.saveAll:
      cmds.addCommand( 'no mka key-server priority' )

   if profile.sessionReKeyPeriod != profile.sessionReKeyPeriodDefault:
      cmds.addCommand( 'mka session rekey-period %d' % profile.sessionReKeyPeriod )
   elif options.saveAll:
      cmds.addCommand( 'no mka session rekey-period' )

   if macsecToggle.toggleMacsecTrafficBlockOnMkaFailureEnabled():
      if profile.trafficPolicyOnNoMka == 'unprotected':
         cmds.addCommand( 'traffic unprotected allow' )
      elif profile.trafficPolicyOnNoMka == 'blocked':
         cmds.addCommand( 'traffic unprotected drop' )
      elif options.saveAll:
         cmds.addCommand( 'traffic unprotected allow active-sak' )
   else:
      if profile.allowUnprotected != profile.allowUnprotectedDefault:
         cmds.addCommand( 'traffic unprotected allow' )
      elif options.saveAll:
         cmds.addCommand( 'no traffic unprotected allow' )

   if profile.lpnValueZero != profile.lpnValueZeroDefault:
      cmds.addCommand( 'mka session lpn advertisement disabled' )
   elif options.saveAll:
      cmds.addCommand( 'no mka session lpn advertisement disabled' )

   if profile.dot1xEnabled != profile.dot1xEnabledDefault:
      cmds.addCommand( 'key source dot1x' )
   elif options.saveAll:
      cmds.addCommand( 'no key source dot1x' )
 
   if profile.groupCak.enabled != profile.groupCak.enabledDefault:
      cmd = 'key source group-cak '
      if profile.groupCak.lifetime != profile.groupCak.lifetimeDefault:
         cmd += str( profile.groupCak.lifetime )
      cmds.addCommand( cmd )
   elif options.saveAll:
      cmds.addCommand( 'no key source group-cak' )
     
   if profile.keyRetire:
      cmds.addCommand( 'key retirement immediate' )
   elif options.saveAll:
      cmds.addCommand( 'no key retirement immediate' )

   if profile.includeSci != profile.includeSciDefault:
      cmds.addCommand( 'sci' )

   if profile.bypassLldp != profile.bypassLldpDefault:
      cmds.addCommand( 'l2-protocol lldp bypass' )

   if profile.replayProtection != profile.replayProtectionDefault:
      cmds.addCommand( 'replay protection disabled' )
   if profile.replayProtectionWindow != profile.replayProtectionWindowDefault:
      cmds.addCommand( 'replay protection window ' +
         str( profile.replayProtectionWindow ) )

   # Save static sak config
   # Save secure channel tx config
   staticMode = mode[ MacsecStaticSakConfigMode ].getOrCreateModeInstance(
                                                      ( profile.name, 'tx' ) )
   cmds = staticMode[ 'Macsec.config.profile.sc' ]
   txSci = profile.txStaticSci
   txStaticSak = profile.txStaticSak
   if txSci:
      cmds.addCommand( 'identifier %s::%s' % ( txSci.addr, txSci.portNum ) )
   if txStaticSak:
      cmds.addCommand( 'an %d key 7 %s' % ( txStaticSak.an, txStaticSak.key ) )

   # Save secure channel rx config
   staticMode = mode[ MacsecStaticSakConfigMode ].getOrCreateModeInstance( 
                                                      ( profile.name, 'rx' ) )
   cmds = staticMode[ 'Macsec.config.profile.sc' ]
   rxSci = profile.rxStaticSci
   if rxSci:
      cmds.addCommand( 'identifier %s::%s' % ( rxSci.addr, rxSci.portNum ) )
   if profile.rxStaticSak:
      for i in sorted( profile.rxStaticSak.keys() ):
         rxSak = profile.rxStaticSak[ i ]
         cmds.addCommand( 'an %d key 7 %s' % ( rxSak.an, rxSak.key ) )

def saveIntfConfig( intfConfig, root, config, sysdbRoot, saveAll ):
   mode = root[ IntfConfigMode ].getOrCreateModeInstance( intfConfig.intfId )
   cmds = mode[ 'Macsec.intf' ]

   cmds.addCommand( 'mac security profile %s' % intfConfig.profileName )

def saveLicenseConfig( root, config, sysdbRoot, options ):
   if config.licenseConfig != Tac.Value( "Macsec::LicenseConfig" ) or \
         options.saveAll:
      mode = root[ MacsecConfigMode ].getOrCreateModeInstance( None )
      cmds = mode[ 'Macsec.config' ]
      licensee = config.licenseConfig.licenseeName
      authKey = config.licenseConfig.authKey
      
      if config.licenseConfig != Tac.Value( "Macsec::LicenseConfig" ):
         cmds.addCommand( 'license %s %s' % ( licensee,
            CliSave.sanitizedOutput( options, "%x" % authKey ) ) )
      elif options.saveAll:
         cmds.addCommand( 'no license' )

def saveDelayProtectionConfig( root, config, sysdbRoot, options ):
   if config.delayProtection != config.delayProtectionDefault or \
         options.saveAll:
      mode = root[ MacsecConfigMode ].getOrCreateModeInstance( None )
      cmds = mode[ 'Macsec.config' ]
      if config.delayProtection:
         cmds.addCommand( 'delay protection' )
      elif options.saveAll:
         cmds.addCommand(' no delay protection' )

def saveFipsRestrictionsConfig( root, config, sysdbRoot, options ):
   if config.fipsRestrictions != config.fipsRestrictionsDefault or \
         options.saveAll:
      mode = root[ MacsecConfigMode ].getOrCreateModeInstance( None )
      cmds = mode[ 'Macsec.config' ]
      if config.fipsRestrictions:
         cmds.addCommand( 'fips restrictions' )
      elif options.saveAll:
         cmds.addCommand('no fips restrictions' )

def saveShutdownConfig( root, config, sysdbRoot, options ):
   if config.shutdown or options.saveAll:
      mode = root[ MacsecConfigMode ].getOrCreateModeInstance( None )
      cmds = mode[ 'Macsec.config' ]

      if config.shutdown:
         cmds.addCommand( 'shutdown' )
      elif options.saveAll:
         cmds.addCommand( 'no shutdown' )

def saveEapolAttributes( root, config, sysdbRoot, options ):
   if config.eapolAttr.destinationMac != config.eapolAttr.destinationMacDefault or\
      config.eapolAttr.etherType != config.eapolAttr.etherTypeDefault or\
      options.saveAll:
      mode = root[ MacsecConfigMode ].getOrCreateModeInstance( None )
      cmds = mode[ 'Macsec.config' ]

      if macsecToggle.toggleMacsecConfigurableEapolDmacEnabled():
         if config.eapolAttr.destinationMac != \
               config.eapolAttr.destinationMacDefault:
            cmds.addCommand( 'eapol mac destination %s' %
                  config.eapolAttr.destinationMac )
         elif options.saveAll:
            cmds.addCommand( 'no eapol mac destination' )

      if macsecToggle.toggleMacsecConfigurableEapolEtherTypeEnabled():
         if config.eapolAttr.etherType != config.eapolAttr.etherTypeDefault:
            cmds.addCommand( 'eapol ethertype %d' %
                          config.eapolAttr.etherType )
         elif options.saveAll:
            cmds.addCommand( 'no eapol ethertype' )

@CliSave.saver( 'Macsec::Config', 'macsec/input/cli' )
def saveConfig( config, root, sysdbRoot, options ):
   # Save profile configs.
   for profileName in sorted( config.profile ):
      saveProfileConfig( config.profile[ profileName ], root, config, sysdbRoot,
            options )

   # Save the interface configs.
   for intfId in config.intfConfig:
      saveIntfConfig( config.intfConfig[ intfId ], root, config, sysdbRoot,
            options.saveAll )
   
   # Save the license config.
   saveLicenseConfig( root, config, sysdbRoot, options )

   # Save the delay protection config
   saveDelayProtectionConfig( root, config, sysdbRoot, options )

   # Save the FIPS config
   saveFipsRestrictionsConfig( root, config, sysdbRoot, options )

   # Save the shutdown config
   saveShutdownConfig( root, config, sysdbRoot, options )

   # Save the EAPoL attributes config
   saveEapolAttributes( root, config, sysdbRoot, options )
