# Copyright (c) 2013 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import CliSave
from CliMode.LdpMode import (
   LdpMode,
)
from IntfCliSave import IntfConfigMode
from IpLibConsts import DEFAULT_VRF
import Toggles.LdpToggleLib
import Toggles.MplsToggleLib
import Tac
MplsLabel = Tac.Type( 'Arnet::MplsLabel' )
LdpProtoParam = Tac.Type( 'Ldp::LdpProtoParam' )
LdpLabelLocalTerminationMode = Tac.Type( 'Ldp::LdpLabelLocalTerminationMode' )

#-------------------------------------------------------------------------------
# Object used for saving commands in "mpls-ldp" mode.
#-------------------------------------------------------------------------------
class LdpConfigMode( LdpMode, CliSave.Mode ):
   def __init__( self, param ):
      LdpMode.__init__( self )
      CliSave.Mode.__init__( self, param )

CliSave.GlobalConfigMode.addChildMode( LdpConfigMode, after=[ 'Mpls.config' ] )

LdpConfigMode.addCommandSequence( 'Ldp.config' )
IntfConfigMode.addCommandSequence( 'Ldp.intf', after=[ 'Ira.ipIntf' ] )

if Toggles.LdpToggleLib.toggleP2mpToggleEnabled():
   from CliMode.LdpMode import MldpMode
   class MldpConfigMode( MldpMode, CliSave.Mode ):
      def __init__( self, param ):
         MldpMode.__init__( self )
         CliSave.Mode.__init__( self, param )

   LdpConfigMode.addChildMode( MldpConfigMode, after=[ 'Ldp.config'] )
   MldpConfigMode.addCommandSequence( 'Mldp.config' )

if Toggles.LdpToggleLib.toggleGrToggleEnabled():
   from CliMode.LdpMode import (
      GrHelperMode,
      GrSpeakerMode,
   )

   class GrHelperConfigMode( GrHelperMode, CliSave.Mode ):
      def __init__( self, param ):
         GrHelperMode.__init__( self )
         CliSave.Mode.__init__( self, param )

   class GrSpeakerConfigMode( GrSpeakerMode, CliSave.Mode ):
      def __init__( self, param ):
         GrSpeakerMode.__init__( self )
         CliSave.Mode.__init__( self, param )

   LdpConfigMode.addChildMode( GrHelperConfigMode, after=[ 'Ldp.config' ] )
   GrHelperConfigMode.addCommandSequence( 'GrHelper.config' )
   LdpConfigMode.addChildMode( GrSpeakerConfigMode, after=[ 'Ldp.config' ] )
   GrSpeakerConfigMode.addCommandSequence( 'GrSpeaker.config' )

def saveIntfConfig( root, ldpIntfConfigColl, options ):
   for intfName in ldpIntfConfigColl:
      mode = root[ IntfConfigMode ].getOrCreateModeInstance( intfName )
      cmds = mode[ "Ldp.intf" ]
      igpSync = ldpIntfConfigColl[ intfName ].igpSync
      if igpSync == "on":
         cmds.addCommand( "mpls ldp igp sync" )
      elif igpSync == "off":
         cmds.addCommand( "no mpls ldp igp sync" )
      elif options.saveAll:
         cmds.addCommand( "default mpls ldp igp sync" )

def saveIntfLdpEnabled( root, ldpEnabledInterfaces, options ):
   for intfName in ldpEnabledInterfaces:
      mode = root[ IntfConfigMode ].getOrCreateModeInstance( intfName )
      cmds = mode[ "Ldp.intf" ]
      cmds.addCommand( "mpls ldp interface" )

# pylint: disable-msg=C0322
@CliSave.saver( 'Ldp::LdpConfigColl', 'mpls/ldp/ldpConfigColl',
                requireMounts=( 'interface/config/all',
                                'interface/status/all',
                                'acl/cpconfig/cli' ) )
def saveLdpConfig ( entity, root, sysdbRoot, options, requireMounts ):
   config = entity.config.get( 'default' )
   if config is None:
      return

   ldpServiceAclConfig = None
   aclCpConfig = requireMounts[ 'acl/cpconfig/cli' ]
   serviceAclVrfConfig = aclCpConfig.cpConfig[ 'ip' ].serviceAcl
   if DEFAULT_VRF in serviceAclVrfConfig:
      ldpServiceAclConfig = serviceAclVrfConfig[ DEFAULT_VRF ].service.get( 'ldp' )

   if config.enabled or config.routerId != '0.0.0.0' or config.routerIdIntfId != '' \
         or config.ipTransportAddrIntfId != '' \
         or config.md5Password != '' \
         or config.linkReadyTimeout != config.linkReadyTimeoutDefault \
         or config.linkReadyDelay != config.linkReadyDelayDefault \
         or config.ldpLabelAdvDspl != 'unSolicited' \
         or config.ldpLabelRetention == 'conservativeRetention' \
         or config.ldpProtoParamSetting \
         or config.staticFecBindingColl.keys() != [] \
         or config.staticTarget.keys() != [] \
         or config.fecFilterPrefixListName != '' \
         or config.helloHoldTime != LdpProtoParam.defaultBasicHelloHoldTime \
         or config.helloInterval != LdpProtoParam.defaultBasicHelloInterval \
         or config.targetedHelloHoldTime != LdpProtoParam.defaultTargetHelloHoldTime\
         or config.targetedHelloInterval != LdpProtoParam.defaultTargetHelloInterval\
         or config.mLdpEnabled != False \
         or config.staticP2mpFecBindingColl.keys() != [] \
         or ( ldpServiceAclConfig and ldpServiceAclConfig.aclName ) \
         or config.grOperFlag.value \
         or config.helloRedundancy \
         or config.ldpLabelLocalTerminationMode != (
            LdpLabelLocalTerminationMode.implicitNull ) \
         or config.entropyLabel is True \
         or config.endOfLib \
         or options.saveAll:
      mode = root[ LdpConfigMode ].getOrCreateModeInstance( 'default' )
   else:
      saveIntfConfig( root, config.intfConfigColl, options )
      saveIntfLdpEnabled( root, config.ldpEnabledIntf, options )
      return

   if config.routerId != '0.0.0.0':
      mode[ 'Ldp.config' ].addCommand( 'router-id ' + config.routerId )
   elif config.routerIdIntfId != '':
      mode[ 'Ldp.config' ].addCommand( 'router-id interface '
                                       + config.routerIdIntfId )
   elif options.saveAll:
      mode[ 'Ldp.config' ].addCommand( 'no router-id' )

   if config.ipTransportAddrIntfId != '':
      mode[ 'Ldp.config' ].addCommand( 'transport-address interface ' +
                                       config.ipTransportAddrIntfId )
   elif options.saveAll:
      mode[ 'Ldp.config' ].addCommand( 'no transport-address' )

   if config.md5Password != '':
      mode[ 'Ldp.config' ].addCommand( 'password 7 ' + 
            CliSave.sanitizedOutput( options, config.md5Password ) )
   elif options.saveAll:
      mode[ 'Ldp.config' ].addCommand( 'no password' )

   if config.linkReadyTimeout == 0:
      mode[ 'Ldp.config' ].addCommand( 'no igp sync holddown' )
   elif config.linkReadyTimeout == Tac.endOfTime:
      mode[ 'Ldp.config' ].addCommand( 'igp sync holddown until-established' )
   elif config.linkReadyTimeout != config.linkReadyTimeoutDefault:
      mode[ 'Ldp.config' ].addCommand( 'igp sync holddown ' + \
              str( int( config.linkReadyTimeout ) ) )
   elif options.saveAll:
      mode[ 'Ldp.config' ].addCommand( 'igp sync holddown ' + \
              str( int( config.linkReadyTimeoutDefault ) ) )

   if config.linkReadyDelay == 0:
      mode[ 'Ldp.config' ].addCommand( 'no igp sync delay' )
   elif config.linkReadyDelay != config.linkReadyDelayDefault:
      mode[ 'Ldp.config' ].addCommand( 'igp sync delay ' + \
              str( int( config.linkReadyDelay ) ) )
   elif options.saveAll:
      mode[ 'Ldp.config' ].addCommand( 'igp sync delay ' + \
              str( int( config.linkReadyDelayDefault ) ) )

   # Following are for hidden commands. Only show when it's configured
   if config.ldpLabelAdvDspl == 'onDemand':
      mode[ 'Ldp.config' ].addCommand( 'adv-discipline mode on-demand' )
   if config.ldpLabelRetention == 'conservativeRetention':
      mode[ 'Ldp.config' ].addCommand( 'retention mode conservative' )
   if config.ldpProtoParamSetting:
      mode[ 'Ldp.config' ].addCommand( 'proto-param ' + 
                                       hex( config.ldpProtoParamSetting ) )

   for key, binding in config.staticFecBindingColl.items():
      cmdStr = ''
      if MplsLabel( binding.fecBindingInfo.inputLabel ).isValid():
         cmdStr += ' input %d' % binding.fecBindingInfo.inputLabel
      if MplsLabel( binding.fecBindingInfo.outputLabel ).isValid():
         cmdStr += ' output ' + binding.fecBindingInfo.nexthopIntfIp.stringValue
         if binding.fecBindingInfo.outputLabel == 0:
            cmdStr += ' explicit-null'
         elif binding.fecBindingInfo.outputLabel == 3:
            cmdStr += ' implicit-null'
         else:
            cmdStr += ' %d' % binding.fecBindingInfo.outputLabel
      mode[ 'Ldp.config' ].addCommand( 'mpls static binding ipv4 ' +
                                       key.stringValue + cmdStr )

   for targetIpAddr in config.staticTarget.keys():
      mode[ 'Ldp.config' ].addCommand( 'neighbor %s targeted' % str( targetIpAddr ) )

   if config.fecFilterPrefixListName != '':
      mode[ 'Ldp.config' ].addCommand( 'fec filter prefix-list %s' %
                                                     config.fecFilterPrefixListName )
   elif options.saveAll:
      mode[ 'Ldp.config' ].addCommand( 'no fec filter' )

   # Link hello configuration
   if config.helloHoldTime != LdpProtoParam.defaultBasicHelloHoldTime:
      if config.helloHoldTime == LdpProtoParam.infiniteHelloHoldTime:
         mode[ 'Ldp.config' ].addCommand( 'discovery hello hold-time infinite')
      else:
         mode[ 'Ldp.config' ].addCommand( 'discovery hello hold-time ' +
                                          str( config.helloHoldTime ) )
   elif options.saveAll:
      mode[ 'Ldp.config' ].addCommand( 'discovery hello hold-time ' +
                                    str( LdpProtoParam.defaultBasicHelloHoldTime ) )

   if config.helloInterval != LdpProtoParam.defaultBasicHelloInterval:
      mode[ 'Ldp.config' ].addCommand( 'discovery hello interval ' +
                                       str( config.helloInterval ) )
   elif options.saveAll:
      mode[ 'Ldp.config' ].addCommand( 'discovery hello interval ' +
                                    str( LdpProtoParam.defaultBasicHelloInterval ) )


   # Targeted hello configuration
   if config.targetedHelloHoldTime != LdpProtoParam.defaultTargetHelloHoldTime:
      if config.targetedHelloHoldTime == LdpProtoParam.infiniteHelloHoldTime:
         mode[ 'Ldp.config' ].addCommand( \
            'discovery targeted-hello hold-time infinite' )
      else:
         mode[ 'Ldp.config' ].addCommand( 'discovery targeted-hello hold-time ' +
                                          str( config.targetedHelloHoldTime ) )
   elif options.saveAll:
      mode[ 'Ldp.config' ].addCommand( 'discovery targeted-hello hold-time ' +
                                    str( LdpProtoParam.defaultTargetHelloHoldTime ) )

   if config.targetedHelloInterval != LdpProtoParam.defaultTargetHelloInterval:
      mode[ 'Ldp.config' ].addCommand( 'discovery targeted-hello interval ' +
                                       str( config.targetedHelloInterval ) )
   elif options.saveAll:
      mode[ 'Ldp.config' ].addCommand( 'discovery targeted-hello interval ' +
                                    str( LdpProtoParam.defaultTargetHelloInterval ) )

   # Service ACL
   if ldpServiceAclConfig and ldpServiceAclConfig.aclName:
      mode[ 'Ldp.config' ].addCommand( 'ip access-group %s' %
                                       ldpServiceAclConfig.aclName )
   elif options.saveAll:
      mode[ 'Ldp.config' ].addCommand( 'no ip access-group' )

   if Toggles.LdpToggleLib.toggleP2mpToggleEnabled():
      if config.mLdpEnabled or config.staticP2mpFecBindingColl.keys() != [] :
         mldpMode = mode[ MldpConfigMode ].getOrCreateModeInstance( 'mldp' )
         if config.mLdpEnabled: 
            mldpMode[ 'Mldp.config' ].addCommand( 'no shutdown' ) 
         else:
            mldpMode[ 'Mldp.config' ].addCommand( 'shutdown' ) 
         for p2mpid in config.staticP2mpFecBindingColl.items():
            mldpMode[ 'Mldp.config' ].addCommand( 'p2mp-lsp root %s opaque-id %d' 
                 %( p2mpid[0].rootIp, p2mpid[0].gOpaqueId ) )
      elif options.saveAll: 
         mldpMode = mode[ MldpConfigMode ].getOrCreateModeInstance( 'mldp' )
         mldpMode[ 'Mldp.config' ].addCommand( 'shutdown' ) 
         for p2mpid in config.staticP2mpFecBindingColl.items():
            mldpMode[ 'Mldp.config' ].addCommand( 'p2mp-lsp root %s opaque-id %d' 
                 %( p2mpid[0].rootIp, p2mpid[0].gOpaqueId ) )

   # Graceful restart configuration
   if Toggles.LdpToggleLib.toggleGrToggleEnabled():
      if config.grOperFlag.grFlagHelper:
         ldpGrHelperMode = mode[ GrHelperConfigMode ].getOrCreateModeInstance(
            'gr-helper' )
         if config.grMaxRecoveryTimeout != LdpProtoParam.defaultGrMaxRecoveryTimeout:
            ldpGrHelperMode[ 'GrHelper.config' ].addCommand(
               'timer recovery %d' % config.grMaxRecoveryTimeout )
         if ( config.grNeighborLivenessTimeout !=
              LdpProtoParam.defaultGrNeighborLivenessTimeout ):
            ldpGrHelperMode[ 'GrHelper.config' ].addCommand(
               'timer neighbor-liveness %d' % config.grNeighborLivenessTimeout )
      elif options.saveAll:
         mode[ 'Ldp.config' ].addCommand( 'no graceful-restart role helper' )
      if config.grOperFlag.grFlagSpeaker:
         ldpGrSpeakerMode = mode[ GrSpeakerConfigMode ].getOrCreateModeInstance(
            'gr-speaker' )
         if config.grHoldingTimeout != LdpProtoParam.defaultGrHoldingTimeout:
            ldpGrSpeakerMode[ 'GrSpeaker.config' ].addCommand(
               'timer state-holding %d' % config.grHoldingTimeout )
         if config.grReconnectTimeout != LdpProtoParam.defaultGrReconnectTimeout:
            ldpGrSpeakerMode[ 'GrSpeaker.config' ].addCommand(
               'timer reconnect %d' % config.grReconnectTimeout )
      elif options.saveAll:
         mode[ 'Ldp.config' ].addCommand( 'no graceful-restart role speaker' )

   if config.helloRedundancy:
      if config.helloRedundancyTimeout == Tac.endOfTime:
         mode[ 'Ldp.config' ].addCommand(
               'neighbor hello-redundancy duration infinite' )
      elif ( config.helloRedundancyTimeout ==
             LdpProtoParam.defaultHelloRedundancyTimeout ):
         mode[ 'Ldp.config' ].addCommand( 'neighbor hello-redundancy' )
      else:
         mode[ 'Ldp.config' ].addCommand( 'neighbor hello-redundancy duration %d' %
               config.helloRedundancyTimeout )
   elif options.saveAll:
      mode[ 'Ldp.config' ].addCommand( 'neighbor hello-redundancy none' )

   if config.onlyLdpEnabledIntfs:
      mode[ 'Ldp.config' ].addCommand( 'interface disabled default' )
   elif options.saveAll:
      mode[ 'Ldp.config' ].addCommand( 'no interface disabled default' )

   if Toggles.LdpToggleLib.toggleLdpExplicitNullToggleEnabled():
      if config.ldpLabelLocalTerminationMode == (
         LdpLabelLocalTerminationMode.explicitNull ):
         mode[ 'Ldp.config' ].addCommand( 'label local-termination explicit-null' )
      elif options.saveAll:
         mode[ 'Ldp.config' ].addCommand( 'label local-termination implicit-null' )

   if Toggles.MplsToggleLib.toggleEntropyLabelSignalEnabled():
      if config.entropyLabel:
         mode[ 'Ldp.config' ].addCommand( 'entropy-label' )
      elif options.saveAll:
         mode[ 'Ldp.config' ].addCommand( 'no entropy-label' )

   if Toggles.LdpToggleLib.toggleEndOfLibEnabled():
      if config.endOfLib:
         mode[ 'Ldp.config' ].addCommand( 'neighbor end-of-lib' )
      elif options.saveAll:
         mode[ 'Ldp.config' ].addCommand( 'no neighbor end-of-lib' )

   # This should be the last line
   if config.enabled == True:
      mode[ 'Ldp.config' ].addCommand( 'no shutdown' )
   elif options.saveAll:
      mode[ 'Ldp.config' ].addCommand( 'shutdown' )

   # Save interface-level parameters:
   #     * "mpls ldp igp sync"
   #     * "mpls ldp interface"
   saveIntfConfig( root, config.intfConfigColl, options )
   saveIntfLdpEnabled( root, config.ldpEnabledIntf, options )

def Plugin( entityManager ):
   pass
