# Copyright (c) 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import CliSave
from CliMode.LauncherDaemonMode import LauncherDaemonMode
from IpLibConsts import DEFAULT_VRF
import LauncherLib
import Tac
from TypeFuture import TacLazyType
from LauncherDaemonConstants import DEFAULT_OOM_SCORE_ADJ
from ProcMgrLib import OOM_NONESSENTIAL_PROC

AGENT_TYPE = TacLazyType( 'GenericAgent::AgentTypeEnum' )

class DaemonConfigMode( LauncherDaemonMode, CliSave.Mode ):
   def enterCmd( self ):
      return 'daemon %s' % self.param_

   def __init__( self, param ):
      LauncherDaemonMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

CliSave.GlobalConfigMode.addChildMode( DaemonConfigMode ) 
DaemonConfigMode.addCommandSequence( 'Launcher.daemonConfig' )

@CliSave.saver( 'Launcher::AgentConfigDir', LauncherLib.agentConfigCliDirPath,
                requireMounts=( 'daemon/agent/config', 'daemon/acl/config' ) )
def saveAgentConfig( entity, root, sysdbRoot, options, requireMounts ):
   genericAgentCfgDir = requireMounts[ 'daemon/agent/config' ]
   aclConfigDir = requireMounts[ 'daemon/acl/config' ]
   for agentName, agent in entity.agent.iteritems():
      if agent.userDaemon and agentName in genericAgentCfgDir:
         # There shouldn't be a case where we have a userDaemon entry
         # in the Launcher config without a corresponding entry in the
         # genericAgentCfgDir, unless somebody has manually been
         # messing around in Sysdb.
         genericAgentCfg = genericAgentCfgDir[ agentName ]
         # if the agent isn't a "genericAgent" don't generate the daemon config for
         # it. It should take care of itself.
         if genericAgentCfg.agentType != AGENT_TYPE.genericAgent:
            continue

         mode = root[ DaemonConfigMode ].getOrCreateModeInstance( agentName )
         cmds = mode[ 'Launcher.daemonConfig' ]

         # Add the 'executable <exe> arguments <args>' string first,
         # as it is the most useful configuration to inspect.
         argString = ' '.join( arg for arg in agent.argv.itervalues() if arg )
         if not genericAgentCfg.defaultEnabled:
            if agent.exe:
               cmds.addCommand( 'exec %s %s' % ( agent.exe, argString ) )
            elif options.saveAll:
               # If this was default enabled we will add 'no command'
               # at the end of this function in saveAll mode.
               cmds.addCommand( 'no exec' )


         # Save oom_score_adj value if it differs from default. Since TerminAttr's
         # default oom_score_adj is special cased, treat it differently here as well.
         if agentName == "TerminAttr":
            if agent.oomScoreAdj != OOM_NONESSENTIAL_PROC or options.saveAll:
               cmds.addCommand( 'oom score adjustment %s' % agent.oomScoreAdj )
         else:
            if agent.oomScoreAdj != DEFAULT_OOM_SCORE_ADJ or options.saveAll:
               cmds.addCommand( 'oom score adjustment %s' % agent.oomScoreAdj )

         heartbeatCmd = None
         if agent.heartbeatPeriod or options.saveAll:
            heartbeatCmd = 'heartbeat %d' % agent.heartbeatPeriod
         if ( agent.startupGracePeriod != agent.startupGracePeriodDefault 
              or options.saveAll ):
            if heartbeatCmd != None:
               heartbeatCmd += ' initialization-period %d' % agent.startupGracePeriod
            else:
               heartbeatCmd = 'heartbeat 0 initialization-period %d' % (
                  agent.startupGracePeriod )
         if heartbeatCmd:
            cmds.addCommand( heartbeatCmd )

         for option, value in sorted( genericAgentCfg.option.iteritems() ):
            cmds.addCommand( 'option %s value %s' % ( option, value ) )

         # Add 'command /xx/xxx arg1 arg2'. Aesthetically I'd like
         # this to be the first command in the submode, like the
         # 'executable' CLI, but since 'command' starts the agent, we
         # should run it after we set the heartbeat and options.
         if genericAgentCfg.defaultEnabled:
            if agent.exe:
               cmds.addCommand( 'command %s %s' % ( agent.exe, argString ) )
            elif options.saveAll:
               cmds.addCommand( 'no command' )

         # Add 'ip/ipv6 access-group name'.
         aclConfig = aclConfigDir.get( agentName )
         if aclConfig:
            for aclType in ( 'ip', 'ipv6' ):
               aclTypeVrf = Tac.Value( 'Acl::AclTypeAndVrfName', aclType,
                                       DEFAULT_VRF )
               aclName = aclConfig.aclName.get( aclTypeVrf )
               if aclName:
                  cmds.addCommand( '%s access-group %s' % ( aclType, aclName ) )

         # It is important that the 'no shutdown' command appears last
         # in the submode so that the agent starts with all the other
         # configuration already set.
         if genericAgentCfg.defaultEnabled:
            # If it is enabled by default, we need to explicitly show
            # a shutdown when not enabled.
            if not genericAgentCfg.enabled:
               cmds.addCommand( 'shutdown' )
            elif options.saveAll or options.saveAllDetail:
               cmds.addCommand( 'no shutdown' )
         else:
            if genericAgentCfg.enabled:
               cmds.addCommand( 'no shutdown' )
            elif options.saveAll or options.saveAllDetail:
               cmds.addCommand( 'shutdown' )
