#!/usr/bin/env python
# Copyright (c) 2009, 2010, 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import CliSave
import Tac
from CliMode.Lanz import LanzStreamingMode
import Arnet
from IpLibConsts import DEFAULT_VRF
import IraVrfCliSave
import LanzLib

DEF_MAX_CONS = 10

LanzEnabled = Tac.Type( 'Lanz::LanzEnabled' )

class StreamingConfigMode( LanzStreamingMode, CliSave.Mode ):
   def __init__( self, param ):
      LanzStreamingMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

CliSave.GlobalConfigMode.addCommandSequence( 'Lanz.config' )
CliSave.GlobalConfigMode.addCommandSequence( 'Lanz.log' )
CliSave.GlobalConfigMode.addChildMode( StreamingConfigMode,
        after=[ IraVrfCliSave.VrfDefinitionCliSaveMode ] )
StreamingConfigMode.addCommandSequence( 'Lanz.streaming' )

CliSave.GlobalConfigMode.addCommandSequence( 'Lanz.globalThresholdsConfig' )
CliSave.GlobalConfigMode.addCommandSequence( 'Lanz.cpuThresholdsConfig' )
CliSave.GlobalConfigMode.addCommandSequence( 'Lanz.fabricThresholdsConfig' )
CliSave.GlobalConfigMode.addCommandSequence( 'Lanz.globalLogInterval' )
CliSave.GlobalConfigMode.addCommandSequence( 'Lanz.qLenInterval' )
CliSave.GlobalConfigMode.addCommandSequence( 'Lanz.mirroringConfig' )

# Temporary disabling for Berlin release
# CliSave.GlobalConfigMode.addCommandSequence( 'Lanz.percentageThresholdsConfig' )

def saveMirrorDestIntfs( lanzConfig, destIntfs, mirroringHwConfig, cmdSeq ):
   for intf in destIntfs:
      if intf.startswith( 'Gre' ):
         # Mirroring to Gre Tunnel
         greCfg = lanzConfig.greTunnelConfig
         if not greCfg:
            continue

         cmdMirrorGre = 'queue-monitor length mirror destination tunnel '\
                        'mode gre source %s destination %s' % \
                        ( greCfg.srcIp, greCfg.dstIp )
         cmdMirrorGre += ' ttl %d' % greCfg.ttl
         cmdMirrorGre += ' dscp %d' % greCfg.dscp
         cmdMirrorGre += ' protocol %d' % greCfg.protocol
         cmdMirrorGre += ' vrf %s' % greCfg.vrf
         cmdSeq.addCommand( cmdMirrorGre )
      else:
         # Mirorring to a regular interface
         cmdSeq.addCommand( 'queue-monitor length mirror destination %s' % intf )


@CliSave.saver( 'Lanz::Config', 'lanz/config',
                requireMounts=( 'lanz/hardware/status', 'acl/cpconfig/cli',
                                'mirroring/hwconfig' ) )
def saveLanzConfig( entity, root, sysdbRoot, options,
                    requireMounts ):
   saveAll = options.saveAll
   saveAllDetail = options.saveAllDetail

   lanzHwStatus = requireMounts[ 'lanz/hardware/status' ]
   mirroringHwConfig = requireMounts[ 'mirroring/hwconfig' ]

   forceSaveLanzEnabled = entity.enabled == LanzEnabled.platformDefault and saveAll
   if entity.enabled == LanzEnabled.enabledByCli \
         or ( forceSaveLanzEnabled and lanzHwStatus.lanzEnabledByDefault ):
      root[ 'Lanz.config' ].addCommand( 'queue-monitor length' )
   elif entity.enabled == LanzEnabled.disabledByCli \
         or ( forceSaveLanzEnabled and not lanzHwStatus.lanzEnabledByDefault ):
      root[ 'Lanz.config' ].addCommand( 'no queue-monitor length' )

   if entity.forceNotifying:
      root[ 'Lanz.config' ].addCommand( 'queue-monitor length notifying' )
   elif saveAll:
      root[ 'Lanz.config' ].addCommand( 'no queue-monitor length notifying' )

   if entity.logInterval:
      root[ 'Lanz.log' ].addCommand(
                           'queue-monitor length log %d' % entity.logInterval )

   if entity.txLatencyEnabled:
      root[ 'Lanz.config' ].addCommand( 'queue-monitor length tx-latency' )
   elif saveAll:
      root[ 'Lanz.config' ].addCommand( 'no queue-monitor length tx-latency' )

   if lanzHwStatus.updateIntervalSupported:
      defaultUsecs = int( lanzHwStatus.defaultQlenUpdateInterval * 1000000 )
      if entity.qLenInterval != defaultUsecs or saveAll:
         root[ 'Lanz.qLenInterval' ].addCommand(
            'queue-monitor length update-interval %d' % entity.qLenInterval )

   cmdSeq = root[ 'Lanz.config' ]
   if entity.defaultThresholdsConfig.isActive:
      defHighThreshold = entity.defaultThresholdsConfig.defHighThreshold
      if lanzHwStatus.lowThresholdSupported:
         defLowThreshold = entity.defaultThresholdsConfig.defLowThreshold
         cmdSeq.addCommand( 'queue-monitor length default thresholds %d %d'
                            %( defHighThreshold, defLowThreshold ) )
      else:
         cmdSeq.addCommand( 'queue-monitor length default threshold %d'
                            %( defHighThreshold ) )
   elif saveAll:
      if LanzLib.lanzEnabled( entity, lanzHwStatus ):
         if lanzHwStatus.lowThresholdSupported:
            cmdSeq.addCommand( 'default queue-monitor length default thresholds' )
         else:
            cmdSeq.addCommand( 'default queue-monitor length default threshold' )

   aclCpConfig = requireMounts[ 'acl/cpconfig/cli' ]
   streamingVrf = entity.streamingVrf

   ipAclConfig = aclCpConfig.cpConfig[ 'ip' ].serviceAcl.get( DEFAULT_VRF )
   ip6AclConfig = aclCpConfig.cpConfig[ 'ipv6' ].serviceAcl.get( DEFAULT_VRF )
   lanzAclConfig = ipAclConfig.service.get( 'lanz' ) if ipAclConfig else None
   lanzIp6AclConfig = ip6AclConfig.service.get( 'lanz' ) if ip6AclConfig else None

   if entity.streaming or entity.maxConnections != DEF_MAX_CONS or \
          streamingVrf != DEFAULT_VRF or \
          ( lanzAclConfig and lanzAclConfig.aclName ) or \
          ( lanzIp6AclConfig and lanzIp6AclConfig.aclName ) or \
          saveAllDetail:
      mode = root[ StreamingConfigMode ].getOrCreateModeInstance( entity.streaming )
      cmdSeq = mode[ 'Lanz.streaming' ]

      if entity.maxConnections != DEF_MAX_CONS or \
            ( entity.streaming and saveAll ) or saveAllDetail:
         cmdSeq.addCommand( 'max-connections %d' % entity.maxConnections )

      if streamingVrf != DEFAULT_VRF or saveAllDetail:
         cmdSeq.addCommand( 'vrf %s' % streamingVrf )

      # Service Acl IPv4
      if lanzAclConfig and lanzAclConfig.aclName:
         cmdSeq.addCommand( 'ip access-group %s' % lanzAclConfig.aclName )
      elif saveAll:
         cmdSeq.addCommand( 'no ip access-group' )

      # Service Acl IPv6
      if lanzIp6AclConfig and lanzIp6AclConfig.aclName:
         cmdSeq.addCommand( 'ipv6 access-group %s' % lanzIp6AclConfig.aclName )
      elif saveAll:
         cmdSeq.addCommand( 'no ipv6 access-group' )

      if entity.streaming:
         cmdSeq.addCommand( 'no shutdown' )
      elif saveAllDetail:
         cmdSeq.addCommand( 'shutdown' )

   cmdSeq = root[ 'Lanz.globalLogInterval' ]
   if entity.globalLogInterval:
      cmdSeq.addCommand( 'queue-monitor length global-buffer log %d'
            %( entity.globalLogInterval ) ) 

   cmdSeq = root[ 'Lanz.globalThresholdsConfig' ]
   if not entity.globalBufferEnabled:
      cmdSeq.addCommand( 'no queue-monitor length global-buffer' )
   elif saveAll:
      cmdSeq.addCommand( 'default queue-monitor length global-buffer' )

   if entity.globalThresholdsConfig.valid and \
      ( entity.globalThresholdsConfig.highThreshold != \
            lanzHwStatus.defGlobalBufHighThreshold or \
        entity.globalThresholdsConfig.lowThreshold != \
            lanzHwStatus.defGlobalBufLowThreshold ):
      cmdSeq.addCommand( 'queue-monitor length global-buffer thresholds %d %d'
                %( entity.globalThresholdsConfig.highThreshold,
                   entity.globalThresholdsConfig.lowThreshold ) )
   elif saveAll:
      cmdSeq.addCommand( 'queue-monitor length global-buffer thresholds %d %d'
                %( lanzHwStatus.defGlobalBufHighThreshold,
                   lanzHwStatus.defGlobalBufLowThreshold ) )

   cmdSeq = root[ 'Lanz.cpuThresholdsConfig' ]   
   if lanzHwStatus.cpuPortMonitoringSupported:
      portConfig = entity.portConfig.get ( "Cpu" )
      if portConfig:
         if lanzHwStatus.lowThresholdSupported and \
               ( portConfig.highThreshold != lanzHwStatus.defCpuHighThreshold or \
                 portConfig.lowThreshold != lanzHwStatus.defCpuLowThreshold ):
            cmdSeq.addCommand( 'queue-monitor length cpu thresholds %d %d'
                            %( portConfig.highThreshold, portConfig.lowThreshold ) )

         if portConfig.disabled:
            cmdSeq.addCommand( 'no queue-monitor length cpu' )

   cmdSeq = root[ 'Lanz.fabricThresholdsConfig' ]   
   if lanzHwStatus.fabricMonitoringSupported:
      portConfig = entity.portConfig.get ( "Fabric" )
      if portConfig:
         if lanzHwStatus.lowThresholdSupported:
            cmdSeq.addCommand( 'queue-monitor length fabric thresholds %d %d'
                            %( portConfig.highThreshold, portConfig.lowThreshold ) )

         if portConfig.disabled:
            cmdSeq.addCommand( 'no queue-monitor length fabric' )

   cmdSeq = root[ 'Lanz.mirroringConfig' ]
   if entity.mirroringEnabled:
      cmdSeq.addCommand( 'queue-monitor length mirror' )
   elif saveAll:
      cmdSeq.addCommand( 'no queue-monitor length mirror' )

   targetIntfs = Arnet.sortIntf( entity.mirrorDestIntf )
   saveMirrorDestIntfs( entity, targetIntfs, mirroringHwConfig, cmdSeq )

# Temporary disabling for Berlin release
'''
   cmdSeq = root[ 'Lanz.percentageThresholdsConfig' ]
   if entity.percentageThresholdsConfig.isActive:
      if lanzHwStatus.lowThresholdSupported:
         cmdSeq.addCommand(
               'queue-monitor length default thresholds %d percent %d percent'
               %( entity.percentageThresholdsConfig.percentHighThreshold,
                  entity.percentageThresholdsConfig.percentLowThreshold ) )
      else:
         cmdSeq.addCommand( 'queue-monitor length default threshold %d percent'
                         % entity.percentageThresholdsConfig.percentHighThreshold )
   elif saveAll:
      cmdSeq.addCommand( 'no queue-monitor length default thresholds' )
'''
