# Copyright (c) 2009, 2010 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

#-------------------------------------------------------------------------------
# This module implements saving the Lag CLI, including Port-Channel interfaces
# and interface-specific configuration.
#-------------------------------------------------------------------------------
import CliSave, IntfCliSave
from IntfCliSave import IntfConfigMode
import LacpConstants
import re
import Tac
import LacpLib
from EthIntfLib import ethPhyIntfConfigIter
from Toggles.LagToggleLib import toggleLacpTimeoutMultiplierEnabled
import Ethernet

isRecirc = Tac.Type( "Arnet::PortChannelIntfId" ).isRecirc
MinLinksTimeoutBase = Tac.Type( "Lag::MinLinksTimeoutBase" )
LacpTimeoutMultiplier = Tac.Type( "Lacp::TimeoutMultiplier" )

CliSave.GlobalConfigMode.addCommandSequence( 'Lag.global',
                                             before=[ IntfConfigMode ] )

IntfConfigMode.addCommandSequence( 'Lag.lagIntf', after=[ 'Arnet.intf' ] )
IntfConfigMode.addCommandSequence( 'Lag.lagConfig', after=[ 'Ebra.switchport' ] )

@CliSave.saver( 'Interface::EthLagIntfConfig', 'interface/config/eth/lag',
                attrName='intfConfig',
                requireMounts=( 'interface/status/all', 'bridging/hwcapabilities' ) )
def saveEthLagIntfConfig( entity, root, sysdbRoot, options,
                          requireMounts ):

   # Save the baseclass (Arnet::IntfConfig) attributes.
   IntfCliSave.saveIntfConfig( entity, root, sysdbRoot, options,
                               requireMounts )

   mode = root[ IntfConfigMode ].getOrCreateModeInstance( entity.intfId )
   cmds = mode[ 'Lag.lagIntf' ]
   saveAll = options.saveAll

   if entity.minLinks != entity.minLinksDefault:
      cmds.addCommand( 'port-channel min-links %d' % entity.minLinks )
   elif saveAll:
      cmds.addCommand( 'no port-channel min-links' )

   if entity.fallback == 'fallbackStatic':
      cmds.addCommand( 'port-channel lacp fallback static' )
   elif entity.fallback == 'fallbackIndividual':
      cmds.addCommand( 'port-channel lacp fallback individual' )
   elif saveAll:
      cmds.addCommand( 'no port-channel lacp fallback' )

   if entity.fallbackTimeout != entity.fallbackTimeoutDefault or saveAll:
      cmds.addCommand( 'port-channel lacp fallback timeout %d' % (
                        entity.fallbackTimeout ) )
      
   if entity.l2Mtu != 0:
      cmds.addCommand( 'l2 mtu %u' % entity.l2Mtu )
   elif saveAll:
      cmds.addCommand( 'no l2 mtu' )

   if entity.l2Mru != 0:
      cmds.addCommand( 'l2 mru %u' % entity.l2Mru )
   elif saveAll:
      cmds.addCommand( 'no l2 mru' )

   if isRecirc( entity.intfId ):
      if len( entity.recircFeature ) > 0:
         vxlanVal = Tac.enumValue(
            "Lag::Recirc::RecircFeature", "vxlan" )
         teleInbVal = Tac.enumValue(
            "Lag::Recirc::RecircFeature", "telemetryinband" )
         openFlowVal = Tac.enumValue(
            "Lag::Recirc::RecircFeature", "openflow" )
         cpumirrorVal = Tac.enumValue(
            "Lag::Recirc::RecircFeature", "cpumirror" )
   
         if vxlanVal in entity.recircFeature:
            cmds.addCommand( 'switchport recirculation features %s' %
                             ( 'vxlan' ) )
         elif teleInbVal in entity.recircFeature:
            cmds.addCommand( 'switchport recirculation features %s' %
                             ( 'telemetry inband' ) )
         elif openFlowVal in entity.recircFeature:
            cmds.addCommand( 'switchport recirculation features %s' %
                             ( 'openflow' ) )
         elif cpumirrorVal in entity.recircFeature:
            cmds.addCommand( 'switchport recirculation features %s' %
                             ( 'cpu-mirror' ) )
      elif saveAll:
         cmds.addCommand( 'no switchport recirculation features' )

   if entity.lacpLoopEnable:
      cmds.addCommand( 'port-channel lacp loopback' )
   elif saveAll:
      cmds.addCommand( 'no port-channel lacp loopback' )

   hwCaps = requireMounts[ 'bridging/hwcapabilities' ]
   if hwCaps.staticMacOnRoutedIntfSupported:
      if entity.addr != '00:00:00:00:00:00':
         rmac = Ethernet.convertMacAddrCanonicalToDisplay( entity.addr )
         cmds.addCommand( 'mac-address router %s' % rmac )
      elif saveAll:
         cmds.addCommand( 'no mac-address router' )

   # # Display defaults when this command is supported.
   # if entity.maxBundle != entity.maxBundleDefault:
   #    cmds.addCommand( 'lacp max-bundle %d' % entity.maxBundle )
   # elif saveAll:
   #    cmds.addCommand( 'no lacp max-bundle' )

@CliSave.saver( 'Lag::Input::Config', 'lag/input/config/cli',
                requireMounts = ( 'interface/config/eth/phy/slice',
                                  'lag/lacp/input/config/cli' ) )
def saveLagConfig( entity, root, sysdbRoot, options,
                   requireMounts ):

   cmds = root[ 'Lag.global' ]
   # Get the list of interfaces based saveAll type
   ethPhyIntfConfigSliceDir = requireMounts[ 'interface/config/eth/phy/slice' ]

   allEthPhyIntfConfigs = ethPhyIntfConfigIter( ethPhyIntfConfigSliceDir )
   
   if entity.memberStatusLogEnabled == False:
      cmds.addCommand( 'no logging event port-channel member-status global' )
   elif options.saveAll:
      cmds.addCommand( 'logging event port-channel member-status global' )

   if entity.minLinksTimeoutBase != MinLinksTimeoutBase.timeoutDefault:
      cmds.addCommand( 'port-channel min-links review interval %d' \
                       % entity.minLinksTimeoutBase )
   elif options.saveAll:
      cmds.addCommand( 'no port-channel min-links review interval' )

   if options.saveAll:
      cfgIntfNames = []

      cfgIntfNames = [ intfName for intfName in allEthPhyIntfConfigs
                       if LacpLib.intfSupportsLag( intfName ) ]
   else:
      cfgIntfNames = entity.phyIntf

   for intfName in cfgIntfNames:
      intfConfig = entity.phyIntf.get( intfName )
      if not intfConfig:
         if options.saveAll:
            intfConfig = Tac.newInstance( 'Lag::EthPhyIntfLagConfig', intfName )
         else:
            continue
      # save intfConfig
      saveEthPhyIntfLagConfig( intfConfig, root, sysdbRoot, options, requireMounts )

def saveEthPhyIntfLagConfig( entity, root, sysdbRoot, options, requireMounts ):
   mode = root[ IntfConfigMode ].getOrCreateModeInstance( entity.intfId )
   cmds = mode[ 'Lag.lagConfig' ]
   saveAll = options.saveAll

   lacpCliConfig = requireMounts[ 'lag/lacp/input/config/cli' ]
   # save lacp port-id before channel-group cli command
   if entity.portId != LacpConstants.LacpPortIdDefault:
      cmds.addCommand( 'lacp port-id %d' % entity.portId )
   elif saveAll:
      cmds.addCommand( 'no lacp port-id' )

   if entity.lag:
      if not isRecirc( entity.lag.intfId ):
         m = re.match( 'Port-Channel(\d+)', entity.lag.intfId )
         if m:
            modemap = { 'lacpModeOff': 'on',
                        'lacpModeActive': 'active', 'lacpModePassive': 'passive' }
            cmds.addCommand( 'channel-group %s mode %s' % (
                             m.group(1), modemap[ entity.mode ] ) )
      else:
         m = re.match( 'Recirc-Channel(\d+)', entity.lag.intfId )
         if m:
            cmds.addCommand( 'channel-group recirculation %s' % m.group( 1 ) )
   elif saveAll:
      cmds.addCommand( 'no channel-group' )

   if entity.timeout == 'lacpShortTimeout':
      cmds.addCommand( 'lacp timer fast' )
   elif saveAll:
      cmds.addCommand( 'lacp timer normal' )
   if toggleLacpTimeoutMultiplierEnabled():
      if entity.timeoutMult != LacpTimeoutMultiplier.defaultValue:
         cmds.addCommand( 'lacp timer multiplier %d' % entity.timeoutMult )
      elif saveAll:
         cmds.addCommand( 'lacp timer multiplier %d'
                          % LacpTimeoutMultiplier.defaultValue )
   if entity.priority != LacpConstants.PortPriorityDefault or saveAll:
      cmds.addCommand( 'lacp port-priority %d' % entity.priority )

   rateLimitIntfEnable = lacpCliConfig.rateLimitIntfEnable
   if entity.intfId in rateLimitIntfEnable:
      if rateLimitIntfEnable[ entity.intfId ]:
         cmds.addCommand( 'lacp rate-limit' )
      else:
         cmds.addCommand( 'no lacp rate-limit' )
   elif saveAll:
      cmds.addCommand( 'default lacp rate-limit' )

   if entity.fwdEnabled == False:
      cmds.addCommand( 'forwarding port-channel disabled' )
   elif saveAll:
      cmds.addCommand( 'no forwarding port-channel disabled' )

@CliSave.saver( 'Lacp::CliConfig', 'lag/lacp/input/config/cli' )
def saveLacpConfig( entity, root, sysdbRoot, options ):
   cmds = root[ 'Lag.global' ]

   if entity.priority != LacpConstants.SystemPriorityDefault or options.saveAll:
      cmds.addCommand( 'lacp system-priority %d' % entity.priority )
   if entity.portIdRange != Tac.Value( 'Lacp::PortIdRange' ):
      cmds.addCommand( 'lacp port-id range %d %d' % \
                ( entity.portIdRange.portIdMin, entity.portIdRange.portIdMax ) )
   elif options.saveAll:
      cmds.addCommand( 'no lacp port-id range' )
   if entity.enableAudit != False: 
      # don't show enableAudit if False (unset) even if saveAll --- it's hidden!
      cmds.addCommand( 'lacp audit' )
   if not entity.rateLimitEnable:
      cmds.addCommand( 'no lacp rate-limit default' )
   elif options.saveAll:
      cmds.addCommand( 'lacp rate-limit default' )

@CliSave.saver( 'Lag::Input::LacpOverrideDir', 'lag/input/lacpoverride/cli' )
def saveLacpOverride( entity, root, sysdbRoot, options ):
   cmds = root[ 'Lag.global' ]
   systemIds = entity.systemId

   for intfId in systemIds:
      mode = root[ IntfConfigMode ].getOrCreateModeInstance( intfId )
      cmds = mode[ 'Lag.lagIntf' ]
      ethAddr = Tac.Value( 'Arnet::EthAddr', 
                           stringValue=systemIds[ intfId ] )
      cmds.addCommand( 'lacp system-id %s' % ethAddr.displayString )
