# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.
import Tac
import CliSave
import EthIntfUtil
from IntfCliSave import IntfConfigMode
from CliMode.L2Protocol import L2ProtocolModeBase, L2ProtocolProfileModeBase
from EbraLib import ( l2ProtocolFromMacAddress, tacL2PtTagFormat,
                      l2PtProtocolToCliToken )

#-------------------------------------------------------------------------------
# Object used for saving commands in "l2-protocol" mode.
#-------------------------------------------------------------------------------
class L2ProtocolMode( L2ProtocolModeBase, CliSave.Mode ):
   def __init__( self, param ):
      L2ProtocolModeBase.__init__( self, param )
      CliSave.Mode.__init__( self, param )

CliSave.GlobalConfigMode.addChildMode( L2ProtocolMode )
L2ProtocolMode.addCommandSequence( 'Ebra.l2protocol' )

#-------------------------------------------------------------------------------
# Object used for saving commands in "forwarding profile <profileName>" mode.
#-------------------------------------------------------------------------------
class L2ProtocolProfileMode( L2ProtocolProfileModeBase, CliSave.Mode ):
   def __init__( self, param ):
      L2ProtocolProfileModeBase.__init__( self, param )
      CliSave.Mode.__init__( self, self.longModeKey )
L2ProtocolMode.addChildMode( L2ProtocolProfileMode )
L2ProtocolProfileMode.addCommandSequence( 'Ebra.forwardingProfile' )

def _hasComments( commentKey, requireMounts ):
   return commentKey in requireMounts[ 'cli/config' ].comment

@CliSave.saver( 'Ebra::L2Pt::L2PtProfileConfig',
                'l2protocolforwarding/profileconfig',
                requireMounts = ( 'cli/config', ) )
def saveConfig( l2PtProfileConfig, root, sysdbRoot, options,
                requireMounts ):
   if not l2PtProfileConfig.profile and not _hasComments( 'l2-protocol',
                                                          requireMounts ):
      return
   profileRoot = root[ L2ProtocolMode ].getOrCreateModeInstance( None )
   for profileName, profile in l2PtProfileConfig.profile.iteritems():
      l2ProtocolProfileMode = profileRoot[ L2ProtocolProfileMode
                                         ].getOrCreateModeInstance( profileName )
      cmds = l2ProtocolProfileMode[ 'Ebra.forwardingProfile' ]
      for protocolInfo in profile.seqToProtocolInfo.itervalues():
         protocol = l2PtProtocolToCliToken[ l2ProtocolFromMacAddress(
            protocolInfo.l2ProtocolMatch.addr ) ]
         tagFormat = protocolInfo.l2ProtocolMatch.tagFormat
         action = protocolInfo.action.action
         if tagFormat == tacL2PtTagFormat.all:
            cmd = '%s %s' % ( protocol, action )
         else:
            cmd = '%s %s %s' % ( protocol, tagFormat, action )
         cmds.addCommand( cmd )

IntfConfigMode.addCommandSequence( 'Ebra.IntfL2ProtocolForwardingProfile' )

@CliSave.saver( 'Ebra::L2Pt::L2PtIntfConfig',
                'l2protocolforwarding/intfconfig',
                requireMounts = ( 'interface/config/eth/phy/all', ) )
def saveIntfConfigs( intfConfig, root, sysdbRoot, options, requireMounts ):
   if options.saveAll:
      # Get all eligible physical ports
      phyIntfConfig = requireMounts[ 'interface/config/eth/phy/all' ]
      phyIntfNames = EthIntfUtil.allPhyEthernetInterfaceNames( phyIntfConfig )
   else:
      phyIntfNames = intfConfig.intfToProfile

   for intfName in phyIntfNames:
      mode = root[ IntfConfigMode ].getOrCreateModeInstance( intfName )
      cmds = mode[ 'Ebra.IntfL2ProtocolForwardingProfile' ]
      profileName = intfConfig.intfToProfile.get( intfName )
      if not profileName:
         if options.saveAll:
            cmds.addCommand( 'no l2-protocol forwarding profile' )
      else:
         cmds.addCommand( 'l2-protocol forwarding profile %s' % profileName )
