# Copyright (c) 2006-2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import CliSave
import Tac

from CliMode.Ira import RouterKernelMode
from CliMode.Ira import NexthopGroupConfigBase
from Arnet.MplsLib import getLabelStack
from Arnet.MplsLib import labelStackToString
from IntfCliSave import IntfConfigMode
from IpLibConsts import DEFAULT_VRF
from RoutingIntfUtils import allIpIntfNames

from RoutingConsts import (
      defaultStaticRouteName,
      defaultStaticRoutePreference,
      defaultStaticRouteTag,
      defaultStaticRouteMetric
)
from NexthopGroupConsts import (
      EntryCounterType,
      NexthopGroupType,
      ipTunnelNexthopGroupTypes,
      greKeyIngressIntf,
      GreKeyType,
)
from NexthopGroupUtils import (
      nexthopGroupCliString,
      nexthopGroupCliStringToTacType,
)

ipIntfCmdSeq = 'Ira.ipIntf'
IntfConfigMode.addCommandSequence( ipIntfCmdSeq,
   after=[ 'Arnet.intf', 'Arnet.l2l3barrier' ] )
# The service routing command must appear in config before the
# interface configuration as there are interface specific commands
# starting with "service". When "service routing" is parsed while in
# inteface configuration mode it results in an ambiguous command
# error (see BUG163576).
CliSave.GlobalConfigMode.addCommandSequence( 'Ira.serviceRouting',
                                             before=[ IntfConfigMode ] )
CliSave.GlobalConfigMode.addCommandSequence( 'Ira.routes', after=[ IntfConfigMode ] )
CliSave.GlobalConfigMode.addCommandSequence( 'Ira.routing',
                                             after=[ IntfConfigMode ] )

optimizeProfileType = Tac.Type( 'Routing::Hardware::OptimizeProfileType' )
tristateBoolEnum = Tac.Type( "Ip::TristateBool" )

#---------------------------------------------------------------------------------
# Cli savers
#---------------------------------------------------------------------------------

routingConsts = Tac.Value( "Ira::RoutingConsts", defaultStaticRoutePreference,
                           defaultStaticRouteTag, defaultStaticRouteName,
                           defaultStaticRouteMetric )

@CliSave.saver( 'Routing::Config', 'routing/config' )
def saveRoutingAttr( entity, root, sysdbRoot, options ):
   if options.intfFilter:
      return
   saveRoutingAttribute( entity, root, sysdbRoot, options, vrfName=None )

@CliSave.saver( 'Routing::VrfConfigDir', 'routing/vrf/config' )
def saveVrfRoutingAttr( entity, root, sysdbRoot, options ):
   if options.intfFilter:
      return
   for vrfname in sorted( entity.vrfConfig ):
      saveRoutingAttribute( entity.vrfConfig[ vrfname ], root, sysdbRoot,
            options, vrfname )

def saveRoutingAttribute( entity, root, sysdbRoot, options, vrfName ):
   vrfString = ' vrf %s' % vrfName if vrfName else ''
   if entity.routing:
      if entity.addresslessForwarding == tristateBoolEnum.isTrue:
         root[ 'Ira.routing' ].addCommand( "ip routing ipv6 interfaces %s" %
               vrfString )
      else:
         root[ 'Ira.routing' ].addCommand( "ip routing%s" % vrfString )
   else:
      root[ 'Ira.routing' ].addCommand( "no ip routing%s" % vrfString )

@CliSave.saver( 'Routing::RouteListConfig', 'routing/routelistconfig' )
def saveRoutingTable( entity, root, sysdbRoot, options ):
   if options.intfFilter:
      return
   saveStaticRoutes( entity, root, sysdbRoot, options, vrfName=None )

@CliSave.saver( 'Routing::VrfRouteListConfigDir', 'routing/vrf/routelistconfig' )
def saveVrfRoutingTables( entity, root, sysdbRoot, options ):
   if options.intfFilter:
      return
   for vrfname in sorted( entity.vrfConfig ):
      saveStaticRoutes( entity.vrfConfig[ vrfname ], root, sysdbRoot,
                        options, vrfname )

def saveStaticRoutes( entity, root, sysdbRoot, options, vrfName ):
   saveAll = options.saveAll
   vrfString = ' vrf %s' % vrfName if vrfName else ''
   routeCmd = 'ip route%s ' % vrfString

   saver = Tac.Value( "Ira::StaticRoutesCliSaver" )
   cmds = saver.save( entity, saveAll, routeCmd, routingConsts )
   for cmd in cmds.split( '\n' ):
      if cmd != '':
         root[ 'Ira.routes' ].addCommand( cmd )

@CliSave.saver( 'Routing::Config', 'routing/config' )
def saveIpIcmpRedirectConfig( entity, root, sysdbRoot, options ):
   saveAll = options.saveAll
   if saveAll and entity.sendRedirects:
      root[ 'Ira.routing' ].addCommand( "ip icmp redirect")
   elif not entity.sendRedirects:
      root[ 'Ira.routing' ].addCommand( "no ip icmp redirect")

@CliSave.saver( 'Routing::Config', 'routing/config' )
def saveIpIcmpSrcIntfConfigDefault( entity, root, sysdbRoot, options ):
   saveIpIcmpSrcIntfConfig( entity, root, sysdbRoot, options, vrfName=None )

@CliSave.saver( 'Routing::Config', 'routing/vrf/config', attrName = 'vrfConfig' )
def saveIpIcmpSrcIntfConfigVrf( entity, root, sysdbRoot, options ):
   saveIpIcmpSrcIntfConfig( entity, root, sysdbRoot, options, entity.name )

# requires that entity be a Routing::Config (as opposed to working on
# Routing::ConfigGen)
def saveIpIcmpSrcIntfConfig( entity, root, sysdbRoot, options, vrfName ):
   if vrfName:
      vrfString = ' vrf %s' % vrfName
   else:
      vrfString = ''
   saveAll = options.saveAll
   if( entity.icmpSrcIntf != "" ):
      root[ 'Ira.routing' ].addCommand( "ip icmp source-interface %s%s" %
                                        ( entity.icmpSrcIntf, vrfString ) )
   elif( saveAll ):
      root[ 'Ira.routing' ].addCommand( "no ip icmp source-interface%s" %
                                        vrfString )

@CliSave.saver( 'Routing::Hardware::Config', 'routing/hardware/config',
     requireMounts=( 'routing/hardware/status', ) )
def saveFibFilterConfig( entity, root, sysdbRoot, options, requireMounts ):
   routingHwStatus = requireMounts[ 'routing/hardware/status' ]
   if not routingHwStatus.fibSuppressionSupported:
      return
   saveAll = options.saveAll
   if( entity.fibSuppression ):
      root[ 'Ira.routing' ].addCommand(
                            "ip fib compression redundant-specifics filter" )
   elif( saveAll ):
      root[ 'Ira.routing' ].addCommand(
                            "no ip fib compression redundant-specifics filter" )

@CliSave.saver( 'Ip::Config', 'ip/config',
                requireMounts=( 'interface/config/all', 'interface/status/all',
                                'l3/intf/config' ) )
def saveIpConfig( entity, root, sysdbRoot, options, requireMounts ):
   # Interface config
   if options.saveAllDetail:
      cfgIntfNames = allIpIntfNames( sysdbRoot, includeEligible=True,
                                     requireMounts=requireMounts )
   elif options.saveAll:
      # IP configuration is allowed on switchport interfaces as well.
      # Display config on all ip interfaces and switchports with non-default
      # config.
      ipIntfNames = allIpIntfNames( sysdbRoot, requireMounts=requireMounts )
      cfgIntfNames = set( ipIntfNames + entity.ipIntfConfig.keys() )
   else:
      cfgIntfNames = entity.ipIntfConfig

   for intfName in cfgIntfNames:
      if options.intfFilter and intfName not in options.intfFilter:
         continue
      intfConfig = entity.ipIntfConfig.get( intfName )
      if not intfConfig:
         if options.saveAll:
            intfConfig = Tac.newInstance( 'Ip::IpIntfConfig', intfName )
            intfConfig.l3Config = Tac.newInstance( 'L3::Intf::Config', intfName )
         else:
            continue

      saveIpIntfConfig( intfConfig, root, sysdbRoot, options )

def saveIpIntfConfig( entity, root, sysdbRoot, options ):
   saveAll = options.saveAll
   if entity.intfId.startswith( "Internal" ):
      # Internal interface configuration. Abort
      # BUG944 - need a more general way of doing this
      return

   mode = root[ IntfConfigMode ].getOrCreateModeInstance( entity.intfId )
   cmds = mode[ 'Ira.ipIntf' ]

   if entity.vrf != DEFAULT_VRF:
      assert entity.vrf != ''
      cmds.addCommand( "vrf %s" % entity.vrf )

   if entity.proxyArpEnabled:
      cmds.addCommand( 'ip proxy-arp' )
   elif saveAll:
      # proxy arp is disabled by default
      cmds.addCommand( 'no ip proxy-arp' )

   if entity.localProxyArpEnabled:
      cmds.addCommand( 'ip local-proxy-arp' )
   elif saveAll:
      cmds.addCommand( 'no ip local-proxy-arp' )

   if entity.gratuitousArpAccepted:
      cmds.addCommand( 'arp gratuitous accept' )
   elif saveAll:
      cmds.addCommand( 'no arp gratuitous accept' )

   if entity.addrSource == 'manual':
      if entity.addrWithMask.address != '0.0.0.0':
         cmds.addCommand( "ip address %s/%d" % ( entity.addrWithMask.address,
                                                 entity.addrWithMask.len ) )
      elif entity.unnumberedIntfId:
         cmds.addCommand( 'ip address unnumbered %s' % entity.unnumberedIntfId )
      elif saveAll:
         cmds.addCommand( "no ip address" )
   elif entity.addrSource == 'dhcp':
      cmds.addCommand( "ip address dhcp" )

   # Secondary addresses are not supported with DHCP at this moment
   if entity.addrSource != 'dhcp':
      for a in entity.secondaryWithMask:
         cmds.addCommand( "ip address %s/%d secondary" % ( a.address, a.len ) )

   if entity.addrSource == 'dhcp':
      if entity.defaultRouteSource == 'dhcp':
         cmds.addCommand( "dhcp client accept default-route" )
      elif saveAll:
         cmds.addCommand( "no dhcp client accept default-route" )

   if entity.urpf.mode != 'disable':
      if entity.urpf.mode == 'strict':
         cmd = 'ip verify unicast source reachable-via rx'
      elif entity.urpf.mode == 'loose':
         cmd = 'ip verify unicast source reachable-via any'
      else:
         cmd = 'ip verify unicast source reachable-via rx allow-default'

      if entity.urpf.acl != "":
         cmd += ' exception-list ' + entity.urpf.acl
      cmds.addCommand( cmd )
   elif saveAll:
      cmds.addCommand( "no ip verify unicast" )

   intfId = entity.intfId
   if ( intfId.startswith( "Ethernet" ) or
      intfId.startswith( "Port-Channel" ) or
      intfId.startswith( "Vlan" ) or
      intfId.startswith( "Test" ) ):
      if entity.directedBroadcastEnabled:
         cmds.addCommand( "ip directed-broadcast" )
      elif saveAll:
         cmds.addCommand( "no ip directed-broadcast" )

   if not entity.attachedRoutes:
      cmds.addCommand( 'no ip attached-routes' )
   elif saveAll:
      cmds.addCommand( 'ip attached-routes' )

@CliSave.saver( 'Routing::Hardware::Config', 'routing/hardware/config',
                requireMounts=( 'routing/hardware/status', ) )
def saveResilientEcmpConfig( entity, root, sysdbRoot, options,
                             requireMounts ):
   routingHwStatus = requireMounts[ 'routing/hardware/status' ]
   if not routingHwStatus.resilientEcmpSupported:
      return
   cmds = root['Ira.routing']
   if not entity.resilientEcmpPrefix.keys():
      if options.saveAll or options.saveAllDetail:
         cmds.addCommand( 'no ip hardware fib ecmp resilience' )
      return
   for prefix, ecmpConfig in entity.resilientEcmpPrefix.items() :
      cmds.addCommand(
         "ip hardware fib ecmp resilience %s capacity %d redundancy %d" \
            % ( prefix.stringValue , ecmpConfig.capacity, \
                   ecmpConfig.redundancy ) )

@CliSave.saver( 'Routing::Hardware::DlbConfig', 'routing/hardware/dlb/config',
                requireMounts=( 'routing/hardware/status', ) )
def saveDlbConfig( entity, root, sysdbRoot, options, requireMounts ):
   routingHwStatus = requireMounts[ 'routing/hardware/status' ]
   if not routingHwStatus.dlbEcmpSupported:
      return
   cmds = root[ 'Ira.routing' ]
   if entity.globalDlbEcmpEnable:
      cmds.addCommand( 'ip hardware fib load-balance distribution dynamic' )
   elif options.saveAll or options.saveAllDetail:
      cmds.addCommand( 'no ip hardware fib load-balance distribution' )

@CliSave.saver( 'Routing::Hardware::Config', 'routing/hardware/config',
      requireMounts = ( 'routing/config', 'routing/hardware/status', ) )
def saveIpRoutingHardwareConfig( entity, root, sysdbRoot, options, requireMounts ):
   saveAll = options.saveAll
   routingConfig = requireMounts[ 'routing/config' ]
   routingHwStatus = requireMounts[ 'routing/hardware/status' ]
   if not entity.icmpUnreachable:
      cmd = 'ip icmp rate-limit-unreachable 0'
      root[ 'Ira.routing' ].addCommand( cmd )
   elif ( saveAll and routingConfig.routing ) or options.saveAllDetail:
      cmd = 'no ip icmp rate-limit-unreachable 0'
      root[ 'Ira.routing' ].addCommand( cmd )
   if ( routingHwStatus.optimizeProfileSettingList and
        len( routingHwStatus.optimizeProfileSettingList.optimizeProfileSetting ) and
        entity.optimizeProfileName and
        entity.optimizeProfileName in
        routingHwStatus.optimizeProfileSettingList.optimizeProfileSetting ):
      profileName = entity.optimizeProfileName
      if profileName == optimizeProfileType.urpfInternet:
         profileName = "urpf-internet"
      cmd = 'ip hardware fib optimize prefixes profile %s' % \
            profileName
      root[ 'Ira.routing' ].addCommand( cmd )
   elif routingHwStatus.prefixLenSupportedInExactMatch:
      prefixes = ''
      for prefixLen in routingHwStatus.prefixLenSupportedInExactMatch:
         if prefixLen in entity.prefixLenInExactMatch and \
                entity.prefixLenInExactMatch[ prefixLen ] == prefixLen:
            prefixes = '%s %d' % ( prefixes, prefixLen )
      expandPrefixes = ''
      compressPrefixes = ''
      urpf = ''
      for prefixLen in entity.prefixLenInExactMatch:
         if entity.prefixLenInExactMatch[ prefixLen ] != prefixLen:
            if entity.prefixLenInExactMatch[ prefixLen ] > prefixLen:
               expandPrefixes = '%s %d' % ( expandPrefixes, prefixLen )
            else:
               compressPrefixes = '%s %d' % ( compressPrefixes, prefixLen )
      if routingHwStatus.urpfExactMatchSupported:
         if entity.urpfExactMatchEnabled:
            urpf = 'urpf'
      if len( prefixes ):
         cmd = 'ip hardware fib optimize prefix-length%s' % prefixes
         if len( expandPrefixes ):
            cmd = '%s expand %s' % ( cmd, expandPrefixes )
         if len( compressPrefixes ):
            cmd = '%s compress %s' % ( cmd, compressPrefixes )
         if urpf:
            cmd = '%s %s' % ( cmd, urpf )
         root[ 'Ira.routing' ].addCommand( cmd )
      elif saveAll or options.saveAllDetail:
         cmd = 'no ip hardware fib optimize prefixes'
         root[ 'Ira.routing' ].addCommand( cmd )
   disabledVrfs = [ vrf for vrf in entity.optimizeDisabledVrf ]
   if disabledVrfs:
      cmd = 'ip hardware fib optimize disable-vrf ' + ' '.join( disabledVrfs )
      root[ 'Ira.routing' ].addCommand( cmd )
   elif saveAll or options.saveAllDetail:
      cmd = 'no ip hardware fib optimize disable-vrf'
      root[ 'Ira.routing' ].addCommand( cmd )


class NexthopGroupConfigSaveMode( NexthopGroupConfigBase, CliSave.Mode ):
   def __init__( self, ngParam ):
      nexthopGroupName, nexthopGroupType = ngParam
      NexthopGroupConfigBase.__init__( self, nexthopGroupName, nexthopGroupType )
      CliSave.Mode.__init__( self, nexthopGroupName )

   def __cmp__( self, other ):
      return cmp( self.nexthopGroupName_, other.nexthopGroupName_)

CliSave.GlobalConfigMode.addCommandSequence( 'NexthopGroup.global',
                                             after=[ IntfConfigMode ] )
CliSave.GlobalConfigMode.addChildMode( NexthopGroupConfigSaveMode,
                                       after=[ 'NexthopGroup.global' ] )
NexthopGroupConfigSaveMode.addCommandSequence( 'NexthopGroup.cfg' )

def getNh( ng, idx ):
   destIpIntf = ng.destinationIpIntf[ idx ]
   nh = destIpIntf.destIp.stringValue
   if destIpIntf.destIp.isLinkLocal:
      nh = "%s%%%s" % ( nh, destIpIntf.intfId )
   return nh

@CliSave.saver( 'Routing::NexthopGroup::ConfigInput',
                'routing/nexthopgroup/input/cli' )
def saveNexthopGroupConfig( entity, root, sysdbRoot, options ):
   for nexthopGroupName in sorted( entity.nexthopGroup ):
      ng = entity.nexthopGroup[ nexthopGroupName ]
      if ng.dynamic:
         continue
      mode = root[ NexthopGroupConfigSaveMode ].getOrCreateModeInstance(
         ( nexthopGroupName, nexthopGroupCliString( ng.type ) ) )
      cmds = mode[ 'NexthopGroup.cfg' ]
      if ng.size:
         cmds.addCommand( 'size %s ' % ng.size )
      elif options.saveAll:
         cmds.addCommand( 'no size ' )
      if ng.type in ipTunnelNexthopGroupTypes and ng.ttl:
         cmds.addCommand( 'ttl %s ' % ng.ttl )
      if ng.type in ipTunnelNexthopGroupTypes:
         if not ng.sourceIp.isAddrZero:
            cmds.addCommand( 'tunnel-source %s ' % ng.sourceIp )
         elif ng.intfId != "":
            cmds.addCommand( 'tunnel-source intf %s ' % ng.intfId )
      if ng.entryCounterType == EntryCounterType.unshared:
         cmds.addCommand( 'entry counters unshared' )
      if ng.hierarchicalFecsEnabled:
         cmds.addCommand( 'fec hierarchical' )
      if ng.dlbEcmpEnable:
         cmds.addCommand( 'dynamic-load-balancing' )
      elif options.saveAll:
         cmds.addCommand( 'no dynamic-load-balancing' )

      # save GRE key config
      assert ng.greKeyType in ( GreKeyType.noGreKey, GreKeyType.ingressIntf )
      if ng.greKeyType == GreKeyType.ingressIntf:
         cmds.addCommand( 'tunnel-key %s ' % greKeyIngressIntf )

      ngType = nexthopGroupCliStringToTacType( mode.nexthopGroupType )
      size = max( ng.destinationIpIntf.keys() ) + 1 if \
                  ng.destinationIpIntf.keys() else 0
      for idx in range ( ng.size if ng.size != 0 else size ):
         if not ng.destinationIp( idx ).isAddrZero:
            if ngType == NexthopGroupType.ip:
               cmds.addCommand( 'entry %s nexthop %s ' %
                                ( idx, ng.destinationIp( idx ).stringValue ) )
            elif ngType == NexthopGroupType.mpls:
               labelStack = getLabelStack( ng.mplsLabelStack[ idx ] )
               labelStackString = labelStackToString( labelStack )
               if labelStack != []:
                  nh = getNh( ng, idx )
                  cmds.addCommand(
                     'entry %d push label-stack %s nexthop %s' %
                     ( idx, labelStackString, nh ) )
               else:
                  cmds.addCommand(
                     'entry %d nexthop %s' %
                     ( idx, ng.destinationIp( idx ).stringValue ) )
            elif ngType == NexthopGroupType.mplsOverGre:
               labelStack = getLabelStack( ng.mplsLabelStack[ idx ] )
               labelStackString = labelStackToString( labelStack )
               cmds.addCommand(
                  'entry %d push label-stack %s tunnel-destination %s' %
                  ( idx, labelStackString, ng.destinationIp( idx ).stringValue ) )
            elif ngType == NexthopGroupType.mplsOverUdp:
               labelStack = getLabelStack( ng.mplsLabelStack[ idx ] )
               labelStackString = labelStackToString( labelStack )
               cmds.addCommand(
                  'entry %d push label-stack %s tunnel-destination %s' %
                  ( idx, labelStackString, ng.destinationIp( idx ).stringValue ) )
            else:
               cmds.addCommand( 'entry %s tunnel-destination %s ' %
                                ( idx, ng.destinationIp( idx ).stringValue ) )

@CliSave.saver( 'L3::Config', 'l3/config' )
def saveServiceRoutingConfig( entity, root, sysdbRoot, options ):
   # Show this command always, even if it is the default
   # Down the lane one day multi-agent will become the default
   # instead of ribd. When that happens the default will be masked
   # again.
   cmd = 'service routing protocols model %s' % entity.configuredProtocolAgentModel
   root[ 'Ira.serviceRouting' ].addCommand( cmd )

class RouterKernelCliSaveMode( RouterKernelMode, CliSave.Mode ):
   def __init__( self, vrfName ):
      RouterKernelMode.__init__( self, vrfName )
      CliSave.Mode.__init__( self, vrfName )

CliSave.GlobalConfigMode.addChildMode( RouterKernelCliSaveMode,
                                       after=[ IntfConfigMode ] )

RouterKernelCliSaveMode.addCommandSequence( 'Router.kernel' )

CliSave.GlobalConfigMode.addCommandSequence( 'Tunnel.nexthopgroup.input.cli',
                                             after=[ 'Ira.routing' ] )

@CliSave.saver( 'Tunnel::NexthopGroup::ConfigInput',
                'tunnel/nexthopgroup/input/cli' )
def saveNexthopGroupTunnelConfig( entity, root, sysdbRoot, options ):
   baseCmd = 'ip tunnel'
   for name in entity.entry:
      cmd = []
      nexthopGroupTunnelConfigEntry = entity.entry[ name ]
      cmd.append( nexthopGroupTunnelConfigEntry.tep.stringValue )
      cmd.append( 'nexthop-group' )
      cmd.append( nexthopGroupTunnelConfigEntry.nhgName )
      if nexthopGroupTunnelConfigEntry.igpPref or \
         nexthopGroupTunnelConfigEntry.igpMetric:
         cmd.append( 'igp-cost' )
         if nexthopGroupTunnelConfigEntry.igpPref:
            cmd.append( 'preference %d' % ( nexthopGroupTunnelConfigEntry.igpPref ) )
         if nexthopGroupTunnelConfigEntry.igpMetric:
            cmd.append( 'metric %d' % ( nexthopGroupTunnelConfigEntry.igpMetric ) )
      root[ 'Tunnel.nexthopgroup.input.cli' ].addCommand(
            baseCmd + ' ' + ' '.join( cmd ) )
