# Copyright (c) 2015 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import CliSave, Tac, Tracing
from CliMode.Ipsec import IkePolicyMode, IpsecProfileMode
from CliMode.Ipsec import IpSecurityMode, SecurityAssociationMode
from CliMode.Ipsec import SecurityAssociationVxlanMode
from CliMode.Ipsec import KeyControllerMode
from IntfCliSave import IntfConfigMode
import Toggles.IpsecToggleLib

traceHandle = Tracing.Handle( 'IpsecCli' )
t0 = traceHandle.trace0

IpsecAuthTypeEnum = Tac.Type( "Ipsec::Ike::IpsecAuthType" )
IpsecEspAlgorithmEnum = Tac.Type( "Ipsec::IpsecEspAlgorithm" )
IpsecDhGroupEnum = Tac.Type( "Ipsec::IpsecDhGroup" )
IpsecEncapProtocolEnum = Tac.Type( "Ipsec::Ike::IpsecEncapProtocol" )
IpsecSaType = Tac.Type( "Ipsec::IpsecSaType" )

encapProtocolMap = {
   IpsecEncapProtocolEnum.udp : "udp",
   }

dhGroupMap = { 
         IpsecDhGroupEnum.modp768 : '1',
         IpsecDhGroupEnum.modp1024 : '2',
         IpsecDhGroupEnum.modp1536 : '5',
         IpsecDhGroupEnum.modp2048 : '14',
         IpsecDhGroupEnum.modp3072 : '15',
         IpsecDhGroupEnum.modp4096 : '16',
         IpsecDhGroupEnum.modp6144 : '17',
         IpsecDhGroupEnum.ecp384 : '20',
         IpsecDhGroupEnum.ecp521 : '21',
         IpsecDhGroupEnum.modp2048s256 : '24'
         }

ipsecEncryptionMap = { IpsecEspAlgorithmEnum.des : '3des',
      IpsecEspAlgorithmEnum.aes128 : 'aes128',
      IpsecEspAlgorithmEnum.aes256 : 'aes256',
      IpsecEspAlgorithmEnum.aes128gcm64 : 'aes128gcm64',
      IpsecEspAlgorithmEnum.aes128gcm128 : 'aes128gcm128',
      IpsecEspAlgorithmEnum.aes256gcm128 : 'aes256gcm128',
      IpsecEspAlgorithmEnum.nullesp : 'null', }

ipsecAuthTypeMap = { IpsecAuthTypeEnum.pre_share : 'pre-share',
                   IpsecAuthTypeEnum.rsa_sig : 'rsa-sig', }

class IpsecMode ( IpSecurityMode, CliSave.Mode ):
   def __init__( self, param ):
      IpSecurityMode.__init__( self )
      CliSave.Mode.__init__( self, 1 )

class IkePolicyConfigMode ( IkePolicyMode, CliSave.Mode ):
   def __init__( self, param ):
      IkePolicyMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

class IpsecSaMode ( SecurityAssociationMode, CliSave.Mode ):
   def __init__( self, param ):
      SecurityAssociationMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

class IpsecProfileConfigMode ( IpsecProfileMode, CliSave.Mode ):
   def __init__( self, param ):
      IpsecProfileMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

class IpsecKeyControllerConfigMode ( KeyControllerMode, CliSave.Mode ):
   def __init__( self, param ):
      KeyControllerMode.__init__( self )
      CliSave.Mode.__init__( self, 1 )

class IpsecSaVxlanMode ( SecurityAssociationVxlanMode, CliSave.Mode ):
   def __init__( self, param ):
      SecurityAssociationVxlanMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )


#Save the IKE config 
def saveIpsecIsakmpConfig( mode, cmds, ipsecConfig, policy, 
                           root, sysdbRoot, options ):
   saveAll = options.saveAll
   ikeParams = policy.ikeParams
   if ikeParams.integrity != ikeParams.integrityDefault or saveAll:
      integrity = str( ikeParams.integrity )
      cmds.addCommand( 'integrity %s' % integrity )
   if ikeParams.auth != ikeParams.authDefault or saveAll:
      cmds.addCommand( 'authentication %s' % ipsecAuthTypeMap[ikeParams.auth] )
   if ikeParams.ikeLifetime != ikeParams.ikeLifetimeDefault or saveAll:
      cmds.addCommand( 'ike-lifetime %s' % ikeParams.ikeLifetime )
   if ikeParams.encryption != ikeParams.encryptionDefault or saveAll:
      encryption = ipsecEncryptionMap[ikeParams.encryption.encryption]
      cmds.addCommand( 'encryption %s' % encryption )
   if ikeParams.dhGroup != ikeParams.dhGroupDefault or saveAll:
      cmds.addCommand( 'dh-group %s' % dhGroupMap[ikeParams.dhGroup] )
   if ikeParams.version != ikeParams.versionDefault or saveAll:
      cmds.addCommand( 'version %s' % ikeParams.version )
   if ikeParams.localId != ikeParams.idDefault or saveAll:
      cmds.addCommand( 'local-id %s' % ikeParams.localId.stringValue )
   if ikeParams.remoteId != ikeParams.idDefault or saveAll:
      cmds.addCommand( 'remote-id %s' % ikeParams.remoteId.stringValue )

#Save Security Associations
def saveIpsecSaConfig( mode, cmds, ipsecConfig, sa, root, sysdbRoot, options ):

   saveAll = options.saveAll
   saParams = sa.saParams
   if saParams.espAes != saParams.espAesDefault or saveAll:
      cmds.addCommand( 'esp encryption %s' % ipsecEncryptionMap[saParams.espAes] )
   if saParams.espSha != saParams.espShaDefault or saveAll:
      espSha = str( saParams.espSha )
      if espSha == 'nullhash':
         espSha = 'null'
      cmds.addCommand( 'esp integrity %s' % espSha )
   if saParams.saLifetime != saParams.saLifetimeDefault or saveAll:
      cmds.addCommand( 'sa lifetime %d hours' % saParams.saLifetime )
   if saParams.packetLimit != saParams.packetLimitDefault or saveAll:
      cmds.addCommand( 'sa lifetime %d thousand-packets' %
                       long( saParams.packetLimit / 1000 ) )
   if saParams.byteLimit != saParams.byteLimitDefault or saveAll:
      if saParams.byteLimitUnit == 'MegaBytes':
         cmds.addCommand( 'sa lifetime %d megabytes' % 
                          long( saParams.byteLimit / ( 1024 ** 2 ) ) )
      elif saParams.byteLimitUnit == 'GigaBytes':
         cmds.addCommand( 'sa lifetime %d gigabytes' % 
                          long( saParams.byteLimit / ( 1024 ** 3 ) ) )
   if saParams.pfsGroup != IpsecDhGroupEnum.IpsecDhGroupNone:
      cmds.addCommand( 'pfs dh-group %s' %  dhGroupMap[saParams.pfsGroup] )
   if saParams.replayWindowSize:
      if saParams.replayWindowSize != saParams.replayWindowSizeDefault:
         cmds.addCommand( 'anti-replay window %s' %  saParams.replayWindowSize )
      elif saveAll:
         cmds.addCommand( 'anti-replay detection' )
   else:
      cmds.addCommand( 'no anti-replay detection' )

#Save Ipsec profile
def saveIpsecProfile( mode, cmds, ipsecConfig, profile, root,
                      sysdbRoot, options ):
   saveAll = options.saveAll
   command = 'ike-policy '
   flag = False
   for policy in profile.ikePolicy:
      flag = True 
      command = command + policy + ' '
   if flag:
      cmds.addCommand( command )

   flag = False
   command = 'sa-policy '
   for ts in profile.securityAssoc:
      flag = True
      command = command + ts + ' '
   if flag:
      cmds.addCommand( command )

   profileParams = profile.profileParams
   if profileParams.connectionType != profileParams.connectionTypeDefault:
      cmds.addCommand( 'connection %s' % profileParams.connectionType ) 
   elif saveAll:
      cmds.addCommand( 'connection %s' % profileParams.connectionTypeDefault ) 

   ikeSharedKey = profileParams.ikeSharedKey.sharedKey
   obscuredSharedKey = profileParams.ikeSharedKey.obscuredSharedKey
   if ikeSharedKey != '' :
      if obscuredSharedKey != '':
         cmds.addCommand( "shared-key 7 %s" % 
            CliSave.sanitizedOutput( options, obscuredSharedKey ) )
      else:
         cmds.addCommand( "shared-key %s" % ikeSharedKey )
   elif saveAll:
      cmds.addCommand( 'no shared-key' )

   if profileParams.dpd.action != 'none' :
      dpd = profileParams.dpd
      cmds.addCommand( "dpd %s %s %s" % ( dpd.interval, dpd.timeout, dpd.action ) ) 
   elif saveAll:
      cmds.addCommand( 'no dpd' )

   if profileParams.flowParallelization.encapProtocol != \
      IpsecEncapProtocolEnum.no_encap:
      cmds.addCommand( 'flow parallelization encapsulation %s' % \
            encapProtocolMap[profileParams.flowParallelization.encapProtocol] )
   elif saveAll:
      cmds.addCommand( 'no flow parallelization encapsulation' )

   if profileParams.mode != profileParams.modeDefault:
      cmds.addCommand( 'mode %s' % profileParams.mode )
   elif saveAll:
      cmds.addCommand( 'mode %s' % profileParams.modeDefault )

#Save Ipsec key controller
def saveIpsecKeyController( mode, cmds, ipsecConfig, root,
                            sysdbRoot, options ):
   saveAll = options.saveAll

   keyControllerCfg = ipsecConfig.keyController

   if keyControllerCfg.profileName != '':
      cmds.addCommand( 'profile %s' % keyControllerCfg.profileName ) 
   elif saveAll:
      cmds.addCommand( 'no profile' )

   if keyControllerCfg.dhGroup != keyControllerCfg.dhGroupDefault \
         or saveAll:
      cmds.addCommand( 'dh-group %s' % dhGroupMap[ keyControllerCfg.dhGroup ] )

   if keyControllerCfg.dhLifetime != keyControllerCfg.dhLifetimeDefault \
         or saveAll:
      cmds.addCommand( 'lifetime %s hours' % keyControllerCfg.dhLifetime ) 

#Save Security Associations for Ipsec Vxlan
def saveIpsecVxlanSaConfig( mode, cmds, ipsecConfig, sa, root, sysdbRoot, options ):
   saveAll = options.saveAll
   saParams = sa.saParams
   if saParams.vxlanEncap:
      cmds.addCommand( 'encapsulation vxlan security' )
   elif saveAll:
      cmds.addCommand( 'no encapsulation vxlan security' )
   if saParams.unprotectedTraffic:
      cmds.addCommand( 'traffic unprotected allow' )
   elif saveAll:
      cmds.addCommand( 'no traffic unprotected allow' )
   if saParams.saLifetime != saParams.saLifetimeDefault or saveAll:
      cmds.addCommand( 'sa lifetime %d hours' % saParams.saLifetime )

CliSave.GlobalConfigMode.addChildMode( IpsecMode, 
                                       before=[ IntfConfigMode ] )
IpsecMode.addCommandSequence('ip.security')

#Adds the Ike mode as a child mode  of the global config mode.
IpsecMode.addChildMode( IkePolicyConfigMode, 
                        before=[ IpsecProfileConfigMode ] )
IkePolicyConfigMode.addCommandSequence( 'Ipsec.IkePolicy.config' )

@CliSave.saver( 'Ipsec::Ike::Config', 'ipsec/ike/config' )
def saveIsakmpConfig( entity, root, sysdbRoot, options ):

   for policyName in entity.ikePolicy:
      policy = entity.ikePolicy.get( policyName )
      parent = root[ IpsecMode ].getOrCreateModeInstance( 1 )
      parentMode = parent[ IkePolicyConfigMode ].\
                        getOrCreateModeInstance( policyName )

      cmds = parentMode[ 'Ipsec.IkePolicy.config' ]
      saveIpsecIsakmpConfig( parentMode, cmds, entity, policy, 
                             root, sysdbRoot, options )

IpsecMode.addChildMode( IpsecSaMode, 
                        before=[ IpsecProfileConfigMode ] )
IpsecSaMode.addCommandSequence( 'Ipsec.Sa.config' )

@CliSave.saver( 'Ipsec::Ike::Config', 'ipsec/ike/config' )
def saveSaConfig( entity, root, sysdbRoot, options ):
   for name in entity.securityAssoc:
      sa = entity.securityAssoc.get( name )
      if sa.saParams.saType != IpsecSaType.IpsecKeyMgmtSa:
         continue
      parent = root[ IpsecMode ].getOrCreateModeInstance( 1 )
      parentMode = parent[ IpsecSaMode ].getOrCreateModeInstance( name )

      cmds = parentMode[ 'Ipsec.Sa.config' ]
      saveIpsecSaConfig( parentMode, cmds, entity, sa, 
                         root, sysdbRoot, options )

#Adds the Ipsec Profile mode as a child mode of the global config mode.
IpsecMode.addChildMode( IpsecProfileConfigMode, 
                        after = [ IkePolicyConfigMode ],
                        before = [ IpsecKeyControllerConfigMode ] )
IpsecProfileConfigMode.addCommandSequence( 'Ipsec.Profile.config' )

@CliSave.saver( 'Ipsec::Ike::Config', 'ipsec/ike/config' )
def saveIpsecProfileConfig( entity, root, sysdbRoot, options ):

   for name in entity.ipsecProfile:
      profile = entity.ipsecProfile.get( name )
      parent = root[ IpsecMode ].getOrCreateModeInstance( 1 )
      parentMode = parent[ IpsecProfileConfigMode ].\
                     getOrCreateModeInstance( name )

      cmds = parentMode[ 'Ipsec.Profile.config' ]
      saveIpsecProfile( parentMode, cmds, entity, profile, 
                        root, sysdbRoot, options )

#Adds the Ipsec Key Controller mode as a child mode of the global config mode.
IpsecMode.addChildMode( IpsecKeyControllerConfigMode, 
                        after = [ IpsecProfileConfigMode ] )
IpsecKeyControllerConfigMode.addCommandSequence( 'Ipsec.KeyController.config' )


@CliSave.saver( 'Ipsec::Ike::Config', 'ipsec/ike/config' )
def saveIpsecKeyControllerConfig( entity, root, sysdbRoot, options ):

   if not entity.keyController.configured:
      return
   parent = root[ IpsecMode ].getOrCreateModeInstance( 1 )
   parentMode = parent[ IpsecKeyControllerConfigMode ].\
                getOrCreateModeInstance( 1 )

   cmds = parentMode[ 'Ipsec.KeyController.config' ]
   saveIpsecKeyController( parentMode, cmds, entity,
                           root, sysdbRoot, options )

#Adds the Ipsec Vxlan Sa mode as a child mode of the global config mode.
IpsecMode.addChildMode( IpsecSaVxlanMode, 
                        before=[ IpsecProfileConfigMode ] )
IpsecSaVxlanMode.addCommandSequence( 'Ipsec.Sa.config' )

@CliSave.saver( 'Ipsec::Ike::Config', 'ipsec/ike/config' )
def saveVxlanSaConfig( entity, root, sysdbRoot, options ):
   if not Toggles.IpsecToggleLib.toggleTunnelSecAssocEnabled():
      return
   for name in entity.securityAssoc:
      sa = entity.securityAssoc.get(name)
      if sa.saParams.saType != IpsecSaType.IpsecVxlanSa:
         continue
      parent = root[ IpsecMode ].getOrCreateModeInstance( 1 )
      parentMode = parent[ IpsecSaVxlanMode ].getOrCreateModeInstance( name )
      cmds = parentMode[ 'Ipsec.Sa.config' ]
      saveIpsecVxlanSaConfig( parentMode, cmds, entity, sa, 
                         root, sysdbRoot, options )


