# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

import Toggles.IpLockingToggleLib as ILTL

import CliSave
import EthIntfUtil
import IpUtils
import Tracing
from CliMode.IpLocking import AddressLockingBaseMode
from CliSavePlugin.IntfCliSave import IntfConfigMode
from IpLockingLib import intfSupportsIpLocking

t0 = Tracing.trace0

class AddressLockingConfigMode( AddressLockingBaseMode, CliSave.Mode ):
   def __init__( self, param ):
      AddressLockingBaseMode.__init__( self )
      CliSave.Mode.__init__( self, param )

   def skipIfEmpty( self ):
      return True

CliSave.GlobalConfigMode.addChildMode( AddressLockingConfigMode,
                                       before=[ IntfConfigMode ] )
IntfConfigMode.addCommandSequence( 'IpLocking.switchport',
                                   after=[ 'Ebra.switchport' ] )
AddressLockingConfigMode.addCommandSequence( 'IpLocking.config' )

def saveIpLockingIntfConfig( ipLockingConfig, root, sysdbRoot, options,
                             requireMounts ):
   saveAll = options.saveAll
   saveAllDetail = options.saveAllDetail

   # Get the list of interfaces on which config needs to be displayed.
   bridgingConfig = requireMounts[ 'bridging/config' ]
   if saveAllDetail:
      cfgIntfNames = EthIntfUtil.allSwitchportNames( bridgingConfig,
                                                     includeEligible=True )
   elif saveAll:
      switchportNames = EthIntfUtil.allSwitchportNames( bridgingConfig )
      cfgIntfNames = set( switchportNames + ipLockingConfig.interface.keys() )
   else:
      cfgIntfNames = ipLockingConfig.interface
   for intf in cfgIntfNames:
      if intfSupportsIpLocking( intf ):
         intfMode = root[ IntfConfigMode ].getOrCreateModeInstance( intf )
         cmds = intfMode[ 'IpLocking.switchport' ]
         intfConfig = ipLockingConfig.interface.get( intf, None )
         if intfConfig:
            af = ''
            af += 'ipv4' if intfConfig.ipv4 else ''
            af += ' ipv6' if intfConfig.ipv6 else ''
            cmds.addCommand( 'address locking {}'.format( af ) )
         else:
            # TODO Update to 'no address locking' when IPv6 is supported.
            cmds.addCommand( 'no address locking ipv4' )

@CliSave.saver( 'IpLocking::CliConfig', 'iplocking/cliConfig',
                requireMounts=( 'bridging/config', ) )
def saveAddressLockingConfig( ipLockingConfig, root, sysdbRoot, options,
                              requireMounts ):
   saveIpLockingIntfConfig( ipLockingConfig, root, sysdbRoot, options,
                            requireMounts )
   saveAll = options.saveAll
   saveAllDetail = options.saveAllDetail

   if ipLockingConfig.configured == ipLockingConfig.configuredDefault \
         and not saveAllDetail:
      return

   mode = root[ AddressLockingConfigMode ].getSingletonInstance()
   cmds = mode[ 'IpLocking.config' ]

   if ipLockingConfig.disabled != ipLockingConfig.disabledDefault:
      cmds.addCommand( 'disabled' )
   elif saveAll:
      cmds.addCommand( 'no disabled' )

   if ipLockingConfig.localInterface != ipLockingConfig.localInterfaceDefault:
      cmds.addCommand( 'local-interface {}'.format(
                        ipLockingConfig.localInterface ) )
   elif saveAll:
      cmds.addCommand( 'no local-interface' )

   if ipLockingConfig.dhcpV4Server:
      leaseQueryServers = [ x for x, y in ipLockingConfig.dhcpV4Server.iteritems()
                               if y == 'leaseQuery' ]
      leaseQueryServers.sort( IpUtils.compareIpAddress )
      for ds in leaseQueryServers:
         cmds.addCommand( 'dhcp server ipv4 {}'.format( ds ) )
      arIpQueryServers = [ x for x, y in ipLockingConfig.dhcpV4Server.iteritems()
                              if y == 'arIpQuery' ]
      arIpQueryServers.sort( IpUtils.compareIpAddress )
      for ds in arIpQueryServers:
         cmds.addCommand( 'dhcp server ipv4 {} protocol arista-query'.format( ds ) )
   elif saveAll:
      cmds.addCommand( 'no dhcp server ipv4' )

   for staticLease in ipLockingConfig.staticLease.values():
      cmds.addCommand( 'lease {} mac {}'.format(
         staticLease.ip, staticLease.mac ) )

   if ILTL.toggleIpLockingMacAgingCleanUpEnabled():
      if ( ipLockingConfig.macAgingCleanUp !=
           ipLockingConfig.macAgingCleanUpDefault ):
         cmds.addCommand( 'locked-address expiration mac disabled' )
      elif saveAll:
         cmds.addCommand( 'no locked-address expiration mac disabled' )

   if ILTL.toggleIpLockingEnforcementEnabled():
      if ( ipLockingConfig.ipv6Enforced !=
           ipLockingConfig.ipv6EnforcedDefault ):
         cmds.addCommand( 'locked-address ipv6 enforcement disabled' )
      elif saveAll:
         cmds.addCommand( 'no locked-address ipv6 enforcement disabled' )

   if ILTL.toggleIpLockingSyncEnabled():
      if ipLockingConfig.reflector:
         ipv4Reflectors = [ r for r in ipLockingConfig.reflector.keys()
                            if r.af == 'ipv4' ]
         ipv6Reflectors = [ r for r in ipLockingConfig.reflector.keys()
                            if r.af == 'ipv6' ]
         ipv4Reflectors.sort( IpUtils.compareIpAddress )
         ipv6Reflectors.sort( IpUtils.compareIp6Address )
         for rs in ipv4Reflectors + ipv6Reflectors:
            cmds.addCommand( 'reflector address {}'.format( rs ) )
      elif saveAll:
         cmds.addCommand( 'no reflector address' )

      if ipLockingConfig.reflectorRole != ipLockingConfig.reflectorRoleDefault:
         cmds.addCommand( 'reflector' )
      elif saveAll:
         cmds.addCommand( 'no reflector' )
