# Copyright (c) 2006-2010, 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

#-------------------------------------------------------------------------------
# This module implements the framework for saving interface configuration.
#-------------------------------------------------------------------------------
import Tac
import CliSave, Arnet
from CliMode.Intf import IntfMode
from CliMode.InterfaceDefaults import InterfaceDefaultsMode

#-------------------------------------------------------------------------------
# Object used for saving commands in "config-if" mode.
#-------------------------------------------------------------------------------
class IntfConfigMode( IntfMode, CliSave.Mode ):
   def __cmp__( self, other ):
      return ( cmp( self.saveOrder_, other.saveOrder_ ) or
               Arnet.compareIntfName( self.param_, other.param_ ) )

   orderMap_ = {
      'Port-Channel': 20,
      'Recirc-Channel' : 20,
   }

   def __init__( self, param ):
      """The interface name is looked up in orderMap; if the key
      matches a prefix of the interface name, the given save order
      is used (lower is earlier).  If there is no match, the default
      order value of 50 is used."""
      self.saveOrder_ = 50
      for k, v in self.orderMap_.iteritems():
         if param.startswith( k ):
            self.saveOrder_ = v
            break
      # This is just like the short name in the CliPlugin
      shortName = IntfMode.getShortname( param )
      # Whether this interface is an inactive interface lane.
      self.inactive_ = False
      self.unconnected_ = False
      # this speeds up things a bit
      self.profileSupported_ = False
      IntfMode.__init__( self, param, shortName )
      CliSave.Mode.__init__( self, param )

   def writeMode( self, param, prefix, cliModel=None ):
      profileCmds = self.getIntfProfileCmds( param )
      if profileCmds and not param.showProfileExpanded:
         self.processProfileCmds( profileCmds )
      return CliSave.Mode.writeMode( self, param, prefix, cliModel=cliModel )

   def getIntfProfileCmds( self, param ):
      if self.profileSupported_:
         profileConfig = CliSave.getSessionEntity( 'interface/profile/config',
                                                   param.sessionRoot,
                                                   param.sysdbRoot,
                                                   param.sStatus,
                                                   param.cleanConfig
         )
         profileAppCfg = profileConfig.intfToAppliedProfile.get( self.param_ )
         if profileAppCfg:
            return profileAppCfg.profileCmdsApplied.splitlines()
      return []

   def hideInactive( self, param ):
      if not self.inactive_:
         return False
      globalInactiveExpose = CliSave.getSessionEntity(
         'interface/config/global',
         param.sessionRoot,
         param.sysdbRoot,
         param.sStatus,
         param.cleanConfig ).exposeInactiveLanes
      return ( not globalInactiveExpose  and self.emptyCmds( param ) )

   def hideUnconnected( self, param ):
      if not self.unconnected_:
         return False
      globalUnconnectedExpose = CliSave.getSessionEntity(
         'interface/config/global',
         param.sessionRoot,
         param.sysdbRoot,
         param.sStatus,
         param.cleanConfig ).exposeUnconnectedLanes
      return ( not globalUnconnectedExpose
               and self.emptyCmds( param ) )

CliSave.GlobalConfigMode.addChildMode( IntfConfigMode )
IntfConfigMode.addCommandSequence( 'Arnet.intf' )

# 'Arnet.l2l3barrier' makes sure that l3 things comes after l2.
IntfConfigMode.addCommandSequence( 'Arnet.l2l3barrier' )

#------------------------------------------------------------------------------
# Helper methods
#-----------------------------------------------------------------------------

# Returns True if the interface supports routing.
# if includeEligible is set then this method returns True even for interfaces which
# are 'capable' of routing, irrespective of their configured state, e.g. For ethernet
# interface configured as switchport, this method returns True, if includeEligible
# is set.
def _isRoutingInterface( intfName, sysdbRoot, includeEligible=False,
                         requireMounts=None ):
   if requireMounts:
      intfStatusDir = requireMounts[ 'interface/status/all' ]
   else:
      intfStatusDir = sysdbRoot[ 'interface' ][ 'status' ][ 'all' ]
   intfStatus = intfStatusDir.intfStatus.get( intfName )
   if not intfStatus:
      return False

   if includeEligible:
      # XXX. This is ugly. We should check some attribute in sysdb to figure out if
      # the interface is capable of supporting routing or not. But nothing is
      # available right now. All interfaces are capable of supporting routing. So
      # returning True as default.
      return True
   else:
      return intfStatus.forwardingModel == 'intfForwardingModelRouted' 

# Returns the string for the load-interval command
def getLoadIntervalCommandStr( saveAll, li, extra='' ):
   if saveAll:
      if li.useDefault:
         return 'default load-interval' + extra
      else:
         return 'load-interval' + extra + ( ' %d' % li.val )
   elif not li.useDefault:
      return 'load-interval' + extra + ( ' %d' % li.val )
   return None

IntfEnabledState = Tac.Type( 'Interface::IntfEnabledState' )

#-------------------------------------------------------------------------------
# Saves the state of the baseclass attributes of an object derived from
# Intf::IntfConfig.  This function should be called from any other CliSave
# function that saves that state of a subclass of Intf::IntfConfig.
#-------------------------------------------------------------------------------
def saveIntfConfig( entity, root, sysdbRoot, options,
                    requireMounts=None ):
   mode = root[ IntfConfigMode ].getOrCreateModeInstance( entity.intfId )
   cmds = mode[ 'Arnet.intf' ]
   saveAll = options.saveAll
   saveAllDetail = options.saveAllDetail
   unconditionalShowAdminStatus = False

   if requireMounts:
      intfStatusDir = requireMounts[ 'interface/status/all' ]
      try:
         intfGlobalConfig = requireMounts[ 'interface/config/global' ]
         unconditionalShowAdminStatus = \
            intfGlobalConfig.defaultEthernetShutdown == IntfEnabledState.shutdown
      except Exception: # pylint: disable-msg=W0703
         # not all interface types need/set this mount (only 'ethernet' needs it)
         pass
   else:
      intfStatusDir = sysdbRoot[ 'interface' ][ 'status' ][ 'all' ]
   
   intfStatus = intfStatusDir.intfStatus.get( entity.intfId )
   if intfStatus:
      mode.inactive_ = not intfStatus.active

   mode.unconnected_ = entity.prettyName.startswith( ( 'Un', 'Ue' ) )
   mode.profileSupported_ = entity.prettyName.startswith( 'Et' )

   if entity.description != '':
      cmds.addCommand( 'description %s' % entity.description )
   elif saveAll:
      cmds.addCommand( 'no description' )

   # Deciding whether to save the enabled state is a bit tricky, because
   # the default can vary across different interface types.  If there is
   # no defaultConfig attached to the IntfConfig or if the defaultConfig
   # doesn't initialize its enabledStateLocal atribute, then we fall all
   # the way back to the IntfConfig::enabledDefault constAttr.  Thankfully,
   # the IntfConfig.enabled attribute encapsulates most of this.  We just
   # have to figure out which default value to compare against, as this is
   # not as well encapsulated.  To really get this, make sure that you look
   # through the innards of IntfConfig::enabled, IntfConfig::enabledState,
   # and the IntfEnabledState enum (where the default value is
   # 'unknownEnabledState'.
   enabled = entity.adminEnabled
   if entity.defaultConfig:
      enabledDefault = entity.defaultConfig.adminEnabled
   else:
      enabledDefault = entity.enabledDefault

   if enabled != enabledDefault or saveAll or unconditionalShowAdminStatus:
      if enabled:
         cmds.addCommand( 'no shutdown' )
      else:
         cmds.addCommand( 'shutdown' )

   loadIntervalCmdStr = getLoadIntervalCommandStr( saveAll, entity.loadInterval )
   if loadIntervalCmdStr is not None:
      cmds.addCommand( loadIntervalCmdStr )
   
   # If saveAll, display the default only on current L3 interfaces.
   # If saveAllDetail, display the default on all L3 capable interfaces.
   globalIntfConfig = Tac.root.newEntity( 'Interface::GlobalIntfConfig',
                                          'globalIntfConfig' )
   if ( entity.mtu and ( ( entity.mtu != globalIntfConfig.l3MtuDefault ) or \
        ( saveAll and _isRoutingInterface( entity.intfId, sysdbRoot,
                                           includeEligible=saveAllDetail,
                                           requireMounts=requireMounts ) ) ) ):
      cmds.addCommand( 'mtu %d' % entity.mtu )

   if entity.linkStatusLogging == 'on':
      cmds.addCommand( 'logging event link-status' )
   elif entity.linkStatusLogging == 'off':
      cmds.addCommand( 'no logging event link-status' )
   elif entity.linkStatusLogging == 'useGlobal' and saveAll:
      cmds.addCommand( 'logging event link-status use-global' )

#-------------------------------------------------------------------------------
# Save the state of the GlobalIntfConfig
#-------------------------------------------------------------------------------

CliSave.GlobalConfigMode.addCommandSequence( 'Log.linkStatus' )

def saveLogFacilityConfig( entity, root, sysdbRoot, options ):
   cmds = root[ 'Log.linkStatus' ]
   if entity.linkStatusLogging == 'off':
      cmds.addCommand( 'no logging event link-status global' )
   elif options.saveAll:
      cmds.addCommand( 'logging event link-status global' )

CliSave.GlobalConfigMode.addCommandSequence( 'Interface.loadInterval' )
  
def saveGlobalLoadInterval( entity, root, sysdbRoot, options ):
   cmds = root[ 'Interface.loadInterval' ]
   loadIntervalCmdStr = getLoadIntervalCommandStr( options.saveAll, 
                                                   entity.loadInterval,
                                                   ' default' )
   if ( loadIntervalCmdStr is not None ):
      cmds.addCommand( loadIntervalCmdStr )

CliSave.GlobalConfigMode.addCommandSequence( 'Interface.exposeInactiveLanes' )
  
def saveExposeInactiveLanes( entity, root, sysdbRoot, options ):
   cmds = root[ 'Interface.exposeInactiveLanes' ]
   if entity.exposeInactiveLanes:
      cmds.addCommand( 'service interface inactive expose' )
   elif options.saveAll:
      cmds.addCommand( 'no service interface inactive expose' )

CliSave.GlobalConfigMode.addCommandSequence( 'Interface.exposeUnconnectedLanes' )
  
def saveExposeUnconnectedLanes( entity, root, sysdbRoot, options ):
   cmds = root[ 'Interface.exposeUnconnectedLanes' ]
   if entity.exposeUnconnectedLanes:
      cmds.addCommand( 'service interface unconnected expose' )
   elif options.saveAll:
      cmds.addCommand( 'no service interface unconnected expose' )

# pylint: disable-msg=C0322
# XXX Workaround for a pylint bug with decorators.
@CliSave.saver( 'Interface::GlobalIntfConfig', 'interface/config/global' )
def saveGlobalIntfConfig( entity, root, sysdbRoot, options ):
   saveLogFacilityConfig( entity, root, sysdbRoot, options )
   saveGlobalLoadInterval( entity, root, sysdbRoot, options )
   saveExposeInactiveLanes( entity, root, sysdbRoot, options )
   saveExposeUnconnectedLanes( entity, root, sysdbRoot, options )

#-------------------------------------------------------------------------------
# Object used for saving commands in "interface-defaults" mode.
#-------------------------------------------------------------------------------
class InterfaceDefaultsConfigMode( InterfaceDefaultsMode, CliSave.Mode ):

   def __init__( self, param ):
      InterfaceDefaultsMode.__init__( self )
      CliSave.Mode.__init__( self, param )

   def skipIfEmpty( self ):
      return True

CliSave.GlobalConfigMode.addChildMode( InterfaceDefaultsConfigMode,
                                       before=[ IntfConfigMode ] )
InterfaceDefaultsConfigMode.addCommandSequence( 'Interface.defaults' )

@CliSave.saver( 'Interface::GlobalIntfConfig', 'interface/config/global' )
def saveGlobalMtu( entity, root, sysdbRoot, options ):
   mode = root[ InterfaceDefaultsConfigMode ].getOrCreateModeInstance( '' )
   cmd = mode[ 'Interface.defaults' ]
   if entity.l3Mtu and entity.l3Mtu != entity.l3MtuDefault:
      cmd.addCommand( 'mtu %d' % entity.l3Mtu )

# This function is registered with the CliSave module.
# It provides an alternative to 'interface/config/all'
# For CLI save plug-ins while in a configuration session.
# This is necessary since the reactors that maintain
# that collection will not react until the session is
# committed.

def syntheticRequireMountGenerator( sysdbRoot, sessionRoot, pathHasSessionPrefix ):
   synthIntfDict = {}
   walk = Tac.newInstance( "Interface::CliSaveIntfConfigWalk" )
   intfConfigPath = 'interface/config'
   walk.root = sysdbRoot[ intfConfigPath ]
   rootName = sysdbRoot.fullName( intfConfigPath )
   for entity in walk.match.itervalues():
      path = entity.fullName.replace( rootName + "/", "", 1 )
      if ( not pathHasSessionPrefix( path ) ):
         synthIntfDict[ entity.prettyName ] = entity
   walk = Tac.newInstance( "Interface::CliSaveIntfConfigWalk" )
   walk.root = sessionRoot.entity[ "interface/config" ]
   for entity in walk.match.itervalues():
      synthIntfDict[ entity.prettyName ] = entity
   return synthIntfDict

CliSave.registerSyntheticRequireMountGenerator(
   'interface/config/all',
   syntheticRequireMountGenerator )
