#!/usr/bin/env python
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tac
import CliSave, IntfCliSave
from IntfCliSave import IntfConfigMode
from SamplePolicyCliSave import SamplePolicySaveMode, SamplePolicySaver
from TrafficPolicyCliSave import ( MatchRuleSaveMode, ActionsSaveMode )
from CliMode.InbandTelemetry import ( FEATURE, InbandTelemetryModeBase,
                                      ProfileModeBase )
from InbandTelemetryTypes import tacCollectorType, tacPortProfileType
import Toggles.InbandTelemetryCommonToggleLib

class InbandTelemetryConfigMode( InbandTelemetryModeBase, CliSave.Mode ):
   def __init__( self, param ):
      InbandTelemetryModeBase.__init__( self )
      CliSave.Mode.__init__( self, self.longModeKey )

CliSave.GlobalConfigMode.addCommandSequence( 'InbandTelemetry.Config' )
IntfCliSave.IntfConfigMode.addCommandSequence( 'InbandTelemetry.Config' )

CliSave.GlobalConfigMode.addChildMode( InbandTelemetryConfigMode,
                                       after=[ IntfConfigMode ] )
InbandTelemetryConfigMode.addCommandSequence( 'InbandTelemetry.Config' )

class ProfileConfigMode( ProfileModeBase, CliSave.Mode ):
   def __init__( self, param ):
      ProfileModeBase.__init__( self, param )
      CliSave.Mode.__init__( self, self.longModeKey )

if Toggles.InbandTelemetryCommonToggleLib.\
   toggleFeatureInbandTelemetrySamplePolicyEnabled():
   InbandTelemetryConfigMode.addChildMode( SamplePolicySaveMode,
                                           after=[ 'InbandTelemetry.Config' ] )
   SamplePolicySaveMode.addCommandSequence( 'InbandTelemetry.Policy' )

   SamplePolicySaveMode.addChildMode( MatchRuleSaveMode )
   MatchRuleSaveMode.addCommandSequence( 'IntSamplePolicy.MatchRule' )

   ActionsSaveMode.addCommandSequence( 'IntSamplePolicy.Actions' )
   InbandTelemetryConfigMode.addChildMode( ProfileConfigMode,
                                           after=[ SamplePolicySaveMode ] )
else:
   InbandTelemetryConfigMode.addChildMode( ProfileConfigMode,
                                           after=[ 'InbandTelemetry.Config' ] )

ProfileConfigMode.addCommandSequence( 'InbandTelemetry.profile' )

class IntSamplePolicySaver( SamplePolicySaver ):
   trafficPoliciesMode = InbandTelemetryConfigMode

   def __init__( self, entity, root, sysdbRoot, options, requireMounts, feature,
                 commentKey ):
      SamplePolicySaver.__init__( self, entity, root, sysdbRoot, options,
                                  requireMounts, feature, commentKey )
      self.policyMapType = 'inband-sample-policy'

   def matchModeCmds( self, matchMode ):
      return matchMode[ 'IntSamplePolicy.MatchRule' ]

   def actionModeCmds( self, actionsMode ):
      return actionsMode[ 'IntSamplePolicy.Actions' ]

def _hasComments( commentKey, cliConfig ):
   return commentKey in cliConfig.comment

@CliSave.saver( 'InbandTelemetry::Config', 'inbandtelemetry/config',
                requireMounts=( 'cli/config', 'inbandtelemetry/hwCapability',
                                'inbandtelemetry/samplePolicies/input/cli',
                              ) )
def saveConfig( entity, root, sysdbRoot, options, requireMounts ):
   hwCapability = requireMounts[ 'inbandtelemetry/hwCapability' ]
   cliConfig = requireMounts[ 'cli/config' ]
   saveAll = options.saveAll

   for intfName in entity.intfToEdgePortProfiles:
      profileName = entity.intfToEdgePortProfiles[ intfName ]
      intfMode = root[ IntfCliSave.IntfConfigMode ].getOrCreateModeInstance(
         intfName )
      intfCmds = intfMode[ 'InbandTelemetry.Config' ]
      intfCmds.addCommand( 'telemetry inband profile edge %s' % profileName )
   for intfName in entity.intfToCorePortProfiles:
      profileName = entity.intfToCorePortProfiles[ intfName ]
      intfMode = root[ IntfCliSave.IntfConfigMode ].getOrCreateModeInstance(
         intfName )
      intfCmds = intfMode[ 'InbandTelemetry.Config' ]
      intfCmds.addCommand( 'telemetry inband profile core %s' % profileName )

   def addCommandIntProfile( profile ):
      profileName = profile.name
      profileType = 'edge' if profile.type == tacPortProfileType.IntEdge else 'core'
      param = ( profileName, profile.type )
      cmdList = []
      isDefaultProfile = profileName == 'default'
      if profile:
         if not saveAll and isDefaultProfile:
            return
         profileMode = intConfigMode[ ProfileConfigMode ].\
                       getOrCreateModeInstance( param )
         if profileType == 'edge':
            if profile.sampleRate != 1048576 or saveAll:
               cmdList.append( 'ingress sample rate %d' % profile.sampleRate )
            if profile.samplePolicy != '':
               cmdList.append( 'ingress sample policy %s'
                               % profile.samplePolicy )
            if not profile.egressDrop:
               cmdList.append( 'egress drop disabled' )
            elif saveAll:
               cmdList.append( 'default egress drop disabled' )
         if profile.collectorName:
            collectionName = profile.collectorName
            collection = entity.collectors[ collectionName ]
            # TODO: when collector log is supported uncomment the below
            # if collection.type == tacCollectorType.CollectorLog:
            #   cmdList.append( 'egress collection log' )
            if collection.type == tacCollectorType.CollectorFlowTracker:
               cmdList.append( 'egress collection flow tracker %s' % collectionName )
         if cmdList:
            cmdSeq = profileMode[ 'InbandTelemetry.profile' ]
            for cmd in cmdList:
               cmdSeq.addCommand( cmd )

   def isDefaultIntParam( param ):
      return bool( param.deviceId == param.defaultDeviceId and param.probeMarker ==
                   param.defaultProbeMarker and
                   param.probeProtocol == param.defaultProbeProtocol )

   def isDefaultConfig( entity ):
      policiesCliConfig = requireMounts[ 'inbandtelemetry/samplePolicies/input/cli' ]
      if len( entity.corePortProfiles.keys() ) == 1 and \
         len( entity.edgePortProfiles.keys() ) == 1 and \
         isDefaultIntParam( entity.intParam ) and \
         not policiesCliConfig.pmapType.pmapInput.keys() and \
         not _hasComments( 'mon-telemetry-inband', cliConfig ) and \
         not entity.enable:
         # TODO: when collector log is supported uncomment the below
         # entity.collectors[ 'DefaultLog' ].maxPktsToBuffer == 64:
         return True
      return False

   if ( Toggles.InbandTelemetryCommonToggleLib.
        toggleFeatureInbandTelemetryEnabled() ) and \
        hwCapability.inbandTelemetrySupported and \
        ( not isDefaultConfig( entity ) or saveAll ):
      intConfigMode = root[ InbandTelemetryConfigMode ].getOrCreateModeInstance(
         None )
      intCmds = intConfigMode[ 'InbandTelemetry.Config' ]
      if entity.enable:
         intCmds.addCommand( 'no disabled' )
      elif saveAll:
         intCmds.addCommand( 'disabled' )
      if entity.intParam.deviceId != entity.intParam.defaultDeviceId or saveAll:
         intCmds.addCommand( 'device-id %s' % entity.intParam.deviceId )

      markerA = entity.intParam.probeMarker >> 32
      markerB = entity.intParam.probeMarker - ( markerA << 32 )
      if hwCapability.defaultIntDetectionMethod == \
            Tac.Type( "InbandTelemetry::IntDetectionMethod" ).ProbeMarkerBased:
         if entity.intParam.probeMarker != entity.intParam.defaultProbeMarker or \
            saveAll:
            hexA = hex( markerA )[ : -1 ] \
                  if 'L' in hex( markerA ) else hex( markerA )
            hexB = hex( markerB )[ : -1 ] \
                  if 'L' in hex( markerB ) else hex( markerB )
            intCmds.addCommand( 'probe marker %s %s' % ( hexA, hexB ) )
      else:
         if entity.intParam.probeProtocol != entity.intParam.defaultProbeProtocol or\
               saveAll:
            intCmds.addCommand( 'probe marker ip protocol %s'
                  % entity.intParam.probeProtocol )
      if Toggles.InbandTelemetryCommonToggleLib.\
         toggleFeatureInbandTelemetrySamplePolicyEnabled():
         cliDumper = IntSamplePolicySaver( requireMounts[
            'inbandtelemetry/samplePolicies/input/cli' ],
                                           root, sysdbRoot, options,
                                           requireMounts, FEATURE,
                                           'mon-telemetry-inband' )
         cliDumper.save()
      # TODO: when collector log is supported uncomment the below
      #logCollection = entity.collectors[ 'DefaultLog' ]
      # if logCollection.maxPktsToBuffer != 64:
      #   intCmds.addCommand( 'egress collection log %d' %
      #                    logCollection.maxPktsToBuffer )
      # elif saveAll:
      #   intCmds.addCommand( 'egress collection log 64' )

      for portProfiles in [ entity.corePortProfiles, entity.edgePortProfiles ]:
         for profile in sorted( portProfiles ):
            addCommandIntProfile( portProfiles[ profile ] )
