# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import cStringIO
import Tac
import Tracing
import CliSave
from CliMode.FlowTracker import TrackerModeBase
from CliMode.FlowExporter import ExporterModeBase
from CliMode.FlowGroups import (
      GroupsModeBase,
      GroupModeBase
)
from FlowTrackerCliUtil import (
      ftrTypeKwStr,
      showFtr,
      showExporter,
      showGroup
)
from FlowTrackingCliSave import FlowTrackingCliSaveMode
import Toggles.InbandTelemetryCommonToggleLib

traceHandle = Tracing.Handle( 'FlowTrackerCliSave' )
t0 = traceHandle.trace0
t1 = traceHandle.trace1
t2 = traceHandle.trace2

#---------------------------------------------------------------------------------
# Cli savers
#---------------------------------------------------------------------------------

class TrackerCliSaveMode( TrackerModeBase, CliSave.Mode ):
   def __init__( self, param ):
      t0( 'TrackerCliSaveMode: ', param )
      TrackerModeBase.__init__( self, param )
      CliSave.Mode.__init__( self, param )

   def skipIfEmpty( self ):
      return False

FlowTrackingCliSaveMode.addChildMode( TrackerCliSaveMode,
                                      before=[ 'FlowTracking.trailer' ] )
TrackerCliSaveMode.addCommandSequence( 'FlowTracker.config' )

class ExporterSaveMode( ExporterModeBase, CliSave.Mode ):
   def __init__( self, param ):
      t0( 'FlowTrackerExporterSaveMode: ', param )
      ExporterModeBase.__init__( self, param )
      CliSave.Mode.__init__( self, param )

   def skipIfEmpty( self ):
      return False

TrackerCliSaveMode.addChildMode( ExporterSaveMode, after=[ 'FlowTracker.config' ] )
ExporterSaveMode.addCommandSequence( 'FlowTracker.exporterConfig' )

class GroupsSaveMode( GroupsModeBase, CliSave.Mode ):
   def __init__( self, param ):
      t0( 'FlowTrackerGroupsSaveMode: ', param )
      GroupsModeBase.__init__( self, param )
      CliSave.Mode.__init__( self, param )

   def skipIfEmpty( self ):
      return False

TrackerCliSaveMode.addChildMode( GroupsSaveMode, after=[ ExporterSaveMode ] )
GroupsSaveMode.addCommandSequence( 'FlowTracker.groupsConfig' )

class GroupSaveMode( GroupModeBase, CliSave.Mode ):
   def __init__( self, param ):
      t0( 'FlowTrackerGroupSaveMode: ', param )
      GroupModeBase.__init__( self, param )
      CliSave.Mode.__init__( self, param )

   def skipIfEmpty( self ):
      return False

GroupsSaveMode.addChildMode( GroupSaveMode, after=[ 'FlowTracker.groupsConfig' ] )
GroupSaveMode.addCommandSequence( 'FlowTracker.groupConfig' )

@CliSave.saver( 'FlowTracking::FlowTrackerConfig', 'flowtracking/config/sampled',
                attrName='flowTrackerConfig' )
@CliSave.saver( 'FlowTracking::FlowTrackerConfig', 'flowtracking/config/hardware',
                attrName='flowTrackerConfig' )
@CliSave.saver( 'FlowTracking::FlowTrackerConfig',
                'flowtracking/config/inbandTelemetry', attrName='flowTrackerConfig' )
@CliSave.saver( 'FlowTracking::FlowTrackerConfig',
                'flowtracking/config/mirrorOnDrop', attrName='flowTrackerConfig' )

def saveFlowTrackerConfig( trConfig, root, sysdbRoot, options ):
   trName = trConfig.name
   if trName == "inbandTelemetry":
      if not Toggles.InbandTelemetryCommonToggleLib. \
            toggleFeatureInbandTelemetryEnabled():
         return
   saveAll = options.saveAll
   saveAllDetail = options.saveAllDetail
   t0( 'saveFlowTrackerConfig', trConfig, trConfig.parent,
       root, sysdbRoot, options, trName )
   ftrType = ftrTypeKwStr[ trConfig.parent.name ]
   ftrMode = root[ FlowTrackingCliSaveMode ].getOrCreateModeInstance( ftrType )
   trMode = ftrMode[ TrackerCliSaveMode ].getOrCreateModeInstance(
                                                      ( ftrType, trName ) )
   t1( 'ftrMode', ftrMode, 'trMode', trMode )
   trCmds = trMode[ 'FlowTracker.config' ]
   trOutput = cStringIO.StringIO()
   showFtr( trOutput, trName, trConfig,
               saveAll=saveAll, saveAllDetail=saveAllDetail )
   for cmd in trOutput.getvalue().splitlines():
      trCmds.addCommand( cmd )

   for expName in sorted( trConfig.expConfig.iterkeys() ):
      expMode = trMode[ ExporterSaveMode ].getOrCreateModeInstance(
                                             ( ftrType, trName, expName ) )
      expCmds = expMode[ 'FlowTracker.exporterConfig' ]
      output = cStringIO.StringIO()
      showExporter( output, expName, trConfig.expConfig[ expName ],
                    saveAll=saveAll, saveAllDetail=saveAllDetail )
      for cmd in output.getvalue().splitlines():
         expCmds.addCommand( cmd )

   if trConfig.fgConfig:
      grpsMode = trMode[ GroupsSaveMode ].getOrCreateModeInstance(
                                                ( ftrType, trName ) )
      seqnoToGroupMap = {}
      for groupName, groupConfig in trConfig.fgConfig.items():
         seqnoToGroupMap[ groupConfig.seqno ] = groupName
      for groupSeqno in sorted( seqnoToGroupMap ):
         groupName = seqnoToGroupMap[ groupSeqno ]
         grpMode = grpsMode[ GroupSaveMode ].getOrCreateModeInstance(
                                                   ( ftrType, trName, groupName ) )
         grpCmds = grpMode[ 'FlowTracker.groupConfig' ]
         output = cStringIO.StringIO()
         showGroup( output, groupName, trConfig.fgConfig[ groupName ],
                    saveAll=saveAll, saveAllDetail=saveAllDetail )
         for cmd in output.getvalue().splitlines():
            grpCmds.addCommand( cmd )
