# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from CliMode.FlexEncap import EncapsulationBaseMode
import CliSave
from IntfCliSave import IntfConfigMode
from Toggles.EbraToggleLib import toggleMplsEvpnVpwsP2Enabled
from TypeFuture import TacLazyType

Dot1qEncapMode = TacLazyType( "Interface::SubIntfDot1qEncapConfigMode" )
FieldType = TacLazyType( "Ebra::FlexEncapField::FieldType" )
SpecType = TacLazyType( "Ebra::FlexEncapSpec::SpecType" )

class EncapsulationConfigMode( EncapsulationBaseMode, CliSave.Mode ):
   def __init__( self, intfName ):
      EncapsulationBaseMode.__init__( self, intfName )
      CliSave.Mode.__init__( self, intfName )

IntfConfigMode.addChildMode( EncapsulationConfigMode, after=[ "EthIntf.ethIntf" ] )
EncapsulationConfigMode.addCommandSequence( "Encapsulation.config" )

def fieldStr( field ):
   if field.type == FieldType.anyOrNone:
      return None
   elif field.type == FieldType.dot1q:
      # We only support single VLAN tag
      # Note: range collection is 1-based, see BUG495934 more for details
      return field.range[ 1 ].lower
   else:
      assert False
      return None

def specStr( spec ):
   if spec.type == SpecType.client:
      return 'client'
   elif spec.type == SpecType.anyOrNone:
      return None
   elif spec.type == SpecType.tagged:
      outer = fieldStr( spec.outer )
      inner = fieldStr( spec.inner )
      if outer and inner:
         # We only support dot1q TPID
         return 'dot1q outer {} inner {}'.format( outer, inner )
      elif outer:
         return 'dot1q {}'.format( outer )
      else:
         # invalid
         assert False
         return None
   else:
      # invalid
      assert False
      return None

def entryStr( entry ):
   client = specStr( entry.client )
   network = specStr( entry.network )
   cmd = 'client {}'.format( client )
   if network:
      cmd += ' network {}'.format( network )
   return cmd

@CliSave.saver( "Interface::SubIntfConfig", "interface/config/subintf",
                requireMounts=( "interface/status/all", ) )
def saveIntfEncapsulationConfig( entity, root, sysdbRoot, options, requireMounts ):
   if not toggleMplsEvpnVpwsP2Enabled():
      # return here otherwise CliSaveTests that don't have the toggle enabled
      # will trip over "no encapsulation vlan" added by this saver
      return
   intfId = entity.intfId
   mode = root[ IntfConfigMode ].getOrCreateModeInstance( intfId )
   cmds = mode[ 'EthIntf.ethIntf' ]
   # We can unconditionally create the mode instance because we have defined
   # skipIfEmpty. This means the submode will be printed even if only comments are
   # configured.
   encapMode = mode[ EncapsulationConfigMode ].getOrCreateModeInstance( intfId )
   flexEncapMode = ( entity.dot1qEncapConfigMode ==
                     Dot1qEncapMode.subIntfDot1qEncapConfigFlexEncap )
   if flexEncapMode:
      cmds = encapMode[ "Encapsulation.config" ]
      if entity.flexEncap:
         for entry in entity.flexEncap.entry:
            cmd = entryStr( entry )
            cmds.addCommand( cmd )
   elif options.saveAll:
      cmds.addCommand( 'no encapsulation vlan' )
