#!/usr/bin/env python
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import CliMode.Firewall
import CliMode.Classification
import Tac
import CliSave
from IntfCliSave import IntfConfigMode

class FirewallConfigMode( CliMode.Firewall.FirewallConfigMode,
                          CliSave.Mode ):
   def __init__( self, param ):
      CliMode.Firewall.FirewallConfigMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

CliSave.GlobalConfigMode.addChildMode( FirewallConfigMode, after=[ IntfConfigMode ] )
FirewallConfigMode.addCommandSequence( 'firewall' )

@CliSave.saver( 'Firewall::Config', 'firewall/config/cli' )
def saveRouterFirewallConfig( entity, root, sysdbRoot, options ):
   if entity.enabled:
      fwMode = root[ FirewallConfigMode ].getOrCreateModeInstance( 'firewall' )
      fwMode[ 'firewall' ].addCommand( "no shutdown" )
      if entity.defaultPolicyDrop == 'enable':
         fwMode[ 'firewall' ].addCommand(
            "segment policy policy-drop-all default" )
      elif entity.defaultPolicyDrop == 'disable':
         fwMode[ 'firewall' ].addCommand(
            "no segment policy policy-drop-all default" )
   elif options.saveAll:
      fwMode = root[ FirewallConfigMode ].getOrCreateModeInstance( 'firewall' )
      fwMode[ 'firewall' ].addCommand( "shutdown" )
      if entity.defaultPolicyDrop == 'enable':
         fwMode[ 'firewall' ].addCommand(
            "segment policy policy-drop-all default" )
      elif entity.defaultPolicyDrop == 'disable':
         fwMode[ 'firewall' ].addCommand(
            "no segment policy policy-drop-all default" )

class FirewallPolicyConfigMode( CliMode.Firewall.FirewallPolicyConfigMode,
                                CliSave.Mode ):
   def __init__( self, param ):
      CliMode.Firewall.FirewallPolicyConfigMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

FirewallConfigMode.addChildMode( FirewallPolicyConfigMode, after=[ 'firewall' ] )
FirewallPolicyConfigMode.addCommandSequence( 'policy' )

@CliSave.saver( 'Firewall::Config', 'firewall/config/cli' )
def savePolicyConfig( entity, root, sysdbRoot, options ):
   for policyName in entity.policy:
      if entity.policy[ policyName ].readonly:
         continue
      fwMode = root[ FirewallConfigMode ].getOrCreateModeInstance( 'firewall' )
      cmds = fwMode[ FirewallPolicyConfigMode
                             ].getOrCreateModeInstance( policyName )[ 'policy' ]
      for seq in entity.policy[ policyName ].rule.keys():
         cmd = '%s ' % str( seq )
         service = entity.policy[ policyName ].rule[ seq ].serviceName
         action = entity.policy[ policyName ].rule[ seq ].action
         log = entity.policy[ policyName ].rule[ seq ].log
         if action == 'deny':
            cmd += 'application %s action drop' % service
         elif action == 'statelessDeny':
            cmd += 'application %s action drop stateless' % service
         elif action == 'permit':
            cmd += 'application %s action forward' % service
         elif action == 'statelessPermit':
            cmd += 'application %s action forward stateless' % service
         if log:
            cmd += ' log'
         cmds.addCommand( cmd )

class FirewallVrfConfigMode( CliMode.Firewall.FirewallVrfConfigMode,
                             CliSave.Mode ):
   def __init__( self, param ):
      CliMode.Firewall.FirewallVrfConfigMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

FirewallConfigMode.addChildMode( FirewallVrfConfigMode, after=[ 'firewall' ] )
FirewallVrfConfigMode.addCommandSequence( 'vrf' )

class FirewallVrfSegmentConfigMode( CliMode.Firewall.FirewallVrfSegmentConfigMode,
                                    CliSave.Mode ):
   def __init__( self, param ):
      CliMode.Firewall.FirewallVrfSegmentConfigMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

FirewallVrfConfigMode.addChildMode( FirewallVrfSegmentConfigMode )
FirewallVrfSegmentConfigMode.addCommandSequence( 'segment' )

class FirewallSegmentConfigMode( CliMode.Firewall.FirewallSegmentConfigMode,
                                 CliSave.Mode ):
   def __init__( self, param ):
      CliMode.Firewall.FirewallSegmentConfigMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

FirewallConfigMode.addChildMode( FirewallSegmentConfigMode )
FirewallSegmentConfigMode.addCommandSequence( 'segment' )

class ClassMapFirewallVrfConfigMode( CliMode.Firewall.ClassMapFirewallVrfConfigMode,
                                     CliSave.Mode ):

   def __init__( self, param ):
      CliMode.Firewall.ClassMapFirewallVrfConfigMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

FirewallVrfSegmentConfigMode.addChildMode( ClassMapFirewallVrfConfigMode )
ClassMapFirewallVrfConfigMode.addCommandSequence( 'segmentDef' )

class ClassMapFirewallConfigMode( CliMode.Firewall.ClassMapFirewallConfigMode,
                                  CliSave.Mode ):

   def __init__( self, param ):
      CliMode.Firewall.ClassMapFirewallConfigMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

FirewallSegmentConfigMode.addChildMode( ClassMapFirewallConfigMode )
ClassMapFirewallConfigMode.addCommandSequence( 'segmentDef' )

@CliSave.saver( 'Firewall::Config', 'firewall/config/cli' )
def saveCmapVrfConfig( entity, root, sysdbRoot, options ):
   for cmapName in entity.classMap:
      fwMode = root[ FirewallConfigMode ].getOrCreateModeInstance( 'firewall' )
      if '__VRF_' not in cmapName:
         continue
      segIndex = cmapName.find( '_segment_' )
      vrfName = cmapName[ len( '__VRF_' ) : segIndex ]
      segmentName = cmapName[ segIndex + len( '_segment_' ) : ]
      firewallMode = fwMode[ FirewallVrfConfigMode
                             ].getOrCreateModeInstance( vrfName )
      segMode = firewallMode[ FirewallVrfSegmentConfigMode
                              ].getOrCreateModeInstance( segmentName )
      if entity.classMap[ cmapName ].intfVlanData:
         cmds = segMode[ ClassMapFirewallVrfConfigMode
                   ].getOrCreateModeInstance( 'segmentDef' )[ 'segmentDef' ]
         cmd = 'match interface '
         cmd += ' '.join( sorted( entity.classMap[ cmapName ].intfVlanData.keys() ) )
         cmds.addCommand( cmd )

      if vrfName in entity.vrf:
         if segmentName in entity.vrf[ vrfName ].segmentDir.segment:
            cmds = segMode[ ClassMapFirewallVrfConfigMode
                   ].getOrCreateModeInstance( 'segmentDef' )[ 'segmentDef' ]
            if entity.vrf[ vrfName ].segmentDir.segment[
                  segmentName ].ipv4PrefixList:
               ipv4Prefix = entity.vrf[ vrfName ].segmentDir.segment[
                     segmentName ].ipv4PrefixList
               cmd = 'match prefix-ipv4 %s' % ipv4Prefix
               cmds.addCommand( cmd )
            if entity.vrf[ vrfName ].segmentDir.segment[
                  segmentName ].ipv6PrefixList:
               ipv6Prefix = entity.vrf[ vrfName ].segmentDir.segment[
                     segmentName ].ipv6PrefixList
               cmd = 'match prefix-ipv6 %s' % ipv6Prefix
               cmds.addCommand( cmd )

@CliSave.saver( 'Firewall::Config', 'firewall/config/cli' )
def saveCmapConfig( entity, root, sysdbRoot, options ):
   def intfVlanRangesToStrings( intfVlanRanges ):
      vlanRanges = [ ( x.vlanBegin, x.vlanEnd )
                      for x in intfVlanRanges.vlanRange ]
      vlanRangeStrings = []
      for vlanRange in sorted( vlanRanges, key=lambda x: x[ 0 ] ):
         if vlanRange[ 0 ] == vlanRange[ 1 ]:
            vlanRangeStrings.append( str( vlanRange[ 0 ] ) )
         else:
            vlanRangeStrings.append( '%d-%d' %
                                     ( vlanRange[ 0 ], vlanRange[ 1 ] ) )
      return vlanRangeStrings

   for cmapName in entity.classMap:
      firewallMode = root[ FirewallConfigMode ].getOrCreateModeInstance( 'firewall' )
      segIndex = cmapName.find( '__segment_' )
      if segIndex == -1:
         continue
      segmentName = cmapName[ segIndex + len( '__segment_' ) : ]
      segMode = firewallMode[ FirewallSegmentConfigMode
                              ].getOrCreateModeInstance( segmentName )
      if entity.classMap[ cmapName ].intfVlanData:
         cmds = segMode[ ClassMapFirewallConfigMode
                   ].getOrCreateModeInstance( 'segmentDef' )[ 'segmentDef' ]

         # All interface-only match criteria to be saved in one line
         # All interface-vlan match criteria to be saved in one line per interface
         intfOnlyCmd = ''
         intfVlanCmds = []
         for intf, intfVlan in entity.classMap[ cmapName ].intfVlanData.iteritems():
            if intfVlan.vlanRange:
               vlanRangeStrings = intfVlanRangesToStrings( intfVlan )
               cmd = 'match interface %s vlan %s' % (
                      intf, ', '.join( vlanRangeStrings ) )
               intfVlanCmds.append( cmd )
            else:
               intfOnlyCmd += ' %s' % intf

         if intfOnlyCmd:
            intfOnlyCmd = 'match interface' + intfOnlyCmd
            cmds.addCommand( intfOnlyCmd )
         if intfVlanCmds:
            for cmd in intfVlanCmds:
               cmds.addCommand( cmd )

class VrfSegmentPoliciesConfigMode( CliMode.Firewall.VrfSegmentPolicyMode,
                                 CliSave.Mode ):
   def __init__( self, param ):
      CliMode.Firewall.VrfSegmentPolicyMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

FirewallVrfSegmentConfigMode.addChildMode( VrfSegmentPoliciesConfigMode )
VrfSegmentPoliciesConfigMode.addCommandSequence( 'segPolicies' )

class SegmentPoliciesConfigMode( CliMode.Firewall.SegmentPolicyMode,
                                 CliSave.Mode ):
   def __init__( self, param ):
      CliMode.Firewall.SegmentPolicyMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

FirewallSegmentConfigMode.addChildMode( SegmentPoliciesConfigMode )
SegmentPoliciesConfigMode.addCommandSequence( 'segPolicies' )

@CliSave.saver( 'Firewall::Config', 'firewall/config/cli' )
def saveVrfSegmentPolicies( entity, root, sysdbRoot, options ):
   for vrfName in entity.vrf:
      fwMode = root[ FirewallConfigMode ].getOrCreateModeInstance( 'firewall' )
      firewallMode = fwMode[ FirewallVrfConfigMode
                             ].getOrCreateModeInstance( vrfName )
      for segmentName in entity.vrf[ vrfName ].segmentDir.segment:
         segMode = firewallMode[ FirewallVrfSegmentConfigMode
                                ].getOrCreateModeInstance( segmentName )
         segment = entity.vrf[ vrfName ].segmentDir.segment[ segmentName ]
         if segment.policy:
            cmds = segMode[ VrfSegmentPoliciesConfigMode
                      ].getOrCreateModeInstance( 'segPolicies' )[ 'segPolicies' ]
            for fromSegment in segment.policy.keys():
               cmd = 'from %s policy %s' % ( fromSegment,
                     segment.policy[ fromSegment ] )
               cmds.addCommand( cmd )

@CliSave.saver( 'Firewall::Config', 'firewall/config/cli' )
def saveSegmentPolicies( entity, root, sysdbRoot, options ):
   if not entity.segmentDir or not entity.segmentDir.segment:
      return
   firewallMode = root[ FirewallConfigMode ].getOrCreateModeInstance( 'firewall' )
   for segmentName in entity.segmentDir.segment:
      segMode = firewallMode[ FirewallSegmentConfigMode
                             ].getOrCreateModeInstance( segmentName )
      segment = entity.segmentDir.segment[ segmentName ]
      if segment.policy:
         cmds = segMode[ SegmentPoliciesConfigMode
                   ].getOrCreateModeInstance( 'segPolicies' )[ 'segPolicies' ]
         for fromSegment in segment.policy:
            cmd = 'from %s policy %s' % ( fromSegment,
                  segment.policy[ fromSegment ] )
            cmds.addCommand( cmd )
