# Copyright (c) 2016 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import CliSave
import Tac
from CliMode.Evpn import EthernetSegmentMode
from IntfCliSave import IntfConfigMode
from CliSavePlugin.RoutingBgpCliSave import (
   RouterBgpBaseConfigMode,
   RouterBgpVrfConfigMode,
   saveAfConfigCallbackDict )
from CliSavePlugin.RoutingBgpCliSave import RouterBgpBaseAfConfigMode
from CliToken.Evpn import nexthopSelfSupportNlris
from RouteMapLib import isAsdotConfigured
from Toggles import (
   BgpCommonToggleLib,
   EvpnToggleLib,
   EvpnLibToggleLib,
)

ethIntfId = Tac.Type( 'Arnet::EthIntfId' )
portChannelIntfId = Tac.Type( 'Arnet::PortChannelIntfId' )
dfElectionHoldTimeDefault = \
      Tac.Value( 'Evpn::DFElectionHoldTime' ).defaultHoldTime
dfElectionDelayTimeDefault = 0
multihomeModeToRedundancyMap = \
      { 'multihomeModeAllActive' : 'all-active',
        'multihomeModeSingleActive' : 'single-active' }
defaultRt = Tac.Value( 'Arnet::EthAddr' )

class EthernetSegmentConfigMode( EthernetSegmentMode, CliSave.Mode ):
   def __init__( self, param ):
      EthernetSegmentMode.__init__( self )
      CliSave.Mode.__init__( self, param )

IntfConfigMode.addChildMode( EthernetSegmentConfigMode )
EthernetSegmentConfigMode.addCommandSequence( 'Evpn.intf' )

def allEthernetSegmentIntfNames( ethernetSegmentConfig, includeEligible=False,
                                 requireMounts=None ):
   # if includeEligible, then all Ethernet and Port-Channel interfaces are
   # include; otherwise, only L2 Ethernet and Port-Channel interfaces are
   # included or interface with this feature configured
   # For eg:
   # int Ethernet1
   #   no switchport
   #     evpn ethernet-segment
   #       esi ...
   # int Ethernet2
   #   no switchport
   #   ...
   # int Ethernet3
   #   switchport 
   # 
   # Result:
   # "show running-config all" -- display default and actually configured esi
   # for Ethernet1 and Ethernet3
   # "show running-config all detail" -- display default and actually
   # configured esi for Ethernet1, Ethernet2 and Ethernet3
   assert requireMounts
   intfConfigDir = requireMounts[ 'interface/config/all' ]
   brConfig = requireMounts[ 'bridging/config' ]
   intfNames = []
   for intfName in intfConfigDir:
      if portChannelIntfId.isPortChannelIntfId( intfName ) or \
            ethIntfId.isEthIntfId( intfName ):
         if includeEligible:
            intfNames.append( intfName )
         else:
            sic = brConfig.switchIntfConfig.get( intfName )
            if ( ( sic and \
                   ( sic.switchportMode == 'access' or
                     sic.switchportMode == 'trunk' ) ) or
                 intfName in ethernetSegmentConfig.intfConfig ):
               intfNames.append( intfName )
   return intfNames

@CliSave.saver( 'Evpn::EthernetSegmentCliConfigDir', 'evpn/interface/config',
                requireMounts=( 'interface/config/all',
                                'bridging/config', ) )
def saveEthernetSegmentConfig( ethernetSegmentConfig,
                               root, sysdbRoot, options, requireMounts ):
   configuredIntfNames = []
   saveAll = options.saveAll
   if options.saveAllDetail:
      configuredIntfNames = \
         allEthernetSegmentIntfNames( ethernetSegmentConfig, includeEligible=True,
                                      requireMounts=requireMounts )
   elif options.saveAll:
      configuredIntfNames = \
         allEthernetSegmentIntfNames( ethernetSegmentConfig,
                                      requireMounts=requireMounts )
   else:
      configuredIntfNames = ethernetSegmentConfig.intfConfig.keys()
   for intf in configuredIntfNames:
      intfMode = root[ IntfConfigMode ].getOrCreateModeInstance( intf )
      ethernetSegmentMode = \
            intfMode[ EthernetSegmentConfigMode ].getOrCreateModeInstance( None )
      cmds = ethernetSegmentMode[ 'Evpn.intf' ]
      intfConfig = ethernetSegmentConfig.intfConfig.get( intf, None )
      if not intfConfig:
         if saveAll:
            intfConfig = Tac.newInstance( 'Evpn::EthernetSegmentCliConfig', intf )
         else:
            continue
      if not intfConfig.esid.none:
         cmds.addCommand( 'identifier %s' % intfConfig.esid.value )
      elif saveAll:
         cmds.addCommand( 'no identifier' )

      if EvpnToggleLib.toggleEvpnSingleActiveToggleEnabled():
         if intfConfig.multihomeMode != intfConfig.multihomeModeDefault:
            cmds.addCommand( 'redundancy %s' %
                             multihomeModeToRedundancyMap[
                                intfConfig.multihomeMode ] )
         elif saveAll:
            cmds.addCommand( 'no redundancy' )

      if intfConfig.holdTime != dfElectionHoldTimeDefault and \
         intfConfig.dfElectionDelayTime != dfElectionDelayTimeDefault:
         cmds.addCommand( 'designated-forwarder election hold-time %s'
                          ' subsequent-hold-time %d' %
                          ( intfConfig.holdTime,
                            intfConfig.dfElectionDelayTime * 1000 ) )
      elif intfConfig.holdTime != dfElectionHoldTimeDefault:
         cmds.addCommand( 'designated-forwarder election hold-time %s' %
                          intfConfig.holdTime )
      elif intfConfig.dfElectionDelayTime != dfElectionDelayTimeDefault:
         cmds.addCommand( 'designated-forwarder election hold-time %s'
                          ' subsequent-hold-time %d' %
                          ( dfElectionHoldTimeDefault,
                            intfConfig.dfElectionDelayTime * 1000 ) )
      elif saveAll:
         cmds.addCommand( 'no designated-forwarder election hold-time' )

      if EvpnToggleLib.toggleEvpnEncapBumFilterToggleEnabled():
         if intfConfig.encapBumFilterHoldTime != 0:
            cmds.addCommand( 'mpls tunnel flood filter time %d' %
                             ( intfConfig.encapBumFilterHoldTime * 1000 ) )
         elif saveAll:
            cmds.addCommand( 'no mpls tunnel flood filter time' )

      if EvpnLibToggleLib.toggleSharedEsiLabelEnabled():
         if intfConfig.sharedEsiLabelIndex != 0:
            cmds.addCommand( 'mpls shared index %d' %
                             intfConfig.sharedEsiLabelIndex )
         elif saveAll:
            cmds.addCommand( 'no mpls shared index' )

      if intfConfig.esImportRouteTarget != defaultRt.stringValue:
         cmds.addCommand( 'route-target import %s' % intfConfig.esImportRouteTarget )
      elif saveAll:
         cmds.addCommand( 'no route-target import' )


@CliSave.saver( 'Routing::Bgp::MacPlugin::EvpnConfig', 'routing/bgp/evpnconfig',
                requireMounts = ( 'routing/bgp/config',
                                  'routing/bgp/asn/config', ) )
def saveEvpnHostFlapConfig( hostFlapConfig, root, sysdbRoot, options,
                            requireMounts ):
   bgpConfig = requireMounts[ 'routing/bgp/config' ]
   asnConfig = requireMounts[ 'routing/bgp/asn/config' ]

   if bgpConfig.asNumber == 0:
      return

   window = hostFlapConfig.duplicateTimeout
   threshold = hostFlapConfig.duplicateThreshold
   isDefault = ( window == hostFlapConfig.duplicateTimeoutDefault and
                 threshold == hostFlapConfig.duplicateThresholdDefault )
   if isDefault and not options.saveAll:
      return

   bgpMode = root[ RouterBgpBaseConfigMode ].getOrCreateModeInstance( (
         bgpConfig.asNumber, isAsdotConfigured( asnConfig ), ) )
   bgpAfMode = bgpMode[ RouterBgpBaseAfConfigMode
         ].getOrCreateModeInstance( 'evpn' )

   if window == 0 and threshold == 0:
      cmd = 'no host-flap detection'
   else:
      cmd = 'host-flap detection window %s threshold %s' % (window, threshold)
   bgpAfMode[ 'Bgp.afConfig.evpn' ].addCommand( cmd )

def saveEvpnNexthopSelf( cmds, config, saveAll ):
   for rType in nexthopSelfSupportNlris:
      attrName = 'nexthopSelfEvpnType' + str( rType )
      val = getattr( config, attrName )
      valDefault = getattr( config, attrName + 'Default' )
      if val != valDefault or saveAll:
         if val == 'isTrue':
            cmds.append( 'neighbor default next-hop-self '
                         'received-evpn-routes route-type ip-prefix' )
         elif val == 'isInvalid':
            cmds.append( 'no neighbor default next-hop-self '
                         'received-evpn-routes route-type ip-prefix' )

def saveEvpnConfig( bgpConfig, af, options=None ):
   cmds = []
   saveAll = options.saveAll if options else False
   saveAllDetail = options.saveAllDetail if options else False
   if bgpConfig.domainIdEvpn != bgpConfig.domainIdEvpnDefault:
      cmds.append( 'domain identifier %s' % bgpConfig.domainIdEvpn.toStrepCli() )
   elif saveAll or saveAllDetail:
      cmds.append( 'no domain identifier' )

   if BgpCommonToggleLib.toggleEvpnVxlanL3UseNexthopIgpCostEnabled():
      if bgpConfig.afEvpnVxlanL3UseNexthopIgpCost != \
            bgpConfig.afEvpnVxlanL3UseNexthopIgpCostDefault:
         cmds.append( 'encapsulation vxlan layer-3 set next-hop igp-cost' )
      elif saveAll or saveAllDetail:
         cmds.append( 'no encapsulation vxlan layer-3 set next-hop igp-cost' )

   if bgpConfig.afEvpnNexthopResolutionDisabled:
      cmds.append( 'next-hop resolution disabled' )
   elif saveAll or saveAllDetail:
      cmds.append( 'no next-hop resolution disabled' )

   if bgpConfig.vpnPruningEnabledEvpn != bgpConfig.vpnPruningEnabledEvpnDefault:
      cmds.append( 'route import match-failure action discard' )
   elif saveAll or saveAllDetail:
      cmds.append( 'no route import match-failure action discard' )

   if BgpCommonToggleLib.toggleArBgpAllowEvpnNexthopSelfEnabled():
      saveEvpnNexthopSelf( cmds, bgpConfig, saveAll )

   return cmds

# saveAfConfigCallbackDict would be used to display Evpn specific config in
# "show bgp config active" output.
# For  "show running" the config is displayed through saveEvpnAfConfig with
# CliSave.saver decorator.
saveAfConfigCallbackDict[ 'evpn' ] = saveEvpnConfig

# Save address-family evpn configuration commands. Majority of the
# evpn config commands are defined and cli saved in BgpCommon package.
@CliSave.saver( 'Routing::Bgp::Config', 'routing/bgp/config',
                requireMounts=( 'routing/bgp/asn/config', ) )
def saveEvpnAfConfig( bgpConfig, root, sysdbRoot, options, requireMounts ):
   if bgpConfig.asNumber == 0:
      return

   asnConfig = requireMounts[ 'routing/bgp/asn/config' ]
   bgpMode = root[ RouterBgpBaseConfigMode ].getOrCreateModeInstance( (
      bgpConfig.asNumber, isAsdotConfigured( asnConfig ), ) )
   cmds = saveEvpnConfig( bgpConfig, 'evpn', options )
   if cmds:
      bgpAfMode = bgpMode[ RouterBgpBaseAfConfigMode
      ].getOrCreateModeInstance( 'evpn' )
      for cmd in cmds:
         bgpAfMode[ 'Bgp.afConfig.evpn' ].addCommand( cmd )

@CliSave.saver( 'Routing::Bgp::VrfConfigDir', 'routing/bgp/vrf/config',
                requireMounts=( 'routing/bgp/config', 'routing/bgp/asn/config', ) )
def saveEvpnVrfConfig( vrfConfigDir, root, sysdbRoot, options, requireMounts ):
   bgpConfig = requireMounts[ 'routing/bgp/config' ]
   if bgpConfig.asNumber == 0:
      return
   if not BgpCommonToggleLib.toggleArBgpOismToggleEnabled():
      # evpn multicast is the only per-VRF EVPN config that needs
      # saving today.
      return
   asnConfig = requireMounts[ 'routing/bgp/asn/config' ]
   bgpMode = root[ RouterBgpBaseConfigMode ].getOrCreateModeInstance( (
      bgpConfig.asNumber, isAsdotConfigured( asnConfig ), ) )
   for vrfName in vrfConfigDir.vrfConfig:
      vrfBgpConfig = vrfConfigDir.vrfConfig[ vrfName ]
      bgpVrfMode = bgpMode[ RouterBgpVrfConfigMode
            ].getOrCreateModeInstance( vrfName )
      cmd = ''
      if vrfBgpConfig.afEvpnOismEnabled:
         cmd = 'evpn multicast'
      elif options.saveAll or options.saveAllDetail:
         cmd = 'no evpn multicast'
      if cmd:
         bgpVrfMode[ 'Bgp.vrf.config' ].addCommand( cmd )
