# Copyright (c) 2006-2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

#-------------------------------------------------------------------------------
# This module implements saving the EthIntf CLI.
#-------------------------------------------------------------------------------
import Tac, Tracing, CliSave, IntfCliSave
from IntfCliSave import IntfConfigMode, InterfaceDefaultsConfigMode
from EthIntfLib import tokenToFecEncodingAttrs
import CliMode.EthIntf                    # pylint: disable-msg=W0611
from CliMode.EthIntf import InterfaceDefaultsEthernetMode
import Ethernet

traceDetail = Tracing.trace2

__defaultTraceHandle__ = Tracing.Handle( "EthCli" )
t0 = Tracing.trace0

IntfConfigMode.addCommandSequence( 'EthIntf.ethIntf', after=[ 'Arnet.intf' ] )

#-------------------------------------------------------------------------------
# Saves the state of an Interface::EthIntfConfig object.
#-------------------------------------------------------------------------------

flowcontrolStrings = { 'flowControlConfigOn' : 'on',
                       'flowControlConfigOff' : 'off',
                       'flowControlConfigDesired' : 'desired' }

# XXX_LWR: These should move somewhere better -- eg EthIntfCliLib.py.
linkModeStrings = {
   'linkModeAutoneg' : 'auto',
   'linkModeAuto40GbpsFull' : 'auto 40gfull',
   'linkMode10MbpsHalf' : '10half',
   'linkMode10MbpsFull' : '10full',
   'linkMode100MbpsHalf' : '100half',
   'linkMode100MbpsFull' : '100full',
   'linkModeForced10MbpsHalf' : 'forced 10half',
   'linkModeForced10MbpsFull' : 'forced 10full',
   'linkModeForced100MbpsHalf' : 'forced 100half',
   'linkModeForced100MbpsFull' : 'forced 100full',
   'linkModeForced1GbpsHalf' : 'forced 1000half',
   'linkModeForced1GbpsFull' : 'forced 1000full',
   'linkModeForced10GbpsFull' : 'forced 10000full',
   'linkModeForced25GbpsFull' : 'forced 25gfull',
   'linkModeForced40GbpsFull' : 'forced 40gfull',
   'linkModeForced50GbpsFull' : 'forced 50gfull',
   'linkModeForced50GbpsFull1Lane' : '50g-1',
   'linkModeForced100GbpsFull' : 'forced 100gfull',
   'linkModeForced100GbpsFull2Lane' : '100g-2',
   'linkModeForced200GbpsFull4Lane' : '200g-4',
   'linkModeForced400GbpsFull8Lane' : '400g-8',
}

uniLinkModeStrings = {
   'uniLinkModeSendOnly' : 'send-only',
   'uniLinkModeReceiveOnly' : 'receive-only',
   'uniLinkModeSendReceive' : 'send-receive'
}

# XXX_LWR: brutal hack to support configuring 1000BASE-T SFPs to
#          advertise only 100/full during autoneg. Full fix is to
#          implement the RFE described in BUG282.
_sfp1000BaseTAutonegLinkModes = Tac.Value( 'Interface::EthLinkModeSet' )
_sfp1000BaseTAutonegLinkModes.mode100MbpsFull = True

def saveErrorCorrection( entity, cmds, saveAll ):

   # Track if any of the encoding values are set. This determines if we are in the
   # default config.
   allDefault = True
   for attr in entity.fecEncodingConfig.attributes:
      if getattr(  entity.fecEncodingConfig, attr ):
         allDefault = False

   if allDefault:
      if saveAll and not entity.isCoherentFecConfigured:
         cmds.addCommand( 'default error-correction encoding' )
   elif entity.fecEncodingConfig.fecEncodingDisabled:
      cmds.addCommand( 'no error-correction encoding' )
   else:
      for ( token, attrList ) in tokenToFecEncodingAttrs.iteritems():
         if any( getattr( entity.fecEncodingConfig, attr ) for attr in attrList ):
            cmds.addCommand( 'error-correction encoding %s' % token )
         elif saveAll:
            cmds.addCommand( 'default error-correction encoding %s' % token )

   # Save the bypass mode configuration.
   FecBypassMode = Tac.Type( 'Interface::EthFecBypassMode' )
   if( entity.fecBypass == FecBypassMode.fecBypassDisabled and
       saveAll ):
      cmds.addCommand( 'no error-correction reed-solomon bypass' )
   elif entity.fecBypass == FecBypassMode.fecBypassIndication:
      cmds.addCommand( 'error-correction reed-solomon bypass indication' )
   elif entity.fecBypass == FecBypassMode.fecBypassCorrection:
      cmds.addCommand( 'error-correction reed-solomon bypass correction' )

def saveNegotiation25gMode( entity, cmds, saveAll ):
   # Determines if we are in 'default' config.
   allDefault = True
   for mode in entity.negotiation25gModeConfig.attributes:
      if not getattr( entity.negotiation25gModeConfig, mode ):
         allDefault = False

   if not allDefault:
      if entity.negotiation25gModeConfig.consortium25g:
         cmds.addCommand( 'phy media 25gbase-cr negotiation standard consortium' )
      elif entity.negotiation25gModeConfig.ieee25g:
         cmds.addCommand( 'phy media 25gbase-cr negotiation standard ieee' )
   elif saveAll:
      cmds.addCommand( 'phy media 25gbase-cr negotiation standard consortium ieee' )

# These should move somewhere better -- eg EthIntfCliLib.py.
def advertisedModesToStrings( advertisedModes ) :
   advertisedString = ''
   advertisedString += '10half ' if advertisedModes.mode10MbpsHalf else ''
   advertisedString += '10full ' if advertisedModes.mode10MbpsFull else ''
   advertisedString += '100half ' if advertisedModes.mode100MbpsHalf else ''
   advertisedString += '100full ' if advertisedModes.mode100MbpsFull else ''
   advertisedString += '1000half ' if advertisedModes.mode1GbpsHalf else ''
   advertisedString += '1000full ' if advertisedModes.mode1GbpsFull else ''
   advertisedString += '2.5gfull ' if advertisedModes.mode2p5GbpsFull else ''
   advertisedString += '5gfull ' if advertisedModes.mode5GbpsFull else ''
   advertisedString += '10gfull ' if advertisedModes.mode10GbpsFull else ''
   advertisedString += '25gfull ' if advertisedModes.mode25GbpsFull else ''
   advertisedString += '40gfull ' if advertisedModes.mode40GbpsFull else ''
   advertisedString += '50gfull ' if advertisedModes.mode50GbpsFull else ''
   advertisedString += '50g-1 ' if advertisedModes.mode50GbpsFull1Lane else ''
   advertisedString += '100gfull ' if advertisedModes.mode100GbpsFull else ''
   advertisedString += '100g-2 ' if advertisedModes.mode100GbpsFull2Lane else ''
   advertisedString += '200g-4 ' if advertisedModes.mode200GbpsFull4Lane else ''
   advertisedString += '400g-8 ' if advertisedModes.mode400GbpsFull8Lane else ''
   return advertisedString

@CliSave.saver( 'Tac::Dir',
                'interface/config/eth/phy/slice',
                attrName = None,
                requireMounts = ( 'interface/status/all',
                                  'interface/config/global', ) )
def saveEthIntfConfig( baseDir, root, sysdbRoot, options, requireMounts ):
   for ethPhyIntfConfigDir in baseDir.values():
      for entity in ethPhyIntfConfigDir.intfConfig.values():
         if options.intfFilter and entity.intfId not in options.intfFilter:
            continue
         if entity.intfId.startswith( "Internal" ):
            # Internal interface. Abort
            # BUG944 - need a more general way of doing this. Should 'internal' be
            # an attribute of the IntfConfig intead?
            continue

         # Save the baseclass (Arnet::IntfConfig) attributes.
         IntfCliSave.saveIntfConfig( entity, root, sysdbRoot, options,
                                     requireMounts )

         # Save the EthIntf attributes.
         mode = root[ IntfConfigMode ].getOrCreateModeInstance( entity.intfId )
         cmds = mode[ 'EthIntf.ethIntf' ]
         saveAll = options.saveAll

         if entity.linkUpDebouncePeriod != 0 or entity.linkDownDebouncePeriod != 0 :
            cmds.addCommand( 'link-debounce time %d %d' \
                             % ( entity.linkUpDebouncePeriod,
                                 entity.linkDownDebouncePeriod ) )
         elif saveAll:
            cmds.addCommand( 'no link-debounce' )

         if entity.txFlowcontrolLocal != 'flowControlConfigUnknown':
            cmds.addCommand( 'flowcontrol send %s'
                             % flowcontrolStrings[ entity.txFlowcontrolLocal ] )
         elif saveAll:
            cmds.addCommand( 'no flowcontrol send' )

         if entity.rxFlowcontrolLocal != 'flowControlConfigUnknown':
            cmds.addCommand( 'flowcontrol receive %s'
                             % flowcontrolStrings[ entity.rxFlowcontrolLocal ] )
         elif saveAll:
            cmds.addCommand( 'no flowcontrol receive' )

         if entity.timestampMode == 'timestampModeBeforeFcs':
            cmds.addCommand( 'mac timestamp before-fcs' )
         elif entity.timestampMode == 'timestampModeReplaceFcs':
            cmds.addCommand( 'mac timestamp replace-fcs' )
         elif entity.timestampMode == 'timestampModeHeader':
            cmds.addCommand( 'mac timestamp header' )
         else:
            assert entity.timestampMode == 'timestampModeDisabled'
            if saveAll:
               cmds.addCommand( 'no mac timestamp' )

         if entity.linkModeLocal != 'linkModeUnknown':
            # XXX_LWR: ugly special case for configuring 1000BASE-T SFPs to
            #          negotiate 100/full, using the command
            #
            #             'speed sfp-1000baset auto 100full'
            #
            #          Currently, this is the *ONLY* piece of hardware on
            #          which we support explicitly configuring what modes
            #          are advertised during autoneg. All other pieces of
            #          hardware advertise their full set of capabilities
            #          during autoneg. We have an RFE (BUG282) to support
            #          configuring this on all hardware, but that's not
            #          going to happen anytime soon, because it affects the
            #          code for all our PHYs and MACs.
            if ( entity.linkModeLocal == 'linkModeAutoneg' ) and \
                   entity.advertisedModesConfiguredLocal:
               # Use the deprecated sfp-1000baset command only if the advertised
               # capabilities are configured using this command. Otherwise use
               # the "speed auto" command.
               if entity.advertisedModesConfiguredForSfp1000BaseT :
                  assert entity.advertisedModesLocal == _sfp1000BaseTAutonegLinkModes
                  cmds.addCommand( 'speed sfp-1000baset auto 100full' )
               else:
                  cmds.addCommand( 'speed auto %s' % advertisedModesToStrings(
                     entity.advertisedModesLocal ) )
            else:
               cmds.addCommand(
                      'speed %s' % linkModeStrings[ entity.linkModeLocal ] )
         elif saveAll:
            cmds.addCommand( 'no speed' )

         if entity.l2Mtu != 0:
            cmds.addCommand( 'l2 mtu %u' % entity.l2Mtu )
         elif saveAll:
            cmds.addCommand( 'no l2 mtu' )

         if entity.l2Mru != 0:
            cmds.addCommand( 'l2 mru %u' % entity.l2Mru )
         elif saveAll:
            cmds.addCommand( 'no l2 mru' )

         if entity.congestionDropsLoggingMode == 'on':
            cmds.addCommand( 'logging event congestion-drops' )
         elif entity.congestionDropsLoggingMode == 'off':
            cmds.addCommand( 'no logging event congestion-drops' )
         elif saveAll:
            cmds.addCommand( 'default logging event congestion-drops' )

         if entity.unidirectionalLinkMode != 'uniLinkModeDisabled':
            cmds.addCommand( 'unidirectional %s' % \
                  uniLinkModeStrings[ entity.unidirectionalLinkMode ] )
         elif saveAll:
            cmds.addCommand( 'default unidirectional' )

         if entity.loopbackMode != 'loopbackNone':
            if entity.loopbackMode == 'loopbackMac':
               cmds.addCommand( 'traffic-loopback source system device mac' )
            elif entity.loopbackMode == 'loopbackPhy':
               cmds.addCommand( 'traffic-loopback source system device phy' )
            else:
               assert entity.loopbackMode == 'loopbackPhyRemote'
               cmds.addCommand( 'traffic-loopback source network device phy' )
         elif saveAll:
            cmds.addCommand( 'no traffic-loopback' )

         if entity.addr != '00:00:00:00:00:00':
            if entity.routerMacConfigured:
               rmac = Ethernet.convertMacAddrCanonicalToDisplay( entity.addr )
               cmds.addCommand( 'mac-address router %s' % rmac )
            else:
               cmds.addCommand( 'mac-address %s' % entity.addr )
         elif saveAll:
            cmds.addCommand( 'no mac-address' )

         saveErrorCorrection( entity, cmds, saveAll )
         saveNegotiation25gMode( entity, cmds, saveAll )

# Returns the string for congestion-drops interval command
def getLogIntervalCommandStr( saveAll, logInterval, extra='' ):
   if logInterval:
      return 'logging event congestion-drops interval %d' % logInterval
   elif saveAll:
      return 'default logging event congestion-drops'
   return None

CliSave.GlobalConfigMode.addCommandSequence( 'Interface.congestionDropsLogInterval' )

@CliSave.saver( 'Interface::EthPhyIntfGlobalDefaultConfigDir',
                'interface/config/eth/phy/globalDefault' )
def saveCongestionDropsLogInterval( entity, root, sysdbRoot, options ):
   cmds = root[ 'Interface.congestionDropsLogInterval' ]
   logIntervalCmdStr = getLogIntervalCommandStr( options.saveAll,
                                                 entity.congestionDropsLogInterval,
                                                 ' default' )
   if logIntervalCmdStr is not None:
      cmds.addCommand( logIntervalCmdStr )

CliSave.GlobalConfigMode.addCommandSequence( 'Interface.xcvrModeConfig' )

@CliSave.saver( 'Interface::EthPhyIntfGlobalDefaultConfigDir',
                'interface/config/eth/phy/globalDefault' )
def saveXcvrGlobalMode( entity, root, sysdbRoot, options, requireMounts ):
   cmds = root[ 'Interface.xcvrModeConfig' ]
   xcvrModeType = Tac.Type( 'Interface::XcvrMode' )

   qsfpModeConfig = entity.xcvrModeConfig[ 'qsfp' ]
   if qsfpModeConfig.isSestoFixedSystem():
      return
   if( qsfpModeConfig.globalMode == xcvrModeType.noXcvrMode and
       qsfpModeConfig.isSupportedPlatform() ):
      if options.saveAll:
         cmds.addCommand( 'default transceiver qsfp default-mode' )
   else:
      cmds.addCommand( 'transceiver qsfp default-mode 4x10G' )

class InterfaceDefaultsEthernetConfigMode( InterfaceDefaultsEthernetMode,
                                           CliSave.Mode ):

   def __init__( self, param ):
      InterfaceDefaultsEthernetMode.__init__( self )
      CliSave.Mode.__init__( self, param )

InterfaceDefaultsConfigMode.addChildMode( InterfaceDefaultsEthernetConfigMode )
InterfaceDefaultsEthernetConfigMode.addCommandSequence( 'interfaceDefaultsEthernet' )
IntfEnabledState = Tac.Type( 'Interface::IntfEnabledState' )

@CliSave.saver( 'Interface::GlobalIntfConfig', 'interface/config/global' )
def saveEthernetShutdown( entity, root, sysdbRoot, options ):
   if entity.defaultEthernetShutdown == IntfEnabledState.shutdown or options.saveAll:
      shutCmd = "shutdown" if entity.defaultEthernetShutdown == \
                                         IntfEnabledState.shutdown else "no shutdown"
      mode = root[ InterfaceDefaultsConfigMode ].getOrCreateModeInstance( '' )
      ethMode = mode[ InterfaceDefaultsEthernetConfigMode
                    ].getOrCreateModeInstance( '' )
      ethCmds = ethMode[ 'interfaceDefaultsEthernet' ]
      ethCmds.addCommand( shutCmd )
