# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import CliSave
from CliMode.Dps import ( RouterPathSelectionModeBase,
                          DpsPathGroupModeBase,
                          DpsPathGroupRemoteRouterBase,
                          DpsPolicyModeBase,
                          DpsPolicyRuleKeyBase,
                          DpsPolicyDefaultRuleBase,
                          DpsLoadBalanceProfileConfigModeBase,
                          DpsVrfConfigModeBase,
                          DEFAULT_PRIORITY,
                          DEFAULT_JITTER, JITTER_SCALE,
                          DEFAULT_LATENCY, LATENCY_SCALE,
                          DEFAULT_LOSSRATE, LOSS_RATE_SCALE, LOSS_RATE_ADJUSTMENT,
                          DEFAULT_UDP_PORT )

class RouterPathSelectionSaveMode( RouterPathSelectionModeBase, CliSave.Mode ):
   def __init__( self, param ):
      RouterPathSelectionModeBase.__init__( self )
      CliSave.Mode.__init__( self, param )

CliSave.GlobalConfigMode.addChildMode( RouterPathSelectionSaveMode )
RouterPathSelectionSaveMode.addCommandSequence( 'Dps.RouterPathSelection' )

class DpsPathGroupSaveMode( DpsPathGroupModeBase, CliSave.Mode ):
   def __init__( self, param ):
      DpsPathGroupModeBase.__init__( self, param )
      CliSave.Mode.__init__( self, param )

RouterPathSelectionSaveMode.addChildMode( DpsPathGroupSaveMode )
DpsPathGroupSaveMode.addCommandSequence( 'Dps.DpsPathGroup' )

class DpsPathGroupRemoteRouterSaveMode( DpsPathGroupRemoteRouterBase,
                                        CliSave.Mode ):
   def __init__( self, param ):
      self.routerIp, self.pgName = param
      DpsPathGroupRemoteRouterBase.__init__( self, self.routerIp, self.pgName )
      CliSave.Mode.__init__( self, param )

DpsPathGroupSaveMode.addChildMode( DpsPathGroupRemoteRouterSaveMode )
DpsPathGroupRemoteRouterSaveMode.addCommandSequence(
                                       'Dps.DpsPathGroupRemoteRouter' )

class DpsPolicySaveMode( DpsPolicyModeBase, CliSave.Mode ):
   def __init__( self, param ):
      DpsPolicyModeBase.__init__( self, param )
      CliSave.Mode.__init__( self, param )

RouterPathSelectionSaveMode.addChildMode( DpsPolicySaveMode )
DpsPolicySaveMode.addCommandSequence( 'Dps.DpsPolicy' )

class DpsPolicyRuleKeySaveMode( DpsPolicyRuleKeyBase,
                                        CliSave.Mode ):
   def __init__( self, param ):
      DpsPolicyRuleKeyBase.__init__( self, param[ 0 ], param[ 1 ], param[ 2 ] )
      CliSave.Mode.__init__( self, param )

DpsPolicySaveMode.addChildMode( DpsPolicyRuleKeySaveMode )
DpsPolicyRuleKeySaveMode.addCommandSequence(
                                       'Dps.DpsPolicyRuleKey' )

class DpsPolicyDefaultRuleSaveMode( DpsPolicyDefaultRuleBase,
                                        CliSave.Mode ):
   def __init__( self, param ):
      DpsPolicyDefaultRuleBase.__init__( self, param[ 0 ], param[ 1 ] )
      CliSave.Mode.__init__( self, param )

DpsPolicySaveMode.addChildMode( DpsPolicyDefaultRuleSaveMode )
DpsPolicyDefaultRuleSaveMode.addCommandSequence(
                                       'Dps.DpsPolicyDefaultRule' )

class DpsLoadBalanceProfileConfigSaveMode( DpsLoadBalanceProfileConfigModeBase,
                                          CliSave.Mode ):
   def __init__( self, param ):
      DpsLoadBalanceProfileConfigModeBase.__init__( self, param )
      CliSave.Mode.__init__( self, param )

RouterPathSelectionSaveMode.addChildMode( DpsLoadBalanceProfileConfigSaveMode,
                                          after=[ DpsPathGroupSaveMode ] )
DpsLoadBalanceProfileConfigSaveMode.addCommandSequence(
        'Dps.DpsLoadBalanceProfile' )

class DpsVrfConfigSaveMode( DpsVrfConfigModeBase, CliSave.Mode ):
   def __init__( self, param ):
      DpsVrfConfigModeBase.__init__( self, param )
      CliSave.Mode.__init__( self, param )

RouterPathSelectionSaveMode.addChildMode( DpsVrfConfigSaveMode )
DpsVrfConfigSaveMode.addCommandSequence( 'Dps.DpsVrfConfig' )

def isDpsConfigPresent( entity ):
   return entity.udpPortConfig != DEFAULT_UDP_PORT or \
          len( entity.pathGroupConfig.keys() ) != 0 or \
          len( entity.loadBalanceProfile.keys() ) != 0 or \
          len( entity.vrfConfig.keys() ) != 0 or \
          len( entity.policyConfig.keys() ) != 0

@CliSave.saver( 'Dps::DpsCliConfig', 'dps/input/cli' )
def saveConfig( entity, root, sysdbRoot, options ):
   if not isDpsConfigPresent( entity ):
      return

   dpsMode = root[ RouterPathSelectionSaveMode ].getOrCreateModeInstance(
                                                      'Dps.RouterPathSelection' )

   udpPort = entity.udpPortConfig
   if udpPort != DEFAULT_UDP_PORT:
      encapCmds = dpsMode[ 'Dps.RouterPathSelection' ]
      encapCmds.addCommand( 'encapsulation path-telemetry udp port %d' % udpPort )

   for pgName in entity.pathGroupConfig.keys():
      pgMode = dpsMode[ DpsPathGroupSaveMode ].getOrCreateModeInstance( pgName )
      pgCmds = pgMode[ 'Dps.DpsPathGroup' ]
      pgCfg = entity.pathGroupConfig[ pgName ]
      if pgCfg.ipsecProfile != "":
         pgCmds.addCommand( 'ipsec profile %s' % pgCfg.ipsecProfile )
      for intf in pgCfg.localIntf:
         pgCmds.addCommand( 'local interface %s' % intf )
      for ip in pgCfg.localIp:
         pgCmds.addCommand( 'local ip %s' % ip )
      if pgCfg.remoteDynamic:
         pgCmds.addCommand( 'peer dynamic' )
      if pgCfg.mssEgress:
         pgCmds.addCommand( 'tcp mss ceiling ipv4 %s egress' % pgCfg.mssEgress )
      for viaCfg in pgCfg.remoteViaConfig.keys():
         viaMode = \
            pgMode[ DpsPathGroupRemoteRouterSaveMode ].getOrCreateModeInstance(
                                                         ( viaCfg, pgName ) )
         viaCmds = viaMode[ 'Dps.DpsPathGroupRemoteRouter' ]
         viaEntry = pgCfg.remoteViaConfig[ viaCfg ]
         peerName = viaEntry.peerName
         if peerName and peerName != "" and \
            peerName != viaEntry.remoteAddr.stringValue:
            viaCmds.addCommand( 'name %s' % peerName )
         for via in viaEntry.remoteVia.keys():
            viaCmds.addCommand( 'ipv4 address %s' % via )

   for policy in entity.policyConfig.keys():
      polMode = dpsMode[ DpsPolicySaveMode ].getOrCreateModeInstance( policy )
      polCfg = entity.policyConfig[ policy ]
      for ruleKey in polCfg.appProfilePolicyRuleList.keys():
         ruleKeyEntry = polCfg.appProfilePolicyRuleList[ ruleKey ]
         ruleKeyMode = \
            polMode[ DpsPolicyRuleKeySaveMode ].getOrCreateModeInstance(
                     ( policy, ruleKey, ruleKeyEntry.appProfileName ) )
         ruleKeyCmds = ruleKeyMode[ 'Dps.DpsPolicyRuleKey' ]
         if ruleKeyEntry.lbGrpName != "":
            ruleKeyCmds.addCommand( 'load-balance %s' % ruleKeyEntry.lbGrpName )

      if polCfg.defaultRuleCfgd:
         defaultLbGrp = polCfg.defaultLbGrpName
         defaultRuleMode = \
            polMode[ DpsPolicyDefaultRuleSaveMode ].getOrCreateModeInstance(
                  ( policy, polCfg.defaultRuleCfgd ) )
         defaultRuleCmds = defaultRuleMode[ 'Dps.DpsPolicyDefaultRule' ]
         if defaultLbGrp != "":
            defaultRuleCmds.addCommand( 'load-balance %s' % defaultLbGrp )

   for profileName in entity.loadBalanceProfile:
      profileMode = dpsMode[ DpsLoadBalanceProfileConfigSaveMode ].\
                                        getOrCreateModeInstance( profileName )
      profileCmds = profileMode[ 'Dps.DpsLoadBalanceProfile' ]
      profile = entity.loadBalanceProfile[ profileName ]
      if profile.latency != DEFAULT_LATENCY:
         profileCmds.addCommand( 'latency %d' %
                 ( profile.latency / LATENCY_SCALE ) )
      if profile.jitter != DEFAULT_JITTER:
         profileCmds.addCommand( 'jitter %d' %
                 ( profile.jitter / JITTER_SCALE ) )
      if profile.lossRate != DEFAULT_LOSSRATE:
         loss = profile.lossRate
         loss -= LOSS_RATE_ADJUSTMENT
         profileCmds.addCommand( 'loss-rate %.2f' %
                 ( float( loss ) / LOSS_RATE_SCALE ) )

      for pgName in profile.pathGroupPriority:
         priority = profile.pathGroupPriority[ pgName ]
         cmd = 'path-group %s' % pgName
         if priority != DEFAULT_PRIORITY:
            cmd += ' priority %d' % priority
         profileCmds.addCommand( cmd )

   for vrfName in entity.vrfConfig.keys():
      vrfMode = dpsMode[ DpsVrfConfigSaveMode ].getOrCreateModeInstance( vrfName )
      vrfCmds = vrfMode[ 'Dps.DpsVrfConfig' ]

      vrfCfg = entity.vrfConfig[ vrfName ]

      if vrfCfg.policyName:
         vrfCmds.addCommand( 'path-selection-policy %s' % vrfCfg.policyName )
