#!/usr/bin/env python
# Copyright (c) 2008-2013 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import CliSave, Tracing, Tac
import Toggles.Dot1xToggleLib as Dot1xToggle
from MultiRangeRule import multiRangeToCanonicalString
from IntfCliSave import IntfConfigMode
from Dot1xLib import portControlEnum2Str, hostModeEnum2Str, Dot1xConsts
from CliMode.Dot1x import Dot1xBaseMode
__defaultTraceHandle__ = Tracing.Handle( 'Dot1xCliSave' )

CliSave.GlobalConfigMode.addCommandSequence( 'dot1x.global' )
IntfConfigMode.addCommandSequence( 'dot1x.intf' )

class Dot1xConfigMode( Dot1xBaseMode, CliSave.Mode ):
   def __init__( self, param ):
      Dot1xBaseMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

CliSave.GlobalConfigMode.addChildMode( Dot1xConfigMode )
Dot1xConfigMode.addCommandSequence( 'dot1x.config' )

@CliSave.saver( 'Dot1x::Config', 'dot1x/config' )
def saveDot1xConfig( entity, root, sysdbRoot, options ):
   saveAll = options.saveAll

   cmds = root[ 'dot1x.global' ]
   if entity.dot1xEnabled:
      cmds.addCommand( 'dot1x system-auth-control' )
   elif saveAll:
      cmds.addCommand( 'no dot1x system-auth-control' )
   if entity.lldpBypass:
      cmds.addCommand( 'dot1x protocol lldp bypass' )
   elif saveAll:
      cmds.addCommand( 'no dot1x protocol lldp bypass' )
   if entity.dot1xDynAuthEnabled:
      cmds.addCommand( 'dot1x dynamic-authorization' )
   elif saveAll:
      cmds.addCommand( 'no dot1x dynamic-authorization' )

   cacheResults = False
   cacheResultsForPhone = False
   if Dot1xToggle.toggleDot1xCachedAuthEnabled():
      if entity.aaaTimeoutPhoneApplyCachedResults == 'aaaTimeoutApplyCached':
         cacheResultsForPhone = True
      if entity.aaaTimeoutApplyCachedResults:
         cacheResults = True
   if Dot1xToggle.toggleDot1xDefaultPhoneVlanEnabled() and ( cacheResultsForPhone or
         entity.aaaTimeoutPhoneAllow or saveAll ):
      mode = root[ Dot1xConfigMode ].getOrCreateModeInstance( None )
      cmds = mode[ 'dot1x.config' ]
      if entity.aaaTimeoutPhoneAllow and cacheResultsForPhone:
         cmds.addCommand( 'aaa unresponsive phone action apply cached-results '
                          'else traffic allow' )
      elif entity.aaaTimeoutPhoneAllow:
         cmds.addCommand( 'aaa unresponsive phone action traffic allow' )
      elif cacheResultsForPhone:
         cmds.addCommand( 'aaa unresponsive phone action apply cached-results' )
      else:
         cmds.addCommand( 'no aaa unresponsive phone action' )


   if Dot1xToggle.toggleDot1xAaaUnresponsiveVlanEnabled() and ( cacheResults or
         entity.aaaTimeoutTrafficAllow or saveAll ):
      mode = root[ Dot1xConfigMode ].getOrCreateModeInstance( None )
      cmds = mode[ 'dot1x.config' ]
      if entity.aaaTimeoutTrafficAllow:
         if cacheResults:
            cmdStr = ( 'aaa unresponsive action apply cached-results else '
                       'traffic allow' )
         else:
            cmdStr = 'aaa unresponsive action traffic allow'
         if entity.aaaTimeoutTrafficAllowVlan:
            cmdStr += ' vlan %d' % entity.aaaTimeoutTrafficAllowVlan
         cmds.addCommand( cmdStr )
      elif cacheResults:
         cmds.addCommand( 'aaa unresponsive action apply cached-results' )
      else:
         cmds.addCommand( 'no aaa unresponsive action' )

   if entity.accountingUpdateInterval != entity.defaultAccountingUpdateInterval:
      mode = root[ Dot1xConfigMode ].getOrCreateModeInstance( None )
      cmds = mode[ 'dot1x.config' ]
      cmds.addCommand( 'aaa accounting update interval %d seconds' % \
                       entity.accountingUpdateInterval )
   elif saveAll: 
      mode = root[ Dot1xConfigMode ].getOrCreateModeInstance( None )
      cmds = mode[ 'dot1x.config' ]
      cmds.addCommand( 'no aaa accounting update interval' )

   if entity.defaultServiceType != entity.serviceType:
      mode = root[ Dot1xConfigMode ].getOrCreateModeInstance( None )
      cmds = mode[ 'dot1x.config' ]
      cmds.addCommand( 'radius av-pair service-type' )
   elif saveAll:
      mode = root[ Dot1xConfigMode ].getOrCreateModeInstance( None )
      cmds = mode[ 'dot1x.config' ]
      cmds.addCommand( 'no radius av-pair service-type' )

   if Dot1xToggle.toggleDot1xGuestVlanEnabled() and ( saveAll or entity.guestVlan ):
      mode = root[ Dot1xConfigMode ].getOrCreateModeInstance( None )
      cmds = mode[ 'dot1x.config' ]
      guestVlanBaseCmd = 'eapol unresponsive action traffic allow'
      if entity.guestVlan:
         cmds.addCommand( '%s vlan %d' % ( guestVlanBaseCmd, entity.guestVlan ) )
      elif saveAll:
         cmds.addCommand( 'no ' + guestVlanBaseCmd )

   if entity.defaultFramedMtu != entity.framedMtu:
      mode = root[ Dot1xConfigMode ].getOrCreateModeInstance( None )
      cmds = mode[ 'dot1x.config' ]
      cmds.addCommand( 'radius av-pair framed-mtu %d' % entity.framedMtu )
   elif saveAll:
      mode = root[ Dot1xConfigMode ].getOrCreateModeInstance( None )
      cmds = mode[ 'dot1x.config' ]
      cmds.addCommand( 'no radius av-pair framed-mtu' )

   if entity.mbaUserName != Tac.Value( "Dot1x::MbaUserName" ):
      mode = root[ Dot1xConfigMode ].getOrCreateModeInstance( None )
      cmds = mode[ 'dot1x.config' ]
      msg = 'mac-based-auth radius av-pair user-name delimiter'
      delim = 'none'
      if entity.mbaUserName.userNameDelim == '-':
         delim = 'hyphen'
      elif entity.mbaUserName.userNameDelim == ':':
         delim = 'colon'
      elif entity.mbaUserName.userNameDelim == '.':
         delim = 'period'
      case = 'uppercase' if entity.mbaUserName.userNameIsUpperCase else 'lowercase'
      cmds.addCommand( '%s %s %s' % ( msg, delim, case ) )
   elif saveAll:
      mode = root[ Dot1xConfigMode ].getOrCreateModeInstance( None )
      cmds = mode[ 'dot1x.config' ]
      cmds.addCommand( 'no mac-based-auth radius av-pair user-name' )

   if entity.disconnectTimeout != entity.disconnectTimeoutDefault:
      mode = root[ Dot1xConfigMode ].getOrCreateModeInstance( None )
      cmds = mode[ 'dot1x.config' ]
      cmds.addCommand( 'supplicant disconnect cached-results timeout %d seconds' %
                       entity.disconnectTimeout )
   elif saveAll:
      mode = root[ Dot1xConfigMode ].getOrCreateModeInstance( None )
      cmds = mode[ 'dot1x.config' ]
      cmds.addCommand( 'no supplicant disconnect cached-results timeout' )

   if Dot1xToggle.toggleDot1xWebAuthEnabled(): 
      if entity.captivePortal != Tac.Value( "Dot1x::CaptivePortal" ):
         mode = root[ Dot1xConfigMode ].getOrCreateModeInstance( None )
         cmds = mode[ 'dot1x.config' ]
         cmd = 'captive-portal'
         if entity.captivePortal.url:
            cmd += ' url %s' % entity.captivePortal.url
         cmds.addCommand( cmd )
      elif saveAll:
         mode = root[ Dot1xConfigMode ].getOrCreateModeInstance( None )
         cmds = mode[ 'dot1x.config' ]
         cmds.addCommand( 'no captive-portal' )

   if Dot1xToggle.toggleDot1xAclWebAuthEnabled():
      mode = root[ Dot1xConfigMode ].getOrCreateModeInstance( None )
      cmds = mode[ 'dot1x.config' ]
      if entity.captivePortalIpv4Acl != "":
         cmd = 'captive-portal access-list ipv4 %s' % entity.captivePortalIpv4Acl
         cmds.addCommand( cmd )
      elif saveAll:
         cmds.addCommand( 'no captive-portal access-list ipv4' )

   if Dot1xToggle.toggleDot1xRandomVlanAssignmentEnabled():
      if entity.vlanGroup.keys():
         mode = root[ Dot1xConfigMode ].getOrCreateModeInstance( None )
         cmds = mode[ 'dot1x.config' ]
         for groupName in entity.vlanGroup:
            vlans = multiRangeToCanonicalString( 
                     entity.vlanGroup[ groupName ].vlanId.keys() )
            cmd = 'vlan assignment group ' + groupName + ' members ' + vlans
            cmds.addCommand( cmd )

   if Dot1xToggle.toggleDot1xIpTrackingEnabled():
      if entity.ipTrackingEnabled != False:
         mode = root[ Dot1xConfigMode ].getOrCreateModeInstance( None )
         cmds = mode[ 'dot1x.config' ]
         cmd = 'address tracking ipv4'
         cmds.addCommand( cmd )
      elif saveAll:
         mode = root[ Dot1xConfigMode ].getOrCreateModeInstance( None )
         cmds = mode[ 'dot1x.config' ]
         cmds.addCommand( 'no address tracking ipv4' )

   for intfName, dot1xIntfConfig in entity.dot1xIntfConfig.iteritems():
      mode = root[ IntfConfigMode ].getOrCreateModeInstance( intfName )
      cmds = mode[ 'dot1x.intf' ]

      if dot1xIntfConfig.dot1xEnabled:
         cmds.addCommand( 'dot1x pae authenticator' )
      elif saveAll:
         cmds.addCommand( 'no dot1x pae authenticator' )

      if dot1xIntfConfig.authFailVlan:
         cmds.addCommand( 'dot1x authentication failure action traffic allow vlan %d'
                          % dot1xIntfConfig.authFailVlan )
      elif dot1xIntfConfig.afDropConfigured:
         cmds.addCommand( 'dot1x authentication failure action traffic drop' )
      elif saveAll:
         cmds.addCommand(
            'no dot1x authentication failure action traffic allow vlan' )

      if dot1xIntfConfig.reauth:
         cmds.addCommand( 'dot1x reauthentication' )
      elif options.saveAll:
         cmds.addCommand( 'no dot1x reauthentication' )

      if dot1xIntfConfig.portCtrlSetting != 'forceAuth' or saveAll:
         cmds.addCommand( 'dot1x port-control %s' 
               % portControlEnum2Str[ dot1xIntfConfig.portCtrlSetting ] )
      
      if dot1xIntfConfig.hostMode != 'multiHost' or saveAll:
         cmds.addCommand( 'dot1x host-mode %s'
               % hostModeEnum2Str[ dot1xIntfConfig.hostMode ] )

      if dot1xIntfConfig.eapolDisabled:
         cmds.addCommand( 'dot1x eapol disabled' )
      elif saveAll:
         cmds.addCommand( 'no dot1x eapol disabled' )
      
      if dot1xIntfConfig.mbaHostMode:
         cmds.addCommand( 'dot1x mac based authentication host-mode common' )
      elif dot1xIntfConfig.mbaEnabled:
         cmds.addCommand( 'dot1x mac based authentication' )
      elif saveAll:
         cmds.addCommand( 'no dot1x mac based authentication' )

      if dot1xIntfConfig.quietPeriod != \
            dot1xIntfConfig.defaultQuietPeriod or saveAll:
         cmds.addCommand( 'dot1x timeout quiet-period %d' 
                                 % dot1xIntfConfig.quietPeriod )
     
      if dot1xIntfConfig.reauthTimeoutIgnore != False:
         cmds.addCommand( 'dot1x timeout reauth-timeout-ignore always' )

      if dot1xIntfConfig.txPeriod != dot1xIntfConfig.defaultTxPeriod or saveAll:
         cmds.addCommand( 'dot1x timeout tx-period %d' 
                                 % dot1xIntfConfig.txPeriod )
      if dot1xIntfConfig.reauthOpts == 'useSetTimeout':
         if dot1xIntfConfig.reauthPeriod != dot1xIntfConfig.defaultReauthPeriod \
            or saveAll:
            cmds.addCommand( 'dot1x timeout reauth-period %d' 
                                 % dot1xIntfConfig.reauthPeriod )
      else:
         cmds.addCommand( 'dot1x timeout reauth-period server' )

      if dot1xIntfConfig.maxReauthReq != dot1xIntfConfig.defaultMaxReauthReq:
         cmds.addCommand( 'dot1x reauthorization request limit %d'
                          % dot1xIntfConfig.maxReauthReq )
      elif saveAll:
         # the default case should always have the new syntax
         cmds.addCommand( 'dot1x reauthorization request limit %d'
                          % dot1xIntfConfig.maxReauthReq )

      if dot1xIntfConfig.unauthorizedAccessVlanEgress:
         cmds.addCommand( 'dot1x unauthorized access vlan membership egress' )
      elif saveAll:
         cmds.addCommand( 'no dot1x unauthorized access vlan membership egress' )

      if dot1xIntfConfig.unauthorizedNativeVlanEgress:
         cmds.addCommand( 'dot1x unauthorized native vlan membership egress' )
      elif saveAll:
         cmds.addCommand( 'no dot1x unauthorized native vlan membership egress' )

      if dot1xIntfConfig.mbaFallback:
         if dot1xIntfConfig.mbaTimeout != Dot1xConsts.mbaTimeoutDefault:
            cmds.addCommand( 'dot1x eapol authentication failure fallback mba'\
                  ' timeout %d' % dot1xIntfConfig.mbaTimeout )
         else:
            cmds.addCommand( 'dot1x eapol authentication failure fallback mba' )
      elif saveAll:
         cmds.addCommand( 'no dot1x eapol authentication failure fallback mba' )

      # pylint: disable-msg=W0105
      '''
      we don't support guest vlan restricted now
      if dot1xIntfConfig.guestVlan != 0:
         cmds.addCommand( 'dot1x guest-vlan %d' % dot1xIntfConfig.guestVlan )
      elif saveAll:
         cmds.addCommand( 'no dot1x guest-vlan' )

      if dot1xIntfConfig.authFailVlan != 0:
         cmds.addCommand( 'dot1x auth-fail vlan %d' % dot1xIntfConfig.authFailVlan )
      elif saveAll:
         cmds.addCommand( 'no dot1x auth-fail vlan' )

      if dot1xIntfConfig.authFailMaxAttempt != \
            dot1xIntfConfig.defaultAuthFailMaxAttempt or saveAll:
         cmds.addCommand( 'dot1x auth-fail max-attempts %d' 
                              % dot1xIntfConfig.authFailMaxAttempt )
      '''
