#!/usr/bin/env python
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function
import CliSave
# Need to import this to get the DhcpRelay.dhcp config sequence
import CliSavePlugin.DhcpRelayHelperCliSave # pylint: disable=unused-import
from CliSavePlugin.IntfCliSave import IntfConfigMode
# Need to import this to get the Ira.ipIntf config sequence (which has the dhcp
# client command) BUG368914
import CliSavePlugin.IraCliSave # pylint: disable=unused-import
from RoutingIntfUtils import allRoutingProtocolIntfNames
from EosDhcpServerLib import convertLeaseSeconds
from EosDhcpServerLib import featureOption43Status
from EosDhcpServerLib import getSubOptionData
from EosDhcpServerLib import subOptionCmd
from EosDhcpServerLib import featureStaticIPv4AddressStatus
from CliMode.DhcpServer import DhcpServerBaseMode, DhcpServerSubnetBaseMode
from CliMode.DhcpServer import DhcpServerVendorOptionBaseMode
from CliMode.DhcpServer import DhcpServerReservationsBaseMode
from CliMode.DhcpServer import DhcpServerReservationsMacAddressBaseMode
import Tac

ipAddr = Tac.Value( 'Arnet::IpAddr' )

# dhcp relay mode
class DhcpServerCliSaveMode( DhcpServerBaseMode, CliSave.Mode ):
   def __init__( self, param ):
      DhcpServerBaseMode.__init__( self )
      CliSave.Mode.__init__( self, param )

# subnet mode
class DhcpServerSubnetCliSaveMode( DhcpServerSubnetBaseMode, CliSave.Mode ):
   def __init__( self, param ):
      DhcpServerSubnetBaseMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

   def commentKey( self ):
      return "dhcp-subnet-%s" % self.subnet

class DhcpServerSubnetCliSaveV4Mode( DhcpServerSubnetCliSaveMode ):
   def __init__( self, param ):
      param = ( param, 'ipv4' )
      DhcpServerSubnetCliSaveMode.__init__( self, param )

class DhcpServerSubnetCliSaveV6Mode( DhcpServerSubnetCliSaveMode ):
   def __init__( self, param ):
      param = ( param, 'ipv6' )
      DhcpServerSubnetCliSaveMode.__init__( self, param )

# vendor-option mode
class DhcpServerVendorOptionCliSaveMode( DhcpServerVendorOptionBaseMode,
                                         CliSave.Mode ):
   def __init__( self, param ):
      DhcpServerVendorOptionBaseMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

   def commentKey( self ):
      return "dhcp-vendor-%s %s" % ( self.af, self.vendorId )

class DhcpServerVendorOptionCliSaveV4Mode( DhcpServerVendorOptionCliSaveMode ):

   def __init__( self, param ):
      # modeCmd is 'ipv4 vendorId' therefore we need to flip it
      param = tuple( reversed( param ) )
      DhcpServerVendorOptionCliSaveMode.__init__( self, param )

# reservations mode
class DhcpServerReservationsCliSaveMode( DhcpServerReservationsBaseMode,
                                         CliSave.Mode ):
   def __init__( self, param ):
      DhcpServerReservationsBaseMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

   def commentKey( self ):
      return self.longModeKey

   def skipIfEmpty( self ):
      return True

class DhcpServerReservationsCliSaveV4Mode( DhcpServerReservationsCliSaveMode ):

   def __init__( self, param ):
      # getOrCreateModeInstance requires param, and it will then initialize
      # this class with param=('ipv4'). For that reason we need param in __init__
      DhcpServerReservationsCliSaveMode.__init__( self, param )

# reservations mac-address mode
class DhcpServerReservationsMacAddressCliSaveMode(
                            DhcpServerReservationsMacAddressBaseMode,
                            CliSave.Mode ):
   def __init__( self, param ):
      DhcpServerReservationsMacAddressBaseMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

   def commentKey( self ):
      return "dhcp-mac-address-%s" % self.macAddr

class DhcpServerReservationsMacAddressCliSaveV4Mode(
                            DhcpServerReservationsMacAddressCliSaveMode ):
   def __init__( self, param ):
      param = ( param, 'ipv4' )
      DhcpServerReservationsMacAddressCliSaveMode.__init__( self, param )

CliSave.GlobalConfigMode.addChildMode( DhcpServerCliSaveMode,
                                       before=[ IntfConfigMode ],
                                       after=[ 'DhcpRelay.dhcp' ] )

DhcpServerCliSaveMode.addCommandSequence( 'DhcpServer.config' )

DhcpServerCliSaveMode.addChildMode( DhcpServerSubnetCliSaveV4Mode )
DhcpServerCliSaveMode.addChildMode( DhcpServerSubnetCliSaveV6Mode )
DhcpServerSubnetCliSaveV4Mode.addCommandSequence( 'DhcpServer.subnetV4' )
DhcpServerSubnetCliSaveV6Mode.addCommandSequence( 'DhcpServer.subnetV6' )
if featureOption43Status():
   DhcpServerCliSaveMode.addChildMode( DhcpServerVendorOptionCliSaveV4Mode )
   DhcpServerVendorOptionCliSaveV4Mode.addCommandSequence(
                                       'DhcpServer.vendorOptionIpv4' )

if featureStaticIPv4AddressStatus():
   DhcpServerSubnetCliSaveV4Mode.addChildMode( DhcpServerReservationsCliSaveV4Mode,
                                               before=[ 'DhcpServer.subnetV4' ] )
   DhcpServerReservationsCliSaveV4Mode. \
         addCommandSequence( 'DhcpServer.reservationsV4' )
   DhcpServerReservationsCliSaveV4Mode. \
         addChildMode( DhcpServerReservationsMacAddressCliSaveV4Mode )
   DhcpServerReservationsMacAddressCliSaveV4Mode.\
         addCommandSequence( 'DhcpServer.reservationsMacAddrV4' )

IntfConfigMode.addCommandSequence( 'DhcpServer.intf', after=[ 'Ira.ipIntf' ] )

def saveReservationsMacAddr( entity, root, saveMode, macAddr, cmdKey, saveAll ):
   mode = root[ saveMode ].getOrCreateModeInstance( macAddr.displayString )
   cmds = mode[ cmdKey ]

   if entity.ipAddr != ipAddr.ipAddrZero:
      cmds.addCommand( "ipv4-address {}".format( entity.ipAddr ) )
   elif saveAll:
      cmds.addCommand( "no ipv4-address" )

def saveVendorOption( entity, root, saveMode, vendorId, cmdKey, saveAll, af ):
   # vendor-option <modeCmd>, this is to enter the mode
   modeCmd = ( af, vendorId )
   mode = root[ saveMode ].getOrCreateModeInstance( modeCmd )
   cmds = mode[ cmdKey ]

   # Similar to subnetRange (entity.ranges)
   for subOption in entity.subOptionConfig.values():
      optionData = getSubOptionData( subOption, raw=True )
      cmd = subOptionCmd( subOption.code, subOption.type,
                           optionData, subOption.isArray )
      cmds.addCommand( cmd )

def saveIpv4SubnetConfig( entity, cmds, saveAll, root ):
   if entity.defaultGateway != ipAddr.ipAddrZero:
      cmds.addCommand( "default-gateway {}".format( entity.defaultGateway ) )
   elif saveAll:
      cmds.addCommand( "no default-gateway" )

   if entity.tftpServerOption66 != entity.tftpServerDefault:
      cmds.addCommand( 'tftp server option 66 {}'.format(
         entity.tftpServerOption66 ) )
   elif saveAll:
      cmds.addCommand( 'no tftp server option 66' )

   if entity.tftpServerOption150.values():
      cmds.addCommand( 'tftp server option 150 {}'.format(
         ' '.join( entity.tftpServerOption150.values() ) ) )
   elif saveAll:
      cmds.addCommand( 'no tftp server option 150' )

   if entity.tftpBootFileName != entity.tftpBootFileNameDefault:
      cmds.addCommand( "tftp server file {}".format(
         entity.tftpBootFileName ) )
   elif saveAll:
      cmds.addCommand( "no tftp server file" )

   # reservations mac-address
   if featureStaticIPv4AddressStatus():
      # go to reservations mode
      mode = root[ DhcpServerReservationsCliSaveV4Mode ].\
                                         getOrCreateModeInstance( ( 'ipv4' ) )
      for macAddr in entity.reservationsMacAddr:
         reservationEntity = entity.reservationsMacAddr[ macAddr ]
         macAddr = reservationEntity.getRawAttribute( "hostId" )
         saveReservationsMacAddr( reservationEntity,
                                  mode,
                                  DhcpServerReservationsMacAddressCliSaveV4Mode,
                                  macAddr, 'DhcpServer.reservationsMacAddrV4',
                                  saveAll )

def saveSubnetConfig( entity, root, saveMode, subnet, cmdKey, saveAll, af ):
   mode = root[ saveMode ].getOrCreateModeInstance( subnet )
   cmds = mode[ cmdKey ]

   if entity.subnetName:
      cmds.addCommand( "name {}".format( entity.subnetName ) )
   elif saveAll:
      cmds.addCommand( "no name" )

   if entity.dnsServers:
      servers = list( entity.dnsServers.values() )
      cmds.addCommand( "dns server {}".format( " ".join( map( str, servers ) ) ) )
   elif saveAll:
      cmds.addCommand( "no dns server" )

   if entity.leaseTime:
      days, hours, minutes = convertLeaseSeconds( entity.leaseTime )
      leaseTimeCmd = "lease time {} days {} hours {} minutes".format(
            days, hours, minutes )
      cmds.addCommand( leaseTimeCmd )
   elif saveAll:
      cmds.addCommand( "no lease time" )

   if af == 'ipv4':
      saveIpv4SubnetConfig( entity, cmds, saveAll, mode )

   # Since the range is ordered, we don't need to wrap this in sorted
   for subnetRange in entity.ranges:
      cmds.addCommand( "range {} {}".format( subnetRange.start,
                                             subnetRange.end ) )

def saveInterfaceConfig( entity, root, sysdbRoot, options, requireMounts ):
   if options.saveAllDetail:
      cfgIntfNames = allRoutingProtocolIntfNames( sysdbRoot, includeEligible=True,
                                                  requireMounts=requireMounts )
   elif options.saveAll:
      cfgIntfNames = set(
            allRoutingProtocolIntfNames( sysdbRoot, requireMounts=requireMounts ) +
            entity.interfacesIpv4.keys() + entity.interfacesIpv6.keys() )
   else:
      cfgIntfNames = set( entity.interfacesIpv4.keys() +
                          entity.interfacesIpv6.keys() )

   for intfId in cfgIntfNames:
      intfMode = root[ IntfConfigMode ].getOrCreateModeInstance( intfId )
      cmds = intfMode[ 'DhcpServer.intf' ]
      if intfId in entity.interfacesIpv4:
         cmds.addCommand( "dhcp server ipv4" )
      elif options.saveAll:
         cmds.addCommand( "no dhcp server ipv4" )
      if intfId in entity.interfacesIpv6:
         cmds.addCommand( "dhcp server ipv6" )
      elif options.saveAll:
         cmds.addCommand( "no dhcp server ipv6" )

@CliSave.saver( 'DhcpServer::Config', 'dhcpServer/config',
                requireMounts=( 'interface/config/all', 'interface/status/all',
                                'l3/intf/config' ) )
def saveDhcpServerConfig( entity, root, sysdbRoot, options, requireMounts ):
   saveInterfaceConfig( entity, root, sysdbRoot, options, requireMounts )

   if entity.dhcpServerMode == entity.dhcpServerModeDefault:
      return

   mode = root[ DhcpServerCliSaveMode ].getSingletonInstance()
   cmds = mode[ 'DhcpServer.config' ]

   if entity.disabled != entity.disabledDefault:
      cmds.addCommand( "disabled" )
   elif options.saveAll:
      cmds.addCommand( "no disabled" )

   days, hours, minutes = convertLeaseSeconds( entity.leaseTimeIpv4 )
   leaseTimeCmd = "lease time ipv4 {} days {} hours {} minutes".format(
      days, hours, minutes )
   if ( entity.leaseTimeIpv4 != entity.leaseTimeIpv4Default ) or options.saveAll:
      cmds.addCommand( leaseTimeCmd )

   if entity.domainNameIpv4 != "":
      cmds.addCommand( "dns domain name ipv4 {}".format( entity.domainNameIpv4 ) )
   elif options.saveAll:
      cmds.addCommand( "no dns domain name ipv4" )

   if entity.dnsServersIpv4:
      servers = list( entity.dnsServersIpv4.values() )
      cmds.addCommand( "dns server ipv4 {}".format( " ".join( servers ) ) )
   elif options.saveAll:
      cmds.addCommand( "no dns server ipv4" )

   days, hours, minutes = convertLeaseSeconds( entity.leaseTimeIpv6 )
   leaseTimeCmd = "lease time ipv6 {} days {} hours {} minutes".format(
      days, hours, minutes )
   if ( entity.leaseTimeIpv6 != entity.leaseTimeIpv6Default ) or options.saveAll:
      cmds.addCommand( leaseTimeCmd )

   if entity.domainNameIpv6 != "":
      cmds.addCommand( "dns domain name ipv6 {}".format( entity.domainNameIpv6 ) )
   elif options.saveAll:
      cmds.addCommand( "no dns domain name ipv6" )

   if entity.dnsServersIpv6:
      servers = list( entity.dnsServersIpv6.values() )
      cmds.addCommand( "dns server ipv6 {}".format(
         " ".join( map( str, servers ) ) ) )
   elif options.saveAll:
      cmds.addCommand( "no dns server ipv6" )

   if entity.tftpServerOption66Ipv4 != entity.tftpServerDefault:
      cmds.addCommand( "tftp server option 66 ipv4 {}".format(
         entity.tftpServerOption66Ipv4 ) )
   elif options.saveAll:
      cmds.addCommand( "no tftp server option 66 ipv4" )

   if entity.tftpServerOption150Ipv4.values():
      cmds.addCommand( "tftp server option 150 ipv4 {}".format(
         ' '.join( entity.tftpServerOption150Ipv4.values() ) ) )
   elif options.saveAll:
      cmds.addCommand( "no tftp server option 150 ipv4" )

   if entity.tftpBootFileNameIpv4 != entity.tftpBootFileNameDefault:
      cmds.addCommand( "tftp server file ipv4 {}".format(
         entity.tftpBootFileNameIpv4 ) )
   elif options.saveAll:
      cmds.addCommand( "no tftp server file ipv4" )

   # Since the subnetConfigIpv4 is ordered, we don't need to wrap this in sorted
   for subnet in entity.subnetConfigIpv4:
      subnetEntity = entity.subnetConfigIpv4[ subnet ]
      saveSubnetConfig( subnetEntity, mode, DhcpServerSubnetCliSaveV4Mode,
                        subnet, 'DhcpServer.subnetV4', options.saveAll, 'ipv4' )

   for subnet in entity.subnetConfigIpv6:
      subnetEntity = entity.subnetConfigIpv6[ subnet ]
      saveSubnetConfig( subnetEntity, mode, DhcpServerSubnetCliSaveV6Mode,
                        subnet, 'DhcpServer.subnetV6', options.saveAll, 'ipv6' )

   # vendor-option
   if featureOption43Status():
      for vendorId in entity.vendorOptionIpv4:
         vendorEntity = entity.vendorOptionIpv4[ vendorId ]
         saveVendorOption( vendorEntity, mode, DhcpServerVendorOptionCliSaveV4Mode,
                           vendorId, 'DhcpServer.vendorOptionIpv4', options.saveAll,
                           'ipv4' )

   if entity.debugLogPath != entity.debugLogPathDefault:
      cmds.addCommand( "debug log {}".format( entity.debugLogUrl ) )
   elif options.saveAll:
      cmds.addCommand( "no debug log" )
