# Copyright (c) 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.


from CliMode.DhcpRelay import DhcpRelayGlobalBaseMode
import CliSave
import DhcpRelayVrfCliLib
import IpUtils
from IpLibConsts import DEFAULT_VRF
from IntfCliSave import IntfConfigMode
from RoutingIntfUtils import allRoutingProtocolIntfNames
import Tac
import Toggles.DhcpRelayToggleLib as DhcpRelayToggleLib

class DhcpRelayGlobalCliSaveMode( DhcpRelayGlobalBaseMode, CliSave.Mode ):
   def __init__( self, param ):
      DhcpRelayGlobalBaseMode.__init__( self )
      CliSave.Mode.__init__( self, param )

if DhcpRelayToggleLib.toggleDhcpRelayGlobalDhcpRelayModeEnabled():
   CliSave.GlobalConfigMode.addChildMode( DhcpRelayGlobalCliSaveMode,
                                          before=[ 'DhcpRelay.dhcp' ] )
   DhcpRelayGlobalCliSaveMode.addCommandSequence( 'DhcpRelay.globalMode'  )

def saveDhcpRelayGlobalModeConfig( entity, root, saveMode, cmdKey, saveAll,
                                   requireMounts ):
   # don't save anything if the user has not entered the mode
   if not entity.dhcpRelayGlobalMode:
      return

   mode = root[ saveMode ].getSingletonInstance()
   cmds = mode[ cmdKey ]

   entity = entity.dhcpRelayGlobalModeConfig
   if DhcpRelayToggleLib.toggleDhcpRelayTunnelReqDisableEnabled():
      if entity:
         if entity.tunnelReqDisable:
            cmds.addCommand( "tunnel requests disabled" )
      elif saveAll:
         cmds.addCommand( "no tunnel requests disabled" )

   if not entity:
      return

   if DhcpRelayToggleLib.toggleDhcpRelayGlobalModeServerCommandEnabled():
      # server ( IP_GENADDR | HOSTNAME )
      # ipv4
      for key in sorted( entity.serverIpGlobal, 
                         cmp=DhcpRelayVrfCliLib.compareIpAddrVrf ):
         cmds.addCommand( "server {}".format( key.ip ) )

      # hostname
      for key in sorted( entity.serverHostnameGlobal,
                         cmp=DhcpRelayVrfCliLib.compareHostnameVrf ):
         cmds.addCommand( "server {}".format( key.hostname ) )
      
   if DhcpRelayToggleLib.toggleDhcpRelayGlobalModeServerCommandIPv6Enabled():
      # ipv6
      for key in sorted( entity.serverIp6Global,
                         cmp=DhcpRelayVrfCliLib.compareIpv6AddrVrf ):
         cmds.addCommand( "server {}".format( key.ip ) )

IntfConfigMode.addCommandSequence( 'DhcpRelay.intf', after=[ 'Ira.ipIntf' ] )
CliSave.GlobalConfigMode.addCommandSequence( 'DhcpRelay.dhcp',
                                             before=[ IntfConfigMode ] )

@CliSave.saver( 'Ip::Helper::DhcpRelay::Config', 'ip/helper/dhcprelay/config',
                requireMounts = ( 'ip/dhcp/relay/config', 'ip/config',
                                  'interface/config/all', 'interface/status/all',
                                  'ip/vrf/config', 'hwEpoch/status', 
                                  'acl/cpconfig/cli' ) )
def saveDhcpRelayHelperConfig( entity, root, sysdbRoot, options,
                               requireMounts ):
   # do nothing if forwarder is configured
   dhcpFwdConfig = requireMounts[ 'ip/dhcp/relay/config' ]
   if dhcpFwdConfig.serverIp or dhcpFwdConfig.serverIntf or \
          dhcpFwdConfig.logVerbose or dhcpFwdConfig.intfConfig:
      return
   
   # dhcp relay mode
   if DhcpRelayToggleLib.toggleDhcpRelayGlobalDhcpRelayModeEnabled():
      saveDhcpRelayGlobalModeConfig( entity, root, DhcpRelayGlobalCliSaveMode,
                                     'DhcpRelay.globalMode', options.saveAll,
                                     requireMounts )

   cmds = root[ 'DhcpRelay.dhcp' ]
   if entity.circuitIdOptOn:
      cmds.addCommand( 'ip dhcp relay information option' )
   elif options.saveAll:
      cmds.addCommand( 'no ip dhcp relay information option' )
   if entity.alwaysOn:
      cmds.addCommand( 'ip dhcp relay always-on' )
   elif options.saveAll:
      cmds.addCommand( 'no ip dhcp relay always-on' )
   if entity.smartRelayGlobal:
      cmds.addCommand( 'ip dhcp relay all-subnets default' )
   elif options.saveAll:
      cmds.addCommand( 'no ip dhcp relay all-subnets default' )
   if entity.allSubnetsV6Global:
      cmds.addCommand( 'ipv6 dhcp relay all-subnets default' )
   elif options.saveAll:
      cmds.addCommand( 'no ipv6 dhcp relay all-subnets default' )
   if entity.dscpV4 != entity.dscpValueDefault:
      cmds.addCommand( 'ip dhcp relay qos dscp %s' % entity.dscpV4 )
   elif options.saveAll:
      cmds.addCommand( 'no ip dhcp relay qos dscp' )
   if entity.dscpV6 != entity.dscpValueDefault:
      cmds.addCommand( 'ipv6 dhcp relay qos dscp %s' % entity.dscpV6 )
   elif options.saveAll:
      cmds.addCommand( 'no ipv6 dhcp relay qos dscp' )
   if entity.clientLinkLayerAddrOptOn:
      cmds.addCommand( 'ipv6 dhcp relay option link-layer address' )
   elif options.saveAll:
      cmds.addCommand( 'no ipv6 dhcp relay option link-layer address' )
   if entity.remoteIdEncodingFormat != entity.remoteIdEncodingFormatDefault or \
         options.saveAll:
      cmds.addCommand( 'ipv6 dhcp relay option remote-id format %s' % \
            entity.remoteIdEncodingFormat )

   # Interface config
   if options.saveAllDetail:
      # This applies to all interfaces that potentially have layer 3 capability,
      # except loopback and management interfaces. These happen to be the 'routing
      # protocol' interfaces.
      cfgIntfNames = allRoutingProtocolIntfNames( sysdbRoot, includeEligible=True,
                                                  requireMounts=requireMounts )
   elif options.saveAll:
      # We allow IP configuration on switchport interfaces as well. Therefore, intfs
      # for config are all IP intfs and switchports with non-default config. 
      cfgIntfNames = set(
            allRoutingProtocolIntfNames( sysdbRoot, requireMounts=requireMounts ) +
            entity.intfConfig.keys() )
   else:
      cfgIntfNames = entity.intfConfig

   for intfName in cfgIntfNames:
      intfConfig = entity.intfConfig.get( intfName )
      if not intfConfig:
         if options.saveAll:
            intfConfig = Tac.newInstance( 'Ip::Helper::DhcpRelay::IntfConfig',
                  intfName )
         else:
            continue
      saveDhcpRelayHelperIntf( intfConfig, root, sysdbRoot, options, requireMounts )

   # save for service acl
   aclCpConfig = requireMounts[ 'acl/cpconfig/cli' ]
   allVrfConfig = requireMounts[ "ip/vrf/config" ]
   vrfNames = allVrfConfig.vrf.members()
   vrfNames.append( DEFAULT_VRF )
 
   for aclType in ( 'ip', 'ipv6' ):
      serviceAclConfig = aclCpConfig.cpConfig[ aclType ].serviceAcl
      for vrf in vrfNames:
         vrfCmd =  '' if vrf == DEFAULT_VRF else ' vrf %s' % vrf
         if ( vrf in serviceAclConfig and 
              'dhcpRelay' in serviceAclConfig[ vrf ].service and 
              serviceAclConfig[ vrf ].service[ 'dhcpRelay' ].aclName ):
            cmds.addCommand( '%s dhcp relay access-group %s%s' % ( aclType,
                             serviceAclConfig[ vrf ].service[ 'dhcpRelay' ].aclName,
                             vrfCmd ) )
         elif options.saveAll:
            cmds.addCommand( 'no %s dhcp relay access-group%s' % 
                             ( aclType, vrfCmd ) )

def saveDhcpRelayHelperIntf( entity, root, sysdbRoot, options, requireMounts ):
   mode = root[ IntfConfigMode ].getOrCreateModeInstance( entity.name )
   cmds = mode[ 'DhcpRelay.intf' ]

   if entity.smartRelay == 'srOn':
      cmds.addCommand( 'ip dhcp relay all-subnets' )
   elif entity.smartRelay == 'srOff':
      cmds.addCommand( 'no ip dhcp relay all-subnets' )
   if options.saveAll and entity.smartRelay == 'srDefault':
      cmds.addCommand( 'default ip dhcp relay all-subnets' )

   if entity.allSubnetsV6 == 'srOn':
      cmds.addCommand( 'ipv6 dhcp relay all-subnets' )
   elif entity.allSubnetsV6 == 'srOff':
      cmds.addCommand( 'no ipv6 dhcp relay all-subnets' )
   if options.saveAll and entity.allSubnetsV6 == 'srDefault':
      cmds.addCommand( 'default ipv6 dhcp relay all-subnets' )

   ipConfig = requireMounts[ 'ip/config' ]
   ipIntfConfig = ipConfig.ipIntfConfig.get( entity.name )
   vrfName = DEFAULT_VRF if ipIntfConfig is None or ipIntfConfig.vrf == "" \
       else ipIntfConfig.vrf
   for i in sorted( entity.serverIp, cmp=DhcpRelayVrfCliLib.compareIpAddrVrf ):
      optKeywords = ""
      sourceIntf = i.srcIntf
      sourceIp = i.srcIp
      if i.vrfName != vrfName:
         optKeywords = optKeywords + " vrf %s" % ( i.vrfName )
      if sourceIntf != "":
         optKeywords = optKeywords + " source-interface %s" % ( sourceIntf )
      elif sourceIp.stringValue != '0.0.0.0':
         optKeywords = optKeywords + " source-address %s" % ( sourceIp )
      cmds.addCommand( 'ip helper-address %s%s' % ( i.ip, optKeywords ) )

   for i in sorted( entity.serverIp6, cmp=DhcpRelayVrfCliLib.compareIpv6AddrVrf ):
      optKeyword = ""
      sourceIp = i.srcIp
      sourceIntf = i.srcIntf
      linkAddr = entity.serverIp6[ i ]
      if i.vrfName != vrfName:
         optKeyword = optKeyword + " vrf %s" % ( i.vrfName )
      if sourceIp.stringValue != '::':
         optKeyword = optKeyword + " source-address %s" % ( sourceIp )
      if sourceIntf != "":
         optKeyword = optKeyword + " local-interface %s" % ( sourceIntf )
      if linkAddr.stringValue != "::":
         optKeyword = optKeyword + " link-address %s" % ( linkAddr )
      cmds.addCommand( 'ipv6 dhcp relay destination %s%s' % ( i.ip, optKeyword ) )

   for i in sorted( entity.ccapCores, cmp=IpUtils.compareIp6Address ):
      cmds.addCommand( 'ipv6 dhcp relay add vendor-option ccap-core %s' %
                       i.stringValue )

   for i in sorted( entity.serverHostname, \
                       cmp=DhcpRelayVrfCliLib.compareHostnameVrf ):
      optKeywords = ""
      sourceIntf = i.srcIntf
      sourceIp = i.srcIp

      if i.vrfName != vrfName:
         optKeywords = optKeywords + " vrf %s" % ( i.vrfName )
      if sourceIntf != "":
         optKeywords = optKeywords + " source-interface %s" % ( sourceIntf )
      elif sourceIp.stringValue != "0.0.0.0":
         optKeywords = optKeywords + " source-address %s" % ( sourceIp )
      cmds.addCommand( 'ip helper-address %s%s' % ( i.hostname, optKeywords ) )
   
   if options.saveAll and not entity.serverIp and not entity.serverHostname:
      cmds.addCommand( 'no ip helper-address' )

   if options.saveAll and not entity.serverIp6:
      cmds.addCommand( 'no ipv6 dhcp relay destination' )

   if options.saveAll and not entity.ccapCores:
      cmds.addCommand( 'no ipv6 dhcp relay add vendor-option ccap-core' )

   if entity.route6Install:
      cmds.addCommand( 'ipv6 dhcp relay install routes' )
   elif options.saveAll and not entity.route6Install:
      cmds.addCommand( 'no ipv6 dhcp relay install routes' )

   if entity.circuitId != entity.name:
      cmds.addCommand( 'ip dhcp relay information option circuit-id %s' 
                       % entity.circuitId )
   elif options.saveAll:
      if len( entity.name ) <= 15:
         cmds.addCommand( 'ip dhcp relay information option circuit-id %s'
                          % entity.name )
   if DhcpRelayToggleLib.toggleDhcpRelayGlobalModeServerCommandEnabled(): 
      if entity.disabledV4:
         cmds.addCommand( 'dhcp relay ipv4 disabled' )
      elif options.saveAll:
         cmds.addCommand( 'no dhcp relay ipv4 disabled' )
