# Copyright (c) 2017 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import CliSave
import Tac
from CliMode.ConnectivityMonitor import ConnectivityMonitorBaseMode, \
   ConnectivityMonitorHostMode, ConnectivityMonitorVrfMode
from IntfCliSave import IntfConfigMode

defaultVrf = "default"

class ConnMonitorConfigMode( ConnectivityMonitorBaseMode, CliSave.Mode ):
   def __init__( self, param ):
      ConnectivityMonitorBaseMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

ConnMonitorConfigMode.addCommandSequence( 'ConnectivityMonitor.config' )
ConnMonitorConfigMode.addCommandSequence( 'ConnectivityMonitor.config.intfs' )
CliSave.GlobalConfigMode.addChildMode( ConnMonitorConfigMode,
                                       after=[ IntfConfigMode ] )

class ConnMonitorHostConfigMode( ConnectivityMonitorHostMode, CliSave.Mode ):
   def __init__( self, param ):
      ConnectivityMonitorHostMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

ConnMonitorHostConfigMode.addCommandSequence( 'ConnectivityMonitor.config.host' )
ConnMonitorHostConfigMode.addCommandSequence(
                                          'ConnectivityMonitor.config.vrf.host' )
ConnMonitorConfigMode.addChildMode( ConnMonitorHostConfigMode,
                                    after=[ 'ConnectivityMonitor.config.intfs' ] )

class ConnMonitorVrfConfigMode( ConnectivityMonitorVrfMode, CliSave.Mode ):
   def __init__( self, param ):
      ConnectivityMonitorVrfMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

ConnMonitorVrfConfigMode.addCommandSequence( 'ConnectivityMonitor.config.vrf' )
ConnMonitorVrfConfigMode.addCommandSequence( 'ConnectivityMonitor.config.vrf.intfs' )
ConnMonitorVrfConfigMode.addChildMode( ConnMonitorHostConfigMode,
                                 after=[ 'ConnectivityMonitor.config.vrf.intfs' ] )
# Add ConnMonitorVrfConfigMode as a childMode to ConnMonitorConfigMode
ConnMonitorConfigMode.addChildMode( ConnMonitorVrfConfigMode )

def saveHostConfig( hostConfig, root, config ):
   parentMode = root[ ConnMonitorConfigMode ].getOrCreateModeInstance( None )
   vrfName = hostConfig.key.vrfName
   if vrfName != defaultVrf:
      parentMode = parentMode[ ConnMonitorVrfConfigMode ].getOrCreateModeInstance(
         vrfName )
   mode = parentMode[ ConnMonitorHostConfigMode ].getOrCreateModeInstance(
      ( hostConfig.key.hostName, vrfName ) )
   cmds =  mode[ 'ConnectivityMonitor.config.host' ]
   if vrfName != defaultVrf:
      cmds = mode[ 'ConnectivityMonitor.config.vrf.host' ]
   if hostConfig.description != "":
      cmds.addCommand( 'description' )
      cmds.addCommand( hostConfig.description )
   if hostConfig.intfSetKey.setName != "":
      if config.vrf[ vrfName ].defaultIntfSet != hostConfig.intfSetKey:
         cmds.addCommand( 'local-interfaces %s address-only' %
                          hostConfig.intfSetKey.setName )
   if not hostConfig.ipv4Addr.isAddrZero:
      cmds.addCommand( 'ip %s' % hostConfig.ipv4Addr.stringValue )
   if hostConfig.url != "":
      cmds.addCommand( 'url %s' % hostConfig.url )

def saveIntfSet( intfSet, root ):
   mode = root[ ConnMonitorConfigMode ].getOrCreateModeInstance( None )
   cmds = mode[ 'ConnectivityMonitor.config.intfs' ]
   vrfName = intfSet.key.vrfName
   if vrfName != defaultVrf:
      mode = mode[ ConnMonitorVrfConfigMode ].getOrCreateModeInstance( vrfName )
      cmds = mode[ 'ConnectivityMonitor.config.vrf.intfs' ]
   cmd = 'interface set %s ' % intfSet.key.setName
   cmd += ','.join( sorted( intfSet.intf ) )
   cmds.addCommand( cmd )

def saveVrfConfig( vrfConfig, root ):
   mode = root[ ConnMonitorConfigMode ].getOrCreateModeInstance( None )
   intfCmds = mode[ 'ConnectivityMonitor.config.intfs' ]
   if vrfConfig.vrfName != defaultVrf:
      mode = mode[ ConnMonitorVrfConfigMode ].getOrCreateModeInstance(
         vrfConfig.vrfName )
      intfCmds = mode[ 'ConnectivityMonitor.config.vrf.intfs' ]
   if vrfConfig.description != '':
      intfCmds.addCommand( 'description' )
      intfCmds.addCommand( vrfConfig.description )
   if vrfConfig.defaultIntfSet.setName != "":
      intfCmds.addCommand( 'local-interfaces %s address-only default' %
                           vrfConfig.defaultIntfSet.setName )

@CliSave.saver( 'ConnectivityMonitor::Config', 'connectivityMonitor/config' )
def saveConfig( config, root, sysdbRoot, options ):
   cmds = []
   for intfSetKey in sorted( config.intfSet ):
      saveIntfSet( config.intfSet[ intfSetKey ], root )
   for vrfName in sorted( config.vrf ):
      saveVrfConfig( config.vrf[ vrfName ], root )
   for hostKey in sorted( config.hostConfig ):
      saveHostConfig( config.hostConfig[ hostKey ], root, config )

   if config.probeInterval != config.probeIntervalDefault:
      cmds.append( 'interval %d' % config.probeInterval )
   elif options.saveAll:
      cmds.append( 'no interval' )

   if config.enabled:
      cmds.append( 'no shutdown' )
   elif options.saveAll:
      cmds.append( 'shutdown' )

   if cmds:
      mode = root[ ConnMonitorConfigMode ].getOrCreateModeInstance( None )
      modeCmds = mode[ 'ConnectivityMonitor.config' ]
      for cmd in cmds:
         modeCmds.addCommand( cmd )

def saveHostConfigV2( hostConfig, root, config, sysdbRoot, options ):
   parentMode = root[ ConnMonitorConfigMode ].getOrCreateModeInstance( None )
   mode = parentMode[ ConnMonitorHostConfigMode ].getOrCreateModeInstance(
         hostConfig.hostName )
   cmds = mode[ 'ConnectivityMonitor.config.host' ]

   if hostConfig.description != "":
      cmds.addCommand( 'description' )
      text = hostConfig.description
      cmds.addCommand( text )

   for interface in sorted( hostConfig.intf ):
      cmds.addCommand( 'source interface %s' % interface )

   if not hostConfig.ipv4Addr.isAddrZero:
      cmds.addCommand( 'ip %s' % hostConfig.ipv4Addr.stringValue )

   if hostConfig.url != "":
      cmds.addCommand( 'url %s' % hostConfig.url )

def saveConfigV2( config, root, sysdbRoot, options ):
   cmds = []

   for hostName in sorted( config.hostConfig ):
      saveHostConfigV2( config.hostConfig[ hostName ], root, config, sysdbRoot,
         options )

   if config.probeInterval != config.probeIntervalDefault:
      cmds.append( 'interval %d' % config.probeInterval )
   elif options.saveAll:
      cmds.append( 'no interval' )

   if config.enabled:
      cmds.append( 'no shutdown' )
   elif options.saveAll:
      cmds.append( 'shutdown' )

   if cmds:
      mode = root[ ConnMonitorConfigMode ].getOrCreateModeInstance( None )
      modeCmds = mode[ 'ConnectivityMonitor.config' ]

      for cmd in cmds:
         modeCmds.addCommand( cmd )



