#!/usr/bin/env python
# Copyright (c) 2017 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import CliSave
from CliMode.CloudHaModes import (
      CloudHaAwsMode,
      CloudHaAzureMode,
      CloudHaGcpMode,
      CloudHaMode,
      CloudHaPeerMode,
      CloudProxyMode,
)
import Tac
from CloudUtil import getHypervisor, defaultRecoveryWaitTime
import Tracing

t0 = Tracing.trace0
cloudType = getHypervisor()

class CloudProxyConfigMode( CloudProxyMode, CliSave.Mode ):
   def __init__( self, param ):
      CloudProxyMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

class CloudHaConfigMode( CloudHaMode, CliSave.Mode ):
   def __init__( self, param ):
      CloudHaMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

class CloudHaConfigPeerMode( CloudHaPeerMode, CliSave.Mode ):
   def __init__( self, param ):
      CloudHaPeerMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

class CloudHaConfigPeerAwsMode( CloudHaAwsMode, CliSave.Mode ):
   def __init__( self, param ):
      CloudHaAwsMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

class CloudHaConfigPeerAzureMode( CloudHaAzureMode, CliSave.Mode ):
   def __init__( self, param ):
      CloudHaAzureMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

class CloudHaConfigPeerGcpMode( CloudHaGcpMode, CliSave.Mode ):
   def __init__( self, param ):
      CloudHaGcpMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

CliSave.GlobalConfigMode.addChildMode( CloudProxyConfigMode ) 
CloudProxyConfigMode.addCommandSequence( 'Cloud.global' )

CliSave.GlobalConfigMode.addChildMode( CloudHaConfigMode ) 

CloudHaConfigMode.addCommandSequence( 'CloudHa.global' )
CloudHaConfigMode.addChildMode( CloudHaConfigPeerMode ) 
CloudHaConfigPeerMode.addCommandSequence( 'CloudHaPeer.global' )

CloudHaConfigPeerMode.addChildMode( CloudHaConfigPeerAwsMode ) 
CloudHaConfigPeerAwsMode.addCommandSequence( 'CloudHaAws.global' )

CloudHaConfigPeerMode.addChildMode( CloudHaConfigPeerAzureMode ) 
CloudHaConfigPeerAzureMode.addCommandSequence( 'CloudHaAzure.global' )

CloudHaConfigPeerMode.addChildMode( CloudHaConfigPeerGcpMode ) 
CloudHaConfigPeerGcpMode.addCommandSequence( 'CloudHaGcp.global' )

@CliSave.saver( 'Cloud::HA::ProxyConfig', 'cloudha/proxyConfig' )
def saveCloudProxy( cfg, root, sysdbRoot, options, requireMounts ):
   saveAll = options.saveAll
   if cloudType not in ( 'AWS', 'Azure', 'GCP' ):
      t0( 'unsupported platform' )
   toRun = []
   for i in cfg.proxy.values():
      proxyName = i.name
      cmd = None
      if i.httpProxy:
         cmd = 'http %s %s ' % ( i.httpProxy, i.httpPort )  
         if i.httpUsername:
            cmd += 'username %s ' % \
                CliSave.sanitizedOutput( options, i.httpUsername )
            if i.httpPassword:
               cmd += 'password 7 %s' % \
                CliSave.sanitizedOutput( options, i.httpPassword )
         toRun.append( cmd )
      elif saveAll:
         toRun.append( 'no http' )
      if i.httpsProxy:
         cmd = 'https %s %s ' % ( i.httpsProxy, i.httpsPort )  
         if i.httpsUsername:
            cmd += 'username %s ' % \
                CliSave.sanitizedOutput( options, i.httpsUsername )
            if i.httpsPassword:
               cmd += 'password 7 %s' % \
                    CliSave.sanitizedOutput( options, i.httpsPassword )
         toRun.append( cmd ) 
      elif saveAll:
         toRun.append( 'no https' )
      if not toRun and saveAll:
         toRun.append( 'no http' )
         toRun.append( 'no https' )

      if not toRun:   
         toRun.append( '!' )
      t0( toRun )
      mode = root[ CloudProxyConfigMode ].getOrCreateModeInstance( proxyName )
      cmds = mode[ 'Cloud.global' ]
      for c in toRun:
         cmds.addCommand( c )

# Handles for aws, azure and gcp config.
# Also handles all embedded submodes peer->aws/azure/gcp.
@CliSave.saver( 'Cloud::HA::AwsHaConfig', 'cloudha/awsconfig' )
@CliSave.saver( 'Cloud::HA::AzureHaConfig', 'cloudha/azureconfig' )
@CliSave.saver( 'Cloud::HA::GcpHaConfig', 'cloudha/gcpconfig' )
def saveCloudHa( config, root, sysdbRoot, options, requireMounts ):
   saveAll = options.saveAll
   # This is used mainl to deal with Eos test issues. See comments below.
   testOnly = False
   # This is used to dump config even if there are no
   # values configured within the peer submode.
   peerExist = False
   if cloudType not in ( 'AWS', 'Azure', 'GCP' ):
      testOnly = True
      t0( 'unsupported platform' )
   toRun = []
   peerToRun = []
   peerAwsSubmode = False
   peerAwsSubmodeCmds = []
   peerAzureSubmode = False
   peerAzureSubmodeCmds = []
   peerGcpSubmode = False
   peerGcpSubmodeCmds = []
   awsConfig = azureConfig = gcpConfig = None
   cfg = None
   assert config
   # This is a bit of mess wrt CLi test infra....
   # The test code calls us with none of the  supported platforms ie 
   # Azure/AWS/GCP and doesn't respect CLI guards. We need to contort to 
   # satisfy it for various CLI infra checks. I couldn't find a way to 
   # influence the EosImage/CliModeTests.py to use a platform specfic 
   # guard etc. So what we do is find the true config based on cloudType 
   # as set by environment if any otherwise use gcp which seems to be 
   # default. Since this routine is called for aws, azure and gcp on any
   # of the three platforms, we just ignore the non-applicable config.
   t0( 'Options passed: ', config, root, sysdbRoot, options, requireMounts )
   if config.tacType.fullTypeName == 'Cloud::HA::AzureHaConfig':
      azureConfig = config
   elif config.tacType.fullTypeName == 'Cloud::HA::AwsHaConfig':
      awsConfig = config
   elif config.tacType.fullTypeName == 'Cloud::HA::GcpHaConfig':
      gcpConfig = config
   if testOnly:
      cfg = gcpConfig
   elif cloudType == 'AWS':
      cfg = awsConfig
   elif cloudType == 'Azure':
      cfg = azureConfig
   elif cloudType == 'GCP':
      cfg = gcpConfig
   else:
      assert 0
   # Ignore the non-applicable mode
   if cfg != config:
      t0( ' passed config %s , computed config %s' % ( config, cfg ) )
      t0( 'bailing out as NA for this platform' )
      return
   
   if cfg.enable: 
      toRun.append( 'no shutdown' )
   elif saveAll:
      toRun.append( 'shutdown' )
   if len( cfg.peer ): 
      peerExist = True
      for v in cfg.peer.values(): 
         t0( ' Cloudha peer : ', v )
         defaultIp = Tac.Value( 'Arnet::IpGenAddr' )
         if v.peerIp and v.peerIp != defaultIp:
            peerToRun.append( 'peer address %s' % str( v.peerIp ) ) 
         elif saveAll:
            peerToRun.append( 'no peer address' ) 
         if v.recoveryWaitTime != defaultRecoveryWaitTime or options.saveAll:
            peerToRun.append( 'recovery wait-time %s' % v.recoveryWaitTime )
         if v.bfdConfig.intf:
            peerToRun.append( 'bfd source-interface %s %s' % ( 
                 str( v.bfdConfig.intf ), 'single-hop' if not v.bfdConfig.multihop \
                       else '' ) )
         elif saveAll:
            peerToRun.append( 'no bfd' )

         if cfg.tacType.fullTypeName == 'Cloud::HA::AwsHaConfig':
            if cfg.peerRoutes or cfg.localRoutes:
               peerAwsSubmode = True
               for i in cfg.peerRoutes.values():
                  peerAwsSubmodeCmds.append( 
                       'backup-gateway %s %s local-cloud-interface %s' % \
                          ( str( i.routeTableDestKey.routeTableId ), 
                          str( i.routeTableDestKey.destination ),
                          str( i.nextHopRouteTarget ) ) )
               for i in cfg.localRoutes.values():
                  peerAwsSubmodeCmds.append( 
                       'primary-gateway %s %s local-cloud-interface %s' % \
                          ( str( i.routeTableDestKey.routeTableId ), 
                          str( i.routeTableDestKey.destination ),
                          str( i.nextHopRouteTarget ) ) )
            elif saveAll:
               # Needed to reflect a user entered comment even though config
               # is empty.
               peerAwsSubmode = True
               peerAwsSubmodeCmds.append( '!' )
         if cfg.tacType.fullTypeName == 'Cloud::HA::AzureHaConfig':
            if cfg.peerRoutes or cfg.localRoutes:
               peerAzureSubmode = True
               for i in cfg.peerRoutes.values():
                  peerAzureSubmodeCmds.append( 
                       'backup-gateway %s %s %s resource-group %s' % \
                          ( str( i.routeTableDestKey.routeTableId ), 
                          str( i.routeTableDestKey.destination ),
                          str( i.nextHopRouteTarget ), 
                          i.resourceGroup ) )
               for i in cfg.localRoutes.values():
                  peerAzureSubmodeCmds.append( 
                       'primary-gateway %s %s %s resource-group %s' % \
                          ( str( i.routeTableDestKey.routeTableId ), 
                          str( i.routeTableDestKey.destination ),
                          str( i.nextHopRouteTarget ), 
                          i.resourceGroup ) )
            elif saveAll:
               # Needed to reflect a user entered comment even though config
               # is empty.
               peerAzureSubmode = True
               peerAzureSubmodeCmds.append( '!' )
         if cfg.tacType.fullTypeName == 'Cloud::HA::GcpHaConfig':
            def routeCmd( route, local=True ):
               if local:
                  cmd = 'primary-gateway '
               else:
                  cmd = 'backup-gateway '
               cmd += str( route.destination )
               intfName = str( route.interface )
               vpc = str( route.vpc )
               tag = str( route.tag )
               if intfName:
                  cmd += ' interface %s' % intfName
               if vpc:
                  cmd += ' vpc %s' % vpc
               if tag:
                  cmd += ' tag %s' % tag
               return cmd

            if cfg.peerRoutes or cfg.localRoutes:
               peerGcpSubmode = True
               for i in cfg.peerRoutes:
                  peerGcpSubmodeCmds.append( routeCmd( i, False ) )
               for i in cfg.localRoutes:
                  peerGcpSubmodeCmds.append( routeCmd( i, True ) )
            elif saveAll:
               # Needed to reflect a user entered comment even though config
               # is empty.
               peerGcpSubmode = True
               peerGcpSubmodeCmds.append( '!' )
         if testOnly:
            # Needed to reflect a user entered comment even though config is empty.
            peerAwsSubmode = True
            peerAzureSubmode = True
            peerGcpSubmode = True

   t0( 'Flags: ', toRun, peerToRun, peerAwsSubmode, peerAzureSubmode, peerGcpSubmode,
       peerExist, testOnly )
   if ( toRun or peerToRun or peerExist or
        peerAwsSubmode or peerAzureSubmode or peerGcpSubmode ):
      t0( 'emitting config' )
      mode = root[ CloudHaConfigMode ].getOrCreateModeInstance( None )
      cmds = mode[ 'CloudHa.global' ]
      for c in toRun:
         cmds.addCommand( c )
      for c in cfg.peer.values():
         t0( 'Cloud HA Peer: ', c.name )
         peerMode = mode[ CloudHaConfigPeerMode ].getOrCreateModeInstance( c.name )
         cmds = peerMode[ 'CloudHaPeer.global' ]
         for cmd in peerToRun:
            cmds.addCommand( cmd )
         if peerAzureSubmode:
            peerAzureMode = peerMode[ CloudHaConfigPeerAzureMode ].\
               getOrCreateModeInstance( c.name )
            t0( 'peerAzureMode: ' , c.name, peerAzureMode  )
            azureCmds = peerAzureMode[ 'CloudHaAzure.global' ]
            for azureCmd in peerAzureSubmodeCmds:
               t0( 'azureCmd: ', azureCmd )
               azureCmds.addCommand( azureCmd ) 
         if peerAwsSubmode:
            peerAwsMode = peerMode[ CloudHaConfigPeerAwsMode ].\
               getOrCreateModeInstance( c.name )
            t0( 'peer aws mode: ', c.name, peerAwsMode  )
            awsCmds = peerAwsMode[ 'CloudHaAws.global' ]
            for awsCmd in peerAwsSubmodeCmds:
               t0( 'awsCmd: ', awsCmd )
               awsCmds.addCommand( awsCmd ) 
         if peerGcpSubmode:
            peerGcpMode = peerMode[ CloudHaConfigPeerGcpMode ].\
               getOrCreateModeInstance( c.name )
            t0( 'peerGcpMode: ' , c.name, peerGcpMode  )
            gcpCmds = peerGcpMode[ 'CloudHaGcp.global' ]
            for gcpCmd in peerGcpSubmodeCmds:
               t0( 'gcpCmd: ', gcpCmd )
               gcpCmds.addCommand( gcpCmd ) 
