#!/usr/bin/env python
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import CliSave
import Tac
from CliMode.Classification import ( AppTrafficRecModeBase, AppProfileModeBase,
                                     FieldSetPrefixModeBase,
                                     FieldSetL4PortModeBase, AppModeBase,
                                     AppTrafficRecConfigMode )
from ClassificationLib import ( numericalRangeToRangeString, getProtocolNumToNameMap,
                                numericalRangeToSet, rangeSetToNumericalRange )
from IntfCliSave import IntfConfigMode

class AppRecognitionSaveMode( AppTrafficRecModeBase, CliSave.Mode ):
   def __init__( self, param ):
      AppTrafficRecModeBase.__init__( self )
      CliSave.Mode.__init__( self, param )

CliSave.GlobalConfigMode.addChildMode( AppRecognitionSaveMode,
                                       after=[ IntfConfigMode ] )
AppRecognitionSaveMode.addCommandSequence( 'Classification.AppRecognition' )

class FieldSetL4PortSaveMode( FieldSetL4PortModeBase, CliSave.Mode ):
   def __init__( self, param ):
      FieldSetL4PortModeBase.__init__( self, param )
      CliSave.Mode.__init__( self, param )

AppRecognitionSaveMode.addChildMode( FieldSetL4PortSaveMode )
FieldSetL4PortSaveMode.addCommandSequence( 'Classification.FieldSetL4Port' )

class FieldSetIpPrefixSaveMode( FieldSetPrefixModeBase, CliSave.Mode ):
   def __init__( self, param ):
      FieldSetPrefixModeBase.__init__( self, param )
      CliSave.Mode.__init__( self, param )

AppRecognitionSaveMode.addChildMode( FieldSetIpPrefixSaveMode )
FieldSetIpPrefixSaveMode.addCommandSequence( 'Classification.FieldSetIpPrefix' )

class AppSaveMode( AppModeBase, CliSave.Mode ):
   def __init__( self, param ):
      AppModeBase.__init__( self, param )
      CliSave.Mode.__init__( self, param )

AppRecognitionSaveMode.addChildMode( AppSaveMode )
AppSaveMode.addCommandSequence( 'Classification.App' )

class AppProfileSaveMode( AppProfileModeBase, CliSave.Mode ):
   def __init__( self, param ):
      AppProfileModeBase.__init__( self, param )
      CliSave.Mode.__init__( self, param )

AppRecognitionSaveMode.addChildMode( AppProfileSaveMode )
AppProfileSaveMode.addCommandSequence( 'Classification.AppProfile' )

@CliSave.saver( 'Classification::AppRecognitionConfig',
                'classification/app-recognition/config',
                requireMounts=( 'classification/app-recognition/fieldset', ) )
def saveAppRecognitionConfig( entity, root, sysdbRoot, options, requireMounts ):
   fieldSetConfig = requireMounts[ 'classification/app-recognition/fieldset' ]
   for fieldSetName in sorted( fieldSetConfig.fieldSetL4Port ):
      appRecogMode = root[ AppRecognitionSaveMode ].getOrCreateModeInstance(
         'Classification.AppRecognition' )
      param = ( "app-recog", "l4-port", fieldSetName, AppTrafficRecConfigMode )
      cmds = appRecogMode[ FieldSetL4PortSaveMode ].getOrCreateModeInstance(
         param )[ 'Classification.FieldSetL4Port' ]
      fieldSetL4PortCfg = fieldSetConfig.fieldSetL4Port[ fieldSetName ]

      if fieldSetL4PortCfg.currCfg and fieldSetL4PortCfg.currCfg.ports:
         cmds.addCommand( numericalRangeToRangeString(
            fieldSetL4PortCfg.currCfg.ports ) )

   for fieldSetName in sorted( fieldSetConfig.fieldSetIpPrefix ):
      appRecogMode = root[ AppRecognitionSaveMode ].getOrCreateModeInstance(
         'Classification.AppRecognition' )

      fieldSetIpPrefixCfg = fieldSetConfig.fieldSetIpPrefix[ fieldSetName ]
      param = ( "app-recog", fieldSetIpPrefixCfg.af,
                fieldSetName, AppTrafficRecConfigMode )
      cmds = appRecogMode[ FieldSetIpPrefixSaveMode ].getOrCreateModeInstance(
         param )[ 'Classification.FieldSetIpPrefix' ]
      if fieldSetIpPrefixCfg.currCfg and fieldSetIpPrefixCfg.currCfg.prefixes:
         prefixes = sorted( prefix.stringValue for prefix in
                            fieldSetIpPrefixCfg.currCfg.prefixes )
         cmdStr = " ".join( prefixes )
         cmds.addCommand( cmdStr )

   for appName in sorted( entity.app.keys() ):
      if entity.app[ appName ].readonly:
         continue
      appRecogMode = root[ AppRecognitionSaveMode ].getOrCreateModeInstance(
         'Classification.AppRecognition' )
      cmds = appRecogMode[ AppSaveMode ].getOrCreateModeInstance(
         appName )[ 'Classification.App' ]
      if entity.app[ appName ].srcPrefixFieldSet:
         cmds.addCommand( 'source prefix field-set %s' %
                          entity.app[ appName ].srcPrefixFieldSet )
      if entity.app[ appName ].dstPrefixFieldSet:
         cmds.addCommand( 'destination prefix field-set %s' %
                          entity.app[ appName ].dstPrefixFieldSet )
      if entity.app[ appName ].srcPortFieldSet or \
         entity.app[ appName ].dstPortFieldSet:
         protoSet = numericalRangeToSet( entity.app[ appName ].proto )
         cmd = 'protocol'
         if 6 in protoSet:
            cmd += ' tcp'
         if 17 in protoSet:
            cmd += ' udp'
         if entity.app[ appName ].srcPortFieldSet:
            srcFieldSet = entity.app[ appName ].srcPortFieldSet
            cmd += ' source port field-set %s' % srcFieldSet
         if entity.app[ appName ].dstPortFieldSet:
            dstFieldSet = entity.app[ appName ].dstPortFieldSet
            cmd += ' destination port field-set %s' % dstFieldSet
         cmds.addCommand( cmd )
      elif entity.app[ appName ].proto:
         protoNoToName = getProtocolNumToNameMap()
         protoSet = numericalRangeToSet( entity.app[ appName ].proto )
         for proto in protoNoToName:
            if proto in protoSet:
               cmds.addCommand( 'protocol %s' % protoNoToName[ proto ] )
               protoSet.remove( proto )
         if protoSet:
            numRange = rangeSetToNumericalRange( protoSet,
                                                 "Classification::ProtocolRange" )
            protoStr = numericalRangeToRangeString( numRange )
            cmds.addCommand( 'protocol %s' % protoStr )

   for appProfileName in sorted( entity.appProfile.keys() ):
      appRecogMode = root[ AppRecognitionSaveMode ].getOrCreateModeInstance(
         'Classification.AppRecognition' )
      cmds = appRecogMode[ AppProfileSaveMode ].getOrCreateModeInstance(
         appProfileName )[ 'Classification.AppProfile' ]
      for appName in sorted( entity.appProfile[ appProfileName ].app.keys() ):
         cmds.addCommand( 'application %s' % appName )
