#!/usr/bin/env python
# Copyright (c) 2016 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from CliMode.Cfm import CfmModeBase, MaintenanceDomainModeBase, CfmProfileModeBase, \
   MaintenanceAssociationModeBase, LocalMaintenanceEndPointModeBase
import CliSave
from CfmLib import ccmTxIntervalToCliToken
from CfmTypes import defaultCfmProfileName, tacMepDirection, defaultPrimaryVlanId

CliSave.GlobalConfigMode.addCommandSequence( 'Cfm.config' )

#---------------------------------------------------------------------
# Modes
#---------------------------------------------------------------------
class CfmConfigMode( CfmModeBase, CliSave.Mode ):
   def __init__( self, param ):
      CfmModeBase.__init__( self, param )
      CliSave.Mode.__init__( self, param )

class MaintenanceDomainConfigMode( MaintenanceDomainModeBase, CliSave.Mode ):
   def __init__( self, param ):
      MaintenanceDomainModeBase.__init__( self, param )
      CliSave.Mode.__init__( self, param )

class CfmProfileConfigMode( CfmProfileModeBase, CliSave.Mode ):
   def __init__( self, param ):
      CfmProfileModeBase.__init__( self, param )
      CliSave.Mode.__init__( self, param )

class MaintenanceAssociationConfigMode( MaintenanceAssociationModeBase,
                                        CliSave.Mode ):
   def __init__( self, param ):
      MaintenanceAssociationModeBase.__init__( self, param )
      CliSave.Mode.__init__( self, param )

class LocalMaintenanceEndPointConfigMode( LocalMaintenanceEndPointModeBase,
                                          CliSave.Mode ):
   def __init__( self, param ):
      LocalMaintenanceEndPointModeBase.__init__( self, param )
      CliSave.Mode.__init__( self, param )

#---------------------------------------------------------------------
# SaveBlocks
#---------------------------------------------------------------------
CliSave.GlobalConfigMode.addChildMode( CfmConfigMode )
CfmConfigMode.addChildMode( MaintenanceDomainConfigMode )
MaintenanceDomainConfigMode.addCommandSequence( 'Cfm.mdConfig' )

CfmConfigMode.addChildMode( CfmProfileConfigMode )
CfmProfileConfigMode.addCommandSequence( 'Cfm.cfmProfileConfig' )

MaintenanceDomainConfigMode.addChildMode( MaintenanceAssociationConfigMode )
MaintenanceAssociationConfigMode.addCommandSequence( 'Cfm.maConfig' )

MaintenanceAssociationConfigMode.addChildMode( LocalMaintenanceEndPointConfigMode )
LocalMaintenanceEndPointConfigMode.addCommandSequence( 'Cfm.localMepConfig' )

#---------------------------------------------------------------------
# Handlers
#---------------------------------------------------------------------
@CliSave.saver( 'CfmAgent::Config', 'cfm/config' )
def saveCfmConfig( entity, root, sysdbRoot, options ):
   saveAll = options.saveAll
   saveAllDetail = options.saveAllDetail

   def getCfmConfigMode( root ):
      return root[ CfmConfigMode ].getOrCreateModeInstance( None )

   def getMdConfigMode( mdConfig, root ):
      cfmConfigMode = getCfmConfigMode( root )
      return cfmConfigMode[ MaintenanceDomainConfigMode ].getOrCreateModeInstance(
         mdConfig.mdLevel )

   def getCfmProfileConfigMode( cfmProfileConfig, root ):
      cfmConfigMode = getCfmConfigMode( root )
      return cfmConfigMode[ CfmProfileConfigMode ].getOrCreateModeInstance(
         cfmProfileConfig.cfmProfileName )

   def getMaConfigMode( maConfig, root ):
      mdConfigMode = getMdConfigMode( maConfig.mdConfig, root )
      return mdConfigMode[
         MaintenanceAssociationConfigMode ].getOrCreateModeInstance(
            ( maConfig.mdConfig.mdLevel, maConfig.maNameId ) )

   def getLocalMepConfigMode( localMepConfig, root ):
      maConfigMode = getMaConfigMode( localMepConfig.maConfig, root )
      return maConfigMode[
         LocalMaintenanceEndPointConfigMode ].getOrCreateModeInstance(
            ( localMepConfig.maConfig.mdConfig.mdLevel,
              localMepConfig.maConfig.maNameId, localMepConfig.localMepId ) )

   def saveMdConfig( mdConfig, root, options ):
      mdConfigMode = getMdConfigMode( mdConfig, root )
      cmds = mdConfigMode[ 'Cfm.mdConfig' ]

      if mdConfig.mdName.name:
         cmds.addCommand( 'name %s' % mdConfig.mdName.name )
      elif saveAll or saveAllDetail:
         cmds.addCommand( 'no name' )

      if mdConfig.defaultMip:
         cmds.addCommand( 'intermediate-point' )
      elif saveAll or saveAllDetail:
         cmds.addCommand( 'no intermediate-point' )

   def saveMaConfig( maConfig, root, options ):
      maConfigMode = getMaConfigMode( maConfig, root )
      cmds = maConfigMode[ 'Cfm.maConfig' ]

      direction = 'up' if maConfig.direction == tacMepDirection.MepDirectionUp \
                  else 'down'
      cmds.addCommand( 'direction %s' % direction )
      cmds.addCommand( 'profile %s' % maConfig.cfmProfileName )
      if maConfig.primaryVlanId != defaultPrimaryVlanId:
         cmds.addCommand( 'vlan %d' % maConfig.primaryVlanId )
      elif saveAll or saveAllDetail:
         cmds.addCommand( 'no vlan' )

   def saveLocalMepConfig( localMepConfig, root, options ):
      localMepConfigMode = getLocalMepConfigMode(
         localMepConfig, root )
      cmds = localMepConfigMode[ 'Cfm.localMepConfig' ]

      if localMepConfig.intfId:
         cmds.addCommand( 'interface %s' % localMepConfig.intfId )
      elif saveAll or saveAllDetail:
         cmds.addCommand( 'no interface' )

      for remoteMepId in sorted( localMepConfig.remoteMepConfig ):
         cmds.addCommand( 'remote endpoint add %d' % remoteMepId )

   def saveCfmProfileConfig( cfmProfile, root, options ):
      cfmProfileConfigMode = getCfmProfileConfigMode( cfmProfile, root )
      cmds = cfmProfileConfigMode[ 'Cfm.cfmProfileConfig' ]

      if cfmProfile.ccmEnable:
         cmds.addCommand( 'continuity-check' )
      elif saveAll or saveAllDetail:
         cmds.addCommand( 'no continuity-check' )

      ccmTxIntervalWithUnit = ccmTxIntervalToCliToken[ cfmProfile.ccmTxInterval ]
      cmds.addCommand( 'continuity-check tx-interval %g %s'
                       % ( ccmTxIntervalWithUnit.ccmTxInterval,
                           ccmTxIntervalWithUnit.unit ) )

   # Use of get() instead of items(), since this python code do not run
   # in context of the Activity Loop. Refer Bug 86221 for more info.
   for mdLevel in entity.mdConfig:
      mdConfig = entity.mdConfig.get( mdLevel )
      if mdConfig is None:
         continue
      saveMdConfig( mdConfig, root, options )

      for maNameId in mdConfig.maConfig:
         maConfig = mdConfig.maConfig.get( maNameId )
         if maConfig is None:
            continue
         saveMaConfig( maConfig, root, options )

         for localMepId in maConfig.localMepConfig:
            localMepConfig = maConfig.localMepConfig.get( localMepId )
            if localMepConfig is None:
               continue
            saveLocalMepConfig( localMepConfig, root, options )

   for cfmProfileName in entity.cfmProfile:
      cfmProfile = entity.cfmProfile.get( cfmProfileName )
      if cfmProfile is None or cfmProfile.cfmProfileName == defaultCfmProfileName:
         continue
      saveCfmProfileConfig( cfmProfile, root, options )
