# Copyright (c) 2017 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import HttpServiceConstants
from CliMode.Capi import CapiVrfConfigModeBase
import CliSave
import Management
import Tac
import CliSavePlugin.HttpService as HttpService

class CapiConfigMode( Management.MgmtConfigMode ):

   def __init__( self, param ):
      Management.MgmtConfigMode.__init__( self, "api http-commands" )

CliSave.GlobalConfigMode.addChildMode( CapiConfigMode )
CapiConfigMode.addCommandSequence( 'Mgmt.api-http-cmds' )

class CapiVrfConfigMode( CapiVrfConfigModeBase, CliSave.Mode ):

   def __init__( self, param ):
      CapiVrfConfigModeBase.__init__( self, param )
      CliSave.Mode.__init__( self, param )

CapiConfigMode.addChildMode( CapiVrfConfigMode,
                             after=[ 'Mgmt.api-http-cmds' ] )
CapiVrfConfigMode.addCommandSequence( 'Mgmt.api-http-cmds-vrf' )

@CliSave.saver( 'HttpService::Config', 'mgmt/capi/config',
                requireMounts = ( 'cli/config', ) )
def saveConfig( config, root, sysdbRoot, options, requireMounts ):

   # Build list of commands
   capiCmds = [] 

   if not config.useHttpServiceCli:
      capiCmds = HttpService.saveCommonHttpServerCmds( config, options )
   capiCmds += saveCapiConfigCmds( config, options, requireMounts )

   if capiCmds:
      mode = root[ CapiConfigMode ].getOrCreateModeInstance(
             'api http-commands' )
      currCmds = mode[ 'Mgmt.api-http-cmds' ]
      for cmd in capiCmds:
         currCmds.addCommand( cmd )
   saveVrf( config, root, options )

def saveVrf( config, root, options ):
   # Save the VRF in the proper sub-submode
   serviceAclTypeVrfMap = config.serviceAclTypeVrfMap
   aclVrfs = set()
   for key, aclName in serviceAclTypeVrfMap.aclName.iteritems():
      aclVrfs.add( key.vrfName )

   for ( vrf, vrfConfig ) in config.vrfConfig.items():
      if 'http-commands' in vrfConfig.vrfService:
         mode = root[ CapiConfigMode 
                    ].getOrCreateModeInstance( 'api http-commands' )
         vrfMode = mode[ CapiVrfConfigMode 
                       ].getOrCreateModeInstance( ( vrf, config ) )
         vrfCmd = vrfMode[ 'Mgmt.api-http-cmds-vrf' ]
         vrfCmd.addCommand( 'no shutdown' )
   
   for vrf in aclVrfs:
      mode = root[ CapiConfigMode ].getOrCreateModeInstance( 'api http-commands' )
      vrfMode = mode[ CapiVrfConfigMode ].getOrCreateModeInstance( ( vrf, config ) )
      vrfCmd = vrfMode[ 'Mgmt.api-http-cmds-vrf' ]
      for aclType in [ 'ip', 'ipv6' ]:
         key = Tac.Value( "Acl::AclTypeAndVrfName", aclType, vrf )
         aclName = ( serviceAclTypeVrfMap and 
                     serviceAclTypeVrfMap.aclName.get( key ) )
         if aclName is None or aclName == '':
            continue
         vrfCmd.addCommand( '%s access-group %s' % ( aclType, aclName ) )

def saveCapiConfigCmds( config, options, requireMounts ):

   cmds = []

   # We can't configure cert/cipher when using HttpService cli so don't save it
   if not config.useHttpServiceCli:
      saveHttpsCert( cmds, options,
                     config.httpsConfig.sslCertificate,
                     config.httpsConfig.sslKey )
      saveHttpsCipher( cmds, options, config.httpsConfig )

   # Add the "no shutdown" last, so we don't continuously restart the
   # server on startup.
   if ( 'http-commands' in config.service and
         config.service[ 'http-commands' ].enabled ):
      cmds.append( "no shutdown" )
   elif options.saveAll:
      cmds.append( "shutdown" )
   cliConfig = requireMounts[ 'cli/config' ]
   if cliConfig.validateOutput:
      cmds.append( "validate-output" )

   return cmds

def saveHttpsCert( cmds, options, cert, key ):
   if not cert and not key:
      if options.saveAll:
         cmds.append( 'no protocol https certificate'  )
   else:
      lines = [ 'protocol https certificate' ]
      for value in ( cert, key ):
         saveOutput = CliSave.sanitizedOutput( options, value ).rstrip( "\n" )
         for line in saveOutput.split( "\n" ):
            lines.append( line )
         lines.append( "EOF" )
      # in json, make it a single command with embedded <CR> (see BUG170442)
      if options.showJson: 
         cmds.append( "\n".join( lines ) )
      else: 
         for line in lines:
            cmds.append( line )

def sslConfigs( cmds, cmd, config, options, defaultValues, hasConfig ):
   if config:
      cmds.append( cmd % " ".join( config.itervalues() ) )
   elif hasConfig:
      cmds.append( cmd % " ".join( defaultValues ) )

def saveHttpsCipher( cmds, options, httpsConfig ):
   # If options are not configured then nothing is saved. This is to
   # differentiate the case where user has enabled all the options ( vs )
   # has not configured any thing. If no option is configured  we allow
   # openssl to use all the builtin CipherSuites.
   sslConfigs( cmds, "protocol https cipher %s", httpsConfig.ciphers,
               options, HttpServiceConstants.CapiCipherSuites.CIPHERS,
               httpsConfig.hasCipherConfig )
   sslConfigs( cmds, "protocol https key-exchange %s", 
               httpsConfig.keyExchange, options, 
               HttpServiceConstants.CapiCipherSuites.KEYEXCHANGE,
               httpsConfig.hasKeyExchangeConfig )
   sslConfigs( cmds, "protocol https mac %s", httpsConfig.mac,
               options, HttpServiceConstants.CapiCipherSuites.MAC,
               httpsConfig.hasMacConfig )

