# Copyright (c) 2016 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.
import CliSave
import Tac
from CliMode.Bmp import RoutingBmpStationMode
from CliSavePlugin import RoutingBgpCliSave
from CliSavePlugin.RoutingBgpCliSave import RouterBgpBaseConfigMode
from IpLibConsts import DEFAULT_VRF
from ReversibleSecretCli import encodeKey
from RouteMapLib import isAsdotConfigured

import Toggles.BmpToggleLib

ConnectionMode = Tac.Type( 'Routing::Bmp::ConnectionMode' )
TimestampMode = Tac.Type( 'Routing::Bmp::TimestampMode' )

#------------------------------------------------------------------------------------
# monitoring station mode
#------------------------------------------------------------------------------------
class RouterBmpStationConfigMode( RoutingBmpStationMode, CliSave.Mode ):
   def __init__( self, param ):
      RoutingBmpStationMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

RouterBgpBaseConfigMode.addChildMode( RouterBmpStationConfigMode )
RouterBmpStationConfigMode.addCommandSequence( 'Bgp.bmp.station' )

def saveBmpStationConfig( stationConfig, root, options, stationName, parentMode,
                          securityConfig ):

   mode = parentMode[ RouterBmpStationConfigMode ].getOrCreateModeInstance(
         stationName )
   cmds = mode[ 'Bgp.bmp.station' ]

   def isConfiguredAttribute( attrStr ):
      attr = getattr( stationConfig, attrStr )
      attrDefault = getattr( stationConfig, attrStr + 'Default' )
      return attr != attrDefault

   def addSimpleSaveCommand( attrStr, cmd, trail=None, add=True, integer=False ):
      attr = getattr( stationConfig, attrStr )
      attrDefault = getattr( stationConfig, attrStr + 'Default' )
      present = False
      saveCmd = ''

      if isConfiguredAttribute( attrStr ):
         saveCmd = '%s' % ( cmd )
         if attrStr == 'shutdown':
            shutdownMsgAttr = stationConfig.shutdownMsg
            shutdownMsgAttrDefault = stationConfig.shutdownMsgDefault
            if shutdownMsgAttr != shutdownMsgAttrDefault:
               saveCmd += ' reason %s' % shutdownMsgAttr
         if attr == attrDefault:
            saveCmd = 'no ' + saveCmd
         elif not isinstance( attr, bool ):
            if integer:
               attr = int( attr )
            saveCmd = saveCmd + ' ' + str( attr )
            present = True
            if trail:
               saveCmd = saveCmd + ' ' + trail
      elif options.saveAll:
         saveCmd = 'no %s' % ( cmd )
         present = False
      if add and saveCmd:
         cmds.addCommand( saveCmd )
      return saveCmd, present

   addSimpleSaveCommand( 'description', 'description' )
   addSimpleSaveCommand( 'statsInterval', 'statistics-interval', integer=True )
   addSimpleSaveCommand( 'shutdown', 'shutdown' )
   addSimpleSaveCommand( 'sourceIntf', 'update-source' )
   if isConfiguredAttribute( 'authenticationKey' ):
      if securityConfig.commonKeyEnabled:
         encryptedPasswd = encodeKey( stationConfig.authenticationKey,
                                      algorithm='MD5' )
      else:
         encryptedPasswd = encodeKey( stationConfig.authenticationKey,
                                      key=stationName + '_passwd',
                                      algorithm='MD5' )
      cmds.addCommand( 'authentication-key 7 %s' %
                        CliSave.sanitizedOutput( options, encryptedPasswd ) )
   elif options.saveAll:
      cmds.addCommand( 'no authentication-key' )

   #------------------------------------------------------------------------------
   # CliSave for "[no|default] connection address <ip|ipv6|hostname> [vrf <VRF>]"
   # command in "monitoring station" mode.
   #------------------------------------------------------------------------------

   addrCmd = 'connection address'
   ( vrfCmd, vrfPresent ) = addSimpleSaveCommand( 'vrfName', 'vrf', add=False )
   if isConfiguredAttribute( 'remoteHost' ):
      if vrfPresent:
         ( addrCmd, _ ) = addSimpleSaveCommand( 'remoteHost', addrCmd,
                                                add=False )
         cmds.addCommand( '%s %s' % ( addrCmd, vrfCmd ) )
      else:
         ( _, _ ) = addSimpleSaveCommand( 'remoteHost', addrCmd, add=True )
   else:
      if options.saveAll:
         cmds.addCommand( 'no %s' % addrCmd )

   #-------------------------------------------------------------------------------
   # CliSave for command
   # "[no|default] connection mode {active port <N> [reconnect-timer <N>]} |
   #                               {passive} "
   # in "monitoring station" mode.
   #-------------------------------------------------------------------------------
   modeCmd = 'connection mode'
   if stationConfig.connectionMode == ConnectionMode.passive:
      cmds.addCommand( '%s passive' % modeCmd )
   elif stationConfig.connectionMode == ConnectionMode.active \
         and isConfiguredAttribute( 'remotePort' ):
      remotePort = stationConfig.remotePort
      if isConfiguredAttribute( 'reconnectTime' ):
         reconnectTime = stationConfig.reconnectTime
         cmds.addCommand( '%s active port %s reconnect-timer %s' %
                          ( modeCmd, remotePort, int( reconnectTime ) ) )
      else:
         cmds.addCommand( '%s active port %s' % ( modeCmd, remotePort ) )
   elif options.saveAll:
      cmds.addCommand( 'no connection mode' )

   keepAlive = stationConfig.keepaliveOption

   idleTime = keepAlive.idleTime 
   probeInterval = keepAlive.probeInterval
   probeCount = keepAlive.probeCount 

   if keepAlive.enabled:
      cmds.addCommand( 'connection keepalive %s %s %s' %
            ( idleTime, probeInterval, probeCount ) )
   elif options.saveAll:
      cmds.addCommand( 'no connection keepalive' )

   stationExportPolicy = stationConfig.bmpExportPolicyStationConfig
   cmd = "export-policy received routes"

   if stationExportPolicy.isSet:
      if stationExportPolicy.prePolicyExport:
         cmd += " pre-policy"
      if stationExportPolicy.postPolicyExport:
         cmd += " post-policy"
      if not stationExportPolicy.postPolicyExport \
            and not stationExportPolicy.prePolicyExport:
         cmd = "no " + cmd
      cmds.addCommand( cmd )
   elif options.saveAll:
      cmd = "default " + cmd
      cmds.addCommand( cmd )

RouterBgpBaseConfigMode.addCommandSequence( 'Bgp.config.bmp' )

@CliSave.saver( 'Routing::Bmp::BmpConfig', 'routing/bmp/config',
                requireMounts=( 'routing/bgp/config', 'routing/bgp/asn/config',
                'mgmt/security/config', ) )
def saveBmpConfig( bmpConfig, root, sysdbRoot, options, requireMounts ):
   bgpConfig = requireMounts[ 'routing/bgp/config' ]
   asnConfig = requireMounts[ 'routing/bgp/asn/config' ]
   securityConfig = requireMounts[ 'mgmt/security/config' ]
   if bgpConfig.asNumber == 0:
      return

   parentMode = root[ RouterBgpBaseConfigMode ].getOrCreateModeInstance( (
         bgpConfig.asNumber, isAsdotConfigured( asnConfig ), ) )

   for stationName in bmpConfig.bmpStation:
      bmpStationConfig = bmpConfig.bmpStation.get( stationName )
      if not bmpStationConfig:
         continue
      saveBmpStationConfig( bmpStationConfig, root, options, stationName,
                            parentMode, securityConfig )

   cmds = parentMode[ 'Bgp.config.bmp' ]
   # BUG157364: Add non-default VRF support
   vrfName = DEFAULT_VRF
   port = bmpConfig.localPort.get( vrfName )
   if port:
      cmds.addCommand( 'monitoring port %s' % port )
   elif options.saveAll:
      cmds.addCommand( 'no monitoring port' )

   prePolicyExport = bmpConfig.bmpExportPolicyConfig.prePolicyExport
   postPolicyExport = bmpConfig.bmpExportPolicyConfig.postPolicyExport

   if not prePolicyExport:
      cmds.addCommand( 'no monitoring received routes pre-policy' )
   elif options.saveAll:
      cmds.addCommand( 'monitoring received routes pre-policy' )

   if not postPolicyExport:
      cmds.addCommand( 'no monitoring received routes post-policy' )
   elif options.saveAll:
      cmds.addCommand( 'monitoring received routes post-policy' )

   afiSafi = Tac.Value( 'Routing::Bgp::AfiSafi', 'afiIpv4', 'safiUnicast' )
   if not bmpConfig.afiSafiExport[ afiSafi ]:
      cmds.addCommand( 'monitoring received routes address-family '
                       'ipv4 unicast disable' )
   elif options.saveAll:
      cmds.addCommand( 'monitoring received routes address-family ipv4 unicast' )

   afiSafi = Tac.Value( 'Routing::Bgp::AfiSafi', 'afiIpv6', 'safiUnicast' )
   if not bmpConfig.afiSafiExport[ afiSafi ]:
      cmds.addCommand( 'monitoring received routes address-family '
                       'ipv6 unicast disable' )
   elif options.saveAll:
      cmds.addCommand( 'monitoring received routes address-family ipv6 unicast' )

   afiSafi = Tac.Value( 'Routing::Bgp::AfiSafi', 'afiIpv6', 'safiMplsLabels' )
   if afiSafi in bmpConfig.afiSafiExport and bmpConfig.afiSafiExport[ afiSafi ]:
      cmd = [ 'monitoring received routes address-family ipv6 labeled-unicast' ]
      if bmpConfig.exportSixPe:
         cmd.append( '6pe' )
      if bmpConfig.exportIpv6LuTunnel:
         cmd.append( 'tunnel' )
      cmds.addCommand( ' '.join( cmd ) )
   elif options.saveAll:
      cmds.addCommand( ( 'no monitoring received routes address-family'
                         ' ipv6 labeled-unicast' ) )
   if Toggles.BmpToggleLib.toggleBmpArBgpVpnToggleEnabled():
      afiSafi = Tac.Value( 'Routing::Bgp::AfiSafi', 'afiIpv4', 'safiMplsVpn' )
      if afiSafi in bmpConfig.afiSafiExport and bmpConfig.afiSafiExport[ afiSafi ]:
         cmds.addCommand( 'monitoring received routes address-family '
                          'vpn-ipv4' )
      elif options.saveAll:
         cmds.addCommand( 'no monitoring received routes address-family vpn-ipv4' )

      afiSafi = Tac.Value( 'Routing::Bgp::AfiSafi', 'afiIpv6', 'safiMplsVpn' )
      if afiSafi in bmpConfig.afiSafiExport and bmpConfig.afiSafiExport[ afiSafi ]:
         cmds.addCommand( 'monitoring received routes address-family '
                          'vpn-ipv6' )
      elif options.saveAll:
         cmds.addCommand( 'no monitoring received routes address-family vpn-ipv6' )

   timestamp = bmpConfig.timestampMode
   if timestamp == TimestampMode.none:
      cmds.addCommand( 'monitoring timestamp none' )
   elif options.saveAll:
      cmds.addCommand( 'monitoring timestamp send-time' )

   if bmpConfig.bmpActivate:
      cmds.addCommand( 'bgp monitoring' )
   elif options.saveAll:
      cmds.addCommand( 'no bgp monitoring' )

# Bgp CliSave for "neighbor * monitoring"
def saveNeighborMonitoring( peer, peerConfig, saveAll ):
   cmd = 'neighbor %s monitoring' % peer
   if peerConfig.bmpActivatePresent:
      if peerConfig.bmpActivate == True:
         return cmd
      else:
         return 'no ' + cmd
   elif saveAll and not peerConfig.isPeerGroupPeer:
      return 'default ' + cmd
   return None
RoutingBgpCliSave.neighborMonitoringHook.addExtension( saveNeighborMonitoring )
