#!/usr/bin/env python
# Copyright (c) 2016 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import CliSave
import MultiRangeRule
from CliSavePlugin.RoutingBgpCliSave import RouterBgpBaseConfigMode
from CliMode.BgpMacVrfConfigMode import (
   BgpMacVrfMode,
   BgpMacVrfVpwsPwMode,
)

from BgpLib import routeTargetToPrint
from ArnetLib import formatRd
from RouteMapLib import isAsdotConfigured, bgpFormatAsn
from Toggles.ArBgpToggleLib import toggleVpwsFlowLabelToggleEnabled
from Toggles.ArBgpToggleLib import toggleRedistRouterMacPrimaryCliEnabled
from Toggles.PseudowireToggleLib import toggleEvpnVpwsFxc1Enabled
from TypeFuture import TacLazyType

import Tracing

th = Tracing.Handle( 'BgpMacVrfConfigCliSave' )
t0 = th.trace0

ControlWordOverride = TacLazyType( "Pseudowire::Hardware::ControlWordOverride" )
EvpnEtid = TacLazyType( "Evpn::Etid" )
FxcMode = TacLazyType( "Pseudowire::Fxc::Mode" )

class BgpMacVrfConfigSaveMode( BgpMacVrfMode, CliSave.Mode ):
   def __init__( self, params ):
      BgpMacVrfMode.__init__( self, *params )
      CliSave.Mode.__init__( self, params[ 0 ] )

RouterBgpBaseConfigMode.addChildMode( BgpMacVrfConfigSaveMode )
BgpMacVrfConfigSaveMode.addCommandSequence( 'Bgp.macvrf.config' )

class BgpMacVrfVpwsPwSaveMode( BgpMacVrfVpwsPwMode, CliSave.Mode ):
   def __init__( self, pwName ):
      BgpMacVrfVpwsPwMode.__init__( self, pwName )
      CliSave.Mode.__init__( self, pwName )

BgpMacVrfConfigSaveMode.addChildMode( BgpMacVrfVpwsPwSaveMode )
BgpMacVrfVpwsPwSaveMode.addCommandSequence( 'Bgp.macvrf.pw.config' )

def saveRouteTarget( cmds, macVrfConfig, asdotConfigured ):
   # route-target < import | export | both >
   # add route-target import or both
   for rt in macVrfConfig.importRtList:
      # Check for "concat" cases: "import export", and "both"
      concat = rt in macVrfConfig.bothRtList or rt in macVrfConfig.importExportRtList
      if rt in macVrfConfig.exportRtList and concat:
         direction = 'both' if rt in macVrfConfig.bothRtList else 'import export'
      else:
         direction = 'import'
      cmds.addCommand(
         'route-target %s %s' % ( direction,
            routeTargetToPrint( rt, asdotConfigured ) ) )

   # add route-target export, unless it's already been printed above
   for rt in macVrfConfig.exportRtList:
      concat = rt in macVrfConfig.bothRtList or rt in macVrfConfig.importExportRtList
      alreadyPrinted = ( rt in macVrfConfig.importRtList and concat )
      if not alreadyPrinted:
         cmds.addCommand( 
            'route-target export ' + routeTargetToPrint( rt, asdotConfigured ) )

   # add route-target export auto
   if macVrfConfig.autoExportRt:
      cmds.addCommand( 'route-target export auto' )

   # add route-target import auto < asn >
   for asn in macVrfConfig.autoImportRtList:
      cmds.addCommand( 'route-target import auto %s' % (
                       bgpFormatAsn( asn, asdotConfigured ) ) )

class VpwsMacVrfConfigSaver( object ):
   def __init__( self, macVrfConfig, macVrfMode, bgpConfig, asdotConfigured ):
      self.macVrfConfig = macVrfConfig
      self.macVrfMode = macVrfMode
      self.bgpConfig = bgpConfig
      self.asdotConfigured = asdotConfigured

   def saveMplsControlWord( self, cmds, options ):
      """mpls control-word"""
      if self.macVrfConfig.mplsControlWord:
         cmds.addCommand( 'mpls control-word' )
      elif options.saveAll:
         cmds.addCommand( 'no mpls control-word' )

   def saveFlowLabel( self, cmds, options ):
      """label flow"""
      if self.macVrfConfig.flowLabel:
         cmds.addCommand( 'label flow' )
      elif options.saveAll:
         cmds.addCommand( 'no label flow' )

   def saveMtu( self, cmds, options ):
      """mtu ( ignore | MTU )"""
      if self.macVrfConfig.pseudowireMtu != 0:
         cmds.addCommand( 'mtu {}'.format( self.macVrfConfig.pseudowireMtu ) )
      elif options.saveAll:
         cmds.addCommand( 'mtu ignore' )

   def saveFxcSignaling( self, cmds, options ):
      """flexible-cross-connect"""
      t0( 'flexible-cross-connect: macVrfConfig.fxcMode', self.macVrfConfig.fxcMode )
      if self.macVrfConfig.fxcMode == FxcMode.defaultFxc:
         t0( "flexible-cross-connect: saving as defaultFxc" )
         cmds.addCommand( 'flexible-cross-connect' )
      elif options.saveAll:
         t0( "flexible-cross-connect: saving as notFxc for saveAll" )
         cmds.addCommand( 'no flexible-cross-connect' )
      else:
         t0( "flexible-cross-connect: not saving" )

   def savePwEvpnVpwsId( self, cmds, options, pwConfig ):
      """evpn vpws id local LOCAL remote REMOTE"""
      if not ( pwConfig.vpwsIdLocal == EvpnEtid.max and
               pwConfig.vpwsIdRemote == EvpnEtid.max ):
         cmds.addCommand( 'evpn vpws id local {} remote {}'.format(
            pwConfig.vpwsIdLocal, pwConfig.vpwsIdRemote ) )
      elif options.saveAll:
         cmds.addCommand( 'no evpn vpws id' )

   @staticmethod
   def _saveOneOverride( cmds, options, direction, value ):
      status = {
         ControlWordOverride.cwDisabled: 'disabled',
         ControlWordOverride.cwAlways: 'always',
      }.get( value )

      cmd = 'mpls control-word data-plane {direction}'
      if status:
         cmds.addCommand( cmd.format( direction=direction ) + " " + status )
      elif options.saveAll:
         cmds.addCommand( 'no ' + cmd.format( direction=direction ) )

   def savePwControlWordDataPlaneOverride( self, cmds, options, pwConfig ):
      """mpls control-word data-plane (receive|transmit) (always|disabled)"""
      override = pwConfig.controlWordOverride
      self._saveOneOverride( cmds, options, 'receive', override.rxOverride )
      self._saveOneOverride( cmds, options, 'transmit', override.txOverride )

   def savePw( self, options, pwName ):
      """pseudowire NAME"""
      pwConfig = self.macVrfConfig.pseudowireConfig.get( pwName )
      if not pwConfig:
         return
      pwMode = self.macVrfMode[ BgpMacVrfVpwsPwSaveMode ].getOrCreateModeInstance(
            pwName )
      cmds = pwMode[ 'Bgp.macvrf.pw.config' ]
      self.savePwEvpnVpwsId( cmds, options, pwConfig )
      self.savePwControlWordDataPlaneOverride( cmds, options, pwConfig )

   def save( self, cmds, options ):
      self.saveMplsControlWord( cmds, options )
      if toggleVpwsFlowLabelToggleEnabled():
         self.saveFlowLabel( cmds, options )
      self.saveMtu( cmds, options )
      if toggleEvpnVpwsFxc1Enabled():
         self.saveFxcSignaling( cmds, options )
      for pwName in sorted( self.macVrfConfig.pseudowireConfig ):
         self.savePw( options, pwName )

def saveRouteTargetEvpnVpws( cmds, macVrfConfig, asdotConfigured ):
   # route-target evpn < import | export >
   # add route-target import or "import export"
   for rt in macVrfConfig.importRtList:
      if rt in macVrfConfig.exportRtList and rt in macVrfConfig.importExportRtList:
         direction = 'import export'
      else:
         direction = 'import'
      cmds.addCommand( 'route-target %s evpn %s' % (
         direction, routeTargetToPrint( rt, asdotConfigured ) ) )

   # add route-target export
   for rt in macVrfConfig.exportRtList:
      alreadyPrinted = ( rt in macVrfConfig.importRtList and
                         rt in macVrfConfig.importExportRtList )
      if not alreadyPrinted:
         cmds.addCommand( 'route-target export evpn %s' % (
            routeTargetToPrint( rt, asdotConfigured ) ) )

def saveMacVrfConfig( macVrfConfig, parentMode, bgpConfig, asdotConfigured,
                      options ):
   isVniMacVrf = macVrfConfig.isVniMacVrf()
   isVpwsMacVrf = macVrfConfig.isVpwsMacVrf()
   mode = parentMode[ BgpMacVrfConfigSaveMode ].getOrCreateModeInstance(
      ( macVrfConfig.name, macVrfConfig.isBundle, isVniMacVrf, isVpwsMacVrf ) )
   cmds = mode[ 'Bgp.macvrf.config' ]

   rd = macVrfConfig.rd
   if rd and rd != 'INVALID':
      cmds.addCommand( 'rd ' + formatRd( rd, asdotConfigured ) )
   if macVrfConfig.autoRd:
      cmds.addCommand( 'rd auto' )

   # EVPN VPWS MAC-VRFs use a different route-target syntax and so have a different
   # save helper method:
   # - extra "evpn" token
   # - no "both" (uses "import export" instead)
   if isVpwsMacVrf:
      saveRouteTargetEvpnVpws( cmds, macVrfConfig, asdotConfigured )
   else:
      saveRouteTarget( cmds, macVrfConfig, asdotConfigured )

   # redistribute < learned [remote] | static | dot1x |
   #                link-local ipv6 |
   #                router-mac [system | (next-hop vtep primary)] ) |
   #                host-route | all >
   if macVrfConfig.isRedistributeAll():
      cmds.addCommand( 'redistribute all' )
   else:
      if macVrfConfig.isInRedistribute( 'redistributeLearned' ):
         cmds.addCommand( 'redistribute learned' )
      if macVrfConfig.isInRedistribute( 'redistributeStatic' ):
         cmds.addCommand( 'redistribute static' )
      if macVrfConfig.isInRedistribute( 'redistributeDot1x' ):
         cmds.addCommand( 'redistribute dot1x' )
      if macVrfConfig.isInRedistribute( 'redistributeLinkLocal' ):
         cmds.addCommand( 'redistribute link-local ipv6' )
      if macVrfConfig.isInRedistribute( 'redistributeIgmp' ):
         cmds.addCommand( 'redistribute igmp' )
      if macVrfConfig.isInRedistribute( 'redistributeSysMac' ):
         cmds.addCommand( 'redistribute router-mac system' )
      if not macVrfConfig.isInRedistribute( 'redistributeHostRoute' ):
         cmds.addCommand( 'no redistribute host-route' )
      if not macVrfConfig.isInRedistribute( 'redistributeRouterMac' ):
         cmds.addCommand( 'no redistribute router-mac' )
      if macVrfConfig.isInRedistribute( 'redistributeRemoteLearned' ):
         cmds.addCommand( 'redistribute learned remote' )
      if macVrfConfig.isInRedistribute( 'redistributeRouterMacPrimary' ):
         if toggleRedistRouterMacPrimaryCliEnabled():
            cmds.addCommand( 'redistribute router-mac next-hop vtep primary' )

   if macVrfConfig.isVniMacVrf():
      if macVrfConfig.redistributeVcs:
         cmds.addCommand( 'redistribute service vxlan' )

   if macVrfConfig.isBundle:
      # vlan-aware-bundle configuration
      if isVniMacVrf:
         cmdStr = 'vni'
      else:
         cmdStr = 'vlan'
      if macVrfConfig.brIdToEtId:
         defaultEtidKeys = []
         nonDefaultEtidKeys = []
         for bdId in sorted( macVrfConfig.brIdToEtId.iterkeys() ):
            etid = macVrfConfig.brIdToEtId.get( bdId )
            if etid == 0:
               if macVrfConfig.isVniMacVrf():
                  defaultEtidKeys.append( bdId.vni )
               else:
                  defaultEtidKeys.append( bdId.vlanId )
            else:
               nonDefaultEtidKeys.append( bdId )
         rangeStr = MultiRangeRule.multiRangeToCanonicalString( defaultEtidKeys )
         if rangeStr:
            cmds.addCommand( '%s ' % ( cmdStr ) + rangeStr )
         for bdId in nonDefaultEtidKeys:
            etid = macVrfConfig.brIdToEtId[ bdId ]
            cmds.addCommand( '%s %s etid %s' % ( cmdStr, bdId.vlanId, etid ) )
   elif isVpwsMacVrf:
      # VPWS
      vpws = VpwsMacVrfConfigSaver( macVrfConfig, mode, bgpConfig, asdotConfigured )
      vpws.save( cmds, options )

   # maximum-routes configuration
   if macVrfConfig.maxRoutes != 0:
      cmds.addCommand( 'maximum-routes %d' % macVrfConfig.maxRoutes )
      
@CliSave.saver( 'Routing::Bgp::MacVrfConfigDir', 'routing/bgp/macvrf',
                requireMounts = ( 'routing/bgp/config',
                                  'routing/bgp/asn/config' ) )
def saveBgpMacVrfConfig( macVrfConfigDir, root, sysdbRoot, options, requireMounts ):
   bgpConfig = requireMounts[ 'routing/bgp/config' ]
   asnConfig = requireMounts[ 'routing/bgp/asn/config' ]
   if bgpConfig.asNumber == 0:
      return
   
   asdotConfigured = isAsdotConfigured( asnConfig )
   parentMode = root[ RouterBgpBaseConfigMode ].getOrCreateModeInstance( (
         bgpConfig.asNumber, asdotConfigured ) )
   for vrfConfig in macVrfConfigDir.config.itervalues():
      saveMacVrfConfig( vrfConfig, parentMode, bgpConfig, asdotConfigured, options )
