#!/usr/bin/env python
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

import CliSave
import Tac
from CliSavePlugin.RoutingBgpCliSave import (
      RouterBgpBaseConfigMode,
      RouterBgpVrfAfConfigMode,
      RouterBgpVrfConfigMode,
      saveAfConfigCallbackDict,
      )
from BgpLib import (
      bgpConfigAttrAfList,
      vpnAfTypes,
      vpnAfTypeMap,
      vpnAfToAfiSafi,
      )
from RouteMapLib import isAsdotConfigured
from Tracing import Handle

__defaultTraceHandle__ = Handle( 'DefaultExportSavePlugin' )

RouterBgpVrfConfigMode.addCommandSequence(
      'Bgp.vrf.default-route-export' )
RouterBgpVrfAfConfigMode.addCommandSequence(
      'Bgp.vrf.afConfig.default-route-export' )

def vpnAfiSafiAllowedInAddressFamily( vpnAfiSafi, addrFamily ):
   if addrFamily is None:
      return True
   elif addrFamily == 'ipv4':
      return vpnAfiSafi != vpnAfToAfiSafi(
                              Tac.Type( 'Routing::Bgp::VpnAf' ).vpnAfVpnV6 )
   elif addrFamily == 'ipv6':
      return vpnAfiSafi != vpnAfToAfiSafi(
                              Tac.Type( 'Routing::Bgp::VpnAf' ).vpnAfVpnV4 )
   return False

def saveDefaultExportConfig( bgpVrfConfig, addrFamily, options=None ):
   cmds = []
   saveAll = options.saveAll if options else False
   saveAllDetail = options.saveAllDetail if options else False
   if addrFamily is None:
      defaultExport = bgpVrfConfig.defaultExport
   elif addrFamily == 'ipv4':
      defaultExport = bgpVrfConfig.defaultExportAfV4
   elif addrFamily == 'ipv6':
      defaultExport = bgpVrfConfig.defaultExportAfV6
   else:
      return cmds

   for vpnAfiSafi in vpnAfTypes:
      defaultExportCfg = defaultExport.get( vpnAfiSafi )

      if defaultExportCfg is None:
         if saveAll or saveAllDetail:
            if vpnAfiSafiAllowedInAddressFamily( vpnAfiSafi, addrFamily ):
               cmds.append( 'no default-route export %s' %
                                 vpnAfTypeMap[ vpnAfiSafi ] )
         continue

      cmd = 'default-route export %s' % vpnAfTypeMap[ vpnAfiSafi ]
      if defaultExportCfg.always:
         cmd += ' always'
      if defaultExportCfg.routeMap:
         cmd += ' route-map %s' % defaultExportCfg.routeMap
      cmds.append( cmd )

   return cmds

saveAfConfigCallbackDict[ 'default-route-export' ] = saveDefaultExportConfig

@CliSave.saver( 'Routing::Bgp::VrfConfigDir', 'routing/bgp/vrf/config',
                requireMounts=( 'routing/bgp/config',
                                  'routing/bgp/vrf/config',
                                  'routing/bgp/asn/config' ) )
def saveDefaultExportAfConfig( vrfConfigDir, root, sysdbRoot, options,
                              requireMounts ):
   bgpConfig = requireMounts[ 'routing/bgp/config' ]
   asnConfig = requireMounts[ 'routing/bgp/asn/config' ]

   if bgpConfig.asNumber == 0:
      return

   bgpMode = root[ RouterBgpBaseConfigMode ].getOrCreateModeInstance( (
      bgpConfig.asNumber, isAsdotConfigured( asnConfig ), ) )

   def saveCmds( bgpVrfConfig, bgpVrfMode, vrf, cmds ):
      if cmds:
         for cmd in cmds:
            bgpVrfMode[ 'Bgp.vrf.config' ].addCommand( cmd )
      for af in bgpConfigAttrAfList():
         cmds = saveDefaultExportConfig( bgpVrfConfig, af, options )
         if cmds:
            bgpVrfAfMode = bgpVrfMode[ RouterBgpVrfAfConfigMode
                  ].getOrCreateModeInstance( ( af, vrf ) )
            for cmd in cmds:
               bgpVrfAfMode[ 'Bgp.vrf.afConfig.%s' % af ].addCommand( cmd )

   cmds = saveDefaultExportConfig( bgpConfig, None, options )
   if cmds: 
      bgpVrfMode = bgpMode[ RouterBgpVrfConfigMode ].getOrCreateModeInstance(
                                                      'default' )
      saveCmds( bgpConfig, bgpVrfMode, 'default', cmds )

   for vrf in vrfConfigDir.vrfConfig:
      bgpVrfConfig = vrfConfigDir.vrfConfig[ vrf ]
      cmds = saveDefaultExportConfig( bgpVrfConfig, None, options )
      if cmds:
         bgpVrfMode = \
            bgpMode[ RouterBgpVrfConfigMode ].getOrCreateModeInstance( vrf )
         saveCmds( bgpVrfConfig, bgpVrfMode, vrf, cmds )
