# Copyright (c) 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tac
import CliSave
import IpUtils
from IntfCliSave import IntfConfigMode
from IpLibConsts import DEFAULT_VRF
from RoutingIntfUtils import allIpIntfNames
from Toggles.ArpToggleLib import (
   toggleArpCacheCapacityEnabled,
)

IntfConfigMode.addCommandSequence( 'Arp.config',
      after=[ 'Ira.ipIntf' ] )
CliSave.GlobalConfigMode.addCommandSequence( 'Ira.arp', after=[ 'Ira.routes' ] )
CliSave.GlobalConfigMode.addCommandSequence( 'Ira.staticNeighbor6',
      after=[ IntfConfigMode ] )
tristateBool = Tac.Type( "Ip::TristateBool" )

defaultEthernetTimeout = Tac.Type( 'Arp::ArpIntfConfig' ).defaultEthernetTimeout

@CliSave.saver( 'Arp::Config', 'arp/config',
                requireMounts = ( 'interface/config/all', 'interface/status/all' ) )
def saveArpConfig( entity, root, sysdbRoot, options,
                   requireMounts ):
   if entity.restoreOnReboot:
      root[ 'Ira.arp' ].addCommand( "arp cache persistent" )

   if entity.ip6RestoreOnReboot:
      root[ 'Ira.staticNeighbor6' ].addCommand( 
         "ipv6 neighbor cache persistent" )

   if entity.ip4PersistanceEnabled:
      root[ 'Ira.arp' ].addCommand( 
            "arp persistent refresh-delay %d" % entity.refreshDelay )

   if entity.ip6PersistanceEnabled:
      root[ 'Ira.staticNeighbor6' ].addCommand( 
            "ipv6 neighbor persistent refresh-delay %d" % entity.refreshDelay )

   if entity.proxyMaxDelay != entity.proxyMaxDelayDefault:
      root[ 'Ira.arp' ].addCommand(
         "arp proxy max-delay %s" % entity.proxyMaxDelay )
   elif options.saveAll:
      root[ 'Ira.arp' ].addCommand(
         "arp proxy max-delay %s" % entity.proxyMaxDelayDefault )

   if entity.globalMonitorMac:
      root[ 'Ira.arp' ].addCommand( "arp monitor mac-address" )
   elif options.saveAll:
      root[ 'Ira.arp' ].addCommand( "no arp monitor mac-address" )
   
   if entity.globalArpTimeout != defaultEthernetTimeout:
      root[ 'Ira.arp' ].addCommand( 
            "arp aging timeout default %d" % entity.globalArpTimeout )
   elif options.saveAll:
      root[ 'Ira.arp' ].addCommand( "default arp aging timeout default" )

   if options.saveAllDetail:
      cfgIntfNames = allIpIntfNames( sysdbRoot, includeEligible=True,
                                     requireMounts=requireMounts )
   elif options.saveAll:
      cfgIntfNames = set(
            allIpIntfNames( sysdbRoot, requireMounts=requireMounts ) +
            entity.arpIntfConfig.keys() )
   else:
      cfgIntfNames = entity.arpIntfConfig.keys()
   for intfName in cfgIntfNames:
      intfConfig = entity.arpIntfConfig.get( intfName )
      if not intfConfig:
         if options.saveAll:
            intfConfig = Tac.newInstance( 'Arp::ArpIntfConfig', intfName )
         else:
            continue
      saveArpIntfConfig( intfConfig, root, sysdbRoot, options )

def saveArpIntfConfig( entity, root, sysdbRoot, options ):
   mode = root[ IntfConfigMode ].getOrCreateModeInstance( entity.name )
   cmds = mode[ 'Arp.config' ]
   if entity.timeoutV4:
      cmds.addCommand( "arp aging timeout %d" % entity.timeoutV4 )
   elif options.saveAll:
      cmds.addCommand( "default arp aging timeout" )
   if entity.arpCacheCapacity != entity.cacheCapacityDefault \
         and toggleArpCacheCapacityEnabled():
      cmds.addCommand( "arp cache dynamic capacity %d" % entity.arpCacheCapacity )
   elif options.saveAll and toggleArpCacheCapacityEnabled():
      cmds.addCommand( "default arp cache dynamic capacity" )
   if entity.timeoutV6:
      cmds.addCommand( "ipv6 nd cache expire %d" % entity.timeoutV6 )
   elif options.saveAll:
      cmds.addCommand( "default ipv6 nd cache expire" )
   if entity.nbrCacheCapacity != entity.cacheCapacityDefault \
         and toggleArpCacheCapacityEnabled():
      cmds.addCommand( "ipv6 nd cache dynamic capacity %d" %
                       entity.nbrCacheCapacity )
   elif options.saveAll and toggleArpCacheCapacityEnabled():
      cmds.addCommand( "default ipv6 nd cache dynamic capacity" )
   if entity.monitorMac == tristateBool.isTrue:
      cmds.addCommand( "arp monitor mac-address" )
   elif entity.monitorMac == tristateBool.isFalse:
      cmds.addCommand( "no arp monitor mac-address" )
   elif options.saveAll and entity.name.startswith( 'Vlan' ):
      cmds.addCommand( "default arp monitor mac-address" )
   if entity.proxyNdEnabled:
      if entity.proxyNdPrefixConnected:
         cmds.addCommand( "ipv6 nd proxy prefix connected" )
      for prefix in sorted( entity.proxyNdPrefix ):
         cmds.addCommand( "ipv6 nd proxy prefix %s" % prefix )
   elif options.saveAll and entity.name.startswith( "Vlan" ):
      cmds.addCommand( "no ipv6 nd proxy" )

@CliSave.saver( 'Arp::ArpInputVrfConfig', 'arp/input/config/cli',
                attrName = 'vrf' )
def saveVrfArpIntfConfig( entity, root, sysdbRoot, options ):
   saveStaticArpEntries( entity, root, sysdbRoot, options, entity.name )
   saveStaticIp6NeighEntries( entity, root, sysdbRoot, options, entity.name )

def saveStaticArpEntries( entity, root, sysdbRoot, options, vrfName ):
   arpCmd = 'arp'
   if vrfName != DEFAULT_VRF:
      arpCmd += ' vrf %s' % vrfName

   for ipAddr in sorted( entity.ipv4, cmp=IpUtils.compareIpAddress ):
      staticArp = entity.ipv4[ ipAddr ]
      cmd = "%s %s %s arpa" % ( arpCmd, ipAddr, staticArp.ethAddr )
      root[ 'Ira.arp' ].addCommand( cmd )

def saveStaticIp6NeighEntries( entity, root, sysdbRoot, options, vrfName ):
   snCmd = 'ipv6 neighbor'
   if vrfName != DEFAULT_VRF:
      snCmd += ' vrf %s' % vrfName

   for sn in sorted( entity.ipv6 ):
      neighbor = entity.ipv6[ sn ]
      cmd = '%s %s %s %s' % ( snCmd, neighbor.ip6Addr.stringValue, 
                              neighbor.intfId, neighbor.ethAddr )
      root[ 'Ira.staticNeighbor6' ].addCommand( cmd )

