# Copyright (c) 2015 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import os

import Tac
import CliExtensions
import CliSave

from AleFlexCounterTypes import (
   counterFeatureMap,
   counterFeatureConfigCommand,
   unitsCountModeToTokenMap,
)
from CliMode.AleCounters import (
   MonitorCounterMode,
   MonitorCounterEthernetMode,
   MonitorQueueCounterMode,
)

CliConfigConstants = Tac.Type( 'Ale::Counters::CliConfigConstants' )

CliSave.GlobalConfigMode.addCommandSequence( 'AleCounters' )

@CliSave.saver( 'Ale::Counters::InternalDrop::CliConfig',
                'hardware/counter/internalDrop/config/cli' )
def saveInternalDropCliConfig( entity, root, sysdbRoot, options ):
   syslogEventThresholds = sorted( entity.syslogEventThreshold.keys() )
   for counterName in syslogEventThresholds:
      root[ 'AleCounters' ].addCommand( 'hardware counter drop log %s %d'
            % ( counterName, entity.syslogEventThreshold[ counterName ] ) )

def flexCounterSaveCounterCliConfig( entity, root, sysdbRoot, options,
                                     requireMounts, fcFeatureStatusDir=None ):
   def addCommand( prefix, featureId, suffix="" ):
      if featureId == 'Route':
         return

      command = prefix + counterFeatureConfigCommand( featureId ) + suffix
      root[ "AleCounters" ].addCommand( command )

   if fcFeatureStatusDir is None:
      return

   defaultFeatAttr = Tac.Value( 'FlexCounters::FeatureAttributeState' )
   for featureId, featureStatus in fcFeatureStatusDir.entityPtr.iteritems():
      featureInfo = counterFeatureMap.get( featureId )
      if featureInfo is None or not featureInfo.autogenCliCfg:
         continue

      featureConfig = entity.feature.get( featureId )
      featureAttrConfig = entity.featureAttribute.get( featureId, defaultFeatAttr )
      if featureConfig is not None:
         if (
               # Feature is explicitly enabled.
               ( not featureStatus.defaultState and featureConfig == "enabled" ) or
               # Feature at default state, but all/allDetail requested.
               ( featureStatus.defaultState and
                 ( options.saveAll or options.saveAllDetail ) ) or
               # Feature at default state, but has a non-default featureAttribute.
               ( featureStatus.defaultState and
                 featureAttrConfig != defaultFeatAttr )
         ):
            featAttrSuffix = ""
            unitsTypeToken = unitsCountModeToTokenMap[ featureAttrConfig.countMode ]
            if unitsTypeToken is not None:
               featAttrSuffix += " units " + unitsTypeToken
            addCommand( "", featureId, featAttrSuffix )
      else:
         if featureStatus.defaultState or options.saveAllDetail:
            addCommand( "no ", featureId )

# Hook for platform featureStatusDir.
# The hook extension must be a callable such that:
# - it accepts the following arguments: `sysdbRoot`;
# - it returns the featureStatusDir.
# No more than 1 extension shall be registered at the same time.
counterFeatureStatusDirHook = CliExtensions.CliHook()

def getFeatureStatusDir( sysdbRoot ):
   extensions = counterFeatureStatusDirHook.extensions()
   if not extensions:
      return None
   if os.environ.get( "ALECOUNTERS_CLIHOOK_STRICT" ):
      assert len( extensions ) <= 1, "More than 1 hook registered"
   # Take the first extension.
   extension = next( iter( extensions ) )
   return extension( sysdbRoot )

@CliSave.saver( 'Ale::FlexCounter::FeatureConfigDir',
                'flexCounter/featureConfigDir/cliAgent' )
def saveCounterCliConfig( entity, root, sysdbRoot, options, requireMounts ):
   flexCounterSaveCounterCliConfig(
      entity, root, sysdbRoot, options, requireMounts,
      fcFeatureStatusDir=getFeatureStatusDir( sysdbRoot ) )

def flexCounterSaveRouteCounterCliConfig( entity, root, sysdbRoot,
                                          options, requireMounts ):
   def addCommand( routeType, vrfName, prefix ):
      if prefix is None:
         return

      command = None

      if vrfName == 'default':
         command = "hardware counter feature route %s %s" % ( routeType, prefix )
      else:
         command = "hardware counter feature route %s vrf %s %s" % (
                       routeType, vrfName, prefix )

      root[ "AleCounters" ].addCommand( command )

   for vrfNameAndPrefixObj in entity.route.keys():
      vrfName = vrfNameAndPrefixObj.vrfName
      prefixString = vrfNameAndPrefixObj.ipGenPrefix.stringValue
      routeType = vrfNameAndPrefixObj.ipGenPrefix.af

      addCommand( routeType, vrfName, prefixString )

#-----------------------------------------------------------------
# CliSave Plugin for monitor counter mode
#-----------------------------------------------------------------
class MonitorCounterConfigMode( MonitorCounterMode, CliSave.Mode ):
   def __init__( self, param ):
      MonitorCounterMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

#-----------------------------------------------------------------
# CliSave Plugin for ethernet interfaces mode
#-----------------------------------------------------------------
class MonitorCounterEthernetConfigMode( MonitorCounterEthernetMode, CliSave.Mode ):
   def __init__( self, param ):
      MonitorCounterEthernetMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

#-----------------------------------------------------------------
# CliSave Plugin for queue counters mode
#-----------------------------------------------------------------
class MonitorQueueCounterConfigMode( MonitorQueueCounterMode, CliSave.Mode ):
   def __init__( self, param ):
      MonitorQueueCounterMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

CliSave.GlobalConfigMode.addChildMode( MonitorCounterConfigMode )
MonitorCounterConfigMode.addChildMode( MonitorCounterEthernetConfigMode )
MonitorCounterEthernetConfigMode.addCommandSequence( 'AleCounters.intfconfig' )
MonitorCounterConfigMode.addChildMode( MonitorQueueCounterConfigMode )
MonitorQueueCounterConfigMode.addCommandSequence( 'AleCounters.queueconfig' )

def savablePollPeriod( config ):
   # Don't save if config.periodPoll in not in range. The only way to get to
   # this state is through Acons for debug purposes, and saving it breaks ASU
   return ( config.periodPoll != config.defaultPeriodPoll and
            config.periodPoll >= CliConfigConstants.minPeriodPoll and
            config.periodPoll <= CliConfigConstants.maxPeriodPoll )

@CliSave.saver( 'Ale::Counters::CliConfig', 'counter/global/intfconfig' )
def saveIntfGlobalConfig( config, root, sysdbRoot, options,
                      requireMounts ):
   savePollPeriod = savablePollPeriod( config )
   saveFastPollPeriod = config.periodFastPoll != config.defaultPeriodFastPoll
   
   # only create the mode if there is anything to save
   if savePollPeriod or saveFastPollPeriod:
      monMode = root[ MonitorCounterConfigMode ].getSingletonInstance()
      mode = monMode[ MonitorCounterEthernetConfigMode ].getSingletonInstance()

      cmds = mode[ 'AleCounters.intfconfig' ]

      # normal and fast polling are separate config, so only add non-defaults
      if savePollPeriod:
         cmds.addCommand( 'update interval %.1f' % config.periodPoll )
      if saveFastPollPeriod:
         cmds.addCommand( 'update fast-interval %d' % config.periodFastPoll )


@CliSave.saver( 'Ale::Counters::CliConfig', 'counter/global/queueconfig' )
def saveQueueGlobalConfig( config, root, sysdbRoot, options,
                      requireMounts ):
   # only create the mode if there is anything to save
   if savablePollPeriod( config ):
      monMode = root[ MonitorCounterConfigMode ].getSingletonInstance()
      mode = monMode[ MonitorQueueCounterConfigMode ].getSingletonInstance()

      cmds = mode[ 'AleCounters.queueconfig' ]
      cmds.addCommand( 'update interval %.1f' % config.periodPoll )
