# Copyright (c) 2006-2010, 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import CliSave, re, MultiRangeRule
from CliMode.Aaa import RoleMode
from AaaDefs import authzBuiltinRoles
import SecretCli

CliSave.GlobalConfigMode.addCommandSequence( 'Aaa.global' )
CliSave.GlobalConfigMode.addCommandSequence( 
   'Aaa.encrypt',
   # root password needs to be high-priority 
   before=[ 'config.priority' ],
   after=[ 'Aaa.global' ] )
CliSave.GlobalConfigMode.addCommandSequence( 'Aaa.users', after=[ 'Aaa.encrypt' ] )
CliSave.GlobalConfigMode.addCommandSequence( 'Aaa.netaccount' )

class RoleConfigMode( RoleMode, CliSave.Mode ):
   def __init__( self, param ):
      RoleMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

CliSave.GlobalConfigMode.addChildMode( RoleConfigMode, after=[ 'Aaa.global' ] )
RoleConfigMode.addCommandSequence( 'Aaa.role' )

# I can't use a specific CliSave.saver for Aaa::AuthenMethodList because I
# don't know from the type whether it's a login method list or an enable method
# list, so I do that from the main Aaa::Config saver.
@CliSave.saver( 'Aaa::Config', 'security/aaa/config' )
def saveAaaConfig( cfg, root, sysdbRoot, options ):
   cmds = root[ 'Aaa.global' ]
   saveAll = options.saveAll

   def _genAuthnCmd( ml, type ):
      m = ml.method
      s = " ".join( [ m[ i ] for i in range( 0, len( m ) ) ] )
      # 'login' is a special alias for 'console'
      name = ml.name if ml.name != 'login' else 'console'
      return "aaa authentication %s %s %s" %( type, name, s ) 

   # If the default login method list has been configured to have anything
   # other than "local" in its list, we need to have it show up in
   # running-config. 
   ml = cfg.defaultLoginMethodList
   if len( ml.method ) != 1 or ml.method[ 0 ] != "local" or saveAll:
      cmds.addCommand( _genAuthnCmd( ml, "login" ) )

   # Output the named login method lists
   if len( cfg.loginMethodList ) == 0:
      if saveAll:
         # there is only default login method
         cmds.addCommand( 'no aaa authentication login console' )
   else:
      for name, ml in cfg.loginMethodList.iteritems():
         cmds.addCommand( _genAuthnCmd( ml, "login" ) )

   # Similar to the default login method list, only stick a command in
   # running-config if the values differ from the defaults except if saveAll is set.
   ml = cfg.defaultEnableMethodList
   if len( ml.method ) != 1 or ml.method[ 0 ] != "local" or saveAll:
      cmds.addCommand( _genAuthnCmd( ml, "enable" ) )
      
   # Output dot1x method lists
   ml = cfg.defaultDot1xMethodList
   if len( ml.method ):
      cmds.addCommand( _genAuthnCmd( ml, "dot1x" ) )
      
   # aaa authentication policy logging on success/failure
   if cfg.loggingOnSuccess:
      cmds.addCommand( 'aaa authentication policy on-success log' )
   elif saveAll:
      cmds.addCommand( 'no aaa authentication policy on-success log' )

   if cfg.loggingOnFailure:
      cmds.addCommand( 'aaa authentication policy on-failure log' )
   elif saveAll:
      cmds.addCommand( 'no aaa authentication policy on-failure log' )

   # aaa authentication policy lockout failure <> [ window <> ] duration <>
   if cfg.lockoutTime:
      lockoutCmd = 'aaa authentication policy lockout %s'
      if cfg.lockoutWindow != cfg.lockoutWindowDefault or saveAll:
         lockAttrs = 'failure %d window %d duration %d' % ( cfg.maxLoginAttempts,
                                                            cfg.lockoutWindow,
                                                            cfg.lockoutTime )
         cmds.addCommand( lockoutCmd % lockAttrs )
      else:
         lockAttrs = 'failure %d duration %d' % ( cfg.maxLoginAttempts,
                                                  cfg.lockoutTime )
         cmds.addCommand( lockoutCmd % lockAttrs )
   elif saveAll:
      cmds.addCommand( 'no aaa authentication policy lockout' )

   # aaa authorization console
   if cfg.consoleAuthz != cfg.consoleAuthzDefault:
      cmds.addCommand( "aaa authorization serial-console" )
   elif saveAll:
      cmds.addCommand( "no aaa authorization serial-console" )

   def _genAuthzCmd( ml, type ):
      if len( ml.method ) == 1 and ml.method[ 0 ] == "none":
         if saveAll:
            return "no aaa authorization %s default" % type
         else:
            return
      m = ml.method
      s = " ".join( [ m[ i ] for i in range( 0, len( m ) ) ] )
      return "aaa authorization %s default %s" %( type, s )

   def _getCmdMethodLists( methods, field, areMlsEqualFunc ):
      cmdRe = re.compile( "command(\\d+)" )
      cmdMls = {}
      for name in sorted( methods.keys() ):
         ml = methods[ name ]
         match = re.match( cmdRe, name )
         if match is not None:
            level = int( match.group( 1 ), 10 )
            if level >= 0 and level <= 15:
               cmdMls[ level ] = ml

      commonLevels = []
      firstMl = None
      while cmdMls:
         firstLevel = cmdMls.keys()[ 0 ]
         firstMl = cmdMls[ firstLevel ]
         del cmdMls[ firstLevel ]
         commonLevels = [ firstLevel ]
         for level, ml in cmdMls.items():
            if areMlsEqualFunc( ml, firstMl ):
               commonLevels.append( level )
               del cmdMls[ level ]
         yield commonLevels, firstMl
      
   ml = cfg.authzMethod.get( 'exec' )
   if ml:
      cmd = _genAuthzCmd( ml, "exec" )
      if cmd:
         cmds.addCommand( cmd )

   for commonLevels, ml in \
          _getCmdMethodLists( cfg.authzMethod, 'method',
                              lambda ml1, ml2:
                                 ml1.method.items() == ml2.method.items() ):
          if len( commonLevels ) == 16:
             cmd = _genAuthzCmd( ml, "commands all" )
             if cmd:
                cmds.addCommand( cmd )
          else:
             levels = MultiRangeRule.multiRangeToCanonicalString( commonLevels )
             cmd = _genAuthzCmd( ml, "commands %s" % levels )
             if cmd:
                cmds.addCommand( cmd )
      
   if cfg.suppressConfigCommandAuthz:
      cmds.addCommand( "no aaa authorization config-commands" )
   elif saveAll:
      cmds.addCommand( "aaa authorization config-commands" )

   # aaa accounting
   def _actionToToken( action ):
      if action == 'startStop':
         return 'start-stop'
      elif action == 'stopOnly':
         return 'stop-only'
      else:
         assert False, 'invalid action!'
   
   def _genMethodsStr( methodMap, multicastMap ):
      """method names with optional 'multicast' keyword suffix"""
      maybeMulticast = lambda m: " multicast" if multicastMap.get( m ) else ""
      orderedMethods = ( methodMap[ k ] for k in sorted( methodMap ) )
      return " ".join( m + maybeMulticast( m ) for m in orderedMethods )

   def _genConsoleAcctCmd( ml, type ):
      if not ml.consoleUseOwnMethod:
         if saveAll:
            # generate default for console accounting not set.
            return 'no aaa accounting %s console' % type
         else:
            return
      if ml.consoleAction == 'none':
         return 'aaa accounting %s console none' % type
      methodsStr = _genMethodsStr( ml.consoleMethod, ml.consoleMethodMulticast )
      return 'aaa accounting %s console %s %s' % (
         type, _actionToToken( ml.consoleAction ), methodsStr )

   def _genDefaultAcctCmd( ml, type ):
      if ml.defaultAction == 'none':
         if saveAll:
            return 'no aaa accounting %s default' % type
         else:
            return
      methodsStr = _genMethodsStr( ml.defaultMethod, ml.defaultMethodMulticast )
      return 'aaa accounting %s default %s %s' % (
         type, _actionToToken( ml.defaultAction ), methodsStr )

   def _genAcctCmd( ml, type, service ):
      if service == 'console':
         if type != 'system' and type != 'dot1x':
            return _genConsoleAcctCmd( ml, type )
         else:
            return
      else:
         return _genDefaultAcctCmd( ml, type )

   def _compareAcctMls( ml1, ml2, service ):
      if service == 'console':
         return ml1.consoleUseOwnMethod == ml2.consoleUseOwnMethod and \
             ml1.consoleAction == ml2.consoleAction and \
             ml1.consoleMethod.items() == ml2.consoleMethod.items()
      return ml1.defaultAction == ml2.defaultAction and \
          ml1.defaultMethod.items() == ml2.defaultMethod.items()

   for service in ( 'console', 'default' ):
      ml = cfg.acctMethod.get( 'exec' )
      if ml:
         cmd = _genAcctCmd( ml, "exec", service )

         if cmd:
            cmds.addCommand( cmd )

      ml = cfg.acctMethod.get( 'system' )
      if ml:
         cmd = _genAcctCmd( ml, "system", service )
         
         if cmd:
            cmds.addCommand( cmd )

      ml = cfg.acctMethod.get( 'dot1x' )
      if ml:
         cmd = _genAcctCmd( ml, "dot1x", service )
         
         if cmd:
            cmds.addCommand( cmd )

      for commonLevels, ml in \
             _getCmdMethodLists( cfg.acctMethod, '%sMethod' % service,
                                 lambda ml1, ml2:
                                    _compareAcctMls( ml1, ml2, service ) ):
         if len( commonLevels ) == 16:
            cmd = _genAcctCmd( ml, "commands all", service )
            if cmd:
               cmds.addCommand( cmd )
         else:
            levels = MultiRangeRule.multiRangeToCanonicalString( commonLevels )
            cmd = _genAcctCmd( ml, "commands %s" % levels, service )
            if cmd:
               cmds.addCommand( cmd )

@CliSave.saver( 'LocalUser::Config', 'security/aaa/local/config' )
def saveLocalUserConfig( entity, root, sysdbRoot, options ):
   cmds = root[ 'Aaa.encrypt' ]
   saveAll = options.saveAll

   if entity.encryptedEnablePasswd != entity.encryptedEnablePasswdDefault:
      secretString = SecretCli.getSecretString( entity.encryptedEnablePasswd,
                                                options )
      if not secretString:
         cmds.addCommand( "% unknown hash algorithm for enable secret" )
         return
      cmd = 'enable password %s' % secretString
      cmds.addCommand( cmd )
   elif saveAll:
      # there is no enable password by default
      cmds.addCommand( 'no enable password' )

   # AaaSetPassword finds one of the following lines in startup-config
   # and sets root password accordingly. To avoid confusion with the content
   # of other commands (e.g., banner), we make sure that those aaa commands
   # are at the beginning of the running-config and always present (therefore
   # we generate 'no aaa root' even in the default case).
   aaaRootCommands = []
   if entity.encryptedRootPasswd != entity.encryptedRootPasswdDefault:
      if entity.encryptedRootPasswd == '':
         aaaRootCommands.append( 'aaa root nopassword' )
      else:
         secretString = SecretCli.getSecretString( entity.encryptedRootPasswd,
                                                   options )
         if not secretString:
            cmds.addCommand( "% unknown hash algorithm for root user" )
            return
         cmd = 'aaa root secret ' + secretString
         aaaRootCommands.append( cmd )
   
   if entity.rootSshKey != entity.rootSshKeyDefault:
      cmd = 'aaa root ssh-key ' + entity.rootSshKey
      aaaRootCommands.append( cmd )
   
   if not aaaRootCommands:
      aaaRootCommands.append( 'no aaa root' )

   for cmd in aaaRootCommands:
      cmds.addCommand( cmd )

   if entity.allowRemoteLoginWithEmptyPassword:
      cmds.addCommand(
         'aaa authentication policy local allow-nopassword-remote-login' )
   elif saveAll:
      # entity.allowRemoteLoginWithEmptyPassword is False by default
      cmds.addCommand(
         'no aaa authentication policy local allow-nopassword-remote-login' )

   if 'admin' not in entity.acct:
      cmds.addCommand( "no username admin" )

   # Add the SSH principals if set for root
   if entity.rootPrincipal:
      cmd = "username root ssh principal %s" % ( entity.rootPrincipal )
      cmds.addCommand( cmd )
   elif saveAll:
      cmds.addCommand( "no username root ssh principal" )

   # output username configuration in alphabetical order
   for user in sorted( entity.acct.values(), key=lambda u: u.userName ):
      saveUserAccount( user, root, sysdbRoot, options )

   # aaa authorization policy local default-role <role-name>
   c = "aaa authorization policy local default-role"
   if entity.defaultRole != entity.defaultRoleDefault:
      cmds.addCommand( c + " " + entity.defaultRole )
   elif saveAll:
      cmds.addCommand( "no " + c )

   # Roles
   for name in sorted( entity.role.keys() ):
      if not saveAll and name in authzBuiltinRoles:
         continue
      role = entity.role[ name ]
      mode = root[ RoleConfigMode ].getOrCreateModeInstance( name )
      cmds = mode[ 'Aaa.role' ]
      for seq, rule in role.rule.iteritems():
         modeKey = ' mode %s' % rule.modeKey if rule.modeKey != '' else ''
         cmd = '%d %s%s command %s' % ( seq, rule.action, modeKey, rule.regex )
         cmds.addCommand( cmd )

# These definitions an go away once we have constAttr access in Python
defaultPrivLevel = 1

def saveUserAccount( entity, root, sysdbRoot, options ):
   cmds = root[ 'Aaa.users' ]
   saveAll = options.saveAll

   # The "admin" account is special: it always exists, even if it has
   # not been configured. If it has not explicitly been configured, it
   # has a blank password, and does not show up in the running-config
   # or startup-config.
   if( entity.name == "admin" and entity.role == 'network-admin' and
         not entity.encryptedPasswd and entity.privilegeLevel == defaultPrivLevel
         and not entity.sshAuthorizedKey and not saveAll ):
      return
   cmd = "username %s" % entity.name
   if entity.privilegeLevel != entity.privilegeLevelDefault or saveAll:
      cmd += " privilege %d" % entity.privilegeLevel
   if entity.role != entity.roleDefault:
      cmd += " role " + entity.role
   if entity.shell:
      cmd += " shell " + entity.shell
   if entity.encryptedPasswd == '':
      cmd += " nopassword"
   else:
      secretString = SecretCli.getSecretString( entity.encryptedPasswd,
                                                options )
      if not secretString:
         cmds.addCommand( "% unknown hash algorithm for user %s" % entity.name )
         return
      cmd += " secret " + secretString
   cmds.addCommand( cmd )

   # Add the ssh key if it is set for this user
   if entity.sshAuthorizedKey:
      cmd = "username %s ssh-key %s" % ( entity.name, entity.sshAuthorizedKey )
      cmds.addCommand( cmd )

   # Add the SSH principals if set for this user
   if entity.principal:
      cmd = "username %s ssh principal %s" % ( entity.name, entity.principal )
      cmds.addCommand( cmd )

