# Copyright (c) 2006-2010, 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

from collections import namedtuple
import cPickle
import inspect
import os

import CliCommon
import CliSession as CS
import Plugins
import PyClient
import ShowRunOutputModel
import Tac
import Tracing

th = Tracing.defaultTraceHandle()
trace = th.trace0
traceTopSort = th.trace1
traceDetail = th.trace2

# Maps pathPrefix to a dict, which maps typename to a list of functions to be called
# to save each entity of type typename in the subtree rooted at pathPrefix.
saveHandlers_ = {}
noTypeSaveHandlers_ = []
# for secure CliSave
secureMonitorSaveHandlers_ = {}

# Marker for commands that we want to skip saving.
SKIP_COMMAND_MARKER = '\xff'

# Type to store handler related information.
TypeHandlerInfo = namedtuple( 'TypeHandlerInfo',
                              'pathPrefix attrName kind requireMounts' )
NoTypeHandlerInfo = namedtuple( 'NoTypeHandlerInfo',
                                'pathPrefix func kind requireMounts '
                                'requireActivityLock' )
# The 'showDeprecatedCmdAllowed' is going to be set to False in the future
# in which case running-config should stop showing deprecated commands.
showDeprecatedCmdAllowedDefault = True

ShowRunningConfigOptions = namedtuple( 'option', [ 'saveAll', 'saveAllDetail',
                                                   'showSanitized', 'showJson',
                                                   'showNoSeqNum',
                                                   'secureMonitor',
                                                   'showProfileExpanded',
                                                   'intfFilter',
                                                   # keep this the last
                                                   'showDeprecatedCmdAllowed' ] )
# the last ones (secureMonitor, showProfileExpanded, intfFilter, showDeprecated) have
# a default value
ShowRunningConfigOptions.__new__.__defaults__ = \
           ( False, False, None, showDeprecatedCmdAllowedDefault, )

def getSaverFunc( func ):
   # The saver functions may not handle the requireMounts parameter
   # (which is a shame, but I guess nobody wants to fix them all).
   # Add a wrapper function to take care of the inconsistency so
   # _callSaver() doesn't have to deal with it.
   if 'requireMounts' in inspect.getargspec( func ).args:
      return func

   # the function does not take requireMounts; let's just add a wrapper
   def saverWrapper( entity, root, sysdbRoot, options, requireMounts=None ):
      assert not requireMounts.requireMounts_
      return func( entity, root, sysdbRoot, options )

   return saverWrapper

def saver( typename, pathPrefix, attrName=None, kind=None, requireMounts=(),
           secureMonitor=False ):
   """A decorator that marks a function as being a CliSave function
   for entities of type 'typename' in the subtree rooted at the entity
   whose name (relative to the Sysdb root) is 'pathPrefix'.  If
   'attrName' is specified, only entities within the subtrees rooted
   at the collection attribute with this name are considered.
   'attrName' must be an instantiating collection attribute containing
   Entity pointers.

   A typical use might be like this:

     @CliSave.saver( 'Ip::VrfConfig', 'ip/config' )
     def saveVrfConfig( entity, root ):
        ...

   or like this:

      @CliSave.saver( 'Interface::EthIntfConfig', 'interface/config/ethernet',
                      attrName='ethIntfConfig' )
      def saveEthIntfConfig( entity, root ):
         ...

   The root argument passed to the saver is an instance of
   GlobalConfigMode.

   If 'kind' is specified then the saver is not invoked as part of the
   normal CliSave, but only when saveTree() is invoked with a matching
   'kind'.  This is used to allow a subtrees to be conditionally
   saved.

   If 'requireMounts' is specified then CliSave supplies the named entities
   to the saver via the optional dict argument also named 'requireMounts'.
   The saver get the entities via this dict, not via the sysdbRoot argument.

   This is useful when the saver needs information from other entities in Sysdb
   other that those under the primary path. For e.g. focalpoint saver requires
   'hardware/focalpoint/config' to check the platform type before saving focalpoint
   commands.

   If 'secureMonitor' is True, this is also used for generating secure-monitor config
   that does not show up in normal running-config.
   """
   assert pathPrefix
   assert not pathPrefix.startswith( '/' )  # Path prefixes must be relative.
   for mountPath in requireMounts:
      assert not mountPath.startswith( '/' ) # Path must be relative.

   def decorator( func ):
      # Register func as a handler for entities of type typename in the subtree
      # rooted at pathPrefix.
      typeHandlerInfo = TypeHandlerInfo( pathPrefix, attrName, kind, requireMounts )
      handlers = [ saveHandlers_ ]
      if secureMonitor:
         handlers.append( secureMonitorSaveHandlers_ )

      newFunc = getSaverFunc( func )

      for h in handlers:
         typeList = h.setdefault( typeHandlerInfo, {} ).setdefault( typename, [] )
         typeList.append( newFunc )
      return newFunc
   return decorator

def simpleSaver( path, kind=None, requireMounts=(), requireActivityLock=True ):
   """A decorator that marks a function as being a CliSave function
   for the entity at 'path'.

   A typical use might be like this:

     @CliSave.simpleSaver( 'hardware/config/diags' )
     def maybeSaveDiags( entity, root ):
        ...

   maybeSaveDiags will be called once, with the entity mounted at
   hardware/config/diags as the first argument and a GlobalConfigMode
   instance as the second argument.

   If 'kind' is specified then the saver is not invoked as part of the
   normal CliSave, but only when saveTree() is invoked with a matching
   'kind'.  This is used to allow a subtree to be conditionally saved.

   If 'requireMounts' is specified then CliSave supplies the named entities
   to the saver via the optional dict argument also named 'requireMounts'.
   The saver get the entities via this dict, not via the sysdbRoot argument.

   This is useful when the saver needs information from other entities in Sysdb
   other that those under the primary path. For e.g. focalpoint saver requires
   'hardware/focalpoint/config' to check the platform type before saving focalpoint
   commands.

   if 'requireActivityLock' is True the activity lock will be held while the save
   function is run, this prevents entities being updated while the saver is running.
   This flag is provided to allow a saver to perform operations (such as mounting)
   which require the activity lock not to be held.
   """
   assert path
   assert not path.startswith( '/' )  # Path prefixes must be relative.
   for mountPath in requireMounts:
      assert not mountPath.startswith( '/' ) # Path must be relative.

   def decorator( func ):
      newFunc = getSaverFunc( func )
      # Register func as a handler for the entity at path.
      if not path:
         print( "noType: empty path", func )
      noTypeHandlerInfo = NoTypeHandlerInfo( path, newFunc, kind, requireMounts,
                                             requireActivityLock )
      noTypeSaveHandlers_.append( noTypeHandlerInfo )
      return newFunc

   return decorator

_pluginsLoaded = False
_pluginsWanted = True

def pluginsWantedIs( w ):
   '''Pass True if you want me to load all plugins, or False if you plan to
   manually load them or otherwise provide the CliSave handlers yourself.'''
   global _pluginsWanted
   if w == _pluginsWanted:
      return
   if not w and _pluginsLoaded:
      raise Exception( "Too late, I already loaded plugins" )
   _pluginsWanted = w

def maybeLoadCliSavePlugins( entityManager ):
   # Load the CliSave plugins the first time this function is called.
   global _pluginsLoaded
   if not _pluginsLoaded and _pluginsWanted:
      # CliSavePlugins or IntfRangePlugins can call ConfigMount.mount()
      Plugins.loadPlugins( 'CliSavePlugin', context=entityManager )
      _pluginsLoaded = True

class EntityRoot( object ):
   ''' Provides a transparent access to entities in Sysdb. '''

   def __init__( self, entityManager ):
      self.em_ = entityManager
      self.sysdbRoot_ = entityManager.root()

   def __getitem__( self, path ):
      if self.em_.localEntityExists( path ):
         return self.em_.getLocalEntity( path )
      if path in self.sysdbRoot_.entity:
         return self.sysdbRoot_.entity[ path ]

      raise KeyError( path )

   def fullName( self, path ):
      if self.em_.localEntityExists( path ):
         return ''
      elif path in self.sysdbRoot_.entity:
         return self.sysdbRoot_.fullName

   def root( self ):
      return self.sysdbRoot_

def saveRunningConfig( entityManager, f, saveAll, saveAllDetail,
                       showSanitized=False, showJson=False, showNoSeqNum=False,
                       errorObj=None,
                       secureMonitor=False,
                       showDeprecatedCmdAllowed=showDeprecatedCmdAllowedDefault,
                       showProfileExpanded=False, intfFilter=None,
                       showHeader=True ):
   """Saves the running-config to the file-like object f."""
   entityRoot = EntityRoot( entityManager )
   saveRunningConfigInternal( entityManager, entityRoot, f,
                              saveAll, saveAllDetail,
                              showSanitized=showSanitized,
                              showJson=showJson,
                              showNoSeqNum=showNoSeqNum,
                              errorObj=errorObj,
                              secureMonitor=secureMonitor,
                              showDeprecatedCmdAllowed=showDeprecatedCmdAllowed,
                              showProfileExpanded=showProfileExpanded,
                              intfFilter=intfFilter,
                              showHeader=showHeader )

def closeFile( f, errorObj ):
   try:
      f.close()
   except ( EOFError, EnvironmentError ), e:
      trace( 'Exception while closing file', e )
      if errorObj:
         cPickle.dump( e, errorObj )
      # re-raise the exception so the client receives an RpcError
      raise

def saveRunningConfigInternal( entityManager, entityRoot, f,
                               saveAll, saveAllDetail, showSanitized=False,
                               showJson=False, showNoSeqNum=False,
                           showDeprecatedCmdAllowed=showDeprecatedCmdAllowedDefault,
                               errorObj=None,
                               secureMonitor=False,
                               showProfileExpanded=False,
                               intfFilter=None,
                               showHeader=True ):
   options = ShowRunningConfigOptions( saveAll, saveAllDetail, showSanitized,
                                       showJson, showNoSeqNum,
                                       secureMonitor,
                                       showProfileExpanded,
                                       intfFilter,
                                       showDeprecatedCmdAllowed )
   trace( 'options:%r', options )

   maybeLoadCliSavePlugins( entityManager )

   if secureMonitor:
      handlers = secureMonitorSaveHandlers_
   else:
      handlers = saveHandlers_

   configRootFilter = CS.activeConfigRootFilter( entityManager )

   # Extract the set of commands to save.
   with Tac.ActivityLockHolder():
      root = GlobalConfigMode()
      for typeHandlerInfo, typeHandlers in sorted( handlers.iteritems() ):
         if typeHandlerInfo.kind is not None:
            continue
         if ( configRootFilter and
              configRootFilter.rootTrie.hasPrefix( typeHandlerInfo.pathPrefix ) ):
            continue
         requireMountsDict = RequireMountsDict( typeHandlerInfo.requireMounts,
                                                entityManager, entityRoot )
         e = entityRoot[ typeHandlerInfo.pathPrefix ]
         traceDetail( 'saving tree of', e )
         _saveTree( e, root, typeHandlers, typeHandlerInfo.attrName, entityRoot,
                    options, typeHandlerInfo.requireMounts, requireMountsDict )
   for noTypeHandlerInfo in noTypeSaveHandlers_:
      if noTypeHandlerInfo.kind is not None:
         continue
      if ( configRootFilter and
           configRootFilter.rootTrie.hasPrefix( noTypeHandlerInfo.pathPrefix ) ):
         continue
      requireMountsDict = RequireMountsDict( noTypeHandlerInfo.requireMounts,
                                             entityManager, entityRoot )
      pathPrefix = noTypeHandlerInfo.pathPrefix
      e = entityRoot[ pathPrefix ] if pathPrefix else entityRoot
      if noTypeHandlerInfo.requireActivityLock:
         with Tac.ActivityLockHolder():
            noTypeHandlerInfo.func( e, root, None, options,
                                    requireMounts=requireMountsDict )
      else:
         noTypeHandlerInfo.func( e, root, None, options,
                                 requireMounts=requireMountsDict )

   # We special case file system errors to pass a more meaningful error.
   # Any other exception is going end up as an RpcError which will
   # result on an Internal Error
   try:
      if showJson:
         cliModel = ShowRunOutputModel.Mode()
         trace( 'cliModel:%r', cliModel )
         param = SaveParam( None, entityRoot, entityRoot,
                            showProfileExpanded=showProfileExpanded,
                            showHeader=showHeader )
         root.write( param, cliModel=cliModel )
         cPickle.dump( cliModel, f )
         f.flush()
      else:
         # Write the commands to the file-like object.
         trace( 'Writing running-config to', f )
         param = SaveParam( f, entityRoot, entityRoot,
                            showProfileExpanded=showProfileExpanded,
                            showHeader=showHeader )
         root.write( param )
   except ( EOFError, EnvironmentError ), e:
      trace( 'Exception while writing CliSave output', e )
      if errorObj:
         cPickle.dump( e, errorObj )
      # re-raise the exception so the client receives an RpcError
      raise

def saveSessionConfigInternal( entityManager, sessionName, sessionRoot,
                               sysdbRoot, f,
                               saveAll, saveAllDetail, showSanitized=False,
                               cleanConfig=False, showJson=False,
                               showNoSeqNum=False,
                           showDeprecatedCmdAllowed=showDeprecatedCmdAllowedDefault,
                               errorObj=None,
                               showProfileExpanded=False ):
   """Saves the config of the session sessionName to the file-like object f."""
   options = ShowRunningConfigOptions( saveAll, saveAllDetail, showSanitized,
                                       showJson, showNoSeqNum,
                                       False, # secureMonitor always False
                                       showProfileExpanded,
                                       None,
                                       showDeprecatedCmdAllowed )
   trace( 'Saving session-config of %s rooted at %s' % ( sessionName,
                                                         sessionRoot ) )
   maybeLoadCliSavePlugins( entityManager )
   if cleanConfig:
      sStatus = None
   else:
      sStatus = CS.sessionStatus.pendingChangeStatusDir.status.get( sessionName )

   sysdbRoot = entityManager.root()
   entityRoot = EntityRoot( entityManager )

   # Generate the mapping for the registered synthetic requireMount generators.
   traceDetail( "generating synthetic requireMount map" )
   syntheticRequireMountMap = _generateSyntheticRequireMountMap( entityRoot,
                                                                 sessionRoot,
                                                                 sStatus,
                                                                 cleanConfig )

   root = None
   traceDetail( "constructing session config" )
   # Extract the set of commands to save.
   with Tac.ActivityLockHolder():
      root = GlobalConfigMode()
      for typeHandlerInfo, typeHandlers in sorted( saveHandlers_.iteritems() ):
         if typeHandlerInfo.kind is not None:
            continue
         requireMountsDict = SessionRequireMountsDict( typeHandlerInfo.requireMounts,
                                                       entityManager,
                                                       entityRoot, sessionRoot,
                                                       sStatus, cleanConfig,
                                                       syntheticRequireMountMap )
         pathPrefix = typeHandlerInfo.pathPrefix
         e = getSessionEntity( pathPrefix, sessionRoot, entityRoot,
                               sStatus, cleanConfig )
         _saveTree( e, root, typeHandlers, typeHandlerInfo.attrName, entityRoot,
                    options, typeHandlerInfo.requireMounts, requireMountsDict )

   for noTypeHandlerInfo in noTypeSaveHandlers_:
      if noTypeHandlerInfo.kind is not None:
         continue
      requireMountsDict = SessionRequireMountsDict( noTypeHandlerInfo.requireMounts,
                                                    entityManager,
                                                    entityRoot, sessionRoot,
                                                    sStatus, cleanConfig,
                                                    syntheticRequireMountMap )
      pathPrefix = noTypeHandlerInfo.pathPrefix
      e = getSessionEntity( pathPrefix, sessionRoot, entityRoot,
                            sStatus, cleanConfig )
      if noTypeHandlerInfo.requireActivityLock:
         with Tac.ActivityLockHolder():
            noTypeHandlerInfo.func( e, root, None, options,
                                    requireMounts=requireMountsDict )

      else:
         noTypeHandlerInfo.func( e, root, None, options,
                                 requireMounts=requireMountsDict )

   # Same as with saveRunningConfigInternal we special case file system errors
   # to pass a more meaningful error. Any other exception is going end up as
   # an RpcError which will result on an Internal Error
   try:
      if showJson:
         cliModel = ShowRunOutputModel.Mode()
         trace( 'cliModel:%r', cliModel )
         param = SaveParam( None, sessionRoot, entityRoot, sStatus=sStatus,
                            cleanConfig=cleanConfig,
                            showProfileExpanded=showProfileExpanded )
         root.write( param, cliModel=cliModel )
         cPickle.dump( cliModel, f )
         f.flush()
      else:
         # Write the commands to the file-like object.
         trace( 'Writing session-config to', f )
         param = SaveParam( f, sessionRoot, entityRoot, sStatus=sStatus,
                            cleanConfig=cleanConfig,
                            showProfileExpanded=showProfileExpanded )
         root.write( param )
   except ( EOFError, EnvironmentError ), e:
      trace( 'Exception while writing saveSessionConfigInternal output', e )
      if errorObj:
         cPickle.dump( e, errorObj )
      # re-raise the exception so the client receives an RpcError
      raise

def saveRunningConfigRemote( pyClient, saveAll, saveAllDetail,
                             showSanitized=False, showJson=False,
                             showNoSeqNum=False, dstFile=None,
                             intfFilter=None ):
   '''Do saveRunningConfig by PyClient. This is much faster than having to
   mount everything. It returns the text of the running-config.
   '''
   if dstFile:
      # If the caller passed us a file might as well use it to copy the output
      # there and avoid copying things to a cStringIo buffer, pass via a socket
      fileIO = "f = file( '%s', 'wb' )\n" % ( dstFile )
   else:
      fileIO = "f = cStringIO.StringIO()\n"

   # On the product, ConfigAgent generates the running-config. Since a request
   # to generate the running-config comes from the ConfigAgent itself, it make a
   # local call to generate the config. The non cohabiting tests call this
   # method to generate the running config.

   # This method is called only in the tests
   assert 'A4_CHROOT' in os.environ

   if pyClient.agent_ == 'ConfigAgent':
      # We assume the agent is already warm
      assert pyClient.agentRoot()[ pyClient.agent_ ].warm
      em = ( "import Tac, ConfigAgent\n"
             "Tac.setproctitle( 'show-running' )\n"
             "with file( '/proc/self/oom_score_adj', 'w' ) as oomf:\n"
             "   oomf.write( '0' )\n"
             "em = ConfigAgent.__myEntityManager__\n" )
   else:
      em = ( "import Sysdb, LazyMount\n"
             "LazyMount.directModeIs( True )\n"
             "em = Sysdb.container.entityManager()\n" )

   code = ( "import CliSave, cStringIO\n" + fileIO + em +
            "errorObj = cStringIO.StringIO()\n"
            "CliSave.saveRunningConfig( em, f, %s, %s, %s, %s, %s, errorObj, "
            "intfFilter=%r )\n"
            % ( saveAll, saveAllDetail, showSanitized, showJson, showNoSeqNum,
                intfFilter ) )
   try:
      pyClient.execute( code )

      if dstFile:
         return pyClient.eval( 'CliSave.closeFile( f, errorObj )' )
      else:
         return pyClient.eval( 'f.getvalue()' )
   except PyClient.RpcError:
      error = pyClient.eval( 'errorObj.getvalue()' )
      if error:
         # if we captured an exception raise it
         excp = cPickle.loads( error )
         raise excp
      else:
         # if we haven't captured and exception then just reraise
         # the RpcError
         raise

def sanitizedOutput( options, originalString ):
   if options and options.showSanitized:
      return '<removed>'
   else:
      return originalString

def _saveTree( entity, root, typeHandlers, attrName, sysdbRoot,
               options, requireMounts, requireMountsDict ):
   """Saves the state of the instantiating subtree rooted at 'entity' by calling the
   handler functions in 'typeHandlers[ type ]' for each entity in the tree.  We do a
   preorder traversal of the tree."""

   walk = Tac.newInstance( "Cli::CliSaveWalk" )
   if attrName:
      # EntityWalk with an attribute filter will not search down an
      # attribute unless it's an instantiating Entity attribute.  We
      # should not be looking a non-instantiating attribute
      # instantiating anyway -- we should go to the instantiating
      # collection directly and walk it.
      a = entity.tacType.attr( attrName )
      assert a.instantiating and a.memberType.isEntity
      walk.attribute = attrName

   for k in typeHandlers:
      walk.handledType[ k ] = True

   walk.root = entity

   for entity in walk.match.itervalues():
      handlers = typeHandlers.get( entity.tacType.fullTypeName )
      for func in handlers:
         traceDetail( 'Calling handler function', func, 'on entity', entity )
         func( entity, root, None, options,
               requireMounts=requireMountsDict )

   traceDetail( 'Done saving', entity )

def saveTree( entity, root, kind, sysdbRoot, options,
              requireMountsDict,
              attrName=None,
              requireMounts=() ):
   """Save the tree rooted at 'entity' explicitly.  Only the
   typeHandlers registered for *exactly* the provided entity, attrName
   and kind are invoked."""

   if requireMountsDict:
      rootPath = requireMountsDict.getRootPathname( entity )
   else:
      rootPath = sysdbRoot.fullName
   path = entity.fullName.replace( rootPath + "/", "", 1 )
   typeHandlerInfo = TypeHandlerInfo( path, attrName, kind, requireMounts )
   typeHandlers = saveHandlers_.get( typeHandlerInfo )
   requireMountsDict = RequireMountsDict( typeHandlerInfo.requireMounts,
                                          requireMountsDict.getEntityManager(),
                                          requireMountsDict.getRoot() )
   if typeHandlers:
      _saveTree( entity, root, typeHandlers, attrName, sysdbRoot, options,
                 typeHandlerInfo.requireMounts, requireMountsDict )
   for noTypeHandlerInfo in noTypeSaveHandlers_:
      if path == noTypeHandlerInfo.pathPrefix and kind == noTypeHandlerInfo.kind:
         noTypeHandlerInfo.func( entity, root, None, options,
                                 requireMounts=requireMountsDict )

class RequireMountsDict( object ):
   """ Maps decorator-specified requireMounts paths
   to the corresponding entities. An instance is passed
   to savers as the optional kwarg 'requireMounts'. """

   def __init__( self, requireMounts, entityManager, entityRoot ):
      self.requireMounts_ = requireMounts
      self.entityManager_ = entityManager
      self.entityRoot_ = entityRoot

   def __getitem__( self, requireMount ):
      if requireMount not in self.requireMounts_:
         raise Exception, "attempt to access undeclared requireMount " + requireMount

      # check if the path exists in the entityManager local coll
      if self.entityManager_.localEntityExists( requireMount ):
         return self.entityManager_.getLocalEntity( requireMount )

      return self.entityRoot_[ requireMount ]

   def getRootPathname( self, entity ):
      rootPath = self.entityRoot_.root().fullName
      assert entity.fullName.startswith( rootPath )
      return rootPath

   def getEntityManager( self ):
      return self.entityManager_

   def getRoot( self ):
      return self.entityRoot_

class SessionRequireMountsDict( RequireMountsDict ):
   """ Maps decorator-specified requireMounts paths
   to the corresponding entities, taking into account
   the session configuration. An instance is passed
   to savers as the optional kwarg 'requireMounts'. """

   def __init__( self, requireMounts,
                 entityManager, entityRoot,
                 sessionRoot,
                 sStatus, cleanConfig,
                 syntheticRequireMountMap ):
      self.sessionRoot_ = sessionRoot
      self.sStatus_ = sStatus
      self.cleanConfig_ = cleanConfig
      self.syntheticRequireMountMap_ = syntheticRequireMountMap
      RequireMountsDict.__init__( self, requireMounts, entityManager, entityRoot )

   def __getitem__( self, requireMount ):
      if requireMount not in self.requireMounts_:
         raise Exception, "attempt to access undeclared requireMount " + requireMount

      # Check for a synthetic requireMount mapping.
      if requireMount in self.syntheticRequireMountMap_:
         return self.syntheticRequireMountMap_[ requireMount ]

      # check if the path exists in the entityManager local coll
      if self.entityManager_.localEntityExists( requireMount ):
         return self.entityManager_.getLocalEntity( requireMount )

      return getSessionEntity( requireMount, self.sessionRoot_, self.entityRoot_,
                               self.sStatus_, self.cleanConfig_ )

   def getRootPathname( self, entity ):
      # There is only one session root unlike 'Sysdb' root
      return self.sessionRoot_.fullName

def registerSyntheticRequireMountGenerator(
      requireMount, func ):
   """ CLI Save modules use this to register a function
   for synthesizing a require mount object. These are
   necessary in a config session for objects normally
   produced by reactors in other agents. The func
   argument is called with three arguments:
   func( sysdbRoot, sessionRoot, pathHasSessionPrefix ).
   The last is a predicate on an entity path name.
   If it is true, use the session entity (if any),
   not the sysdb entity. """

   syntheticRequireMountGenerators_[ requireMount ] = func

syntheticRequireMountGenerators_ = {}

def _generateSyntheticRequireMountMap( sysdbRoot,
                                       sessionRoot,
                                       sStatus,
                                       cleanConfig ):
   """ This is called by saveSessionConfigInternal,
   given to instances of SessionRequireMountsDict. """

   synthMap = {}
   for ( requireMount, func ) in syntheticRequireMountGenerators_.iteritems():
      synthMap[ requireMount ] = func( sysdbRoot,
                                       sessionRoot,
                                       lambda path:
                                             _pathHasSessionPrefix(
                                                   path,
                                                   sStatus,
                                                   cleanConfig ) )
   return synthMap

def getSessionEntity( path, sessionRoot, entityRoot,
                      sStatus, cleanConfig ):
   """ Given an entity path, return the entity appropriate to
   the session state. """

   if path:
      hasRoot = _pathHasSessionPrefix( path, sStatus, cleanConfig )
      if hasRoot:
         e = sessionRoot.entity[ path ]
      else:
         e = entityRoot[ path ]
   else:
      e = sessionRoot
   return e

def _pathHasSessionPrefix( path, sStatus, cleanConfig ):
   """ Consult the session trie to see if it has
   a prefix for the given path. The trie may be null,
   implying no prefixes. Special-case if clean config,
   using the presence of the path in cleanConfigRoot
   as a predicate. """

   if cleanConfig:
      return CS.sessionStatus.cleanConfigStatus.cleanConfigRoot.get( path )
   if sStatus and sStatus.localPrefix:
      return sStatus.localPrefix.hasPrefix( path )
   return False

#------------------------------------------------------------------------------------
# The model here is that the CLI save output comprises a set of SaveBlocks, of two
# kinds: CommandSequences and ModeCollections.  A CommandSequence comprises an
# ordered sequence of CLI commands.  A ModeCollection comprises a set of Mode
# instances, each of which recursively contains a set of SaveBlocks.
#
# SaveBlocks may have dependencies between them which are used to compute the order
# in which they are output.
#
# For examples of how to write a CliSave plugin, see AID95.
#------------------------------------------------------------------------------------
class SaveBlock( object ):
   """Abstract baseclass for all SaveBlocks, the basic unit from which the CLI save
   output is produced."""

   def write( self, param, prefix, cliModel=None ):
      """ Write out the commands in this SaveBlock. prefix parameter is used
      to achieve whitespace indentation.
      configRoot should be used to access config entities and SysdbRoot for
      everything else. While in config session configRoot points to sessionRoot
      otherwise it's same as sysdbRoot """
      raise NotImplementedError

   def empty( self, param ):
      """ Returns True if there are no commands in this SaveBlock."""
      raise NotImplementedError

   def content( self, other, param ):
      """The content function returns a tuple to facilitate the range merge
      feature where multiple instances of a certain config mode (e.g., VLAN)
      can be displayed as a range if they have identical content."""
      raise NotImplementedError

   def separator( self ):
      """Whether to print a '!' separator between save blocks."""
      return False

class SaveParam( object ):
   __slots__ = ( 'stream', 'sessionRoot', 'sysdbRoot', 'sStatus', 'cleanConfig',
                 'cliConfig', 'showProfileExpanded', 'showHeader' )

   def __init__( self, stream, sessionRoot, sysdbRoot, sStatus=None,
                 cleanConfig=False, showProfileExpanded=False,
                 showHeader=True ):
      self.stream = stream
      self.sessionRoot = sessionRoot
      self.sysdbRoot = sysdbRoot
      self.sStatus = sStatus
      self.cleanConfig = cleanConfig
      # this is used for CLI comments - just initialize it here
      self.cliConfig = getSessionEntity( "cli/config",
                                         sessionRoot,
                                         sysdbRoot,
                                         sStatus,
                                         cleanConfig )
      # Used for checking if we should expand the profile config for interfaces
      # or show the processed version
      self.showProfileExpanded = showProfileExpanded
      self.showHeader = showHeader

class CommandSequence( SaveBlock ):
   """A SaveBlock that comprises a simple ordered sequence of CLI commands.  Commands
   may be added to the sequence via the 'addCommand' method, or via one of the
   convenience methods 'writeAttr' or 'writeBoolAttr'.

   This class should not be instantiated directly by plugins; instead, a command
   sequence name should be registered with a Mode subclass by calling
   addCommandSequence( name ) on that Mode subclass, and then a CommandSequence
   object should be obtained by doing mode[ name ] on an instance of that Mode
   subclass."""

   def __init__( self ):
      SaveBlock.__init__( self )
      self.commands_ = []

   def write( self, param, prefix, cliModel=None ):
      stream = param.stream
      for c in self.commands_:
         if c != SKIP_COMMAND_MARKER:
            if stream:
               stream.write( prefix )
               stream.write( c )
               stream.write( '\n' )
            else:
               cliModel.cmds[ c ] = None

   def addCommand( self, command ):
      # make sure people don't inadvertently create empty lines by adding a \n to
      # their commands, but do allow multiline commands that have "inner carriage
      # returns" in json output format (like 'banner motd' or capi's certificates)
      assert not command[ -1 : ] == '\n'
      self.commands_.append( command )

   def empty( self, param ):
      return self.commands_ == []

   def content( self, param ):
      return tuple( self.commands_ )

   def writeAttr( self, entity, attr, command ):
      """Writes an attribute using the syntax '<command> <value>', provided that the
      value is not equal to the default."""

      val = getattr( entity, attr )
      default = getattr( entity, attr + "Default" )
      if val != default:
         self.addCommand( "%s %s" % ( command, val ) )

   def writeBoolAttr( self, entity, attr, command ):
      """Writes an boolean-valued attribute using the syntax '[no] <command>',
      provided that the value is not equal to the default.  The command is prefixed
      with 'no' if the value is False.  In other words:

      If the default is False:
        -  if the value is False, nothing is written.
        -  if the value is True, '<command>' is written.

      If the default is True:
        -  if the value is False, 'no <command>' is written.
        -  if the value is True, nothing is written."""

      val = getattr( entity, attr )
      default = getattr( entity, attr + "Default" )
      assert ( default is True ) or ( default is False )
      if val != default:
         if val == False:
            self.addCommand( 'no ' + command )
         else:
            self.addCommand( command )

class ModeCollection( SaveBlock ):
   """A SaveBlock that comprises a set of Mode instances for a particular CLI
   mode.

   This class should not be instantiated directly by plugins; instead, a child Mode
   subclass should be registered with a parent Mode subclass by calling
   addChildMode( clazz ) on that parent Mode subclass, and then a ModeCollection
   object should be obtained by doing mode[ clazz ] on an instance of that parent
   Mode subclass."""

   def __init__( self, modeClass ):
      assert issubclass( modeClass, Mode )
      SaveBlock.__init__( self )
      self.modeClass_ = modeClass
      self.modeInstanceMap_ = {}

   def writeRange( self, param, prefix, cliModel=None ):
      # Merge modes with the same block together.
      # contentMap maps content to the first mode with unique commands.
      contentMap = {}
      # modeMap maps the first mode with unique commands to a list of
      # modes with identical content.
      modeMap = {}
      for i in sorted( self.modeInstanceMap_.values() ):
         if i.hideInactive( param ):
            continue
         if i.hideUnconnected( param ):
            continue
         if i.empty( param ):
            continue
         if not i.canMergeRange():
            modeMap[ i ] = None
            continue
         content = i.content( param )
         mode = contentMap.get( content )
         if mode:
            # we can merge
            modeMap[ mode ].append( i )
         else:
            contentMap[ content ] = i
            modeMap[ i ] = [ i ]

      stream = param.stream
      firstTime = True
      for i in sorted( modeMap.keys() ):
         if stream and not firstTime:
            # Write a newline to separate this instance of the Mode subclass
            # from the previous instance in the ModeCollection.
            stream.write( prefix + '!\n' )
         firstTime = False

         m = modeMap[ i ]
         if m:
            enterCmd = i.enterRangeCmd( m )
         else:
            enterCmd = i.enterCmd()
         i.write( param, prefix, cliModel=cliModel, enterCmd=enterCmd )

   def write( self, param, prefix, cliModel=None ):
      if self.modeClass_.mergeRange and len( self.modeInstanceMap_ ) > 1:
         return self.writeRange( param, prefix, cliModel=cliModel )

      stream = param.stream
      firstTime = True
      prevNeedSeparator = False
      for i in sorted( self.modeInstanceMap_.values() ):
         if i.hideInactive( param ):
            continue
         if i.hideUnconnected( param ):
            continue
         if i.empty( param ):
            continue
         needSeparator = i.modeSeparator()
         if stream and not firstTime and ( prevNeedSeparator or needSeparator ):
            # Write a newline to separate this instance of the Mode subclass
            # from the previous instance in the ModeCollection.
            stream.write( prefix + '!\n' )
         firstTime = False
         prevNeedSeparator = needSeparator
         i.write( param, prefix, cliModel=cliModel )

   def empty( self, param ):
      """The ModeCollection is empty (no commands) if all Modes in the collection
      are empty. Note that currently, Modes are never empty, since they always
      consist of at least their enterCmd(), but it felt wrong relying on that."""

      for i in self.modeInstanceMap_.itervalues():
         if not i.empty( param ):
            return False
      return True

   def getOrCreateModeInstance( self, param ):
      if not self.modeInstanceMap_.has_key( param ):
         if param is not None:
            assert None not in self.modeInstanceMap_, \
               "singleton instance has to use getSingletonInstance()"
         self.modeInstanceMap_[ param ] = self.modeClass_( param )
      return self.modeInstanceMap_[ param ]

   def getSingletonInstance( self ):
      # This is a special case for singleton modes that don't have keys
      # We just use a fixed key.
      if ( len( self.modeInstanceMap_ ) == 1 and
           self.modeInstanceMap_.keys()[ 0 ] is not None or
           len( self.modeInstanceMap_ ) > 1 ):
         assert False, "singleton instance has to use getSingletonInstance()"
      return self.getOrCreateModeInstance( None )

   def separator( self ):
      """Always print a separator between modes."""
      return True

class Mode( object ):
   """A container for the SaveBlocks that are to be output in a instance of a
   particular mode.  Plugins should create a subclass of this class for each CLI mode
   they define (that is, for each subclass of CliParser.Mode they define)."""

   # all save blocks are sorted by sort key; if absent, the classname is used
   sortKey = ""
   printComments = True
   # The following can be set to True if we want to merge modes with identical
   # commands into one range block. In addition it has to do the following:
   # 1. implement a static method of enterRangeCmd() that takes a list of modes
   #    and returns a command that enters the range.
   # 2. overload the canMergeRange() function to return True when conditions
   #    are met.
   mergeRange = False

   def __init__( self, param ):
      self.param_ = param

      if not hasattr( self, 'saveBlockGenerators_' ):
         self.__class__._sortSaveBlockGenerators()

      # Instantiate all the SaveBlocks for this Mode instance.
      self.saveBlocks_ = []
      self.saveBlockMap_ = {}
      for generator in self.saveBlockGenerators_:
         if type( generator ) == str:
            saveBlock = CommandSequence()
         else:
            saveBlock = ModeCollection( generator )
         self.saveBlocks_.append( saveBlock )
         self.saveBlockMap_[ generator ] = saveBlock

   def skipIfEmpty( self ):
      # This indicates that we shold not print the mode in running-config if it is
      # empty (and no comments) even if someone called getOrCreateModeInstance().
      return False

   def hideInactive( self, param ):
      """This can be overridden if certain modes could be hidden at run time."""
      return False

   def hideUnconnected( self, param ):
      """This can be overridden if certain modes could be hidden at run time."""
      return False

   def commentKey( self ):
      # pylint: disable-msg=E1101
      if hasattr( self, 'longModeKey' ):
         return self.longModeKey
      else:
         trace( "Returning dummy comment Key since none exists for this mode" )
         return None

   @classmethod
   def modeSortKey( clazz ):
      if clazz.sortKey:
         return clazz.sortKey
      else:
         return clazz.__name__

   def addCommandSequence( clazz, name, before=[], after=[] ):
      """Registers a command sequence name for instances of this Mode subclass.
      'before' and 'after' are lists of command sequence names or other Mode subclass
      objects.  'before' represents those save blocks that this command sequence must
      come before.  'after' represents those save blocks that this command sequence
      must come after.  Therefore:

       MyParentMode.addCommandSequence( 'seq1' )
       MyParentMode.addCommandSequence( 'seq2' )
       MyParentMode.addCommandSequence( 'seq3', before=[ 'seq1' ], after=[ 'seq2' ] )

      will cause the save blocks to be output in the order: seq2, seq3, seq1."""
      assert type( name ) == str
      clazz._addSaveBlockGeneratorRecord( name, before, after )
   addCommandSequence = classmethod( addCommandSequence )

   def addChildMode( clazz, childModeClass, before=[], after=[] ):
      """Registers a Mode subclass as being a child mode of this Mode subclass.
      'before' and 'after' are lists of command sequence names or other Mode subclass
      objects.  'before' represents those save blocks that the child mode must come
      before.  'after' represents those save blocks that the child mode must come
      after.  Therefore:

       MyParentMode.addCommandSequence( 'seq1' )
       MyParentMode.addCommandSequence( 'seq2' )
       MyParentMode.addChildMode( MyChildMode, before=[ 'seq1' ], after=[ 'seq2' ] )

      will cause the save blocks to be output in the order: seq2, MyChildMode,
      seq1."""
      assert issubclass( childModeClass, Mode )
      clazz._addSaveBlockGeneratorRecord( childModeClass, before, after )
   addChildMode = classmethod( addChildMode )

   def _addSaveBlockGeneratorRecord( clazz, generator, before, after ):
      """Adds a 'save block generator record' to this Mode subclass.  A 'save block
      generator' is either the name of a command sequence (a string) or a child mode
      class object.  A 'save block generator record' contains the generator, plus
      information about its dependencies on other save blocks."""
      if 'DEBUG_CLISAVE' in os.environ:
         # Get information about the caller of this function's caller.
         import traceback
         ( file, lineNo, fn, line ) = traceback.extract_stack()[ -3 ]
         debug = ( file, lineNo )
      else:
         debug = None
      record = ( generator, before, after, debug )
      try:
         clazz.saveBlockGeneratorRecords_.append( record )
      except AttributeError:
         clazz.saveBlockGeneratorRecords_ = [ record ]
   _addSaveBlockGeneratorRecord = classmethod( _addSaveBlockGeneratorRecord )

   def _sortSaveBlockGenerators( clazz ):
      """Sorts the save block generators for this mode class according to their
      dependencies.  This function is called the first time that an instance of this
      Mode subclass is created."""
      try:
         generatorRecords = clazz.saveBlockGeneratorRecords_
      except AttributeError:
         # There are no child mode classes.
         generatorRecords = []

      generators = []
      for ( generator, before, after, debug ) in generatorRecords:
         generators.append( generator )

      partialOrder = []
      for ( generator, before, after, debug ) in generatorRecords:
         partialOrder.extend( [ ( generator, b ) for b in before ] )
         partialOrder.extend( [ ( a, generator ) for a in after ] )

         # Check that none of the SaveBlocks have been given a dependency on a
         # non-existent SaveBlock.
         for b in before + after:
            if not b in generators:
               location = "an unknown location " \
                   "(run with DEBUG_CLISAVE=1 for more debug info)"
               try:
                  ( file, line ) = debug
                  location = "%s:%d" % ( file, line )
               except ValueError:
                  pass
               except TypeError:
                  pass
               msg = "At %s\nyou called %s.addChildMode() or " \
                     "%s.addCommandSequence(), " \
                     "passing %r as an entry in the 'before' or 'after' lists.\n" \
                     "However, %r is not a valid CLI save block for mode %s.\n" \
                     "Perhaps you passed the mode name rather than the mode class " \
                     "object?" % \
                     ( location, clazz.__name__, clazz.__name__,
                       b, b, clazz.__name__ )
               raise Exception, msg

      # Sort the generators into a deterministic order; the
      # topological sort is not fully constraining.
      def _generatorKey( g ):
         if type( g ) == str:
            return g
         else:
            return g.modeSortKey()

      generators.sort( key=_generatorKey )
      traceTopSort( 'Sorting save blocks %s for class %s' %
              ( generators, clazz.__name__ ) )
      traceTopSort( 'Using partial order %s' % partialOrder )
      generators = Tac.topologicalSort( generators, partialOrder )
      traceTopSort( 'Sorted as %s' % generators )
      assert not hasattr( clazz, 'saveBlockGenerators_' )
      clazz.saveBlockGenerators_ = generators

   _sortSaveBlockGenerators = classmethod( _sortSaveBlockGenerators )

   def comments( self, param ):
      if self.printComments:
         commentKey = self.commentKey()
         if commentKey:
            return param.cliConfig.comment.get( commentKey )
      return None

   def emptyCmds( self, param ):
      for b in self.saveBlocks_:
         if not b.empty( param ):
            return False
      return not self.comments( param )

   def empty( self, param ):
      """We are empty if no commands have been added to this Mode
      instance, and the enterCmd() is empty."""
      return ( ( self.skipIfEmpty() or not self.enterCmd() ) and
               self.emptyCmds( param ) )

   def canMergeRange( self ):
      """Whether this mode instance can participate in range merge.
      For example, VLAN 1 is not supposed to merge with other VLAN
      instances, partly due to its different defaults."""
      return False

   def instanceKey( self ):
      """Used by range merge."""
      return self.param_

   def content( self, param ):
      """Returns comments + saveBlocks as a tuple"""
      blocks = [ self.comments( param ) ]
      for i in self.saveBlocks_:
         blocks.append( i.content( param ) )
      return tuple( blocks )

   def writeMode( self, param, prefix, cliModel=None ):
      stream = param.stream
      firstTime = True
      for b in self.saveBlocks_:
         if b.empty( param ):
            continue
         if stream and not firstTime and ( not prefix or
                                           b.separator() ):
            # Write a newline to separate it from the previous SaveBlock,
            # but only for global mode or if I am a new mode.
            stream.write( prefix + '!\n' )
         firstTime = False
         b.write( param, prefix, cliModel=cliModel )

   def write( self, param, prefix, cliModel=None, enterCmd=None ):
      """configRoot should be used to access config entities and SysdbRoot for
      everything else. While in config session configRoot points to sessionRoot
      otherwise it's same as sysdbRoot """
      stream = param.stream
      if enterCmd is None:
         enterCmd = self.enterCmd()

      if stream:
         stream.write( prefix )
         stream.write( enterCmd )
         stream.write( '\n' )
      else:
         cliModel.cmds[ enterCmd ] = newModel = ShowRunOutputModel.Mode()
         # Get rid of the header field, this is only present on the
         # global config
         newModel.header = None
         cliModel = newModel

      # prefix gets one more indentation level
      prefix += '   '

      comment = self.comments( param )
      if comment:
         for line in comment.splitlines():
            if stream:
               stream.write( prefix + CliCommon.commentAppendStr +
                          ' %s\n' % line )
            else:
               cliModel.comments.append( line )

      self.writeMode( param, prefix, cliModel=cliModel )

   def processProfileCmds( self, pcmds ):
      # Based on profile cmds, update our save blocks to reflect the
      # real running-config.
      # Note the first save block should be a CommandSequence with
      # the 'profile' command itself
      firstSb = self.saveBlocks_[ 0 ]
      assert isinstance( firstSb, CommandSequence )
      assert firstSb.commands_[ 0 ].startswith( 'profile ' )
      overriddenPcmds = list( pcmds )
      for sb in self.saveBlocks_[ 1 : ]:
         if isinstance( sb, CommandSequence ):
            for idx, cmd in enumerate( sb.commands_ ):
               if cmd in overriddenPcmds:
                  # Slight efficiency trick, check marker in CommandSequence.write()
                  sb.commands_[ idx ] = SKIP_COMMAND_MARKER
                  overriddenPcmds.remove( cmd )
      for cmd in overriddenPcmds:
         if cmd.startswith( 'default ' ):
            continue
         baseCmd = cmd.lstrip( "no " )
         # insert default commands after the 'profile' command
         self.saveBlocks_[ 0 ].commands_.append( 'default ' + baseCmd )

   def modeSeparator( self ):
      """For printing modes inside a mode collection, whether a separator is needed
      between two modes."""
      return True

   def __cmp__( self, other ):
      """Compares two instances of this Mode subclass.  Subclasses should override
      this method if they wish their mode instances to be output in a different order
      to this default."""
      assert self.__class__ == other.__class__
      return cmp( self.instanceKey(), other.instanceKey() )

   def enterCmd( self ):
      """Subclasses should either override this method to return the
      command to be used to enter this CLI mode instance, or inherit a
      CliMode to implement this method. Otherwise, NotImplementedError
      is raised."""
      raise NotImplementedError

   def __getitem__( self, key ):
      """Returns the SaveBlock for a particular key.  The key may be either the name
      of a command sequence (a string) or a child mode class object."""
      return self.saveBlockMap_[ key ]

class GlobalConfigMode( Mode ):
   def __init__( self ):
      # Instances of GlobalConfigMode do not have parameters.
      Mode.__init__( self, None )

   headerHooks_ = []

   @staticmethod
   def addHeaderHook( callable ):
      '''Add a "header hook", which gets called at the very beginning to write
      a header at the start of running-config.  Eos uses this to write a header
      with information about the version of EOS that wrote the running-config
      (see Eos/CliSavePlugin/EosRunningConfigHeader.py).

      The header hook is passed a file-like object that it can write to, and
      anything written there goes before all of the save blocks.'''
      GlobalConfigMode.headerHooks_.append( callable )

   def write( self, param, cliModel=None ):
      # Or we are printing to a stream or to a cliModel
      stream = param.stream
      assert bool( stream ) != bool( cliModel )
      if param.showHeader:
         for h in GlobalConfigMode.headerHooks_:
            h( param, cliModel=cliModel )
      self.writeMode( param, '', cliModel=cliModel )
      if stream:
         stream.write( '!\nend\n' )

# Add a blank command sequence so we can enable relative dependencies.
# For example, all low-priority sequences should use
# after=[ 'config.priority' ], and all high-priority sequences should
# use before=[ 'config.priority' ].
# By default, command sequences are not ordered wrt config.priority.
GlobalConfigMode.addCommandSequence( 'config.priority' )
