# Copyright (c) 2015 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import os
import random
import re

import EapiClientLib

# Do not eval pythonExpression if it attempts to use any of these.
# The python expression is expected to be from trusted source and so we create a 
# blacklist to prevent access to obviously risky builtins.
unallowedBuiltins = [
   '__import__',
   'compile',
   'delattr',
   'dir',
   'eval', 
   'execfile', 
   'file',
   'getattr', 
   'globals', 
   'hasattr',
   'input',
   'locals',
   'open',
   'raw_input',
   'reload',
   'setattr',
   'vars'
   ]

# pass in key and json show command output, and returns list of all values of that
# key.  
# Use this function to get all values for a keyName anywhere in the json
# hierarchy.  Possible use cases, traversing show inventory looking for 'modelName'
# or for show interface status to loop over all interface status-es to find all the
# existing interfaceForwardingModel
def findAllJson( needle, haystack ):
   result = set([])

   if type( haystack ) == type( dict() ):
      if needle in haystack.keys():
         result.add( haystack[ needle ] )
      for key in haystack.keys():
         result.update( set( findAllJson( needle, haystack[ key ] ) ) )
   elif type( haystack ) == type( [] ):
      for bale in haystack:
         result.update( set( findAllJson( needle, bale ) ) )

   return list( result )

# returns list of all matching groups in the text that match the pattern.  
# Possible use case would be to return all types of transceivers or for checking
# whether ip routing is enabled.
def findAllText( pattern, text ):
   return re.findall( pattern, text )

# additional methods provided to aid in writing python expression for 
# describing tag.
jsonFunctionsAvailableDict = {
   'findAll' : findAllJson
   }
textFunctionsAvailableDict = {
   'findAll' : findAllText
   }
builtinsAvailableDict = { k : v for k, v in __builtins__.items() \
                                  if k not in unallowedBuiltins }

jsonFunctionsAvailableDict.update( builtinsAvailableDict )
textFunctionsAvailableDict.update( builtinsAvailableDict )

# Opens a socket to run the show command via FastClidServer and block on the
# response
def runShowCommand( sysname, command, revision,
                    responseFormat="json", requestTimeout=None ):
   if os.environ.get( 'CR_BTEST_RUN' ):
      # Breadth test run.
      # Can't run show commands here, randomly return one of the 3 possible
      # return values. This api can either return the command output ( for test
      # we'll just echo command ), noSuchCommand or an exception
      showCommandException = \
            "csocket\nerror\np0\n(I2\nS'No such file or directory'\np1\ntp2\nRp3\n."
      if random.choice( [ True, False ] ):
         return ( command, None )
      else:
         return ( None, showCommandException )

   unpickledStuffForDebuggingOnly = None
   result = None
   try:
      eapiClient = EapiClientLib.EapiClient( sysname=sysname,
                                             disableAaa=True,
                                             privLevel=15 )
      response = eapiClient.runCmds( version=1, format=responseFormat,
                                     cmds=[ { "cmd": command,
                                              "revision": revision } ],
                                     requestTimeout=requestTimeout,
                                     timestamps=True )
   except EapiClientLib.EapiException as err:
      unpickledStuffForDebuggingOnly = err.data if err.data else err
   except Exception as err: # pylint: disable-msg=W0703
      unpickledStuffForDebuggingOnly = err
   else:
      result = response[ "result" ][ 0 ] if response.get( "result" ) else None
      if response.get( "error" ):
         unpickledStuffForDebuggingOnly = response.get( "error" )
   outputTuple = ( result, unpickledStuffForDebuggingOnly )
   assert any( outputTuple ), "We must have atleast one output entry"
   return outputTuple

def evalPythonExpression( pythonExpression, commandResult, responseFormat="json" ):
   pyResult = None
   unpickledStuffForDebuggingOnly = None
   if os.environ.get( 'CR_BTEST_RUN' ):
      # Breadth test run.
      if random.choice( [ True, False ] ):
         return pythonExpression, None
      else:
         return None, "NameError: name 'len' is not defined"
   try:
      if not pythonExpression:
         raise ValueError( "python expression cannot be empty" )
      pyResult = eval( pythonExpression, { "__builtins__":None }, 
                       dict( jsonFunctionsAvailableDict, json=commandResult )
                       if responseFormat=="json" else 
                       dict( textFunctionsAvailableDict, 
                             text=commandResult[ 'output' ] ) )
   except Exception as err: # pylint: disable-msg=W0703
      # The base Exception class has __reduce__() method  to make it picklable
      unpickledStuffForDebuggingOnly = err
   return pyResult, unpickledStuffForDebuggingOnly
