# Copyright (c) 2015 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Agent, Tac, Tracing
import CliRelaySwitchUtil, CliRelaySwitchLib
from collections import deque
import cPickle

t0 = Tracing.trace0
t1 = Tracing.trace1
t2 = Tracing.trace2

class ShowCommand( object ):
   """ Acts on the input ShowCommand entity and executes the show command
   periodically. It effectively manages all the output attributes in output
   ShowCommandResult entity """

   notifierTypeName = 'CliRelay::ShowCommand'

   def __init__( self, sysname, showCommand, showCommandResult ):
      self.sysname = sysname
      self.showCommand = showCommand
      self.showCommandResult = showCommandResult

   def run( self ):
      t2( "Attempt to run command", self.showCommand.showCommand, 
          "Capi revision", self.showCommand.revision )

      ( showCommandResult, unpickledStuffForDebuggingOnly ) = \
            CliRelaySwitchUtil.runShowCommand( self.sysname,
                                               self.showCommand.showCommand,
                                               self.showCommand.revision,
                                               self.showCommand.responseFormat,
                                               60 )
      t2( "Output from running show command", showCommandResult,
          unpickledStuffForDebuggingOnly )

      # default cPickle protocol produces the larger sized human readable ASCII
      # Set timestamp in pickledException
      if showCommandResult:
         ( pythonEval, unpickledStuffForDebuggingOnly ) = \
             CliRelaySwitchUtil.evalPythonExpression( 
                   self.showCommand.pythonExpression, showCommandResult,
                   self.showCommand.responseFormat )
         t0( "Python expression", self.showCommand.pythonExpression )
         t0( type( self.showCommand.pythonExpression ) )
         t2( "Python expression eval result", pythonEval, 
               unpickledStuffForDebuggingOnly )
         if pythonEval is not None:
            self.showCommandResult.stuffForDebuggingOnly = cPickle.dumps( \
               ( Tac.now(), ) )
            self.showCommandResult.pythonExpressionResult = str( pythonEval )
      if unpickledStuffForDebuggingOnly:
         self.showCommandResult.stuffForDebuggingOnly = \
             cPickle.dumps( ( Tac.now(), unpickledStuffForDebuggingOnly ) )
         self.showCommandResult.pythonExpressionResult = ""

class CliRelayClientReactor( Tac.Notifiee ):
   """ Create ShowCommandResult entity in showCommandResultDir for every 
   ShowCommand entity that appears in the input ShowCommandDir."""

   notifierTypeName = 'CliRelay::ShowCommandDir'
   
   def __init__( self, sysname, cliRelaySwitchConfig, showCommandDir, 
                 showCommandResultDir ):
      Tac.Notifiee.__init__( self, showCommandDir )
      self.sysname = sysname
      self.showCommandResultDir = showCommandResultDir
      self.cliRelaySwitchConfig = cliRelaySwitchConfig
      self.showCommand = {}
      # For each command in queue, there is corresponding element in
      # list to store meta data
      #   ( lastRunTimestamp, lastRunTime )
      self.commandList = {}
      self.commandQueue = deque()
      self.refreshCmdActivity_ = Tac.ClockNotifiee( self.refreshCommandResult )
      self.refreshCmdActivity_.timeMin = Tac.endOfTime
      # How much time did refreshActivity take last time. We need this to 
      # decide how often should we run to avoid load on the system.
      self.refreshActivityLastRunTime = None
      # Flag to indicate if we are in middle of processing commandQueue
      self.refreshActivityInProgress = False
      # Timestamp when refreshActivity started processing the commandQueue
      self.refreshActivityStartTime = None
      t0( "Time interval", self.cliRelaySwitchConfig.refreshTime )
      self.handleInitialized()

   def handleInitialized( self ):
      # Handle initial state of showCommandDir
      for key in self.notifier().showCommand:
         self.handleShowCommand( key )
      # Cleanup any stale state in output showCommandResultDir
      staleResultKeys = set( self.showCommandResultDir.showCommandResult.keys() ) - \
            set( self.notifier().showCommand.keys() )
      for key in staleResultKeys:
         t0( "Stale command", key, "deleted from showCommandResultDir" )
         del self.showCommandResultDir.showCommandResult[ key ]
      t0( "Initialized reactor for", self.notifier() )

   def printCommandQueue( self ):
      print "Dumping command Q ( ", len( self.commandQueue ), " )"
      for key in self.commandQueue:
         assert key in self.commandList
         print "Command ", key

   @Tac.handler( 'showCommand' )
   def handleShowCommand( self, key ):
      # key is the id. If we have non-zero keys, then start monitoring
      if key in self.notifier().showCommand:
         showCmd = self.notifier().showCommand[ key ]
         t1( "handleShowCommand ", showCmd, "( ", key, " )",
               " is added to showCommand collection" )
         showCommandResult = self.showCommandResultDir.newShowCommandResult( key )
         self.showCommand[ key ] = ShowCommand( self.sysname, showCmd, \
                                                   showCommandResult )

         # Check if this is going to be the first element on the queue. 
         # Trigger refreshActivity to process the queue.
         if not len( self.commandQueue ):
            self.refreshCmdActivity_.timeMin = 0

         # Add to the Queue
         self.commandQueue.append( key )
         self.commandList[ key ] = ( None, None )
      else:
         showCmd = self.showCommand.pop( key, None )
         t1( "handleShowCommand", showCmd.showCommand.showCommand, "( ", key, " )",
               " revision ", showCmd.showCommand.revision, 
               "removed from showCommand collection" )
         del self.showCommandResultDir.showCommandResult[ key ]
         # Remove from the Queue
         self.commandQueue.remove( key )
         del self.commandList[ key ]

   # Timer activity scheduled at configured time interval
   def refreshCommandResult( self ):

      t2( "refreshActivity scheduled ", Tac.now() )
      if not len( self.commandQueue ):
         # Empty queue
         t2( "Empty command queue" )
         return

      # Get the first element in queue but don't remove yet
      showCommandId = self.commandQueue[ 0 ]
      ( lastRunTimestamp, _lastRunDuration ) = self.commandList[ showCommandId ]

      # Process show command if it's newly added or hasn't been run
      # in this run of refreshActivity
      if lastRunTimestamp is None or \
            lastRunTimestamp < self.refreshActivityStartTime:

         # Set the flag to indicate refreshActivity is in middle of processing queue
         # if not set
         if not self.refreshActivityInProgress:
            self.refreshActivityInProgress = True
            # Store the start time
            self.refreshActivityStartTime = Tac.now()

         # Run the command
         startTime = Tac.now()
         self.showCommand[ showCommandId ].run()
         endTime = Tac.now()
         self.commandList[ showCommandId ] = ( endTime, ( endTime - startTime ) )
         t2( "command ", self.showCommand[ showCommandId ], "( ", showCommandId, 
               " )", " took ", _lastRunDuration, " last time and now ", 
               ( endTime - startTime ) )

         # Remove and add command to the tail of the queue
         self.commandQueue.popleft()
         self.commandQueue.append( showCommandId )

         # Run as soon as possible
         self.refreshCmdActivity_.timeMin = 0
      else:
         # We are done processing the whole queue. This must be the first element 
         # in the queue
         self.refreshActivityInProgress = False

         if self.refreshActivityStartTime:
            self.refreshActivityLastRunTime = \
                  Tac.now() - self.refreshActivityStartTime

         # Algorithm to schedule when next refreshActivity should be run is very
         # simple. All we want is to avoid load on the system in case some show
         # command take longer to run.
         # Check how much time we took last to process the whole queue. Multiply by
         # factor of 10 and add the default refreshTime. e.g. if default refreshTime
         # is 30 seconds and last run we took 5 minutes to process then we don't 
         # want to schedule again at 5 minutes 30 seconds since we know that 
         # probably next run will also take similar time. Instead sleep for 50 
         # minutes ( factor of 10 ) so that we know we don't run more than 10% of 
         # the time
         weightOfLastRun = 0
         if self.refreshActivityLastRunTime:
            t2( " RefreshActivity run took ", self.refreshActivityLastRunTime, 
                  " seconds" )
            # Backing off delayes our test run ( breadth tests ) so reduce the 
            # delay if we are called in breadth test environment
            weightOfLastRun = self.refreshActivityLastRunTime * \
                  self.cliRelaySwitchConfig.refreshBackOffFactor

         # Set self.refreshActivityStartTime to the time when refreshActivity is
         # scheduled next
         nextRunInterval = weightOfLastRun + self.cliRelaySwitchConfig.refreshTime
         self.refreshActivityStartTime = Tac.now() + nextRunInterval
         t2( " Scheduling RefreshActivity to run after ", 
               nextRunInterval, " seconds" )

         # Schedule RefreshActivity 
         self.refreshCmdActivity_.timeMin = self.refreshActivityStartTime
         t2( " RefreshActivity scheduled to run at ", 
               self.refreshActivityStartTime )

class CliRelayDirReactor( Tac.Notifiee ):
   """ Create status entities for top level config entity created by each
   service agent mounted under
   'cliRelay/version1/fromCvx'. This reactor keeps the
   output cliRelayStatus.showCommandResultDir collection in sync by
   creating entities that match the entrys that get instantiated under input
   showCommandDir. """
   
   notifierTypeName = 'Tac::Dir'
   
   def __init__( self, sysname, cliRelaySwitchConfig, showCommandDir, 
                 cliRelayStatus ):
      Tac.Notifiee.__init__( self, showCommandDir )
      self.sysname = sysname
      self.cliRelayStatus = cliRelayStatus
      self.cliRelaySwitchConfig = cliRelaySwitchConfig
      # Dictionary to keep the collection of reactors for each input entity
      # under showCommandDir
      self.cliRelayClient = {} 
      self.handleInitialized()

   def handleInitialized( self ):
      for key in self.notifier().entityPtr:
         self.handleCliRelayClient( key )

      staleKeys = set( self.cliRelayStatus.showCommandResultDir.keys() ) - \
            set( self.notifier().entityPtr.keys() )
      for key in staleKeys:
         t0( "Stale client", key, "deleted from CliRelayStatus" )
         del self.cliRelayStatus.showCommandResultDir[ key ]
         self.cliRelayClient.pop( key, None )
      t0( "Initialized reactor for", self.notifier() )

   @Tac.handler( 'entityPtr' )
   def handleCliRelayClient( self, key ):
      # key has to exist in entityPtr collection and its value in
      # entityPtr collection should not be NULL
      if( key in self.notifier().entityPtr ) and \
            ( self.notifier().entityPtr[ key ] ): 
         t0( "handleCliRelayClient", key, "added to request" )
         assert self.notifier().entityPtr[ key ].tacType.fullTypeName == \
               "CliRelay::ShowCommandDir", "Unexpected entity type"
         showCommandResultDir = \
               self.cliRelayStatus.newShowCommandResultDir( key )
         self.cliRelayClient[ key ] = CliRelayClientReactor( 
            self.sysname,
            self.cliRelaySwitchConfig, self.notifier().entityPtr[ key ], 
            showCommandResultDir )
      else:
         t0( "handleCliRelayClient", key, "removed from request" ) 
         self.cliRelayClient.pop( key, None )
         del self.cliRelayStatus.showCommandResultDir[ key ]

class CliRelayFromCvxReactor( CliRelaySwitchLib.CliRelayFromCvxReactor ):
   ''' CVX publish mount appear under the Tac::Dir which the switch agent mounts
   from Sysdb. SCS switch agent requires commandRequest entry to appear for
   processing the show commands requested from SCS service agent '''

   def __init__( self, fromCvx, toCvx, config, sysname ):
      # Tac::Dir containing appNames and config.runnability published fromCvx
      self.toCvx = toCvx     # CliRelay::CliRelayStatus
      self.config = config   # CliRelaySwitch::Config
      self.sysname = sysname
      self.cliRelayDirReactor = None
      self.key = 'commandDir'
      CliRelaySwitchLib.CliRelayFromCvxReactor.__init__( self, fromCvx )

   def handleKey( self, key, created ):
      # Ignore entries which we don't care about
      if key != self.key:
         return

      t0( "CliRelayFromCvxReactor : handle", key, "created", created )
      if created:
         assert self.notifier().entityPtr[ key ].tacType.fullTypeName == \
               'Tac::Dir'
         t0( "Spawning child reactor to handle client apps" )
         self.cliRelayDirReactor = \
               CliRelayDirReactor( self.sysname, 
                                      self.config,
                                      self.notifier()[ self.key ],
                                      self.toCvx )
      else:
         t0( "CliRelayFromCvxReactor : Cleaning up child reactor" )
         self.cliRelayDirReactor = None
   
class CliRelaySwitchAgent( Agent.Agent ):
   """ The CliRelaySwitchAgent allows service agents in CVX controller to
   monitor the output of show commands. Eg: BugAlert service agent """

   def __init__( self, entityManager ):
      self.switchConfig = None
      self.fromCvx = None
      self.toCvx = None
      self.cliRelayFromCvxReactor = None
      self.sysname = entityManager.sysname()
      Agent.Agent.__init__( self, entityManager )
      t0( "CliRelaySwitchAgent initialized" )

   def doInit( self, entityManager ):
      t0( "doInit called" )
      mg = entityManager.mountGroup()
      # Write to our config the state CVX publishes for our runnability
      self.switchConfig = mg.mount(
            'cliRelay/switchConfig',
            'CliRelaySwitch::Config', 'w' )
      self.fromCvx = mg.mount( 
            'cliRelay/version1/fromCvx', 
            'Tac::Dir', "ri" )
      self.toCvx = mg.mount(
            'cliRelay/version1/toCvx',
            'CliRelay::CliRelayStatus', "w" )

      def _finishMounts():
         self.cliRelayFromCvxReactor = \
               CliRelayFromCvxReactor( self.fromCvx,
                                          self.toCvx,
                                          self.switchConfig,
                                          self.sysname )
         t0( "Finished mounts" )

      mg.close( _finishMounts )

