# Copyright (c) 2012 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

import re

from BasicCli import ConfigModeBase
from CliCommon import AlreadyHandledError

# Range expansion for range modes deriving from ConfigModeBase.
# Used by Ebra/CliPlugin/VlanCli.py and Intf/CliPlugin/IntfRangeCli.py

def rangeError( mode, msg ):
   mode.addError( str( msg ) )
   mode.addError( "Range format: {start,end|$,step=1,pad=1} "
                  "(max 1 per command)" )
   raise AlreadyHandledError

def expandRange( mode, tokens, numModes ):
   # Returns a list of tokens (list(list())) if tokens can successfully be
   # expanded, otherwise None.

   # See if we want range expansion: {start,end,step=1,pad=1}
   # Try to match a syntactically valid range.
   rangeRe = re.compile( r"(\{\d+,(\$|\d+)(,\d+){0,2}\})" )
   rangeMatch = None
   rangedTokenIndex = 0
   for i in range( len( tokens ) ):
      # findall returns a list of tuples, each tuple capturing a subgroup match
      matches = rangeRe.findall( tokens[ i ] )
      if len( matches ):
         # First match, take note of it
         if not rangeMatch and len( matches ) == 1:
            rangeMatch = matches[ 0 ][ 0 ]
            rangedTokenIndex = i
         # More than one match found
         else:
            rangeError( mode, "Only one range expansion is supported." )

   # A single match found for a range expression
   if rangeMatch:
      rangedToken = tokens[ rangedTokenIndex ]
      rangeArgs = rangeMatch[ 1 : -1 ].split( "," )
      rangeArgsLen = len( rangeArgs )

      # Get range step
      if rangeArgsLen == 2:
         rangeStep = 1
      elif rangeArgsLen >= 3:
         rangeStep = int( rangeArgs[ 2 ] )

      # Autocomplete end '$' token
      if rangeArgs[ 1 ] == '$':
         rangeArgs[ 1 ] = int( rangeArgs[ 0 ] ) + \
                          numModes * rangeStep - 1

      # Build range
      fullRange = range( int( rangeArgs[ 0 ] ),
                         int( rangeArgs[ 1 ] ) + 1,
                         rangeStep )

      # Verify that the output of range() is not empty. This can happen
      # in syntactically valid but logically invalid ranges such as
      # range(1, 0).
      if not len( fullRange ):
         rangeError( mode, "Invalid range (end < start)." )

      # Padding length, used as arg for "%.*d"
      if rangeArgsLen == 4:
         rangePadding = int( rangeArgs[ 3 ] )
      else:
         rangePadding = 1

      # Specified range matches the number of modes that are in the
      # current range mode.
      if len( fullRange ) == numModes:
         # Build the list of tokens to execute
         tokensList = list()
         for i in fullRange:
            tokens[ rangedTokenIndex ] = \
               rangeRe.sub( "%.*d" % ( rangePadding, i ), rangedToken )
            tokensList.append( list( tokens ) )

         return tokensList
      else:
         rangeError( mode,
                     "Specified range's elements do not match the number "
                     "of modes in the current range mode (range %d, "
                     "expected %d)" % ( len( fullRange ), numModes ) )

   # No syntactically valid range match. No checking for mismatched braces.
   # We assume if there is an error in which braces are at fault, this is
   # behavior intended by the user.
   return None

def tryRangeParse( mode, tokens, **kwargs ):
   # Try to parse a range expansion in a range mode.
   # Returns True if we successfully get each mode to parse.
   # Returns False if we don't do any handling at all.
   # Raises AlreadyHandledError iff we run into an error while trying to handle
   # a range expansion

   assert isinstance( mode, ConfigModeBase )

   # Check that we're in a range mode
   if not mode.multiInstance_:
      return None

   tokensList = expandRange( mode, tokens, len( mode.individualModes_ ) )
   if tokensList:
      # pylint: disable-msg=W0212
      # Get each mode to parse. If we do get a list from expandRange(), then
      # we have as many child modes as we have expanded tokens in the list.
      for i in range( len( tokensList ) ):
         submode = mode.individualModes_[ i ]
         result = submode.parse( tokensList[ i ], **kwargs )
         submode.session._invokeValueFunc( submode, result, kwargs[ "authz" ],
                                           kwargs[ "acct" ] )
      return True

   return False
