# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from ctypes import cdll
import os
import sys
from CliPrintCTypes import decorateFunctions

class CliPrint:
   '''Wrapper for ctypes based access to members defined in libCliPrint'''
   def __init__( self ):
      self.mlib_ = cdll.LoadLibrary( "libCliPrint.so" )
      # Decorate the loaded ctype members using the generated decorator
      decorateFunctions( self.mlib_ )

   @property
   def lib( self ):
      return self.mlib_

# The CliPrint module allows to print cli text (printf with format string)
# or json from the same line of code, depending on the needed output-format.
# For an example, look for _example_  below (Note that such models needs to inherit
# from DeferredModel -- a missnomer: we just never create a python model instance,
# we produce the json directly, in a streamed fashion -> cli output order needs
# to agree with model order, else code has to bifurk into TEXT and JSON paths.
# Example usage in Cli/test/ModelTest.py function pyCliPrintHandler().


# Output Formats. This should be in sync with the ofUnset/OfText/OfJson
# definitions of ofJson and ofText from AgentCommandRequest.tac
TEXT = 1
JSON = 2

JSON_NUMBER = 0
JSON_STRING = 1
JSON_BOOLEAN = 2
JSON_FLOAT = 3

# pylint not happy about the nop function overwrite, so disable warning
#   E0202(method-hidden):An attribute defined in CliPrint line 49 hides this method
# pylint: disable-msg=E0202

class Printer():
   def __init__( self, outputFormat ):
      self.outputFormat = outputFormat
      # The first attribute will not need to terminate previous line (with comma).
      # Set each time a structure is started, set false when closing that
      # struct since it means parent struct has at least one attr.
      self.firstAttr = True # e.g. still waiting for first attribute
      self.fd = sys.stdout.fileno()

      if outputFormat == TEXT:
         self.start = self.nop
         self.end = self.nop
         self.startList = self.nop2args
         self.endList = self.nop
         self.startListEntry = self.nop
         self.endListEntry = self.nop
         self.startSubmodel = self.nop2args
         self.endSubmodel = self.nop
         self.startDict = self.nop2args
         self.endDict = self.nop
         self.startDictEntry = self.nop2args
         self.endDictEntry = self.nop

   def nop( self ):
      pass

   def nop2args( self, dummy ):
      pass

   # starts/ends JSON stream
   def start( self ):
      os.write( self.fd, "{" )

   def end( self ):
      os.write( self.fd, "}" )

   # List Structure Functions
   def startList( self, listName ):
      if not self.firstAttr:
         os.write( self.fd, "," )
         self.firstAttr = True
      os.write( self.fd, "\"%s\":[" % listName )

   def endList( self ):
      os.write( self.fd, "]" )
      self.firstAttr = False

   def startListEntry( self ):
      if not self.firstAttr:
         os.write( self.fd, "," )
         self.firstAttr = True
      os.write( self.fd, "{" )

   def endListEntry( self ):
      os.write( self.fd, "}" )
      self.firstAttr = False

   # utility for dict and submodel
   def _startObj( self, attrName=None ):
      if not self.firstAttr:
         os.write( self.fd, "," )
         self.firstAttr = True
      if attrName:
         os.write( self.fd, "\"%s\":{" % attrName )
      else:
         os.write( self.fd, "{" )

   def _endObj( self ):
      os.write( self.fd, "}" )
      self.firstAttr = False

   # Submodel Structure Functions
   def startSubmodel( self, submodelName ):
      self._startObj( submodelName )

   def endSubmodel( self ):
      self._endObj()

   # Dict Structure Functions
   def startDict( self, dictName ):
      self._startObj( dictName )

   def endDict( self ):
      self._endObj()

   def startDictEntry( self, attr ):
      if attr.jsonType != JSON_STRING:
         assert False, "Dict '%s' key is not a string but type %d" % (
                          attr.name, attr.jsonType )
      else:
         self._startObj( attr.value )

   def endDictEntry( self ):
      self._endObj()

   # Functions that define Attributes (values)
   def addAttributes( self, fmtStr, *args ):
      if self.outputFormat == TEXT:
         os.write( self.fd, fmtStr % args )
         return
      for info in args:
         if info.isKey_:
            continue
         if not self.firstAttr:
            os.write( self.fd, "," )
         else:
            self.firstAttr = False # will print first one right below
         if info.jsonType == JSON_STRING:
            if info.name:
               os.write( self.fd, "\"%s\":\"%s\"" % ( info.name, info.value ) )
            else:
               os.write( self.fd, "\"%s\"" % info.value )
         elif info.jsonType == JSON_NUMBER:
            if info.name:
               os.write( self.fd, "\"%s\":%d" % ( info.name, info.value ) )
            else:
               os.write( self.fd, "%d" % info.value )
         elif info.jsonType == JSON_FLOAT:
            if info.name:
               os.write( self.fd, "\"%s\":%f" % ( info.name, info.value ) )
            else:
               os.write( self.fd, "%f" % info.value )
         elif info.jsonType == JSON_BOOLEAN:
            if info.name:
               if info.value:
                  os.write( self.fd, "\"%s\":true" % info.name )
               else:
                  os.write( self.fd, "\"%s\":false" % info.name )
            else:
               if info.value:
                  os.write( self.fd, "true" )
               else:
                  os.write( self.fd, "false" )
         else:
            assert False, "Unknown JSON type %d for '%s'" % (
                             info.jsonType, info.name )

   # This is to print banners/headers: a no-op in json output format
   def addFrills( self, fmtStr, argsTuple=None ):
      if self.outputFormat == TEXT:
         if argsTuple:
            os.write( self.fd, fmtStr % argsTuple )
         else:
            os.write( self.fd, fmtStr )

   # cooler (less efficient) python api for usage with "with"

   # with Lists
   def list( self, listName ):
      return self.List( self, listName )

   class List( object ):
      def __init__( self, printer, listName ):
         self.printer = printer
         self.listName = listName

      def __enter__( self ):
         self.printer.startList( self.listName )
         return self

      def __exit__( self, excType, value, excTraceback ):
         self.printer.endList( )

   def listEntry( self ):
      return self.ListEntry( self )

   class ListEntry( object ):
      def __init__( self, printer ):
         self.printer = printer

      def __enter__( self ):
         self.printer.startListEntry( )
         return self

      def __exit__( self, excType, value, excTraceback ):
         self.printer.endListEntry( )

   # with Dicts
   def dict( self, dictName ):
      return self.Dict( self, dictName )

   class Dict( object ):
      def __init__( self, printer, dictName ):
         self.printer = printer
         self.dictName = dictName

      def __enter__( self ):
         self.printer.startDict( self.dictName )
         return self

      def __exit__( self, excType, value, excTraceback ):
         self.printer.endDict( )

   def dictEntry( self, attr ):
      return self.DictEntry( self, attr )

   class DictEntry( object ):
      def __init__( self, printer, attr ):
         self.printer = printer
         self.attr = attr

      def __enter__( self ):
         self.printer.startDictEntry( self.attr )
         return self

      def __exit__( self, excType, value, excTraceback ):
         self.printer.endDictEntry( )

class AttrInfo():
   def __init__( self, printer, name, jsonType, value=None ):
      self.name = name         # Attribute Name in JSON
      self.jsonType = jsonType # Attribute Type in JSON
      self.isKey_ = False      # only print in TEXT since JSON prints it separately
      self.hasValue = False    # make sure we print Attr that were inited
      self.outputFormat = printer.outputFormat
      self.value = value
      if value:
         self.hasValue = True

   # For outputFormat JSON, returns self
   # For outputformat TEXT, returns self.value
   def getArg( self ):
      assert self.hasValue, "Trying to print '%s' but value not set" % self.name
      if self.outputFormat == TEXT:
         return self.value
      else:
         return self

   def setValue( self, value ):
      self.value = value
      self.hasValue = True

   def isKey( self ):
      self.isKey_ = True

class IntAttrInfo( AttrInfo ):
   def __init__( self, printer, name, value=None ):
      AttrInfo.__init__( self, printer, name, JSON_NUMBER, value=value )

class StringAttrInfo( AttrInfo ):
   def __init__( self, printer, name, value=None ):
      AttrInfo.__init__( self, printer, name, JSON_STRING, value=value )

class FloatAttrInfo( AttrInfo ):
   def __init__( self, printer, name, value=None ):
      AttrInfo.__init__( self, printer, name, JSON_FLOAT, value=value )
