# Copyright (c) 2010-2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, print_function

import time
import os

import BasicCli
import Cell
import CliCommand
import CliMatcher
import FileCliUtil
import FileUrl
import LazyMount
import ShowCommand
import SimpleConfigFile
import Tac
import Url
from CliModel import Enum, Int, Model

helpZeroTouch = 'ZeroTouch configuration'
helpScriptExecTimeout = ( 'Change timeout for the downloaded script '
                          'to finish execution' )
helpTimeout = 'Time allowed in seconds for the downloaded script to finish execution'
helpCancel = 'Cancel ZeroTouch and reload the switch'
helpEnable = 'Enable ZeroTouch'
helpDisable = 'Disable ZeroTouch and reload the switch'

zeroTouchConfigPath = "flash:/zerotouch-config"

ztConfig = None
ztStatus = None
reloadConfig = None
bannerConfig = None

#--------------------------------------------------
# `[no] zerotouch enable` command
#--------------------------------------------------
class ZeroTouchEnable( CliCommand.CliCommandClass ):
   syntax = '''zerotouch enable'''
   noSyntax = syntax
   hidden = True
   data = {
      'zerotouch': helpZeroTouch,
      'enable' : helpEnable,
   }

   @staticmethod
   def handler( mode, args ):
      if mode.session_.startupConfig():
         ztConfig.enabled = True
      else:
         # 'zerotouch enable' can only be done from startup-config
         mode.addError( "Configuration ignored: "
                        "ZeroTouch cannot be enabled interactively. "
                        "To enable ZeroTouch, delete startup-config"
                        " and reload the switch." )

   @staticmethod
   def noHandler( mode, args ):
      bannerConfig.loginBanner = ""
      ztConfig.enabled = False

BasicCli.GlobalConfigMode.addCommandClass( ZeroTouchEnable )

#--------------------------------------------------
# `zerotouch cancel` command
#--------------------------------------------------
class ZeroTouchCancel( CliCommand.CliCommandClass ):
   syntax = '''zerotouch cancel'''
   data = {
      'zerotouch': helpZeroTouch,
      'cancel': helpCancel,
   }

   @staticmethod
   def handler( mode, args ):
      ztConfig.cancelled = True

BasicCli.UnprivMode.addCommandClass( ZeroTouchCancel )

#--------------------------------------------------
# `zerotouch disable` command
#--------------------------------------------------
class ZeroTouchDisable( CliCommand.CliCommandClass ):
   syntax = '''zerotouch disable'''
   data = {
      'zerotouch': helpZeroTouch,
      'disable': helpDisable,
   }

   @staticmethod
   def handler( mode, args ):
      wasEnabled = False
      if ztConfig.enabled:
         wasEnabled = True
         ztConfig.enabled = False

      urlContext = Url.Context( mode.entityManager, mode.session_.disableAaa_ )
      zeroTouchConfigUrl = Url.parseUrl( zeroTouchConfigPath, urlContext )
      try:
         filename = zeroTouchConfigUrl.localFilename()
         config = SimpleConfigFile.SimpleConfigFileDict( filename, True )
         config[ 'DISABLE' ] = "True"
      except ( IOError, SimpleConfigFile.ParseError ):
         mode.addError( "Error writing to %s" % zeroTouchConfigPath )
         return

      if wasEnabled:
         # Sleep for 1 second before rebooting
         time.sleep( 1 )

         if Cell.cellType() != "fixed":
            reloadConfig.all = True

         reloadConfig.reloadTime = Tac.now()

BasicCli.UnprivMode.addCommandClass( ZeroTouchDisable )

#--------------------------------------------------
# `zerotouch script-exec-timeout <timeout>` command
#--------------------------------------------------
class ZeroTouchSET( CliCommand.CliCommandClass ):
   syntax = '''zerotouch script-exec-timeout <timeout>'''
   data = {
      'zerotouch': helpZeroTouch,
      'script-exec-timeout': helpScriptExecTimeout,
      '<timeout>': CliMatcher.IntegerMatcher( 1, 24*60*60, helpdesc=helpTimeout ),
   }

   @staticmethod
   def handler( mode, args ):
      ztConfig.scriptExecTimeout = args[ '<timeout>' ]

BasicCli.UnprivMode.addCommandClass( ZeroTouchSET )

#-----------------------------------------------------------------------------------
# show zerotouch
#-----------------------------------------------------------------------------------
class ZeroTouch( Model ):
   mode = Enum( values=( "active", "enabled", "disabled" ), help="ZeroTouch Mode" )
   scriptTimeout = Int( help="Script timeout in secs" )

   def render( self ):
      print()
      print( "ZeroTouch Mode:", self.mode.capitalize() )
      print( "Script Exec Timeout:", self.scriptTimeout )

class ShowZeroTouch( ShowCommand.ShowCliCommandClass ):
   syntax = 'show zerotouch'
   data = {
            'zerotouch': 'ZeroTouch status',
          }
   cliModel = ZeroTouch

   @staticmethod
   def handler( mode, args ):
      if ztConfig.enabled:
         if ztStatus.enabled:
            state = 'active'
         else:
            state = 'enabled'
      else:
         state = 'disabled'

      return ZeroTouch( mode=state, scriptTimeout=ztConfig.scriptExecTimeout )

BasicCli.addShowCommandClass( ShowZeroTouch )

#-------------------------------------------------------------
# Plugin method - Mount the objects we need from Sysdb
#-------------------------------------------------------------
def Plugin( entityManager ):
   global ztConfig, ztStatus, reloadConfig, bannerConfig
   ztConfig = LazyMount.mount( entityManager,
                               "zerotouch/config",
                               "ZeroTouch::Config",
                               "w" )
   ztStatus = LazyMount.mount( entityManager,
                               "zerotouch/status",
                               "ZeroTouch::Status",
                               "r" )
   reloadConfig = LazyMount.mount( entityManager,
                                   Cell.path( "sys/reload/config" ),
                                   "System::Reload::Config",
                                   "w" )
   bannerConfig = LazyMount.mount( entityManager,
                                   "sys/banner/config/cli",
                                   "System::BannerConfig",
                                   "w" )

   # Add to checkCopyFileAllowed functions in Cli.FileCliUtil
   def checkZeroTouch( mode, surl, durl ):
      startupConfig = FileUrl.localStartupConfig( *Url.urlArgsFromMode( mode ) )
      if ( durl == startupConfig and
           ztConfig.enabled and
           "ZTP_EXEC_SCRIPT" not in os.environ ):
         mode.addError( "Cannot copy to startup-config when ZeroTouch is enabled" )
         return False
      return True

   FileCliUtil.checkCopyFileAllowed.addExtension( checkZeroTouch )
