#!/usr/bin/env python
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

import CliCommand
import CliMatcher
import CliPlugin.AaaCli as AaaCli
from CliPlugin.IntfCli import Intf
import CliPlugin.VrfCli as VrfCli
import CliPlugin.Xmpp as Xmpp
import DscpCliLib
import HostnameCli
from ReversibleSecretCli import reversibleSecretCliExpression

matcherSourceInterface = CliMatcher.KeywordMatcher( 'source-interface',
      helpdesc='Interface providing source IP address for XMPP' )

#--------------------------------------------------------------------------------
# [ no | default ] connection unencrypted permit
#--------------------------------------------------------------------------------
class ConnectionUnencryptedPermitCmd( CliCommand.CliCommandClass ):
   syntax = 'connection unencrypted permit'
   noOrDefaultSyntax = 'connection unencrypted permit ...'
   data = {
      'connection': 'Set connection parameters',
      'unencrypted': 'Set options related to unencrypted connections',
      'permit': 'Permit option',
   }
   handler = Xmpp.XmppConfigMode.configurePermitUnencrypted
   noOrDefaultHandler = Xmpp.XmppConfigMode.defaultPermitUnencrypted

Xmpp.XmppConfigMode.addCommandClass( ConnectionUnencryptedPermitCmd )

#--------------------------------------------------------------------------------
# [ no | default ] domain DOMAIN
#--------------------------------------------------------------------------------
class DomainCmd( CliCommand.CliCommandClass ):
   syntax = 'domain DOMAIN'
   noOrDefaultSyntax = 'domain ...'
   data = {
      'domain': 'Set the default domain name',
      'DOMAIN': CliMatcher.PatternMatcher( pattern='\\S+',
         helpdesc='XMPP DomainName', helpname='WORD' ),
   }

   @staticmethod
   def handler( mode, args ):
      Xmpp.XmppConfigMode.configureDomainName( mode, args[ 'DOMAIN' ] )

   noOrDefaultHandler = Xmpp.XmppConfigMode.noDomainName

Xmpp.XmppConfigMode.addCommandClass( DomainCmd )

#--------------------------------------------------------------------------------
# [ no | default ] qos dscp DSCP
#--------------------------------------------------------------------------------
DscpCliLib.addQosDscpCommandClass( Xmpp.XmppConfigMode,
                                   Xmpp.XmppConfigMode.setDscp,
                                   Xmpp.XmppConfigMode.noDscp )

#--------------------------------------------------------------------------------
# [ no | default ] server SERVER [ port PORT ]
#--------------------------------------------------------------------------------
class ServerCmd( CliCommand.CliCommandClass ):
   syntax = 'server SERVER [ port PORT ]'
   noOrDefaultSyntax = 'server ...'
   data = {
      'server': 'Configure the XMPP server',
      'SERVER': HostnameCli.IpAddrOrHostnameMatcher(
         helpdesc='Hostname or IP address of XMPP server' ),
      'port': 'Configure the XMPP server port',
      'PORT': CliMatcher.IntegerMatcher( 1, 65535, helpdesc='XMPP server port' ),
   }
   handler = Xmpp.XmppConfigMode.configureServer
   noOrDefaultHandler = Xmpp.XmppConfigMode.noServer

Xmpp.XmppConfigMode.addCommandClass( ServerCmd )

#--------------------------------------------------------------------------------
# [ no | default ] session privilege PRIVILEGE
#--------------------------------------------------------------------------------
class SessionPrivilegeCmd( CliCommand.CliCommandClass ):
   syntax = 'session privilege PRIVILEGE'
   noOrDefaultSyntax = 'session privilege ...'
   data = {
      'session': 'Set parameters specific to CLI sessions created for XMPP',
      'privilege': 'Set a default session privilege level for XMPP commands',
      'PRIVILEGE': CliMatcher.IntegerMatcher( 0, 15,
         helpdesc='Set a default session privilege level for XMPP commands' ),
   }
   handler = Xmpp.XmppConfigMode.configurePrivLevel
   noOrDefaultHandler = Xmpp.XmppConfigMode.noPrivLevel

Xmpp.XmppConfigMode.addCommandClass( SessionPrivilegeCmd )

#--------------------------------------------------------------------------------
# [ no | default ] shutdown
#--------------------------------------------------------------------------------
class ShutdownCmd( CliCommand.CliCommandClass ):
   syntax = 'shutdown'
   noOrDefaultSyntax = 'shutdown'
   data = {
      'shutdown': 'Disable XMPP',
   }
   handler = Xmpp.XmppConfigMode.shutdown
   noOrDefaultHandler = Xmpp.XmppConfigMode.noShutdown

Xmpp.XmppConfigMode.addCommandClass( ShutdownCmd )

#--------------------------------------------------------------------------------
# source-interface IPINTF
#--------------------------------------------------------------------------------
class SourceInterfaceIpintfCmd( CliCommand.CliCommandClass ):
   syntax = 'source-interface IPINTF'
   noOrDefaultSyntax = 'source-interface ...'
   data = {
      'source-interface': matcherSourceInterface,
      'IPINTF': Intf.matcherWithIpSupport,
   }
   handler = Xmpp.XmppConfigMode.setSrcIntfName
   noOrDefaultHandler = Xmpp.XmppConfigMode.noSrcIntfName

Xmpp.XmppConfigMode.addCommandClass( SourceInterfaceIpintfCmd )

#--------------------------------------------------------------------------------
# [ no | default ] vrf VRF
#--------------------------------------------------------------------------------
class VrfCmd( CliCommand.CliCommandClass ):
   syntax = 'vrf VRF'
   noOrDefaultSyntax = 'vrf ...'
   data = {
      'vrf': 'Configure VRFs',
      'VRF': VrfCli.vrfMatcher,
   }
   handler = Xmpp.XmppConfigMode.configureVrf
   noOrDefaultHandler = Xmpp.XmppConfigMode.noVrf

Xmpp.XmppConfigMode.addCommandClass( VrfCmd )

#--------------------------------------------------------------------------------
# [ no | default ] switch-group GROUPNAME [ password PASSWORD ]
#--------------------------------------------------------------------------------
class SwitchGroupCommand( CliCommand.CliCommandClass ):
   syntax = "switch-group GROUPNAME [ password PASSWORD ]"
   noOrDefaultSyntax = "switch-group [ { GROUPNAME } ]"
   data = {
      'switch-group' : 'Configure switch-groups to join',
      'GROUPNAME' : CliMatcher.PatternMatcher( '.+',
                                             helpname='WORD',
                                             helpdesc='Group name(s) to add' ),
      'password' : 'Assign the XMPP password',
      'PASSWORD' : reversibleSecretCliExpression( 'PASSWORD' )
   }
   handler = Xmpp.XmppConfigMode.addGroup
   noOrDefaultHandler = Xmpp.XmppConfigMode.removeGroups

Xmpp.XmppConfigMode.addCommandClass( SwitchGroupCommand )

#--------------------------------------------------------------------------------
# [ no | default ] switch-group GROUPNAME [ password PASSWORD ]
#--------------------------------------------------------------------------------
class XmppUsernameCommand( CliCommand.CliCommandClass ):
   syntax = "username USERNAME password PASSWORD"
   noOrDefaultSyntax = "username ..."
   data = {
      'username' : 'Define your XMPP username and password',
      'USERNAME' : AaaCli.usernameMatcher,
      'password' : 'Assign the XMPP password',
      'PASSWORD' : reversibleSecretCliExpression( 'PASSWORD' )
   }
   handler = Xmpp.XmppConfigMode.configureUsername
   noOrDefaultHandler = Xmpp.XmppConfigMode.noUsername

Xmpp.XmppConfigMode.addCommandClass( XmppUsernameCommand )
