#!/usr/bin/env python
# Copyright (c) 2013 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import time

import Ark
import CliModel
import IpLibConsts
import Tac
import ArnetModel
import IntfModels


def _formatStatusStr( status, maxLen ):
   """Formats the presence status string for CLI display."""
   # Convert newlines to spaces so multi-line statuses display correctly
   # and strip off trailing whitespace.
   status = ' '.join( status.splitlines() )
   # Truncate the string to fit within the maximum display length
   if len( status ) > maxLen:
      return status[ :maxLen-3 ] + '...'
   return status


NEIGHBOR_STATES = { 'unknown' : 'unknown',
                    'notPresent' : 'not present',
                    'present' : 'present' }

CONNECTION_STATES = { 'unknown': 'unknown',
                      'notConnected': 'not connected',
                      'connected': 'connected',
                      'encryptionFailed': 'Encryption Failed',
                      'authenticationFailed': 'authentication failed' }

def _neighborStateStr( state ):
   return NEIGHBOR_STATES.get( state, 'unknown' )


def _connStateStr( state ):
   return CONNECTION_STATES.get( state, 'unknown' )


class Neighbor( CliModel.Model ):
   """Information about an XMPP neighbor."""

   state = CliModel.Enum(
      help='The presence state of this neighbor learnt from XMPP',
      values=NEIGHBOR_STATES.keys() )

   statusMessage = CliModel.Str( help='The XMPP status message of this neighbor' )
   lastUpdateTimestamp = CliModel.Float( help='Last time the neighbor status '
                                         'changed in seconds since epoch, UTC' )


class Group( CliModel.Model ):
   """Information about a XMPP multi user chat room."""

   enabled = CliModel.Bool( help='Whether we are connected to this room' )


class Neighbors( CliModel.Model ):

   neighbors = CliModel.Dict(
      help='Collection of XMPP neighbor details keyed by neighbor base JID',
      valueType=Neighbor )

   def render( self ):
      fmt = '%-30.30s %-15.15s %-25.25s'
      print fmt % ( 'Neighbor', 'State', 'Time Since Last Change' )
      print fmt % ( '-' * 30, '-' * 15, '-' * 25 )
      neighbors = sorted( self.neighbors.items() )
      for name, neighbor in neighbors:
         if neighbor.lastUpdateTimestamp:
            timestamp = Ark.timestampToStr( time.time() -
                                            neighbor.lastUpdateTimestamp )
         else:
            timestamp = 'never'
         print fmt % ( name, _neighborStateStr( neighbor.state ), timestamp )
      print ''
      fmt = '%-30.30s %-40.40s'
      print fmt % ( 'Neighbor', 'Status Message' )
      print fmt % ( '-' * 30, '-' * 40 )
      for name, neighborStatus in neighbors:
         print fmt % ( name,
                       _formatStatusStr( neighborStatus.statusMessage, 40 ) )


class Groups( CliModel.Model ):

   groups = CliModel.Dict(
      help='Collection of group details keyed by switch group name '
      '(aka XMPP chat room JIDs)', valueType=Group )

   def render( self ):
      if not self.groups:
         print 'Not in any switch-groups'
      for group in sorted( self.groups ):
         print group


class XmppStatus( CliModel.Model ):

   enabled = CliModel.Bool( help='Whether the XMPP agent is enabled' )
   serverAddress = CliModel.Str(
      help='The IP address or hostname of the XMPP server' )
   username = CliModel.Str( help='The XMPP base jid (email address) '
                            'for the XMPP server' )
   domainName = CliModel.Str( help='Default domain name appended to bare usernames' )

   port = CliModel.Int( help='The TCP port number used for XMPP connections' )
   vrfName = CliModel.Str( help='The current VRF name the agent is running in' )
   srcInterface = IntfModels.Interface( help='Interface providing source IP address'
                               'for XMPP' )
   srcIpAddr = ArnetModel.Ip4Address( help='Source ip address of Xmpp packets' )
   connectionState = CliModel.Enum(
      help='XMPP server connection status',
      values=CONNECTION_STATES.keys() )
   statusMessage = CliModel.Str( help="The XMPP agent's current status message" )
   initialPrivLevel = CliModel.Int( help='The initial privilege level for '
                                    'command authorization when not using AAA' )

   def render( self ):
      print 'XMPP Server: %s port %s' % ( self.serverAddress, self.port )
      if self.vrfName != IpLibConsts.DEFAULT_VRF:
         print 'Using VRF:', self.vrfName
      if self.srcInterface != Tac.Value( "Arnet::IntfId" ):
         print 'Using source interface: %s (ip address: %s)' \
               % ( self.srcInterface, self.srcIpAddr )
      print 'Client username:', self.username
      print 'Default domain:', self.domainName
      print 'Default privilege level for received commands: %d' % (
         self.initialPrivLevel )
      print 'Connection status:', _connStateStr( self.connectionState )
