# Copyright (c) 2013 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, print_function, division
import Arnet
from CliModel import Bool
from CliModel import Dict
from CliModel import OrderedDict
from CliModel import Enum
from CliModel import Float
from CliModel import Model
from CliModel import Str
from CliModel import Int
from CliModel import List 
from CliModel import Submodel
import CliPlugin.IntfCli
import CliPlugin.EthIntfModel as EthIntfModel
import Hexdump
from IntfModels import Interface
from collections import OrderedDict as ordDict
from collections import namedtuple
from XcvrLib import getXcvrSlotName

import Ark 
import Tac 
import datetime
import re

SUPPORTED_FORMATS = frozenset( [ "default", "csv" ] )

paramTypeToStr = {
      'temperature' : 'Temperature',
      'caseTemperature': 'Case temperature',
      'voltage' : 'Voltage',
      'txBias' : 'TX bias current',
      'txPower' : 'Optical TX power',
      'rxPower' : 'Optical RX power',
      'snr' : 'SNR',
      'residualISI' : 'Residual ISI',
      'pam4Transitions' : 'Level transitions',
      'tecCurrent' : 'TEC current error',
      'laserFreq' : 'Frequency error',
      'laserTemp' : 'Temperature error',
      'laserTemperature' : 'Laser temperature',
      'laserTempAux' : 'Laser temperature',
      'preFecBERCurr' : 'Pre-FEC BER',
      'uncorrectedBERCurr' : 'Uncorrected BER',
      'preFECBERAvg' : 'Pre-FEC BER avg',
      'preFECBERMax' : 'Pre-FEC BER max',
      'preFECBERMin' : 'Pre-FEC BER min',
      'preFECBERHighAlarmExceeded' : 'Pre-FEC BER high alarm exceeded',
      'preFECBERHighWarnExceeded' : 'Pre-FEC BER high warn exceeded',
      'uncorrectedBERAvg' : 'Uncorrected BER avg',
      'uncorrectedBERMax' : 'Uncorrected BER max',
      'uncorrectedBERMin' : 'Uncorrected BER min',
      'uncorrectedBERHighAlarmExceeded' : 'Uncorrected BER high alarm exceeded',
      'uncorrectedBERHighWarnExceeded' : 'Uncorrected BER high warn exceeded',
      'errFramesAvg' : 'Post-FEC errored frames ratio avg',
      'errFramesMin' : 'Post-FEC errored frames ratio min',
      'errFramesMax' : 'Post-FEC errored frames ratio max',
      'errFramesCur' : 'Post-FEC errored frames ratio',
      'tempAlarmExceeded' : 'Temperature alarm exceeded',
      'tempWarnExceeded' : 'Temperature warn exceeded',
      'netFecUncBlkCount' : 'Post-FEC UCB',
      'rxChromDispersion' : 'Chromatic dispersion',
      'rxDiffGroupDelay' : 'Differential group delay',
      'rxStateOfPolarization' : 'State of polarization',
      'rxNetworkPDL': 'Network PDL',
      'rxQFactor' : 'Q-factor',
      'rxCarrierFreqOffset' : 'Carrier frequency offset',
      'netOsnrEstimate' : 'Network OSNR estimate',
      'laserAge' : 'Laser age',
      'txLaserAge' : 'TX laser age',
      'rxLaserAge' : 'RX laser age',
      'txLaserPower' : 'TX laser power',
      'rxLaserPower' : 'RX laser power',
      'txLaserTemperature': 'TX laser temperature',
      'rxLaserTemperature': 'RX laser temperature',
      'rxChannelPower': 'RX channel power',
      'coherentTxPower': 'TX power',
      'totalRxPower': 'RX total power',
      'rollOff' : 'Roll-off factor',
      'aggrTxPwr' : 'Aggregate TX Power',
      'aggrRxPwr' : 'Aggregate RX Power',
      'cdHighShortLink' : 'Chromatic dispersion (short link)',
      'cdLowLongLink' : 'Chromatic dispersion (long link)',
      'osnr' : 'Received OSNR estimate',
      'esnr' : 'Received ESNR estimate',
      'dgd' : 'Differential group delay',
      'sopmd' : 'SOPMD',
      'pdl' : 'Polarization dependent loss',
      'cfo' : 'Carrier frequency offset',
      'evm' : 'Error vector magnitude',
      'sopRoc' : 'SOP rate of change',
      'mer' : 'Modulation error ratio',
      'txChannelPower' : 'TX power',
      'cmisLaserFrequency' : 'Laser frequency',
      'unknown' : 'Unknown',
      }

# Format of this is { channel : { paramType : label } }
polsDomAdditionalLabel = {
      1 : { 'txBias' : '(Booster)',
            'txPower' : '(Booster output)',
            'rxPower' : '(Pre-amp input)',
            'laserTemperature' : '(Booster)',
          },
      2 : { 'txBias' : '(Pre-amp)',
            'txPower' : '(Pre-amp output)',
            'rxPower' : '(Booster input)',
            'laserTemperature' : '(Pre-amp)',
          },
      }

# Helper to return a presentable string for a parameter type
def _parameterTypeToStr( paramType ):
   if paramType in paramTypeToStr.keys():
      return paramTypeToStr[ paramType ]
   return paramTypeToStr[ 'unknown' ]

#----------------------------------------------------------------------------
#
# Helpers for "show interface [ <interface> ] transceiver [csv]"
#
#----------------------------------------------------------------------------

def _printHeaderDefault():
   headerFormat = "%-10s%-11s%-10s%-10s%-10s%-10s%-13s"
   print( "If device is externally calibrated, only calibrated values are printed." )
   print( "N/A: not applicable, Tx: transmit, Rx: receive." )
   print( "mA: milliamperes, dBm: decibels (milliwatts)." )
   print( headerFormat % ( "", "", "", "Bias", "Optical", "Optical", "" ) )
   print( headerFormat % ( "", "Temp", "Voltage", "Current",
                          "Tx Power", "Rx Power", "" ) )
   print( headerFormat % ( "Port", "(Celsius)", "(Volts)", "(mA)",
                          "(dBm)", "(dBm)", "Last Update" ) )
   print( headerFormat % ( "-" * 5, "-" * 9, "-" * 8, "-" * 8,
                          "-" * 8, "-" * 8, "-" * 19 ) )

def _printHeaderCsv():
   print ( "Last Update,Port (Interface Name),Xcvr Serial Number,"
           "Media type,Temperature (Celsius),Voltage (Volts),Current (mA),"
           "Tx Power (dBm),Rx Power (dBm)" )

#----------------------------------------------------------------------------
#
# Helpers for "show interfaces [ <interface> ] transceiver dom [thresholds]"
#
#----------------------------------------------------------------------------

def _printDomHeaderDefault():
   print( "Ch: Channel, N/A: not applicable, TX: transmit, RX: receive" )
   print( "mA: milliamperes, dBm: decibels (milliwatts), C: Celsius, V: Volts" )

def indicator( val, highAlarm, highWarn, lowWarn, lowAlarm ):
   # If value is -inf(not supported) or the BER is initialized to it's max value
   # or if high alarm is same as low alarm (which means high warn is same as low
   # warn, which means all thresholds are same) then don't print the indicator.
   pm = Tac.Value( 'Xcvr::Sff8436PerformanceMonitoring' )
   if val == float( "-inf" ) or val == pm.berMaxValue or highAlarm == lowAlarm:
      return ''
   if ( highAlarm and val >= highAlarm ) or ( lowAlarm and val <= lowAlarm ):
      return 'ALARM'
   if ( highWarn and val >= highWarn ) or ( lowWarn and val <= lowWarn ):
      return 'WARN'
   return ''

def _printDomThresHeaderDefault():
   print( "Ch: Channel, mA: milliamperes, dBm: decibels (milliwatts)," )
   print( "C: Celsius, V: Volts, NA or N/A: not applicable." )

#----------------------------------------------------------------------------
#
# Helpers for "show interface [ <interface> ] transceiver detail [csv]"
#
#----------------------------------------------------------------------------
def _printThresHeaderDefault():
   print( "mA: milliamperes, dBm: decibels (milliwatts), " 
          + "NA or N/A: not applicable." )
   print( "A2D readouts (if they differ), are reported in parentheses." )
   print( "The threshold values are calibrated." )
   
def _printThresHeaderPerParameterDefault( name, unit ):
   headerFormat = "%-11s%-14.14s%-12.12s%-12.12s%-12.12s%-12.12s"
   units = "(" + unit + ")"
   # If the name of parameter is longer than 12 letters, use the white space as
   # a delimiter to break it into two lines.
   print( headerFormat % ( "", "" if len( name ) < 13 
         else name[ : name.rfind( ' ', 0, 13 ) ],
         "High Alarm", "High Warn", "Low Alarm", "Low Warn" ) )
   print( headerFormat % ( "", name if len( name ) < 13
         else name[ name.rfind( ' ', 0, 13 ) + 1 : ], 
         "Threshold", "Threshold", "Threshold", "Threshold" ) )
   print( headerFormat % ( "Port", units, units, units, units, units ) )
   print( headerFormat % ( "-" * 7, "-" * 12, "-" * 10, "-" * 10,
                          "-" * 10, "-" * 10 ) )

def _printThresHeaderPerParameterCsv( name, unit ):
   units = '(' + unit + ')'
   print( 'Port (Interface Name), ' + name + units + 
          ' High Alarm Threshold, ' + units + 
          ' High Warn Threshold, ' + units + 
          ' Low Alarm Threshold, ' + units + 
          ' Low Warn Threshold, ' + units )


#----------------------------------------------------------------------------
#
# Models for "show interface [ <interface> ] transceiver [detail] [csv]"
#
#----------------------------------------------------------------------------
class InterfaceTransceiverBase( Model ):
   def renderModelNoDetail( self, intfName, printFmt ):
      assert printFmt in SUPPORTED_FORMATS, "Unrecognized DOM output format"
      shortName = CliPlugin.IntfCli.Intf.getShortname( intfName )

      numFields = { "default" : 6, "csv" : 8 }

      naFmt = { "default" : "%-11s%-11s%-10s%-10s%-10s%-9s%-10s",
                "csv" : "%s,%s,%s,%s,%s,%s,%s,%s,%s", }

      if printFmt == 'default':
         print( naFmt[ printFmt ] % ( ( shortName, ) + 
                                     ( "N/A", ) * numFields[ printFmt ] ) )
      elif printFmt == 'csv':
         print( naFmt[ printFmt ] % ( ( "N/A", ) + ( shortName, ) + 
                                     ( "N/A", ) * ( numFields[ printFmt ] - 1 ) ) )

   def renderModelDetailed( self, intfName, param ):
      shortName = CliPlugin.IntfCli.Intf.getShortname( intfName )

      numDataFields = 5
      naFmt = "%-11.11s%-14.14s%-12.12s%-12.12s%-12.12s%-12.12s"

      print( naFmt % ( ( shortName, ) + ( "N/A", ) * numDataFields ) )


class InterfaceTransceiverDetailThresholds( Model ):
   highAlarm = Float( help="High alarm threshold for parameter", optional=True )
   highWarn = Float( help="High warning threshold for parameter", optional=True )
   lowAlarm = Float( help="Low alarm threshold for parameter", optional=True )
   lowWarn = Float( help="Low warning threshold for parameter", optional=True )


class InterfaceTransceiverDetails( Model ):
   temperature = \
       Submodel( valueType=InterfaceTransceiverDetailThresholds,
                 help="Temperature alarm and warning thresholds in Celsius.",
                 optional=True )

   voltage = \
       Submodel( valueType=InterfaceTransceiverDetailThresholds,
                 help="Voltage alarm and warning thresholds in Volts.",
                 optional=True )

   txBias = \
       Submodel( valueType=InterfaceTransceiverDetailThresholds,
                 help="Bias current alarm and warning thresholds in mA.",
                 optional=True )

   txPower = \
       Submodel( valueType=InterfaceTransceiverDetailThresholds,
                 help="TX power alarm and warning thresholds in dBm.",
                 optional=True )

   rxPower = \
       Submodel( valueType=InterfaceTransceiverDetailThresholds,
                 help="RX power alarm and warning thresholds in dBm.",
                 optional=True )

   totalRxPower = \
       Submodel( valueType=InterfaceTransceiverDetailThresholds,
                 help="Total RX power alarm and warning thresholds in dBm.",
                 optional=True )

   _domThresholdOverridden = Bool(
      help="Indicates when the DOM thresholds are set by configuration" )

class InterfaceTransceiver( InterfaceTransceiverBase ):
   vendorSn = Str( help="Transceiver serial number" )
   mediaType = Str( help="Media type" )
   narrowBand = Bool( help="Narrowband Transceiver", optional=True )
   updateTime = Float( help="Last update time in UTC" )
   temperature = Float( help="Temperature in Celsius", optional=True )
   voltage = Float( help="Voltage in Volts", optional=True )
   txBias = Float( help="Current in mA" , optional=True)
   txPower = Float( help="TX power in dBm", optional=True )
   rxPower = Float( help="RX power in dBm", optional=True )
   totalRxPower = Float( help="Total RX power in dBm", optional=True )

   details = Submodel( valueType=InterfaceTransceiverDetails,
                       help="Warning and alarm thresholds",
                       optional=True )

   def _opt( self, value ):
      return value if value is not None else float( "-inf" )

   def renderModelNoDetail( self, intfName, printFmt ):
      assert printFmt in SUPPORTED_FORMATS, "Unrecognized DOM output format"

      fmt = { "default" : "%-11s%-11.2f%-10.2f%-9.2f%-10.2f%-10.2f%-10s",
              "csv": "%s,%s,%s,%s,%.2f,%.2f,%.2f,%.2f,%.2f", }

      shortName = CliPlugin.IntfCli.Intf.getShortname( intfName )
      updateTime = Ark.timestampToStr( self.updateTime, now=Tac.utcNow() )
      if printFmt == 'default':
         fieldValues = ( shortName,
                         self._opt( self.temperature ),
                         self._opt( self.voltage ),
                         self._opt( self.txBias ),
                         self._opt( self.txPower ),
                         self._opt( self.rxPower ),
                         updateTime )
      elif printFmt == 'csv':
         fieldValues = ( updateTime,
                         shortName,
                         self.vendorSn,
                         self.mediaType,
                         self._opt( self.temperature ),
                         self._opt( self.voltage ),
                         self._opt( self.txBias ),
                         self._opt( self.txPower ),
                         self._opt( self.rxPower ) )
   
      domLine = fmt[ printFmt ] % fieldValues
      print( domLine.replace( "-inf", " N/A" ) )

   def renderModelDetailed( self, intfName, param ):
      shortName = CliPlugin.IntfCli.Intf.getShortname( intfName )

      # Note that even though the CLI accepts the optional "csv"
      # keyword, the only difference between "CSV" and non-"CSV"
      # output is the heading. There is no difference in the
      # per-interface output.

      operValue = float( "-inf" )
      highAlarm = float( "-inf" )
      highWarn = float( "-inf" )
      lowAlarm = float( "-inf" )
      lowWarn = float( "-inf" )

      thresh = None
      if param == "Temperature":
         thresh = self.details.temperature
         operValue = self.temperature
      elif param == "Voltage":
         thresh = self.details.voltage
         operValue = self.voltage
      elif param == "Current":
         thresh = self.details.txBias
         operValue = self.txBias
      elif param == "Tx Power":
         thresh = self.details.txPower
         operValue = self.txPower
      elif param == "Rx Power":
         thresh = self.details.rxPower
         operValue = self.rxPower
      elif param == "Total Rx Power":
         thresh = self.details.totalRxPower
         operValue = self.totalRxPower

      if thresh:
         # Some xcvr types support only a subset of the thresholds in
         # the TAC model.
         highAlarm = thresh.highAlarm
         highWarn = thresh.highWarn
         lowAlarm = thresh.lowAlarm
         lowWarn = thresh.lowWarn

      fmt = "%-11s%-14.2f%-12.2f%-12.2f%-12.2f%-10.2f"
      fieldValues = ( shortName,
                      self._opt( operValue ),
                      self._opt( highAlarm ),
                      self._opt( highWarn ),
                      self._opt( lowAlarm ),
                      self._opt( lowWarn ) )

      domLine = fmt % fieldValues
      # pylint: disable-msg=W0212
      if self.details._domThresholdOverridden and \
         param in [ "Tx Power", "Rx Power" ]:
         domLine += "*"
      if self.narrowBand or param != "Total Rx Power":
         # Show "Total Rx Power" section only if transceiver is narrowband.
         print( domLine.replace( "-inf", " N/A" ) )


class InterfacesTransceiver( Model ):
   _printFmt = Enum( values=SUPPORTED_FORMATS, help="Type of print format" )
   _detailed = Bool( help="Include warning and alarm thresholds" )
   interfaces = Dict( keyType=Interface, valueType=InterfaceTransceiverBase,
                     help="Mapping between Interface name and the transceiver info" )
   _domThresholdOverrideEnabled = Bool( help="Indicates whether the DOM threshold"
                                             " override is enabled" )

   def render( self ):
      if self._detailed:
         self._renderDetailed()
      else:
         self._renderNotDetailed()

   def _renderDetailed( self ):
      assert self._printFmt in SUPPORTED_FORMATS, "Unrecognized DOM output format"
      printThresHdrFn = { "default" : _printThresHeaderPerParameterDefault,
                          "csv" : _printThresHeaderPerParameterCsv, }

      if not self.interfaces:
         return

      if self._printFmt == "default":
         _printThresHeaderDefault()
         if self._domThresholdOverrideEnabled:
            print( "(*) One or more thresholds are set by configuration." )

      paramUnitList = [ ( 'Temperature', 'Celsius' ),
                        ( 'Voltage', 'Volts' ), 
                        ( 'Current', 'mA' ),
                        ( 'Tx Power', 'dBm' ),
                        ( 'Rx Power', 'dBm' ) ]
      
      for interface in self.interfaces.itervalues():
         if type( interface ) is InterfaceTransceiver and interface.narrowBand:
            paramUnitList.append( ( 'Total Rx Power', 'dBm' ) )
            break

      for (param, unit) in paramUnitList:
         printThresHdrFn[ self._printFmt ]( param, unit )

         for intf in Arnet.sortIntf( self.interfaces ):
            self.interfaces[ intf ].renderModelDetailed( intf, param )


   def _renderNotDetailed( self ):
      assert self._printFmt in SUPPORTED_FORMATS, "Unrecognized DOM output format"
      printHeaderFn = { "default" : _printHeaderDefault,
                        "csv" : _printHeaderCsv }

      if not self.interfaces:
         # If there are no interfaces, skip even printing headers.
         return

      printHeaderFn[ self._printFmt ]()

      for intf in Arnet.sortIntf( self.interfaces ):
         self.interfaces[ intf ].renderModelNoDetail( intf, self._printFmt )

#----------------------------------------------------------------------------
#
# Models for "show interface [ <interface> ] transceiver dom [thresholds]"
#
#----------------------------------------------------------------------------

def subPort( intfName ):
   return re.sub( 'Ethernet|Management', 'Port ', getXcvrSlotName( intfName ) )

class InterfaceTransceiverDomBase( Model ):
   displayName = Str( help="Interfaces corresponding to the channel of the"
                           " transceiver", optional=True )
   def renderModelNoThresholds( self, intfName, printFmt, printCommonDomInfo ):
      assert printFmt == 'default', "Unrecognized DOM output format"
      print( "" )
      print( subPort( intfName ) )
      print( "Last update: %s" % "N/A" )
      print( "%50s" % ( "Value" ) )
      print( "%54s" % ( "------------" ) )
      fmtValue = "   %-40s%5s %s"
      print( fmtValue % ( "Temperature", "N/A", "N/A" ) )
      print( fmtValue % ( "Voltage", "N/A", "N/A" ) )
      print( fmtValue % ( "TX bias current", "N/A", "N/A" ) )
      print( fmtValue % ( "Optical TX power",  "N/A", "N/A" ) )
      print( fmtValue % ( "Optical RX power", "N/A", "N/A" ) )

   def renderModelThresholds( self, intfName, printCommonDomInfo ):
      print( "" )
      print( subPort( intfName ) )
      print( "Last update: %s" % "N/A" )

      print( "%-38s%11s%11s%11s%11s" % \
            ( " ", "High Alarm", "High Warn", "Low Warn", "Low Alarm" ) )
      print( "%-32s%-8s%-11s%-11s%-11s%-9s%6s%11s" % ( " ", "Value", "Threshold",
                  "Threshold", "Threshold", "Threshold", "Unit", "Indicator" ) )
      print( "%-26s%-77s" % ( " ", "-" * 75 ) )
      fmtValue = "   %-23s%11s%12s%11s%11s%11s %5s%11s"

      print( fmtValue % ( "Temperature", 
            "N/A", "N/A", "N/A", "N/A", "N/A", "N/A", "N/A" ) )
      print( fmtValue % ( "Voltage", 
            "N/A", "N/A", "N/A", "N/A", "N/A", "N/A", "N/A" ) )
      print( fmtValue % ( "TX bias current", 
            "N/A", "N/A", "N/A", "N/A", "N/A", "N/A", "N/A" ) )
      print( fmtValue % ( "Optical TX power", 
            "N/A", "N/A", "N/A", "N/A", "N/A", "N/A", "N/A" ) )
      print( fmtValue % ( "Optical RX power", 
            "N/A", "N/A", "N/A", "N/A", "N/A", "N/A", "N/A" ) )


class InterfaceTransceiverDomParameter( Model ):
   unit = Str( help="Parameter unit" )
   channels = Dict( keyType=str, valueType=float, 
         help="Per channel parameter value. "+\
               "'-' for the channel indicates a channel independent parameter" )
   threshold = Submodel( valueType=InterfaceTransceiverDetailThresholds,
                       help="Threshold for monitored parameter",
                       optional=True )

# pylint complains that instances of OrderdedDict have not iteritems member -
# which is true at the time of compilation
# pylint: disable-msg=E1101
class InterfaceTransceiverDom( InterfaceTransceiverDomBase ):
   vendorSn = Str( help="Transceiver serial number" )
   mediaType = Str( help="Media type" )
   updateTime = Float( help="Last update time in UTC" )
   _hasMultiChannelDomParam = Bool( help="Has atleast one dom parameter that is"
                                    " channel specific", optional=True )
   _dualLaserModulePresent = Bool( help="Dual laser DCO module requires special "
                                        "dom formatting" )
   parameters = \
         Dict( keyType=str, valueType=InterfaceTransceiverDomParameter, 
               help="A mapping of parameter name to per channel value and "+\
                     "thresholds" )
   _paramOrder = \
         List( valueType=str, help="List of parameters in the order they "+\
               "should be printed" )

   _usePolsLabels = Bool( help="Whether to render using labels specific to "
                               "POLS modules", optional=True )

   def _getCommonDomParams( self ):
      return [ paramType for paramType in self._paramOrder \
               if '-' in self.parameters[ paramType ].channels ]

   def _getSingleChannelDomParamChannelLabel( self, intfName ):
      channel = ""
      if "Channel" not in self.displayName:
         # Get channel number from any of the channel specific DOM parameters
         for param in self.parameters.itervalues():
            if '-' not in param.channels:
               # We found one, get the channel number entry.
               channelNum = int( next( iter( param.channels ) ) )
               # Since we are in a single channel domain,
               # we know that all channel specific DOM parameters will have the same
               # channel number inserted through the LaneMapping CLI infra
               # Breaking here since we found one entry, although it's not necessary.
               break
         channel = "(Channel %d)" % channelNum
         if self._usePolsLabels:
            if channelNum == 1:
               channel = "(Line)"
            elif channelNum == 2:
               channel = "(Local)"
      return channel

   def _getPerChannelDomParamLabel( self, paramType, channel ):
      label = _parameterTypeToStr( paramType )
      if self._usePolsLabels:
         channelLabels = polsDomAdditionalLabel.get( int( channel ) )
         if channelLabels and paramType in channelLabels:
            label = " ".join( [ label, channelLabels[ paramType ] ] )
      return label

   def _opt( self, value ):
      return value if value is not None else float( "-inf" )

   def renderModelNoThresholds( self, intfName, printFmt, printCommonDomInfo ):
      assert printFmt == 'default', "Unrecognized DOM output format"
      commonDomParams = self._getCommonDomParams()
      # ---------------------- Common-Transceiver-Info----------------------- #
      # Do not print DOM headers and common params if it is already printed
      # for the port this interface belongs to.
      # Essentially this:
      #
      # Port 1
      # Last update: x:xx:xx ago
      #                           Value       Unit
      #                          --------    -------
      # Temperature                 x           C
      # Voltage                     x           V
      # Aggregate TX Power          x           dBm
      # Aggregate RX Power          x           dBm
      # .....
      # ...
      # once per port.
      if printCommonDomInfo:
         print( "" )
         # ----------------------DOM - Header-------------------------------- #
         updateTime = Ark.timestampToStr( self.updateTime, now=Tac.utcNow() )
         print( subPort( intfName ) )
         print( "Last update: %s" % updateTime )
         print( "%50s" % ( "Value" ) )
         print( "%56s" % ( 16*'-' ) )
         # ---------------------Common-DOM-Params---------------------------- #
         for commonDomParam in commonDomParams:
            if any( berFmt in _parameterTypeToStr( commonDomParam ) for
                    berFmt in [ 'BER', 'Post-FEC errored frames' ] ):
               fmtValue = "   %-38s%4.2e %s"
            elif "BlkCount" in commonDomParam:
               fmtValue = "   %-38s%8d %s"
            else:
               fmtValue = "   %-40s%6.2f %s"
            domLine = fmtValue % ( _parameterTypeToStr( commonDomParam ),
                                   self.parameters[ commonDomParam ].channels[ '-' ],
                                   self.parameters[ commonDomParam ].unit )
            print( domLine.replace( "-inf", " N/A" ) )
      # -----------------------Channel-Specific-DOM-Info--------------------- #

      if self._dualLaserModulePresent or \
         self.mediaType == "400GBASE-ZR":
         # dual laser dco module has only a single channel and should not group
         # parameters under a single interface header
         return

      # ---------------------Handle-Interface-Display-Name------------------- #
      # Essentially this:
      # EthernetX/Ethernet... Lane X ( Here Lane label is optional ) ( Channel X )
      # ( Channel X is optional as well ). Lane and Channel are dependent
      # on hasMultiChannelDomParam being False.
      # This is printed for every interface.

      channel = ""
      fmtValue = "      %-8s%8s %26.2f %s"
      if not self._hasMultiChannelDomParam:
         channel = self._getSingleChannelDomParamChannelLabel( intfName )
         fmtValue = "   %-34s%12.2f %s"
      print( "%s %s" % ( self.displayName, channel ) )

      # ----------------------Channel-Specific-DOM-Params-------------------- #
      # Essentially this:
      # If self._hasMultiChannelDomParam is True ...
      #    TX bias current
      #       Channel    1                 x           mA
      #       Channel    2                 x           mA
      #       Channel    ....
      #       ...
      #    Optical TX power
      #       Channel    1                 x           dBm
      #       Channel    2                 x           dBm
      #       Channel    ....
      #       ....
      #    Optical RX power
      #       ....
      # Else ...
      #    TX bias current                 x           mA
      #    Optical TX power                x           dBm
      #    Optical RX power                x           dBm
      # This is printed for every interface as well.
      for paramType in self._paramOrder:
         # Do not print common dom params again.
         if paramType in commonDomParams:
            continue
         param = self.parameters[ paramType ]
         unit = param.unit
         # Print paramType label only if it is a channel specific dom param
         if self._hasMultiChannelDomParam:
            print( "   %s" % _parameterTypeToStr( paramType ) )
         for chan, val in sorted( param.channels.iteritems(),
                                  key = lambda ( chan ): int( chan[ 0 ] ) ):
            if self._hasMultiChannelDomParam:
               domLine = fmtValue % ( "Channel", chan, val, unit )
            else:
               paramTypeStr = self._getPerChannelDomParamLabel( paramType, chan )
               domLine = fmtValue % ( paramTypeStr, val, unit )
            print( domLine.replace( "-inf", " N/A" ) )

   def renderModelThresholds( self, intfName, printCommonDomInfo ):
      commonDomParams = self._getCommonDomParams()
      # ---------------------Common-Transceiver-Info------------------------------- #
      # Do not print DOM threshold headers and common params if it is already printed
      # for the port this interface belongs to.
      # Essentially this:
      #
      # Port 1
      # Last update: x:xx:xx ago
      #              High Alarm   High Warn   Low Warn   Low Alarm
      #      Value    Threshold   Threshold  Threshold   Threshold   Unit   Indicator
      #
      # Temp ......
      # Vol  ....
      # .....
      # ...
      # once per port ( Format fit to 85 chars for comments )
      if printCommonDomInfo:
         print( "" )
         # ----------------------DOM-Theshold-Header------------------------------- #
         updateTime = Ark.timestampToStr( self.updateTime, now=Tac.utcNow() )
         print( subPort( intfName ) )
         print( "Last update: %s" % updateTime )
         print( "%-45s%11s%11s%11s%11s" % \
            ( " ", "High Alarm", "High Warn", "Low Warn", "Low Alarm" ) )
         print( "%-39s%-8s%-11s%-11s%-11s%-9s%6s%11s" % ( " ", "Value", "Threshold",
                     "Threshold", "Threshold", "Threshold", "Unit", "Indicator" ) )
         print( "%-33s%-77s" % ( " ", "-" * 75 ) )
         # -------------------------Common-DOM-Params------------------------------ #
         for commonDomParam in commonDomParams:
            param = self.parameters[ commonDomParam ]
            thresh = param.threshold
            if not param.threshold:
               continue
            if 'BER' in commonDomParam:
               fmtValue = "   %-30s%11.2e%12.2e%11.2e%11.2e%11.2e %5s%11s"
            else:
               fmtValue = "   %-30s%11.2f%12.2f%11.2f%11.2f%11.2f %5s%11s"
            parameterStr = _parameterTypeToStr( commonDomParam )
            domLine = fmtValue % ( parameterStr,
                         param.channels[ '-' ], self._opt( thresh.highAlarm ),
                         self._opt( thresh.highWarn ),  self._opt( thresh.lowWarn ),
                         self._opt( thresh.lowAlarm ), param.unit,
                         indicator( param.channels[ '-' ], thresh.highAlarm,
                                    thresh.highWarn, thresh.lowWarn,
                                    thresh.lowAlarm ) )
            print( domLine.replace( "-inf", " N/A" ) )

      # ------------------Channel-Specific-DOM-Threshold-Info--------------------- #

      if self._dualLaserModulePresent or \
         self.mediaType == "400GBASE-ZR":
         # dual laser dco module has only a single channel and should not group
         # parameters under a single interface header
         return

      # -----------------------Handle-Interface-Display-Name---------------------- #
      # Essentially this:
      # EthernetX/Ethernet... Lane X ( Here Lane lable is optional ) ( Channel X )
      # ( Channel X is optional as well ). Lane and Channel are dependent
      # on hasMultiChannelDomParam being False.
      # This is printed for every interface.
      channel = ""
      fmtValue = "      %-15s%8s%15.2f%12.2f%11.2f%11.2f%11.2f %5s%11s"
      if not self._hasMultiChannelDomParam:
         channel = self._getSingleChannelDomParamChannelLabel( intfName )
         fmtValue = "   %-34s%7.2f%12.2f%11.2f%11.2f%11.2f %5s%11s"
      print( "%s %s" % ( self.displayName, channel ) )

      # --------------------Channel-Specific-DOM-Threshold-Params---------------- #
      # Essentially this:
      # If self._hasMultiChannelDomParam is True ...
      #    TX bias current
      #       Channel    1     x     x      x     x     x     mA
      #       Channel    2     .....
      #       Channel    ....
      #       ...
      #    Optical TX power
      #       Channel    1     x     x      x     x     x     dBm
      #       Channel    2    ......
      #       Channel    ....
      #       ....
      #    Optical RX power
      #       ....
      # Else ...
      #    TX bias  current    x     x      x     x     x     mA
      #    Optical TX power    x     x      x     x     x     dBm
      #    Optical RX power    .....
      # This is printed for every interface as well.
      for paramType in self._paramOrder:
         # Do not print common dom params again.
         if paramType in commonDomParams:
            continue
         param = self.parameters[ paramType ]
         thresh = param.threshold
         if not param.threshold:
            continue
         unit = param.unit
         # Print paramType label only if it is a channel specific dom param
         if self._hasMultiChannelDomParam:
            print( "   %s" % _parameterTypeToStr( paramType ) )
         for chan, val in sorted( param.channels.iteritems(),
                                  key = lambda ( chan ): int( chan[ 0 ] ) ):
            indicatorLabel = indicator( val, thresh.highAlarm, thresh.highWarn,
                                        thresh.lowWarn, thresh.lowAlarm )
            if self._hasMultiChannelDomParam:
               domLine = fmtValue % ( "Channel", chan, val,
                                      self._opt( thresh.highAlarm ),
                                      self._opt( thresh.highWarn ),
                                      self._opt( thresh.lowWarn ),
                                      self._opt( thresh.lowAlarm ), unit,
                                      indicatorLabel )
            else:
               paramTypeStr = self._getPerChannelDomParamLabel( paramType, chan )
               domLine = fmtValue % ( paramTypeStr,
                            val, self._opt( thresh.highAlarm ),
                            self._opt( thresh.highWarn ),
                            self._opt( thresh.lowWarn ),
                            self._opt( thresh.lowAlarm ), unit,
                            indicatorLabel )
            print( domLine.replace( "-inf", " N/A" ) )

class InterfacesTransceiverDom( Model ):
   _printFmt = Enum( values=SUPPORTED_FORMATS, help="Type of print format" )
   _detailed = Bool( help="Include warning and alarm thresholds" )
   interfaces = Dict( keyType=Interface, valueType=InterfaceTransceiverDomBase,
                     help="Mapping between interface name and the transceiver \
                           information" )
   _xcvrNames = List( valueType=str,
                      help="Transceiver slot names corresponding to the \
                            requested interfaces", optional=True )
   _interfacesOrder = List( valueType=str,
                            help="Order of interfaces to be displayed",
                            optional=True )

   def render( self ):
      if self._detailed:
         self._renderThresholds()
      else:
         self._renderNotThresholds()

   def _renderThresholds( self ):
      assert self._printFmt == 'default', "Unrecognized DOM output format"

      if not self.interfaces:
         # If there are no interfaces, skip even printing headers.
         return

      _printDomThresHeaderDefault()

      # Print according to interfacesOrder and not sorted( self.interfaces ... )
      for intf in self._interfacesOrder:
         xcvrName = getXcvrSlotName( intf )
         self.interfaces[ intf ].renderModelThresholds( intf,
                                            xcvrName not in self._xcvrNames )
         # Having duplicates won't affect the output.
         self._xcvrNames.append( xcvrName )

   def _renderNotThresholds( self ):
      assert self._printFmt == 'default', "Unrecognized DOM output format"
      printHeaderFn = _printDomHeaderDefault

      if not self.interfaces:
         # If there are no interfaces, skip even printing headers.
         return

      printHeaderFn()

      # Print according to interfacesOrder and not sorted( self.interfaces ... )
      for intf in self._interfacesOrder:
         xcvrName = getXcvrSlotName( intf )
         self.interfaces[ intf ].renderModelNoThresholds( intf, self._printFmt,
                                                   xcvrName not in self._xcvrNames )
         # Having duplicates won't affect the output.
         self._xcvrNames.append( xcvrName )

#--------------------------------------------------------------------------------
#
# Models for 
# "show interfaces [ <interface> ] transceiver performance-monitoring [thresholds]"
#
#--------------------------------------------------------------------------------

def _printPerformanceMonitoringHeaderDefault():
   print( "Ch: Channel, N/A: not applicable" )

class InterfaceTransceiverPerformanceMonitoringInterval( Model ):
   intervalStartTime = Float( help="Time when this interval started in UTC", 
         optional=True )
   updateTime = Float( help="Last update time in UTC", optional=True )
   parameters = \
         Dict( keyType=str, 
               valueType=InterfaceTransceiverDomParameter, 
                      help="A mapping of performance monitoring parameter names "+\
                            "to per channel value and thresholds in this interval",
                            optional=True )

class InterfaceTransceiverPerformanceMonitoring( Model ):
   performanceMonitoringIntervals = \
         Dict( keyType=int, 
               valueType=InterfaceTransceiverPerformanceMonitoringInterval, 
                      help="A mapping of performance monitoring interval number to \
per interval information. Interval numbers are integers starting from 0. Interval 0 \
is the interval in progress,Interval 1 is the most recently completed interval and \
so on", optional=True )
   _paramOrder = \
         List( valueType=str, help="List of parameters in the order they "+\
               "should be printed" )
   
   def _opt( self, value ):
      return value if value is not None else float( "-inf" )
  
   def _printPerformanceMonitoring( self, paramOrder, pm ):
      fmtValueBER = "  %-43s%3s%13.2e"
      fmtValueCount = "  %-43s%3s%13d"
      fmtTime = "  %-43s%3s%13s"
      intervalStartTime = \
            Ark.timestampToStr( pm.intervalStartTime, now=Tac.utcNow() )
      updateTime = Ark.timestampToStr( pm.updateTime, now=Tac.utcNow() )
      print( fmtTime % ( "Started", "-", intervalStartTime ) )
      print( fmtTime % ( "Last update", "-", updateTime ) )
      for _paramType in paramOrder:
         param = pm.parameters[ _paramType ]
         alreadyPrintedParamType = False
         for chan, value in param.channels.iteritems():
            # Print parameter name only once if this parameter is same as previous
            paramType = _parameterTypeToStr( _paramType ) \
                  if not alreadyPrintedParamType else ''
            fmtValue = fmtValueCount \
                  if 'Exceeded' in _paramType else fmtValueBER
            pmLine = fmtValue % ( paramType, chan, value )
            print( pmLine.replace( "2.05e+10", "0.00e+00" ) )
            alreadyPrintedParamType = True

   def _printPerformanceMonitoringThreshold( self, paramOrder, pm ):
      fmtValueBER = "  %-43s%3s%13.2e%11.2e%11.2e%11.2e%11.2e%10s"
      fmtValueCount = "  %-43s%3s%13d%11s%11s%11s%11s%10s"
      fmtTime = "  %-43s%3s%13s%11s%11s%11s%11s%10s"
      intervalStartTime = \
            Ark.timestampToStr( pm.intervalStartTime, now=Tac.utcNow() )
      updateTime = Ark.timestampToStr( pm.updateTime, now=Tac.utcNow() )
      print( fmtTime % ( "Started", "-", intervalStartTime, " ", 
                         " ", " ", " ", " " ) )
      print( fmtTime % ( "Last update", "-", updateTime, " ", " ", " ", " ", " " ) )
      for _paramType in paramOrder:
         param = pm.parameters[ _paramType ]
         thresh = param.threshold
         # Print parameter name only once if this parameter is same as previous
         alreadyPrintedParamType = False
         for chan, value in param.channels.iteritems():
            paramType = _parameterTypeToStr( _paramType ) \
                  if not alreadyPrintedParamType else ''
            fmtValue = fmtValueCount \
                  if 'Exceeded' in _paramType else fmtValueBER
            pmLine = fmtValue % ( paramType, chan, value, 
               self._opt( thresh.highAlarm ), self._opt( thresh.highWarn ), 
               self._opt( thresh.lowWarn ), self._opt(  thresh.lowAlarm ),
               indicator( value, thresh.highAlarm, thresh.highWarn,
                  thresh.lowWarn, thresh.lowAlarm ) )
            pmLine = pmLine.replace( "2.05e+10", "0.00e+00" )
            pmLine = pmLine.replace( "-inf", " N/A" )
            print( pmLine )
            alreadyPrintedParamType = True

   def renderModelNoThresholds( self, intfName ):
      print( intfName )
      fmtHeader = "%-45s%3s%13s"
      print( fmtHeader % ( "Parameter", "Ch", "Value" ) )
      print( "-" * 61 )
      # Print current interval's information
      if 0 in self.performanceMonitoringIntervals.keys():
         print( "Current Interval 0" )
         self._printPerformanceMonitoring( 
               self._paramOrder, self.performanceMonitoringIntervals[ 0 ] )
      
      if 1 in self.performanceMonitoringIntervals.keys():
      # Print previous interval 1 information
         print( "Interval 1" )
         self._printPerformanceMonitoring( 
               self._paramOrder, self.performanceMonitoringIntervals[ 1 ] )

   def renderModelThresholds( self, intfName ):
      print( intfName )
      fmtHeader = "%-45s%3s%13s%11s%11s%11s%11s%10s"
      print( fmtHeader % ( " ", " ", " ", "High Alarm", "High Warn", 
            "Low Warn", "Low Alarm", " " ) )
      print( fmtHeader % ( "Parameter", 
            "Ch", "Value", "Threshold", "Threshold", "Threshold", 
            "Threshold", "Indicator" ) )
      print( "-" * 116 )
      
      if 0 in self.performanceMonitoringIntervals.keys():
         print( "Current Interval 0" )
         self._printPerformanceMonitoringThreshold( 
               self._paramOrder, self.performanceMonitoringIntervals[ 0 ] )
      
      if 1 in self.performanceMonitoringIntervals.keys():
         print( "Interval 1" )
         self._printPerformanceMonitoringThreshold( 
               self._paramOrder, self.performanceMonitoringIntervals[ 1 ] )

class InterfacesTransceiverPerformanceMonitoring( Model ):
   _detailed = Bool( help="Include warning and alarm thresholds" )
   performanceMonitoringPeriodSeconds = \
         Int( help="Performance monitoring period in seconds",
              optional=True ) 
   interfaces = Dict( keyType=Interface, 
                     valueType=InterfaceTransceiverPerformanceMonitoring,
                     help="Mapping between interface name and the "+\
                           "performance monitoring information" )
   _performanceMonitoringConfigured = Bool( help="Performance monitoring "+\
                                           "configured on any interface or not" )
   def render( self ):
      
      if self._performanceMonitoringConfigured:
         # If there are interfaces to print, print the key first.
         if self.interfaces:
            _printPerformanceMonitoringHeaderDefault()
            print( "\n" )
         # Now print the performance monitoring period in format days, hh:mm:ss
         print( "Performance monitoring period : %s " % \
                str( datetime.timedelta( 
                  seconds = self.performanceMonitoringPeriodSeconds ) ) )

      if not self.interfaces:
         # If there are no interfaces, skip even printing headers.
         return
      
      if self._detailed:
         self._renderThresholds()
      else:
         self._renderNotThresholds()

   def _renderThresholds( self ):
      
      for intf in Arnet.sortIntf( self.interfaces ):
         print( "\n" )
         self.interfaces[ intf ].renderModelThresholds( intf )

   def _renderNotThresholds( self ):

      for intf in Arnet.sortIntf( self.interfaces ):
         print( "\n" )
         self.interfaces[ intf ].renderModelNoThresholds( intf )

#--------------------------------------------------------------------------------
#
# Models for "show interfaces [ <interface> ] transceiver hardware"
#
#--------------------------------------------------------------------------------

class InterfacesTransceiverHardwarePower( Model ):
   configuredTxPower = Float( help="TX power in dBm", optional=True )
   _configuredTxPowerDefault = Bool( help="Indicates if the TX power is "
                                         "configured to its default value",
                                    optional=True )
   configuredRxPower = Float( help="RX power in dBm", optional=True )
   _configuredRxPowerDefault = Bool( help="Indicates if the RX power is "
                                         "configured to its default value",
                                    optional=True )
   operationalRxAttenuation = Float( help="RX attenuation in dB", optional=True )
   _tunableLaserEnabled = Bool( help="Indicates if (laser disabled) should be "
                                     "displayed for coherent optics" )
   _dualLaserModulePresent = Bool( help="Indicates that the receiver tuning"
                                        "should be displayed for coherent optics" )

   def configuredTxPowerDefaultIs( self, val ):
      self._configuredTxPowerDefault = val
   def configuredRxPowerDefaultIs( self, val ):
      self._configuredRxPowerDefault = val

   def renderModel( self ):
      printFmt = "{0}: {1}{2}"
      laserDisabledMsg = ""
      if not self._tunableLaserEnabled and not self._dualLaserModulePresent:
         laserDisabledMsg = " (laser disabled)"
      if self.configuredTxPower is not None:
         if self._tunableLaserEnabled:
            print( printFmt.format( "Configured TX power (dBm)",
                                    self.configuredTxPower, " (default)" if \
                                    self._configuredTxPowerDefault else "" ) )
         else:
            print( printFmt.format( "Configured TX power (dBm)",
                                    str( self.configuredTxPower ) + \
                                    laserDisabledMsg, "" ) )
      if self.configuredRxPower is not None:
         print( printFmt.format( "Configured Rx Power (dBm)", self.configuredRxPower,
                                 "(Default)" if self._configuredRxPowerDefault else \
                                 "" ) )
      if self.operationalRxAttenuation is not None:
         print( printFmt.format( "Operational Rx Attenuation (dB)",
                                 self.operationalRxAttenuation if \
                                 self._tunableLaserEnabled else \
                                 str( self.operationalRxAttenuation ) + \
                                 laserDisabledMsg, "" ) )

class InterfacesTransceiverHardwareTuning( Model ):
   # Attributes for tuning
   _outputSelect = Bool( help="Selects tuning data set to display" )
   _tunableLaserEnabled = Bool( help="Indicates which display format to use for "
                                     "coherent optics tuning data" )
   _dualLaserModulePresent = Bool( help="Indicates that the receiver tuning"
                                        "should be displayed for coherent optics" )
   configuredFrequency = Float( help="Configured TX frequency in GHz",
                                optional=True )
   configuredRxFrequency = Float( help="Configured RX frequency in GHz",
                                  optional=True )
   configuredRxFineFrequency = Float( help="Configured RX fine frequency in GHz",
                                      optional=True )
   _configuredFrequencyUnsupported = Bool( help="Indicates if the configured "
                                                "frequency is unsupported",
                                           optional=True )
   computedWavelength = Float( help="Computed wavelength in nm", optional=True )
   computedRxWavelength = Float( help="Computed RX wavelength in nm", optional=True )
   operationalFrequency = Float( help="Operational TX frequency in GHz",
                                 optional=True )
   operationalRxFrequency = Float( help="Operational RX frequency in GHz",
                                   optional=True )
   _operationalFrequencyDefault = Bool( help="Indicates if the operational "
                                             "frequency is configured to its "
                                             "default value",
                                        optional=True )
   operationalWavelength = Float( help="Operational TX wavelength in nm",
                                  optional=True )
   operationalRxWavelength = Float( help="Operational RX wavelength in nm",
                                    optional=True )
   configuredChannel = Int( help="Configured channel", optional=True )
   _configuredChannelUnsupported = Bool( help="Indicates if the configured "
                                             "channel is unsupported",
                                         optional=True )
   configuredGrid = Float( help="Configured grid in GHz", optional=True )
   computedFrequency = Float( help="Computed frequency in GHz", optional=True )
   operationalChannel = Int( help="Operational channel", optional=True )
   _operationalChannelDefault = Bool( help="Indicates if the operational "
                                          "channel is configured to its "
                                          "default value",
                                     optional=True )
   operationalGrid = Float( help="Operational grid in GHz", optional=True )
   _operationalGridDefault = Bool( help="Indicates if the operational grid is "
                                       "configured to its default value",
                                  optional=True )
   gridSpacingCapabilities = Str( help="Grid spacing capabilities in GHz",
                                  optional=True )
   _gridSpacingCapabilitiesUnknown = Bool( help="Indicates if the grid-spacing "
                                               "capabilities are unknown",
                                          optional=True )

   def configuredFrequencyUnsupportedIs( self, val ):
      self._configuredFrequencyUnsupported = val
   def operationalFrequencyDefaultIs( self, val ):
      self._operationalFrequencyDefault = val
   def configuredChannelUnsupportedIs( self, val ):
      self._configuredChannelUnsupported = val
   def operationalChannelDefaultIs( self, val ):
      self._operationalChannelDefault = val
   def operationalGridDefaultIs( self, val ):
      self._operationalGridDefault = val
   def gridSpacingCapabilitiesUnknownIs( self, val ):
      self._gridSpacingCapabilitiesUnknown = val

   def renderModel( self ):
      printFmt = "{0}: {1}{2}"
      if self._outputSelect:
         if self._dualLaserModulePresent:
            # dual laser module must also display output for RX laser
            print( printFmt.format( "Configured TX frequency (GHz)",
                                    "{0:,}".format( self.configuredFrequency ) if \
                                       self._tunableLaserEnabled else "none", "" ) )
            print( printFmt.format( "Configured RX frequency (GHz)",
                                    self.configuredRxFrequency, "" ) )
            print( printFmt.format( "Configured RX fine frequency (GHz)",
                                    self.configuredRxFineFrequency, "" ) )
            print( printFmt.format( "Computed TX wavelength (nm)",
                                    "%.2f" % self.computedWavelength if \
                                       self._tunableLaserEnabled else "none", "" ) )
            print( printFmt.format( "Computed RX wavelength (nm)",
                                    "%.2f" % self.computedRxWavelength, "" ) )
            print( printFmt.format( "Operational TX frequency (GHz)",
                                    "{0:,}".format( self.operationalFrequency ) if \
                                       self._tunableLaserEnabled else "none", "" ) )
            print( printFmt.format( "Operational RX frequency (GHz)",
                                    "{0:,}".format( self.operationalRxFrequency ),
                                    "" ) )
            print( printFmt.format( "Operational TX wavelength (nm)",
                                    "%.2f" % self.operationalWavelength if \
                                       self._tunableLaserEnabled else "none", "" ) )
            print( printFmt.format( "Operational RX wavelength (nm)",
                                    "%.2f" % self.operationalRxWavelength, "" ) )
         else:
            print( printFmt.format( "Configured frequency (GHz)",
                                    self.configuredFrequency, "(unsupported)" if \
                                       self._configuredFrequencyUnsupported else \
                                       "" ) )
            print( printFmt.format( "Computed wavelength (nm)",
                                    "%.2f" % self.computedWavelength, "" ) )
            print( printFmt.format( "Operational frequency (GHz)",
                                    "{0:,}".format( self.operationalFrequency ),
                                    "(default)" if self._operationalFrequencyDefault\
                                       else "" ) )
            print( printFmt.format( "Operational wavelength (nm)",
                                    "%.2f" % self.operationalWavelength, "" ) )
      else:
         if self._tunableLaserEnabled:
            print( printFmt.format( "Configured channel", self.configuredChannel,
                                    "(unsupported)" if \
                                    self._configuredChannelUnsupported else "" ) )
            print( printFmt.format( "Configured grid (GHz)", self.configuredGrid, 
                                    "" ) )
            print( printFmt.format( "Computed frequency (GHz)",
                                   "{0:,}".format( self.computedFrequency ), "" ) )
            print( printFmt.format( "Computed wavelength (nm)",
                                   "%.2f" % self.computedWavelength, "" ) )
            print( printFmt.format( "Operational channel", self.operationalChannel,
                                   "(default)" if self._operationalChannelDefault \
                                         else "" ) )
            print( printFmt.format( "Operational grid (GHz)", self.operationalGrid,
                                   "(default)" if self._operationalGridDefault \
                                         else "" ) )
            print( printFmt.format( "Operational frequency (GHz)",
                                   "{0:,}".format( self.operationalFrequency ),
                                   "(default)" if self._operationalFrequencyDefault \
                                         else "" ) )
            print( printFmt.format( "Operational wavelength (nm)",
                                   "%.2f" % self.operationalWavelength, "" ) )
         else:
            print( printFmt.format( "Configured channel", "none (default)", "" ) )
            print( printFmt.format( "Configured grid (GHz)", "none (default)", 
                                    "" ) ) 
            print( printFmt.format( "Computed frequency (GHz)", "none (default)",
                                   "" ) )
            print( printFmt.format( "Computed wavelength (nm)", "none (default)",
                                   "" ) )
            print( printFmt.format( "Operational channel", "none (laser disabled)",
                                   "" ) )
            print( printFmt.format( "Operational grid (GHz)", 
                                    "none (laser disabled)", "" ) )
            print( printFmt.format( "Operational frequency (GHz)",
                                    "none (laser disabled)", "" ) )
            print( printFmt.format( "Operational wavelength (nm)",
                                    "none (laser disabled)", "" ) )
         if self.gridSpacingCapabilities or self._gridSpacingCapabilitiesUnknown:
            print( printFmt.format( "Grid spacing capabilities (GHz)",
                                    self.gridSpacingCapabilities, "(unknown)" if \
                                         self._gridSpacingCapabilitiesUnknown else \
                                         "" ) )

class InterfacesTransceiverHardwareXcvrCapabilities( Model ):
   speed = Enum( values=EthIntfModel.CapabilitiesMixIn.speedValues, 
                 help='Transceiver speed' )
   duplex = Enum( values=( 'full', 'half', 'unknown' ),
                  help='Duplex communication capabilities with this speed' )
   laneCount = Int( optional=True, help='Number of lanes on the interface' )
   _showLanes = Bool( optional=True, help='Lane count is set' )

   def showLanesIs( self, showLanes ):
      self._showLanes = showLanes

   def showLanes( self ):
      return self._showLanes

class InterfacesTransceiverHardware( Model ):
   mediaType = Str( help="The effective media type, normally the detectedMediaType, "
                    "but can be overridden by cli config" )
   detectedMediaType = Str( help="The media type set in device's IDPROM" )
   cableType = Enum( values=[ 'CA-N', 'CA-S', 'CA-L' ],
                     help="Type of copper cable", optional=True )
   wavelength = Float( help="Wavelength of fiber (nm)", optional=True )
   _wavelengthPrecNm = Bool( help="Determines the formatting of the printed "
                                  "wavelength", optional=True )
   tuning = Submodel( valueType=InterfacesTransceiverHardwareTuning,
                      help="Tuning information", optional=True )
   txDisabled = Bool( help="Optical transmitter disabled", optional=True )
   power = Submodel( valueType=InterfacesTransceiverHardwarePower,
                     help="Power information", optional=True )
   powerCharacteristic = Float( help="Maximum module power in watts" )
   maxPowerSupportedCharacteristic = Float(
         help="Maximum slot power in watts" )
                            
   def wavelengthPrecNmIs( self, val ):
      self._wavelengthPrecNm = val

   def renderModel( self, intfName ):
      print( "Name: %s" % CliPlugin.IntfCli.Intf.getShortname( intfName ) )
      print( "Media type: %s" % self.mediaType )
      print( "Maximum module power (W): %s" % self.powerCharacteristic )
      power = self.maxPowerSupportedCharacteristic or 'N/A'
      print( "Maximum slot power (W): %s" % power )
      if self.wavelength is not None:
         # Casting to provide the same output as the unconverted command.
         print( "Wavelength (nm): %s" % ( int( self.wavelength ) if \
               self._wavelengthPrecNm else float( self.wavelength ) ) )
      if self.cableType is not None:
         print( "Cable type: %s" % self.cableType )
      if self.tuning:
         self.tuning.renderModel()
      if self.txDisabled is not None:
         print( "TX laser status: %s" % ( "disabled" if self.txDisabled
                                          else "enabled" ) )
      if self.power:
         self.power.renderModel()
      if self.detectedMediaType != self.mediaType:
         print( "Detected media type: %s" % self.detectedMediaType )
      print( "" )

class InterfacesTransceiverHardwareBase( Model ):
   interfaces = Dict( keyType=Interface, valueType=InterfacesTransceiverHardware,
                      help="Mapping between interface name and the transceiver info"
                      )
   _xcvrNotPresent = Str( help="Print xcvr presence error" )

   def xcvrNotPresentIs( self, intf ):
      self._xcvrNotPresent = intf

   def render( self ):
      if not self.interfaces:
         if self._xcvrNotPresent is not None:
            print( "%s - transceiver not present" % self._xcvrNotPresent )
         return

      for intf in Arnet.sortIntf( self.interfaces ):
         self.interfaces[ intf ].renderModel( intf )

#--------------------------------------------------------------------------------
#
# Models for "show idprom transceiver [<interfaces>]"
#
#--------------------------------------------------------------------------------

class InterfacesTransceiverIdpromData( Model ):
   registers = List( valueType=int, help='Eeprom data as an integer' )
   regWidth = Int( help="Eeprom register width in bits" )

   def dataIs( self, dataStr ):
      self.registers = [ ord( x ) for x in dataStr ]

   def renderModel( self ):
      uniCodeData = ''.join( [ chr( x ) for x in self.registers ] )
      # Don't condense repeating rows.
      print( Hexdump.hexdump( uniCodeData, condense=False ) )

class InterfacesTransceiverIdpromPages( Model ):
   pages = OrderedDict( keyType=str, valueType=InterfacesTransceiverIdpromData,
                        help="Mapping of page name to register" )

   def renderModel( self, intfName, ):
      for ( pageNum, page ) in self.pages.iteritems():
         header = "%s IDPROM Page " % intfName
         if pageNum.startswith( 'upper' ) or pageNum.startswith( 'lower' ):

            m =  re.match( r"(?P<upperOrLower>(upper|lower))(?P<address>\d+)",
                           pageNum )
            address = int( m.groupdict()[ 'address' ] )
            address = format( address, 'x' ).rjust( 2, '0' )
            header += "%sh %s" % ( address, m.groupdict()[ 'upperOrLower' ].title() )
         else:
            header += pageNum
         if page.regWidth:
            header += " Register Width %u" % page.regWidth
         print( header )
         page.renderModel()

class InterfacesTransceiverIdpromBase( Model ):
   interfaces = Dict( keyType=Interface, valueType=InterfacesTransceiverIdpromPages,
                      help='Mapping between interface name and the eeprom' )
   def render( self ):
      if not self.interfaces:
         return
      for intf in Arnet.sortIntf( self.interfaces ):
         self.interfaces[ intf ].renderModel( intf )
         print()

#--------------------------------------------------------------------------------
#
# Models for "show interface [<interfaces>] transceiver eeprom"
#
#--------------------------------------------------------------------------------

def spacedHex( val, n, minWidth=1 ):
   """Converts an integer into space-separated hex form with a space every n numbers
      and left-padded with 0, e.g. (0x123456789, 2) => '0x01 23 45 67 89'"""
   hexed = "{0:0>{1}X}".format( val, minWidth )
   hexed = "0" * ( len( hexed ) % n ) + hexed
   return "0x" + " ".join( hexed[ i : i + n ] for i in range( 0, len( hexed ), n ) )

def collapseRange( ints ):
   """Collapses a list of ints into its shortened string form (e.g. '132-135, 172').
      Assumes input is sorted; will return descending ranges if reverse sorted"""
   ranges = [] 
   for i in ints:
      if ranges and abs( i - ranges[ -1 ][ -1 ] ) <= 1:
         ranges[ -1 ].append( i )
      else:
         ranges.append( [ i ] )
   return ",".join( "%s-%s" % ( r[ 0 ], r[ -1 ] ) if r[ 0 ] != r[ -1 ] else
                    "%s" % r[ 0 ] for r in ranges )

# pylint: disable-msg=attribute-defined-outside-init,dangerous-default-value
def eepromFieldBuilder( fields, name, pages, raw=None, decode=None, valueAttrs={} ):
   """Creates a field with the given name in the current field's subfields.
      Creates it with the given raw value, and a decoded value by passing
      raw to the decode function if it exists."""
   field = InterfacesTransceiverEepromField( _name=name )
   fields[ name ] = field
   for page, byteList in pages:
      field.addBytes( page, *byteList )
   if raw is not None:
      decoded = decode( raw ) if callable( decode ) else None
      if isinstance( decoded, bool ):
         field.value = InterfacesTransceiverEepromBoolValue()
      elif isinstance( decoded, basestring ):
         field.value = InterfacesTransceiverEepromStrValue()
      elif isinstance( decoded, int ):
         field.value = InterfacesTransceiverEepromIntValue()
      elif isinstance( decoded, float ):
         field.value = InterfacesTransceiverEepromFloatValue()
      else:
         # Either there wasn't a decoding function or we didn't understand its
         # output, so don't include the decode value
         field.value = InterfacesTransceiverEepromValue()
      field.value.raw = int( raw )
      if decoded is not None and hasattr( field.value, "decoded" ):
         field.value.decoded = decoded
      for key, val in valueAttrs.iteritems():
         if val is not None and hasattr( field.value, key ):
            setattr( field.value, key, val )
   return field

# TODO: update decode functions to use these constants
class eepromUnits( object ):
   @staticmethod
   def enums():
      return [ v for a, v in vars( eepromUnits ).iteritems()
               if not a.startswith( "_" ) and a != "enums" ]

   # Output values for the different units we support in the output
   megabits = "Mbps"
   megabaud = "MBd"
   decibels = "dB"
   kilometers = "km"
   meters = "m"
   nanometers = "nm"
   celsius = "C"
   watts = "W"

class eepromPrefixes( object ):
   @staticmethod
   def enums():
      return [ v for a, v in vars( eepromPrefixes ).iteritems()
               if not a.startswith( "_" ) and a != "enums" ]

   # Output values for the different prefixes we support in the output
   minus = "-"
   plus = "+"
   plusOrMinus = "+/-"

# Builds the recursive printing and representation based on sub, sortVal, and text
class EepromBaseModel( Model ):
   def renderIndented( self, indentLevel, indentStr ):
      print( indentStr * indentLevel + self.text() )
      if self.sub():
         for item in sorted( self.sub().values(), key=lambda s: s.sortVal() ):
            item.renderIndented( indentLevel + 1, indentStr )

   # Override this with the sub dictionary to iterate over for printing
   def sub( self ):
      return {}

   # Override this with the field to compare when sorting
   def sortVal( self ):
      return self.sub()

   # Override this with the text output for this line (don't include indent)
   def text( self ):
      raise NotImplementedError

class InterfacesTransceiverEepromValue( EepromBaseModel ):
   raw = Int( help="The raw value stored in EEPROM" )
   # This value is needed to consistently print the full width of a raw value as we
   # lose the width information about the int when going from c++ to python
   _numBytes = Int( help="The number of bytes in this raw value.", default=1 )

   def sortVal( self ):
      return int( self.raw )

   def text( self ):
      return spacedHex( self.raw, 2, minWidth=self._numBytes * 2 )

class InterfacesTransceiverEepromBoolValue( InterfacesTransceiverEepromValue ):
   decoded = Bool( help="The decoded value from the EEPROM" )

   def text( self ):
      return str( self.decoded ).lower()

class InterfacesTransceiverEepromStrValue( InterfacesTransceiverEepromValue ):
   decoded = Str( help="The decoded value from the EEPROM" )
   possibleValues = List( help="The possible values that decoded can take on",
                    valueType=str, optional=True )

   def text( self ):
      return self.decoded

class InterfacesTransceiverEepromIntValue( InterfacesTransceiverEepromValue ):
   decoded = Int( help="The decoded value from the EEPROM" )
   units = Enum( help="The units for the decoded value", values=eepromUnits.enums(),
                 optional=True )
   prefix = Enum( help="Prefix such as \"+/-\" for the decoded value",
                  values=eepromPrefixes.enums(), optional=True )

   def text( self ):
      out = self.prefix or ""
      out += str( self.decoded )
      if self.units:
         out += " " + self.units
      return out

class InterfacesTransceiverEepromFloatValue( InterfacesTransceiverEepromValue ):
   decoded = Float( help="The decoded value from the EEPROM" )
   units = Enum( help="The units for the decoded value", values=eepromUnits.enums(),
                 optional=True )
   prefix = Enum( help="Prefix such as \"+/-\" for the decoded value",
                  values=eepromPrefixes.enums(), optional=True )

   def text( self ):
      out = self.prefix or ""
      out += str( self.decoded )
      if self.units:
         out += " " + self.units
      return out

class InterfacesTransceiverEepromBytes( EepromBaseModel ):
   _page = Int( help="The page the byte offsets are for." )
   byteOffsets = List( help="The byte offsets for the field.", valueType=int )

   def sortVal( self ):
      # Have to re-pack the list because CliModel.List doesn't compare properly
      return ( self._page, [ b for b in self.byteOffsets ] )

   def text( self ):
      return "{:0>2X}h:{!s}".format( self._page, collapseRange( self.byteOffsets ) )

# Create a dummy parent class for the recursive fields
# pylint: disable-msg=abstract-method
class InterfacesTransceiverEepromFieldBase( EepromBaseModel ):
   pass

class InterfacesTransceiverEepromField( InterfacesTransceiverEepromFieldBase ):
   _name = Str( help="The name of this field" )
   pages = Dict( help="Mapping between a page and the bytes on the page the field "
                      "is defined over.",
                 keyType=int, valueType=InterfacesTransceiverEepromBytes )
   value = Submodel( help="The value of this field",
                     valueType=InterfacesTransceiverEepromValue, optional=True )
   subfields = Dict( help="Mapping between a subfield name and its data",
                     keyType=str, valueType=InterfacesTransceiverEepromFieldBase,
                     optional=True )
   note = Str( help="A note about this field's interpretation",
               optional=True )

   def sub( self ):
      return self.subfields

   def sortVal( self ):
      # Have to re-pack the list to grab the correct sort value for the pages
      return ( [ pg.sortVal() for pg in self.pages.itervalues() ], self._name )

   def text( self ):
      out = [ self._name ]
      byteStr = ";".join( sorted( pg.text() for pg in self.pages.itervalues() ) )
      out.append( "(" + byteStr + "):" )
      if self.value:
         out.append( self.value.text() )
      if self.note:
         out.append( "(%s)" % self.note )
      return " ".join( out )

   def addBytes( self, page, *byteList ):
      byteListLocal = []
      for byte in byteList:
         if isinstance( byte, list ):
            byteListLocal += byte[:]
         else:
            byteListLocal.append( byte )
      if page not in self.pages:
         self.pages[ page ] = InterfacesTransceiverEepromBytes( _page=page )
      self.pages[ page ].byteOffsets += byteListLocal

   def addField( self, name, byteList, raw=None, decode=None, **valueAttrs ):
      return eepromFieldBuilder( self.subfields, name, byteList, raw=raw,
                                 decode=decode, valueAttrs=valueAttrs )

class InterfacesTransceiverEepromIntf( EepromBaseModel ):
   _name = Str( help="The name of the interface this EEPROM is from" )
   fields = Dict( help="Mapping between a field name and its data",
                  keyType=str, valueType=InterfacesTransceiverEepromField )

   def sub( self ):
      return self.fields

   def text( self ):
      return "%s EEPROM:" % self._name

   def addField( self, name, byteList, raw=None, decode=None, **valueAttrs ):
      return eepromFieldBuilder( self.fields, name, byteList, raw=raw,
                                 decode=decode, valueAttrs=valueAttrs )

class InterfacesTransceiverEepromBase( Model ):
   interfaces = Dict( help="Mapping between interface name and the EEPROM",
                      keyType=Interface, valueType=InterfacesTransceiverEepromIntf )

   def render( self ):
      if self.interfaces:
         for intf in Arnet.sortIntf( self.interfaces ):
            self.interfaces[ intf ].renderIndented( 0, "  " )
            print()

#--------------------------------------------------------------------------------
#
# Models for "show transceiver status [ interfaces <interface> ]"
#
#--------------------------------------------------------------------------------

AttrDesc = namedtuple( 'AttrDesc', 'attr header renderedTrue renderedFalse' )

def fmtGeneric( attr, attrName, fmt="  %-29.29s %-20.20s %7.7s       %s" ):
   if not attr:
      return
   renderableAttrs = attr.getRenderedAttr()
   values = ( attrName,
              renderableAttrs.get( "state", '' ),
              renderableAttrs.get( "changes", '' ),
              renderableAttrs.get( "lastChange", '' ) )
   if any( values ):
      print( fmt % values )

def fmtDescriptive( attrDesc, fmt="  %-29.29s %-20.20s %7.7s       %s" ):
   '''
   Helper function used to print a formatted line in the case when we would
   want to use a more descriptive token for the state, instead of true/false.

   Parameters
   ----------
   attrDesc : AttrDesc object
      Namedtuple object holding the attribute that must be rendered, the line
      header, and the tokens used to replace the boolean state in the output
   '''
   assert isinstance( attrDesc, AttrDesc )
   if not attrDesc or not attrDesc.attr:
      return
   renderableAttrs = attrDesc.attr.getRenderedAttr()
   renderedState = attrDesc.renderedTrue \
                   if "true" in renderableAttrs.get( "state", '' ) \
                   else attrDesc.renderedFalse
   values = ( attrDesc.header,
              renderedState,
              renderableAttrs.get( "changes", '' ),
              renderableAttrs.get( "lastChange", '' ) )
   if any( values ):
      print( fmt % values )

class InterfacesTransceiverStatusValueFormat( Model ):
   __public__ = False
   changes = Int( help="Changes since system boot", optional=True )
   lastChange = Float( help="Last update time in UTC", optional=True )

   def _getRenderedAttr( self ):
      return { "changes" : str( self.changes if self.changes is not None else '' ),
               "lastChange" : ( Ark.timestampToStr( self.lastChange ) if
                                self.lastChange is not None else '' ) }
   def getRenderedAttr( self ):
      return self._getRenderedAttr()

class InterfacesTransceiverStatusValueFormatInt(
      InterfacesTransceiverStatusValueFormat ):
   state = Int( help="Current value" )

   def getRenderedAttr( self ):
      renderDict = self._getRenderedAttr()
      renderDict[ "state" ] = str( self.state )
      return renderDict

class InterfacesTransceiverStatusValueFormatHex(
      InterfacesTransceiverStatusValueFormat ):
   __public__ = False
   state = Int( help="Current value" )

   def getRenderedAttr( self ):
      renderDict = self._getRenderedAttr()
      renderDict[ "state" ] = hex( self.state )
      return renderDict

class InterfacesTransceiverStatusValueFormatStr(
      InterfacesTransceiverStatusValueFormat ):
   __public__ = False
   state = Str( help="Current value" )

   def getRenderedAttr( self ):
      renderDict = self._getRenderedAttr()
      renderDict[ "state" ] = self.state
      return renderDict

class InterfacesTransceiverStatusValueFormatBool(
      InterfacesTransceiverStatusValueFormat ):
   __public__ = False
   state = Bool( help="Current value" )

   def getRenderedAttr( self ):
      renderDict = self._getRenderedAttr()
      renderDict[ "state" ] = str( self.state ).lower()
      return renderDict

class InterfacesTransceiverStatusValueFormatFloat(
      InterfacesTransceiverStatusValueFormat ):
   __public__ = False
   state = Float( help="Current value" )

   def getRenderedAttr( self ):
      renderDict = self._getRenderedAttr()
      renderDict[ "state" ] = str( self.state )
      return renderDict

class InterfacesTransceiverStatusValueFormatScientificFloat(
      InterfacesTransceiverStatusValueFormat ):
   __public__ = False
   state = Float( help="Current value" )
   
   def getRenderedAttr( self ):
      renderDict = self._getRenderedAttr()
      renderDict[ "state" ] = "{:.2e}".format( self.state )
      return renderDict

class InterfacesTransceiverStatuValuesSwizzlerList(
      InterfacesTransceiverStatusValueFormat ):
   __public__ = False
   state = List( valueType=str, help="List of adapters in use" )

   def getRenderedAttr( self ):
      listValues = ", ".join( self.state or [ 'none' ] )
      renderDict = self._getRenderedAttr()
      renderDict[ "state" ] = listValues
      return renderDict

class InterfacesTransceiverStatusValueFormatModuleStateDec(
   InterfacesTransceiverStatusValueFormat ):

   init = Submodel( valueType=InterfacesTransceiverStatusValueFormatStr,
                    help='Module state initialization' )
   lowPwr = Submodel( valueType=InterfacesTransceiverStatusValueFormatStr,
                      help='Module state low power' )
   hiPwrUp = Submodel( valueType=InterfacesTransceiverStatusValueFormatStr,
                       help='Module state high power up' ) 
   txOff = Submodel( valueType=InterfacesTransceiverStatusValueFormatStr,
                     help='Module state TX off' )
   txTurnOn = Submodel( valueType=InterfacesTransceiverStatusValueFormatStr,
                        help='Module state TX turn on' )
   ready = Submodel( valueType=InterfacesTransceiverStatusValueFormatStr,
                     help='Module state ready' )
   fault = Submodel( valueType=InterfacesTransceiverStatusValueFormatStr,
                     help='Module state fault' )
   txTurnOff = Submodel( valueType=InterfacesTransceiverStatusValueFormatStr,
                         help='Module state TX turn off' )
   hiPwrDown = Submodel( valueType=InterfacesTransceiverStatusValueFormatStr,
                         help='Module state high power down' )

   def renderModel( self, fmtState="    %-40.40s %-20.20s %7.7s       %s" ):
      moduleStateAttrs = [
         ( self.init, "Initialization" ),
         ( self.lowPwr, "Low power" ),
         ( self.hiPwrUp, "High power up" ),
         ( self.txOff, "TX off" ),
         ( self.txTurnOn, "TX turn on" ),
         ( self.ready, "Ready" ),
         ( self.fault, "Fault" ),
         ( self.txTurnOff, "TX turn off" ),
         ( self.hiPwrDown, "High power down" ),
      ]
      print( "  Module state" )
      for attr in moduleStateAttrs:
         fmtGeneric( *attr, fmt=fmtState )

class InterfacesTransceiverStatusValueFormatTxTurnUpStateDec(
   InterfacesTransceiverStatusValueFormat ):

   txInit = Submodel( valueType=InterfacesTransceiverStatusValueFormatStr,
                      help='TX turn up state path initialization' )
   txDataPathLock = Submodel(
      valueType=InterfacesTransceiverStatusValueFormatStr,
      help='TX turn up state data path lock' )
   txLasReadyOff = Submodel(
      valueType=InterfacesTransceiverStatusValueFormatStr,
      help='TX turn up state TX laser ready off' )
   txLaserReady = Submodel(
      valueType=InterfacesTransceiverStatusValueFormatStr,
      help='TX turn up state TX laser ready' )
   txModulatorConverge = \
      Submodel( valueType=InterfacesTransceiverStatusValueFormatStr,
                help='TX turn up state TX modulator converge' )
   txOutPwrAdj = Submodel(
      valueType=InterfacesTransceiverStatusValueFormatStr,
      help='TX turn up state TX output power adjustment' )

   def renderModel( self, fmtState="    %-40.40s %-20.20s %7.7s       %s" ):
      txTurnUpStateAttrs = [
         ( self.txInit, "Path initialization" ),
         ( self.txDataPathLock, "Data path lock" ),
         ( self.txLasReadyOff, "Laser ready off" ),
         ( self.txLaserReady, "Laser ready" ),
         ( self.txModulatorConverge, "Modulator converge" ),
         ( self.txOutPwrAdj, "Output power adjustment" ) ]
      print( "  TX turn-up state" )
      for attr in txTurnUpStateAttrs:
         fmtGeneric( *attr, fmt=fmtState )

class InterfacesTransceiverStatusValueFormatRxTurnUpStateDec(
   InterfacesTransceiverStatusValueFormat ):

   rxInit = Submodel( valueType=InterfacesTransceiverStatusValueFormatStr,
                      help='RX turn up state path initialization' )
   rxLoLaserReady = Submodel( valueType=InterfacesTransceiverStatusValueFormatStr,
                              help='RX turn up state RX LO laser ready' )
   rxWaitForInput = Submodel( valueType=InterfacesTransceiverStatusValueFormatStr,
                              help='RX turn up state wait for signal input' )
   adcOutput = Submodel( valueType=InterfacesTransceiverStatusValueFormatStr,
                         help='RX turn up state ADC output' )
   dispersionLock = Submodel( valueType=InterfacesTransceiverStatusValueFormatStr,
                              help='RX turn up state dispersion lock' )
   rxDemodLock = Submodel( valueType=InterfacesTransceiverStatusValueFormatStr,
                           help='RX turn up state demodulator lock' )

   def renderModel( self, fmtState="    %-40.40s %-20.20s %7.7s       %s" ):
      rxTurnUpStateAttrs = [
         ( self.rxInit, "Path initialization" ),
         ( self.rxLoLaserReady, "LO laser ready" ),
         ( self.rxWaitForInput, "Wait for signal input" ),
         ( self.adcOutput, "ADC output" ),
         ( self.dispersionLock, "Dispersion lock" ),
         ( self.rxDemodLock, "Demodulator lock" ),
      ]
      print( "  RX turn-up state" )
      for attr in rxTurnUpStateAttrs:
         fmtGeneric( *attr, fmt=fmtState )

class InterfacesTransceiverStatusValueFormatModuleGeneralStatus(
   InterfacesTransceiverStatusValueFormat ):

   hiPowerOn = Submodel(
      valueType=InterfacesTransceiverStatusValueFormatStr,
      help='Module general status high power on' )
   pmIntervalDone = Submodel(
      valueType=InterfacesTransceiverStatusValueFormatStr,
      help='Module general status PM interval done' )
   outOfAlgn = Submodel(
      valueType=InterfacesTransceiverStatusValueFormatStr,
      help='Module general status out of alignment' )
   rxNetworkLOL = Submodel(
      valueType=InterfacesTransceiverStatusValueFormatStr,
      help='Module general status RX network loss of lock' )
   rxLOS = Submodel(
      valueType=InterfacesTransceiverStatusValueFormatStr,
      help='Module general status RX loss of signal' )
   txHostLOL = Submodel(
      valueType=InterfacesTransceiverStatusValueFormatStr,
      help='Module general status TX host loss of lock' )
   txLOF = Submodel(
      valueType=InterfacesTransceiverStatusValueFormatStr,
      help='Module general status TX loss of signal functionality' )
   hwInterlock = Submodel(
      valueType=InterfacesTransceiverStatusValueFormatStr,
      help='Module general status hw interlock' )

   def renderModel( self, fmtStatus="    %-40.40s %-20.20s %7.7s       %s" ):
      modGeneralStatusAttrs = [
         ( self.hiPowerOn, "High power on" ),
         ( self.outOfAlgn, "Host lane alignment" ),
         ( self.rxNetworkLOL, "RX network LOL" ),
         ( self.rxLOS, "RX LOS" ),
         ( self.txHostLOL, "TX host LOL" ),
         ( self.txLOF, "TX LOSF" ),
         ( self.hwInterlock, "HW interlock" ) ]
      print( "  Module general status" )
      for attr in modGeneralStatusAttrs:
         fmtGeneric( *attr, fmt=fmtStatus )

class InterfacesTransceiverStatusValueFormatModuleFaultStatus(
   InterfacesTransceiverStatusValueFormat ):

   checkSum = Submodel(
      valueType=InterfacesTransceiverStatusValueFormatStr,
      help='Module fault status checksum' )
   powerSupplyFault = Submodel(
      valueType=InterfacesTransceiverStatusValueFormatStr,
      help='Module fault status power supply fault' )
   pldFlashInitFault = Submodel(
      valueType=InterfacesTransceiverStatusValueFormatStr,
      help='Module fault status pld flash init fault' )
   modSpecFault = Submodel(
      valueType=InterfacesTransceiverStatusValueFormatStr,
      help='Module fault status module specific hw fault' )
   modOverTempFault = Submodel(
      valueType=InterfacesTransceiverStatusValueFormatStr,
      help='Module fault status module over temperature fault' )
   powerSupplyFault = Submodel(
      valueType=InterfacesTransceiverStatusValueFormatStr,
      help='Module fault status power supply fault' )

   def renderModel( self, fmtFault="    %-40.40s %-20.20s %7.7s       %s" ):
      modFaultStatusAttrs = [
         ( self.checkSum, "CFP table checksum" ),
         ( self.powerSupplyFault, "Power supply fault" ),
         ( self.pldFlashInitFault, "PLD flash init fault" ),
         ( self.modSpecFault, "Module specific HW fault" ),
         ( self.modOverTempFault, "Module over temp fault" ), ]
      print( "  Module fault status" )
      for attr in modFaultStatusAttrs:
         fmtGeneric( *attr, fmt=fmtFault )

class InterfacesTransceiverStatusValueFormatNetworkLaneVendorFAWS(
   InterfacesTransceiverStatusValueFormat ):

   oaPumpBiasHighAlarm = Submodel(
      valueType=InterfacesTransceiverStatusValueFormatStr,
      help='OA pump bias high alarm' )
   oaPumpBiasHighWarn = Submodel(
      valueType=InterfacesTransceiverStatusValueFormatStr,
      help='OA pump bias high warn' )
   oaPumpBiasLowAlarm = Submodel(
      valueType=InterfacesTransceiverStatusValueFormatStr,
      help='OA pump bias low alarm' )
   oaPumpBiasLowWarn = Submodel(
      valueType=InterfacesTransceiverStatusValueFormatStr,
      help='OA pump bias low warn' )
   rxPhaseCtrlLoopHighAlarm = Submodel(
      valueType=InterfacesTransceiverStatusValueFormatStr,
      help='RX phase control loop high alarm' )
   rxPhaseCtrlLoopHighWarn = Submodel(
      valueType=InterfacesTransceiverStatusValueFormatStr,
      help='RX phase control loop high warn' )
   rxPhaseCtrlLoopLowAlarm = Submodel(
      valueType=InterfacesTransceiverStatusValueFormatStr,
      help='RX phase control loop low alarm' )
   rxPhaseCtrlLoopLowWarn = Submodel(
      valueType=InterfacesTransceiverStatusValueFormatStr,
      help='RX phase control loop low warn' )
   txModBiasVOAHighAlarm = Submodel(
      valueType=InterfacesTransceiverStatusValueFormatStr,
      help='TX mod bias VOA control loop high alarm' )
   txModBiasVOAHighWarn = Submodel(
      valueType=InterfacesTransceiverStatusValueFormatStr,
      help='TX mod bias VOA control loop high warn' )
   rxTunedChPwrHighAlarm = Submodel(
      valueType=InterfacesTransceiverStatusValueFormatStr,
      help='RX tuned channel power high alarm' )
   rxTunedChPwrHighWarn = Submodel(
      valueType=InterfacesTransceiverStatusValueFormatStr,
      help='RX tuned channel power high warn' )
   rxChannelPwrLowAlarm = Submodel(
      valueType=InterfacesTransceiverStatusValueFormatStr,
      help='RX tuned channel power low alarm' )
   rxChannelPwrLowWarn = Submodel(
      valueType=InterfacesTransceiverStatusValueFormatStr,
      help='RX tuned channel power low warn' )
   modBiasConvergeFault = Submodel(
      valueType=InterfacesTransceiverStatusValueFormatStr,
      help='Modulator bias converge fault' )
   adcOutputLow = Submodel(
      valueType=InterfacesTransceiverStatusValueFormatStr,
      help='ADC output low' )

   def renderModel( self, fmtFaws="    %-40.40s %-20.20s %7.7s       %s" ):
      networkLaneVendorFAWSAttrs = [
         ( self.oaPumpBiasHighAlarm, "OA pump bias high alarm" ),
         ( self.oaPumpBiasHighWarn, "OA pump bias high warn" ),
         ( self.oaPumpBiasLowAlarm, "OA pump bias low alarm" ),
         ( self.oaPumpBiasLowWarn, "OA pump bias low warn" ),
         ( self.rxPhaseCtrlLoopHighAlarm, "RX phase control loop high alarm" ),
         ( self.rxPhaseCtrlLoopHighWarn, "RX phase control loop high warn" ),
         ( self.rxPhaseCtrlLoopLowAlarm, "RX phase control loop low alarm" ),
         ( self.rxPhaseCtrlLoopLowWarn, "RX phase control loop low warn" ),
         ( self.txModBiasVOAHighAlarm,
           "TX mod bias VOA ctrl loop high alarm" ),
         ( self.txModBiasVOAHighWarn,
           "TX mod bias VOA ctrl loop high warn" ),
         ( self.rxTunedChPwrHighAlarm,
           "RX tuned channel power high alarm" ),
         ( self.rxTunedChPwrHighWarn,
           "RX tuned channel power high warn" ),
         ( self.rxChannelPwrLowAlarm,
           "RX tuned channel power low alarm" ),
         ( self.rxChannelPwrLowWarn,
           "RX tuned channel power low warn" ),
         ( self.modBiasConvergeFault, "Modulator bias converge fault" ),
         ( self.adcOutputLow, "ADC output low" ),
      ]
      print( "  Network lane vendor specific faults/status" )
      for attr in networkLaneVendorFAWSAttrs:
         fmtGeneric( *attr, fmt=fmtFaws )

class InterfacesTransceiverStatusValueFormatNetworkLaneFaultStatus(
   InterfacesTransceiverStatusValueFormat ):

   rxTECFault = Submodel(
      valueType=InterfacesTransceiverStatusValueFormatStr,
      help='RX TEC fault' )
   rxFIFOFault = Submodel(
      valueType=InterfacesTransceiverStatusValueFormatStr,
      help='RX FIFO fault' )
   rxLOL = Submodel(
      valueType=InterfacesTransceiverStatusValueFormatStr,
      help='RX LOL' )
   rxLOS = Submodel(
      valueType=InterfacesTransceiverStatusValueFormatStr,
      help='RX LOS' )
   txLOL = Submodel(
      valueType=InterfacesTransceiverStatusValueFormatStr,
      help='TX LOL' )
   txLOSF = Submodel(
      valueType=InterfacesTransceiverStatusValueFormatStr,
      help='TX LOSF' )
   wavelengthUnlockFault = Submodel(
      valueType=InterfacesTransceiverStatusValueFormatStr,
      help='Wavelength unlock fault' )
   tecFault = Submodel(
      valueType=InterfacesTransceiverStatusValueFormatStr,
      help='TEC fault' )

   def renderModel( self, fmtAttrs="    %-40.40s %-20.20s %7.7s       %s" ):
      networkLaneFaultStatusAttrs = [
         ( self.rxTECFault, "RX TEC fault" ),
         ( self.rxFIFOFault, "RX FIFO fault" ),
         ( self.rxLOL, "RX LOL" ),
         ( self.rxLOS, "RX LOS" ),
         ( self.txLOL, "TX LOL" ),
         ( self.txLOSF, "TX LOSF" ),
         ( self.wavelengthUnlockFault, "Wavelength unlock fault" ),
         ( self.tecFault, "TEC fault" ),
      ]
      print( "  Network lane fault/status" )
      for attr in networkLaneFaultStatusAttrs:
         fmtGeneric( *attr, fmt=fmtAttrs )

class InterfacesTransceiverStatusValueFormatLaneAlignment(
   InterfacesTransceiverStatusValueFormat ):

   netLaneTxAlignment = Submodel(
       valueType=InterfacesTransceiverStatusValueFormatStr,
       help='Network lane TX alignment' )
   modemLockFault = Submodel(
       valueType=InterfacesTransceiverStatusValueFormatStr,
       help='RX alignment modem lock fault' )
   modemSyncDetectFault = Submodel(
       valueType=InterfacesTransceiverStatusValueFormatStr,
       help='RX alignment modem sync detect fault' )

   def renderModel( self, fmtTx="  %-42.42s %-20.20s %7.7s       %s",
                    fmtRx="    %-40.40s %-20.20s %7.7s       %s" ):
      fmtGeneric( self.netLaneTxAlignment, "Network TX alignment",
                  fmt=fmtTx )
      print( "  Network RX alignment" )
      for attr in [ ( self.modemLockFault, "Modem lock fault" ),
                    ( self.modemSyncDetectFault,
                      "Modem sync detect fault" ), ]:
         fmtGeneric( *attr, fmt=fmtRx )

class InterfacesTransceiverStatusValueFormatNetworkLaneRxOTNStatus(
   InterfacesTransceiverStatusValueFormat ):

   fastPMRxSignalDegradeAlarm = Submodel(
       valueType=InterfacesTransceiverStatusValueFormatStr,
       help='Fast PM RX signal degrade alarm' )
   fastPMRxSignalFailAlarm = Submodel(
       valueType=InterfacesTransceiverStatusValueFormatStr,
       help='Fast PM RX signal fail alarm' )
   fastPMTxSignalDegradeAlarm = Submodel(
       valueType=InterfacesTransceiverStatusValueFormatStr,
       help='Fast PM TX signal degrade alarm' )
   fastPMTxSignalFailAlarm = Submodel(
       valueType=InterfacesTransceiverStatusValueFormatStr,
       help='Fast PM TX signal fail alarm' )
   ingressFDDAlarm = Submodel(
       valueType=InterfacesTransceiverStatusValueFormatStr,
       help='Ingress FDD alarm' )
   ingressFEDAlarm = Submodel(
       valueType=InterfacesTransceiverStatusValueFormatStr,
       help='Ingress FED alarm' )
   oduLCKMaintSignal = Submodel(
       valueType=InterfacesTransceiverStatusValueFormatStr,
       help='ODU LCK maintenance signal' )
   oduAISMaintSignal = Submodel(
       valueType=InterfacesTransceiverStatusValueFormatStr,
       help='ODU AIS maintenance signal' )
   sectionMonitoringBackDefect = Submodel(
       valueType=InterfacesTransceiverStatusValueFormatStr,
       help='Section monitoring back defect' )
   outOfMultiframe = Submodel(
       valueType=InterfacesTransceiverStatusValueFormatStr,
       help='Out of multiframe' )
   lossOfMultiframe = Submodel(
       valueType=InterfacesTransceiverStatusValueFormatStr,
       help='Loss of multiframe' )
   outOfFrame = Submodel(
       valueType=InterfacesTransceiverStatusValueFormatStr,
       help='Out of frame' )
   lossOfFrame = Submodel(
       valueType=InterfacesTransceiverStatusValueFormatStr,
       help='Loss of frame' )

   def renderModel( self, fmtRxOTN="    %-40.40s %-20.20s %7.7s       %s" ):
      networkRxOTNStatusAttrs = [
         ( self.fastPMRxSignalDegradeAlarm, "Fast PM RX signal degrade alarm" ),
         ( self.fastPMRxSignalFailAlarm, "Fast PM RX signal fail alarm" ),
         ( self.fastPMTxSignalDegradeAlarm, "Fast PM TX signal degrade alarm" ),
         ( self.fastPMTxSignalFailAlarm, "Fast PM TX signal fail alarm" ),
         ( self.ingressFDDAlarm, "Ingress FDD alarm" ),
         ( self.ingressFEDAlarm, "Ingress FED alarm" ),
         ( self.oduLCKMaintSignal, "ODU LCK maintenance signal" ),
         ( self.oduAISMaintSignal, "ODU AIS maintenance signal" ),
         ( self.sectionMonitoringBackDefect, "Section monitoring back defect" ),
         ( self.outOfMultiframe, "Out of multiframe" ),
         ( self.lossOfMultiframe, "Loss of multiframe" ),
         ( self.outOfFrame, "Out of frame" ),
         ( self.lossOfFrame, "Loss of frame" ),
      ]
      print( "  Network lane RX OTN status" )
      for attr in networkRxOTNStatusAttrs:
         fmtGeneric( *attr, fmt=fmtRxOTN )

class InterfacesTransceiverStatusValueFormatTxModulatorBiasVOA(
   InterfacesTransceiverStatusValueFormat ):

   loop1Alarm = Submodel(
       valueType=InterfacesTransceiverStatusValueFormatStr,
       help='Net mod bias VOA loop 1 alarm' )
   loop1Warn = Submodel(
       valueType=InterfacesTransceiverStatusValueFormatStr,
       help='Net mod bias VOA loop 1 warn' )
   loop2Alarm = Submodel(
       valueType=InterfacesTransceiverStatusValueFormatStr,
       help='Net mod bias VOA loop 2 alarm' )
   loop2Warn = Submodel(
       valueType=InterfacesTransceiverStatusValueFormatStr,
       help='Net mod bias VOA loop 2 warn' )
   loop3Alarm = Submodel(
       valueType=InterfacesTransceiverStatusValueFormatStr,
       help='Net mod bias VOA loop 3 alarm' )
   loop3Warn = Submodel(
       valueType=InterfacesTransceiverStatusValueFormatStr,
       help='Net mod bias VOA loop 3 warn' )
   loop4Alarm = Submodel(
       valueType=InterfacesTransceiverStatusValueFormatStr,
       help='Net mod bias VOA loop 4 alarm' )
   loop4Warn = Submodel(
       valueType=InterfacesTransceiverStatusValueFormatStr,
       help='Net mod bias VOA loop 4 warn' )
   loop5Alarm = Submodel(
       valueType=InterfacesTransceiverStatusValueFormatStr,
       help='Net mod bias VOA loop 5 alarm' )
   loop5Warn = Submodel(
       valueType=InterfacesTransceiverStatusValueFormatStr,
       help='Net mod bias VOA loop 5 warn' )
   loop6Alarm = Submodel(
       valueType=InterfacesTransceiverStatusValueFormatStr,
       help='Net mod bias VOA loop 6 alarm' )
   loop6Warn = Submodel(
       valueType=InterfacesTransceiverStatusValueFormatStr,
       help='Net mod bias VOA loop 6 warn' )
   loop7Alarm = Submodel(
       valueType=InterfacesTransceiverStatusValueFormatStr,
       help='Net mod bias VOA loop 7 alarm' )
   loop7Warn = Submodel(
       valueType=InterfacesTransceiverStatusValueFormatStr,
       help='Net mod bias VOA loop 7 warn' )
   loop8Alarm = Submodel(
       valueType=InterfacesTransceiverStatusValueFormatStr,
       help='Net mod bias VOA loop 8 alarm' )
   loop8Warn = Submodel(
       valueType=InterfacesTransceiverStatusValueFormatStr,
       help='Net mod bias VOA loop 8 warn' )

   def renderModel( self, fmtAttrs="    %-40.40s %-20.20s %7.7s       %s" ):
      networkModBiasVOAAttrs = [
         ( self.loop1Alarm, "Loop 1 alarm" ),
         ( self.loop1Warn, "Loop 1 warn" ),
         ( self.loop2Alarm, "Loop 2 alarm" ),
         ( self.loop2Warn, "Loop 2 warn" ),
         ( self.loop3Alarm, "Loop 3 alarm" ),
         ( self.loop3Warn, "Loop 3 warn" ),
         ( self.loop4Alarm, "Loop 4 alarm" ),
         ( self.loop4Warn, "Loop 4 warn" ),
         ( self.loop5Alarm, "Loop 5 alarm" ),
         ( self.loop5Warn, "Loop 5 warn" ),
         ( self.loop6Alarm, "Loop 6 alarm" ),
         ( self.loop6Warn, "Loop 6 warn" ),
         ( self.loop7Alarm, "Loop 7 alarm" ),
         ( self.loop7Warn, "Loop 7 warn" ),
         ( self.loop8Alarm, "Loop 8 alarm" ),
         ( self.loop8Warn, "Loop 8 warn" ),
      ]
      print( "  Network TX modulator bias VOA AWS" )
      for attr in networkModBiasVOAAttrs:
         fmtGeneric( *attr, fmt=fmtAttrs )

class InterfacesTransceiverStatusValueFormatVendorSpecificFaultStatus(
   InterfacesTransceiverStatusValueFormat ):

   bootEepromCRCFault = Submodel(
       valueType=InterfacesTransceiverStatusValueFormatStr,
       help='Boot EEPROM CRC fault' )
   devInitFault = Submodel(
       valueType=InterfacesTransceiverStatusValueFormatStr,
       help='Device initialization fault' )
   ctrlProcImgABBootFault = Submodel(
       valueType=InterfacesTransceiverStatusValueFormatStr,
       help='Controller proc image AB boot fault' )
   fwHwMismatchFault = Submodel(
       valueType=InterfacesTransceiverStatusValueFormatStr,
       help='FW HW mismatch fault' )
   powerSupplyRWFault = Submodel(
       valueType=InterfacesTransceiverStatusValueFormatStr,
       help='Power supply RW fault' )
   cryptoDevRWFault = Submodel(
       valueType=InterfacesTransceiverStatusValueFormatStr,
       help='Crypto device RW fault' )
   cpldDevRWFault = Submodel(
       valueType=InterfacesTransceiverStatusValueFormatStr,
       help='CPLD device RW fault' )
   viMonitorDevRWFault = Submodel(
       valueType=InterfacesTransceiverStatusValueFormatStr,
       help='VI monitor device RW fault' )
   dacRefClockRWFault = Submodel(
       valueType=InterfacesTransceiverStatusValueFormatStr,
       help='DAC reference clock RW fault' )
   ctrlProcIntCommFault = Submodel(
       valueType=InterfacesTransceiverStatusValueFormatStr,
       help='Controller proc internal comm fault' )
   modemProcIntCommFault = Submodel(
       valueType=InterfacesTransceiverStatusValueFormatStr,
       help='Modem proc internal comm fault' )
   opticsProcIntCommFault = Submodel(
       valueType=InterfacesTransceiverStatusValueFormatStr,
       help='Optics proc internal comm fault' )
   txITLAFault = Submodel(
       valueType=InterfacesTransceiverStatusValueFormatStr,
       help='TX ITLA fault' )
   rxITLAFault = Submodel(
       valueType=InterfacesTransceiverStatusValueFormatStr,
       help='RX ITLA fault' )
   adcCalibrationFault = Submodel(
       valueType=InterfacesTransceiverStatusValueFormatStr,
       help='ADC calibration fault' )
   dacCalibrationFault = Submodel(
       valueType=InterfacesTransceiverStatusValueFormatStr,
       help='DAC calibration fault' )
   picEepromRWFault = Submodel(
       valueType=InterfacesTransceiverStatusValueFormatStr,
       help='PIC EEPROM RW fault' )
   picEepromCRCfault = Submodel(
       valueType=InterfacesTransceiverStatusValueFormatStr,
       help='PIC EEPROM CRC fault' )
   apexRWFault = Submodel(
       valueType=InterfacesTransceiverStatusValueFormatStr,
       help='APEX RW fault' )
   picXTiaRWFault = Submodel(
       valueType=InterfacesTransceiverStatusValueFormatStr,
       help='PIC X TIA RW fault' )
   picYTiaRWFault = Submodel(
       valueType=InterfacesTransceiverStatusValueFormatStr,
       help='PIC Y TIA RW fault' )
   picDriverRWFault = Submodel(
       valueType=InterfacesTransceiverStatusValueFormatStr,
       help='PIC driver RW fault' )

   def renderModel( self, fmtAttrs="    %-40.40s %-20.20s %7.7s       %s" ):
      vendorSpecificFaultStatusAttrs = [
         ( self.bootEepromCRCFault, "Boot EEPROM CRC fault" ),
         ( self.devInitFault, "Device initialization fault" ),
         ( self.ctrlProcImgABBootFault, "Controller proc img AB boot fault" ),
         ( self.fwHwMismatchFault, "FW HW mismatch fault" ),
         ( self.powerSupplyRWFault, "Power supply RW fault" ),
         ( self.cryptoDevRWFault, "Crypto device RW fault" ),
         ( self.cpldDevRWFault, "CPLD device RW fault" ),
         ( self.viMonitorDevRWFault, "VI monitor device RW fault" ),
         ( self.dacRefClockRWFault, "DAC ref clock RW fault" ),
         ( self.ctrlProcIntCommFault, "Controller proc internal comm fault" ),
         ( self.modemProcIntCommFault, "Modem proc internal comm fault" ),
         ( self.opticsProcIntCommFault, "Optics proc internal comm fault" ),
         ( self.txITLAFault, "TX ITLA fault" ),
         ( self.rxITLAFault, "RX ITLA fault" ),
         ( self.adcCalibrationFault, "ADC calibration fault" ),
         ( self.dacCalibrationFault, "DAC calibration fault" ),
         ( self.picEepromRWFault, "PIC EEPROM RW fault" ),
         ( self.picEepromCRCfault, "PIC EEPROM CRC fault" ),
         ( self.apexRWFault, "APEX RW fault" ),
         ( self.picXTiaRWFault, "PIC X TIA RW fault" ),
         ( self.picYTiaRWFault, "PIC Y TIA RW fault" ),
         ( self.picDriverRWFault, "PIC driver RW fault" ),
      ]
      print( "  Vendor specific fault status" )
      for attr in vendorSpecificFaultStatusAttrs:
         fmtGeneric( *attr, fmt=fmtAttrs )

class InterfacesTransceiverStatusExtCfp2Dco( Model ):
   __public__ = False
   _dualLaserModulePresent = Bool( help="The dual laser DP04 DCO module requires "
                                        "custom CLI status formatting when "
                                        "inserted" )

   def dualLaserModulePresentIs( self, present=False ):
      self._dualLaserModulePresent = present

   def dualLaserModulePresent( self ):
      return self._dualLaserModulePresent

   controllerState = Submodel( valueType=InterfacesTransceiverStatusValueFormatStr,
                               help="Controller state" )
   moduleState = Submodel( valueType=InterfacesTransceiverStatusValueFormatHex,
                           help="Module state" )
   txTurnUpState = Submodel( valueType=InterfacesTransceiverStatusValueFormatHex,
                             help="TX turn-up state" )
   rxTurnUpState = Submodel( valueType=InterfacesTransceiverStatusValueFormatHex,
                             help="RX turn-up state" )
   moduleGeneral = Submodel( valueType=InterfacesTransceiverStatusValueFormatHex,
                             help="Module general status" )
   moduleFaults = Submodel( valueType=InterfacesTransceiverStatusValueFormatHex,
                            help="Module fault status" )
   vendorFaults = Submodel( valueType=InterfacesTransceiverStatusValueFormatHex,
                            help="Vendor specific faults/status" )
   networkVendorFaults = Submodel(
         valueType=InterfacesTransceiverStatusValueFormatHex,
         help="Network vendor-specific faults/status" )
   laserFaultCode = Submodel( valueType=InterfacesTransceiverStatusValueFormatHex,
                              help="Laser fault code" )
   sopFastTracking = Submodel( valueType=InterfacesTransceiverStatusValueFormatStr,
                               help="State of Polarization (SOP) fast tracking" )

   # The following attributes currently apply only to the DP04 module
   moduleStateDec = Submodel(
      valueType=InterfacesTransceiverStatusValueFormatModuleStateDec,
      help="Module state deciphered for cfp2Dco Dp04 module" )
   txTurnUpStateDec = Submodel(
      valueType=InterfacesTransceiverStatusValueFormatTxTurnUpStateDec,
      help="TX turn up state deciphered for cfp2Dco Dp04 module" )
   rxTurnUpStateDec = Submodel(
      valueType=InterfacesTransceiverStatusValueFormatRxTurnUpStateDec,
      help="RX turn up state deciphered for cfp2Dco Dp04 module" )
   moduleGeneralStatus = Submodel(
      valueType=InterfacesTransceiverStatusValueFormatModuleGeneralStatus,
      help="Module general status for cfp2Dco Dp04 module" )
   moduleFaultStatus = Submodel(
      valueType=InterfacesTransceiverStatusValueFormatModuleFaultStatus,
      help="Module fault status for cfp2Dco Dp04 module" )
   txModBiasHighAlarm = Submodel(
      valueType=InterfacesTransceiverStatusValueFormatStr,
      help='TX modulator bias high alarm' )
   txModBiasHighWarn = Submodel(
      valueType=InterfacesTransceiverStatusValueFormatStr,
      help='TX modulator bias high warn' )
   txModBiasLowAlarm = Submodel(
      valueType=InterfacesTransceiverStatusValueFormatStr,
      help='TX modulator bias low alarm' )
   txModBiasLowWarn = Submodel(
      valueType=InterfacesTransceiverStatusValueFormatStr,
      help='TX modulator bias low warn' )
   txLaserPowerHighAlarm = Submodel(
      valueType=InterfacesTransceiverStatusValueFormatStr,
      help='TX laser power high alarm' )
   txLaserPowerHighWarn = Submodel(
      valueType=InterfacesTransceiverStatusValueFormatStr,
      help='TX laser power high warn' )
   txLaserPowerLowAlarm = Submodel(
      valueType=InterfacesTransceiverStatusValueFormatStr,
      help='TX laser power low alarm' )
   txLaserPowerLowWarn = Submodel(
      valueType=InterfacesTransceiverStatusValueFormatStr,
      help='TX laser power low warn' )
   rxLaserPowerHighAlarm = Submodel(
      valueType=InterfacesTransceiverStatusValueFormatStr,
      help='RX laser power high alarm' )
   rxLaserPowerHighWarn = Submodel(
      valueType=InterfacesTransceiverStatusValueFormatStr,
      help='RX laser power high warn' )
   rxLaserPowerLowAlarm = Submodel(
      valueType=InterfacesTransceiverStatusValueFormatStr,
      help='RX laser power low alarm' )
   rxLaserPowerLowWarn = Submodel(
      valueType=InterfacesTransceiverStatusValueFormatStr,
      help='RX laser power low warn' )
   txLaserTempHighAlarm = Submodel(
      valueType=InterfacesTransceiverStatusValueFormatStr,
      help='TX laser temperature high alarm' )
   txLaserTempHighWarn = Submodel(
      valueType=InterfacesTransceiverStatusValueFormatStr,
      help='TX laser temperature high warn' )
   txLaserTempLowAlarm = Submodel(
      valueType=InterfacesTransceiverStatusValueFormatStr,
      help='TX laser temperature low alarm' )
   txLaserTempLowWarn = Submodel(
      valueType=InterfacesTransceiverStatusValueFormatStr,
      help='TX laser temperature low warn' )
   rxLaserTempHighAlarm = Submodel(
      valueType=InterfacesTransceiverStatusValueFormatStr,
      help='RX laser temperature high alarm' )
   rxLaserTempHighWarn = Submodel(
      valueType=InterfacesTransceiverStatusValueFormatStr,
      help='RX laser temperature high warn' )
   rxLaserTempLowAlarm = Submodel(
      valueType=InterfacesTransceiverStatusValueFormatStr,
      help='RX laser temperature low alarm' )
   rxLaserTempLowWarn = Submodel(
      valueType=InterfacesTransceiverStatusValueFormatStr,
      help='RX laser temperature low warn' )
   rxPowerHighAlarm = Submodel(
      valueType=InterfacesTransceiverStatusValueFormatStr,
      help='RX total power high alarm' )
   rxPowerHighWarn = Submodel(
      valueType=InterfacesTransceiverStatusValueFormatStr,
      help='RX total power high warn' )
   rxPowerLowAlarm = Submodel(
      valueType=InterfacesTransceiverStatusValueFormatStr,
      help='RX total power low alarm' )
   rxPowerLowWarn = Submodel(
      valueType=InterfacesTransceiverStatusValueFormatStr,
      help='RX total power low warn' )
   netLaneVendorSpecificFAWS = Submodel(
      valueType=InterfacesTransceiverStatusValueFormatNetworkLaneVendorFAWS,
      help='Network lane vendor specific fault/status' )
   netLaneFaultStatus = Submodel(
      valueType=InterfacesTransceiverStatusValueFormatNetworkLaneFaultStatus,
      help='Network lane fault/status' )
   laneAlignment = Submodel(
      valueType=InterfacesTransceiverStatusValueFormatLaneAlignment,
      help='TX/RX alignment' )
   networkLaneRxOTNStatus = Submodel(
      valueType=InterfacesTransceiverStatusValueFormatNetworkLaneRxOTNStatus,
      help='Network lane RX OTN status' )
   networkTxModulatorBiasVoaAWS = Submodel(
      valueType=InterfacesTransceiverStatusValueFormatTxModulatorBiasVOA,
      help='Network TX modulator bias VOA AWS' )
   adcFaultCode = Submodel(
       valueType=InterfacesTransceiverStatusValueFormatHex,
       help='ADC fault code' )
   dacFaultCode = Submodel(
       valueType=InterfacesTransceiverStatusValueFormatHex,
       help='DAC fault code' )
   modulatorConvergenceIter = Submodel(
       valueType=InterfacesTransceiverStatusValueFormatInt,
       help='Modulator convergence iteration count' )
   modulatorConvergenceStep = Submodel(
       valueType=InterfacesTransceiverStatusValueFormatInt,
       help='Modulator convergence step' )
   modulatorBiasFaultCode = Submodel(
       valueType=InterfacesTransceiverStatusValueFormatHex,
       help='Modulator bias fault code' )
   vendorSpecificFaultStatus = Submodel(
      valueType=InterfacesTransceiverStatusValueFormatVendorSpecificFaultStatus,
      help='Vendor specific fault status' )
   rxLaserFaultCode = Submodel(
       valueType=InterfacesTransceiverStatusValueFormatHex,
       help='RX laser fault code' )

   def renderExt( self ):
      attrs = [ ( self.controllerState, "Controller state" ), ]
      if not self._dualLaserModulePresent:
         attrs.extend( [ ( self.moduleState, "Module state" ),
                         ( self.txTurnUpState, "TX turn-up state" ),
                         ( self.rxTurnUpState, "RX turn-up state" ),
                         ( self.moduleGeneral, "Module general status" ),
                         ( self.moduleFaults, "Module fault status" ),
                         ( self.vendorFaults, "Vendor specific faults/status" ),
                         ( self.networkVendorFaults, "Network vendor-specific "
                                                     "faults/status" ),
                         ( self.laserFaultCode, "Laser fault code" ),
                         ( self.sopFastTracking, "SOP fast tracking" ) ] )
         for attr in attrs:
            fmtGeneric( *attr )
      else:
         fmt = "  %-42.42s %-20.20s %7.7s       %s"
         for attr in attrs:
            fmtGeneric( *attr, fmt=fmt )
         self.moduleStateDec.renderModel()
         self.txTurnUpStateDec.renderModel()
         self.rxTurnUpStateDec.renderModel()
         self.moduleGeneralStatus.renderModel()
         self.moduleFaultStatus.renderModel()
         alarmStatusAttrs = [
            ( self.txModBiasHighAlarm, "TX modulator bias high alarm" ),
            ( self.txModBiasHighWarn, "TX modulator bias high warn" ),
            ( self.txModBiasLowAlarm, "TX modulator bias low alarm" ),
            ( self.txModBiasLowWarn, "TX modulator bias low warn" ),
            ( self.txLaserPowerHighAlarm, "TX laser power high alarm" ),
            ( self.txLaserPowerHighWarn, "TX laser power high warn" ),
            ( self.txLaserPowerLowAlarm, "TX laser power low alarm" ),
            ( self.txLaserPowerLowWarn, "TX laser power low warn" ),
            ( self.rxLaserPowerHighAlarm, "RX laser power high alarm" ),
            ( self.rxLaserPowerHighWarn, "RX laser power high warn" ),
            ( self.rxLaserPowerLowAlarm, "RX laser power low alarm" ),
            ( self.rxLaserPowerLowWarn, "RX laser power low warn" ),
            ( self.txLaserTempHighAlarm, "TX laser temperature high alarm" ),
            ( self.txLaserTempHighWarn, "TX laser temperature high warn" ),
            ( self.txLaserTempLowAlarm, "TX laser temperature low alarm" ),
            ( self.txLaserTempLowWarn, "TX laser temperature low warn" ),
            ( self.rxLaserTempHighAlarm, "RX laser temperature high alarm" ),
            ( self.rxLaserTempHighWarn, "RX laser temperature high warn" ),
            ( self.rxLaserTempLowAlarm, "RX laser temperature low alarm" ),
            ( self.rxLaserTempLowWarn, "RX laser temperature low warn" ),
            ( self.rxPowerHighAlarm, "RX total power high alarm" ),
            ( self.rxPowerHighWarn, "RX total power high warn" ),
            ( self.rxPowerLowAlarm, "RX total power low alarm" ),
            ( self.rxPowerLowWarn, "RX total power low warn" ),
         ]
         for attr in alarmStatusAttrs:
            fmtGeneric( *attr, fmt=fmt )
         self.netLaneVendorSpecificFAWS.renderModel()
         self.netLaneFaultStatus.renderModel()
         self.laneAlignment.renderModel()
         self.networkLaneRxOTNStatus.renderModel()
         self.networkTxModulatorBiasVoaAWS.renderModel()
         for attr in [
            ( self.adcFaultCode, "ADC fault code" ),
            ( self.dacFaultCode, "DAC fault code" ),
            ( self.modulatorConvergenceIter, "Modulator convergence iteration" ),
            ( self.modulatorConvergenceStep, "Modulator convergence step" ),
            ( self.modulatorBiasFaultCode, "Modulator bias fault code" ) ]:
            fmtGeneric( *attr, fmt=fmt )
         self.vendorSpecificFaultStatus.renderModel()
         fmtGeneric( self.laserFaultCode, "TX laser fault code", fmt=fmt )
         fmtGeneric( self.rxLaserFaultCode, "RX laser fault code", fmt=fmt )
         fmtGeneric( self.sopFastTracking, "SOP fast tracking", fmt=fmt )

class InterfacesTransceiverStatusIntfValueFormatIngressAlarmStatus(
   InterfacesTransceiverStatusValueFormat ):
   __public__ = False

   rxLocalFault = Submodel( valueType=InterfacesTransceiverStatusValueFormatBool,
                            help='RX local fault' )
   rxRemoteFault = Submodel( valueType=InterfacesTransceiverStatusValueFormatBool,
                             help='RX remote fault' )
   blockLock = Submodel( valueType=InterfacesTransceiverStatusValueFormatStr,
                            help='PCS block lock' )
   alignmentMarkerLock = Submodel(
      valueType=InterfacesTransceiverStatusValueFormatStr,
      help='PCS alignment marker lock' )
   bipErrorsDetected = Submodel(
      valueType=InterfacesTransceiverStatusValueFormatStr,
      help='PCS BIP errors detected' )
   erroredBlocksDetected = Submodel(
      valueType=InterfacesTransceiverStatusValueFormatStr,
      help='PCS errored blocks detected' )
   loa = Submodel( valueType=InterfacesTransceiverStatusValueFormatStr,
                      help='PCS loss of alignment' )
   los = Submodel( valueType=InterfacesTransceiverStatusValueFormatStr,
                      help='PCS loss of signal' )

   def renderModel( self, fmt="    %-40.40s %-20.20s %7.7s       %s" ):
      print( "  Client ingress alarm status" )
      ingressAlarmStatusAttrs = [
         ( self.rxLocalFault, 'RX local fault' ),
         ( self.rxRemoteFault, 'RX remote fault' ),
         ( self.blockLock, 'PCS block lock' ),
         ( self.alignmentMarkerLock, 'PCS alignment marker lock' ),
         ( self.bipErrorsDetected, 'PCS BIP errors detected' ),
         ( self.erroredBlocksDetected, 'PCS errored blocks detected' ),
         ( self.loa, 'PCS LOA' ),
         ( self.los, 'PCS LOS' ) ]
      for attr in ingressAlarmStatusAttrs:
         fmtGeneric( *attr, fmt=fmt )

class InterfacesTransceiverStatusIntfValueFormatEgressAlarmStatus(
   InterfacesTransceiverStatusIntfValueFormatIngressAlarmStatus ):

   highBer = Submodel( valueType=InterfacesTransceiverStatusValueFormatStr,
                       help='PCS high BER' )

   def renderModel( self, fmt="    %-40.40s %-20.20s %7.7s       %s" ):
      print( "  Client egress alarm status" )
      egressAlarmStatusAttrs = [
         ( self.rxLocalFault, 'RX local fault' ),
         ( self.rxRemoteFault, 'RX remote fault' ),
         ( self.blockLock, 'PCS block lock' ),
         ( self.highBer, 'PCS high BER' ),
         ( self.alignmentMarkerLock, 'PCS alignment marker lock' ),
         ( self.bipErrorsDetected, 'PCS BIP errors detected' ),
         ( self.erroredBlocksDetected, 'PCS errored blocks detected' ),
         ( self.loa, 'PCS LOA' ) ]
      for attr in egressAlarmStatusAttrs:
         fmtGeneric( *attr, fmt=fmt )

class InterfacesTransceiverStatusExtCfp2DcoIntf( Model ):
   __public__ = False
   pcsAlarmStatus = Submodel( valueType=InterfacesTransceiverStatusValueFormatHex,
                              help="Client physical coding sublayer alarm status" )
   _dualLaserModulePresent = Bool( help="The dual laser DP04 DCO module requires "
                                        "custom CLI status formatting when "
                                        "inserted" )
   # The following only apply to DP04 dual laser modules
   laneCount = Submodel( valueType=InterfacesTransceiverStatusValueFormatInt,
                         help='Client speed PMD lane count' )
   speed = Submodel( valueType=InterfacesTransceiverStatusValueFormatStr,
                     help='Client operational speed' )
   fec = Submodel( valueType=InterfacesTransceiverStatusValueFormatStr,
                   help='Client forward error correction type' )
   rsFecCodewordSize = Submodel( valueType=InterfacesTransceiverStatusValueFormatInt,
                                 help='Client Reed-Solomon FEC codeword size' )
   fecUncorrectedBlocks = Submodel(
      valueType=InterfacesTransceiverStatusValueFormatInt,
      help='Client FEC uncorrected blocks' )
   preFecBer = Submodel( valueType=InterfacesTransceiverStatusValueFormatStr,
                         help='Client pre-FEC bit error rate' )
   ingressAlarmStatus = Submodel(
      valueType=InterfacesTransceiverStatusIntfValueFormatIngressAlarmStatus,
      help='Client ingress alarms/status' )
   egressAlarmStatus = Submodel(
      valueType=InterfacesTransceiverStatusIntfValueFormatEgressAlarmStatus,
      help='Client egress alarms/status' )

   def dualLaserModulePresentIs( self, present=False ):
      self._dualLaserModulePresent = present

   def dualLaserModulePresent( self ):
      return self._dualLaserModulePresent

   def renderExt( self, intfFmt="  %-42.42s %-20.20s %7.7s       %s" ):
      if self.dualLaserModulePresent():
         cfp2DcoIntfAttrs = [
            ( self.laneCount, 'Lane count' ),
            ( self.fec, 'Forward error correction' ),
            ( self.rsFecCodewordSize, 'Reed-Solomon codeword size' ),
            ( self.fecUncorrectedBlocks, 'FEC uncorrected blocks' ),
            ( self.preFecBer, 'Pre-FEC bit error rate' ) ]
         for attr in cfp2DcoIntfAttrs:
            fmtGeneric( *attr, fmt=intfFmt )
         self.ingressAlarmStatus.renderModel()
         self.egressAlarmStatus.renderModel()
      else:
         fmtGeneric( self.pcsAlarmStatus, "Client PCS alarm status" )

def renderChAttr( channelsDict, printChLabel=False ):
   if len( channelsDict ) > 1:
      # Follows the format of:
      #   Attribute name
      #     Channel 1
      #     Channel 2
      #     Channel ...
      #     Channel n
      attrs = ordDict( [ ( "RX LOS", "rxLos" ), ( "TX fault", "txFault" ),
                         ( "RX CDR LOL", "rxCdrLol" ),
                         ( "TX power high alarm", "txPowerHiAlarm" ),
                         ( "TX power high warn", "txPowerHiWarn" ),
                         ( "TX power low alarm", "txPowerLoAlarm" ),
                         ( "TX power low warn", "txPowerLoWarn" ),
                         ( "TX bias high alarm", "txBiasHiAlarm" ),
                         ( "TX bias high warn", "txBiasHiWarn" ),
                         ( "TX bias low alarm", "txBiasLoAlarm" ),
                         ( "TX bias low warn", "txBiasLoWarn" ),
                         ( "RX power high alarm", "rxPowerHiAlarm" ),
                         ( "RX power high warn", "rxPowerHiWarn" ),
                         ( "RX power low alarm", "rxPowerLoAlarm" ),
                         ( "RX power low warn", "rxPowerLoWarn" ) ] )

      for attrName, attrValue in attrs.iteritems():
         if any( [ getattr( i, attrValue ) for i in channelsDict.values() ] ):
            print( "  %s" % attrName )
            for ch in sorted( channelsDict ):
               fmtGeneric( getattr( channelsDict[ ch ], attrValue ),
                           "Channel %d" % ch,
                           fmt="    %-27.27s %-20.20s %7.7s       %s" )
      
      lane = 1
      if lane in channelsDict:
         if channelsDict[ lane ].coherentAlarms is not None:
            channelsDict[ lane ].coherentAlarms.renderModel()
         if channelsDict[ lane ].frequencyTuningAlarms is not None:
            channelsDict[ lane ].frequencyTuningAlarms.renderModel()
   elif len( channelsDict ) == 1:
      ch, chModel = list( channelsDict.items() )[ 0 ]
      if printChLabel:
         print( "Channel %d" % ch )
      chModel.renderModel()

def renderEnhancedDomAttrs( hostLanesDict ):
   enhancedDomAttrs = ordDict( [ ( "Pre-FEC bit error rate",
                                   "preFecBERCurrHost" ),
                                 ( "Post-FEC errored frames ratio",
                                   "errFramesCurHost" ) ] )
   for attrName, attrValue in enhancedDomAttrs.iteritems():
      applicableLanes = sum( [ getattr( i, attrValue, None ) is not None \
                               for i in hostLanesDict.values() ] )
      if applicableLanes > 1:  
         print( "  %s" % attrName )
         for lane in sorted( hostLanesDict ):
            fmtGeneric( getattr( hostLanesDict[ lane ], attrValue ),
                        "Host lane %d" % lane,
                        fmt="    %-27.27s %-20.20s %7.7s       %s" )
      elif applicableLanes == 1:
         lane = 1
         fmtGeneric( getattr( hostLanesDict[ lane ], attrValue ),
                     attrName )

def renderHostLaneAttr( hostLanesDict, printHostLaneLabel=False ):
   if len( hostLanesDict ) > 1:
      # Follows the format of:
      #   Attribute name
      #     Host lane 1
      #     Host lane 2
      #     Host lane ...
      #     Host lane n
      renderEnhancedDomAttrs( hostLanesDict )

      attrs = ordDict( [ ( "TX LOS", "txLos" ), ( "TX CDR LOL", "txCdrLol" ),
                         ( "TX adaptive input EQ fault", "txAdaptiveInputEqFault" ),
                       ] )
      for attrName, attrValue in attrs.iteritems():
         if any( [ getattr( i, attrValue ) for i in hostLanesDict.values() ] ):
            print( "  %s" % attrName )
            for lane in sorted( hostLanesDict ):
               fmtGeneric( getattr( hostLanesDict[ lane ], attrValue ),
                           "Host lane %d" % lane,
                           fmt="    %-27.27s %-20.20s %7.7s       %s" )
   elif len( hostLanesDict ) == 1:
      lane, laneModel = list( hostLanesDict.items() )[ 0 ]
      if printHostLaneLabel:
         print( "Host lane %d" % lane )
      laneModel.renderModel()

class InterfacesTransceiverStatusFrequencyTuningChannelSpecific( Model ):
   __public__ = False
   tuningInProgress = Submodel( valueType=InterfacesTransceiverStatusValueFormatBool,
                                help="Frequency tuning in progress", optional=True )
   tuningNotAccepted = Submodel( 
         valueType=InterfacesTransceiverStatusValueFormatBool,
         help="Frequency tuning busy", optional=True )
   invalidChannel = Submodel(
         valueType=InterfacesTransceiverStatusValueFormatBool,
         help="Frequency tuning invalid channel ( configured channel is outside the \
               range for the selected grid spacing )",
         optional=True )
   tuningComplete = Submodel( valueType=InterfacesTransceiverStatusValueFormatBool,
                              help="Frequency tuning completed",
                              optional=True )

   def renderModel( self ):
      attrs = [ AttrDesc( self.tuningInProgress, 'Freq tuning in progress',
                          'in progress', 'idle' ),
                AttrDesc( self.tuningNotAccepted, 'Freq tuning busy', 'alarm',
                          'ok' ),
                AttrDesc( self.invalidChannel, 'Freq tuning invalid channel',
                          'alarm', 'ok' ),
                AttrDesc( self.tuningComplete, 'Freq tuning completed', 'yes',
                          'no' )
              ]

      for attrDesc in attrs:
         fmtDescriptive( attrDesc )

class InterfacesTransceiverStatusCoherentChannelSpecific( Model ):
   __public__ = False
   txLossOfAlignment = Submodel( 
         valueType=InterfacesTransceiverStatusValueFormatBool,
         help="Transmit loss of alignment", optional=True )
   txOutOfAlignment = Submodel( valueType=InterfacesTransceiverStatusValueFormatBool,
                                 help="Transmit out of alignment", optional=True )
   txCmuLossOfLock = Submodel( valueType=InterfacesTransceiverStatusValueFormatBool,
                        help="Transmit clock monitor unit clock loss of lock",
                        optional=True )
   txRefClkLossOfLock = Submodel( 
         valueType=InterfacesTransceiverStatusValueFormatBool,
         help="Transmit reference clock loss of lock",
         optional=True )
   txDeskewLossOfLock = Submodel( 
         valueType=InterfacesTransceiverStatusValueFormatBool,
         help="Transmit deskew clock loss of lock",
         optional=True )
   txFifoErr = Submodel( valueType=InterfacesTransceiverStatusValueFormatBool,
                           help="Transmit FIFO error", optional=True )
   rxDemodulationLossOfLock = Submodel( 
                          valueType=InterfacesTransceiverStatusValueFormatBool,
                          help="Receive demodulator loss of lock",
                          optional=True )
   rxChrDispCompensationLossOfLock = \
         Submodel( valueType=InterfacesTransceiverStatusValueFormatBool,
                   help="Receive chromatic dispersion compensation loss of lock",
                   optional=True )
   rxLossOfAlignment = Submodel( 
         valueType=InterfacesTransceiverStatusValueFormatBool,
         help="Receive loss of alignment", optional=True )
   rxOutOfAlignment = Submodel( valueType=InterfacesTransceiverStatusValueFormatBool,
                                help="Receive out of alignment", optional=True )
   rxDeskewLossOfLock = Submodel( 
         valueType=InterfacesTransceiverStatusValueFormatBool,
         help="Receive deskew loss of lock", optional=True )
   rxFifoErr = Submodel( valueType=InterfacesTransceiverStatusValueFormatBool,
                           help="Receive FIFO error", optional=True )
   fecExcessiveDegrade = Submodel( 
         valueType=InterfacesTransceiverStatusValueFormatBool,
         help="Receive FEC excessive degrade ( indicates that BER exceeded the \
               excessive degrade threshold )",
         optional=True )
   fecDetectedDegrade = Submodel( 
         valueType=InterfacesTransceiverStatusValueFormatBool,
         help="Receive FEC detect degrade ( indicates that BER exceeded the \
               detected degrade threshold )", 
               optional=True )
   
   def renderModel( self ):
      attrs = [ AttrDesc( self.txLossOfAlignment, 'TX loss of alignment',
                          'unaligned', 'ok' ),
                AttrDesc( self.txOutOfAlignment, 'TX out of alignment',
                          'unaligned', 'ok' ),
                AttrDesc( self.txCmuLossOfLock, 'TX clock monitor unit LOL',
                          'no lock', 'ok' ),
                AttrDesc( self.txRefClkLossOfLock, 'TX reference clock LOL',
                          'no lock', 'ok' ),
                AttrDesc( self.txDeskewLossOfLock, 'TX deskew LOL', 'no lock',
                          'ok' ),
                AttrDesc( self.txFifoErr, 'TX FIFO error', 'error', 'ok' ),
                AttrDesc( self.rxDemodulationLossOfLock, 'RX demodulator LOL',
                          'no lock', 'ok' ),
                AttrDesc( self.rxChrDispCompensationLossOfLock,
                          'RX CD compensation LOL', 'no lock', 'ok' ),
                AttrDesc( self.rxLossOfAlignment, 'RX loss of alignment',
                          'unaligned', 'ok' ),
                AttrDesc( self.rxOutOfAlignment, 'RX out of alignment',
                          'unaligned', 'ok' ),
                AttrDesc( self.rxDeskewLossOfLock, 'RX deskew LOL', 'no lock',
                          'ok' ),
                AttrDesc( self.rxFifoErr, 'RX FIFO error', 'error', 'ok' ),
                AttrDesc( self.fecExcessiveDegrade, 'RX FEC excessive degrade',
                          'alarm', 'ok' ),
                AttrDesc( self.fecDetectedDegrade, 'RX FEC detected degrade',
                          'alarm', 'ok' ) 
              ]         
      for attrDesc in attrs:
         fmtDescriptive( attrDesc )

class InterfacesTransceiverStatusChannelSpecific( Model ):
   __public__ = False
   rxLos = Submodel( valueType=InterfacesTransceiverStatusValueFormat,
                     help="Receive loss of signal", optional=True )
   txFault = Submodel( valueType=InterfacesTransceiverStatusValueFormat,
                       help="Transmit fault", optional=True )
   rxCdrLol = Submodel( valueType=InterfacesTransceiverStatusValueFormat,
                        help="Receive clock and data recovery loss of lock",
                        optional=True )
   txPowerHiAlarm = Submodel( valueType=InterfacesTransceiverStatusValueFormat,
                              help="Transmit high power alarm", optional=True )
   txPowerLoAlarm = Submodel( valueType=InterfacesTransceiverStatusValueFormat,
                              help="Transmit low power alarm", optional=True )
   txPowerHiWarn = Submodel( valueType=InterfacesTransceiverStatusValueFormat,
                             help="Transmit high power warning", optional=True )
   txPowerLoWarn = Submodel( valueType=InterfacesTransceiverStatusValueFormat,
                             help="Transmit low power warning", optional=True )
   txBiasHiAlarm = Submodel( valueType=InterfacesTransceiverStatusValueFormat,
                             help="Transmit high bias alarm", optional=True )
   txBiasLoAlarm = Submodel( valueType=InterfacesTransceiverStatusValueFormat,
                             help="Transmit low bias alarm", optional=True )
   txBiasHiWarn = Submodel( valueType=InterfacesTransceiverStatusValueFormat,
                            help="Transmit high bias warning", optional=True )
   txBiasLoWarn = Submodel( valueType=InterfacesTransceiverStatusValueFormat,
                            help="Transmit low bias warning", optional=True )
   rxPowerHiAlarm = Submodel( valueType=InterfacesTransceiverStatusValueFormat,
                              help="Receive high power alarm", optional=True )
   rxPowerLoAlarm = Submodel( valueType=InterfacesTransceiverStatusValueFormat,
                              help="Receive low power alarm", optional=True )
   rxPowerHiWarn = Submodel( valueType=InterfacesTransceiverStatusValueFormat,
                             help="Receive high power warning", optional=True )
   rxPowerLoWarn = Submodel( valueType=InterfacesTransceiverStatusValueFormat,
                             help="Receive low power warning", optional=True )
   coherentAlarms = Submodel( valueType= \
                              InterfacesTransceiverStatusCoherentChannelSpecific,
                              help="Coherent media channel specific alarms",
                              optional=True )
   frequencyTuningAlarms = Submodel( 
         valueType = InterfacesTransceiverStatusFrequencyTuningChannelSpecific,
         help="Frequency tuning media channel specific CMIS alarms",
         optional=True )

   def renderModel( self ):
      fmtGeneric( self.rxLos, "RX LOS" )
      fmtGeneric( self.txFault, "TX fault" )
      fmtGeneric( self.rxCdrLol, "RX CDR LOL" )
      fmtGeneric( self.txPowerHiAlarm, "TX power high alarm" )
      fmtGeneric( self.txPowerHiWarn, "TX power high warn" )
      fmtGeneric( self.txPowerLoAlarm, "TX power low alarm" )
      fmtGeneric( self.txPowerLoWarn, "TX power low warn" )
      fmtGeneric( self.txBiasHiAlarm, "TX bias high alarm" )
      fmtGeneric( self.txBiasHiWarn, "TX bias high warn " )
      fmtGeneric( self.txBiasLoAlarm, "TX bias low alarm" )
      fmtGeneric( self.txBiasLoWarn, "TX bias low warn " )
      fmtGeneric( self.rxPowerHiAlarm, "RX power high alarm" )
      fmtGeneric( self.rxPowerHiWarn, "RX power high warn" )
      fmtGeneric( self.rxPowerLoAlarm, "RX power low alarm" )
      fmtGeneric( self.rxPowerLoWarn, "RX power low warn" )
      
      if self.coherentAlarms is not None:
         self.coherentAlarms.renderModel()

      if self.frequencyTuningAlarms is not None:
         self.frequencyTuningAlarms.renderModel()
   
      
class InterfacesTransceiverStatusHostLaneSpecific( Model ):
   __public__ = False

   preFecBERCurrHost = Submodel( 
         valueType=InterfacesTransceiverStatusValueFormatScientificFloat,
         help="Current value of host-side pre-FEC bit error rate", optional=True )
   errFramesCurHost = Submodel( 
         valueType=InterfacesTransceiverStatusValueFormatScientificFloat,
         help="Current value of host-side post-FEC errored frames ratio",
         optional=True )

   txLos = Submodel( valueType=InterfacesTransceiverStatusValueFormat,
                     help="Transmit loss of signal", optional=True )
   txCdrLol = Submodel( valueType=InterfacesTransceiverStatusValueFormat,
                        help="Transmit clock and data recovery loss of lock",
                        optional=True )
   txAdaptiveInputEqFault = Submodel( 
         valueType=InterfacesTransceiverStatusValueFormat,
         help="Transmit adaptive input equalization fault", optional=True )

   def renderModel( self ):
      fmtGeneric( self.preFecBERCurrHost, "Pre-FEC bit error rate" )
      fmtGeneric( self.errFramesCurHost, "Post-FEC errored frames ratio" )
      fmtGeneric( self.txLos, "TX LOS" )
      fmtGeneric( self.txCdrLol, "TX CDR LOL" )
      fmtGeneric( self.txAdaptiveInputEqFault, "TX adaptive input EQ fault" )

class InterfacesTransceiverStatusInterfaceSpecific( Model ):
   __public__ = False
   interface = Submodel( valueType=InterfacesTransceiverStatusValueFormatStr,
                         help="Interface name", optional=True )
   operSpeed = Submodel( valueType=InterfacesTransceiverStatusValueFormatStr,
                         help="Operational speed of the interface", optional=True )
   channels = Dict( valueType=InterfacesTransceiverStatusChannelSpecific,
                    keyType=int, help="Channel specific attributes indexed by "
                                      "channel number", optional=True )
   cfp2Dco = Submodel( valueType=InterfacesTransceiverStatusExtCfp2DcoIntf,
                       help="CFP2-DCO specific attributes", optional=True )
   hostLanes = Dict( valueType=InterfacesTransceiverStatusHostLaneSpecific,
                    keyType=int, help="Host lane specific attributes indexed by "
                                      "lane number", optional=True )

   def renderModel( self ):
      if self._hasCoherentAttributes():
         renderChAttr( self.channels )
      print( self.interface.state )
      # pylint: disable-msg=protected-access
      if self.cfp2Dco and self.cfp2Dco._dualLaserModulePresent:
         fmt = "  %-42.42s %-20.20s %7.7s       %s"
         fmtGeneric( self.operSpeed, "Operational speed", fmt=fmt )
      else:
         fmtGeneric( self.operSpeed, "Operational speed" )
      if not self._hasCoherentAttributes():
         renderChAttr( self.channels )
      renderHostLaneAttr( self.hostLanes )
      if self.cfp2Dco:
         self.cfp2Dco.renderExt()

   def _hasCoherentAttributes( self ):
      lane = 1
      return lane in self.channels and \
             ( self.channels[ lane ].coherentAlarms is not None or \
               self.channels[ lane ].frequencyTuningAlarms is not None )
         

class InterfacesTransceiverStatusValues( Model ):
   __public__ = False
   mediaType = Submodel( valueType=InterfacesTransceiverStatusValueFormatStr,
                           help="Transceiver media type" )
   serialNumber = Submodel( valueType=InterfacesTransceiverStatusValueFormat,
                            help="Transceiver serial number" )
   presence = Submodel( valueType=InterfacesTransceiverStatusValueFormatStr,
                        help="Indication of transceiver presence", optional=True )
   eepromReadTimeouts = Submodel( valueType=InterfacesTransceiverStatusValueFormat,
                                  help="Number of Eeprom read timeouts",
                                  optional=True )
   badEepromChecksums = Submodel( valueType=InterfacesTransceiverStatusValueFormat,
                                  help="Number of bad Eeprom checksums",
                                  optional=True )
   spuriousDet = Submodel( valueType=InterfacesTransceiverStatusValueFormat,
                               help="Number of spurious detections of the "
                                    "transceiver", optional=True )
   domControlFail = Submodel( valueType=InterfacesTransceiverStatusValueFormat,
                              help="Number of DOM control/status failures",
                              optional=True )
   resets = Submodel( valueType=InterfacesTransceiverStatusValueFormat,
                      help="Number of transceiver resets", optional=True )
   interrupts = Submodel( valueType=InterfacesTransceiverStatusValueFormat,
                          help="Number of transceiver interrupts", optional=True )
   smbusFailures = Submodel( valueType=InterfacesTransceiverStatusValueFormat,
                             help="Number of transceiver system managements bus "
                                  "failures", optional=True )
   mdioFailures = Submodel( valueType=InterfacesTransceiverStatusValueFormat,
                            help="Number of transceiver management data "
                                 "input/output failures", optional=True )
   interfaces = Dict( valueType=InterfacesTransceiverStatusInterfaceSpecific,
                      keyType=str, help="Interface specific attributes indexed "
                                        "by interface name", optional=True )
   channels = Dict( valueType=InterfacesTransceiverStatusChannelSpecific,
                    keyType=int, help="Channel specific attributes indexed by "
                                      "channel number", optional=True )
   cfp2Dco = Submodel( valueType=InterfacesTransceiverStatusExtCfp2Dco,
                       help="CFP2-DCO specific attributes", optional=True )
   adapters = Submodel( valueType=InterfacesTransceiverStatuValuesSwizzlerList,
                        help="Form factor adapter types in use", optional=True )
   dataPathFirmwareFault = Submodel(
                           valueType=InterfacesTransceiverStatusValueFormat,
                           help="Data path firmware fault", optional=True )
   moduleFirmwareFault = Submodel( valueType=InterfacesTransceiverStatusValueFormat,
                                   help="Module firmware fault", optional=True )
   tempHiAlarm = Submodel( valueType=InterfacesTransceiverStatusValueFormat,
                           help="High temperature alarm", optional=True )
   tempLoAlarm = Submodel( valueType=InterfacesTransceiverStatusValueFormat,
                           help="Low temperature alarm", optional=True )
   tempHiWarn = Submodel( valueType=InterfacesTransceiverStatusValueFormat,
                          help="High temperature warn", optional=True )
   tempLoWarn = Submodel( valueType=InterfacesTransceiverStatusValueFormat,
                          help="Low temperature warn", optional=True )
   voltageHiAlarm = Submodel( valueType=InterfacesTransceiverStatusValueFormat,
                              help="High voltage alarm", optional=True )
   voltageLoAlarm = Submodel( valueType=InterfacesTransceiverStatusValueFormat,
                              help="Low voltage alarm", optional=True )
   voltageHiWarn = Submodel( valueType=InterfacesTransceiverStatusValueFormat,
                             help="High voltage alarm", optional=True )
   voltageLoWarn = Submodel( valueType=InterfacesTransceiverStatusValueFormat,
                             help="Low voltage alarm", optional=True )
   moduleState = Submodel( valueType=InterfacesTransceiverStatusValueFormatStr,
                           help="Module state", optional=True )
   dataPathState = Dict( valueType=InterfacesTransceiverStatusValueFormatStr,
                         keyType=int, help="Data path state indexed by "
                                           "host lane number", optional=True )
   hostLanes = Dict( valueType=InterfacesTransceiverStatusHostLaneSpecific,
                    keyType=int, help="Host lane specific attributes indexed by "
                                      "lane number", optional=True )

   def renderModel( self ):
      attrs = [ ( self.mediaType, "Transceiver" ),
                ( self.serialNumber, "Transceiver SN" ),
                ( self.presence, "Presence" ),
                ( self.adapters, "Adapters" ),
                ( self.eepromReadTimeouts, "Xcvr EEPROM read timeout" ),
                ( self.badEepromChecksums, "Bad EEPROM checksums" ),
                ( self.spuriousDet, "Spurious xcvr detection" ),
                ( self.domControlFail, "DOM control/status fail" ),
                ( self.resets, "Resets" ),
                ( self.interrupts, "Interrupts" ),
                ( self.smbusFailures, "Smbus failures" ),
                ( self.mdioFailures, "Mdio failures" ),
                ( self.dataPathFirmwareFault, "Data path firmware fault" ),
                ( self.moduleFirmwareFault, "Module firmware fault" ),
                ( self.tempHiAlarm, "Temperature high alarm" ),
                ( self.tempHiWarn, "Temperature high warn" ),
                ( self.tempLoAlarm, "Temperature low alarm" ),
                ( self.tempLoWarn, "Temperature low warn" ),
                ( self.voltageHiAlarm, "Voltage high alarm" ),
                ( self.voltageHiWarn, "Voltage high warn" ),
                ( self.voltageLoAlarm, "Voltage low alarm" ),
                ( self.voltageLoWarn, "Voltage low warn" ),
                ( self.moduleState, "Module state" ) ]
      for attr in attrs:
         if self.cfp2Dco and self.cfp2Dco.dualLaserModulePresent():
            dualLaserFmt = "  %-42.42s %-20.20s %7.7s       %s"
            fmtGeneric( *attr, fmt=dualLaserFmt )
         else:
            fmtGeneric( *attr )

      if self.cfp2Dco:
         self.cfp2Dco.renderExt()

      for hostLane, dathPathStateModel in sorted( self.dataPathState.iteritems() ):
         dataPathStateAttrName = "Data path %s state" % hostLane
         fmtGeneric( dathPathStateModel, dataPathStateAttrName )
      
      for intf in Arnet.sortIntf( self.interfaces ):
         self.interfaces[ intf ].renderModel()

      # pylint: disable-msg=protected-access
      if self.cfp2Dco and self.cfp2Dco.dualLaserModulePresent():
         # Channel parameters are not populated by dual laser dco module
         return
      renderChAttr( self.channels, printChLabel=True )
      renderHostLaneAttr( self.hostLanes, printHostLaneLabel=True )

class InterfacesTransceiverStatus( Model ):
   __public__ = False
   ports = OrderedDict( keyType=str, valueType=InterfacesTransceiverStatusValues,
                 help='Mapping between port name and transceiver status' )
   def render( self ):
      if not self.ports:
         return

      fmt = "%-31.31s %-20.20s %-13.13s %11.11s"
      for port in self.ports:
         if ( self.ports[ port ].cfp2Dco and
              self.ports[ port ].cfp2Dco.dualLaserModulePresent() ):
            fmt = "  %-42.42s %-20.20s %7.7s       %s"
         print( fmt % ( "", "Current State", "Changes", "Last Change" ) )
         print( fmt % ( "", "-------------", "-------", "-----------" ) )

         print( port )
         self.ports[ port ].renderModel()
         print()

