# Copyright (c) 2020 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.
#
# This file contains the CLI command definition for the
# '[no|default] transceiver diag management' command.

from __future__ import absolute_import, division, print_function
import CliCommand
import CliMatcher
import CliParser
from CliPlugin.EthIntfCli import xcvrKw
from CliPlugin.XcvrConfigCli import XcvrConfigModelet
from CliPlugin.XcvrConfigCli import diagKeyword
from CliPlugin.XcvrConfigCli import getXcvrConfigCliDirHelper
from CliPlugin.XcvrConfigCli import xcvrConfigCliDelete
from CliPlugin.XcvrConfigCli import primaryIntfGuard
import ConfigMount
import LazyMount
import Tac

#--------------------------------------------------------------------------------
# [ no | default ] transceiver diag management
#--------------------------------------------------------------------------------
managementKeyword = CliMatcher.KeywordMatcher( 'management',
      helpdesc='Configure transceiver software management mode' )

def capabilityGuard( mode, token ):
   """
   Guard function for the 'transceiver diag management' command.

   The command is guarded by capability information which comes from a
   platform's FDL.  This means only ports with this special capability can
   run this command.

   Additionally we only allow this command to run on primary interfaces.  We
   invoke the primary interface guard as well.
   """
   xcvrName = xcvrConfigDir.intfToXcvrName.get( mode.intf.name )
   assert xcvrName is not None

   xcvrConfig = xcvrConfigDir.xcvrConfig.get( xcvrName )
   capable = ( isinstance( xcvrConfig, Tac.Type( "Xcvr::QsfpConfig" ) ) and
               xcvrConfig.qsfpSlotCapabilities.whisperPortCapable )
   if capable:
      # Command must be run on a primary interface
      return primaryIntfGuard( mode, token )
   else:
      return CliParser.guardNotThisPlatform

class TransceiverDiagManagementCmd( CliCommand.CliCommandClass ):
   syntax = 'transceiver diag management'
   noOrDefaultSyntax = syntax
   data = {
      'transceiver' : xcvrKw,
      'diag' : diagKeyword,
      'management' : CliCommand.Node( matcher=managementKeyword,
                                      guard=capabilityGuard ),
   }

   @staticmethod
   def handler( mode, args ):
      noOrDefault = CliCommand.isNoOrDefaultCmd( args )
      intfName = mode.intf.name
      xcvrConfigCliDir = getXcvrConfigCliDirHelper( intfName, xcvrConfigCliSliceDir )

      # Create XcvrConfigCli for this interface if it does not exist, unless this
      # is no/default.  In that case the command is a no-op.
      if xcvrConfigCliDir.xcvrConfigCli.get( intfName ) is None:
         if noOrDefault:
            return
         xcvrConfigCliDir.newXcvrConfigCli( intfName )

      xcvrConfigCli = xcvrConfigCliDir.xcvrConfigCli.get( intfName )
      assert xcvrConfigCli

      forceManagement = not noOrDefault
      xcvrConfigCli.forceQsfpManagementMode = forceManagement

      if xcvrConfigCliDelete( xcvrConfigCli ):
         del xcvrConfigCliDir.xcvrConfigCli[ intfName ]

   # [ no | default ] uses the same handler function
   noOrDefaultHandler = handler

XcvrConfigModelet.addCommandClass( TransceiverDiagManagementCmd )


#-----------------------------------------------------------------------------------
# Plugin
#-----------------------------------------------------------------------------------
xcvrConfigCliSliceDir = None
xcvrConfigDir = None

def Plugin( em ):
   global xcvrConfigCliSliceDir
   global xcvrConfigDir

   xcvrConfigCliSliceDir = ConfigMount.mount( em, "hardware/xcvr/cli/config/slice",
                                              "Tac::Dir", "wi" )
   xcvrConfigDir = LazyMount.mount( em, "hardware/xcvr/config/all",
                                    "Xcvr::AllConfigDir", "r" )
