# Copyright (c) 2012 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function
import re
import csv
import os
import BasicCli, CliParser, ConfigMount, Tracing, Tac
from CliPlugin.EthIntfCli import xcvrKw, qsfpKw
from CliPlugin import IntfCli, XcvrAllStatusDir
import CliToken.Monitor
import Cell
import EthIntfLib
import LazyMount
import Toggles.XcvrToggleLib
import MultiRangeRule
import Logging
import Url
from XcvrLib import getXcvrStatus, validMediaType
from XcvrLib import mediaTypeCliToEnums, tokensToHelp
from CliMode.Layer1Monitor import Layer1MonitorBaseMode
import CliCommand
import CliMatcher
from TypeFuture import TacLazyType

TRANSCEIVER_DOM_THRESHOLD_OVERRIDE_WARNING = None
Logging.logD( id="TRANSCEIVER_DOM_THRESHOLD_OVERRIDE_WARNING",
              severity=Logging.logWarning,
              format="Custom-defined DOM thresholds have been enabled, and "
                     "will be used instead of the manufacturer-defined thresholds.",
              explanation="Custom DOM thresholds have been enabled by the "
                          "transceiver dom-threshold command. These thresholds "
                          "will apply to modules meeting the custom-defined "
                          "conditions. The custom DOM thresholds will now be used "
                          "in place of manufacturer-defined thresholds.",
              recommendedAction="Ensure the custom DOM thresholds are appropriate "
                                "for the matching transceivers." )

TRANSCEIVER_DOM_THRESHOLD_OVERRIDE_FAILED = None
Logging.logD( id="TRANSCEIVER_DOM_THRESHOLD_OVERRIDE_FAILED",
              severity=Logging.logError,
              format="%s doesn't exist or contains errors.",
              explanation="The transceiver DOM thresholds weren't overridden.",
              recommendedAction="Check the DOM threshold file"
              "to make sure it exists and has the right format." )

__defaultTraceHandle__ = Tracing.Handle( "XcvrCli" )

# Helper functions

xgc = None
xcvrCliConfigSliceDir = None
xcvrStatusDir = None
xcvrConfigDir = None

sfpHighPowerEnable = Toggles.XcvrToggleLib.toggleSfpHighPowerEnabled()

matcherPerformanceMonitoring = CliMatcher.KeywordMatcher( 'performance-monitoring',
      helpdesc='Set performance-monitoring parameters' )
matcherElectrical = CliMatcher.KeywordMatcher( 'electrical',
      helpdesc='Configure transceiver electrical side parameters' )
matcherTransmitter = CliMatcher.KeywordMatcher( 'transmitter',
      helpdesc='Transmitter parameters' )
diagKeyword = CliMatcher.KeywordMatcher( 'diag',
      helpdesc='Configure transceiver in diagnostics mode' )

XcvrType = TacLazyType( 'Xcvr::XcvrType' )
XcvrMediaType = TacLazyType( 'Xcvr::MediaType' )

def getXcvrConfigCliDir( intfName ):
   """
   Gets the XcvrConfigCli object belonging to the passed in interface.

   Uses the globally scoped xcvrCliConfigSliceDir set in this file's
   CLI plugin function.

   Parameters
   ----------
   intfName : str
      Interface to retrive the XcvrConfigCli for
   Returns
   -------
   Xcvr::XcvrConfigCli
      The XcvrConfigCli object associated with the passed in interface
   """
   return getXcvrConfigCliDirHelper( intfName, xcvrCliConfigSliceDir )

def getXcvrConfigCliDirHelper( intfName, sliceDir ):
   """
   Gets the XcvrConfigCli object belonging to the passed in interface.

   Parameters
   ----------
   intfName : str
      Interface to retrieve the XcvrConfigCli for
   cliConfigDir : Tac::Dir
      Slicified directory containing XcvrConfigCli objects

   Returns
   -------
   Xcvr::XcvrConfigCli
      The XcvrConfigCli object associated with the passed in interface
   """
   # Helper function to get coathanger associated with interface
   isModular = Cell.cellType() == "supervisor"

   # Supervisors having xcvr slots will be present in FixedSystem slice
   if isModular and not Tac.Type( "Arnet::MgmtIntfId" ).isMgmtIntfId( intfName ):
      sliceName = EthIntfLib.sliceName( intfName )
      if not sliceName:
         return None
      xcvrConfigCliDir = sliceDir.get( sliceName )
   else:
      xcvrConfigCliDir = sliceDir[ 'FixedSystem' ]
   return xcvrConfigCliDir

def noOrDefaultInCommand( context ):
   """
   Helper function that determines whether "no" or "default" appear in the context
   of a command. We need to use this for cases when we are using a 
   DynamicKeywordMatcher and we want to always allow no|default commands to go 
   through.

   Parameters
   ----------
   context: ParserCtx object
      Object passed by a DynamicKeywordMatcher to the matching function if 
      passContext=True is given to its constructor.

   Returns
   -------
   boolean
      True if no/default is present, False otherwise
   """
   for node in context.matchedNodes:
      if node.name_ in [ CliCommand.NAME_NO, CliCommand.NAME_DEFAULT ]:
         return True

   return False


class XcvrConfigModelet( CliParser.Modelet ):
   modeletParseTree = CliParser.ModeletParseTree()
   def __init__( self, mode ):
      CliParser.Modelet.__init__( self )
      
   @staticmethod
   def shouldAddModeletRule( mode ):
      return mode.intf.name.startswith( 'Ethernet' )

IntfCli.IntfConfigMode.addModelet( XcvrConfigModelet )

# Support for default interface command (AID905)
class XcvrTuningIntfCleaner( IntfCli.IntfDependentBase ):
   def setDefault( self ):
      del xgc.tuningConfig[ self.intf_.name ]
      del xgc.rxTuningConfig[ self.intf_.name ]
      del xgc.txPowerConfig[ self.intf_.name ]
      del xgc.rxPowerConfig[ self.intf_.name ]
      del xgc.txDisabled[ self.intf_.name ]

# Support for default interface command (AID905)
# clear the config in xcvrConfigCliDir
class XcvrConfigCliIntfCleaner( IntfCli.IntfDependentBase ):
   def setDefault( self ):
      intfName = self.intf_.name
      xcvrConfigCliDir = getXcvrConfigCliDir( intfName )
      if not xcvrConfigCliDir:
         return
      del xcvrConfigCliDir.xcvrConfigCli[ intfName ]

# Support for default interface command (AID905)
class PerformanceMonitoringIntfCleaner( IntfCli.IntfDependentBase ):
   def setDefault( self ):
      if xgc.performanceMonitoringConfig.has_key( self.intf_.name ):
         del xgc.performanceMonitoringConfig[ self.intf_.name ]

##### [no|default] "service unsupported-transceiver <licenseeName> <key>" #####

class XcvrModelet( CliParser.SingletonModelet ):
   modeletParseTree = CliParser.ModeletParseTree()
   def __init__( self, mode ):
      self.mode = mode
      CliParser.SingletonModelet.__init__( self )

BasicCli.GlobalConfigMode.addModelet( XcvrModelet )

#--------------------------------------------------------------------------------
# [ no | default ] service unsupported-transceiver LICENSEE AUTH_KEY
#--------------------------------------------------------------------------------
class ServiceUnsupportedTransceiverCmd( CliCommand.CliCommandClass ):
   syntax = 'service unsupported-transceiver LICENSEE AUTH_KEY'
   noOrDefaultSyntax = 'service unsupported-transceiver ...'
   data = {
      'service' : 'Configure service parameters',
      'unsupported-transceiver' : 'Enable unsupported transceivers, if authorized',
      'LICENSEE' : CliMatcher.PatternMatcher( r'\w+', helpname='WORD',
         helpdesc='Licensee name' ),
      'AUTH_KEY' : CliCommand.Node(
         matcher=CliMatcher.PatternMatcher( r'[a-fA-F0-9]{1,8}', helpname='HEX_KEY',
            helpdesc='Key to authorize unsupported transceivers' ),
         sensitive=True )
   }
   hidden = True

   @staticmethod
   def handler( mode, args ):
      xgc.licensee = args[ 'LICENSEE' ]
      xgc.key = int( args[ 'AUTH_KEY' ], 16 )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      xgc.licensee = ''
      xgc.key = 0

XcvrModelet.addCommandClass( ServiceUnsupportedTransceiverCmd )

#--------------------------------------------------------------------------------
# [ no | default ] transceiver qsfp ethernet-compliance bit7-is-psm4
#--------------------------------------------------------------------------------
class QsfpComplianceBit( CliCommand.CliCommandClass ):
   syntax = 'transceiver qsfp ethernet-compliance bit7-is-psm4'
   noOrDefaultSyntax = syntax
   data = {
      'transceiver' : xcvrKw,
      'qsfp' : qsfpKw,
      'ethernet-compliance' : ( 'Override standard meaning of bits in QSFP '
                                'eth-compliance field' ),
      'bit7-is-psm4' : 'Bit7 identifies 40GBASE-PSM4 media type',
   }
   hidden = True

   @staticmethod
   def handler( mode, args ):
      xgc.ethComplianceBit7IsPsm4 = True

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      xgc.ethComplianceBit7IsPsm4 = False

BasicCli.GlobalConfigMode.addCommandClass( QsfpComplianceBit )

#--------------------------------------------------------------------------------
# [ no | default ] transceiver frequency FREQUENCY
#--------------------------------------------------------------------------------
def mhzFreqToGhz( frequency ):
   return float( frequency ) / 1e3

def ghzFreqToMhz( frequency ):
   return float( frequency ) * 1e3

def getTuningFreq( baseFrequency, channel, grid ):
   return baseFrequency + ( channel - 1 ) * grid

def getTuningChannel( baseFrequency, freq, grid ):
   # Round down to the nearest channel
   return  1 + ( freq - baseFrequency ) / grid

def cmdSetXcvrChannel( mode, args ):
   channel = args.get( 'CHANNEL' )
   grid = args.get( 'GRID' )
   frequency = args.get( 'FREQUENCY' )
   displayUnits = 'ghz' in args
   tuningConfigName = mode.intf.name

   if channel or grid:
      if not xgc.tuningConfig.has_key( tuningConfigName ):
         xgc.newTuningConfig( tuningConfigName )
      tuningConfig = xgc.tuningConfig[ tuningConfigName ]
      tuningConstants = Tac.Value( 'Xcvr::TuningConstants' )
      tuningConfig.grid = int( ghzFreqToMhz( grid ) ) if grid else \
                                         tuningConstants.defaultGrid
      tuningConfig.channel = channel
      tuningConfig.frequency = 0
   elif frequency:
      if not xgc.tuningConfig.has_key( tuningConfigName ):
         xgc.newTuningConfig( tuningConfigName )
      tuningConfig = xgc.tuningConfig[ tuningConfigName ]
      tuningConfig.grid = 0
      tuningConfig.channel = 0
      tuningConfig.frequency = int( round( ghzFreqToMhz( frequency ) ) )
      tuningConfig.displayUnits = displayUnits
   else:
      if xgc.tuningConfig.has_key( tuningConfigName ):
         del xgc.tuningConfig[ tuningConfigName ]

gridKw = CliMatcher.KeywordMatcher( 'grid-spacing',
                          helpdesc='Grid-spacing tuning for optical transceiver' )
matcherGhzUnits = CliMatcher.KeywordMatcher( 'ghz', helpdesc='Units in gigahertz' )

# command for configuring the transmitter tuning frequency
class TransceiverFrequencyFrequencyCmd( CliCommand.CliCommandClass ):
   syntax = 'transceiver frequency FREQUENCY [ ghz ]'
   noOrDefaultSyntax = 'transceiver frequency ...'
   data = {
      'transceiver' : xcvrKw,
      'frequency' : 'Frequency tuning for optical transmitter',
      'FREQUENCY' : CliMatcher.FloatMatcher( 190000, 200000,
         helpdesc='Frequency (GHz)', precisionString='%.9g' ),
      'ghz' : matcherGhzUnits,
   }

   handler = cmdSetXcvrChannel
   noOrDefaultHandler = cmdSetXcvrChannel

XcvrConfigModelet.addCommandClass( TransceiverFrequencyFrequencyCmd )

#--------------------------------------------------------------------------------
# [ no | default ] transceiver channel CHANNEL [ grid-spacing GRID ]
#--------------------------------------------------------------------------------
class TransceiverChannelChannelCmd( CliCommand.CliCommandClass ):
   syntax = 'transceiver channel CHANNEL [ grid-spacing GRID ]'
   noOrDefaultSyntax = 'transceiver channel ...'
   data = {
      'transceiver' : xcvrKw,
      'channel' : 'Channel tuning for optical transceiver',
      'CHANNEL' : CliMatcher.IntegerMatcher( 1, 1000, helpdesc='Channel number' ),
      'grid-spacing' : gridKw,
      'GRID' : CliMatcher.PatternMatcher(
         pattern='(100|50|33|25)(\\.0{0,2})?|12\\.50?|6\\.25',
         helpdesc='Spacing (GHz)', helpname='<100|50|25|12.5|6.25>' ),
   }

   handler = cmdSetXcvrChannel
   noOrDefaultHandler = handler

XcvrConfigModelet.addCommandClass( TransceiverChannelChannelCmd )

#--------------------------------------------------------------------------------
# transceiver receiver signal-power POWER
#--------------------------------------------------------------------------------
matcherRx = CliMatcher.KeywordMatcher( 'receiver', helpdesc='Receiver parameters' )
matcherSignalPower = CliMatcher.KeywordMatcher( 'signal-power',
      helpdesc='Optical signal power' )

class TransceiverReceiverSignalPowerPowerCmd( CliCommand.CliCommandClass ):
   syntax = 'transceiver receiver signal-power POWER'
   noOrDefaultSyntax = 'transceiver receiver signal-power ...'
   data = {
      'transceiver' : xcvrKw,
      'receiver' : matcherRx,
      'signal-power' : matcherSignalPower,
      'POWER' : CliMatcher.FloatMatcher( -30, 10, helpdesc='Power (dBm)',
         precisionString='%.4g' ),
   }

   @staticmethod
   def handler( mode, args ):
      power = args[ 'POWER' ]
      if mode.intf.name not in xgc.rxPowerConfig:
         xgc.newRxPowerConfig( mode.intf.name )

      rxPowerConfig = xgc.rxPowerConfig[ mode.intf.name ]
      rxPowerConfig.signalPower = power

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      del xgc.rxPowerConfig[ mode.intf.name ]

XcvrConfigModelet.addCommandClass( TransceiverReceiverSignalPowerPowerCmd )

#--------------------------------------------------------------------------------
# transceiver receiver frequency FREQUENCY
#--------------------------------------------------------------------------------
def getRxFrequencyTokens( mode, context ):
   keywordDict = {
      'frequency': 'Frequency tuning for optical receiver',
   }

   if noOrDefaultInCommand( context ):
      return keywordDict

   xcvrName = xcvrConfigDir.intfToXcvrName.get( mode.intf.name )
   if not xcvrName:
      return keywordDict
      # If Fru hasn't started, cli will accept all commands.
   xcvrStatus = getXcvrStatus( xcvrStatusDir.xcvrStatus.get( xcvrName ) )
   if ( xcvrStatus.presence != 'xcvrPresent' and
        xcvrStatus.xcvrType in [ XcvrType.cfp2 ] ):
      # If no module is inserted, cli will accept all commands.
      return keywordDict
   if xcvrStatus.mediaType == XcvrMediaType.xcvr400GDwdmDco:
      # Only the Dp04 DCO module is able to configure rx frequency
      return keywordDict
   else:
      return {}

matcherRxFrequency = CliMatcher.DynamicKeywordMatcher( getRxFrequencyTokens,
                                                       passContext=True )

class TransceiverReceiverFrequencyCmd( CliCommand.CliCommandClass ):
   syntax = 'transceiver receiver frequency FREQUENCY ghz'
   noOrDefaultSyntax = 'transceiver receiver frequency ...'
   data = {
      'transceiver' : xcvrKw,
      'receiver' : matcherRx,
      'frequency' : matcherRxFrequency,
      'FREQUENCY' : CliMatcher.FloatMatcher( 190000, 200000,
         helpdesc='Frequency (GHz)', precisionString='%.9g' ),
      'ghz' : matcherGhzUnits,
   }

   @staticmethod
   def handler( mode, args ):
      frequency = args.get( 'FREQUENCY' )
      if mode.intf.name not in xgc.rxTuningConfig:
         xgc.newRxTuningConfig( mode.intf.name )

      rxTuningConfig = xgc.rxTuningConfig[ mode.intf.name ]
      rxTuningConfig.frequency = int( round( ghzFreqToMhz( frequency ) ) )
      rxTuningConfig.grid = 0
      rxTuningConfig.channel = 0

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      del xgc.rxTuningConfig[ mode.intf.name ]

XcvrConfigModelet.addCommandClass( TransceiverReceiverFrequencyCmd )

#--------------------------------------------------------------------------------
# transceiver receiver frequency fine FINE_FREQUENCY
#--------------------------------------------------------------------------------
matcherRxFrequencyFine = CliMatcher.KeywordMatcher( 'fine',
   helpdesc='Bright fine tuning of frequency for optical receiver' )

class TransceiverReceiverFrequencyFineCmd( CliCommand.CliCommandClass ):
   syntax = 'transceiver receiver frequency fine FINE_FREQUENCY ghz'
   noOrDefaultSyntax = 'transceiver receiver frequency fine ...'
   data = {
      'transceiver' : xcvrKw,
      'receiver' : matcherRx,
      'frequency' : matcherRxFrequency,
      'fine' : matcherRxFrequencyFine,
      'FINE_FREQUENCY' : CliMatcher.FloatMatcher( -10.0, 10.0,
         helpdesc='Frequency (GHz)', precisionString='%.9g' ),
      'ghz' : matcherGhzUnits,
   }

   @staticmethod
   def handler( mode, args ):
      fineFrequency = args.get( 'FINE_FREQUENCY' )
      if mode.intf.name not in xgc.rxTuningConfig:
         xgc.newRxTuningConfig( mode.intf.name )

      rxTuningConfig = xgc.rxTuningConfig[ mode.intf.name ]
      rxTuningConfig.fineFrequency = float( round( ghzFreqToMhz( fineFrequency ) ) )
      rxTuningConfig.grid = 0
      rxTuningConfig.channel = 0

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      if mode.intf.name in xgc.rxTuningConfig:
         xgc.rxTuningConfig[ mode.intf.name ].fineFrequency = 0.0

XcvrConfigModelet.addCommandClass( TransceiverReceiverFrequencyFineCmd )

#--------------------------------------------------------------------------------
# transceiver transmitter signal-power POWER
#--------------------------------------------------------------------------------
class TransceiverTransmitterSignalPowerPowerCmd( CliCommand.CliCommandClass ):
   syntax = 'transceiver transmitter signal-power POWER'
   noOrDefaultSyntax = 'transceiver transmitter signal-power ...'
   data = {
      'transceiver' : xcvrKw,
      'transmitter' : matcherTransmitter,
      'signal-power' : matcherSignalPower,
      'POWER' : CliMatcher.FloatMatcher( -30, 10, helpdesc='Power (dBm)',
         precisionString='%.4g' ),
   }

   @staticmethod
   def handler( mode, args ):
      power = args[ 'POWER' ]
      if mode.intf.name not in xgc.txPowerConfig:
         xgc.newTxPowerConfig( mode.intf.name )

      txPowerConfig = xgc.txPowerConfig[ mode.intf.name ]
      powerConstants = Tac.Value( 'Xcvr::PowerConstants' )
      txPowerConfig.signalPower = power if power is not None else \
                                  powerConstants.defaultTxPower

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      if mode.intf.name in xgc.txPowerConfig:
         del xgc.txPowerConfig[ mode.intf.name ]

XcvrConfigModelet.addCommandClass( TransceiverTransmitterSignalPowerPowerCmd )

#--------------------------------------------------------------------------------
# transceiver transmitter DISABLED
#--------------------------------------------------------------------------------
def getTxDisabledToken( mode, context ):
   keywordDict = {
      'DISABLED': 'Optical transmitter disabled',
   }

   if noOrDefaultInCommand( context ):
      return keywordDict

   xcvrName = xcvrConfigDir.intfToXcvrName.get( mode.intf.name )
   if not xcvrName:
      return keywordDict
   # If Fru hasn't started, cli will accept all commands.
   xcvrStatus = getXcvrStatus( xcvrStatusDir.xcvrStatus.get( xcvrName ) )
   if ( xcvrStatus.presence != 'xcvrPresent' and
        xcvrStatus.xcvrType in [ XcvrType.cfp2, XcvrType.osfp, XcvrType.qsfpDd ] ):
      # If no module is inserted, cli will accept all commands.
      return keywordDict
   if xcvrStatus.isCoherent( xcvrStatus.mediaType ):
      # Only the 400G ZR and Dp04 DCO module are able to configure soft TX disable
      return keywordDict
   else:
      return {}

matcherTxDisabled = CliMatcher.DynamicKeywordMatcher( getTxDisabledToken,
                                                      passContext=True )
class TransceiverTransmitterDisabledCmd( CliCommand.CliCommandClass ):
   syntax = 'transceiver transmitter DISABLED'
   noOrDefaultSyntax = syntax
   data = {
      'transceiver' : xcvrKw,
      'transmitter' : matcherTransmitter,
      'DISABLED' : matcherTxDisabled,
   }

   @staticmethod
   def handler( mode, args ):
      if args.get( 'DISABLED' ):
         xgc.txDisabled.add( mode.intf.name )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      del xgc.txDisabled[ mode.intf.name ]

XcvrConfigModelet.addCommandClass( TransceiverTransmitterDisabledCmd )

#--------------------------------------------------------------------------------
# [ no | default ] performance-monitoring period [ SECS | MINS | HOURS | DAYS ]
#--------------------------------------------------------------------------------
class PerformanceMonitoringPeriodCmd( CliCommand.CliCommandClass ):
   syntax = 'performance-monitoring period [ SECS | MINS | HOURS | DAYS ]'
   noOrDefaultSyntax = 'performance-monitoring period ...'
   data = {
      'performance-monitoring' : matcherPerformanceMonitoring,
      'period' : 'Set performance-monitoring period',
      'SECS' : CliMatcher.PatternMatcher( pattern='[0-9]+s', helpdesc='seconds',
         helpname='[0-9]+s' ),
      'MINS' : CliMatcher.PatternMatcher( pattern='[0-9]+m', helpdesc='minutes',
         helpname='[0-9]+m' ),
      'HOURS' : CliMatcher.PatternMatcher( pattern='[0-9]+h', helpdesc='hours',
         helpname='[0-9]+h' ),
      'DAYS' : CliMatcher.PatternMatcher( pattern='[0-9]+d', helpdesc='days',
         helpname='[0-9]+d' ),
   }

   @staticmethod
   def handler( mode, args ):
      monitoringTime = args.get( 'SECS', args.get( 'MINS',
         args.get( 'HOURS', args.get( 'DAYS' ) ) ) )
      no = CliCommand.isNoOrDefaultCmd( args )
      pmPeriod = Tac.Value( 'Xcvr::PerformanceMonitoringPeriodConfig' )
      if monitoringTime is None or no:
         pm = Tac.Value( 'Xcvr::Sff8436PerformanceMonitoring' )
         monitoringTime = '%ds' % pm.defaultPMPeriod
      m = re.match( r'([0-9]+)\s*(s|m|h|d)\s*$', monitoringTime )
      unit = m.group( 2 )
      pmPeriod.unit = unit
      if unit is None:
         pmPeriod.valueInSeconds = Tac.endOfTime
      else:
         factor = 1
         if unit != 's':
            factor *= 60
            if unit != 'm':
               factor *= 60
               if unit != 'h':
                  factor *= 24
                  assert unit == 'd', 'wrong time format'
         pmPeriod.valueInSeconds = int( m.group( 1 ) ) * factor
      xgc.performanceMonitoringPeriod = pmPeriod

   noOrDefaultHandler = handler

BasicCli.GlobalConfigMode.addCommandClass( PerformanceMonitoringPeriodCmd )

#--------------------------------------------------------------------------------
# [ no | default ] performance-monitoring transceiver default
#--------------------------------------------------------------------------------
class PerformanceMonitoringCmd( CliCommand.CliCommandClass ):
   syntax = 'performance-monitoring transceiver default'
   noOrDefaultSyntax = syntax
   data = {
      'performance-monitoring' : matcherPerformanceMonitoring,
      'transceiver' : xcvrKw,
      'default' : 'Performance-monitoring global control',
   }

   @staticmethod
   def handler( mode, args ):
      xgc.performanceMonitoringGlobal = True

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      xgc.performanceMonitoringGlobal = False

BasicCli.GlobalConfigMode.addCommandClass( PerformanceMonitoringCmd )

#--------------------------------------------------------------------------------
# [ no | default ] performance-monitoring transceiver
#--------------------------------------------------------------------------------
class IntfPerformanceMonitoringCmd( CliCommand.CliCommandClass ):
   syntax = 'performance-monitoring transceiver'
   noOrDefaultSyntax = syntax
   data = {
      'performance-monitoring' : matcherPerformanceMonitoring,
      'transceiver' : xcvrKw,
   }

   @staticmethod
   def handler( mode, args ):
      # for default, switch to global configuration
      no = CliCommand.isNoCmd( args )
      if CliCommand.isDefaultCmd( args ):
         no = not xgc.performanceMonitoringGlobal

      pmConfigName = mode.intf.name
      if not xgc.performanceMonitoringConfig.has_key( pmConfigName ):
         xgc.newPerformanceMonitoringConfig( pmConfigName )
      performanceMonitoringConfig = xgc.performanceMonitoringConfig[ pmConfigName ]
      performanceMonitoringConfig.performanceMonitoringEnabled = not no

   noOrDefaultHandler = handler

IntfCli.IntfConfigMode.addCommandClass( IntfPerformanceMonitoringCmd )

#--------------------------------------------------------------------------------
# [ no | default ] transceiver 100GE-DWDM2 temperature cooling-target TEMP celsius
#--------------------------------------------------------------------------------
class CoolingTargetTemperatureCmd( CliCommand.CliCommandClass ):
   syntax = 'transceiver 100GE-DWDM2 temperature cooling-target TEMP celsius'
   noOrDefaultSyntax = 'transceiver 100GE-DWDM2 temperature cooling-target ...'
   data = {
      'transceiver' : xcvrKw,
      '100GE-DWDM2' : 'Set 100GE-DWDM2 media type parameters',
      'temperature' : 'Set temperature related parameters on the transceiver',
      'cooling-target' : 'Set cooling target for the transceiver',
      'TEMP' : CliMatcher.IntegerMatcher( 1, 100, helpdesc='Temperature' ),
      'celsius' : 'Unit Celsius',
   }

   @staticmethod
   def handler( mode, args ):
      # Sets the target temperature config to 'temperature' Celsius
      xgc.targetTemperature100GEDwdm2 = float( args[ 'TEMP' ] )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      # Resets the target temperature config to default value for Unit Celsius
      defaultTargetTemp = Tac.Value( 'Units::Celsius' )
      xgc.targetTemperature100GEDwdm2 = defaultTargetTemp.value 

BasicCli.GlobalConfigMode.addCommandClass( CoolingTargetTemperatureCmd )

#### [no|default] transceiver media override <media type> ###

def mediaIntfReject( intfName, mediaType ):   
   xcvrName = xcvrConfigDir.intfToXcvrName.get( intfName )
   if not xcvrName:
      # if Fru haven't started, cli will accept all commands.
      return False
   xcvrStatus = getXcvrStatus( xcvrStatusDir.xcvrStatus.get( xcvrName ) )
   if xcvrStatus and xcvrStatus.presence == 'xcvrPresent':      
      if not validMediaType( mediaType, xcvrStatus ):
         return True
   return False

def intfCliReject( intfName ):
   isModular = Cell.cellType() == "supervisor"
   if isModular:
      numOfSlash = sum( c == '/' for c in intfName )
      if numOfSlash == 2 and intfName[ -2: ] != '/1':
         return True
   else:
      if '/' in intfName and intfName[ -2: ] != '/1': 
         return True
   return False

def xcvrConfigCliDelete( xcvrConfigCli ):
   """
   Parameters
   ----------
   Xcvr::XcvrConfigCli

   Returns
   -------
   boolean
      True if the passed in xcvrConfigCli object is able to be deleted.
      Otherwise, returns False.
   """
   # We need to check every attribute of xcvrConfigCli equals to default one.
   # Otherwise, we cannot delete xcvrConfigCli.
   if xcvrConfigCli.overrideMediaType != "xcvrUnknown" or \
      xcvrConfigCli.overrideTuningParams:
      return False
   for laneTuningParam in xcvrConfigCli.slotTuningParams.values():
      if laneTuningParam.txInputEqualization.valid or \
         laneTuningParam.txInputAdaptiveEqEnable.valid or \
         laneTuningParam.rxOutputAmplitude.valid or \
         laneTuningParam.rxOutputPreEmphasis.valid or \
         laneTuningParam.rxOutputPostEmphasis.valid or \
         laneTuningParam.txInputEqualization.moduleDefault or \
         laneTuningParam.rxOutputAmplitude.moduleDefault or \
         laneTuningParam.rxOutputPreEmphasis.moduleDefault:
         return False
   if xcvrConfigCli.forceModuleRemoved:
      return False
   if sfpHighPowerEnable and xcvrConfigCli.preventSfpHighPowerEnable:
      return False
   if xcvrConfigCli.forceQsfpManagementMode:
      return False
   return True

def setMediaType( mode, args ):
   intfName = mode.intf.name
   mediaType = mediaTypeCliToEnums.get( args.get( 'MEDIA_TYPE' ) )

   if intfCliReject( intfName ):
      mode.addError( "%s is not supported for overriding the media type. "
                     "Please use its primary interface. " % intfName )
      return

   if mediaType and mediaIntfReject( intfName, mediaType ):
      mode.addError( "The overridden media type doesn't match"
                     " the transceiver type of %s. " % intfName )
      return

   xcvrConfigCliDir = getXcvrConfigCliDir( intfName )
   if mediaType:
      if not xcvrConfigCliDir.xcvrConfigCli.get( intfName ):
         xcvrConfigCliDir.newXcvrConfigCli( intfName )
      xcvrConfigCliDir.xcvrConfigCli[ intfName ].overrideMediaType = mediaType
   else:
      if xcvrConfigCliDir.xcvrConfigCli.get( intfName ):
         xcvrConfigCliDir.xcvrConfigCli[ intfName ].overrideMediaType = "xcvrUnknown"
         if xcvrConfigCliDelete( xcvrConfigCliDir.xcvrConfigCli[ intfName ] ):
            del xcvrConfigCliDir.xcvrConfigCli[ intfName ]

#--------------------------------------------------------------------------------
# [ no | default ] transceiver media override MEDIA_TYPE
#--------------------------------------------------------------------------------
class TransceiverMediaOverrideCmd( CliCommand.CliCommandClass ):
   syntax = 'transceiver media override MEDIA_TYPE'
   noOrDefaultSyntax = 'transceiver media override ...'
   data = {
      'transceiver' : xcvrKw,
      'media' : 'Configure media specific options',
      'override' : 'Override the detected media type',
      'MEDIA_TYPE' : CliMatcher.EnumMatcher( tokensToHelp ),
   }

   handler = setMediaType
   noOrDefaultHandler = handler

XcvrConfigModelet.addCommandClass( TransceiverMediaOverrideCmd )

##### [no | default] transceiver dom-threshold file <default|filename> #####

def enableDomThresholdOverride( mode, fileOrUrl ):
   if fileOrUrl == 'default':
      domThresholdDataSrc = '/usr/share/Xcvr/AristaStandardDOMThresholds.csv'
   elif os.path.isfile( fileOrUrl.localFilename() ):
      domThresholdDataSrc = fileOrUrl.localFilename()
      mode.addWarning(
         "Custom-defined DOM thresholds have been enabled, "
         "and will be used instead of" )
      mode.addWarning( "the manufacturer-defined thresholds." )
      mode.addWarning(
         "Ensure the custom DOM thresholds are appropriate "
         "for the matching transceivers." )
      Logging.log( TRANSCEIVER_DOM_THRESHOLD_OVERRIDE_WARNING )
   else:
      mode.addError( "%s doesn't exist or contains errors." % fileOrUrl )
      Logging.log( TRANSCEIVER_DOM_THRESHOLD_OVERRIDE_FAILED,
                   fileOrUrl.localFilename() )
      return

   domThresholdReader = csv.DictReader(
      open( domThresholdDataSrc, 'rb' ), delimiter=',' )
   domThresholds = [ i for i in domThresholdReader ]
   xgc.domThresholdOverrideData.clear()
   validFormat = True
   for dt in domThresholds:
      if ( not dt[ 'vendorName' ] and not dt[ 'vendorPn' ] ) or \
         len( dt[ 'vendorName' ] ) > 16 or len( dt[ 'vendorPn' ] ) > 16:
         validFormat = False
         break
      # Pad the vendorName and vendorPn with spaces to fit the 16-character length
      # The vendorSn is only used to generate the col key. The vendorSn in Sysdb,
      # i.e. dtData.vendorSn, is non-padded since it is partial match.
      vendorName = '{:<16}'.format( dt[ 'vendorName' ] )
      vendorPn = '{:<16}'.format( dt[ 'vendorPn' ] )
      vendorSn = '{:<16}'.format( dt[ 'vendorSn' ] )
      idx = '%s %s %s' % ( vendorName, vendorPn, vendorSn )
      # The csv content is invalid if it contains entries with
      # the same vendorName, vendorPn, and vendorSn
      if idx in xgc.domThresholdOverrideData:
         validFormat = False
         break
      dtData = xgc.newDomThresholdOverrideData( idx )

      dtData.vendorName = vendorName
      dtData.vendorPn = vendorPn
      dtData.vendorSn = dt[ 'vendorSn' ]
      # Rx power thresholds
      dtData.rxPowerHighAlarm = float( dt[ 'rxPowerHighAlarm' ] )
      dtData.rxPowerLowAlarm = float( dt[ 'rxPowerLowAlarm' ] )
      dtData.rxPowerHighWarn = float( dt[ 'rxPowerHighWarn' ] )
      dtData.rxPowerLowWarn = float( dt[ 'rxPowerLowWarn' ] )
      # Tx power thresholds
      dtData.txPowerHighAlarm = float( dt[ 'txPowerHighAlarm' ] )
      dtData.txPowerLowAlarm = float( dt[ 'txPowerLowAlarm' ] )
      dtData.txPowerHighWarn = float( dt[ 'txPowerHighWarn' ] )
      dtData.txPowerLowWarn = float( dt[ 'txPowerLowWarn' ] )

   if validFormat:
      xgc.domThresholdOverrideEnabled = True
      xgc.domThresholdOverrideSrc = str( fileOrUrl )
   else:
      mode.addError( "%s doesn't exist or contains errors." % fileOrUrl )
      Logging.log( TRANSCEIVER_DOM_THRESHOLD_OVERRIDE_FAILED,
                   fileOrUrl.localFilename() )

def disableDomThresholdOverride( mode ):
   xgc.domThresholdOverrideEnabled = False

class DomThresholdFileCmd( CliCommand.CliCommandClass ):
   syntax = "transceiver dom-threshold file default | URL | ANYFILE"
   noOrDefaultSyntax = "transceiver dom-threshold file ..."
   data = {
      'transceiver' : xcvrKw,
      'dom-threshold' : 'Configure optics transceiver DOM thresholds',
      'file' : 'Override DOM thresholds with values from specified file',
      'default' : 'Default Arista-standardized thresholds',
      'URL' : Url.UrlMatcher( lambda fs: fs.scheme in [ 'flash:' ] and
                                fs.supportsRead(),
                                'File location', acceptSimpleFile=False ),
      'ANYFILE' : CliCommand.Node(
         matcher=Url.UrlMatcher( lambda fs: ( fs.scheme in [ 'file:' ] and
                                              fs.supportsRead() ),
                                 'File location', acceptSimpleFile=False ),
         hidden=True )
      }
   @staticmethod
   def handler( mode, args ):
      enableDomThresholdOverride( mode,
                                  args.get( 'default' ) or args.get( 'URL' ) or
                                  args.get( 'ANYFILE' ) )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      disableDomThresholdOverride( mode )

BasicCli.GlobalConfigMode.addCommandClass( DomThresholdFileCmd )

#--------------------------------------------------------------------------------
# [ no | default ] transceiver electrical lane LANE_LIST
#                       [ { ( tx-input-equalization TX_INPUT_EQUALIZATION ) |
#                           ( rx-output-amplitude RX_OUTPUT_AMPLITUDE ) |
#                           ( rx-output-pre-emphasis RX_OUTPUT_PREEMPHASIS ) |
#                           ( rx-output-post-emphasis RX_OUTPUT_POSTEMPHASIS ) } ]
#--------------------------------------------------------------------------------

def copyCliLaneTuingParam( cliLaneTuningParam=None ):
   laneTuningParam = Tac.Value( "Xcvr::LaneTuningParameter" )
   laneTuningParam.txInputEqualization = Tac.Value( "Xcvr::TuningValue",
      cliLaneTuningParam.txInputEqualization.valid,
      cliLaneTuningParam.txInputEqualization.val,
      cliLaneTuningParam.txInputEqualization.moduleDefault )
   laneTuningParam.rxOutputAmplitude = Tac.Value( "Xcvr::TuningValue",
      cliLaneTuningParam.rxOutputAmplitude.valid,
      cliLaneTuningParam.rxOutputAmplitude.val,
      cliLaneTuningParam.rxOutputAmplitude.moduleDefault )
   laneTuningParam.rxOutputPreEmphasis = Tac.Value( "Xcvr::TuningValue",
      cliLaneTuningParam.rxOutputPreEmphasis.valid,
      cliLaneTuningParam.rxOutputPreEmphasis.val,
      cliLaneTuningParam.rxOutputPreEmphasis.moduleDefault )
   laneTuningParam.rxOutputPostEmphasis = Tac.Value( "Xcvr::TuningValue",
      cliLaneTuningParam.rxOutputPostEmphasis.valid,
      cliLaneTuningParam.rxOutputPostEmphasis.val,
      cliLaneTuningParam.rxOutputPostEmphasis.moduleDefault )
   laneTuningParam.txInputAdaptiveEqEnable = Tac.Value( "Xcvr::TuningValue",
      cliLaneTuningParam.txInputAdaptiveEqEnable.valid,
      cliLaneTuningParam.txInputAdaptiveEqEnable.val,
      cliLaneTuningParam.txInputAdaptiveEqEnable.moduleDefault )

   return laneTuningParam

def primaryIntfGuard( mode, token ):
   intfName = mode.intf.name
   if intfCliReject( intfName ):
      return CliParser.guardNotThisPlatform
   return None

# gyorgym NOTE: BUG504031
# This function is never called.
def moduleDefaultKeywordGuard( mode, token ):
   intfName = mode.intf.name
   xcvrName = xcvrConfigDir.intfToXcvrName.get( intfName )
   if not xcvrName:
      return None
   
   xcvrStatus = getXcvrStatus( xcvrStatusDir.xcvrStatus.get( xcvrName ) )
   if not xcvrStatus:
      return None

   if xcvrStatus.xcvrType != 'qsfpPlus':
      # Guard module default token for non-qsfp ports
      return CliParser.guardNotThisPlatform

   return None

def autoTxInputEqKeywordGuard( mode, token ):
   intfName = mode.intf.name
   xcvrName = xcvrConfigDir.intfToXcvrName.get( intfName )
   if not xcvrName:
      return None
   
   xcvrStatus = getXcvrStatus( xcvrStatusDir.xcvrStatus.get( xcvrName ) )
   if not xcvrStatus:
      return None

   if not xcvrStatus.xcvrType in [ 'osfp', 'qsfpDd' ]:
      # Guard this token for non-osfp and non-qsfpDd ports
      return CliParser.guardNotThisPlatform

   cmisTuningSupp = xcvrStatus.cmisTuningParamCapabilities.tuningParamImplemented
   autoImpl = cmisTuningSupp.txInputAdaptiveEqualizationImpl

   xcvrConfigCliDir = getXcvrConfigCliDir( intfName )
   xcvrConfigCli = xcvrConfigCliDir.xcvrConfigCli.get( intfName )
   overrideEn = xcvrConfigCli and xcvrConfigCli.overrideTuningParams

   if xcvrStatus.presence == 'xcvrPresent' and not ( autoImpl or overrideEn ):
      # When a transceiver is inserted, require it to advertise 'auto' support, or
      # the overrideTuning cli to be enabled
      return CliParser.guardNotThisPlatform

   return None

def valueTxInputEqKeywordGuard( mode, token ):
   intfName = mode.intf.name
   xcvrName = xcvrConfigDir.intfToXcvrName.get( intfName )
   if not xcvrName:
      return None

   xcvrStatus = getXcvrStatus( xcvrStatusDir.xcvrStatus.get( xcvrName ) )
   if not xcvrStatus:
      return None

   cmisEqImpl = False
   if xcvrStatus.xcvrType in [ 'osfp', 'qsfpDd' ]:
      cmisTuningSupp = xcvrStatus.cmisTuningParamCapabilities.tuningParamImplemented
      cmisEqImpl = cmisTuningSupp.txInputFixedEqualizationImpl

   xcvrConfigCliDir = getXcvrConfigCliDir( intfName )
   xcvrConfigCli = xcvrConfigCliDir.xcvrConfigCli.get( intfName )
   overrideEn = xcvrConfigCli and xcvrConfigCli.overrideTuningParams

   if xcvrStatus.xcvrType in [ 'osfp', 'qsfpDd' ] and \
      xcvrStatus.presence == 'xcvrPresent' and not ( cmisEqImpl or overrideEn ):
      # When a osfp/qsfpDd transceiver is inserted, require it to advertise
      # txInputFixedEqualization support, or the overrideTuning cli to be enabled
      return CliParser.guardNotThisPlatform

   return None

def sfpKeywordGuard( mode, token ):
   intfName = mode.intf.name
   xcvrName = xcvrConfigDir.intfToXcvrName.get( intfName )
   if not xcvrName:
      return None
   xcvrStatus = getXcvrStatus( xcvrStatusDir.xcvrStatus.get( xcvrName ) )
   if not xcvrStatus:
      return None

   if xcvrStatus.xcvrType != 'sfpPlus':
      # Guard sfp token for non-sfp ports
      return CliParser.guardNotThisPlatform

   return None

def cmisTuningParamGuardHelper( token, xcvrStatus, overrideEn ):
   if token == 'tx-input-equalization':
      paramSupportedStr = 'txInputFixedEqualizationImpl'
      paramSupportedStrAuto = 'txInputAdaptiveEqualizationImpl'
   elif token == 'rx-output-amplitude':
      paramSupportedStr = 'rxOutputAmplitudeImpl'
   elif token == 'rx-output-pre-emphasis':
      paramSupportedStr = 'rxOutputPreEmphasisImpl'
   elif token == 'rx-output-post-emphasis':
      paramSupportedStr = 'rxOutputPostEmphasisImpl'
   else:
      # Guard unsupported tuning param
      return CliParser.guardNotThisPlatform

   tuningParamImpl = getattr(
      xcvrStatus.cmisTuningParamCapabilities.tuningParamImplemented,
      paramSupportedStr )
   if token == 'tx-input-equalization' and not tuningParamImpl:
      # For tx-input-equalization, we check the fixed equalization first. If it is
      # not supported, we need to check the auto equalization. If both are not
      # supported, token 'tx-input-equalization' will be guarded.
      tuningParamImpl = getattr(
      xcvrStatus.cmisTuningParamCapabilities.tuningParamImplemented,
      paramSupportedStrAuto )

   if xcvrStatus.presence == 'xcvrPresent' and (
         not xcvrStatus.writableEeprom() or not ( tuningParamImpl or overrideEn ) ):
      # Guard this command when the osfp/qsfpDd xcvr is present but it isn't optical
      # or it doesn't support tuning the param without overriding to enable tuning
      return CliParser.guardNotThisPlatform

   return None

def nonCmisTuningParamGuardHelper( token, xcvrStatus, overrideEn ):
   qsfp28Id = Tac.enumValue( "Xcvr::Sff8024::IdentifierCode", "idenQsfp28" )

   if token == 'tx-input-equalization':
      paramSupportedStr = 'txInputEqualizationSupported'
   elif token == 'rx-output-amplitude':
      paramSupportedStr = 'rxAmplitudeSupported'
   elif token == 'rx-output-pre-emphasis':
      paramSupportedStr = 'rxPreEmphasisSupported'
   else:
      # Guard unsupported tuning param
      return CliParser.guardNotThisPlatform

   tuningParamSupp = getattr( xcvrStatus.xcvrTuningParamCapabilities,
                              paramSupportedStr )
   if xcvrStatus.xcvrType != 'qsfpPlus':
      # Guard this command for non-qsfp ports
      return CliParser.guardNotThisPlatform
   elif xcvrStatus.presence == 'xcvrPresent' and \
        ( xcvrStatus.sff8436IdEepromContents.identifier != qsfp28Id or \
          not xcvrStatus.optical or not ( tuningParamSupp or  overrideEn ) ):
      # Guard this command when the qsfp xcvr is present but not qsfp28 or it
      # doesn't support tuning the param without overriding to enable tuning
      return CliParser.guardNotThisPlatform

   return None

def tuningParamGuard( mode, token ):
   # Accept only tunable parameters
   intfName = mode.intf.name
   xcvrName = xcvrConfigDir.intfToXcvrName.get( intfName )
   if not xcvrName:
      return None
   
   xcvrStatus = getXcvrStatus( xcvrStatusDir.xcvrStatus.get( xcvrName ) )
   if not xcvrStatus:
      return None

   xcvrConfigCliDir = getXcvrConfigCliDir( intfName )
   xcvrConfigCli = xcvrConfigCliDir.xcvrConfigCli.get( intfName )
   overrideEn = xcvrConfigCli and xcvrConfigCli.overrideTuningParams
   
   if xcvrStatus.xcvrType in [ 'osfp', 'qsfpDd' ]:
      return cmisTuningParamGuardHelper( token, xcvrStatus, overrideEn )
   else:
      # qsfp check happens inside this function
      return nonCmisTuningParamGuardHelper( token, xcvrStatus, overrideEn )

def setTuningParam( mode, args ):
   intfName = mode.intf.name
   xcvrConfigCliDir = getXcvrConfigCliDir( intfName )

   if not xcvrConfigCliDir.xcvrConfigCli.get( intfName ):
      xcvrConfigCliDir.newXcvrConfigCli( intfName )
   xcvrConfigCli = xcvrConfigCliDir.xcvrConfigCli[ intfName ]
   
   for laneId in args[ 'LANE_LIST' ]:
      # laneId starts from 1
      if xcvrConfigCli.slotTuningParams.get( laneId - 1 ):
         # copy the last cli tuning config
         laneTuningParam = copyCliLaneTuingParam(
            xcvrConfigCli.slotTuningParams.get( laneId - 1 ) )
      else:
         laneTuningParam = Tac.Value( "Xcvr::LaneTuningParameter" )
 
      for tuningParam, tuningValue in args[ 'tuningParams' ]:
         if tuningValue == 'module-default':
            laneTuningValue = Tac.Value( "Xcvr::TuningValue", False, 0, True )
         elif tuningValue == 'auto':
            laneTuningValue = Tac.Value( "Xcvr::TuningValue", True, 1, False )
         else:
            laneTuningValue = Tac.Value( "Xcvr::TuningValue", True, tuningValue,
                                         False )

         if tuningParam == 'txInputEqualization':
            if tuningValue == 'auto':
               laneTuningParam.txInputEqualization = \
                  Tac.Value( "Xcvr::TuningValue", False, 0, False )
               laneTuningParam.txInputAdaptiveEqEnable = laneTuningValue
            elif tuningValue == 'module-default':
               laneTuningParam.txInputEqualization = laneTuningValue
               laneTuningParam.txInputAdaptiveEqEnable = \
                  Tac.Value( "Xcvr::TuningValue", False, 0, False )
            else:
               laneTuningParam.txInputEqualization = laneTuningValue
               # txInputEqualization and txInputAdaptiveEqEnable are mutually
               # exclusive. To make txInputEqualization configured, we need
               # to set txInputAdaptiveEqEnable to 0 and make it valid so as to
               # disable txInputAdaptiveEq
               laneTuningParam.txInputAdaptiveEqEnable = \
                  Tac.Value( "Xcvr::TuningValue", True, 0, False )
         elif tuningParam == 'rxOutputAmplitude':
            laneTuningParam.rxOutputAmplitude  = laneTuningValue
         elif tuningParam == 'rxOutputPreEmphasis':
            laneTuningParam.rxOutputPreEmphasis = laneTuningValue
         elif tuningParam == 'rxOutputPostEmphasis':
            laneTuningParam.rxOutputPostEmphasis = laneTuningValue

      xcvrConfigCli.slotTuningParams[ laneId - 1 ] = laneTuningParam

def noTuningParam( mode, args ):
   intfName = mode.intf.name
   xcvrConfigCliDir = getXcvrConfigCliDir( intfName )

   if not xcvrConfigCliDir.xcvrConfigCli.get( intfName ):
      # no xcvrConfigCli on this interface
      return

   xcvrConfigCli = xcvrConfigCliDir.xcvrConfigCli[ intfName ]
   if 'LANE_LIST' not in args:
      for laneId in xcvrConfigCli.slotTuningParams.keys():
         # clear all the tuning params for all the lanes
         del xcvrConfigCli.slotTuningParams[ laneId ]
   else:
      tuningParams = args[ 'tuningParams' ]
      for laneId in args[ 'LANE_LIST' ]:
         # laneId starts from 1
         if not xcvrConfigCli.slotTuningParams.get( laneId - 1 ):
            # no tuning param on this lane, skip unnecessary default command
            continue

         if not tuningParams:
            # clear all tuning params on this lane
            del xcvrConfigCli.slotTuningParams[ laneId - 1 ]
            continue

         laneTuningParam = copyCliLaneTuingParam(
            xcvrConfigCli.slotTuningParams.get( laneId - 1 ) )
         for tuningParam, _ in tuningParams:
            laneTuningValue = Tac.Value( "Xcvr::TuningValue", False, 0, False )
            if tuningParam == 'txInputEqualization':
               laneTuningParam.txInputEqualization = laneTuningValue
               laneTuningParam.txInputAdaptiveEqEnable = laneTuningValue
            elif tuningParam == 'rxOutputAmplitude':
               laneTuningParam.rxOutputAmplitude  = laneTuningValue
            elif tuningParam == 'rxOutputPreEmphasis':
               laneTuningParam.rxOutputPreEmphasis = laneTuningValue
            elif tuningParam == 'rxOutputPostEmphasis':
               laneTuningParam.rxOutputPostEmphasis = laneTuningValue

         xcvrConfigCli.slotTuningParams[ laneId - 1 ] = laneTuningParam

   if xcvrConfigCliDelete( xcvrConfigCliDir.xcvrConfigCli[ intfName ] ):
      del xcvrConfigCliDir.xcvrConfigCli[ intfName ]

nodeTransceiver = CliCommand.guardedKeyword( 'transceiver',
      helpdesc='configure transceiver settings', guard=primaryIntfGuard )
matcherModuleDefault = CliMatcher.KeywordMatcher( 'module-default',
      helpdesc='Configure tuning parameter to module\'s default' )

class TxInputEqualizationExpr( CliCommand.CliExpression ):
   expression = 'TX_MODULE_DEFAULT | auto | TX_INPUT_EQUALIZATION'
   data = {
      'TX_MODULE_DEFAULT' : CliCommand.Node(
         matcher=matcherModuleDefault,
         alias='TX_INPUT_EQUALIZATION' ),
      'auto' : CliCommand.guardedKeyword( 'auto',
         helpdesc='enable adaptive tx input equalization',
         guard=autoTxInputEqKeywordGuard,
         alias='TX_INPUT_EQUALIZATION' ),
      'TX_INPUT_EQUALIZATION' : CliCommand.Node(
         matcher=CliMatcher.IntegerMatcher( 0, 12, helpdesc='Tuning value' ),
         guard=valueTxInputEqKeywordGuard ),
   }

class RxOutputAmplitudeExpr( CliCommand.CliExpression ):
   expression = 'AMPLITUDE_MODULE_DEFAULT | RX_OUTPUT_AMPLITUDE'
   data = {
      'AMPLITUDE_MODULE_DEFAULT' : CliCommand.Node(
         matcher=matcherModuleDefault,
         alias='RX_OUTPUT_AMPLITUDE' ),
      'RX_OUTPUT_AMPLITUDE' : CliMatcher.IntegerMatcher( 0, 3,
         helpdesc='Tuning value' ),
   }

class RxOutputPreEmphasisExpr( CliCommand.CliExpression ):
   expression = 'EMPHASIS_MODULE_DEFAULT | RX_OUTPUT_PREEMPHASIS'
   data = {
      'EMPHASIS_MODULE_DEFAULT' : CliCommand.Node( matcher=matcherModuleDefault,
         alias='RX_OUTPUT_PREEMPHASIS' ),
      'RX_OUTPUT_PREEMPHASIS' : CliMatcher.IntegerMatcher( 0, 7,
         helpdesc='Tuning value' ),
   }

class TuningParamCmd( CliCommand.CliCommandClass ):
   syntax = ( 'transceiver electrical lane LANE_LIST '
                     '[ { ( tx-input-equalization TX_INPUT_EQUALIZATION ) | '
                         '( rx-output-amplitude RX_OUTPUT_AMPLITUDE ) | '
                         '( rx-output-pre-emphasis RX_OUTPUT_PREEMPHASIS ) | '
                         '( rx-output-post-emphasis RX_OUTPUT_POSTEMPHASIS ) } ]' )
   noOrDefaultSyntax = ( 'transceiver electrical [ lane LANE_LIST '
               '[ { ( tx-input-equalization [ TX_INPUT_EQUALIZATION ] ) | '
                   '( rx-output-amplitude [ RX_OUTPUT_AMPLITUDE ] ) | '
                   '( rx-output-pre-emphasis [ RX_OUTPUT_PREEMPHASIS ] ) | '
                   '( rx-output-post-emphasis [ RX_OUTPUT_POSTEMPHASIS ] ) } ] ]' )
   data = {
      'transceiver' : nodeTransceiver,
      'electrical' : matcherElectrical,
      'lane' : 'Transceiver lane id(s)',
      'LANE_LIST' : MultiRangeRule.MultiRangeMatcher(
         rangeFn=lambda: ( 1, 8 ),
         noSingletons=False,
         value=lambda mode, genericRangeList: genericRangeList.values(),
         helpdesc='lane number(s)' ),
      'tx-input-equalization' : CliCommand.guardedKeyword( 'tx-input-equalization',
         helpdesc='configure tx input equalization tuning',
         guard=tuningParamGuard, maxMatches=1 ),
      'TX_INPUT_EQUALIZATION' : TxInputEqualizationExpr,
      'rx-output-amplitude' : CliCommand.guardedKeyword( 'rx-output-amplitude',
         helpdesc='configure rx output amplitude tuning',
         guard=tuningParamGuard, maxMatches=1 ),
      'RX_OUTPUT_AMPLITUDE' : RxOutputAmplitudeExpr,
      'rx-output-pre-emphasis' : CliCommand.guardedKeyword( 'rx-output-pre-emphasis',
         helpdesc='configure rx output pre-emphasis tuning',
         guard=tuningParamGuard, maxMatches=1 ),
      'RX_OUTPUT_PREEMPHASIS' : RxOutputPreEmphasisExpr,
      'rx-output-post-emphasis' : CliCommand.guardedKeyword(
         'rx-output-post-emphasis',
         helpdesc='configure rx output post-emphasis tuning',
         guard=tuningParamGuard, maxMatches=1 ),
      'RX_OUTPUT_POSTEMPHASIS' : CliMatcher.IntegerMatcher( 0, 7,
         helpdesc='Tuning value' ),
   }

   @staticmethod
   def adapter( mode, args, argsList ):
      if 'tuningParams' not in args:
         args[ 'tuningParams' ] = []

      if 'tx-input-equalization' in args:
         args[ 'tuningParams' ].append( ( 'txInputEqualization',
            args.get( 'TX_INPUT_EQUALIZATION', [ None ] )[ 0 ] ) )
         del args[ 'tx-input-equalization' ]
      if 'rx-output-amplitude' in args:
         args[ 'tuningParams' ].append( ( 'rxOutputAmplitude',
            args.get( 'RX_OUTPUT_AMPLITUDE', [ None ] )[ 0 ] ) )
         del args[ 'rx-output-amplitude' ]
      if 'rx-output-pre-emphasis' in args:
         args[ 'tuningParams' ].append( ( 'rxOutputPreEmphasis',
            args.get( 'RX_OUTPUT_PREEMPHASIS', [ None ] )[ 0 ] ) )
         del args[ 'rx-output-pre-emphasis' ]
      if 'rx-output-post-emphasis' in args:
         args[ 'tuningParams' ].append( ( 'rxOutputPostEmphasis',
            args.get( 'RX_OUTPUT_POSTEMPHASIS', [ None ] )[ 0 ] ) )
         del args[ 'rx-output-post-emphasis' ]

   handler = setTuningParam
   noOrDefaultHandler = noTuningParam

XcvrConfigModelet.addCommandClass( TuningParamCmd )

#--------------------------------------------------------------------------------
# [ no | default ] transceiver electrical tuning override
#--------------------------------------------------------------------------------
class TransceiverElectricalTuningOverrideCmd( CliCommand.CliCommandClass ):
   syntax = 'transceiver electrical tuning override'
   noOrDefaultSyntax = syntax
   data = {
      'transceiver' : nodeTransceiver,
      'electrical' : matcherElectrical,
      'tuning' : 'Configure transceiver tuning capabilities',
      'override' : ( 'Ignore module EEPROM tuning capabilities and treat all as '
                     'tunable' ),
   }

   @staticmethod
   def handler( mode, args ):
      noOrDefault = CliCommand.isNoOrDefaultCmd( args )
      intfName = mode.intf.name
      xcvrConfigCliDir = getXcvrConfigCliDir( intfName )

      if not xcvrConfigCliDir.xcvrConfigCli.get( intfName ):
         if noOrDefault:
            # no xcvrConfigCli to clean on this interface
            return
         xcvrConfigCliDir.newXcvrConfigCli( intfName )

      xcvrConfigCli = xcvrConfigCliDir.xcvrConfigCli[ intfName ]
      if noOrDefault:
         xcvrConfigCli.overrideTuningParams = False
         if xcvrConfigCliDelete( xcvrConfigCliDir.xcvrConfigCli[ intfName ] ):
            del xcvrConfigCliDir.xcvrConfigCli[ intfName ]
      else:
         xcvrConfigCli.overrideTuningParams = True

   noOrDefaultHandler = handler

XcvrConfigModelet.addCommandClass( TransceiverElectricalTuningOverrideCmd )

#--------------------------------------------------------------------------------
# [ no | default ] transceiver diag simulate removed
#--------------------------------------------------------------------------------
class TransceiverDiagSimulateRemovedCmd( CliCommand.CliCommandClass ):
   syntax = 'transceiver diag simulate removed'
   noOrDefaultSyntax = syntax
   data = {
      'transceiver' : xcvrKw,
      'diag' : diagKeyword,
      'simulate' : 'Set simulated configuration',
      'removed' : 'Remove transceiver',
   }

   @staticmethod
   def handler( mode, args ):
      noOrDefault = CliCommand.isNoOrDefaultCmd( args )
      intfName = mode.intf.name
      xcvrConfigCliDir = getXcvrConfigCliDir( intfName )

      if not xcvrConfigCliDir.xcvrConfigCli.get( intfName ):
         if noOrDefault:
            # no xcvrConfigCli to clean on this interface
            return
         xcvrConfigCliDir.newXcvrConfigCli( intfName )

      xcvrConfigCli = xcvrConfigCliDir.xcvrConfigCli[ intfName ]
      if noOrDefault:
         xcvrConfigCli.forceModuleRemoved = False
         xcvrConfigCli.forceModuleRemovedChanged += 1
         if xcvrConfigCliDelete( xcvrConfigCliDir.xcvrConfigCli[ intfName ] ):
            del xcvrConfigCliDir.xcvrConfigCli[ intfName ]
      else:
         xcvrConfigCli.forceModuleRemoved = True
         xcvrConfigCli.forceModuleRemovedChanged += 1

   noOrDefaultHandler = handler

XcvrConfigModelet.addCommandClass( TransceiverDiagSimulateRemovedCmd )

####
# To prevent sfp enable high power:
# transceiver sfp power low  
# Default mode - enable high power:
# no|default transceiver sfp power [low] 
####
sfpMatcher = CliMatcher.KeywordMatcher(
             'sfp', helpdesc='SFP transceiver' )
class configSfpPowerCmd( CliCommand.CliCommandClass ):
   syntax = "transceiver sfp power low"
   noOrDefaultSyntax = "transceiver sfp power [low]"

   data = {
         'transceiver' : xcvrKw,
         'sfp' : CliCommand.Node( matcher=sfpMatcher, guard=sfpKeywordGuard ),
         'power' :  'Configure transceiver power settings',
         'low' : 'Do not permit module to draw more than 1 Watt'
         }
   @staticmethod
   def handler( mode, args ):
      intfName = mode.intf.name
      xcvrConfigCliDir = getXcvrConfigCliDir( intfName )

      if not xcvrConfigCliDir.xcvrConfigCli.get( intfName ):
         xcvrConfigCliDir.newXcvrConfigCli( intfName )
   
      xcvrConfigCli = xcvrConfigCliDir.xcvrConfigCli[ intfName ]
      xcvrConfigCli.preventSfpHighPowerEnable = True

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      intfName = mode.intf.name
      xcvrConfigCliDir = getXcvrConfigCliDir( intfName )

      if not xcvrConfigCliDir.xcvrConfigCli.get( intfName ):
         # no xcvrConfigCli on this interface
         return

      xcvrConfigCli = xcvrConfigCliDir.xcvrConfigCli[ intfName ]
      xcvrConfigCli.preventSfpHighPowerEnable = False
      if xcvrConfigCliDelete( xcvrConfigCliDir.xcvrConfigCli[ intfName ] ):
         del xcvrConfigCliDir.xcvrConfigCli[ intfName ]

if sfpHighPowerEnable:
   XcvrConfigModelet.addCommandClass( configSfpPowerCmd )

# Layer1Monitor mode
class Layer1MonitorMode( Layer1MonitorBaseMode, BasicCli.ConfigModeBase ):
   name = "Layer 1 monitor configuration"
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session ):
      Layer1MonitorBaseMode.__init__( self, None )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

#--------------------------------------------------------------------------------
# monitor layer1
#--------------------------------------------------------------------------------
class MonitorLayer1Cmd( CliCommand.CliCommandClass ):
   syntax = 'monitor layer1'
   data = {
      'monitor' : CliToken.Monitor.monitorMatcher,
      'layer1' : 'Layer 1 monitor configuration',
   }

   @staticmethod
   def handler( mode, args ):
      childMode = mode.childMode( Layer1MonitorMode )
      mode.session_.gotoChildMode( childMode )

BasicCli.GlobalConfigMode.addCommandClass( MonitorLayer1Cmd )

#--------------------------------------------------------------------------------
# [ no | default ] logging transceiver
#--------------------------------------------------------------------------------
class LoggingTransceiverCmd( CliCommand.CliCommandClass ):
   syntax = 'logging transceiver'
   noOrDefaultSyntax = syntax
   data = {
      'logging' : 'Configure logging',
      'transceiver' : 'Enable transceiver digital optical monitoring (DOM) logging',
   }

   @staticmethod
   def handler( mode, args ):
      xgc.domThresholdLogging = True

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      xgc.domThresholdLogging = False

Layer1MonitorMode.addCommandClass( LoggingTransceiverCmd )

#------------------------------------------------------
# Plugin method
#------------------------------------------------------

def Plugin( em ):
   global xgc
   global xcvrCliConfigSliceDir
   global xcvrStatusDir
   global xcvrConfigDir

   xgc = ConfigMount.mount( em, "hardware/xcvr/xgc", "Xcvr::Xgc", "w" )
   xcvrCliConfigSliceDir = ConfigMount.mount( em, "hardware/xcvr/cli/config/slice",
                                              "Tac::Dir", "wi" )
   xcvrStatusDir = XcvrAllStatusDir.xcvrAllStatusDir( em )
   xcvrConfigDir = LazyMount.mount( em, "hardware/xcvr/config/all",
                                    "Xcvr::AllConfigDir", "r" )
   IntfCli.Intf.registerDependentClass( XcvrTuningIntfCleaner )
   IntfCli.Intf.registerDependentClass( XcvrConfigCliIntfCleaner )
   IntfCli.Intf.registerDependentClass( PerformanceMonitoringIntfCleaner )
