# Copyright (c) 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, print_function, division
from collections import defaultdict, OrderedDict, namedtuple
import re
import natsort
import BasicCli, ConfigMount, Tracing, Arnet, Tac
import CliMatcher
import ShowCommand
import Toggles.XcvrToggleLib
from CliCommand import Node
from Intf.IntfRange import IntfRangeMatcher
import CliPlugin.TechSupportCli
import CliPlugin.XcvrAllStatusDir
from CliPlugin.XcvrModel import ( InterfacesTransceiver,
      InterfaceTransceiver, InterfaceTransceiverDetails,
      InterfaceTransceiverDetailThresholds, InterfaceTransceiverBase,
      InterfacesTransceiverDom, InterfaceTransceiverDom,
      InterfaceTransceiverDomBase, InterfacesTransceiverPerformanceMonitoring,
      InterfaceTransceiverPerformanceMonitoring, InterfaceTransceiverDomParameter,
      InterfaceTransceiverPerformanceMonitoringInterval,
      InterfacesTransceiverIdpromPages, InterfacesTransceiverIdpromBase,
      InterfacesTransceiverIdpromData, InterfacesTransceiverEepromBase,
      InterfacesTransceiverEepromIntf, InterfacesTransceiverHardwarePower,
      InterfacesTransceiverHardwareTuning, InterfacesTransceiverHardware,
      InterfacesTransceiverHardwareBase, InterfacesTransceiverStatus,
      InterfacesTransceiverStatusValues, InterfacesTransceiverStatusValueFormat,
      InterfacesTransceiverStatusExtCfp2Dco,
      InterfacesTransceiverStatusExtCfp2DcoIntf,
      InterfacesTransceiverStatusValueFormatModuleStateDec,
      InterfacesTransceiverStatusValueFormatTxTurnUpStateDec,
      InterfacesTransceiverStatusValueFormatRxTurnUpStateDec,
      InterfacesTransceiverStatusValueFormatModuleGeneralStatus,
      InterfacesTransceiverStatusValueFormatModuleFaultStatus,
      InterfacesTransceiverStatusValueFormatNetworkLaneVendorFAWS,
      InterfacesTransceiverStatusValueFormatNetworkLaneFaultStatus,
      InterfacesTransceiverStatusValueFormatLaneAlignment,
      InterfacesTransceiverStatusValueFormatNetworkLaneRxOTNStatus,
      InterfacesTransceiverStatusValueFormatTxModulatorBiasVOA,
      InterfacesTransceiverStatusValueFormatVendorSpecificFaultStatus,
      InterfacesTransceiverStatusIntfValueFormatIngressAlarmStatus,
      InterfacesTransceiverStatusIntfValueFormatEgressAlarmStatus,
      InterfacesTransceiverStatusValueFormatInt,
      InterfacesTransceiverStatusValueFormatStr,
      InterfacesTransceiverStatusInterfaceSpecific,
      InterfacesTransceiverStatusValueFormatBool,
      InterfacesTransceiverStatusValueFormatHex,
      InterfacesTransceiverStatusChannelSpecific,
      InterfacesTransceiverStatusHostLaneSpecific,
      InterfacesTransceiverStatuValuesSwizzlerList, SUPPORTED_FORMATS )
from CliPlugin.XcvrEepromModel import decodeSfp, decodeQsfp, decodeCmis
from CliPlugin.XcvrConfigCli import ( gridKw, mhzFreqToGhz, ghzFreqToMhz,
                                      getTuningFreq, getTuningChannel )
import CliPlugin.IntfCli
from CliPlugin.IntfCli import ShowIntfCommand, Intf, interfaceKwMatcher
import CliPlugin.EthIntfCli
from CliPlugin.EthIntfCli import getAllIntfs, EthPhyIntf, xcvrShowKw
import CliPlugin.EthIntfModel
from XcvrLib import ( getIntfNameByLane, getLineSideChannels, getXcvrSlotName,
                      getXcvrStatus, getLaneId,
                      isOsfpToQsfpSwizzler,
                      isQsfpDdToQsfpSwizzler,
                      isQsfpToSfpSwizzler )
import CliToken.Idprom
import Cell
import EthIntfLib
import LazyMount
from TypeFuture import TacLazyType

RxLosOrTxFaultImplemented = namedtuple( 'rxLosOrTxFaultImplemented',
   'rx_los tx_fault' )
CoherentClientStatusAttr = namedtuple( 'ClientPcsAlarmStatusAttr',
                                       'name trueState falseState' )
LinkMode = TacLazyType( 'Interface::EthLinkMode' )
EthSpeed = TacLazyType( 'Interface::EthSpeed' )
EthLaneCount = TacLazyType( 'Interface::EthLaneCount' )
ethTypes = TacLazyType( 'Interface::EthTypesApi' )
XcvrType = TacLazyType( 'Xcvr::XcvrType' )
XcvrMediaType = TacLazyType( 'Xcvr::MediaType' )
ClientConfig = TacLazyType(
   'Cfp2DcoHal::Cfp2DcoReg::GeneralModeControlLsb::ClientConfig' )

__defaultTraceHandle__ = Tracing.Handle( "XcvrCli" )

xgc = None
xcvrCliConfigSliceDir = None
entityManager_ = None
xcvrConfigDir_ = None
xcvrStatusDir_ = None
xcvrControllerStatusDir_ = None
phyStatus_ = None

xcvrCapCliEnable = Toggles.XcvrToggleLib.toggleXcvrCapCliEnabled()

def _strip( s ):
   if not s:
      return ""
   return s.strip().encode( 'string_escape' )

ethSpeedToStr = {
   'speedUnknown' : 'unknown',
   'speed10Mbps' : '10Mbps',
   'speed100Mbps' : '100Mbps',
   'speed1Gbps' : '1Gbps',
   'speed2p5Gbps' : '2.5Gbps',
   'speed5Gbps' : '5Gbps',
   'speed10Gbps' : '10Gbps',
   'speed25Gbps' : '25Gbps',
   'speed40Gbps' : '40Gbps',
   'speed50Gbps' : '50Gbps',
   'speed100Gbps' : '100Gbps',
   'speed200Gbps' : '200Gbps',
   'speed400Gbps' : '400Gbps',
   }

# Following parameters are exceeded counters.
# These parameters don't have any thresholds.
exceededCountersList = [ 'preFECBERHighAlarmExceeded', 'preFECBERHighWarnExceeded',
      'uncorrectedBERHighAlarmExceeded', 'uncorrectedBERHighWarnExceeded',
      'tempAlarmExceeded', 'tempWarnExceeded' ]
# These parameters are BERs.
berParametersList = [ 'preFecBERCurr', 'uncorrectedBERCurr', 'preFECBERAvg',
      'preFECBERMax', 'preFECBERMin', 'uncorrectedBERAvg', 'uncorrectedBERMax',
      'uncorrectedBERMin' ]

xcvrPresenceToStr = {
   'xcvrPresenceUnknown' : 'unknown',
   'xcvrNotPresent' : 'not present',
   'xcvrPresent' : 'present'
}

dataPathStateToStr = {
   'dataPathUnknown' : 'unknown',
   'dataPathDeactivated' : 'deactivated',
   'dataPathInit' : 'init',
   'dataPathDeinit' : 'deinit',
   'dataPathActivated' : 'activated',
   'dataPathTxTurnOn' : 'TX turn on',
   'dataPathTxTurnOff' : 'TX turn off',
   'dataPathInitialized' : 'initialized'
}

moduleStateToStr = {
   'moduleUnknown' : 'unknown',
   'moduleLowPwr' : 'low power',
   'modulePwrUp' : 'power up',
   'moduleReady' : 'ready',
   'modulePwrDn' : 'power down',
   'moduleFault' : 'fault'
}

# Helper function. Returns a tuple consisting of a list of interfaces
# and their corresponding names
def getAllIntfsWrapper( mode, intf, mod ):
   intfs = getAllIntfs( mode, intf, mod, EthPhyIntf,
                        exposeInactive=True, exposeUnconnected=True )
   if not intfs:
      return ( None, None )
   intfNames = [ intf.name for intf in intfs ]
   return ( intfs, intfNames )

class IntfMapping( object ):
   """
   Class that populates and contain the data necessary to display interfaces
   grouped together based on their current configuration. For example, given a
   100GBASE-LR4 QSFP module in a QSFP port configured to 50G-2 on /1 and /3, we
   would want to display /1 and /2 together, and display /3 and /4 together.

   Attributes:
      intfMap (dict):
         Dictionary (of the type defaultdict( dict )) that will contain the
         final port mapping.The port map ultimately contains the list optical
         channels for each interface that correspond to some interface.
         For example, in a general case of QSFP the port map might resemble:
            { 'Ethernet1/1' : { 'Ethernet1/1' : [ 0, 1, 2, 3 ] } }
         For more obscure cases, we can link interfaces to other interfaces.
         This might be done for Agile or CloudsRest2, and might look like:
            { 'Ethernet1' : { 'Ethernet1' : [ 0 ], 'Ethernet3' : [ 0 ],
                              'Ethernet5' : [ 0 ], 'Ethernet7' : [ 0 ] } }
         This means that we would print information for channel 0 of Et1 & Et3
         if the user requested info pertaining to Et1.
      intfNameOverride (dict):
         Dictionary (of the type defaultdict( dict )) that will override any
         interface names when they are being displayed.
      intfNameOverrideXcvrDom (dict):
         Same as intfNameOverride, except unique to the command 'show int
         transceiver dom'.
      portLaneMapOverride (dict):
         Dictionary (of the type defaultdict( dict )) that will override the
         lane mapping of a port. Sometimes a xcvr will only have one set of dom
         info. In these cases we want to show this info unassociated with any
         particular lane.
   """
   def __init__( self, requestedIntfs ):
      # Create empty mapping objects
      self.reqIntfs = set()
      self.intfMap = defaultdict( dict )
      self.hostMap = defaultdict( dict )

      self.xcvrStatus = xcvrStatusDir_.xcvrStatus
      if not self.xcvrStatus:
         return

      self.reqIntfs = set( i.name for i in requestedIntfs )
      self.reqPorts = set( getXcvrSlotName( i ) for i in self.reqIntfs )

      # Sometimes the lane stats don't correctly correspond to the lane mapping
      # Stores a list of interfaces and the lane to display stats for, or use
      # the key 'xcvr' to indicate lane info for the xcvr as a whole.
      self.portLaneMapOverride = defaultdict( dict )

      # Affects the interface label. Ex: "Interface1" -> "Interface1 (channel 0)"
      self.intfNameOverride = {}
      # Override the interface label for "show interfaces transceiver dom"
      self.intfNameOverrideXcvrDom = {}

      ethIntfStatusDir = LazyMount.mount( entityManager_,
                                          'interface/status/eth/intf',
                                          'Interface::EthIntfStatusDir', 'r' )
      # EthIntfStatusDir throws an exception when an invalid interface name is
      # passed as a key. We just want to use this as a dictionary without this
      # bonus feature, so we convert it to a dict.
      self.epis = dict( ethIntfStatusDir )

      # Populate generic lists and dictionaries used by the intf mapping functions
      self.intfNamesDict = defaultdict( dict )
      self.linkModeStatusDict = {}
      self.anegLanes = {}
      self.anegSpeed = {}
      self.xcvrTypesDict = {}
      self.activeIntfs = []
      self.xcvrsPresent = []
      self.electricalLanesDict = {}
      self.opticalChannelsDict = {}
      self.xcvrTypesDict = {}
      self.xcvrMediaTypesDict = {}

      self.getCoherent = False
      self.modulationDict = {}

      self.getAgile = False
      self.eplByIntf = {}

      self.polsSlots = []

      self._populateInternalData()
      self._populateInternalDataCoherent()
      self._populateInternalDataAgile()
      self._populateInternalDataPols()

      self._getIntfMap()

   def _populateInternalData( self ):
      for slotName, status in self.xcvrStatus.iteritems():
         self.xcvrTypesDict[ slotName ] = status.xcvrType
         self.xcvrMediaTypesDict[ slotName ] = status.mediaType
         self.opticalChannelsDict[ slotName ] = getLineSideChannels( status )
         self.electricalLanesDict[ slotName ] = status.capabilities.maxChannels
         if status.presence == 'xcvrPresent':
            self.xcvrsPresent.append( slotName )
         for lane, intfName in status.xcvrConfig.intfName.iteritems():
            if intfName not in self.epis:
               continue
            self.intfNamesDict[ slotName ][ lane ] = intfName
            lms = self.epis[ intfName ].linkModeStatus
            self.linkModeStatusDict[ intfName ] = lms
            anegLanes = self.epis[ intfName ].autonegStatus.laneCount
            self.anegLanes[ intfName ] = anegLanes
            anegSpeed = self.epis[ intfName ].autonegStatus.speed
            self.anegSpeed[ intfName ] = anegSpeed
            if self.epis[ intfName ].active:
               self.activeIntfs.append( intfName )

   def _populateInternalDataCoherent( self ):
      coherentSlices = []
      # This path exists on systems even if they don't support coherent optics
      coherentDir = LazyMount.mount( entityManager_, 'hardware/phy/status/data/'
                                     'coherent', 'Tac::Dir', 'ri' )
      coherentSliceDir = coherentDir.get( 'slice' )
      if coherentSliceDir:
         self.getCoherent = True
         coherentSlices = coherentSliceDir.values()
      for coherentStatusDir in coherentSlices:
         phyCoherentStatus = coherentStatusDir.phyCoherentStatus
         for intfName in phyCoherentStatus:
            curStatus = phyCoherentStatus[ intfName ]
            self.modulationDict[ intfName ] = curStatus.modulationStatus

   def _populateInternalDataAgile( self ):
      eplConfs = []
      # Mount hardware dir only as 'r' to prevent mounting all other paths
      hwDir = LazyMount.mount( entityManager_, 'hardware', 'Tac::Dir', 'r' )
      if 'focalpointv2' in hwDir:
         fpv2HwConfig = LazyMount.mount( entityManager_, 'hardware/focalpointv2/'
                                         'hwConfig',
                                         'Hardware::FocalPointV2::HwConfig', 'ri' )
         if 'alta0' in fpv2HwConfig.focalPoint:
            self.getAgile = True
            eplConfs = fpv2HwConfig.focalPoint[ 'alta0' ].eplPortConfig.values()
      self.eplByIntf = { c.intfName : c.eplId for c in eplConfs }

   def _populateInternalDataPols( self ):
      self.polsSlots = [ slotName for slotName, mediaType in
                         self.xcvrMediaTypesDict.items() if
                         mediaType == XcvrMediaType.xcvrAmpZr ]

   def _getIntfMap( self ):
      """
      Helper function that calls all mapping functions.
      """
      for curPort in self.reqPorts:
         if curPort not in self.xcvrsPresent:
            continue
         self._getIntfMapBase( curPort, self.xcvrTypesDict[ curPort ],
                               self.opticalChannelsDict[ curPort ],
                               self.electricalLanesDict[ curPort ],
                               self.intfNamesDict[ curPort ] )
      if self.getCoherent:
         self._getCoherentIntfMap()
      if self.getAgile:
         self._getAgileIntfMap()
      if self.polsSlots:
         self._getPolsIntfMap()

   def _getIntfMapBase( self, slotName, xcvrType, opticalChannels, electricalLanes,
                        intfNamesDict ):
      """
      Function to return the most common interface mapping for a single interface.
      It also handles simple special cases.
      """
      remainingChannels = range( opticalChannels )

      if slotName not in self.xcvrsPresent:
         return

      # Handle 10x10 100G cases because it is ambiguous between 10x10G and 4x25G
      baseIntfName = slotName + '/1'

      # Special case for mbo @ 100G, set its mapping to 1-11, exclude 0 and 12
      if( xcvrType == XcvrType.mbo and
          self.linkModeStatusDict[ baseIntfName ] ==
            LinkMode.linkModeForced100GbpsFull ):
         self.intfMap[ baseIntfName ][ baseIntfName ] = range( 1, 11 )
         self.hostMap[ baseIntfName ] = range( 1, 11 )
         return

      # Special case for the remaining 10 lane 100G case, which is supported
      # only on CFP2 SR10
      if( xcvrType == XcvrType.cfp2 and opticalChannels == 10 and
          self.linkModeStatusDict[ baseIntfName ] ==
            LinkMode.linkModeForced100GbpsFull ):
         self.intfMap[ baseIntfName ][ baseIntfName ] = remainingChannels
         self.hostMap[ baseIntfName ] = remainingChannels
         return

      # Lane is a zero-indexed lane number that corresponds to the current interface.
      # For example, when intfName == 'Et1/1', lane == 0
      elecIdx = 0
      for lane, intfName in intfNamesDict.iteritems():
         if intfName not in self.activeIntfs:
            # Could be transient state or inactive; don't allocate lanes
            continue
         linkMode = self.linkModeStatusDict.get( intfName, LinkMode.linkModeUnknown )
         if linkMode == LinkMode.linkModeAutoneg:
            if( self.anegLanes.get( intfName, EthLaneCount.laneCountUnknown ) !=
                EthLaneCount.laneCountUnknown ):
               ethLaneCount = self.anegLanes[ intfName ]
            else:
               anegSpeed = self.anegSpeed.get( intfName, EthSpeed.speedUnknown )
               ethLaneCount = ethTypes.nrzSpeedToEthLaneCount( anegSpeed )
         else:
            ethLaneCount = ethTypes.linkModeToEthLaneCount( linkMode )
         hostLanes = ethTypes.laneCountNumber( ethLaneCount )
         if not hostLanes:
            # lmUnknown
            continue

         if electricalLanes < opticalChannels:
            # We expected electricalLanes >= opticalChannels. In the unlikely
            # event that we get here, we want to prevent the cli from crashing
            # and still try and print relevant data, so we will use the lower
            # value to try and prevent indexing where we shouldn't.
            print( "Unhandled case. Format might not reflect actual configuration." )
            opticalChannels = electricalLanes

         if not opticalChannels or not electricalLanes:
            # Unexpected condition
            continue

         startLane = lane * opticalChannels / electricalLanes
         while remainingChannels and remainingChannels[ 0 ] < startLane:
            # If we aren't starting from /1, then we will want to dump the lanes
            # that we skipped over. We might also need to do this if we've skipped
            # past a lane as a result of the link mode being unknown.
            remainingChannels.pop( 0 )

         # Invariant: At this point, lane should be eligible to use
         # remainingChannels[ 0 ]

         if lane > elecIdx:
            # lane can sometimes jump farther ahead (for example, if a QSFP port
            # only has interfaces Et1/1 and Et1/3 available).
            elecIdx = lane

         opticalChIdxs = []
         # Determine the number of lanes in the group. We want to evenly distribute
         # lanes if opticalChannels != electricalLanes. For example, if we are
         # operating at 50G-2 on a 2 lane QSFP port (lanes = 2, opticalChannels = 2,
         # electricalLanes = 4) then we want to assign /1 to lane 0 and /3 to lane 1.
         # Also, we will always enter this loop, even if the math resolves to less
         # than 0. This could happen if we are printing a single lane speed. In the
         # previous example, set speed to 25G-1 (lanes = 1) and this will occur.
         lanesGroupedFloat = float( hostLanes * opticalChannels ) / electricalLanes
         lanesGrouped = int( lanesGroupedFloat )
         if lanesGrouped != lanesGroupedFloat or lanesGrouped == 0:
            lanesGrouped = hostLanes
         opticalChIdxs += remainingChannels[ : lanesGrouped ]
         remainingChannels = remainingChannels[ lanesGrouped : ]
         if intfName not in self.reqIntfs:
            # We break out here and not above because Alta does not set
            # sub-interfaces inactive ever. This allows /1 to run and allocate
            # remainingChannels, even though it wasn't requested by the user.
            # We then simply disregard the channels it allocated by continuing here.
            continue
         if opticalChIdxs:
            self.intfMap[ intfName ][ intfName ] = opticalChIdxs
            self.hostMap[ intfName ] = range( elecIdx, elecIdx + hostLanes )
         elecIdx += hostLanes
      return

   def _getCoherentIntfMap( self ):
      """
      Updates the mapping for cohorent interfaces (CloudsRest2 and DCO modules).
      """
      for slotName in self.intfNamesDict:
         xcvrMediaType = self.xcvrMediaTypesDict[ slotName ]
         if xcvrMediaType not in ( XcvrMediaType.xcvr100GDwdmDco,
                                   XcvrMediaType.xcvr100GDwdmCoherentE,
                                   XcvrMediaType.xcvr100GDwdmCoherent,
                                   XcvrMediaType.xcvr400GDwdmDco ):
            continue
         intfNames = self.intfNamesDict[ slotName ].values()

         # There is only one set of rxlos/txfault data for the xcvr
         self.portLaneMapOverride[ slotName ][ 'xcvr' ] = [ 0 ]
         for intfName in intfNames:
            self.portLaneMapOverride[ slotName ][ intfName ] = []
         modulation = self.modulationDict.get( slotName + '/1' )
         if xcvrMediaType == XcvrMediaType.xcvr100GDwdmDco:
            if( slotName in self.reqPorts and
                   modulation in [ 'modulation8Qam', 'modulation16Qam' ] ):
               self.intfMap[ slotName + '/1' ] = { slotName + '/1' : [ 0 ],
                                                   slotName + '/2' : [ 1 ] }
            continue
         if xcvrMediaType == XcvrMediaType.xcvr400GDwdmDco:
            if( slotName in self.reqPorts and
                   modulation in [ 'modulation16Qam' ] ):
               self.intfMap[ slotName + '/1' ] = { slotName + '/1' : [ 0 ],
                                                   slotName + '/2' : [ 1 ],
                                                   slotName + '/3' : [ 2 ],
                                                   slotName + '/4' : [ 3 ] }
            continue
         # CR2 in 8QAM groups two physical ports together to send 3 lanes
         # (3 interfaces) of 100G. The groups are interfaces adjacent,
         # such as EtX/1/1, EtX/1/2, and EtX/2/1 sending traffic and EtX/2/2
         # inactive, not sending anything. In the context of populating lane data,
         # we will refer to EtX/1/1 as the primary intf, EtX/1 as the primary
         # port, EtX/2/1 as the sub intf and EtX/2 as the sub port.
         # Only the primary port will contain lane 2 configured to 8qam
         intfName = slotName + '/2'
         if intfName not in intfNames:
            return
         modulation = self.modulationDict.get( intfName )
         if( modulation == 'modulation8Qam' and
              intfName.endswith( '/2' ) ):
            # Get the subsumed port name, we do this by adding 1 to the primary
            # port number (i.e. primary: EtX/1 sub: EtX/(1 + 1) => EtX/2)
            subPort = slotName.split( '/' )
            subPort[ -1 ] = str( int( subPort[ -1 ] ) + 1 )
            subPort = '/'.join( subPort )
            # Create interface names from port names by appending lane id
            subIntfName = '%s/%s' % ( subPort, 1 )
            inactiveSubIntf = '%s/%s' % ( subPort, 2 )
            primaryIntfName = '%s/%s' % ( slotName, 1 )

            if not self.reqIntfs.intersection( [ primaryIntfName, intfName,
                                                 subIntfName ] ):
               if inactiveSubIntf in self.reqIntfs:
                  # To get here we must have requested the inactive sub intf.
                  # As a result, the generic mapping will have been created for the
                  # interfaces of this port, and we want to clear them because we
                  # don't actually want to display them.
                  self.intfMap.update( { subIntfName : {}, inactiveSubIntf : {} } )
               continue

            # Set lane stats for the sub port to be xcvr specific
            self.portLaneMapOverride[ subPort ][ 'xcvr' ] = [ 0 ]
            self.portLaneMapOverride[ subPort ][ subIntfName ] = []
            self.portLaneMapOverride[ subPort ][ inactiveSubIntf ] = []
            # Display subIntf as part of group
            self.portLaneMapOverride[ primaryIntfName ][ subIntfName ] = [ 0 ]
            dwdmMapping = { primaryIntfName : [ 0 ], intfName : [ 0 ],
                            subIntfName : [ 0 ] }
            self.intfMap.update( { primaryIntfName : dwdmMapping,
                              subIntfName : dwdmMapping,
                              intfName : dwdmMapping,
                              inactiveSubIntf : {} } )
            cr2ShortName = primaryIntfName + ',' + intfName \
                              + ',' + subIntfName + ' (Channel 1)'
            cr2ShortName = cr2ShortName.replace( 'Ethernet', 'Et' )
            self.intfNameOverrideXcvrDom[ primaryIntfName ] = cr2ShortName
            self.intfNameOverrideXcvrDom[ intfName ] = cr2ShortName
            self.intfNameOverrideXcvrDom[ subIntfName ] = cr2ShortName
            self.hostMap.update( dwdmMapping )
      return

   def _getAgileIntfMap( self ):
      """
      Updates the mapping for agile interfaces (SFP ports acting as one QSFP port).
      """
      intfsByEpl = defaultdict( list )
      for intfName, eplId in self.eplByIntf.iteritems():
         intfsByEpl[ eplId ].append( intfName )

      for intfName in self.activeIntfs:
         # Only the primary intfs will appear to be operating at 40G
         if self.linkModeStatusDict[ intfName ] == LinkMode.linkModeForced40GbpsFull:
            if self.xcvrTypesDict[ getXcvrSlotName( intfName ) ] == 'qsfpPlus':
               continue
            epl = self.eplByIntf[ intfName ]

            if not self.reqIntfs.intersection( intfsByEpl[ epl ] ):
               continue

            sortedEpl = Arnet.sortIntf( intfsByEpl[ epl ] )
            for laneId, curIntfName in enumerate( sortedEpl ):
               # Display rxlos/txfault data for the xcvr and not mapped to an intf
               self.portLaneMapOverride[ curIntfName ][ 'xcvr' ] = [ 0 ]
               self.portLaneMapOverride[ curIntfName ][ curIntfName ] = []
               # Overriding intfName->curIntfName is how we will prevent displaying
               # lane information grouped together and instead display the interfaces
               # separately (since each is its own port).
               self.portLaneMapOverride[ intfName ][ curIntfName ] = []
               self.intfMap[ curIntfName ] = {}
               self.intfMap[ sortedEpl[ 0 ] ][ curIntfName ] = [ 0 ]
               # The display name for the intf should contain the lane id and will
               # display the primary intf name
               self.intfNameOverride[ curIntfName ] = '%s (Lane %d)' % ( intfName,
                                                                         laneId )
               self.intfNameOverrideXcvrDom[ curIntfName ] = \
                     '%s Lane %d (Channel 1)' % ( intfName, laneId )
               self.hostMap[ curIntfName ] = [ 0 ]
      return

   def _getPolsIntfMap( self ):
      for polsSlot in self.polsSlots:
         for i in range( 8 ):
            laneId = i + 1
            intfId = "%s/%d" % ( polsSlot, laneId )
            if laneId in ( 1, 2 ) and intfId in self.reqIntfs:
               # Mark /1 and /2 as "owning" lane0 and lane1 respectively
               self.intfMap[ intfId ] = { intfId: [ i ] }
            else:
               self.intfMap.pop( intfId, None )

#----------------------------------------------------------------------------
#
# "show interface [ <interface> ] transceiver [detail] [csv]"
#
#----------------------------------------------------------------------------

def _showInterfacesXcvr( mode, intf=None, mod=None, outputFmt=None, detailed=False ):
   res = InterfacesTransceiver( _printFmt=outputFmt, _detailed=detailed,
      _domThresholdOverrideEnabled=xgc.domThresholdOverrideEnabled )
   assert outputFmt in SUPPORTED_FORMATS, "Unrecognized DOM output format"
   ( intfs, intfNames ) = getAllIntfsWrapper( mode, intf, mod )
   if not intfs:
      return res

   for status in xcvrStatusDir_.xcvrStatus.values():
      status = getXcvrStatus( status )
      if not status.xcvrMgrCapabilities.domDynamicValues:
         continue
      assert status.xcvrConfig.intfName
      lineSideChannels = getLineSideChannels( status )
      for laneId in range( lineSideChannels ):
         printName = getIntfNameByLane( laneId, status, lineSideChannels, intfNames )
         if not printName:
            continue

         if not status.domInfoValid[ laneId ]:
            res.interfaces[ printName ] = InterfaceTransceiverBase()
         else:
            res.interfaces[ printName ] = _xcvrDomInfo( status,
                                                        status.domInfo[ laneId ] )
            if detailed:
               intfXcvr = res.interfaces[ printName ]
               intfXcvr.details = _xcvrDomDetails( status )
               # pylint: disable-msg=W0212
               intfXcvr.details._domThresholdOverridden = \
                  status.domThresholdOverridden

   return res

def getXcvrChannel( status, channel ):
   """
   Helper method for reformatting appropriate channel to be printed.
   """
   if status.isCoherent( status.mediaType ):
      channel = '-'
   return channel

def getXcvrAddThreshold( status, paramType, addThreshold ):
   if _isDualLaserModule( status ):
      if paramType == Tac.Type( "Xcvr::EnhancedDomParamType" ).txPower:
         addThreshold = False
      elif paramType == Tac.Type( "Xcvr::EnhancedDomParamType" ).rxPower:
         addThreshold = False
   return addThreshold

def getXcvrParamType( status, paramType ):
   """
   Helper method for reformatting parameter names to be used in
   paramTypeToStr to get exact output format
   """

   domParamType = Tac.Type( "Xcvr::EnhancedDomParamType" )

   # Convert some params common for all coherent systems
   if status.isCoherent( status.mediaType ):
      paramTypeToLabelCoherent = {
         domParamType.temperature : 'caseTemperature',
         domParamType.txPower : 'coherentTxPower',
         domParamType.laserTemp : 'laserTemperature',
         domParamType.rxSignalPower : 'rxChannelPower',
         domParamType.rxTotalPower : 'totalRxPower',
         domParamType.preFecBERAvg : 'preFECBERAvg',
         domParamType.preFecBERMin : 'preFECBERMin',
         domParamType.preFecBERMax : 'preFECBERMax',
      }
      if paramType in paramTypeToLabelCoherent:
         return paramTypeToLabelCoherent[ paramType ]

   # Convert some params of dual laser systems
   if _isDualLaserModule( status ):
      paramTypeToLabelDualLaser = {
         domParamType.rxPower : 'rxChannelPower',
         domParamType.laserAge : 'txLaserAge',
      }
      if paramType in paramTypeToLabelDualLaser:
         return paramTypeToLabelDualLaser[ paramType ]

   # Default case - no conversion
   return paramType

def _opt( value, defaultValue=None ):
   if ( value is not None and value != float( "-inf" ) ):
      return value
   else:
      return defaultValue

def _xcvrDomInfo( status, domInfo ):
   # The following checks prevent the TX and RX Power from being
   # reported as N/A when the DomInfo is valid.
   domCapabilities = status.domCapabilities

   if ( domInfo.txPower < Tac.Value("Xcvr::PowerConstants").minimumTxPower \
        and domCapabilities.txPower ):
      txPower = Tac.Value("Xcvr::PowerConstants").minimumTxPower
   else:
      txPower = domInfo.txPower

   minimumRxPowerConstant = Tac.Value("Xcvr::PowerConstants").minimumRxPower
   if _isDualLaserModule( status ):
      minimumRxPowerConstant = \
         Tac.Value("Xcvr::PowerConstants").minimumCoherentRxPower

   if ( domInfo.rxPower < minimumRxPowerConstant and
        domCapabilities.rxPower ):
      rxPower = minimumRxPowerConstant
   else:
      rxPower = domInfo.rxPower

   if ( domInfo.totalRxPower < minimumRxPowerConstant and
        domCapabilities.totalRxPower ):
      totalRxPower = minimumRxPowerConstant
   else:
      totalRxPower = domInfo.totalRxPower

   # Used for active copper where type is optical but txBias is not supported
   if not domCapabilities.txBias:
      txBias = float( "-inf" )
   else:
      txBias = domInfo.txBias

   model = InterfaceTransceiver()
   model.vendorSn = status.vendorInfo.vendorSn.strip()
   model.mediaType = status.mediaTypeString.strip()
   model.narrowBand = status.narrowBand
   model.updateTime = Tac.utcNow() - ( Tac.now() - domInfo.updateTime )
   model.temperature = _opt( domInfo.temperature )
   model.voltage = _opt( domInfo.voltage )
   model.txBias = _opt( txBias )
   model.txPower = _opt( txPower )
   model.rxPower = _opt( rxPower )
   model.totalRxPower = _opt( totalRxPower )
   return model

def _getDetailThresholdModel( domThreshold, attr ):
   highAlarm = _opt( getattr( domThreshold, attr + "HighAlarm" ) )
   highWarn = _opt( getattr( domThreshold, attr + "HighWarn" ) )
   lowAlarm = _opt( getattr( domThreshold, attr + "LowAlarm" ) )
   lowWarn = _opt( getattr( domThreshold, attr + "LowWarn" ) )

   return InterfaceTransceiverDetailThresholds( highAlarm=highAlarm,
                                                highWarn=highWarn, lowAlarm=lowAlarm,
                                                lowWarn=lowWarn )

def _getEnhancedThresholdModel( domThreshold, attr ):

   highAlarm = getattr( domThreshold, attr + "HighAlarm", None )
   highWarn = getattr( domThreshold, attr + "HighWarn", None )
   lowAlarm = getattr( domThreshold, attr + "LowAlarm", None )
   lowWarn = getattr( domThreshold, attr + "LowWarn", None )
   # check that all attributes are available
   if None not in ( highAlarm, highWarn, lowAlarm, lowWarn ):
      return InterfaceTransceiverDetailThresholds( highAlarm=_opt( highAlarm ),
                                                   highWarn=_opt( highWarn ),
                                                   lowAlarm=_opt( lowAlarm ),
                                                   lowWarn=_opt( lowWarn ) )
   else:
      return None

def _xcvrDomDetails( status ):
   model = InterfaceTransceiverDetails()

   if not status.domThresholdValid:
      return model

   domThreshold = status.domThreshold
   domCapabilities = status.domCapabilities

   model.temperature = _getDetailThresholdModel( domThreshold, "temp" )
   model.voltage = _getDetailThresholdModel( domThreshold, "voltage" )
   if domCapabilities.txBias:
      model.txBias = _getDetailThresholdModel( domThreshold, "txBias" )
   model.txPower = _getDetailThresholdModel( domThreshold, "txPower" )
   model.rxPower = _getDetailThresholdModel( domThreshold, "rxPower" )
   model.totalRxPower = _getDetailThresholdModel( domThreshold, "totalRxPower" )
   return model


def showInterfacesXcvrDefault( mode, args ):
   intf = args.get( 'INTF' )
   mod = args.get( 'MOD' )
   outputFmt = args.get( 'csv', 'default' )
   detailed = 'detail' in args
   return _showInterfacesXcvr( mode, intf, mod, outputFmt=outputFmt,
                               detailed=detailed )

class ShowIntfXcvr( ShowIntfCommand ):
   syntax = 'show interfaces transceiver [ csv | detail ]'
   data = dict( transceiver=xcvrShowKw,
                csv='Show interface transceiver in CSV format',
                detail='Show interface transceiver detail' )
   handler = showInterfacesXcvrDefault
   cliModel = InterfacesTransceiver

BasicCli.addShowCommandClass( ShowIntfXcvr )

#----------------------------------------------------------------------------
#
# "show interfaces [ <interface> ] transceiver dom [ thresholds ]"
#
#----------------------------------------------------------------------------
def _showInterfacesXcvrDom( mode, intf=None,
      mod=None, outputFmt=None, thresholds=False ):
   res = InterfacesTransceiverDom( _printFmt=outputFmt, _detailed=thresholds )
   assert outputFmt in SUPPORTED_FORMATS, "Unrecognized DOM output format"
   ( intfs, _ ) = getAllIntfsWrapper( mode, intf, mod )

   if not intfs:
      return res

   # Get the laneMappings for all the active interfaces
   intfMapping = IntfMapping( intfs )
   intfLaneMapping = intfMapping.intfMap
   if not intfLaneMapping:
      return res

   # Get the overriden display Names of the interfaces
   intfNameOverrideXcvrDom = intfMapping.intfNameOverrideXcvrDom

   if not intfLaneMapping:
      return res

   for intfToDisplay in Arnet.sortIntf( intfLaneMapping ):
      for intfName in Arnet.sortIntf( intfLaneMapping[ intfToDisplay ] ):
         channels = intfLaneMapping[ intfToDisplay ][ intfName ]
         xS = xcvrStatusDir_.xcvrStatus.get( getXcvrSlotName( intfName ) )
         xS = getXcvrStatus( xS )

         if not xS.xcvrMgrCapabilities.domDynamicValues:
            continue

         if xS.presence != "xcvrPresent":
            # Skip transceivers that aren't present
            continue

         if intfName in res.interfaces.keys() or not channels:
            continue

         laneId = getLaneId( intfName )
         laneId = laneId - 1 if laneId else laneId

         domModel = _getIntfXcvrDomInfo( xS, laneId,
                                         channels, thresholds )
         domModel.displayName = intfNameOverrideXcvrDom.get( intfName ) or intfName
         res.interfaces[ intfName ] = domModel
         # pylint:disable-msg=protected-access
         res._interfacesOrder.append( intfName )
   return res

# Populate and return the DOM model for the given xcvrStatus.
def _getIntfXcvrDomInfo( status, laneId, channels, thresholds ):
   # Populate N/A for all params on interfaces with invalid domInfo.
   xcvrDomModel = InterfaceTransceiverDomBase()
   if status.domInfoValid[ laneId ]:
      # Populate the legacy DOM parameters such as temperature, voltage etc.
      xcvrDomModel = _xcvrCommonDomInfo( status, status.domInfo[ laneId ],
                                         laneId, channels, thresholds )
      # Append enhanced dom parameters if they are present ( 100GE-DWDM2 )
      if status.mediaType == XcvrMediaType.xcvr100GEDwdm2:
         return _xcvrEnhancedDomInfo( status, status.domInfo[ laneId ],
                                      laneId, thresholds, xcvrDomModel )

      # Append enhanced dom parameters if they are present ( 400GBASE-ZR )
      elif status.mediaType == XcvrMediaType.xcvr400GBaseZr:
         return _xcvrCmisEnhancedDomInfo( status, laneId, thresholds, xcvrDomModel )

      # Append DCO DOM parameters.
      elif ( status.mediaType in ( XcvrMediaType.xcvr100GDwdmDco,
                                   XcvrMediaType.xcvr400GDwdmDco ) ):
         return _xcvrCfp2DcoDomInfo( status, xcvrDomModel, thresholds )
   return xcvrDomModel

# Helper function to insert a parameter in a dict
# If parameter exists, we are probably adding value for another channel
# If parameter doesn't exist, create one.
def insertParameterInDict( parameters, paramType, unit, channel, value,
      threshold=None ):
   if paramType in parameters.keys():
      parameters[ paramType ].channels[ channel ] = value
   else:
      newParam = InterfaceTransceiverDomParameter()
      newParam.unit = unit
      newParam.channels[ channel ] = value
      if threshold is not None:
         newParam.threshold = threshold
      parameters[ paramType ] = newParam
   return ( parameters, len( parameters[ paramType ].channels ) > 1 )

def _addTotalRxPowerToModel( status, newParam, isMultiChannel, paramList ):
   if _isDualLaserModule( status ):
      paramType = 'totalRxPower'
      paramList.append( paramType )
      paramChannel = '-'
      paramUnit = 'dBm'
      domInfo = status.domInfo[ 0 ]
      minimumRxPowerConstant = \
         Tac.Value("Xcvr::PowerConstants").minimumCoherentRxPower
      if ( domInfo.totalRxPower < minimumRxPowerConstant and
           status.domCapabilities.totalRxPower ):
         totalRxPower = minimumRxPowerConstant
      else:
         totalRxPower = domInfo.totalRxPower
      _threshold = _getEnhancedThresholdModel( status.domThreshold, paramType )
      ( newParam, isMultiChannel ) = insertParameterInDict( newParam, paramType,
                                                            paramUnit, paramChannel,
                                                            totalRxPower,
                                                            threshold=_threshold )
   return ( newParam, isMultiChannel, paramList )

# Create and populate legacy DOM parameters such as temperature, voltage etc...
# and return the model.
def _xcvrCommonDomInfo( status, domInfo, laneId, channels, thresholds ):
   model = InterfaceTransceiverDom(
      _dualLaserModulePresent=_isDualLaserModule( status ) )
   model.vendorSn = status.vendorInfo.vendorSn.strip()
   model.mediaType = status.mediaTypeString.strip()
   polsModule = status.mediaType == XcvrMediaType.xcvrAmpZr
   model.updateTime = Tac.utcNow() - ( Tac.now() - domInfo.updateTime )
   domCapabilities = status.domCapabilities
   domThreshold = status.domThreshold
   _threshold = None
   addThreshold = thresholds and status.domThresholdValid
   paramList = []
   newParam = {}
   # True when at least one dom parameter exists for multiple channels.
   isMultiChannel = False
   # Insert temperature in the model
   paramType = getXcvrParamType( status, 'temperature' )
   addThreshold = getXcvrAddThreshold( status, 'temperature', addThreshold )
   paramUnit = 'C'
   paramValue = _opt( domInfo.temperature, float( "-inf" ) )
   paramChannel = '-'
   paramList.append( paramType )
   if addThreshold:
      _threshold = _getEnhancedThresholdModel( domThreshold, "temp" )
   ( newParam, isMultiChannel ) = \
         insertParameterInDict( newParam, paramType, paramUnit,
                                paramChannel, paramValue, _threshold )

   # Insert voltage in the model
   paramType = 'voltage'
   paramUnit = 'V'
   paramValue = _opt( domInfo.voltage, float( "-inf" ) )
   paramChannel = "-"
   paramList.append( paramType )
   if addThreshold:
      _threshold = _getEnhancedThresholdModel( domThreshold, "voltage" )
   ( newParam, isMultiChannel ) = \
         insertParameterInDict( newParam, paramType, paramUnit,
                                paramChannel, paramValue, _threshold )

   if domCapabilities.aggTxPower:
      paramType = 'aggrTxPwr'
      paramUnit = 'dBm'
      paramValue = status.domAggPower.aggTxPower
      paramChannel = "-"
      paramList.append( paramType )
      ( newParam, _ ) = \
            insertParameterInDict( newParam, paramType, paramUnit,
                                   paramChannel, paramValue, None )

   if domCapabilities.aggRxPower:
      paramType = 'aggrRxPwr'
      paramUnit = 'dBm'
      paramValue = status.domAggPower.aggRxPower
      paramChannel = "-"
      paramList.append( paramType )
      ( newParam, _ ) = \
            insertParameterInDict( newParam, paramType, paramUnit,
                                   paramChannel, paramValue, None )

   for chan in channels:
      # channels passed in are indexed starting from 0...
      ch = chan + 1
      paramChannel = getXcvrChannel( status, str( ch ) )
      channelThreshold = domThreshold
      if polsModule and ch == 2:
         # POLS modules have alternative thresholds for the splitter channel
         channelThreshold = status.eepromContents.altDomThreshold

      # Insert Tx-Bias in the model
      paramType = getXcvrParamType( status, 'txBias' )
      paramUnit = 'mA'
      if not status.isCoherent( status.mediaType ):
         # Add paramType to our list of parameters only if it doesn't already exist.
         if paramType not in paramList:
            paramList.append( paramType )
         paramValue = _opt( status.domInfo[ chan ].txBias, float( "-inf" ) )
         if not domCapabilities.txBias:
            paramValue = float( "-inf" )
         _threshold = None
         if addThreshold:
            if domCapabilities.txBias:
               _threshold = _getEnhancedThresholdModel( channelThreshold, "txBias" )
            else:
               _threshold = InterfaceTransceiverDetailThresholds( highAlarm=None,
                                                   highWarn=None, lowAlarm=None,
                                                   lowWarn=None )
         ( newParam, isMultiChannel ) = \
               insertParameterInDict( newParam, paramType, paramUnit,
                                      paramChannel, paramValue, _threshold )
      # Insert Tx Power in the model
      addThreshold = getXcvrAddThreshold( status, 'txPower', addThreshold )
      paramType = getXcvrParamType( status, 'txPower' )
      paramUnit = 'dBm'
      # Add paramType to our list of parameters only if it doesn't already exist.
      if paramType not in paramList:
         paramList.append( paramType )
      paramValue = status.domInfo[ chan ].txPower
      if ( ( paramValue == float( "-inf" ) or
             paramValue < Tac.Value("Xcvr::PowerConstants").minimumTxPower ) and
            domCapabilities.txPower ):
         paramValue = Tac.Value("Xcvr::PowerConstants").minimumTxPower
      _threshold = None
      if addThreshold:
         _threshold = _getEnhancedThresholdModel( channelThreshold, paramType )
      ( newParam, isMultiChannel ) = \
            insertParameterInDict( newParam, paramType, paramUnit,
                                   paramChannel, paramValue, _threshold )

      # Insert Rx total power in the model if applicable
      ( newParam, isMultiChannel, paramList ) = \
         _addTotalRxPowerToModel( status, newParam, isMultiChannel, paramList )

      # Insert Rx Power in the model
      addThreshold = getXcvrAddThreshold( status, 'rxPower', addThreshold )
      paramType = getXcvrParamType( status, 'rxPower' )
      paramUnit = 'dBm'

      if not status.isCoherent( status.mediaType ) or _isDualLaserModule( status ):
         # Add paramType to our list of parameters only if it doesn't already exist.
         if paramType not in paramList:
            paramList.append( paramType )

         paramValue = status.domInfo[ chan ].rxPower
         minimumRxPowerConstant = Tac.Value( "Xcvr::PowerConstants" ).minimumRxPower
         if _isDualLaserModule( status ):
            minimumRxPowerConstant = \
                           Tac.Value( "Xcvr::PowerConstants" ).minimumCoherentRxPower
         if ( ( paramValue == float( "-inf" ) or
                paramValue < minimumRxPowerConstant ) and
              domCapabilities.rxPower ):
            paramValue = minimumRxPowerConstant
         _threshold = None
         if addThreshold:
            _threshold = _getEnhancedThresholdModel( channelThreshold, paramType )
         ( newParam, isMultiChannel ) = \
               insertParameterInDict( newParam, paramType, paramUnit,
                                     paramChannel, paramValue, _threshold )

      # If POLS, include Aux2/3 as laser temp for booster/pre-amp respectively
      if polsModule and chan in range( 2 ):
         paramType = 'laserTemperature'
         if paramType not in paramList:
            paramList.append( paramType )
         paramUnit = 'C'
         paramName = { 0: "aux2", 1: "aux3" }[ chan ]
         paramValue = getattr( status.auxInfo, "%sVal" % paramName )
         if addThreshold:
            _threshold = _getEnhancedThresholdModel( status.auxThreshold, paramName )
         ( newParam, isMultiChannel ) = \
            insertParameterInDict( newParam, paramType, paramUnit,
                                   paramChannel, paramValue, _threshold )

   model.parameters = newParam
   # pylint: disable-msg=protected-access
   model._paramOrder = paramList
   model._hasMultiChannelDomParam = isMultiChannel
   model._usePolsLabels = polsModule

   return model

# Appends the Cmis Enhanced DOM parameters to the xcvr DOM model that is passed in.
def _xcvrCmisEnhancedDomInfo( status, laneId, thresholds, model ):
   paramList = []
   newParam = {}
   # True when at least one dom parameter exists for multiple channels.
   isMultiChannel = False
   _threshold = None
   scope = Tac.Type( 'Xcvr::EnhancedDomParamScope' )
   if status.enhancedDom:
      for idx, extDomConfig in status.enhancedDom.domConfig.items():
         extDomInfo = status.enhancedDom.domInfo[ idx ]
         if ( extDomInfo.valid and
              extDomConfig.lane == laneId and
              extDomConfig.scope != scope.enhancedDomScopeHost and
              not extDomInfo.perfMon ):
            paramType = getXcvrParamType( status, extDomConfig.paramType )
            # Check if this paramType is not already in our list. Well, this is a
            # little inefficient if the list is long but the sample space
            # is quite small here (in tens) so it doesn't hurt much to do a
            # lookup everytime.
            if paramType not in paramList:
               paramList.append( paramType )
            paramUnit = extDomConfig.paramUnit
            paramValue = extDomInfo.paramValue
            # 400G-ZR shows all media parameters as 'globals'
            paramChannel = str( extDomConfig.lane ) if extDomConfig.scope not in \
                                          [ scope.enhancedDomScopeModule,
                                            scope.enhancedDomScopeMedia ] else '-'
            ( newParam, isMultiChannel ) = \
                           insertParameterInDict( newParam, paramType, paramUnit,
                                          paramChannel, paramValue, _threshold )

   model.parameters.update( newParam )
   # pylint:disable-msg=protected-access
   model._paramOrder.extend( paramList )
   model._hasMultiChannelDomParam = isMultiChannel

   return model

# Appends the DCO DOM parameters to the xcvr DOM model that is passed in.
def _xcvrCfp2DcoDomInfo( status, model, thresholds ):
   cfp2DcoStatus = status.cfp2DcoNewStatus
   paramList = []
   newParam = {}
   # True when at least one dom parameter exists for multiple channels.
   isMultiChannel = False
   isDp04 = ( cfp2DcoStatus.dcoType ==
              Tac.Type( "Xcvr::Cfp2DcoType" ).cfp2DcoTypeDp04 )
   addThreshold = thresholds and status.domThresholdValid and isDp04
   domThreshold = cfp2DcoStatus.dcoDomThreshold

   def addToModel( model, status, attrKey, valueDict, paramList, newParam,
                   isMultiChannel, threshold ):
      isDp04Attr, attr = attrKey
      if isDp04 and isDp04Attr:
         # pylint: disable-msg=protected-access
         if attr in model._paramOrder:
            # Simply update parameter ordering with attribute populated in
            # _xcvrCommonDomInfo
            model._paramOrder.remove( attr )
            paramList.append( attr )
            return ( newParam, isMultiChannel )
      if not isDp04 and isDp04Attr:
         return ( newParam, isMultiChannel )
      if attr not in cfp2DcoStatus.attributes:
         assert "Attribute {} is not a member of Cfp2DcoStatus".format( attr )

      paramType = getXcvrParamType( status, valueDict[ attrKey ][ 0 ] )
      paramList.append( paramType )
      paramUnit = valueDict[ attrKey ][ 1 ]
      paramValue = float( getattr( cfp2DcoStatus, attr ) )
      paramChannel = '-'
      return insertParameterInDict( newParam, paramType, paramUnit,
            paramChannel, paramValue, threshold )

   domAttr = OrderedDict( [
         ( ( False, 'networkBer' ), ( 'preFecBERCurr', '', None ) ),
         ( ( False, 'netFecUncBlkCount' ), ( 'netFecUncBlkCount', '', None ) ),
         ( ( False, 'netOsnrEstimate' ), ( 'netOsnrEstimate', 'dB', None ) ),
         ( ( False, 'rxDiffGroupDelay' ), ( 'rxDiffGroupDelay', 'ps', None ) ),
         ( ( False, 'laserAge' ), ( 'laserAge', '%', None ) ),
         ( ( True, 'rxLaserAge' ), ( 'rxLaserAge', '%', None ) ),
         ( ( False, 'rxSnr' ), ( 'snr', 'dB', None ) ),
         ( ( False, 'rxStateOfPolarization' ), ( 'rxStateOfPolarization', 'rad/sec',
                                                 None ) ),
         ( ( True , 'rxNetworkPDL' ), ( 'rxNetworkPDL', 'dB', None ) ),
         ( ( False, 'rxQFactor' ), ( 'rxQFactor', 'dB', None ) ),
         ( ( False, 'rxChromDispersion' ), ( 'rxChromDispersion', 'ps/nm', None ) ),
         ( ( False, 'rxCarrierFreqOffset' ), ( 'rxCarrierFreqOffset', 'MHz',
                                               None ) ),
         ( ( False, 'rollOff' ), ( 'rollOff', '', None ) ),
         # These are commonDomParams being populated elsewhere, but mut be specified
         # here for proper ordering in output. Indicate this behavior by setting the
         # desired parameter type below to None.
         ( ( True, 'coherentTxPower' ), ( None, None, None ) ),
         ( ( True, 'totalRxPower' ), ( None, None, None ) ),
         ( ( True, 'rxChannelPower' ), ( None, None, None ) ),
         # END commonDomParams
         ( ( True, 'txLaserPower' ), ( 'txLaserPower', 'dBm', 'txLaserPower' ) ),
         ( ( True, 'rxLaserPower' ), ( 'rxLaserPower', 'dBm', 'rxLaserPower' ) ),
         ( ( True, 'txLaserTemperature' ), ( 'txLaserTemperature', 'C',
                                             'txLaserTemp' ) ),
         ( ( True, 'rxLaserTemperature' ), ( 'rxLaserTemperature', 'C',
                                             'rxLaserTemp' ) ),
   ] )
   for attr, ( _, _, thresholdAttr ) in domAttr.items():
      _threshold = None
      if thresholdAttr and addThreshold:
         _threshold = _getEnhancedThresholdModel( domThreshold, thresholdAttr )
      ( newParam, isMultiChannel ) = addToModel( model, status, attr, domAttr,
                                                 paramList, newParam, isMultiChannel,
                                                 _threshold )
   # pylint: disable-msg=protected-access
   model.parameters.update( newParam )
   model._paramOrder.extend( paramList )
   model._hasMultiChannelDomParam = isMultiChannel

   return model

# Appends the Enhanced DOM parameters to the xcvr DOM model that is passed in.
def _xcvrEnhancedDomInfo( status, domInfo, laneId, thresholds, model ):
   paramList = []
   newParam = {}
   # True when at least one dom parameter exists for multiple channels.
   isMultiChannel = False
   _threshold = None
   addThreshold = thresholds and status.domThresholdValid

   if status.isEnhancedDomSupported:
      eMC = Tac.Value( 'Xcvr::Sff8436EnhancedMonitoringContents' )
      totalParameters = eMC.totalParameters
      for idx in range( 1, totalParameters + 1 ):
         extDom = status.enhancedDomInfo[ idx ]
         if extDom.enhancedDomInfoValid and not extDom.isPM:
            paramType = extDom.paramType
            # Check if this paramType is not already in our list. Well, this is a
            # little inefficient if the list is long but the sample space
            # is quite small here (in tens) so it doesn't hurt much to do a
            # lookup everytime.
            if paramType not in paramList:
               paramList.append( paramType )
            paramUnit = extDom.paramUnit
            paramValue = extDom.monitoredParameter
            paramChannel = str( extDom.channel ) if extDom.channelSpecific else '-'
            if addThreshold:
               pt = Tac.Type( "Xcvr::EnhancedDomParamType" )
               # These two parameters don't have low alarm and low warn thresholds
               if paramType == pt.preFecBERCurr or \
                     paramType == pt.uncorrectedBERCurr:
                  _threshold = InterfaceTransceiverDetailThresholds(
                        highAlarm=extDom.highAlarm, highWarn=extDom.highWarn )
               else:
                  _threshold = InterfaceTransceiverDetailThresholds(
                        highAlarm=extDom.highAlarm, highWarn=extDom.highWarn,
                        lowWarn=extDom.lowWarn, lowAlarm=extDom.lowAlarm )
            ( newParam, isMultiChannel ) = \
                  insertParameterInDict( newParam, paramType, paramUnit,
                                         paramChannel, paramValue, _threshold )
   model.parameters.update( newParam )
   model._paramOrder.extend( paramList ) # pylint:disable-msg=protected-access
   # pylint: disable-msg=protected-access
   model._hasMultiChannelDomParam = isMultiChannel

   return model

def showInterfacesXcvrDom( mode, args ):
   intf = args.get( 'INTF' )
   mod = args.get( 'MOD' )
   thresholds = 'thresholds' in args
   return _showInterfacesXcvrDom( mode, intf, mod, outputFmt='default',
                                  thresholds=thresholds )

class ShowIntfXcvrDom( ShowIntfCommand ):
   syntax = 'show interfaces transceiver dom [ thresholds ]'
   data = dict( transceiver=xcvrShowKw,
                dom=( 'Show transceiver Digital Optical Monitoring (DOM) parameters '
                      'for the interface' ),
                thresholds=( 'Show transceiver Digital Optical Monitoring (DOM) '
                             'parameters with thresholds for the interface' ) )
   handler = showInterfacesXcvrDom
   cliModel = InterfacesTransceiverDom

BasicCli.addShowCommandClass( ShowIntfXcvrDom )

#--------------------------------------------------------------------------------
#
# Models for "show int [ <interface> ] trans performance-monitoring [thresholds]"
#
#--------------------------------------------------------------------------------

def _showInterfacesXcvrPerformanceMonitoring( mode, intf=None,
      mod=None, thresholds=False ):
   res = InterfacesTransceiverPerformanceMonitoring( _detailed=thresholds )
   ( intfs, intfNames ) = getAllIntfsWrapper( mode, intf, mod )
   if not intfs:
      return InterfacesTransceiverPerformanceMonitoring()

   isPerformanceMonitoringConfigured = False

   for status in xcvrStatusDir_.xcvrStatus.itervalues():
      if not _isPerformanceMonitoringSupported( status ):
         continue

      if status.xcvrType in ( 'qsfpDd', 'osfp' ) and status.adapterPresent:
         status = status.qsfpStatus

      name = status.xcvrConfig.intfName.get( 0 )
      if name not in intfNames:
         continue

      if not isPerformanceMonitoringConfigured and \
            status.performanceMonitoringConfigured:
         isPerformanceMonitoringConfigured = True
         # At this point, PM is configured on this interface.
         # Store the PM period in the model irrespective of whether PM is
         # enabled or not.
         res.performanceMonitoringPeriodSeconds = \
               xgc.performanceMonitoringPeriod.valueInSeconds

      if not status.performanceMonitoringEnabled:
         continue

      res.interfaces[ name ] = _xcvrPerformanceMonitoring( status, thresholds )

   # pylint: disable-msg=W0212
   res._performanceMonitoringConfigured = isPerformanceMonitoringConfigured
   return res

def _getParameters( period, thresholds ):
   newParam = {}
   paramList = []
   _threshold = None
   pm = Tac.Value( 'Xcvr::Sff8436PerformanceMonitoring' )
   for key in sorted( period.parameter.keys() ):
      thresh = period.parameter[ key ]
      paramType = key
      # Check if this paramType is not already in our list. Well, this is a little
      # inefficient if the list is long but the sample space is quite small
      # here (in tens) so it doesn't hurt much to do a lookup everytime.
      if key not in paramList:
         paramList.append( key )
      paramUnit = ''
      # If the parameter is a BER and it's value is the max possible value for this
      # parameter (which is the initialized value), use value 0 instead.
      paramValue = period.parameter[ key ].value
      if( ( paramType in berParametersList ) and ( paramValue == pm.berMaxValue ) ):
         paramValue = 0.0
      paramChannel = \
            str( period.parameter[ key ].channel ) \
            if period.parameter[ key ].channelSpecific else '-'
      if thresholds:
         # Exceeded counts don't have any thresholds, other BER counts only have
         # high alarm and high warn thresholds.
         if paramType in exceededCountersList:
            _threshold = InterfaceTransceiverDetailThresholds()
         else:
            _threshold = InterfaceTransceiverDetailThresholds(
                  highAlarm=thresh.highAlarm, highWarn=thresh.highWarn )
      ( newParam, _ )  = \
            insertParameterInDict( newParam, paramType,
                                   paramUnit, paramChannel, paramValue, _threshold )

   return newParam, paramList

def _getPerformanceMonitoringInterval( startTime, updateTime, params ):
   interval = InterfaceTransceiverPerformanceMonitoringInterval()
   interval.intervalStartTime = \
         Tac.utcNow() - ( Tac.now() - startTime )
   interval.updateTime = \
      Tac.utcNow() - ( Tac.now() - updateTime )
   interval.parameters = params
   return interval

def _xcvrPerformanceMonitoring( status, thresholds ):
   model = InterfaceTransceiverPerformanceMonitoring()

   intervals = {}
   newParamCurrent = {}
   newParamPrevious = {}
   paramList = []
   currPeriod = status.performanceMonitoring.currentPeriod
   prevPeriod = status.performanceMonitoring.previousPeriod
   if currPeriod.isValid:
      newParamCurrent, paramList = _getParameters( currPeriod, thresholds )
      intervals[ 0 ] = _getPerformanceMonitoringInterval( currPeriod.startTime,
            currPeriod.updateTime, newParamCurrent )

   model._paramOrder = paramList       # pylint: disable-msg=W0212

   if prevPeriod.isValid:
      newParamPrevious, paramList = _getParameters( prevPeriod, thresholds )
      intervals[ 1 ] = _getPerformanceMonitoringInterval( prevPeriod.startTime,
            prevPeriod.updateTime, newParamPrevious )

   model.performanceMonitoringIntervals = intervals
   return model

def showInterfacesXcvrPerformanceMonitoring( mode, args ):
   intf = args.get( 'INTF' )
   mod = args.get( 'MOD' )
   thresholds = 'thresholds' in args
   return _showInterfacesXcvrPerformanceMonitoring( mode, intf, mod,
                                                    thresholds=thresholds )

class ShowIntfXcvrPerf( ShowIntfCommand ):
   syntax = 'show interfaces transceiver performance-monitoring [ thresholds ]'
   data = { 'transceiver' : xcvrShowKw,
            'performance-monitoring' :
            'Show performance-monitoring parameters for the interface',
            'thresholds' : ( 'Show performance-monitoring parameters with '
                             'thresholds for the interface' )
   }
   handler = showInterfacesXcvrPerformanceMonitoring
   cliModel = InterfacesTransceiverPerformanceMonitoring

BasicCli.addShowCommandClass( ShowIntfXcvrPerf )

#----------------------------------------------------------------------------
#
# "show interface [ <interface> ] transceiver tuning [ detail ]"
#
#----------------------------------------------------------------------------

def _printTuningHeaderDefault():
   headerFormat = "%-11s%-14s%-14s%-14s%-14s"
   print( "Transceiver Tuning" )
   print( "* = Unsupported by module, value may be inaccurate" )
   print( headerFormat % ( "Port", "Amplitude", "Pre-emphasis", "Post-emphasis",
                          "Equalization" ) )
   print( headerFormat % ( "-" * 7, "-" * 13, "-" * 13, "-" * 13, "-" * 13 ) )

def _getTuningSupported( status ):
   if status.xcvrType in [ 'osfp', 'qsfpDd', 'qsfpCmis' ]:
      tuningCap = status.cmisTuningParamCapabilities.tuningParamImplemented
      txAdaEquSupp = tuningCap.txInputAdaptiveEqualizationImpl
      txEquSupp = tuningCap.txInputFixedEqualizationImpl
      rxPostEmpSupp = tuningCap.rxOutputPostEmphasisImpl
      rxPreEmpSupp = tuningCap.rxOutputPreEmphasisImpl
      rxAmpSupp = tuningCap.rxOutputAmplitudeImpl
   else:
      tuningCap = status.xcvrTuningParamCapabilities
      txAdaEquSupp = False
      txEquSupp = tuningCap.txInputEqualizationSupported
      rxPostEmpSupp = False
      rxPreEmpSupp = tuningCap.rxPreEmphasisSupported
      rxAmpSupp = tuningCap.rxAmplitudeSupported
   return txAdaEquSupp, txEquSupp, rxPostEmpSupp, rxPreEmpSupp, rxAmpSupp

def _getTuningOverride( status ):
   primaryLaneId = 0
   assert status.xcvrConfig.intfName
   primaryIntfName = status.xcvrConfig.intfName[ primaryLaneId ]
   xcvrConfigCliDir = getXcvrConfigCliDir( primaryIntfName )
   xcvrConfigCli = xcvrConfigCliDir.xcvrConfigCli.get( primaryIntfName )
   overrideEn = xcvrConfigCli and xcvrConfigCli.overrideTuningParams
   return overrideEn

def _showInterfacesXcvrTuning( mode, intf=None, mod=None ):
   ( intfs, intfNames ) = getAllIntfsWrapper( mode, intf, mod )
   if not intfs:
      return
   xcvrStatus = xcvrStatusDir_.xcvrStatus

   fmt = "%-11s%-14s%-14s%-14s%-14s"
   tuningHeaderPrinted = False

   for xcvrName in Arnet.sortIntf( xcvrStatus ):
      status = getXcvrStatus( xcvrStatus[ xcvrName ] )

      if status.presence != "xcvrPresent":
         # Skip transceivers that aren't present
         continue

      txAdaEquSupp, txEquSupp, rxPostEmpSupp, rxPreEmpSupp, rxAmpSupp = \
         _getTuningSupported( status )
      overrideEn = _getTuningOverride( status )

      if not any( [ txAdaEquSupp, txEquSupp, rxPostEmpSupp, rxPreEmpSupp, rxAmpSupp,
                    overrideEn ] ):
         continue

      maxChannels = status.capabilities.maxChannels
      for laneId in range( maxChannels ):
         name = getIntfNameByLane( laneId, status, maxChannels, intfNames )
         if not name:
            continue

         # Generate the output string for each tuning parameter
         # 1. 'N/A' is used only when the tuning param is unsupported by module.
         # 2. '*' is appended only when the tuning param is unsupported by module.
         if status.xcvrType.lower() == "cfp2":
            rxAmpStr = 'N/A'
         else:
            rxAmpStr = str( status.rxOutputAmplitude[ laneId ] ) + \
                       ( '' if rxAmpSupp else '*' )
         rxPreEmpStr = str( status.rxOutputPreEmphasis[ laneId ] ) + \
                       ( '' if rxPreEmpSupp else '*' )
         if status.xcvrType in [ 'osfp', 'qsfpDd', 'qsfpCmis' ]:
            rxPostEmpStr = str( status.rxOutputPostEmphasis[ laneId ] ) + \
                           ( '' if rxPostEmpSupp else '*' )
         else:
            rxPostEmpStr = 'N/A'
         if ( status.xcvrType in [ 'osfp', 'qsfpDd', 'qsfpCmis' ] and
              status.txInputAdaptiveEqEnable[ laneId ] ):
            txEquStr = 'auto' + ( '' if txAdaEquSupp else '*' )
         else:
            txEquStr = str( status.txInputEqualization[ laneId ] ) + \
                       ( '' if txEquSupp else '*' )

         fieldValues = ( Intf.getShortname( name ),
                         rxAmpStr, rxPreEmpStr, rxPostEmpStr, txEquStr )
         if not tuningHeaderPrinted:
            _printTuningHeaderDefault()
            tuningHeaderPrinted = True
         print( fmt % fieldValues )

#----------------------------------------------------------------------------
#
# "show interface [ <interface> ] transceiver tuning detail"
#
#----------------------------------------------------------------------------

def _printTuningParamHeaderDefault( tuningType="" ):
   headerFormat = "%-11s%-18s%-18s%-18s"
   print( tuningType )
   print( headerFormat % ( "Port", "Module Default", "Configured", "Operational" ) )
   print( headerFormat % ( "-" * 7, "-" * 14, "-" * 14, "-" * 14 ) )

def _getCmisTxInputEqStr( txAdaEquSupp, txEquSupp, status, xcvrConfigCli, laneId ):
   starAdaEq = '' if txAdaEquSupp else '*'
   starFixEq = '' if txEquSupp else '*'
   overrideEn = xcvrConfigCli and xcvrConfigCli.overrideTuningParams

   # Module Default Value
   moduleDefExists = status.defaultOsfpTuningParamValue.has_key( laneId )
   if moduleDefExists:
      if status.defaultOsfpTuningParamValue[ laneId ].txInputAdaptiveEqEnable:
         moduleStr = 'auto' + starAdaEq
      else:
         moduleStr = str( status.defaultOsfpTuningParamValue[
                             laneId ].txInputEqualization ) + starFixEq
   else:
      moduleStr = "N/A"

   slotDefExists = status.xcvrConfig.fdlTuningParam.has_key( laneId )
   configExists = xcvrConfigCli.slotTuningParams.has_key( laneId ) \
                  if xcvrConfigCli else False
   configFixEq = xcvrConfigCli.slotTuningParams[ laneId ].txInputEqualization \
                 if configExists else None
   configAdaEq = xcvrConfigCli.slotTuningParams[ laneId ].txInputAdaptiveEqEnable \
                 if configExists else None

   # Configure Value
   if txAdaEquSupp or txEquSupp or overrideEn:
      if ( txAdaEquSupp or overrideEn ) and configExists and configAdaEq.valid and \
         configAdaEq.val:
         configuredStr = 'auto'
      elif ( txEquSupp or overrideEn ) and configExists and configFixEq.valid:
         configuredStr = str( configFixEq.val )
      elif slotDefExists and ( status.xcvrConfig.fdlTuningParam[
                                  laneId ].txInputAdaptiveEqEnable.valid or
                               status.xcvrConfig.fdlTuningParam[
                                  laneId ].txInputEqualization.valid ):
         configuredStr = 'Slot Default'
      else:
         configuredStr = 'Module Default'
   else:
      configuredStr = 'Module Default'

   # Operational Value
   operationalAdaEqEnable = status.txInputAdaptiveEqEnable[ laneId ]
   operationalFixEq = status.txInputEqualization[ laneId ]
   if operationalAdaEqEnable:
      operationalStr = 'auto' + starAdaEq
   else:
      operationalStr = str( operationalFixEq ) + starFixEq

   return moduleStr, configuredStr, operationalStr

def _getTuningParamStr( tuningType, tuningParamSupp, status, xcvrConfigCli, laneId ):
   if tuningType == 'Rx Output Amplitude':
      defaultParamStr = 'defaultRxOutputAmplitude'
      paramStr = 'rxOutputAmplitude'
   elif tuningType == 'Rx Output Pre-emphasis':
      defaultParamStr = 'defaultRxOutputPreEmphasis'
      paramStr = 'rxOutputPreEmphasis'
   elif tuningType == 'Rx Output Post-emphasis':
      defaultParamStr = ''
      paramStr = 'rxOutputPostEmphasis'
   else:
      defaultParamStr = 'defaultTxInputEqualization'
      paramStr = 'txInputEqualization'

   if status.xcvrType.lower() == "cfp2" and paramStr == "rxOutputAmplitude":
      return "N/A", "N/A", "N/A"

   if status.xcvrType not in [ 'osfp', 'qsfpDd', 'qsfpCmis' ] and \
      paramStr == "rxOutputPostEmphasis":
      return "N/A", "N/A", "N/A"

   star = '' if tuningParamSupp else '*'
   overrideEn = xcvrConfigCli and xcvrConfigCli.overrideTuningParams

   moduleStr = "N/A"
   if status.xcvrType in [ 'osfp', 'qsfpDd', 'qsfpCmis' ]:
      slotDefExists = status.xcvrConfig.fdlTuningParam.has_key( laneId )
      slotDefault = getattr( status.xcvrConfig.fdlTuningParam[ laneId ],
                             paramStr ) if slotDefExists else None
      moduleDefExists = status.defaultOsfpTuningParamValue.has_key( laneId )
      if moduleDefExists:
         moduleStr = str( getattr( status.defaultOsfpTuningParamValue[ laneId ],
                                   paramStr ) ) + star
   else:
      slotDefExists = status.xcvrType.lower() == 'qsfpplus' and \
                      getattr( status.xcvrConfig, paramStr ).has_key( laneId )
      slotDefault = getattr( status.xcvrConfig, paramStr ).get( laneId ) \
                    if slotDefExists else None
      moduleDefExists = status.xcvrType.lower() == 'qsfpplus' and \
                        getattr( status, defaultParamStr ).has_key( laneId )
      if moduleDefExists:
         moduleStr = str( getattr( status, defaultParamStr )[ laneId ] ) + star

   configExists = xcvrConfigCli.slotTuningParams.has_key( laneId ) \
                  if xcvrConfigCli else False
   configVal = getattr( xcvrConfigCli.slotTuningParams[ laneId ], paramStr ) \
               if configExists else None

   if tuningParamSupp or overrideEn:
      if configVal and configVal.valid:
         # CLI-configured value
         configuredStr = str( configVal.val )
      elif configVal and configVal.moduleDefault:
         # CLI-configured module-default
         configuredStr = "Module Default"
      elif slotDefault and slotDefault.valid:
         # CLI-default results in slot default
         configuredStr = "Slot Default"
      else:
         # Fall back to module default when no slot default exists
         configuredStr = "Module Default"
   else:
      configuredStr = "Module Default"

   operationalStr = str( getattr( status, paramStr )[ laneId ] ) + star
   return moduleStr, configuredStr, operationalStr

def _showInterfacesXcvrTuningDetail( mode, intf=None, mod=None ):
   ( intfs, intfNames ) = getAllIntfsWrapper( mode, intf, mod )
   if not intfs:
      return
   xcvrStatus = xcvrStatusDir_.xcvrStatus

   # Print out MBO tuning amplitude and preEmphasis settings if supported
   fmt = "%-11s%-18s%-18s%-18s"

   # Check if at least one xcvr support at least one tuning param or override its
   # tuning capabilities
   tuningTypeList = []
   for xcvrName in Arnet.sortIntf( xcvrStatus ):
      status = getXcvrStatus( xcvrStatus[ xcvrName ] )

      if status.presence != "xcvrPresent":
         # Skip transceivers that aren't present
         continue

      # Nothing shows up if all tuning params on all interfaces are not supported.
      txAdaEquSupp, txEquSupp, rxPostEmpSupp, rxPreEmpSupp, rxAmpSupp = \
         _getTuningSupported( status )
      overrideEn = _getTuningOverride( status )

      if any( [ txAdaEquSupp, txEquSupp, rxPostEmpSupp, rxPreEmpSupp,
                rxAmpSupp, overrideEn ] ):
         tuningTypeList = [ 'Rx Output Amplitude', 'Rx Output Pre-emphasis',
                            'Rx Output Post-emphasis', 'Tx Input Equalization' ]
         break

   # Print values of tuning params
   tuningHelpPrinted = False
   for tuningType in tuningTypeList:
      tuningHeaderPrinted = False

      for xcvrName in Arnet.sortIntf( xcvrStatus ):
         status = getXcvrStatus( xcvrStatus[ xcvrName ] )

         if status.presence != "xcvrPresent":
            # Skip transceivers that aren't present
            continue
         assert status.xcvrConfig.intfName

         primaryLaneId = 0
         primaryIntfName = status.xcvrConfig.intfName[ primaryLaneId ]
         xcvrConfigCliDir = getXcvrConfigCliDir( primaryIntfName )
         xcvrConfigCli = xcvrConfigCliDir.xcvrConfigCli.get( primaryIntfName )
         overrideEn = xcvrConfigCli and xcvrConfigCli.overrideTuningParams

         txAdaEquSupp, txEquSupp, rxPostEmpSupp, rxPreEmpSupp, rxAmpSupp = \
            _getTuningSupported( status )
         if not any( [ txAdaEquSupp, txEquSupp, rxPostEmpSupp, rxPreEmpSupp,
                       rxAmpSupp, overrideEn ] ):
            continue

         if tuningType == 'Rx Output Amplitude':
            tuningParamSupp = rxAmpSupp
         elif tuningType == 'Rx Output Pre-emphasis':
            tuningParamSupp = rxPreEmpSupp
         elif tuningType == 'Rx Output Post-emphasis':
            tuningParamSupp = rxPostEmpSupp
         elif tuningType == 'Tx Input Equalization':
            tuningParamSupp = txEquSupp
         else:
            tuningParamSupp = False

         maxChannels = status.capabilities.maxChannels
         for laneId in range( maxChannels ):
            name = getIntfNameByLane( laneId, status, maxChannels, intfNames )
            if not name:
               continue

            if status.xcvrType in [ 'osfp', 'qsfpDd', 'qsfpCmis' ] and \
               tuningType == 'Tx Input Equalization':
               moduleStr, configuredStr, operationalStr = _getCmisTxInputEqStr(
                  txAdaEquSupp, txEquSupp, status, xcvrConfigCli, laneId )
            else:
               moduleStr, configuredStr, operationalStr = _getTuningParamStr(
                  tuningType, tuningParamSupp, status, xcvrConfigCli, laneId )

            fieldValues = ( Intf.getShortname( name ),
                            moduleStr, configuredStr, operationalStr )
            if not tuningHelpPrinted:
               print( "* = Unsupported by module, value may be inaccurate" )
               tuningHelpPrinted = True
            if not tuningHeaderPrinted:
               _printTuningParamHeaderDefault( tuningType )
               tuningHeaderPrinted = True
            print( fmt % fieldValues )

def showInterfacesXcvrTuning( mode, args ):
   intf = args.get( 'INTF' )
   mod = args.get( 'MOD' )
   if 'detail' in args:
      _showInterfacesXcvrTuningDetail( mode, intf, mod )
   else:
      _showInterfacesXcvrTuning( mode, intf, mod )

class ShowIntfXcvrTuning( ShowIntfCommand ):
   syntax = 'show interfaces transceiver tuning [ detail ]'
   data = dict( transceiver=xcvrShowKw,
                tuning='Show interface transceiver tuning',
                detail='Show interface transceiver tuning detail' )
   handler = showInterfacesXcvrTuning

BasicCli.addShowCommandClass( ShowIntfXcvrTuning )

#-------------------------------------------------------------------------------
# The "show interfaces [<name>] transceiver hardware" command, in enable mode.
#   show interfaces [<name>] transceiver hardware
#   show interfaces module <modnum> transceiver hardware
#-------------------------------------------------------------------------------
def computeWavelength( frequency ):
   c = 299792458.0 # speed of light (m/s)
   return c / mhzFreqToGhz( frequency )

def getTuningWavelength( baseFrequency, channel, grid ):
   freq = getTuningFreq( baseFrequency, channel, grid )
   return computeWavelength( freq )

def getWavelength( status ):
   # Wavelength precision can be in nanometers, picometers, or
   # not supported at all (eg, copper media).  The display is
   # based on what that precision is.
   # Returns tuple ( wavelength, if precision is nm )
   if status and status.optical:
      if status.laserWavelengthPrecision == 'laserWavelengthPrecisionNm':
         return status.laserWavelength / 1000.0, True
      elif status.laserWavelengthPrecision == 'laserWavelengthPrecisionPm':
         if status.xcvrType.lower() == "qsfpplus":
            # qsfp directly provides the wavelength in pm. Convert it to nm.
            # This introduces the decimal to get away from integer math and
            # ensures the first decimal is printed.
            return status.laserWavelength / 1000.0, False
         return status.laserWavelength / 1000.0, False
   return None, None

def getGridSpacingCap( status ):
   # in GHz
   gridSpacingCap = status.resolvedTuningStatus.gridSpacingCapabilities
   spacing = ""
   if gridSpacingCap.gridSpacing100000M:
      spacing += "100.0,"
   if gridSpacingCap.gridSpacing50000M:
      spacing += "50.0,"
   if gridSpacingCap.gridSpacing33000M:
      spacing += "33.0,"
   if gridSpacingCap.gridSpacing25000M:
      spacing += "25.0,"
   if gridSpacingCap.gridSpacing12500M:
      spacing += "12.5,"
   if gridSpacingCap.gridSpacing6250M:
      spacing += "6.25"
   return spacing.strip( ',' )

# Helper function to return the cable type based on the media type.
# Returns False for media which do not have a cable type.
def getCableType( mediaType ):
   cableType = None
   if mediaType in ( XcvrMediaType.xcvr50GBaseCr2N,
                     XcvrMediaType.xcvr100GBaseCr4N,
                     XcvrMediaType.xcvr200GBaseCr8N,
                     XcvrMediaType.xcvr400GBaseCr8N,
                     XcvrMediaType.xcvr25GBaseCrN ):
      cableType = 'CA-N'
   elif mediaType in ( XcvrMediaType.xcvr50GBaseCr2S,
                       XcvrMediaType.xcvr100GBaseCr4S,
                       XcvrMediaType.xcvr200GBaseCr8S,
                       XcvrMediaType.xcvr400GBaseCr8S,
                       XcvrMediaType.xcvr25GBaseCrS ):
      cableType = 'CA-S'
   elif mediaType in ( XcvrMediaType.xcvr50GBaseCr2L,
                       XcvrMediaType.xcvr100GBaseCr4,
                       XcvrMediaType.xcvr200GBaseCr8,
                       XcvrMediaType.xcvr400GBaseCr8,
                       XcvrMediaType.xcvr25GBaseCr ):
      cableType = 'CA-L'
   return cableType

def _isDualLaserModule( status ):
   if hasattr( status, "cfp2DcoNewStatus" ) and status.cfp2DcoNewStatus:
      return status.cfp2DcoNewStatus.dcoType == "cfp2DcoTypeDp04"
   return False

def _isTxDisableSupportedModule( status ):
   # Currently, this field only applies to dual laser cfp2 dco modules
   return _isDualLaserModule( status )

def _isPerformanceMonitoringSupported( status ):
   '''
   Helper method for determining if performance monitoring is supported on this
   transceiver. Note: the status object provided must be the parent-most transceiver
   status.
   '''
   return status.typeInUse == 'qsfpPlus'

def _populateHardwareTuning( name, status ):
   tuningConstants = Tac.Value( "Xcvr::TuningConstants" )
   rts = status.resolvedTuningStatus
   sfpPlusPresent = status.xcvrType == 'sfpPlus'
   tunableLaserEnabled = status.ituTuningConfig.laserEnabled
   tuningFrequencyConfigured = bool(
      ( rts.operationalFrequency > 0 and
        rts.operationalFrequency == rts.configuredFrequency and
        tunableLaserEnabled ) or
      _isDualLaserModule( status ) )

   # _tunableLaserEnabled should be true always for tunable SFP+ or when configuring
   # CFP2 with a valid tuning frequency or channel/grid
   model = InterfacesTransceiverHardwareTuning(
      _outputSelect = tuningFrequencyConfigured,
      _tunableLaserEnabled = tunableLaserEnabled or sfpPlusPresent,
      _dualLaserModulePresent = _isDualLaserModule( status ) )

   if tunableLaserEnabled or sfpPlusPresent:
      if tuningFrequencyConfigured:
         model.configuredFrequency = mhzFreqToGhz( rts.configuredFrequency )
      else:
         model.configuredChannel = rts.configuredChannel
         model.configuredChannelUnsupportedIs( sfpPlusPresent and
            rts.configuredChannel != rts.operationalChannel and
            rts.configuredFrequency == 0 )
         model.configuredGrid = mhzFreqToGhz( rts.configuredGrid )
         model.computedFrequency = mhzFreqToGhz( rts.computedFrequency )
         model.operationalChannel = rts.operationalChannel
         model.operationalGrid = mhzFreqToGhz( rts.operationalGrid )
         model.operationalGridDefaultIs(
            rts.operationalGrid == tuningConstants.defaultGrid )
         # (default) label operational channel only shown for 10GBASE-DWDM
         model.operationalChannelDefaultIs( sfpPlusPresent and
            rts.operationalChannel == tuningConstants.defaultChannel and
            rts.operationalGrid == tuningConstants.defaultGrid )
      model.computedWavelength = rts.computedWavelength
      model.operationalFrequency = mhzFreqToGhz( rts.operationalFrequency )
      model.operationalWavelength = rts.operationalWavelength
   else:
      # populate only configured attributes in model when misconfigured
      if rts.configuredFrequency > 0:
         model.configuredFrequency = mhzFreqToGhz( rts.configuredFrequency )
         model.computedWavelength = rts.computedWavelength
      elif ( rts.configuredChannel > 0 and rts.configuredGrid > 0 and
             not _isDualLaserModule( status ) ):
         model.configuredChannel = rts.configuredChannel
         model.configuredGrid = mhzFreqToGhz( rts.configuredGrid )
         model.computedFrequency = mhzFreqToGhz( rts.computedFrequency )
         model.computedWavelength = rts.computedWavelength

   # Dual laser DCO module capable of tuning both tx and rx paths. Rx configuration
   # and status should always be displayed for this module.
   if _isDualLaserModule( status ):
      rrts = status.cfp2DcoNewStatus.resolvedRxTuningStatus
      model.configuredRxFrequency = mhzFreqToGhz( rrts.configuredFrequency )
      model.configuredRxFineFrequency = mhzFreqToGhz(
         rrts.configuredFineFrequency )
      model.computedRxWavelength = rrts.computedWavelength
      model.operationalRxFrequency = mhzFreqToGhz( rrts.operationalFrequency )
      model.operationalRxWavelength = rrts.operationalWavelength
   # Grid spacing capabilites for tunable transceivers only (i.e. CFPX-100G-DWDM,
   # 100G-DWDM-E, 10GBASE-DWDM)
   if rts.gridSpacingCapabilities:
      gridSpacingCap = getGridSpacingCap( status )
      model.gridSpacingCapabilities = gridSpacingCap
      model.gridSpacingCapabilitiesUnknownIs( gridSpacingCap == "" )

   return model

def _populateHardwarePower( status, cap, name ):
   model = InterfacesTransceiverHardwarePower(
      _tunableLaserEnabled = status.ituTuningConfig.laserEnabled,
      _dualLaserModulePresent = _isDualLaserModule( status ) )

   if cap.configurableTxPower:
      powerConstants = Tac.Value( 'Xcvr::PowerConstants' )
      txPowerConfig = xgc.txPowerConfig.get( name, None )
      if txPowerConfig:
         txPower = txPowerConfig.signalPower
      else:
         txPower = powerConstants.defaultTxPower
      model.configuredTxPower = txPower
      model.configuredTxPowerDefaultIs( txPower == powerConstants.defaultTxPower )

   if cap.configurableRxPower:
      powerConstants = Tac.Value( 'Xcvr::PowerConstants' )
      rxPowerConfig =  xgc.rxPowerConfig.get( name, None )
      rxPower = None
      if rxPowerConfig:
         rxPower = rxPowerConfig.signalPower
      model.configuredRxPower = rxPower
      # some transceivers, such as CFP2-DCO, do not support default Rx power and
      # do not provide operational Rx attenuation
      if not cap.directRxPowerConfig:
         model.configuredRxPowerDefaultIs( rxPower == powerConstants.defaultRxPower )
         model.operationalRxAttenuation = status.currRxAttenuation

   return model if model != InterfacesTransceiverHardwarePower() else None

def _populateHardware( status, name ):
   cap = status.capabilities
   model = InterfacesTransceiverHardware()
   model.mediaType = status.mediaTypeString
   model.detectedMediaType = status.detectedMediaTypeString
   model.cableType = getCableType( status.mediaType )
   model.powerCharacteristic = status.powerCharacteristic
   model.maxPowerSupportedCharacteristic = status.maxPowerSupportedCharacteristic
   wavelengthSupported = status.isOpticalDomSupported( status.mediaType )
   if wavelengthSupported and not cap.tunableWavelength:
      ( wavelength, precision ) = getWavelength( status )
      model.wavelength = wavelength
      model.wavelengthPrecNmIs( precision )
   model.power = _populateHardwarePower( status, cap, name )
   if cap.tunableWavelength:
      model.tuning = _populateHardwareTuning( name, status )
   if _isTxDisableSupportedModule( status ):
      model.txDisabled = ( name in xgc.txDisabled or
                           not status.ituTuningConfig.laserEnabled )
   return model

def showInterfacesXcvrHardware( mode, args ):
   intf = args.get( 'INTF' )
   mod = args.get( 'MOD' )
   model = InterfacesTransceiverHardwareBase()
   ( intfs, intfNames ) = getAllIntfsWrapper( mode, intf, mod )
   if not intfs:
      return model
   xcvrStatus = xcvrStatusDir_.xcvrStatus

   for intfName in intfNames:
      slotName = getXcvrSlotName( intfName )
      status = getXcvrStatus( xcvrStatus.get( slotName ) )
      if not status:
         continue

      lineSideChannels = getLineSideChannels( status )
      for laneId in range( lineSideChannels ):
         name = getIntfNameByLane( laneId, status, lineSideChannels, intfNames )
         if not name:
            continue
         if status.presence != "xcvrPresent":
            if len( intfs ) == 1:
               model.xcvrNotPresentIs( intfName )
               return model
            continue
         model.interfaces[ name ] = _populateHardware( status, name )
   return model

class ShowIntfXcvrHardware( ShowIntfCommand ):
   syntax = 'show interfaces transceiver hardware'
   data = dict( transceiver=xcvrShowKw,
                hardware='Show interface transceiver hardware' )
   handler = showInterfacesXcvrHardware
   cliModel = InterfacesTransceiverHardwareBase

BasicCli.addShowCommandClass( ShowIntfXcvrHardware )

#----------------------------------------------------------------------------
#
# "show interfaces [<name>] transceiver channels"
#
#----------------------------------------------------------------------------
def getChannelStr( baseFrequency, freq, grid ):
   assert freq >= baseFrequency
   if ( freq - baseFrequency ) % grid != 0:
      return ""
   return "%d" % getTuningChannel( baseFrequency, freq, grid )

def showInterfacesXcvrChannels( mode, args ):
   intf = args.get( 'INTF' )
   mod = args.get( 'MOD' )
   grid = args.get( 'GRID' )
   minFreq = args.get( 'MIN_FREQ' )
   maxFreq = args.get( 'MAX_FREQ' )

   ( intfs, intfNames ) = getAllIntfsWrapper( mode, intf, mod )
   if not intfs:
      return
   xcvrStatus = xcvrStatusDir_.xcvrStatus
   for intfName in Arnet.sortIntf( xcvrStatus ):
      status = getXcvrStatus( xcvrStatus[ intfName ] )
      for laneId in range( status.capabilities.lineSideChannels ):
         name = status.xcvrConfig.intfName.get( laneId )
         if name not in intfNames:
            continue

         shortName = Intf.getShortname( name )
         if status.presence != "xcvrPresent":
            # Only show "transceiver not present" message if the user requested
            # a single xcvr. Otherwise, don't pollute the output with such messages.
            if len( intfs ) == 1:
               mode.addError( "%s - transceiver not present" % shortName )
            continue

         if status.capabilities.tunableWavelength:
            tuningCap = status.tuningCapabilities
            baseFreq = tuningCap.baseFrequency
            firstFreq = tuningCap.firstFrequency
            lastFreq = tuningCap.lastFrequency
            startFreq = max( baseFreq, firstFreq )

            minimumFreq = int( ghzFreqToMhz( minFreq ) ) if minFreq else startFreq
            maximumFreq = int( ghzFreqToMhz( maxFreq ) ) if maxFreq else lastFreq
            if minimumFreq > maximumFreq:
               if len( intfs ) == 1:
                  mode.addError( "%s - frequency range invalid" % shortName )
               continue

            if grid is None:
               grid = "100,50"
            # Retrieve a unique collection of grid-spacings.
            gridSpacingList = sorted( list( set(
                                 map( float, grid.split( "," ) ) ) ) , reverse=True )

            fmt = "{0:<11} {1:>9}"
            frequencyDict = {}
            for index, gridSpacing in enumerate( gridSpacingList ):
               fmt += " {%d:>8}" % ( index + 2 )
               gridSpacingInMHz = int( ghzFreqToMhz( gridSpacing ) )
               for freq in range( startFreq, lastFreq + gridSpacingInMHz,
                     gridSpacingInMHz ):
                  if minimumFreq <= freq <= maximumFreq:
                     frequencyDict[ mhzFreqToGhz( freq ) ] = \
                           [ computeWavelength( freq ), mhzFreqToGhz( freq ) ]

            for freq in frequencyDict:
               for gridSpacing in gridSpacingList:
                  gridSpacingInMHz = int( ghzFreqToMhz( gridSpacing ) )
                  frequencyDict[ freq ].append(
                        getChannelStr( baseFreq, int( ghzFreqToMhz( freq ) ),
                                       int( ghzFreqToMhz( gridSpacing ) ) ) )

            headerLine1 = [ " " * 10, " " * 9 ]
            headerLine2 = [ "Wavelength", "Frequency" ]
            headerLine3 = [ "   (nm)   ", "  (GHz)  " ]
            headerLine4 = [ "-" * 10, "-" * 9 ]
            for gridSpacing in gridSpacingList:
               headerLine1.append( "%sGHz-" % gridSpacing )
               headerLine2.append( "spacing" )
               headerLine3.append( "Channel" )
               headerLine4.append( "-" * 7 )

            print( "Name: %s" % shortName )
            print( fmt.format( *headerLine1 ) )
            print( fmt.format( *headerLine2 ) )
            print( fmt.format( *headerLine3 ) )

            for freq in sorted( frequencyDict.keys() ):
               print( fmt.format( *frequencyDict[ freq ] ) )
            print( "" )
         else:
            if len( intfs ) == 1:
               mode.addError( "%s - transceiver not tunable" % shortName )
            continue

freqMatcher = CliMatcher.FloatMatcher( 190000, 200000,
                                       helpdesc='frequency (GHz)',
                                       precisionString='%.9g' )

class ShowIntfXcvrChannels( ShowIntfCommand ):
   syntax = 'show interfaces transceiver channels [ grid-spacing GRID ] ' \
            '[ min-frequency MIN_FREQ ] [ max-frequency MAX_FREQ ]'
   data = { 'transceiver' : xcvrShowKw,
            'channels' : 'Show interface transceiver supported channels',
            'grid-spacing' : gridKw,
            'GRID' : CliMatcher.PatternMatcher(
               r'^((100|50|33|25|12\.5|6\.25)\,)*(100|50|33|25|12\.5|6\.25)$',
               helpname='100,50,25,12.5,6.25', helpdesc='grid-spacing (GHz)' ),
            'min-frequency' : 'Specify minimum frequency',
            'MIN_FREQ' : freqMatcher,
            'max-frequency' : 'Specify maximum frequency',
            'MAX_FREQ' : freqMatcher
   }
   handler = showInterfacesXcvrChannels

BasicCli.addShowCommandClass( ShowIntfXcvrChannels )

#----------------------------------------------------------------------------
#
# "show idprom transceiver [ <interface> ] [ extended ]"
#
#----------------------------------------------------------------------------

# Helper collections
#    Headers: headers used in extended dump formatting
#    Contents: the attribute in xcvrStatus that contains the eeprom memory map
#    Pages: the attributes in contents that contain the page data we want
qsfpPageNums = [ "lower0",
                 "upper0",
                 "upper3",
                 "upper32",
                 "upper33" ]
# Note that page20h and page21h only exist when enhanced DOM is supported
# by the module. When not supported, these pages are empty and we don't
# print them out.
qsfpEepromPages = [ "eepromPage00LowerData",
                    "eepromPage00UpperData",
                    "eepromPage03UpperData",
                    "eepromPage20hUpperData",
                    "eepromPage21hUpperData" ]

sfpPageNums = [ "A0", "A2" ]
sfpEepromPages = [ "buf", "eepromPageA2Data" ]

# Static pages
_upper00h = "upper0"
_upper01h = "upper1"
_upper02h = "upper2"
# All OSFP pages
osfpPageNums = [ "lower0",
                 _upper00h,
                 _upper01h,
                 _upper02h,
                 "upper16",
                 "upper17" ]
_osfpBank = Tac.Type( "Xcvr::OsfpBank" )
_osfpPage = Tac.Type( "Xcvr::OsfpPage" )
osfpEepromPages = [ Tac.Value( "Xcvr::OsfpBankAndPage",
                               _osfpBank.bankNa, _osfpPage.lowerPage00 ),
                    Tac.Value( "Xcvr::OsfpBankAndPage",
                               _osfpBank.bankNa, _osfpPage.upperPage00 ),
                    Tac.Value( "Xcvr::OsfpBankAndPage",
                               _osfpBank.bankNa, _osfpPage.upperPage01 ),
                    Tac.Value( "Xcvr::OsfpBankAndPage",
                               _osfpBank.bankNa, _osfpPage.upperPage02 ),
                    Tac.Value( "Xcvr::OsfpBankAndPage",  # banked
                               _osfpBank.bank0, _osfpPage.upperPage10 ),
                    Tac.Value( "Xcvr::OsfpBankAndPage",  # banked
                               _osfpBank.bank0, _osfpPage.upperPage11 ), ]

# Extra pages for tunable wavelength
osfpTunableWavelengthPageNums = [ "upper4",
                                  "upper18" ]


osfpTunableWavelengthEepromPages = [ Tac.Value( "Xcvr::OsfpBankAndPage",
                                                _osfpBank.bankNa,
                                                _osfpPage.upperPage04 ),
                                     Tac.Value( "Xcvr::OsfpBankAndPage", # banked
                                                _osfpBank.bank0,
                                                _osfpPage.upperPage12 ), ]

# Extra pages for VDM Group 1
osfpVersatileDiagsMonitoringCommonPageNums = [ "upper32",
                                               "upper36",
                                               "upper40",
                                               "upper44",
                                               "upper45",
                                               "upper47" ]


osfpVersatileDiagsMonitoringCommonPages = [ Tac.Value( "Xcvr::OsfpBankAndPage",
                                                       _osfpBank.bank0,
                                                       _osfpPage.upperPage20 ),
                                            Tac.Value( "Xcvr::OsfpBankAndPage",
                                                       _osfpBank.bank0,
                                                       _osfpPage.upperPage24 ),
                                            Tac.Value( "Xcvr::OsfpBankAndPage",
                                                       _osfpBank.bank0,
                                                       _osfpPage.upperPage28 ),
                                            Tac.Value( "Xcvr::OsfpBankAndPage",
                                                       _osfpBank.bank0,
                                                       _osfpPage.upperPage2C ),
                                            Tac.Value( "Xcvr::OsfpBankAndPage",
                                                       _osfpBank.bank0,
                                                       _osfpPage.upperPage2D ),
                                            Tac.Value( "Xcvr::OsfpBankAndPage",
                                                       _osfpBank.bank0,
                                                       _osfpPage.upperPage2F ), ]

# Extra pages for VDM Groups 2
osfpVersatileDiagsMonitoringGroup2PageNums = [ "upper33",
                                               "upper37",
                                               "upper41" ]

osfpVersatileDiagsMonitoringGroup2Pages = [ Tac.Value( "Xcvr::OsfpBankAndPage",
                                                       _osfpBank.bank0,
                                                       _osfpPage.upperPage21 ),
                                            Tac.Value( "Xcvr::OsfpBankAndPage",
                                                       _osfpBank.bank0,
                                                       _osfpPage.upperPage25 ),
                                            Tac.Value( "Xcvr::OsfpBankAndPage",
                                                       _osfpBank.bank0,
                                                       _osfpPage.upperPage29 ), ]


# Extra pages for DWDM Coherent (ZR)
osfpCoherentDiagsMonitoringPageNums = [ "upper51",
                                        "upper53" ]

osfpCoherentDiagsMonitoringPages = [ Tac.Value( "Xcvr::OsfpBankAndPage",
                                                _osfpBank.bank0,
                                                _osfpPage.upperPage33 ),
                                     Tac.Value( "Xcvr::OsfpBankAndPage",
                                                _osfpBank.bank0,
                                                _osfpPage.upperPage35 ), ]


# Extra pages for POLS
polsPageNums = [ "upper192",
                 "upper193",
                 "upper208",
                 "upper209",
                 "upper216",
                 "upper217" ]
polsEepromPages = [ Tac.Value( "Xcvr::OsfpBankAndPage",
                               _osfpBank.bankNa, _osfpPage.upperPageC0 ),
                    Tac.Value( "Xcvr::OsfpBankAndPage",
                               _osfpBank.bankNa, _osfpPage.upperPageC1 ),
                    Tac.Value( "Xcvr::OsfpBankAndPage",
                               _osfpBank.bankNa, _osfpPage.upperPageD0 ),
                    Tac.Value( "Xcvr::OsfpBankAndPage",
                               _osfpBank.bankNa, _osfpPage.upperPageD1 ),
                    Tac.Value( "Xcvr::OsfpBankAndPage",
                               _osfpBank.bankNa, _osfpPage.upperPageD8 ),
                    Tac.Value( "Xcvr::OsfpBankAndPage",
                               _osfpBank.bankNa, _osfpPage.upperPageD9 ), ]

def getXcvrConfigCliDir( intfName ):
   # Helper function to get coathanger associated with interface
   isModular = Cell.cellType() == "supervisor"

   # Supervisors having xcvr slots will be present in FixedSystem slice
   if isModular and not Tac.Type( "Arnet::MgmtIntfId" ).isMgmtIntfId( intfName ):
      sliceName = EthIntfLib.sliceName( intfName )
      xcvrConfigCliDir = xcvrCliConfigSliceDir[ sliceName ]
   else:
      xcvrConfigCliDir = xcvrCliConfigSliceDir[ 'FixedSystem' ]
   return xcvrConfigCliDir

# Prepare function to create idprom-dump request. "show idprom transceiver extended"
# works in two parts:
#   1. Prepare the idprom dump. This is done by writing a config (i.e.
#      idEepromDumpRequest), which the XcvrCtrl SMs react to and read additional
#      pages as needed. The preparation is done in createIdpromDumpRequest()
#   2. Print the results of the extended idprom read. This is done primarily
#      _doExtendedIdpromDump(), which waits for the extended read to complete
#      if necessary, then reads attributes in xcvrStatus where the idprom contents
#      is stored.
def createIdpromDumpRequest( mode, intf=None, mod=None ):
   ( _ , intfNames ) = getAllIntfsWrapper( mode, intf, mod )
   if not intfNames:
      return
   # Filter out interface names that are valid to create XcvrConfigCli objs with.
   # Valid names are the primary interface name. If the user specified the dump on
   # any lane of a multi-lane interface, request a dump as if they specified the
   # primary lane.
   xcvrStatuses = xcvrStatusDir_.xcvrStatus
   primaryLaneId = 0
   primaryIntfNames = [ status.xcvrConfig.intfName[ primaryLaneId ] for status in
                        xcvrStatuses.values() if
                        any( intfName in intfNames for intfName in
                             status.xcvrConfig.intfName.values() ) ]
   for intfName in Arnet.sortIntf( primaryIntfNames ):
      xcvrConfigCliDir = getXcvrConfigCliDir( intfName )
      # Write a snapshot request
      if intfName not in xcvrConfigCliDir.xcvrConfigCli:
         xcvrConfigCliDir.newXcvrConfigCli( intfName )
      xcvrConfigCliDir.xcvrConfigCli[ intfName ].eepromDumpRequest = Tac.now()

# Helper function. When called, checks whether we have response for our dump-request
def _dumpRequestCompleted( xcvrStatus, xcvrDir ):
   primaryLaneId = 0
   primaryIntfName = xcvrStatus.xcvrConfig.intfName[ primaryLaneId ]
   xcvrConfigCliDir = getXcvrConfigCliDir( primaryIntfName )
   xcvrConfigCli = xcvrConfigCliDir.xcvrConfigCli[ primaryIntfName ]
   # If timestamp is newer than our request, eeprom is printable
   if( ( xcvrStatus.idEepromDumpTimestamp >= xcvrConfigCli.eepromDumpRequest ) or
       xcvrStatus.name not in xcvrDir ):
      return True
   return False

# Helper function. Prints msg informing user that xcvr is not there
def _printNotPresentMsg( mode, name, lenIntfs ):
   if lenIntfs == 1:
      # Only show "transceiver not present" message if the user requested
      # idprom of a single xcvr. Otherwise, when dumping idproms of all
      # xcvrs, don't pollute the output with such messages.
      mode.addError( "%s: transceiver not present" % name )

# Helper function. Factors out code for issuing a basic IDPROM data dump.
def _doNormalIdpromDump( mode, xcvrStatus, xcvrDir, intfs, model ):
   intfName = xcvrStatus.name

   # The non-extended version of the command only prints if the xcvr is
   # identified as present, so we want to preserve that behavior.
   if xcvrStatus.presence != "xcvrPresent":
      _printNotPresentMsg( mode, intfName, len( intfs ) )
      return

   model.interfaces[ intfName ] = InterfacesTransceiverIdpromPages()
   intfModel = model.interfaces[ intfName ]
   if xcvrStatus.xcvrType.lower() == "qsfpplus":
      intfModel.pages[ qsfpPageNums[ 1 ] ] = InterfacesTransceiverIdpromData()
      intfModel.pages[ qsfpPageNums[ 1 ] ].dataIs(
                           xcvrStatus.sff8436IdEepromContents.eepromPage00UpperData )
      intfModel.pages[ qsfpPageNums[ 1 ] ].regWidth = 8
   elif xcvrStatus.xcvrType.lower() == "sfpplus":
      intfModel.pages[ sfpPageNums[ 0 ] ] = InterfacesTransceiverIdpromData()
      intfModel.pages[ sfpPageNums[ 0 ] ].regWidth = 8
      intfModel.pages[ sfpPageNums[ 0 ] ].dataIs(
                                             xcvrStatus.sff8472IdEepromContents.buf )
   elif xcvrStatus.xcvrType.lower() in [ "osfp", "qsfpdd", "qsfpcmis" ]:
      # Create a temporary mapping from page numbers to OSFP EEPROM page objects
      cmisPageNumsToEeprom = dict( zip( osfpPageNums, osfpEepromPages ) )

      def _populateModel( pageNum ):
         intfModel.pages[ pageNum ] = InterfacesTransceiverIdpromData()
         intfModel.pages[ pageNum ].regWidth = 8
         data = xcvrStatus.rawEeprom.memory.get( cmisPageNumsToEeprom[ pageNum ] )
         # If the data for a page does not exist in our memory shadow,
         # then we use an empty string to represent that data.
         data = data if data is not None else ""
         intfModel.pages[ pageNum ].dataIs( data )

      # UpperPage00h, UpperPage01h, and UpperPage02h are static pages
      # and therefore do not require another SMBus read.  Whether or not
      # transceiver has page 01h and 02h implemented is determined by the
      # flatMem bit in lowerPage00h. Populate only the pages which are implemented.
      _populateModel( _upper00h )
      if xcvrStatus.writableEeprom():
         for pageNum in [ _upper01h, _upper02h ]:
            _populateModel( pageNum )

   elif xcvrStatus.xcvrType.lower() == "cfp2":
      if not xcvrStatus.initialized:
         # Avoid printing the spacing line for a non-initialized cfp2
         return
      memory = xcvrStatus.cfpMsaMisIdEepromContents.memory
      for tableName in sorted( memory.keys() ):
         table = memory[ tableName ]
         if not table.contents:
            # Skip empty memory pages
            continue
         intfModel.pages[ tableName ] = InterfacesTransceiverIdpromData()
         intfModel.pages[ tableName ].dataIs( table.contents )
         intfModel.pages[ tableName ].regWidth = table.registerWidth

# Helper function for the extended idprom dump command.
# Returns a list of tuples of (page number, page data) of type (string, string) when
# the EEPROM data is valid, otherwise returns None.
def _getIdpromData( xcvrStatus ):
   xcvrType = xcvrStatus.xcvrType.lower()

   # Different form-factors have different software EEPROM models,
   # so handle them differently
   if xcvrType in [ 'qsfpplus', 'sfpplus' ]:
      if xcvrType == 'qsfpplus':
         pageNums = qsfpPageNums
         eepromAttrName = "sff8436IdEepromContents"
         pages = qsfpEepromPages
      else:
         pageNums = sfpPageNums
         eepromAttrName = "sff8472IdEepromContents"
         pages = sfpEepromPages

      eeprom = getattr( xcvrStatus, eepromAttrName )
      # Check the validity of the EEPROM data
      if eeprom.lastValidData < xcvrStatus.idEepromDumpTimestamp:
         return None

      data = [ getattr( eeprom, page ) for page in pages ]
      return zip( pageNums, data )

   elif xcvrType in [ 'osfp', 'qsfpdd', 'qsfpcmis' ]:
      # Check the validity of the EEPROM data
      if xcvrStatus.rawEeprom.lastValidData < xcvrStatus.idEepromDumpTimestamp:
         return None

      rawPageData = []
      for page in osfpEepromPages:
         pgData = xcvrStatus.rawEeprom.memory.get( page )
         rawPageData.append( pgData if pgData is not None else '' )

      idpromData = zip( osfpPageNums, rawPageData )

      # For transceivers with tunable wavelength, show advertise and control pages
      rawTunableWlPageData = []
      if xcvrStatus.capabilities.tunableWavelength:
         for page in osfpTunableWavelengthEepromPages:
            pgData = xcvrStatus.rawEeprom.memory.get( page )
            rawTunableWlPageData.append( pgData if pgData is not None else '' )

         idpromData += zip( osfpTunableWavelengthPageNums, rawTunableWlPageData )

      # For transceiver that support Versatile Diagnostics Monitoring
      # avolinsk - only VDM groups 1 and 2 are supported
      rawVdmPageData = []
      if xcvrStatus.vdmCapabilities.vdmSupported:
         if xcvrStatus.vdmCapabilities.maxPageGroup >= 2:
            for page in osfpVersatileDiagsMonitoringGroup2Pages:
               pgData = xcvrStatus.rawEeprom.memory.get( page )
               rawVdmPageData.append( pgData if pgData is not None else '' )

               idpromData += zip( osfpVersatileDiagsMonitoringGroup2PageNums,
                                  rawVdmPageData )

         # Append VDM Group 1 and common pages
         rawVdmPageData = []
         for page in osfpVersatileDiagsMonitoringCommonPages:
            pgData = xcvrStatus.rawEeprom.memory.get( page )
            rawVdmPageData.append( pgData if pgData is not None else '' )

            idpromData += zip( osfpVersatileDiagsMonitoringCommonPageNums,
                            rawVdmPageData )

      # For Coherent CMIS modules
      rawCmisCoherentPageData = []
      if xcvrStatus.isCoherent( xcvrStatus.mediaType ):
         for page in osfpCoherentDiagsMonitoringPages:
            pgData = xcvrStatus.rawEeprom.memory.get( page )
            rawCmisCoherentPageData.append( pgData if pgData is not None else '' )

         idpromData += zip( osfpCoherentDiagsMonitoringPageNums,
                            rawCmisCoherentPageData )


      # POLS follow CMIS4.0 but has more customized pages
      rawPageData = []
      if xcvrStatus.mediaType == XcvrMediaType.xcvrAmpZr:
         for page in polsEepromPages:
            pgData = xcvrStatus.rawEeprom.memory.get( page )
            rawPageData.append( pgData if pgData is not None else '' )

         idpromData += zip( polsPageNums, rawPageData )
      # Pages could be added in any order, sort them by page number
      return natsort.natsorted( idpromData )

   else:
      assert 0, "Unexpected xcvrType %s in Idprom dump" % xcvrType
      return None

# Helper function. Factors out common code for issuing an extended IDPROM dump rather
# than a basic one.
def _doExtendedIdpromDump( mode, xcvrStatus, xcvrDir, intfs, model ):
   intfName = xcvrStatus.name

   if xcvrStatus.xcvrType.lower() == 'cfp2':
       # Cfp2 doesn't need to create a request for extended, so print "normal" dump
      _doNormalIdpromDump( mode, xcvrStatus, xcvrDir, intfs, model )
      return

   # Try to print. If the request becomes stale, timeout
   try:
      # Non-Arista hardware mostly lacks fancy SMBus accelerators.
      # We'll be more lenient with the timeout on non-Arista hardware.
      timeout = 120 if runningOnWhiteboxHardware() else 15

      ds = "idprom-dump on intf %s" % intfName
      Tac.waitFor( lambda: _dumpRequestCompleted( xcvrStatus, xcvrDir ),
                   sleep=True, timeout=timeout, description=ds, maxDelay=1.0 )

      # Check to see if the linecard has been removed/the name is valid
      if xcvrStatus.name not in xcvrDir:
         _printNotPresentMsg( mode, intfName, len( intfs ) )
         return

      # Get the actual EEPROM data from xcvrStatus
      eeprom = _getIdpromData( xcvrStatus )

      # Check if there was an error during the read. The value of 'eeprom' will be
      # None if the lastValidData timestamp predates the dump request timestamp.
      if xcvrStatus.presence == "xcvrPresent" and eeprom is None:
         print( "%s IDPROM: error retrieving data; read failed" % intfName )
      # The extended version will try to read as much as it can, regardless of xcvr
      # presence for sfp+, qsfp+, osfp and qsfpDd. Example: if we insert a faulty or
      # third-party xcvr, I believe status.presence gets stuck at xcvrPresenceUnknown
      else:
         anyData = eeprom and any( [ bool( pageData ) for _, pageData in eeprom ] )
         if not anyData:
            if len( intfs ) == 1:
               print( '%s: no data to print\n' % intfName )
            return
         else:
            model.interfaces[ intfName ] = InterfacesTransceiverIdpromPages()
            intfModel = model.interfaces[ intfName ]
            for ( pageNum, pageData ) in eeprom:
               # Check that the page has data loaded
               if pageData:
                  intfModel.pages[ pageNum ] = InterfacesTransceiverIdpromData()
                  intfModel.pages[ pageNum ].dataIs( pageData )
                  intfModel.pages[ pageNum ].regWidth = 8

   except Tac.Timeout:
      print( "%s: Timeout" % intfName )

# Dump IDPROM contents.
def xcvrIdpromDumpFunc( mode, func, model, intf=None, mod=None ):
   ( intfs, intfNames ) = getAllIntfsWrapper( mode, intf, mod )
   if not intfs:
      return
   xcvrStatuses = xcvrStatusDir_.xcvrStatus

   # Filter the xcvr names based on the specified interfaces
   filteredXcvrNames = [ xcvr.name for xcvr in xcvrStatuses.values()
                         if any( i in intfNames
                                 for i in xcvr.xcvrConfig.intfName.values() )
                         or xcvr.name in intfNames ]

   for xcvrName in Arnet.sortIntf( filteredXcvrNames ):
      # Call our library function to get the nested-most XcvrStatus object.
      xcvrStatus = getXcvrStatus( xcvrStatuses.get( xcvrName ) )

      # Check to see if the linecard has been removed
      if not xcvrStatus or xcvrName not in xcvrStatuses:
         _printNotPresentMsg( mode, xcvrName, len( intfs ) )
         continue

      # Check that the xcvr is supported
      if xcvrStatus.xcvrType.lower() not in [ "sfpplus", "qsfpplus", "cfp2",
            "osfp", "qsfpdd", "qsfpcmis" ]:
         continue

      func( mode, xcvrStatus, xcvrStatuses, intfs, model )

def runningOnWhiteboxHardware():
   # pkgdeps: library HwPlutoLibrary
   hwPlutoLibConfig = LazyMount.mount( entityManager_,
                                       'hardware/pluto/library/config',
                                       'Hardware::PlutoLibrary::Config', 'r' )
   return hwPlutoLibConfig.isWhitebox

def showInterfaceXcvrIdprom( mode, args ):
   intf = args.get( 'INTFS' )
   model = InterfacesTransceiverIdpromBase()
   if args.get( 'extended' ):
      createIdpromDumpRequest( mode, intf=intf )
      xcvrIdpromDumpFunc( mode, _doExtendedIdpromDump, model, intf=intf )
   else:
      xcvrIdpromDumpFunc( mode, _doNormalIdpromDump, model, intf=intf )
   return model

class ShowXcvrIdprom( ShowCommand.ShowCliCommandClass ):
   syntax = 'show idprom ( transceiver | interface ) [ INTFS ] [ extended ]'
   data = {
      'idprom' : CliToken.Idprom.idpromNode,
      'transceiver' : Node( matcher=xcvrShowKw ),
      'interface' : Node( matcher=interfaceKwMatcher, hidden=True ),
      'INTFS' : Node( matcher=IntfRangeMatcher() ),
      'extended' : CliToken.Idprom.extendedNode,
   }
   handler = showInterfaceXcvrIdprom
   cliModel = InterfacesTransceiverIdpromBase

BasicCli.addShowCommandClass( ShowXcvrIdprom )


#----------------------------------------------------------------------------
#
# "show interfaces [ <names> ] transceiver eeprom"
#
#----------------------------------------------------------------------------

# Function to use to decode the EEPROM of a xcvr for a given xcvrType, should take a
# xcvrStatus and a model to fill for just that xcvr
eepromDecodeFuncs = {
                      XcvrType.osfp : ( decodeCmis, "eepromContents" ),
                      XcvrType.qsfpDd : ( decodeCmis, "eepromContents" ),
                      XcvrType.qsfpCmis : ( decodeCmis, "eepromContents" ),
                      XcvrType.qsfpPlus : ( decodeQsfp, "sff8436IdEepromContents" ),
                      XcvrType.sfpPlus : ( decodeSfp, "sff8472IdEepromContents" ),
                    }
def _doDecodedEepromDump( mode, xcvrStatus, xcvrDir, intfs, model ):
   intfName = xcvrStatus.name
   if xcvrStatus.xcvrType not in eepromDecodeFuncs:
      print( "%s: decode not supported for this transceiver type" % intfName )
      return

   # Get the actual EEPROM data from xcvrStatus
   eeprom = _getIdpromData( xcvrStatus )

   # Check if there was an error during the read. The value of 'eeprom' will be
   # None if the lastValidData timestamp predates the dump request timestamp.
   if xcvrStatus.presence == "xcvrPresent" and eeprom is None:
      print( "%s IDPROM: error retrieving data; read failed" % intfName )
   # The decode version will try to read as much as it can, regardless of xcvr
   # presence for sfp+, qsfp+, osfp and qsfpDd. Example: if we insert a faulty or
   # third-party xcvr, I believe status.presence gets stuck at xcvrPresenceUnknown
   else:
      anyData = eeprom and any( bool( pageData ) for _, pageData in eeprom )
      if anyData:
         model.interfaces[ intfName ] = \
            InterfacesTransceiverEepromIntf( _name=intfName )
         decode, eepromName = eepromDecodeFuncs[ xcvrStatus.xcvrType ]
         decode( getattr( xcvrStatus, eepromName ), model.interfaces[ intfName ] )
      else:
         if len( intfs ) == 1:
            print( '%s: no data to print\n' % intfName )


def showInterfaceXcvrEeprom( mode, args ):
   intf = args.get( 'INTF' )
   mod = args.get( 'MOD' )
   model = InterfacesTransceiverEepromBase()
   xcvrIdpromDumpFunc( mode, _doDecodedEepromDump, model,
                       intf=intf, mod=mod )
   return model

class ShowIntfXcvrEeprom( ShowIntfCommand ):
   syntax = 'show interfaces transceiver eeprom'
   data = dict( transceiver=xcvrShowKw,
                eeprom='Show decoded dump of supported EEPROM pages' )
   handler = showInterfaceXcvrEeprom
   cliModel = InterfacesTransceiverEepromBase

BasicCli.addShowCommandClass( ShowIntfXcvrEeprom )


#--------------------------------------------------------------------------------
#
# show transceiver status [ interface INTF ]
#
#--------------------------------------------------------------------------------

registeredXcvrStatusFn = {}
def registerXcvrStatusFn( fnName, fn ):
   assert fnName not in registeredXcvrStatusFn
   registeredXcvrStatusFn[ fnName ] = fn

def _populateXcvrStatusValues( intfName ):
   model = InterfacesTransceiverStatusValues()

   baseXcvrStatus = xcvrStatusDir_.xcvrStatus[ getXcvrSlotName( intfName ) ]
   xcvrStatus = getXcvrStatus( baseXcvrStatus )

   model.serialNumber = InterfacesTransceiverStatusValueFormatStr(
      state=_strip( xcvrStatus.vendorInfo.vendorSn ) )

   model.mediaType = InterfacesTransceiverStatusValueFormatStr(
      state=xcvrStatus.mediaTypeString,
      changes=xcvrStatus.presenceChanges,
      lastChange=xcvrStatus.lastPresenceChange )

   model.presence = InterfacesTransceiverStatusValueFormatStr(
      state=xcvrPresenceToStr[ xcvrStatus.presence ] )

   model.badEepromChecksums = InterfacesTransceiverStatusValueFormat(
      changes=xcvrStatus.xcvrStats.badEepromChecksumEvents,
      lastChange=xcvrStatus.xcvrStats.lastBadEepromChecksumEvent )

   model.resets = InterfacesTransceiverStatusValueFormat(
      changes=xcvrStatus.xcvrStats.resetCount,
      lastChange=xcvrStatus.xcvrStats.lastReset )

   model.interrupts = InterfacesTransceiverStatusValueFormat(
      changes=xcvrStatus.xcvrStats.interruptCount,
      lastChange=xcvrStatus.xcvrStats.lastInterrupt )

   if xcvrStatus.smbusFailCountValid:
      model.smbusFailures = InterfacesTransceiverStatusValueFormat(
         changes=xcvrStatus.xcvrStats.smbusFailCount,
         lastChange=xcvrStatus.xcvrStats.lastSmbusFail )

   swizzlers = []
   curXcvrStatus = baseXcvrStatus
   if isOsfpToQsfpSwizzler( curXcvrStatus ):
      swizzlers.append( 'oqa' )
      curXcvrStatus = curXcvrStatus.qsfpStatus
   if isQsfpDdToQsfpSwizzler( curXcvrStatus ):
      curXcvrStatus = curXcvrStatus.qsfpStatus
   if isQsfpToSfpSwizzler( curXcvrStatus ):
      if xcvrStatus.qsa28:
         swizzlers.append( 'qsa28' )
      else:
         swizzlers.append( 'qsa10' )
   model.adapters = InterfacesTransceiverStatuValuesSwizzlerList(
      state=swizzlers )

   showVoltTempAlarmWarn = xcvrStatus.optical and xcvrStatus.xcvrStatsEnt

   if xcvrStatus.xcvrType in [ "osfp", "qsfpDd" ]:
      showVoltTempAlarmWarn = xcvrStatus.writableEeprom() and xcvrStatus.xcvrStatsEnt
      model.dataPathFirmwareFault = InterfacesTransceiverStatusValueFormatBool(
         state=bool( xcvrStatus.cmisModuleStatsEnt.dataPathFirmwareFault.current ),
         changes=xcvrStatus.cmisModuleStatsEnt.dataPathFirmwareFault.changes,
         lastChange=xcvrStatus.cmisModuleStatsEnt.dataPathFirmwareFault.lastChange )
      model.moduleFirmwareFault = InterfacesTransceiverStatusValueFormatBool(
         state=bool( xcvrStatus.cmisModuleStatsEnt.moduleFirmwareFault.current ),
         changes=xcvrStatus.cmisModuleStatsEnt.moduleFirmwareFault.changes,
         lastChange=xcvrStatus.cmisModuleStatsEnt.moduleFirmwareFault.lastChange )
      model.moduleState = InterfacesTransceiverStatusValueFormatStr(
            state=moduleStateToStr[ xcvrStatus.moduleState ],
            changes=xcvrStatus.cmisModuleStatsEnt.moduleStateChanged.changes,
            lastChange=xcvrStatus.cmisModuleStatsEnt.moduleStateChanged.lastChange )
      for laneId in xcvrStatus.dataPathState:
         laneStats = xcvrStatus.laneStats.get( laneId )
         if laneStats is None:
            continue
         model.dataPathState[ laneId + 1 ] = \
            InterfacesTransceiverStatusValueFormatStr(
               state=dataPathStateToStr[ xcvrStatus.dataPathState[ laneId ] ],
               changes=laneStats.dataPathStateChange.changes,
               lastChange=laneStats.dataPathStateChange.lastChange )   

   if showVoltTempAlarmWarn:
      model.voltageHiAlarm = InterfacesTransceiverStatusValueFormatBool(
         state=bool( xcvrStatus.xcvrStatsEnt.voltageHiAlarm.current ),
         changes=xcvrStatus.xcvrStatsEnt.voltageHiAlarm.changes,
         lastChange=xcvrStatus.xcvrStatsEnt.voltageHiAlarm.lastChange )
      model.voltageLoAlarm = InterfacesTransceiverStatusValueFormatBool(
         state=bool( xcvrStatus.xcvrStatsEnt.voltageLoAlarm.current ),
         changes=xcvrStatus.xcvrStatsEnt.voltageLoAlarm.changes,
         lastChange=xcvrStatus.xcvrStatsEnt.voltageLoAlarm.lastChange )
      model.voltageHiWarn = InterfacesTransceiverStatusValueFormatBool(
         state=bool( xcvrStatus.xcvrStatsEnt.voltageHiWarn.current ),
         changes=xcvrStatus.xcvrStatsEnt.voltageHiWarn.changes,
         lastChange=xcvrStatus.xcvrStatsEnt.voltageHiWarn.lastChange )
      model.voltageLoWarn = InterfacesTransceiverStatusValueFormatBool(
         state=bool( xcvrStatus.xcvrStatsEnt.voltageLoWarn.current ),
         changes=xcvrStatus.xcvrStatsEnt.voltageLoWarn.changes,
         lastChange=xcvrStatus.xcvrStatsEnt.voltageLoWarn.lastChange )
      model.tempHiAlarm = InterfacesTransceiverStatusValueFormatBool(
         state=bool( xcvrStatus.xcvrStatsEnt.tempHiAlarm.current ),
         changes=xcvrStatus.xcvrStatsEnt.tempHiAlarm.changes,
         lastChange=xcvrStatus.xcvrStatsEnt.tempHiAlarm.lastChange )
      model.tempLoAlarm = InterfacesTransceiverStatusValueFormatBool(
         state=bool( xcvrStatus.xcvrStatsEnt.tempLoAlarm.current ),
         changes=xcvrStatus.xcvrStatsEnt.tempLoAlarm.changes,
         lastChange=xcvrStatus.xcvrStatsEnt.tempLoAlarm.lastChange )
      model.tempHiWarn = InterfacesTransceiverStatusValueFormatBool(
         state=bool( xcvrStatus.xcvrStatsEnt.tempHiWarn.current ),
         changes=xcvrStatus.xcvrStatsEnt.tempHiWarn.changes,
         lastChange=xcvrStatus.xcvrStatsEnt.tempHiWarn.lastChange )
      model.tempLoWarn = InterfacesTransceiverStatusValueFormatBool(
         state=bool( xcvrStatus.xcvrStatsEnt.tempLoWarn.current ),
         changes=xcvrStatus.xcvrStatsEnt.tempLoWarn.changes,
         lastChange=xcvrStatus.xcvrStatsEnt.tempLoWarn.lastChange )

   return model

def displayOnlyRxLosTxFaultPinStates( xcvrStatus ):
   ''' Xcvr CLI helper to determine if 'show transceiver status interface'
   command should only render pin states for RX_LOS and TX_FAULT
   instead of all the DOM parameters.
   We want to catch all the SFP's that are not optics but do implement these options.
   For all the other optical media-types, isOpticalDomSupported check
   should be used to determine the eligibility of displaying DOM parameters.
   ----------------------------------------------------------------------------
   params:
      xcvrStatus : Xcvr::XcvrNewStatus
         status object of the transceiver slot.
   '''
   rxLosOrTxFaultImpl = RxLosOrTxFaultImplemented( False, False )
   if xcvrStatus.xcvrType == XcvrType.sfpPlus:
      optionsValue = xcvrStatus.sff8472IdEepromContents.options
      options = Tac.newInstance( "Xcvr::Sff8472Options", optionsValue )
      rxLosOrTxFaultImpl = RxLosOrTxFaultImplemented( options.rxLosImplemented,
            options.txFaultImplemented )
   return rxLosOrTxFaultImpl

def _populateCoherentXcvrStatusValues( intfName ):

   status = xcvrStatusDir_.xcvrStatus[ getXcvrSlotName( intfName ) ]
   baseModel = None

   def populateModel( model, cfp2DcoStatus, attrs, attrType, isEventGroup,
                      trueState, falseState ):
      '''
      Helper method to populate commonly formatted alarm/warning models for the DP04
      dual laser module.
      '''
      for attr in attrs:
         attrValue = getattr( cfp2DcoStatus, attrType ).event[ attr ].stat
         if isEventGroup:
            current = attrValue.current
            lastChange = attrValue.lastChange
         else:
            current = attrValue.value
            lastChange = attrValue.when
         submodel = InterfacesTransceiverStatusValueFormatStr(
               state=trueState if current else falseState,
               changes=attrValue.changes,
               lastChange=lastChange )
         setattr( model, attr, submodel )

   if status.mediaType == XcvrMediaType.xcvr100GDwdmDco:
      cfp2DcoStatus = status.cfp2DcoNewStatus
      baseModel = InterfacesTransceiverStatusExtCfp2Dco()
      baseModel.controllerState = InterfacesTransceiverStatusValueFormatStr(
            state=cfp2DcoStatus.controllerState )
      baseModel.moduleState = InterfacesTransceiverStatusValueFormatHex(
            state=cfp2DcoStatus.moduleState )
      baseModel.txTurnUpState = InterfacesTransceiverStatusValueFormatHex(
            state=cfp2DcoStatus.txTurnUpState )
      baseModel.rxTurnUpState = InterfacesTransceiverStatusValueFormatHex(
            state=cfp2DcoStatus.rxTurnUpState )
      baseModel.moduleGeneral = InterfacesTransceiverStatusValueFormatHex(
            state=cfp2DcoStatus.moduleGeneralStatus )
      baseModel.moduleFaults = InterfacesTransceiverStatusValueFormatHex(
            state=cfp2DcoStatus.moduleFaultStatus )
      baseModel.vendorFaults = InterfacesTransceiverStatusValueFormatHex(
            state=cfp2DcoStatus.vendorSpecificFAWS )
      baseModel.networkVendorFaults = InterfacesTransceiverStatusValueFormatHex(
            state=cfp2DcoStatus.networkVendorSpecificFAWS )
      baseModel.laserFaultCode = InterfacesTransceiverStatusValueFormatHex(
            state=cfp2DcoStatus.laserFaultCode )
      baseModel.sopFastTracking = InterfacesTransceiverStatusValueFormatStr(
            state="enabled" if cfp2DcoStatus.sopFastTracking else "disabled" )

   elif _isDualLaserModule( status ):
      # The following states apply only to the dual laser DP04 module
      baseModel = InterfacesTransceiverStatusExtCfp2Dco()
      baseModel.dualLaserModulePresentIs( True )
      cfp2DcoStatus = status.cfp2DcoNewStatus
      baseModel.controllerState = InterfacesTransceiverStatusValueFormatStr(
            state=cfp2DcoStatus.controllerState )
      baseModel.moduleStateDec = \
         InterfacesTransceiverStatusValueFormatModuleStateDec()
      moduleStateEvents = [ 'init', 'lowPwr', 'hiPwrUp', 'txOff', 'txTurnOn',
                            'ready', 'fault', 'txTurnOff', 'hiPwrDown' ]
      populateModel( baseModel.moduleStateDec, cfp2DcoStatus, moduleStateEvents,
                     'moduleStateDec', True, 'active', 'inactive' )

      # TX turn up state
      baseModel.txTurnUpStateDec = \
         InterfacesTransceiverStatusValueFormatTxTurnUpStateDec()
      txTurnUpState = baseModel.txTurnUpStateDec
      txTurnUpModelInputs = [ ( 'txInit', 'ok', 'not initialized' ),
                              ( 'txDataPathLock', 'ok', 'no lock' ),
                              ( 'txLasReadyOff', 'ok', 'not ready' ),
                              ( 'txLaserReady', 'ok', 'not ready' ),
                              ( 'txModulatorConverge', 'ok', 'not converged' ),
                              ( 'txOutPwrAdj', 'ok', 'not adjusted' ), ]
      for ( txTurnUpAttr, trueState, falseState ) in txTurnUpModelInputs:
         populateModel( txTurnUpState, cfp2DcoStatus, [ txTurnUpAttr ],
                        'txTurnUpStateDec', False, trueState, falseState )

      # RX turn up state
      baseModel.rxTurnUpStateDec = \
         InterfacesTransceiverStatusValueFormatRxTurnUpStateDec()
      rxTurnUpState = baseModel.rxTurnUpStateDec
      rxTurnUpModelInputs = [ ( 'rxInit', 'ok', 'not initialized' ),
                              ( 'rxLoLaserReady', 'ok', 'not ready' ),
                              ( 'rxWaitForInput', 'ok', 'not ready' ),
                              ( 'adcOutput', 'up', 'down' ),
                              ( 'dispersionLock', 'ok', 'no lock' ),
                              ( 'rxDemodLock', 'ok', 'no lock' ), ]
      for ( rxTurnUpAttr, trueState, falseState ) in rxTurnUpModelInputs:
         populateModel( rxTurnUpState, cfp2DcoStatus, [ rxTurnUpAttr ],
                        'rxTurnUpStateDec', False, trueState, falseState )

      # Module general status
      baseModel.moduleGeneralStatus = \
         InterfacesTransceiverStatusValueFormatModuleGeneralStatus()
      moduleGeneralStatus = baseModel.moduleGeneralStatus
      modGenStatModelInputs = [ ( 'hiPowerOn', 'ok', 'power off' ),
                                ( 'outOfAlgn', 'unaligned', 'ok' ),
                                ( 'rxNetworkLOL', 'no lock', 'ok' ),
                                ( 'rxLOS', 'no signal', 'ok' ),
                                ( 'txHostLOL', 'no lock', 'ok' ),
                                ( 'txLOF', 'no signal', 'ok' ),
                                ( 'hwInterlock', 'max capacity', 'ok' ), ]
      for ( modStatAttr, trueState, falseState ) in modGenStatModelInputs:
         populateModel( moduleGeneralStatus, cfp2DcoStatus, [ modStatAttr ],
                        'modGeneralStatus', True, trueState, falseState )

      # Module fault status
      baseModel.moduleFaultStatus = \
         InterfacesTransceiverStatusValueFormatModuleFaultStatus()
      moduleFaultStatus = baseModel.moduleFaultStatus
      checkSum = cfp2DcoStatus.modFaultStat.event[ 'checkSum' ].stat
      moduleFaultStatus.checkSum = InterfacesTransceiverStatusValueFormatStr(
         state='ok' if not checkSum.current else 'failed',
         changes=checkSum.changes,
         lastChange=checkSum.lastChange )
      modFaultStatusAttrs = [ 'powerSupplyFault', 'pldFlashInitFault',
                              'modSpecFault', 'modOverTempFault' ]
      populateModel( baseModel.moduleFaultStatus, cfp2DcoStatus, modFaultStatusAttrs,
                     'modFaultStat', True, 'fault', 'ok' )

      # Dual laser alarms/warnings
      netAlarm1Attrs = [ 'txLaserPowerHighAlarm', 'txLaserPowerLowAlarm',
                         'txLaserTempLowAlarm', 'txLaserTempHighAlarm',
                         'rxPowerHighAlarm', 'rxPowerLowAlarm' ]
      populateModel( baseModel, cfp2DcoStatus, netAlarm1Attrs,
                     'netAlarmWarn1', True, 'alarm', 'ok' )
      netWarning1Attrs = [ 'txLaserPowerHighWarn', 'txLaserPowerLowWarn',
                           'txLaserTempHighWarn', 'txLaserTempLowWarn',
                           'rxPowerHighWarn', 'rxPowerLowWarn' ]
      populateModel( baseModel, cfp2DcoStatus, netWarning1Attrs,
                     'netAlarmWarn1', True, 'warn', 'ok' )
      netAlarm2Attrs = [ 'txModBiasHighAlarm', 'rxLaserPowerHighAlarm',
                         'rxLaserPowerLowAlarm', 'rxLaserTempHighAlarm',
                         'rxLaserTempLowAlarm' ]
      populateModel( baseModel, cfp2DcoStatus, netAlarm2Attrs,
                     'netAlarmWarn2', True, 'alarm', 'ok' )
      netWarning2Attrs = [ 'txModBiasHighWarn', 'rxLaserPowerHighWarn',
                           'rxLaserPowerLowWarn', 'rxLaserTempHighWarn',
                           'rxLaserTempLowWarn' ]
      populateModel( baseModel, cfp2DcoStatus, netWarning2Attrs,
                     'netAlarmWarn2', True, 'warn', 'ok' )

      # Network lane vendor specific faults/status
      baseModel.netLaneVendorSpecificFAWS = \
         InterfacesTransceiverStatusValueFormatNetworkLaneVendorFAWS()
      netLaneVendorSpecificFAWS = baseModel.netLaneVendorSpecificFAWS
      netVendorSpecFAWSModelInputs = [
         ( [ 'oaPumpBiasHighAlarm', 'oaPumpBiasLowAlarm', 'rxPhaseCtrlLoopHighAlarm',
             'rxPhaseCtrlLoopLowAlarm', 'txModBiasVOAHighAlarm',
             'rxTunedChPwrHighAlarm', 'rxChannelPwrLowAlarm' ],
           'alarm', 'ok' ),
         ( [ 'oaPumpBiasHighWarn', 'oaPumpBiasLowWarn', 'rxPhaseCtrlLoopHighWarn',
             'rxPhaseCtrlLoopLowWarn', 'txModBiasVOAHighWarn',
             'rxTunedChPwrHighWarn', 'rxChannelPwrLowWarn' ],
           'warn', 'ok' ),
         ( [ 'modBiasConvergeFault' ], 'fault', 'ok' ),
         ( [ 'adcOutputLow' ], 'low', 'ok' ), ]
      for ( fawsAttrs, trueState, falseState ) in netVendorSpecFAWSModelInputs:
         populateModel( netLaneVendorSpecificFAWS, cfp2DcoStatus,
                        fawsAttrs, 'networkVendorSpecificFAWSDec',
                        True, trueState, falseState )

      # Network lane fault/status
      baseModel.netLaneFaultStatus = \
         InterfacesTransceiverStatusValueFormatNetworkLaneFaultStatus()
      netLaneFaultStatus = baseModel.netLaneFaultStatus
      netLaneFaultAttrs = [ 'rxTECFault', 'rxFIFOFault', 'wavelengthUnlockFault',
                            'tecFault' ]
      populateModel( netLaneFaultStatus, cfp2DcoStatus, netLaneFaultAttrs,
                     'networkFaultStatus', True, 'fault', 'ok' )
      netLaneFaultStatModelInputs = [ ( 'rxLOL', 'no lock', 'ok' ),
                                      ( 'rxLOS', 'no signal', 'ok' ),
                                      ( 'txLOL', 'no lock', 'ok' ),
                                      ( 'txLOSF', 'no signal', 'ok' ), ]
      for ( netLaneFaultStatAttr, trueState,
            falseState ) in netLaneFaultStatModelInputs:
         populateModel( netLaneFaultStatus, cfp2DcoStatus,
                        [ netLaneFaultStatAttr ], 'networkFaultStatus',
                        True, trueState, falseState )

      # TX/RX lane alignment
      baseModel.laneAlignment = InterfacesTransceiverStatusValueFormatLaneAlignment()
      laneAlignment = baseModel.laneAlignment
      netLaneTxAlignment = cfp2DcoStatus.txNetworkAlignmentStatus.event[
         'outOfAlgn' ].stat
      laneAlignment.netLaneTxAlignment = \
         InterfacesTransceiverStatusValueFormatStr(
            state='unaligned' if netLaneTxAlignment.current else 'ok',
            changes=netLaneTxAlignment.changes,
            lastChange=netLaneTxAlignment.lastChange )
      populateModel( laneAlignment, cfp2DcoStatus,
                     [ 'modemLockFault', 'modemSyncDetectFault' ],
                     'rxNetworkAlignmentStatus', True, 'fault', 'ok' )

      # Network lane RX OTN status
      baseModel.networkLaneRxOTNStatus = \
         InterfacesTransceiverStatusValueFormatNetworkLaneRxOTNStatus()
      networkLaneRxOTNStatus = baseModel.networkLaneRxOTNStatus
      rxNetOTNStatusAttrs = [ 'fastPMRxSignalDegradeAlarm',
         'fastPMRxSignalFailAlarm', 'fastPMTxSignalDegradeAlarm',
         'fastPMTxSignalFailAlarm', 'ingressFDDAlarm', 'ingressFEDAlarm' ]
      populateModel( networkLaneRxOTNStatus, cfp2DcoStatus, rxNetOTNStatusAttrs,
                     'rxNetOTNStatus', True, 'alarm', 'ok' )
      netLaneRxOTNStatModelInputs = [
         ( 'oduLCKMaintSignal', 'active', 'inactive' ),
         ( 'oduAISMaintSignal', 'active', 'inactive' ),
         ( 'sectionMonitoringBackDefect', 'defect', 'ok' ),
         ( 'outOfMultiframe', 'out of frame', 'ok' ),
         ( 'lossOfMultiframe', 'no frame', 'ok' ),
         ( 'outOfFrame', 'out of frame', 'ok' ),
         ( 'lossOfFrame', 'no frame', 'ok' ) ]
      for ( netLaneRxOTNStatAttr, trueState,
            falseState ) in netLaneRxOTNStatModelInputs:
         populateModel( networkLaneRxOTNStatus, cfp2DcoStatus,
                        [ netLaneRxOTNStatAttr ], 'rxNetOTNStatus', True, trueState,
                        falseState )

      # Network TX modulator bias VOA AWS
      baseModel.networkTxModulatorBiasVoaAWS = \
         InterfacesTransceiverStatusValueFormatTxModulatorBiasVOA()
      networkTxModulatorBiasVoaAWS = baseModel.networkTxModulatorBiasVoaAWS
      netTxModBiasVOAStatAlarms = [ 'loop1Alarm', 'loop2Alarm', 'loop3Alarm',
                                    'loop4Alarm', 'loop5Alarm', 'loop6Alarm',
                                    'loop7Alarm', 'loop8Alarm' ]
      populateModel( networkTxModulatorBiasVoaAWS, cfp2DcoStatus,
                     netTxModBiasVOAStatAlarms, 'netTxModBiasVOAStat', False,
                     'alarm', 'ok' )
      netTxModBiasVOAStatWarnings = [ 'loop1Warn', 'loop2Warn', 'loop3Warn',
                                      'loop4Warn', 'loop5Warn', 'loop6Warn',
                                      'loop7Warn', 'loop8Warn' ]
      populateModel( networkTxModulatorBiasVoaAWS, cfp2DcoStatus,
                     netTxModBiasVOAStatWarnings, 'netTxModBiasVOAStat', False,
                     'warn', 'ok' )

      # ADC and DAC fault codes
      adcFaultCode = cfp2DcoStatus.ntwkAdcDacFltCodes.event[
         'netADCFaultCode' ].stat
      baseModel.adcFaultCode = \
         InterfacesTransceiverStatusValueFormatHex(
            state=adcFaultCode.value,
            changes=adcFaultCode.changes,
            lastChange=adcFaultCode.when )
      dacFaultCode = cfp2DcoStatus.ntwkAdcDacFltCodes.event[
         'netDACFaultCode' ].stat
      baseModel.dacFaultCode = \
         InterfacesTransceiverStatusValueFormatHex(
            state=dacFaultCode.value,
            changes=dacFaultCode.changes,
            lastChange=dacFaultCode.when )

      # Net mod bias fault information
      convergenceIter = cfp2DcoStatus.netModBiasFaultInfo.event[
         'convergenceIter' ].stat
      baseModel.modulatorConvergenceIter = \
         InterfacesTransceiverStatusValueFormatInt(
            state=int( convergenceIter.value ),
            changes=convergenceIter.changes,
            lastChange=convergenceIter.when )
      convergenceStep = cfp2DcoStatus.netModBiasFaultInfo.event[
         'convergenceStep' ].stat
      baseModel.modulatorConvergenceStep = \
         InterfacesTransceiverStatusValueFormatInt(
            state=int( convergenceStep.value ),
            changes=convergenceStep.changes,
            lastChange=convergenceStep.when )
      faultCode = cfp2DcoStatus.netModBiasFaultInfo.event[
         'faultCode' ].stat
      baseModel.modulatorBiasFaultCode = \
         InterfacesTransceiverStatusValueFormatHex(
            state=faultCode.value,
            changes=faultCode.changes,
            lastChange=faultCode.when )

      # Vendor specific fault status
      vendorSpecificFAWSAttrs = [ 'bootEepromCRCFault', 'devInitFault',
         'ctrlProcImgABBootFault', 'fwHwMismatchFault', 'powerSupplyRWFault',
         'cryptoDevRWFault', 'cpldDevRWFault', 'viMonitorDevRWFault',
         'dacRefClockRWFault' ]
      baseModel.vendorSpecificFaultStatus = \
         InterfacesTransceiverStatusValueFormatVendorSpecificFaultStatus()
      populateModel( baseModel.vendorSpecificFaultStatus, cfp2DcoStatus,
                     vendorSpecificFAWSAttrs, 'vendorSpecificFAWSDec', True,
                     'fault', 'ok' )
      vendorSpecificFAWS2Attrs = [ 'ctrlProcIntCommFault', 'modemProcIntCommFault',
         'opticsProcIntCommFault', 'txITLAFault', 'rxITLAFault',
         'adcCalibrationFault', 'dacCalibrationFault', 'picEepromRWFault',
         'picEepromCRCfault', 'apexRWFault', 'picXTiaRWFault', 'picYTiaRWFault',
         'picDriverRWFault' ]
      populateModel( baseModel.vendorSpecificFaultStatus, cfp2DcoStatus,
                     vendorSpecificFAWS2Attrs, 'vendorSpecificFAWS2', True,
                     'fault', 'ok' )

      # Laser fault codes
      rxLaserFaultCode = cfp2DcoStatus.rxLaserFaultCode.event[
         'rxLaserFaultCode' ].stat
      baseModel.rxLaserFaultCode = InterfacesTransceiverStatusValueFormatHex(
         state=rxLaserFaultCode.value,
         changes=rxLaserFaultCode.changes,
         lastChange=rxLaserFaultCode.when )
      txLaserFaultCode = cfp2DcoStatus.txLaserFaultCode.event[
         'txLaserFaultCode' ].stat
      baseModel.laserFaultCode = InterfacesTransceiverStatusValueFormatHex(
         state=txLaserFaultCode.value,
         changes=txLaserFaultCode.changes,
         lastChange=txLaserFaultCode.when )

   return baseModel

def getXcvrLaneStats( xcvrStatus, channels ):
   chValues = {}
   flagsSupported = xcvrStatus.isOpticalDomSupported( xcvrStatus.mediaType )
   renderOnlyRxLosTxFault = displayOnlyRxLosTxFaultPinStates( xcvrStatus )
   for laneId in channels:
      laneStats = xcvrStatus.laneStats.get( laneId )
      laneStatsEnt = xcvrStatus.laneStatsEnt.get( laneId )
      if ( ( flagsSupported or True in renderOnlyRxLosTxFault ) and
            laneStats is not None ):
         chSpecific = InterfacesTransceiverStatusChannelSpecific()

         if flagsSupported or renderOnlyRxLosTxFault.rx_los:
            chSpecific.rxLos = InterfacesTransceiverStatusValueFormatBool(
               state=laneStats.currentRxLosState,
               changes=laneStats.rxLosEventsSinceInsertion,
               lastChange=laneStats.lastRxLosEvent )

         if flagsSupported or renderOnlyRxLosTxFault.tx_fault:
            chSpecific.txFault = InterfacesTransceiverStatusValueFormatBool(
               state=laneStats.currentTxFaultState,
               changes=laneStats.txFaultEventsSinceInsertion,
               lastChange=laneStats.lastTxFaultEvent )

         # The following parameters should be displayed only if
         # flagsSuppported is set irrespective of renderOnlyRxLosTxFault values.
         if flagsSupported:
            attrs = [ 'rxCdrLol', 'txPowerHiAlarm', 'txPowerLoAlarm',
                      'txPowerHiWarn', 'txPowerLoWarn', 'txBiasHiAlarm',
                      'txBiasLoAlarm', 'txBiasHiWarn', 'txBiasLoWarn',
                      'rxPowerHiAlarm', 'rxPowerLoAlarm', 'rxPowerHiWarn',
                      'rxPowerLoWarn' ]

            if xcvrStatus.xcvrType in [ "osfp", "qsfpDd" ]:
               for attr in attrs:
                  cur = getattr( laneStats, attr )
                  val = InterfacesTransceiverStatusValueFormatBool(
                              state=bool( cur.current ),
                              changes=cur.changes,
                              lastChange=cur.lastChange )
                  setattr( chSpecific, attr, val )
            elif laneStatsEnt is not None:
               for attr in attrs:
                  cur = getattr( laneStatsEnt, attr )
                  val = InterfacesTransceiverStatusValueFormatBool(
                              state=bool( cur.current ),
                              changes=cur.changes,
                              lastChange=cur.lastChange )
                  setattr( chSpecific, attr, val )

         chValues[ laneId ] = chSpecific
   return chValues

def getHostLaneStats( xcvrStatus, hostLanes ):
   hlValues = {}
   flagsSupported = xcvrStatus.isOpticalDomSupported( xcvrStatus.mediaType )
   for laneId in hostLanes:
      laneStats = xcvrStatus.laneStats.get( laneId )
      laneStatsEnt = xcvrStatus.laneStatsEnt.get( laneId )
      if flagsSupported and laneStats is not None:
         hlSpecific = InterfacesTransceiverStatusHostLaneSpecific()

         attrs = [ 'txCdrLol', 'txLos', 'txAdaptiveInputEqFault' ]
         if xcvrStatus.xcvrType in [ "osfp", "qsfpDd" ]:
            for attr in attrs:
               cur = getattr( laneStats, attr )
               val = InterfacesTransceiverStatusValueFormatBool(
                           state=bool( cur.current ),
                           changes=cur.changes,
                           lastChange=cur.lastChange )
               setattr( hlSpecific, attr, val )
         elif laneStatsEnt is not None:
            for attr in attrs:
               cur = getattr( laneStatsEnt, attr )
               val = InterfacesTransceiverStatusValueFormatBool(
                           state=bool( cur.current ),
                           changes=cur.changes,
                           lastChange=cur.lastChange )
               setattr( hlSpecific, attr, val )

         hlValues[ laneId ] = hlSpecific
   return hlValues

def _populateIntfSpecificValues( model, intfName, intfMapping, primaryIntf ):
   intfSpecific = InterfacesTransceiverStatusInterfaceSpecific()
   laneMap = intfMapping.intfMap
   hostMap = intfMapping.hostMap
   portLaneMapOverride = intfMapping.portLaneMapOverride
   intfNameOverride = intfMapping.intfNameOverride
   slotName = getXcvrSlotName( intfName )
   xcvrStatus = getXcvrStatus( xcvrStatusDir_.xcvrStatus[ slotName ] )
   intfLanes = laneMap[ intfName ].get( intfName )

   intfLanes = portLaneMapOverride.get( slotName, {} ).get( intfName, intfLanes )
   hostLanes = hostMap[ intfName ]

   primaryEpis = None
   if primaryIntf in intfMapping.epis:
      primaryEpis = intfMapping.epis[ primaryIntf ]
   else:
      primaryEpis = intfMapping.epis[ getXcvrSlotName( primaryIntf ) ]
   if primaryEpis:
      speed = primaryEpis.speed
      intfSpecific.operSpeed = InterfacesTransceiverStatusValueFormatStr(
            state=ethSpeedToStr[ speed ] )

   xcvrLaneStats = getXcvrLaneStats( xcvrStatus, intfLanes )
   for laneId in xcvrLaneStats:
      intfSpecific.channels[ laneId + 1 ] = xcvrLaneStats[ laneId ]
   hostLaneStats = getHostLaneStats( xcvrStatus, hostLanes )
   for laneId in hostLaneStats:
      intfSpecific.hostLanes[ laneId + 1 ] = hostLaneStats[ laneId ]

   intfDisplayName = intfNameOverride.get( intfName ) or intfName
   intfSpecific.interface = InterfacesTransceiverStatusValueFormatStr(
         state=intfDisplayName )
   model.interfaces[ intfDisplayName ] = intfSpecific

   return model

def _populateCoherentIntfSpecificValues( model, intfName ):
   status = xcvrStatusDir_.xcvrStatus[ getXcvrSlotName( intfName ) ]
   intfSpecific = InterfacesTransceiverStatusExtCfp2DcoIntf()

   if not _isDualLaserModule( status ):
      if status.mediaType == XcvrMediaType.xcvr100GDwdmDco:
         lane = 1
         if status.xcvrConfig.intfName.get( 0 ) == intfName:
            lane = 0
         pcsAlarmState = getattr( status.cfp2DcoNewStatus,
                                  'clientPcsAlarmStatus%d' % lane )
         statusModel = InterfacesTransceiverStatusValueFormatHex(
            state=pcsAlarmState )
         intfSpecific.pcsAlarmStatus = statusModel
         model.interfaces[ intfName ].cfp2Dco = intfSpecific
      return model

   # The current field only applies for DP04 DCO modules
   intfSpecific.dualLaserModulePresentIs( True )
   cfp2DcoStatus = status.cfp2DcoNewStatus
   clientCfg = Tac.Value( 'Cfp2DcoHal::Cfp2DcoReg::GeneralModeControlLsb',
                          cfp2DcoStatus.generalModeControlLsb ).clientCfg
   if clientCfg == ClientConfig.mode400Gaui8:
      laneCount = 8
      fec = 'reed-solomon'
      rsFecCodewordSize = 544
   elif clientCfg == ClientConfig.mode200Gaui4:
      laneCount = 4
      fec = 'reed-solomon'
      rsFecCodewordSize = 544
   elif clientCfg == ClientConfig.mode100Gaui2:
      laneCount = 8
      fec = 'reed-solomon'
      rsFecCodewordSize = 544
   elif clientCfg == ClientConfig.mode100Caui4RsFec:
      laneCount = 4
      fec = 'reed-solomon'
      rsFecCodewordSize = 528
   elif clientCfg == ClientConfig.mode100Caui4NoFec:
      laneCount = 4
      fec = 'none'
      rsFecCodewordSize = None
   else:
      raise Exception(
         'Invalid ClientConfig type identified: {}'.format( clientCfg ) )
   intfSpecific.laneCount = InterfacesTransceiverStatusValueFormatInt(
      state=laneCount )
   intfSpecific.fec = InterfacesTransceiverStatusValueFormatStr(
      state=fec )
   intfStatus = cfp2DcoStatus.cfp2DcoIntfStatus[ intfName ]
   if rsFecCodewordSize:
      intfSpecific.rsFecCodewordSize = InterfacesTransceiverStatusValueFormatInt(
         state=rsFecCodewordSize )
      clientFecUcb = intfStatus.clientFecUcb.event[ 'ucb' ].stat
      intfSpecific.fecUncorrectedBlocks = InterfacesTransceiverStatusValueFormatInt(
         state=clientFecUcb.value,
         changes=clientFecUcb.changes,
         lastChange=clientFecUcb.when )
      intfSpecific.preFecBer = InterfacesTransceiverStatusValueFormatStr(
         state='%.2e' % intfStatus.clientTxBerPm )

   def populateClientPcsModel( status, model, attrs ):
      for ( attr, intfXcvrStatusValueFormat ) in attrs:
         # Construct model attribute name
         alarmStatStr = attr.name.replace( 'ingress', '' ).replace( 'egress', '' )
         cliModelAttrName = alarmStatStr[ 0 ].lower() + alarmStatStr[ 1: ]
         attrValue = status.event[ attr.name ].stat
         attrModel = intfXcvrStatusValueFormat(
            state=attr.trueState if attrValue.current else attr.falseState,
            changes=attrValue.changes,
            lastChange=attrValue.lastChange )
         setattr( model, cliModelAttrName, attrModel )

   intfSpecific.ingressAlarmStatus = \
      InterfacesTransceiverStatusIntfValueFormatIngressAlarmStatus()
   intfSpecific.egressAlarmStatus = \
      InterfacesTransceiverStatusIntfValueFormatEgressAlarmStatus()
   fmtBool = InterfacesTransceiverStatusValueFormatBool
   fmtStr = InterfacesTransceiverStatusValueFormatStr
   ingressClientPcsAttrs = [
      ( CoherentClientStatusAttr( 'ingressRxLocalFault', True, False ), fmtBool ),
      ( CoherentClientStatusAttr( 'ingressRxRemoteFault', True, False ), fmtBool ),
      ( CoherentClientStatusAttr( 'ingressBlockLock', 'no block lock', 'ok' ),
        fmtStr ),
      ( CoherentClientStatusAttr( 'ingressAlignmentMarkerLock', 'unaligned', 'ok' ),
        fmtStr ),
      ( CoherentClientStatusAttr( 'ingressBipErrorsDetected', 'detected', 'ok' ),
        fmtStr ),
      ( CoherentClientStatusAttr( 'ingressErroredBlocksDetected', 'detected', 'ok' ),
        fmtStr ),
      ( CoherentClientStatusAttr( 'ingressLoa', 'unaligned', 'ok' ), fmtStr ),
      ( CoherentClientStatusAttr( 'ingressLos', 'no signal', 'ok' ), fmtStr ) ]
   egressClientPcsAttrs = [
      ( CoherentClientStatusAttr( 'egressHighBer', 'high BER', 'ok' ), fmtStr ),
      ( CoherentClientStatusAttr( 'egressRxLocalFault', True, False ), fmtBool ),
      ( CoherentClientStatusAttr( 'egressRxRemoteFault', True, False ), fmtBool ),
      ( CoherentClientStatusAttr( 'egressBlockLock', 'no block lock', 'ok' ),
        fmtStr ),
      ( CoherentClientStatusAttr( 'egressAlignmentMarkerLock', 'unaligned', 'ok' ),
        fmtStr ),
      ( CoherentClientStatusAttr( 'egressBipErrorsDetected', 'detected', 'ok' ),
        fmtStr ),
      ( CoherentClientStatusAttr( 'egressErroredBlocksDetected', 'detected', 'ok' ),
        fmtStr ),
      ( CoherentClientStatusAttr( 'egressLoa', 'unaligned', 'ok' ),fmtStr ), ]
   for ( client, attrs ) in [ ( 'ingressAlarmStatus', ingressClientPcsAttrs ),
                              ( 'egressAlarmStatus', egressClientPcsAttrs ) ]:
      clientPcsModel = getattr( intfSpecific, client )
      populateClientPcsModel( intfStatus.clientPcsAlarmStatus, clientPcsModel,
                              attrs )
   model.interfaces[ intfName ].cfp2Dco = intfSpecific
   return model

def showInterfacesXcvrStatus( mode, args ):
   intf = args.get( 'INTF' )
   mod = args.get( 'MOD' )
   model = InterfacesTransceiverStatus()

   ( intfs, _ ) = getAllIntfsWrapper( mode, intf, mod )
   if not intfs:
      return model

   intfMapping = IntfMapping( intfs )
   laneMap = intfMapping.intfMap
   if not laneMap:
      return model
   hostMap = intfMapping.hostMap
   portLaneMapOverride = intfMapping.portLaneMapOverride

   for intfToDisplay in Arnet.sortIntf( laneMap ):
      for intfName in Arnet.sortIntf( laneMap[ intfToDisplay ] ):
         key = intfName
         if portLaneMapOverride.get( intfToDisplay, {} ).get( intfName ):
            key = intfToDisplay
         portName = getXcvrSlotName( key )
         portDisplayName = re.sub( 'Ethernet|Management', 'Port ', portName )
         if portDisplayName not in model.ports:
            xcvrModel = _populateXcvrStatusValues( intfToDisplay )
            xcvrModel.cfp2Dco = _populateCoherentXcvrStatusValues( intfToDisplay )

            lanes = portLaneMapOverride.get( portName, {} ).get( 'xcvr' )
            if lanes is not None:
               xcvrStatus = getXcvrStatus( xcvrStatusDir_.xcvrStatus[ portName ] )
               xcvrLaneStats = getXcvrLaneStats( xcvrStatus, lanes )
               for laneId in xcvrLaneStats:
                  xcvrModel.channels[ laneId + 1 ] = xcvrLaneStats[ laneId ]
               hostLanes = hostMap[ portName ]
               hostLaneStats = getHostLaneStats( xcvrStatus, hostLanes )
               for laneId in hostLaneStats:
                  xcvrModel.hostLanes[ laneId + 1 ] = hostLaneStats[ laneId ]
            model.ports[ portDisplayName ] = xcvrModel

         _populateIntfSpecificValues( model.ports[ portDisplayName ], intfName,
                                      intfMapping, intfToDisplay )
         _populateCoherentIntfSpecificValues( model.ports[ portDisplayName ],
                                              intfName )
   return model

class TransceiverStatusCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show transceiver status [ interface INTF ]'
   data = {
      'transceiver' : xcvrShowKw,
      'status' : 'Show transceiver status information',
      'interface' : 'Show internal interface state',
      'INTF' : Intf.rangeMatcher,
   }
   handler = showInterfacesXcvrStatus
   cliModel = InterfacesTransceiverStatus

BasicCli.addShowCommandClass( TransceiverStatusCmd )


#------------------------------------------------------
# "show tech-support" plugin
#------------------------------------------------------
def _showTechCmds():
   return [ 'show interfaces transceiver detail' ]

CliPlugin.TechSupportCli.registerShowTechSupportCmdCallback(
   '2014-02-05 16:54:34', _showTechCmds,
   summaryCmdCallback=lambda: [ 'show interfaces transceiver' ] )

def _showTechCmdsAddon():
   return [ 'show interfaces transceiver tuning detail',
            'show transceiver status' ]

CliPlugin.TechSupportCli.registerShowTechSupportCmdCallback(
   '2018-06-18 16:58:26', _showTechCmdsAddon )

def _showTechCmdsAddon2():
   return [ 'show interfaces transceiver dom',
            'show interfaces transceiver dom thresholds' ]

CliPlugin.TechSupportCli.registerShowTechSupportCmdCallback(
   '2020-09-04 18:00:00', _showTechCmdsAddon2 )

#------------------------------------------------------
# Plugin method
#------------------------------------------------------

def Plugin( em ):
   global xgc, entityManager_
   global xcvrCliConfigSliceDir
   global xcvrConfigDir_
   global xcvrStatusDir_
   entityManager_ = em
   xgc = ConfigMount.mount( em, "hardware/xcvr/xgc", "Xcvr::Xgc", "w" )
   xcvrCliConfigSliceDir = ConfigMount.mount( em, "hardware/xcvr/cli/config/slice",
                                              "Tac::Dir", "wi" )
   xcvrStatusDir_ = CliPlugin.XcvrAllStatusDir.xcvrAllStatusDir( em )
   xcvrConfigDir_ = xcvrStatusDir_.baseConfigDir_

