# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.
from __future__ import absolute_import
import Arnet
import Tac
from CliModel import Model, Enum, Dict, Int, Bool, Str, Submodel, List
from CliPlugin.IntfModel import Interface
import CliPlugin.IntfCli as IntfCli
import CliPlugin.EthIntfModel as EthIntfModel

autonegSpeed = { 'autoSpeed' : 'Auto-speed',
                 'auto' : 'auto' }
autonegSuffix = { 'forced' : 'forced',
                  'autoNegotiated' : 'auto-negotiated' }
autonegDuplex = { 'autoDuplex' : 'Auto-duplex',
                  'auto' : 'auto' }
class TransceiverSpeedLanesDuplex( Model ):
   speed = Enum( values=EthIntfModel.CapabilitiesMixIn.speedValues + 
                        autonegSpeed.keys(),
                 optional=True, help='Interface speed' )
   laneCount = Int( optional=True, help='Number of lanes on the interface' )
   duplex = Enum( values=[ 'full', 'half', 'unconfigured' ] + autonegDuplex.keys(),
                  optional=True,
                  help='Duplex communication capabilities with this speed' )
   _showLanes = Bool( optional=True, help='Lane count is set' )
   autonegSuffix = Enum( values=autonegSuffix.keys(), optional=True,
                         help='Speed forced or auto-negotiated' )
   _autonegActive = Bool( optional=True, help='Whether auto-negotiation is active' )
   def showLanesIs( self, showLanes ):
      self._showLanes = showLanes

   def showLanes( self ):
      return self._showLanes

   def autonegActiveIs( self, autonegActive ):
      self._autonegActive = autonegActive

   def autonegActive( self ):
      return self._autonegActive

class TransceiverCapabilities( Model ):
   speedCapabilities = Submodel( valueType=EthIntfModel.EthSpeedLanesDuplex, 
                                 optional=True,
                                 help='Transceiver speed capabilities' )

class TransceiverProperties( Model ):
   adminSpeed = Submodel( valueType=TransceiverSpeedLanesDuplex, optional=True,
                          help='Administrative speed/lane count/duplex' )
   operSpeed = Submodel( valueType=TransceiverSpeedLanesDuplex, optional=True,
                         help='Operational speed/lane count/duplex' )
   mediaType = Str( optional=True, help='Transceiver type' )
   _transceiverPresence = Bool( optional=True, help = 'Transceiver is present' )
   xcvrCapabilities = List( valueType=TransceiverCapabilities, optional=True,
                            help='Transceiver capabilities' )
   _isSwappableXcvr = Bool( optional=True, help='Transceiver is swappable' )

   def isSwappableXcvrIs( self, isSwappableXcvr ):
      self._isSwappableXcvr = isSwappableXcvr

   def isSwappableXcvr( self ):
      return self._isSwappableXcvr 

   def transceiverPresenceIs( self, xcvrPresence ):
      self._transceiverPresence = xcvrPresence

   def transceiverPresence( self ):
      return self._transceiverPresence

   def speedLanesDuplexToXcvrModel( self, speed, lanes, duplex, showLanes,
                                    autonegActive ):
      tsd = TransceiverSpeedLanesDuplex()
      tsd.autonegActiveIs( autonegActive )
      if autonegActive and speed in autonegSpeed:
         tsd.speed = speed
      else:
         tsd.speed = EthIntfModel.CapabilitiesMixIn.ethSpeedToSpeedValue( speed )
         tsd.showLanesIs( showLanes )
      if tsd.showLanes():
         tsd.laneCount = int( EthIntfModel.CapabilitiesMixIn.ethLaneCountToValue( 
                              lanes ) )
      tsd.duplex = duplex
      return tsd

   def adminSpeedIs( self, speed, lanes, duplex, showLanes, autonegActive ):
      self.adminSpeed = self.speedLanesDuplexToXcvrModel( speed, lanes, duplex,
                                                          showLanes, autonegActive )

   def operSpeedIs( self, speed, lanes, duplex, showLanes, autonegActive,
                    autonegSuffixStr ):
      tsd = TransceiverSpeedLanesDuplex()
      tsd = self.speedLanesDuplexToXcvrModel( speed, lanes, duplex,
                                              showLanes, autonegActive )
      if autonegSuffixStr:
         tsd.autonegSuffix = autonegSuffixStr
      self.operSpeed = tsd

   def formatSpeedStr( self, speed, lanes, showLanes, autonegActive ):
      # Get the speed string according to the info of the speed and laneCount. Ex, 
      # if speed='400Gbps', showLanes=True and lanes=8, the output will be '400G-8'. 
      # if speed='Unknown', the output will be 'unconfigured'.
      if autonegActive and speed in autonegSpeed:
         return speed
      if speed.endswith( 'bps' ):
         tmpStr = speed[ :-len( 'bps' ) ].replace( 'p', '.' )
      else:
         tmpStr = speed.replace( 'p', '.' )
      if lanes and showLanes:
         tmpStr = tmpStr + '-' + str( lanes )
      if tmpStr == 'Unknown':
         tmpStr = 'unconfigured'
      return tmpStr

   def printAdminSpeed( self, speed ):
      return self.formatSpeedStr( speed.speed, speed.laneCount,
                                  speed.showLanes(), speed.autonegActive() )

   def printOperSpeed( self, operSpeed ):
      speedStr = self.formatSpeedStr( operSpeed.speed, operSpeed.laneCount,
                                      operSpeed.showLanes(),
                                      operSpeed.autonegActive() )
      if operSpeed.autonegSuffix in autonegSuffix:
         speedStr += '(' + autonegSuffix[ operSpeed.autonegSuffix ] + ')'
      return speedStr

   def printOperDuplex( self, operSpeed ):
      operDuplex = operSpeed.duplex
      if operSpeed.autonegSuffix in autonegSuffix:
         operDuplex += '(' + autonegSuffix[ operSpeed.autonegSuffix ] + ')'
      return operDuplex
   def _getXcvrCapList( self, linkModeCaps ):
      '''
      input: List of linkModeCapabilities
      output: List of TransceiverCapabilities
      '''
      speedModel = []
      for speedCap in linkModeCaps:
         ethSpeed = EthIntfModel.EthSpeedLanesDuplex()
         xcvrCap = TransceiverCapabilities()
         speedCapSpeed = getattr( speedCap, 'speed', 'speedUnknown' )
         ethSpeed.speed = EthIntfModel.CapabilitiesMixIn.ethSpeedToSpeedValue( 
                          speedCapSpeed )
         speedCapDuplex = getattr( speedCap, 'duplex', 'duplexUnknown' )
         ethSpeed.duplex = speedCapDuplex.lstrip( 'duplex' ).lower()
         ethSpeed.showLanesIs( getattr( speedCap, 'showLanes', False ) )
         if ethSpeed.showLanes():
            ethSpeed.laneCount = int( 
                                 EthIntfModel.CapabilitiesMixIn.ethLaneCountToValue(
                                 speedCap.lanes ) )
         xcvrCap.speedCapabilities = ethSpeed
         if xcvrCap not in speedModel:
            speedModel.append( xcvrCap )
      return speedModel

   def xcvrCapListIs( self, linkModeCaps ):
      if linkModeCaps is None or linkModeCaps == 'none':
         # If linkModeCaps is None:
         # The speed capabilities is empty. Ex, the linkModeCapabilities is empty
         # for fixed 10GBASE-T. We won't print speed capabilities since no 
         # transceiver is installed. 
         # If the linkModeCaps == 'none':
         # There is no available linkModeCapabilities. Ex, for 100GBASE-SRBD,
         # only the '/1' has the availble linkModeCapabilities. We will print
         # 'none' for speed capabilities on '/2-/4' lanes.
         self.xcvrCapabilities = []
      else:
         xcvrCapList = self._getXcvrCapList( linkModeCaps )
         self.xcvrCapabilities = xcvrCapList

   def printSpeedCapabilities( self, speedCapList ):
      if not speedCapList:
         speedStr = 'none'
      else:
         speedStr = ''
         for speedCaps in speedCapList:
            speedCap = speedCaps.speedCapabilities
            speedCapStr = EthIntfModel.CapabilitiesMixIn.formatSpeedLanesDuplex( 
                          speedCap.speed, speedCap.laneCount, speedCap.showLanes, 
                          speedCap.duplex, False, False )
            if speedCapStr not in speedStr:
               if not speedStr:
                  speedStr += speedCapStr
               else:
                  speedStr = speedStr + ',' + speedCapStr
      return speedStr

   def renderModel( self, intfName ):
      print 'Name: %s' % IntfCli.Intf.getShortname( intfName )
      print 'Administrative speed: %s' % self.printAdminSpeed( self.adminSpeed )
      print 'Administrative duplex: %s' % self.adminSpeed.duplex
      print 'Operational speed: %s' % self.printOperSpeed( self.operSpeed )
      print 'Operational duplex: %s' % self.printOperDuplex( self.operSpeed )
      print 'Media type: %s' % self.mediaType
      # If it is a 10GBASE-T fixed port, we will not print speedCapabilities, 
      # since there is no transceiver there.
      if self._transceiverPresence and self._isSwappableXcvr:
         print 'Speed capabilities: %s' % self.printSpeedCapabilities(
                                          self.xcvrCapabilities )
class TransceiverPropertiesBase( Model ):
   interfaces = Dict( keyType=Interface, valueType=TransceiverProperties,
                      help='Mapping between interface name and the'
                           ' transceiver properties' )
   def render( self ):
      for intf in Arnet.sortIntf( self.interfaces ):
         self.interfaces[ intf ].renderModel( intf )

