#!/usr/bin/env python
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

import BasicCli
import CliCommand
import CliMatcher
import CliPlugin.BridgingCli as BridgingCli
import CliPlugin.EthIntfCli as EthIntfCli
import CliPlugin.IpAddrMatcher as IpAddrMatcher
import CliPlugin.IpGenAddrMatcher as IpGenAddrMatcher
import CliPlugin.LagIntfCli as LagIntfCli
import CliPlugin.MacAddr as MacAddr
import CliPlugin.MlagShowCli as MlagShowCli
import CliPlugin.VlanCli as VlanCli
import CliPlugin.VrfCli as VrfCli
import CliPlugin.VxlanCli as VxlanCli
import CliPlugin.VxlanModel as VxlanModel
import Intf.IntfRange as IntfRange
import ShowCommand

matcherAddress = CliMatcher.KeywordMatcher( 'address',
      helpdesc='Filter by MAC address' )
matcherArp = CliMatcher.KeywordMatcher( 'arp',
      helpdesc='ARP entries' )
matcherController = CliMatcher.KeywordMatcher( 'controller',
      helpdesc='VXLAN control service' )
matcherCounters = CliMatcher.KeywordMatcher( 'counters',
      helpdesc='VXLAN Counters' )
matcherIp = CliMatcher.KeywordMatcher( 'ip',
      helpdesc='IP address' )
matcherLearnRestrict = CliMatcher.KeywordMatcher( 'learn-restrict',
      helpdesc='VXLAN learning restrictions' )
matcherName = CliMatcher.KeywordMatcher( 'name',
      helpdesc='Specify logical router name' )
matcherVlan = CliMatcher.KeywordMatcher( 'vlan',
      helpdesc='VLAN configuration' )
matcherVtep = CliMatcher.KeywordMatcher( 'vtep',
      helpdesc='VXLAN Tunnel End Points' )
nodeLogicalRouter = CliCommand.guardedKeyword( 'logical-router',
      helpdesc='Logical router status',
      guard=VxlanCli.vxlanRoutingSupportedGuard )
nodeVxlan = CliCommand.guardedKeyword( 'vxlan',
      helpdesc='Configure VXLAN parameters',
      guard=VxlanCli.vxlanSupportedGuard )
arpIpMatcher = IpGenAddrMatcher.IpGenAddrMatcher(
      helpdesc='IP address (or prefix) of remote VTEP',
      helpdesc4='ARP entry\'s IPv4 address',
      helpdesc6='ARP entry\'s IPv4 address' )
logicalRouterMatcher = CliMatcher.QuotedStringMatcher(
      helpdesc='Logical router name', helpname='WORD' )
directionMatcher = CliMatcher.EnumMatcher( {
      'encap': 'Encapsulation counters',
      'decap': 'Decapsulation counters',
   } )
vrfExprFactory = VrfCli.VrfExprFactory(
      helpdesc='VRF name' )

#--------------------------------------------------------------------------------
# show vxlan address-table [ mlag-peer ] [ HOST_TYPE ]
#                                        [ address MAC_ADDR ] 
#                                        [ vlan VLAN_ID ]
#                                        [ vtep { VTEP } ]
#--------------------------------------------------------------------------------
class VxlanAddressTableCmd( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show vxlan address-table [ mlag-peer ] [ HOST_TYPE ] '
                                                     '[ address MAC_ADDR ] '
                                                     '[ vlan VLAN_ID ] '
                                                     '[ vtep { VTEP } ]' )
   data = {
      'vxlan': nodeVxlan,
      'address-table': BridgingCli.matcherAddressTable,
      'mlag-peer': CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher( 'mlag-peer',
            helpdesc='MAC addresses learned from MLAG peer' ),
         guard=MlagShowCli.mlagSupportedGuard ),
      'address': matcherAddress,
      'HOST_TYPE': CliMatcher.EnumMatcher( {
         'received': 'Received MAC addresses',
         'evpn': 'EVPN Received MAC addresses',
         'static': 'Static entry type',
         'dynamic': 'Dynamic entry type',
         'unicast': 'Unicast entry type',
         'configured': 'Configured MAC entries',
      } ),
      'MAC_ADDR': MacAddr.macAddrMatcher,
      'vlan': BridgingCli.matcherVlan,
      'VLAN_ID': VlanCli.vlanIdMatcher,
      'vtep': matcherVtep,
      'VTEP': IpGenAddrMatcher.IpGenAddrMatcher( helpdesc='Vtep IP address',
                                               helpdesc4="Vtep's IPv4 address",
                                               helpdesc6="Vtep's IPv6 address" ),
   }
   handler = VxlanCli.showVxlanMacAddr
   cliModel = VxlanModel.VxlanMacAddresses

BasicCli.addShowCommandClass( VxlanAddressTableCmd )

#--------------------------------------------------------------------------------
# show vxlan address-table count [ vtep { VTEP } ]
#--------------------------------------------------------------------------------
class VxlanAddressTableCountCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show vxlan address-table count [ vtep { VTEP } ]'
   data = {
      'vxlan': nodeVxlan,
      'address-table': BridgingCli.matcherAddressTable,
      'count': 'Count of MAC addresses per VTEP',
      'vtep': matcherVtep,
      'VTEP': IpGenAddrMatcher.IpGenAddrMatcher( helpdesc='Vtep IP address',
                                               helpdesc4="Vtep's IPv4 address",
                                               helpdesc6="Vtep's IPv6 address" ),
   }
   handler = VxlanCli.showVxlanMacAddrCount
   cliModel = VxlanModel.VxlanAddressTableCount

BasicCli.addShowCommandClass( VxlanAddressTableCountCmd )

#--------------------------------------------------------------------------------
# show vxlan config-sanity [ brief | detail ]
#--------------------------------------------------------------------------------
class VxlanConfigSanityCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show vxlan config-sanity [ brief | detail ]'
   data = {
      'vxlan': nodeVxlan,
      'config-sanity': 'VXLAN Config Sanity',
      'brief': 'Show brief information',
      'detail': 'More comprehensive output'
   }
   handler = VxlanCli.showConfigSanity
   cliModel = VxlanModel.ConfigSanityModel

BasicCli.addShowCommandClass( VxlanConfigSanityCmd )

#--------------------------------------------------------------------------------
# show vxlan controller address-table advertised [ vlan VLAN_ID ]
#                                                [ address MAC_ADDR ]
#--------------------------------------------------------------------------------
class VxlanControllerAddressTableAdvertisedCmd( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show vxlan controller address-table advertised [ vlan VLAN_ID ] '
                                                            '[ address MAC_ADDR ]' )
   data = {
      'vxlan': nodeVxlan,
      'controller': matcherController,
      'address-table': BridgingCli.matcherAddressTable,
      'address': matcherAddress,
      'advertised': 'Advertised MAC addresses',
      'vlan': BridgingCli.matcherVlan,
      'VLAN_ID': VlanCli.vlanIdMatcher,
      'MAC_ADDR': MacAddr.macAddrMatcher,
   }
   handler = VxlanCli.showVxlanVniStatusAdvertised
   cliModel = VxlanModel.VxlanVniStatusDirModel

BasicCli.addShowCommandClass( VxlanControllerAddressTableAdvertisedCmd )

#--------------------------------------------------------------------------------
# show vxlan controller address-table received [ vlan VLAN_ID ] 
#                                              [ address MAC_ADDR ] [ vtep { VTEP } ]
#--------------------------------------------------------------------------------
class VxlanControllerAddressTableReceivedCmd( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show vxlan controller address-table received [ vlan VLAN_ID ] '
                                          '[ address MAC_ADDR ] [ vtep { VTEP } ]' )
   data = {
      'vxlan': nodeVxlan,
      'controller': matcherController,
      'address-table': BridgingCli.matcherAddressTable,
      'address': matcherAddress,
      'received': 'Received MAC addresses',
      'vlan': BridgingCli.matcherVlan,
      'VLAN_ID': VlanCli.vlanIdMatcher,
      'MAC_ADDR': MacAddr.macAddrMatcher,
      'vtep': matcherVtep,
      'VTEP': IpAddrMatcher.IpAddrMatcher( helpdesc='IP address of remote VTEP' ),
   }
   handler = VxlanCli.showVxlanVniStatusReceived
   cliModel = VxlanModel.VxlanVniStatusDirModel

BasicCli.addShowCommandClass( VxlanControllerAddressTableReceivedCmd )

#--------------------------------------------------------------------------------
# show vxlan controller arp advertised [ VRF ] [ vlan VLAN_ID ] [ ip IP ]
#                                                               [ address MAC_ADDR ]
#--------------------------------------------------------------------------------
class VxlanControllerArpAdvertisedCmd( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show vxlan controller arp advertised [ VRF ] '
                                 '[ vlan VLAN_ID ] [ ip IP ] [ address MAC_ADDR ]' )
   data = {
      'vxlan': nodeVxlan,
      'controller': matcherController,
      'arp': matcherArp,
      'advertised': 'Advertised ARP entries',
      'VRF': vrfExprFactory,
      'vlan': BridgingCli.matcherVlan,
      'VLAN_ID': VlanCli.vlanIdMatcher,
      'ip': matcherIp,
      'IP': arpIpMatcher,
      'address': matcherAddress,
      'MAC_ADDR': MacAddr.macAddrMatcher,
   }
   handler = VxlanCli.showVxlanIpToMacAdvertised
   cliModel = VxlanModel.VxlanArpTableModel

BasicCli.addShowCommandClass( VxlanControllerArpAdvertisedCmd )

#--------------------------------------------------------------------------------
# show vxlan controller arp received [ VRF ] [ vlan VLAN_ID ] [ ip IP ]
#                                                               [ address MAC_ADDR ]
#--------------------------------------------------------------------------------
class VxlanControllerArpReceivedCmd( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show vxlan controller arp received [ VRF ] [ vlan VLAN_ID ] '
                                                 '[ ip IP ] [ address MAC_ADDR ]' )
   data = {
      'vxlan': nodeVxlan,
      'controller': matcherController,
      'arp': matcherArp,
      'received': 'Received ARP entries',
      'VRF': vrfExprFactory,
      'vlan': BridgingCli.matcherVlan,
      'VLAN_ID': VlanCli.vlanIdMatcher,
      'ip': matcherIp,
      'IP': arpIpMatcher,
      'address': matcherAddress,
      'MAC_ADDR': MacAddr.macAddrMatcher,
   }
   handler = VxlanCli.showVxlanIpToMacReceived
   cliModel = VxlanModel.VxlanArpTableModel

BasicCli.addShowCommandClass( VxlanControllerArpReceivedCmd )

#--------------------------------------------------------------------------------
# show vxlan controller logical-router [ { ( vni VNI ) | ( name LOGICAL_ROUTER ) } ]
#--------------------------------------------------------------------------------
class VxlanControllerLogicalRouterCmd( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show vxlan controller logical-router '
                                    '[ { ( vni VNI ) | ( name LOGICAL_ROUTER ) } ]' )
   data = {
      'vxlan': nodeVxlan,
      'controller': matcherController,
      'logical-router': nodeLogicalRouter,
      'vni': CliCommand.singleNode( VxlanCli.vniMatcherForConfig ),
      'VNI': VxlanCli.vniMatcher,
      'name': CliCommand.singleNode( matcherName ),
      'LOGICAL_ROUTER': logicalRouterMatcher
   }
   handler = VxlanCli.showVxlanLogicalRouter
   cliModel = VxlanModel.VxlanLogicalRouterModel

BasicCli.addShowCommandClass( VxlanControllerLogicalRouterCmd )

#--------------------------------------------------------------------------------
# show vxlan controller logical-router route [ name LOGICAL_ROUTER ]
#--------------------------------------------------------------------------------
class VxlanControllerLogicalRouterRouteCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show vxlan controller logical-router route [ name LOGICAL_ROUTER ]'
   data = {
      'vxlan': nodeVxlan,
      'controller': matcherController,
      'logical-router': nodeLogicalRouter,
      'route': 'Logical router route',
      'name': matcherName,
      'LOGICAL_ROUTER': logicalRouterMatcher
   }
   handler = VxlanCli.showVxlanRoutes
   cliModel = VxlanModel.VxlanLogicalRouterModel

BasicCli.addShowCommandClass( VxlanControllerLogicalRouterRouteCmd )

#--------------------------------------------------------------------------------
# show vxlan controller logical-router uplink [ name LOGICAL_ROUTER ]
#--------------------------------------------------------------------------------
class VxlanControllerLogicalRouterUplinkCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show vxlan controller logical-router uplink [ name LOGICAL_ROUTER ]'
   data = {
      'vxlan': nodeVxlan,
      'controller': matcherController,
      'logical-router': nodeLogicalRouter,
      'uplink': 'Logical router uplink ports',
      'name': matcherName,
      'LOGICAL_ROUTER': logicalRouterMatcher
   }
   handler = VxlanCli.showVxlanUplinkPorts
   cliModel = VxlanModel.VxlanLogicalRouterModel

BasicCli.addShowCommandClass( VxlanControllerLogicalRouterUplinkCmd )

#--------------------------------------------------------------------------------
# show vxlan controller status
#--------------------------------------------------------------------------------
class VxlanControllerStatusCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show vxlan controller status'
   data = {
      'vxlan': nodeVxlan,
      'controller': matcherController,
      'status': 'VXLAN control service status',
   }
   handler = VxlanCli.showVxlanControlServiceStatus
   cliModel = VxlanModel.VxlanControlServiceStatus

BasicCli.addShowCommandClass( VxlanControllerStatusCmd )

#--------------------------------------------------------------------------------
# show vxlan counters learn-restrict COUNTER [ vlan VLANS ]
#--------------------------------------------------------------------------------
class VxlanCountersLearnRestrictAllCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show vxlan counters learn-restrict COUNTER [ vlan VLANS ]'
   data = {
      'vxlan': nodeVxlan,
      'counters': matcherCounters,
      'learn-restrict': matcherLearnRestrict,
      'COUNTER': CliMatcher.EnumMatcher( {
         'all': 'Show all counters',
         'brief': 'Show brief information',
         'prefix': 'Match a prefix',
      } ),
      'vlan': matcherVlan,
      'VLANS': VlanCli.vlanSetMatcher,
   }
   handler = VxlanCli.showLearnCounters
   cliModel = VxlanModel.VxlanLearnCountersModel

BasicCli.addShowCommandClass( VxlanCountersLearnRestrictAllCmd )

#--------------------------------------------------------------------------------
# show vxlan flood vtep [ vlan VLANS ]
#--------------------------------------------------------------------------------
class VxlanFloodVtepCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show vxlan flood vtep [ vlan VLANS ]'
   data = {
      'vxlan': nodeVxlan,
      'flood': 'VXLAN flooding behavior',
      'vtep': matcherVtep,
      'vlan': matcherVlan,
      'VLANS': VlanCli.vlanSetMatcher,
   }
   handler = VxlanCli.showFloodVtep
   cliModel = VxlanModel.VxlanFloodVtepListModel

BasicCli.addShowCommandClass( VxlanFloodVtepCmd )

#--------------------------------------------------------------------------------
# show vxlan learn-restrict vtep [ vlan VLANS ]
#--------------------------------------------------------------------------------
class VxlanLearnRestrictVtepCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show vxlan learn-restrict vtep [ vlan VLANS ]'
   data = {
      'vxlan': nodeVxlan,
      'learn-restrict': matcherLearnRestrict,
      'vtep': matcherVtep,
      'vlan': matcherVlan,
      'VLANS': VlanCli.vlanSetMatcher,
   }
   handler = VxlanCli.showLearnRestrict
   cliModel = VxlanModel.VxlanLearnRestrictModel

BasicCli.addShowCommandClass( VxlanLearnRestrictVtepCmd )

#--------------------------------------------------------------------------------
# show vxlan counters software
#--------------------------------------------------------------------------------
class VxlanCountersSoftwareCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show vxlan counters software'
   data = {
      'vxlan': nodeVxlan,
      'counters': matcherCounters,
      'software': 'Software forwarding counters',
   }
   handler = VxlanCli.showSwCtrs
   cliModel = VxlanModel.VxlanSwCounters

BasicCli.addShowCommandClass( VxlanCountersSoftwareCmd )

#--------------------------------------------------------------------------------
# show vxlan counters varp
#--------------------------------------------------------------------------------
class VxlanCountersVarpCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show vxlan counters varp'
   data = {
      'vxlan': nodeVxlan,
      'counters': matcherCounters,
      'varp': 'Virtual ARP',
   }
   handler = VxlanCli.showVarpCounters
   cliModel = VxlanModel.VxlanVarpCounters

BasicCli.addShowCommandClass( VxlanCountersVarpCmd )

#--------------------------------------------------------------------------------
# show vxlan counters vni [ VNI ] [ DIRECTION ]
#--------------------------------------------------------------------------------
class VxlanCountersVniCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show vxlan counters vni [ VNI ] [ DIRECTION ]'
   data = {
      'vxlan': nodeVxlan,
      'counters': matcherCounters,
      'vni': CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher( 'vni', helpdesc='VNI counters' ),
         guard=VxlanCli.vniCountersSupported ),
      'VNI': VxlanCli.vniMatcher,
      'DIRECTION': directionMatcher
   }
   handler = VxlanCli.showVniCounters
   cliModel = VxlanModel.VxlanVniCountersModel

BasicCli.addShowCommandClass( VxlanCountersVniCmd )

#--------------------------------------------------------------------------------
# show vxlan counters vtep [ unlearnt | IP ] [ DIRECTION ]
#--------------------------------------------------------------------------------
class VxlanCountersVtepCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show vxlan counters vtep [ unlearnt | IP ] [ DIRECTION ]'
   data = {
      'vxlan': nodeVxlan,
      'counters': matcherCounters,
      'vtep': CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher( 'vtep', helpdesc='VTEP counters' ),
         guard=VxlanCli.vtepCountersSupported ),
      'unlearnt': 'Unlearnt VTEPs',
      'IP': IpGenAddrMatcher.IpGenAddrMatcher( helpdesc='Vtep IP address',
                                               helpdesc4="Vtep's IPv4 address",
                                               helpdesc6="Vtep's IPv6 address" ),
      'DIRECTION': directionMatcher
   }
   handler = VxlanCli.showVtepCounters
   cliModel = VxlanModel.VxlanVtepCountersModel

BasicCli.addShowCommandClass( VxlanCountersVtepCmd )

#--------------------------------------------------------------------------------
# show vxlan qos
#--------------------------------------------------------------------------------
class VxlanQosCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show vxlan qos'
   data = {
      'vxlan': nodeVxlan,
      'qos': 'Qos settings',
   }
   handler = VxlanCli.showVxlanQos
   cliModel = VxlanModel.VxlanQosModel

BasicCli.addShowCommandClass( VxlanQosCmd )

#--------------------------------------------------------------------------------
# show vxlan vni [ VNI ] [ { ( ( interface INTF ) |
#                              ( source SOURCE ) | 
#                              ( dot1q ( untagged | VLANS ) ) ) } ]
#--------------------------------------------------------------------------------
matcherVni = CliMatcher.KeywordMatcher( 'vni', helpdesc='VNI to VLAN mapping' )
class VxlanVniCmd( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show vxlan vni [ VNI ] [ { ( ( interface INTF ) | '
                                           '( source SOURCE ) | '
                                           '( dot1q ( untagged | VLANS ) ) ) } ]' )
   data = {
      'vxlan': nodeVxlan,
      'vni': matcherVni,
      'VNI': VxlanCli.vniMatcher,
      'interface': CliCommand.singleKeyword( 'interface',
         helpdesc='Interface selector' ),
      'INTF': IntfRange.IntfRangeMatcher( noSingletons=False,
         explicitIntfTypes=set( ( EthIntfCli.EthPhyAutoIntfType,
                                  LagIntfCli.LagAutoIntfType ) ) ),
      'source': CliCommand.singleKeyword( 'source', helpdesc='Source filter' ),
      'SOURCE': CliMatcher.PatternMatcher( pattern='\\S+',
         helpdesc='source of VNI to VLAN binding', helpname='WORD' ),
      'dot1q': CliCommand.singleKeyword( 'dot1q', helpdesc='802.1Q Tag filter' ),
      'untagged': 'Untagged packets',
      'VLANS': VlanCli.vlanSetMatcher,
   }
   handler = VxlanCli.showVxlanVni
   cliModel = VxlanModel.VxlanVniModel

BasicCli.addShowCommandClass( VxlanVniCmd )

#--------------------------------------------------------------------------------
# show vxlan vni summary
#--------------------------------------------------------------------------------
class VxlanVniSummaryCmd( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show vxlan vni summary' )
   data = {
      'vxlan': nodeVxlan,
      'vni': matcherVni,
      'summary': 'VXLAN VNI summary'
   }
   handler = VxlanCli.showVxlanVniSummary
   cliModel = VxlanModel.VxlanVniSummaryModel

BasicCli.addShowCommandClass( VxlanVniSummaryCmd )

#--------------------------------------------------------------------------------
# show vxlan vtep [ ( type TUNNELTYPE ) ] [ detail ]
#--------------------------------------------------------------------------------
class VxlanVtepCmd( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show vxlan vtep [ ( type TUNNEL_TYPE ) ] [ detail ]' )
   data = {
      'vxlan': nodeVxlan,
      'vtep': matcherVtep,
      'type' : CliCommand.singleKeyword( 'type', helpdesc='Tunnel type filter' ),
      'TUNNEL_TYPE' : CliMatcher.EnumMatcher( VxlanModel.vxlanTunnelTypes ),
      'detail' : CliCommand.singleKeyword( 'detail',
                                           helpdesc='Detailed VTEP information' ),
   }
   handler = VxlanCli.showVxlanVteps
   cliModel = VxlanModel.VxlanVtepsModel

class VxlanVtepSummaryCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show vxlan vtep summary'
   data = {
      'vxlan': nodeVxlan,
      'vtep': matcherVtep,
      'summary': 'VXLAN VTEP summary',
   }
   handler = VxlanCli.showVxlanVtepsSummary
   cliModel = VxlanModel.VxlanVtepsSummaryModel

BasicCli.addShowCommandClass( VxlanVtepCmd )
BasicCli.addShowCommandClass( VxlanVtepSummaryCmd )
