#!/usr/bin/env python
# Copyright (c) 2012 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from CliModel import Dict, Str, Bool, Model, Int, GeneratorList, GeneratorDict, Enum
from CliPlugin.IntfModel import InterfaceStatus
from VxlanVniLib import VniFormat
from ArnetModel import Ip4Address, Ip6Address, IpGenericPrefix
from operator import itemgetter
import collections
from ArnetModel import MacAddress
import Ethernet
from IntfModels import Interface
from Arnet import IpGenAddr, sortIntf

#-------------------------------------------------------------------------------
# VxlanIntf specific Eapi Models
#-------------------------------------------------------------------------------
class VxlanInterfaceStatus( InterfaceStatus ):
   class VxlanRemoteVtepAddrs( Model ):
      remoteVtepAddr = GeneratorList(
            valueType=Ip4Address, help='List of IPv4 vteps' )
      remoteVtepAddr6 = GeneratorList(
            valueType=Ip6Address, help='List of IPv6 vteps' )

   class VniSourcePair( Model ):
      vni = Int( help="Vni" )
      source = Str( help='Mapping source' )

   class VxlanVtepPrefixes( Model ):
      learnFrom = Enum( values=( 'learnFromAny', 'learnFromList',
                                 'learnFromFloodList', 'learnFromDefault' ),
                        help='Origin of VTEP learn restriction' )
      prefixList = GeneratorList( valueType=IpGenericPrefix,
                                  help='List of vtep (prefixes)' ) 

   vniInDottedNotation = Bool( default=False,
                               help='True if vni is displayed in dotted notation' )
   srcIpIntf = Str( help='Source interface for vxlan', optional=True )
   vArpVtepAddr = Str( help='IP address of virtual vtep', optional=True )
   srcIpAddr = Str( help='IP address of the source interface', optional=True )
   srcIpAddrV6 = Ip6Address( help='IPv6 address of the source interface',
                             optional=True )
   floodMcastGrp = Str( help='IP multicast group address for flooding traffic',
                        optional=True )
   mcastGrpDecap = Str( help='IP multicast group list for decap', optional=True )
   vlanToVniMap = GeneratorDict( keyType=int, valueType=VniSourcePair,
                                 help='Vlan to VNI source pair mapping' )
   vlanToVtepList = Dict( keyType=int, valueType=VxlanRemoteVtepAddrs,
                        help='Per Vlan Headend Vtep list' )
   staticFloodlists = Bool( help='Static Floodlists Configured' )
   replicationMode = Str( help='Replication/Flood Mode', optional=True )
   vlanToLearnRestrict = Dict( keyType=int, valueType=VxlanVtepPrefixes,
                        help='Per Vlan Learn list', optional=True )
   vrfToVniMap = Dict( keyType=str, valueType=long,
                       help='VRF to VNI mapping', optional=True )
   decapVniToVrfMap = Dict( keyType=long, valueType=str,
                            help='Decapsulation VNI to VRF mapping', optional=True )

   dataPathLearningMode = Bool( help='Datapath Learning Mode' )
   controllerClientMode = Bool( help='Controller Client Mode' )
   controllerControlPlane = Bool( help='Controller Control Plane' )
   vccImportVlan = Str( help='Import-enabled VLANs' )
   arpReplyRelayMode = Bool( help='ARP Reply Relay Mode' )
   arpLocalAddress = Bool( help='Enable rewrite of virtual IP with local IP in '
                           'VXLAN ARP requests' )
   portDot1qVniMapping = Bool( help='VNI mapping to port, dot1q tag pairs' )
   mlagSharedRouterMacAddr = MacAddress( help='MAC address of the MLAG Shared Router'
                                         ' Mac' )
   vtepAddrMask = Str( help='IP address mask assigned to VTEP',
                       optional=True )
   mlagSrcIpIntf = Interface( help='Source interface for MLAG VTEP IP',
                              optional=True )
   mlagSrcIpAddr = Ip4Address( help='IP address of the MLAG source interface',
                               optional=True )
   mlagSrcIpAddrV6 = Ip6Address( help='IPv6 address of the MLAG source interface',
                               optional=True )
   use32BitVni = Bool( help='VNI length in the VXLAN header' )
   vtepToVtepBridging = Bool( help='VTEP to VTEP bridging is enabled' )
   vtepSetForSourcePruning = GeneratorList( valueType=Ip4Address,
         help='Source pruning VTEP list' ) 
   vtepSourcePruningAll = Bool(
         help='Source pruning for all VTEPs' )
   decapFilterMode = Enum( values=( 'filterEnabled', 'filterRelaxedAll',
                                    'filterRelaxedIntf', 'filterDisabled' ),
                           help='VXLAN decapsulation filter mode' )
   decapFilterIntf = GeneratorList( valueType=Interface,
                           help='VXLAN decapsulation filter excluded interfaces' )
   mcastRouting = Bool( help='Multicast routing is enabled' )
   vxlanEncapsulation = Enum( values=( 'ipv4', 'ipv6', 'ipv4AndIpv6' ),
                              help='VXLAN encapsulation type' )
   vArpVtepSrcIpIntf = Interface( help='Source interface for virtual VTEP IP',
                                     optional=True )
   vArpVtepAddrV6 = Ip6Address( help='IPv6 address of the virtual VTEP'
                                        'source interface',
                                        optional=True )
   floodLearnedAll = Bool( help='Flood VTEPs learned via datapath learning',
                           optional=True )

   def renderSrcIpAddr( self, mlag=False ):
      if not self.mlagSrcIpIntf and mlag:
         # render is optional for mlagSrcIpIntf
         return

      if mlag:
         # pylint: disable-msg=E1101
         srcIpIntf = self.mlagSrcIpIntf.stringValue if self.mlagSrcIpIntf else ''
         # pylint: enable-msg=E1101
      else:
         srcIpIntf = self.srcIpIntf
      mlagPrefix = "MLAG " if mlag else ""
      srcIntfConfigured = srcIpIntf or "not configured"
      printBuf = "  %sSource interface is %s" % ( mlagPrefix, srcIntfConfigured )

      if srcIpIntf:
         srcIpAddrs = []
         if mlag:
            if self.vxlanEncapsulation != 'ipv6':
               srcIpAddr = str( self.mlagSrcIpAddr )
               if not IpGenAddr( srcIpAddr ).isAddrZero:
                  srcIpAddrs.append( srcIpAddr )

            if self.vxlanEncapsulation != 'ipv4':
               srcIpAddr = str( self.mlagSrcIpAddrV6 )
               if not IpGenAddr( srcIpAddr ).isAddrZero:
                  srcIpAddrs.append( srcIpAddr )
         else:
            if self.vxlanEncapsulation != 'ipv6':
               srcIpAddr = str( self.srcIpAddr )
               if not IpGenAddr( srcIpAddr ).isAddrZero:
                  srcIpAddrs.append( srcIpAddr )

            if self.vxlanEncapsulation != 'ipv4':
               srcIpAddr = str( self.srcIpAddrV6 )
               if not IpGenAddr( srcIpAddr ).isAddrZero:
                  srcIpAddrs.append( srcIpAddr )

         srcIpAddr = ' and '.join( srcIpAddrs )
         if srcIpAddr:
            printBuf += " and is active with %s" % srcIpAddr
         else:
            printBuf += " and is inactive"
      print printBuf

   def renderVxlanEncapsulation( self ):
      if self.vxlanEncapsulation != 'ipv4':
         if self.vxlanEncapsulation == 'ipv4AndIpv6':
            print "  Vxlan Encapsulation is IPv4 and IPv6"
         else:
            print "  Vxlan Encapsulation is IPv6"

   def renderVarpVtepAddr( self ):
      if self.vArpVtepSrcIpIntf:
         print "  Virtual VTEP source interface is %s " % self.vArpVtepSrcIpIntf
         varpVtepAddrs = []
         if self.vxlanEncapsulation != 'ipv6' and self.vArpVtepAddr:
            varpVtepIpAddr = str( self.vArpVtepAddr )
            if not IpGenAddr( varpVtepIpAddr ).isAddrZero:
               varpVtepAddrs.append( varpVtepIpAddr )

         if self.vxlanEncapsulation != 'ipv4' and self.vArpVtepAddrV6:
            varpVtepIpAddr = str( self.vArpVtepAddrV6 )
            if not IpGenAddr( varpVtepIpAddr ).isAddrZero:
               varpVtepAddrs.append( varpVtepIpAddr )
         varpVtepIpAddr = ' and '.join( varpVtepAddrs )
         if varpVtepIpAddr:
            print "  Virtual VTEP IP address is %s" % varpVtepIpAddr
      elif self.vArpVtepAddr:
         print "  Virtual VTEP IP address is %s" % self.vArpVtepAddr

   def renderGroupAddr( self ):
      if self.floodMcastGrp == '0.0.0.0':
         return
      print "  Multicast group address is %s" % self.floodMcastGrp

   def renderVtepAddrMask( self ):
      if not self.vtepAddrMask or self.vtepAddrMask == '255.255.255.255':
         return
      print "  VTEP address mask is %s" % self.vtepAddrMask

   def renderMcastGrpDecapAddr( self ):
      if not self.mcastGrpDecap:
         return

      print "  Multicast group decap address list is: "
      grps = self.mcastGrpDecap.split()
      log = ''
      hdr = ' ' * 4
      for grp in grps:
         log += '%-15s' % grp 
         if len( hdr + log ) > 72:
            print hdr + log 
            log = ''
            hdr = ' ' * 4
      if log:
         print hdr + log 

   def renderDatapathLearningMode( self ):
      if self.controllerClientMode and not self.dataPathLearningMode:
         print "  Remote MAC learning via VCS"
      elif self.controllerControlPlane:
         if self.dataPathLearningMode:
            print "  Remote MAC learning via EVPN and Datapath"
         else:
            print "  Remote MAC learning via EVPN"
      elif self.dataPathLearningMode:
         print "  Remote MAC learning via Datapath"
      else:
         print "  Remote MAC learning is disabled"

   def renderVccImportVlan( self ):
      if not self.controllerClientMode:
         return

      if self.vccImportVlan == '':
         print "  Remote MAC learning from Vxlan Controller Service disabled " \
               "in all VLANs"
      else:
         print "  Remote MAC learning from Vxlan Controller Service enabled in " \
               "VLAN(s): %s" % self.vccImportVlan

   def renderArpReplyRelayMode( self ):
      if self.arpReplyRelayMode:
         print "  ARP Reply relay is enabled"

   def renderArpLocalAddress( self ):
      if self.arpLocalAddress:
         print "  Rewrite of virtual IP with local IP in VXLAN ARP is enabled "

   def renderPortDot1qVniMapping( self ):
      mapping = "port, dot1q tag pairs" if self.portDot1qVniMapping else "VLANs"
      print "  VNI mapping to %s" % mapping

   def renderVlanToVniMap( self ):
      vlanToVniMap = sorted( self.vlanToVniMap, key=itemgetter(0) )
      notConfigured = "not configured" if not vlanToVniMap else ""

      print "  Static VLAN to VNI mapping is %s" % notConfigured

      # Loop through all mapping through sorted Vlan key
      vlanToVniMapBySource = collections.defaultdict( list )
      for vlan, vsp in vlanToVniMap:
         vni = vsp.vni
         source = vsp.source
         vlanToVniMapBySource[ source ].append( ( vlan, vni ) )

      def renderMap( vlanToVniSourceMap ):
         i = 1
         display = "   "
         for vlan, vni in vlanToVniSourceMap:
            vniStr = VniFormat( vni, self.vniInDottedNotation )
            display += "{:18s}".format( " [%d, %s]" % ( vlan, vniStr ) )
            if not i % 4:
               print display
               display = "   "
            i += 1

         if display:
            print display

      # Loop through sorted source key, "" or static should be first
      for source, sourceVlanToVniMap in sorted( vlanToVniMapBySource.iteritems() ):
         if source:
            print "  Dynamic VLAN to VNI mapping for '%s' is" % source

         renderMap( sourceVlanToVniMap )

      if vlanToVniMapBySource:
         print "  Note: All Dynamic VLANs used by VCS are internal VLANs."
         print "        Use 'show vxlan vni' for details."

   def renderFloodVtepList( self ):
      if not self.vlanToVtepList:
         return

      print '  Headend replication flood vtep list is:'
      for ( vlan, vl ) in sorted( self.vlanToVtepList.iteritems(),
                                  key=lambda( k,_ ): k ):
         hdr = '  %4d' % vlan
         trailer = ''
         # Note: we're not sorting the ip addresses
         for ip in vl.remoteVtepAddr:
            trailer += ' %-15s' % ip
            if len( hdr + trailer ) > 72:
               print hdr + trailer
               trailer = ''
               hdr = ' ' * 6
         for ip in vl.remoteVtepAddr6:
            trailer += ' %-15s' % ip.stringValue
            if len( hdr + trailer ) > 72:
               print hdr + trailer
               trailer = ''
               hdr = ' ' * 6
         if trailer:
            print hdr + trailer
 
   def renderReplicationMode( self ):
      mode_to_string = {'unknown' : "not initialized yet",
                        'multicast' :  "multicast",
                        'headendNonVcs' : "headend with Flood List Source: CLI",
                        'headendVcs' : "headend with Flood List Source: %s" % \
                        self.floodlistSourceString() }
      print "  Replication/Flood Mode is %s" % mode_to_string[ self.replicationMode ]
 
   def floodlistSourceString( self ):
      sources = []
      if self.controllerClientMode:
         sources.append( 'VCS' )
         if self.controllerControlPlane:
            sources.append( 'EVPN' )
      elif self.controllerControlPlane:
         sources.append( 'EVPN' )
         if self.staticFloodlists:
            sources.append( 'CLI' )
      if self.floodLearnedAll:
         sources.append( 'Datapath learning' )
      return ', '.join( sources )

   def renderReplicationList( self ):
      mode_to_render = { 'unknown' : None,
                         'multicast' : self.renderGroupAddr,
                         'headendNonVcs' : self.renderFloodVtepList,
                         'headendVcs' : self.renderFloodVtepList }
      if self.replicationMode in mode_to_render and \
            mode_to_render[ self.replicationMode ]:
         mode_to_render[ self.replicationMode ]()

   def renderLearnFrom( self ):
      mode_to_string = {'learnFromAny' : "any IP address",
                        'learnFromFloodList' :  "only those in flood list",
                        'learnFromList' : "only those in explicit learn list",
                        'learnFromDefault' : "<default value>"}
      print "  Learn VTEPS from %s" % mode_to_string[ self.learnFrom ]

   def renderVrfToVniMap( self ):
      notConfigured = "not configured" if not self.vrfToVniMap else ""
      print "  Static VRF to VNI mapping is %s" % notConfigured
      
      # Loop through all mapping through sorted Vrf name
      for ( vrf, vni ) in sorted( self.vrfToVniMap.iteritems(),
                                  key=lambda( k,_ ): k ):
         vniStr = VniFormat( vni, self.vniInDottedNotation )
         print "   [%s, %s]" % ( vrf, vniStr )

   def renderDecapVniToVrfMap( self ):
      if not self.decapVniToVrfMap:
         return
      print "  Static VRF to alternate decapsulation VNIs mapping is"

      vrfToVnisMap = collections.defaultdict( list )
      for vni, vrf in self.decapVniToVrfMap.iteritems():
         vrfToVnisMap[ vrf ].append( vni )

      # Loop through all mapping through sorted Vrf name
      for vrf, vnis in sorted( vrfToVnisMap.iteritems() ):
         vniStr = ', '.join( str( VniFormat( vni, self.vniInDottedNotation ) ) for
                             vni in sorted( vnis ) )
         print "   [%s, %s]" % ( vrf, vniStr )

   def renderMlagSharedRouterMacAddr( self ):
      if self.mlagSharedRouterMacAddr != '00:00:00:00:00:00':
         # pylint: disable-msg=E1101
         print "  MLAG Shared Router MAC is %s" % \
            Ethernet.convertMacAddrCanonicalToDisplay(
               self.mlagSharedRouterMacAddr.stringValue )

   def renderVxlanHeaderVniLength( self ):
      if not self.use32BitVni:
         return
      print "  VXLAN header VNI length is 32 bits"

   def renderVtepToVtepBridging( self ):
      if not self.vtepToVtepBridging:
         return
      print "  VTEP to VTEP bridging is enabled"

      if self.vtepSourcePruningAll:
         print "  Source pruning for all VTEPs"
         return
      try:
         firstIp = next( self.vtepSetForSourcePruning )
      except StopIteration:
         return
      print '  Source pruning VTEP list is:'
      trailer = '   %-15s' % firstIp
      for ip in self.vtepSetForSourcePruning:
         trailer += ' %-15s' % ip
         if len( trailer ) > 72:
            print trailer
            trailer = '  '
      print trailer

   def renderMcastRouting( self ):
      if not self.mcastRouting:
         return
      print "  Multicast routing is enabled"

   def renderDecapFilter( self ):
      if self.decapFilterMode == 'filterEnabled':
         return
      filterStr = '  Decapsulation filter'
      if self.decapFilterMode == 'filterDisabled':
         filterStr += ' is disabled'
      elif self.decapFilterMode == 'filterRelaxedAll':
         filterStr += ' excludes interfaces in multiple VRFs'
      elif self.decapFilterMode == 'filterRelaxedIntf':
         excludedIntfs = ', '.join( str( intf ) for intf in sortIntf(
            self.decapFilterIntf ) )
         filterStr += ' excludes %s' % excludedIntfs
      print filterStr

   def renderLinkTypeSpecific( self ):
      self.renderSrcIpAddr()
      self.renderSrcIpAddr( mlag=True )
      self.renderVxlanEncapsulation()
      self.renderVarpVtepAddr()
      self.renderReplicationMode()
      self.renderDatapathLearningMode()
      self.renderArpReplyRelayMode()
      self.renderArpLocalAddress()
      self.renderPortDot1qVniMapping()
      self.renderVlanToVniMap()
      self.renderVrfToVniMap()
      self.renderDecapVniToVrfMap()
      self.renderReplicationList()
      self.renderMcastGrpDecapAddr()
      self.renderMlagSharedRouterMacAddr()
      self.renderVtepAddrMask()
      self.renderVxlanHeaderVniLength()
      self.renderVtepToVtepBridging()
      self.renderMcastRouting()
      self.renderVccImportVlan()
      self.renderDecapFilter()

   def render( self ):
      self.renderHeader()
      self.renderHardware()
      self.renderLinkTypeSpecific()
      self.renderUptime()
