#!/usr/bin/env python
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

import CliCommand
import CliMatcher
import CliPlugin.IntfCli as IntfCli
import CliPlugin.IpAddrMatcher as IpAddrMatcher
import CliPlugin.VlanCli as VlanCli
import CliPlugin.VxlanCli as VxlanCli
import ConfigMount
import Tac
import Toggles.McastVpnLibToggleLib
import Tracing

t0 = Tracing.trace0

# Global variables for state mounted and shared across command
# classes.
ipTunnelGroupConfig = None

UnderlayRouteType = Tac.Type( "Routing::Multicast::UnderlayRouteType::RouteType" )

matcherGroup = CliMatcher.KeywordMatcher(
   'group', helpdesc="Underlay multicast group address" )
groupNode = CliCommand.Node( matcherGroup,
      guard=VxlanCli.vxlanUnderlayMcastSupportedGuard )

def getIpTunnelConfigEntry( tunnelType, underlayIpAddr ):
   # underlayIpAddr is a string form of the multicast IP address.
   underlayGroup = Tac.Value( 'Routing::Multicast::UnderlayAddr', underlayIpAddr )
   return Tac.Value( 'Routing::Multicast::IpTunnelConfigEntry',
                     tunnelType, underlayGroup )

def maybeCleanupIntfConfig( intfName ):
   if ipTunnelGroupConfig.get( intfName ) and ipTunnelGroupConfig[
         intfName ].isConfigEmpty:
      t0( "Cleanup tunnel intf config", intfName )
      del ipTunnelGroupConfig[ intfName ]

#-------------------------------------------------------------------------------
# config-interface-vxlan: vxlan vlan <id> flood group <ip-addr>
#-------------------------------------------------------------------------------
class McastVpnFloodGroupCmd( CliCommand.CliCommandClass ):
   syntax = "vxlan vlan VLAN_ID flood group MCAST_GROUP_ADDR"
   noOrDefaultSyntax = "vxlan vlan VLAN_ID flood group ..."
   data = {
      'vxlan': VxlanCli.vxlanNode,
      'vlan': CliCommand.Node( matcher=VxlanCli.matcherVlan,
                               guard=VxlanCli.isVxlan1InterfaceGuard ),
      'VLAN_ID': VlanCli.vlanIdMatcher,
      'flood': VxlanCli.matcherFlood,
      'group': groupNode,
      'MCAST_GROUP_ADDR': IpAddrMatcher.IpAddrMatcher(
         helpdesc="IP multicast group address" )
   }

   @staticmethod
   def handler( mode, args ):
      t0( "McastVpnFloodGroupCmd handler args", args )
      mcastGroupAddr = args[ 'MCAST_GROUP_ADDR' ]
      validationError = IpAddrMatcher.validateMulticastIpAddr( mcastGroupAddr,
                                                        allowReserved=False )
      if validationError:
         mode.addError( validationError )
         return
      vlanId = args[ 'VLAN_ID' ].id
      ipTunnelGroupIntfConfig = ipTunnelGroupConfig.newIntfConfig( mode.intf.name )
      ipTunnelGroupIntfConfig.vlanToFloodGroup[ vlanId ] = getIpTunnelConfigEntry(
         'tunnelTypeStatic', mcastGroupAddr )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      t0( "McastVpnFloodGroupCmd noOrDefaultHandler args", args )
      ipTunnelGroupIntfConfig = ipTunnelGroupConfig.intfConfig.get( mode.intf.name )
      if ipTunnelGroupIntfConfig is None:
         t0( mode.intf.name, "doesn't exist to cleanup vlanToFloodGroup" )
         return
      del ipTunnelGroupIntfConfig.vlanToFloodGroup[ args[ 'VLAN_ID' ].id ]
      maybeCleanupIntfConfig( mode.intf.name )

if Toggles.McastVpnLibToggleLib.toggleBullUnderlayMulticastEnabled():
   VxlanCli.VxlanIntfModelet.addCommandClass( McastVpnFloodGroupCmd )

matcherMulticast = CliMatcher.KeywordMatcher(
   'multicast', helpdesc="Underlay multicast group address" )
#-------------------------------------------------------------------------------
# config-interface-vxlan: vxlan vlan <id> multicast group <ip-addr>
#-------------------------------------------------------------------------------
class McastVpnMulticastGroupCmd( CliCommand.CliCommandClass ):
   syntax = "vxlan vlan VLAN_ID multicast group MCAST_GROUP_ADDR"
   noOrDefaultSyntax = "vxlan vlan VLAN_ID multicast group ..."
   data = {
      'vxlan': VxlanCli.vxlanNode,
      'vlan': CliCommand.Node( matcher=VxlanCli.matcherVlan,
                               guard=VxlanCli.isVxlan1InterfaceGuard ),
      'VLAN_ID': VlanCli.vlanIdMatcher,
      'multicast': matcherMulticast,
      'group': groupNode,
      'MCAST_GROUP_ADDR': IpAddrMatcher.IpAddrMatcher(
         helpdesc="IP multicast group address" )
   }

   @staticmethod
   def handler( mode, args ):
      t0( "McastVpnMulticastGroupCmd handler args", args )
      mcastGroupAddr = args[ 'MCAST_GROUP_ADDR' ]
      validationError = IpAddrMatcher.validateMulticastIpAddr( mcastGroupAddr,
                                                        allowReserved=False )
      if validationError:
         mode.addError( validationError )
         return
      vlanId = args[ 'VLAN_ID' ].id
      ipTunnelGroupIntfConfig = ipTunnelGroupConfig.newIntfConfig( mode.intf.name )
      ipTunnelGroupIntfConfig.vlanToMulticastGroup[ vlanId ] = \
            getIpTunnelConfigEntry( 'tunnelTypeStatic', mcastGroupAddr )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      t0( "McastVpnMulticastGroupCmd noOrDefaultHandler args", args )
      ipTunnelGroupIntfConfig = ipTunnelGroupConfig.intfConfig.get( mode.intf.name )
      if ipTunnelGroupIntfConfig is None:
         t0( mode.intf.name, "doesn't exist to cleanup vlanToMulticastGroup" )
         return
      del ipTunnelGroupIntfConfig.vlanToMulticastGroup[ args[ 'VLAN_ID' ].id ]
      maybeCleanupIntfConfig( mode.intf.name )

if Toggles.McastVpnLibToggleLib.toggleMcastVpnUnderlayMulticastEnabled():
   VxlanCli.VxlanIntfModelet.addCommandClass( McastVpnMulticastGroupCmd )

#-------------------------------------------------------------------------------
# config-interface-vxlan: vxlan vrf <name> multicast group <ip-addr>
#-------------------------------------------------------------------------------
class McastVpnVrfGroupCmd( CliCommand.CliCommandClass ):
   syntax = "vxlan VRF multicast group MCAST_GROUP_ADDR"
   noOrDefaultSyntax = "vxlan VRF multicast group ..."
   data = {
      'vxlan': VxlanCli.vxlanNode,
      'VRF': VxlanCli.vrfExprFactory,
      'multicast': CliCommand.Node( matcher=matcherMulticast,
                                    guard=VxlanCli.isVxlan1InterfaceGuard ),
      'group': groupNode,
      'MCAST_GROUP_ADDR': IpAddrMatcher.IpAddrMatcher(
         helpdesc="IP multicast group address" )
   }

   @staticmethod
   def handler( mode, args ):
      t0( "McastVpnVrfGroupCmd handler args", args )
      mcastGroupAddr = args[ 'MCAST_GROUP_ADDR' ]
      validationError = IpAddrMatcher.validateMulticastIpAddr( mcastGroupAddr,
                                                        allowReserved=False )
      if validationError:
         mode.addError( validationError )
         return
      vrfName = args[ 'VRF' ]
      ipTunnelGroupIntfConfig = ipTunnelGroupConfig.newIntfConfig( mode.intf.name )
      ipTunnelGroupIntfConfig.vrfToMulticastGroup[ vrfName ] = \
            getIpTunnelConfigEntry( 'tunnelTypeStatic', mcastGroupAddr )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      t0( "McastVpnVrfGroupCmd noOrDefaultHandler args", args )
      ipTunnelGroupIntfConfig = ipTunnelGroupConfig.intfConfig.get( mode.intf.name )
      if ipTunnelGroupIntfConfig is None:
         t0( mode.intf.name, "doesn't exist to cleanup vrfToMulticastGroup" )
         return
      del ipTunnelGroupIntfConfig.vrfToMulticastGroup[ args[ 'VRF' ] ]
      maybeCleanupIntfConfig( mode.intf.name )

if Toggles.McastVpnLibToggleLib.toggleMcastVpnOISMEnabled():
   VxlanCli.VxlanIntfModelet.addCommandClass( McastVpnVrfGroupCmd )

protocolNode = CliCommand.Node(
      matcher=CliMatcher.KeywordMatcher( 'protocol', helpdesc='Protocol settings' ),
      guard=VxlanCli.vxlanUnderlayMcastSupportedGuard )

multicastModes = {
   'asm': "Any Source Multicast",
   'ssm': "Source Specific Multicast (default)"
}
matcherRouteType = CliMatcher.EnumMatcher( multicastModes )
tacMulticastModeMap = {
   'ssm': UnderlayRouteType.pimssm,
   'asm': UnderlayRouteType.pimasm,
}

def _setModeType( intfName, trafficType, modeType=None ):
   ipTunnelGroupIntfConfig = ipTunnelGroupConfig.intfConfig.get( intfName )
   if not ipTunnelGroupIntfConfig:
      return
   if not modeType:
      modeType = ipTunnelGroupIntfConfig.underlayRouteTypeDefault
   if trafficType == 'multicast':
      ipTunnelGroupIntfConfig.mcastUnderlayRouteType = modeType
   elif trafficType == 'flood':
      ipTunnelGroupIntfConfig.floodUnderlayRouteType = modeType
   maybeCleanupIntfConfig( intfName )

#-------------------------------------------------------------------------------
# config-interface-vxlan:  vxlan multicast protocol pim [ asm | ssm ]
#-------------------------------------------------------------------------------
class McastUnderlayType( CliCommand.CliCommandClass ):
   syntax = "vxlan multicast protocol pim ROUTE_TYPE"
   noOrDefaultSyntax = "vxlan multicast protocol pim ..."
   data = {
      'vxlan': VxlanCli.vxlanNode,
      'multicast': CliCommand.Node( matcher=CliMatcher.KeywordMatcher( 'multicast',
                                              helpdesc='Multicast vxlan commands' ),
                                    guard=VxlanCli.isVxlan1InterfaceGuard ),
      'protocol': protocolNode,
      'pim': 'Protocol Independent Multicast',
      'ROUTE_TYPE': matcherRouteType,
   }

   @staticmethod
   def handler( mode, args ):
      t0( "McastUnderlayMode handler args", args )
      modeArg = args.get( 'ROUTE_TYPE', 'ssm' )
      modeType = tacMulticastModeMap.get( modeArg )
      ipTunnelGroupConfig.newIntfConfig( mode.intf.name )
      _setModeType( mode.intf.name, 'multicast', modeType )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      _setModeType( mode.intf.name, 'multicast' )

if Toggles.McastVpnLibToggleLib.toggleMcastVpnUnderlayAsmEnabled():
   VxlanCli.VxlanIntfModelet.addCommandClass( McastUnderlayType )

#-------------------------------------------------------------------------------
# config-interface-vxlan:  vxlan flood protocol pim [ asm | ssm ]
#-------------------------------------------------------------------------------
class FloodUnderlayType( CliCommand.CliCommandClass ):
   syntax = "vxlan flood protocol pim ROUTE_TYPE"
   noOrDefaultSyntax = "vxlan flood protocol pim ..."
   data = {
      'vxlan': VxlanCli.vxlanNode,
      'flood': CliCommand.Node( matcher=VxlanCli.matcherFlood,
                                guard=VxlanCli.isVxlan1InterfaceGuard ),
      'protocol': protocolNode,
      'pim': 'Protocol Independent Multicast',
      'ROUTE_TYPE': matcherRouteType,
   }
   hidden = True

   @staticmethod
   def handler( mode, args ):
      t0( "McastUnderlayMode handler args", args )
      modeArg = args.get( 'ROUTE_TYPE', 'ssm' )
      modeType = tacMulticastModeMap.get( modeArg )
      ipTunnelGroupConfig.newIntfConfig( mode.intf.name )
      _setModeType( mode.intf.name, 'flood', modeType )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      _setModeType( mode.intf.name, 'flood' )

if Toggles.McastVpnLibToggleLib.toggleMcastVpnTestCliEnabled():
   VxlanCli.VxlanIntfModelet.addCommandClass( FloodUnderlayType )

#-------------------------------------------------------------------------------
# config-interface-vxlan:  vxlan remote vni <VNI> vrf <VRF>
#-------------------------------------------------------------------------------
class StaticRemoteVniToVrfMap( CliCommand.CliCommandClass ):
   syntax = 'vxlan remote vni VNI VRF'
   noOrDefaultSyntax = 'vxlan remote vni VNI...'
   data = {
      'vxlan': VxlanCli.vxlanNode,
      'remote': CliCommand.Node( matcher=CliMatcher.KeywordMatcher( 'remote',
                                            helpdesc='Remote VTEP' ),
                                 guard=VxlanCli.isVxlan1InterfaceGuard ),
      'vni': VxlanCli.vniMatcherForConfig,
      'VNI': VxlanCli.vniMatcher,
      'VRF': VxlanCli.vrfExprFactory,
   }
   hidden = True

   @staticmethod
   def handler( mode, args ):
      vniArg = int( args[ 'VNI' ] )
      vrfArg = args[ 'VRF' ]
      config = ipTunnelGroupConfig.newIntfConfig( mode.intf.name )
      config.remoteVniToVrf[ vniArg ] = vrfArg

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      vniArg = int( args.get( 'VNI' ) )
      if mode.intf.name not in ipTunnelGroupConfig.intfConfig:
         return
      intfConfig = ipTunnelGroupConfig.intfConfig[ mode.intf.name ]
      del intfConfig.remoteVniToVrf[ vniArg ]
      maybeCleanupIntfConfig( mode.intf.name )

if Toggles.McastVpnLibToggleLib.toggleMcastVpnOISMEnabled() :
   VxlanCli.VxlanIntfModelet.addCommandClass( StaticRemoteVniToVrfMap )

class VxlanIntfMcastVpnIntf( IntfCli.IntfDependentBase ):
   """Cleanup config in response to VTI destruction."""
   def setDefault( self ):
      t0( "Destroying ipTunnelGroupConfig for", self.intf_.name )
      del ipTunnelGroupConfig.intfConfig[ self.intf_.name ]

def Plugin( entityManager ):
   global ipTunnelGroupConfig

   typeName = 'Routing::Multicast::IpTunnelGroupConfig'
   ipTunnelGroupConfig = ConfigMount.mount( entityManager,
                                            Tac.Type( typeName ).mountPath,
                                            typeName,
                                            "w" )
   # Registering this cleanup handler is a NOP when toggle is
   # disabled.
   IntfCli.Intf.registerDependentClass( VxlanIntfMcastVpnIntf )
