#!/usr/bin/env python
# Copyright (c) 2012-2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import re
import sys
import Arnet
import Toggles.VxlanToggleLib
import BasicCli
import CliCommand
import CliMatcher
import CliToken.Clear
import CliParser
import CliParserCommon
from CliPlugin import (
      AclCli,
      BfdCli,
      BridgingCli,
      ConfigConvert,
      IntfCli,
      IpAddrMatcher,
      IpGenAddrMatcher,
      IraCommonCli,
      IraIp6Cli,
      IraIpCli,
      IraIpRouteCliLib,
      LoopbackIntfCli,
      MacAddr,
      TechSupportCli,
      VirtualIntfRule,
      VlanCli,
      VrfCli,
)
from CliPlugin.VxlanControllerModel import (
      VxlanMacVtepListPair,
      VxlanMacVtepPair,
      VxlanVniStatus,
      VxlanArpEntry,
      VxlanArpTable,
)
from CliPlugin.IraIp6RouteCliLib import (
      manageIpv6Route,
      noIpv6RouteTableForVrfMsg,
      routeMatcherForIpv6Config,
      ipv6PrefixMatcher,
)
import CliSession
import ConfigMount
import Ethernet
from Intf import IntfRange
import LazyMount
import MultiRangeRule
import SmashLazyMount
import VxlanIntfModel
import VxlanModel
import VxlanCliLib
import RoutingConsts
import Tac
import Tracing
from operator import itemgetter
from collections import Counter, defaultdict
from itertools import chain
from CliExtensions import CliHook
from EthIntfCli import EthPhyIntf
from IntfRangePlugin.VxlanIntf import VxlanAutoIntfType
from MlagMountHelper import MlagStatusLazyMounter
from VxlanVniLib import VniFormat
import VxlanVniLib
from VxlanConfigSanity import (
      ConfigCheckItem,
      GenConfigCheckItems,
      warnExplainedMsg,
      dynVlanVniWarnMsg,
      vlanNotCreatedFmt,
      dynVlanVniConflictFmt,
      noRemoteVtepVlanFloodlistFmt,
      noVniInVrfToVniFormat,
      noVrfInVrfToVniFormat,
      RouteTrie,
      underlayDefaultVrfId,
)

t0 = Tracing.trace0

# pkgdeps library RoutingLib

configSanityFeatureDir = None
vxlanStatusDir = None
vxlanConfigDir = None
vxlanCounterConfigDir = None
vxAgentCounter = None
vxHwStatusDir = None
vtiConfigDir = None
vtiStatusDir = None
bridgingHwCapabilities = None
bridgingConfig = None
smashBridgingStatus = None
routingStatus = None
routing6Status = None
forwardingStatus = None
forwarding6Status = None
arpTableStatus = None
routeTrie = None
fhrpStatus = None
vrMacStatus = None
bridgingCliConfig = None
vlanXlateStatusDir = None
vxlanVniStatusDir = None
vxlanVniFdbStatusDir = None
vxlanClientDir = None
vxlanEvpnDynamicVlans = None
serviceConfigDir = None
vxlanControllerConfig = None
controllerErrorStatus = None
vniToVlanMap = None
mlagStatus = None
mlagHostTable = None
mlagVxlanStatus = None
em = None
vcsStateClientView = None
lRStatus = None
cvxClientStatus = None
hwCounterFeatureStatusDir = None
vxlanHwCounter = None
ipConfig = None
ipStatus = None
vxlanClientConvergenceStatus = None
remoteVniToVlanStatus = None
remoteVtepHwConfig = None
vtepHwStatus = None
ethAddrZero = Tac.Type( "Arnet::EthAddr" ).ethAddrZero
ipAddrZero = Tac.Type( 'Arnet::IpAddr' ).ipAddrZero
ipAddrWithMask = Tac.Type( 'Arnet::IpAddrWithMask' )
zeroAddrWithMask = ipAddrWithMask( '0.0.0.0', 0 )
ipsecConfig = None

vxlanBridgingConfigCheckCallback = []
vxlanRoutingConfigCheckCallback = []

vxlanEncapType = Tac.Type( "Vxlan::VxlanEncap" )
VtepType = Tac.Type( "L2Rib::VtepType" )
VxlanTunnelType = Tac.Type( "Vxlan::VxlanTunnelType" )
BfdInterval = Tac.Type( 'Bfd::BfdInterval' )
BfdMultiplier = Tac.Type( 'Bfd::BfdMultiplier' )

# To box vxlanCtrlConfig 
vxlanCtrlCfgBox = []

DEFAULT_VRF = VrfCli.DEFAULT_VRF

matcherAny = CliMatcher.KeywordMatcher( 'any',
      helpdesc='Learn from any addresses' )
matcherFlood = CliMatcher.KeywordMatcher( 'flood',
      helpdesc='VXLAN flooding behavior' )
matcherLearnRestrict = CliMatcher.KeywordMatcher( 'learn-restrict',
      helpdesc='VXLAN learning restrictions' )
matcherUdpPort = CliMatcher.KeywordMatcher( 'udp-port',
      helpdesc='VXLAN UDP port' )
vniMatcher = VxlanVniLib.VniMatcher( 'VXLAN Network Identifier',
      vxlanCtrlCfgBox )
vniRangeHelp = 'VXLAN Network Identifier (VNI) or range(s) of VNIs'
vniMultiMatcher = MultiRangeRule.MultiRangeMatcher(
      rangeFn=lambda: ( 1, 2**32 - 2 ),
      noSingletons=False,
      dottedRange=True,
      DoParseClass=VxlanVniLib.VniDottedDoParse,
      helpdesc=vniRangeHelp )
matcherVlan = CliMatcher.KeywordMatcher( 'vlan',
      helpdesc='VLAN configuration' )
matcherTxRxIntervalMs = CliMatcher.IntegerMatcher( 50, 60000,
      helpdesc='Rate in milliseconds' )
ipsecProfileNameMatcher = CliMatcher.DynamicNameMatcher(
      lambda mode: [] if ipsecConfig is None else ipsecConfig.ipsecProfile,
      helpdesc='VXLAN security profile name',
      helpname='WORD',
      priority=CliParser.PRIO_LOW )
vniAddRemoveMatcher = CliMatcher.EnumMatcher( { 'add': 'Add a VNI',
                                                'remove': 'Remove a VNI' } )
dynamicMatcherForConfig = CliMatcher.KeywordMatcher( 'dynamic',
      helpdesc='Dynamic' )
ribBypassMatcherForConfig = CliMatcher.KeywordMatcher( 'rib-bypass',
      helpdesc='RIB bypass' )

@Tac.memoize
def vxlanCliHelper():
   rtConfig = IraIpRouteCliLib.ip.routeConfig
   rtVrfConfig = IraIpRouteCliLib.ip.vrfRouteConfigDir
   LazyMount.force( bridgingHwCapabilities )
   return Tac.newInstance("Vxlan::VxlanCliPluginHelper", rtConfig, rtVrfConfig,
         vtiConfigDir, IraIpRouteCliLib.iraIpRouteCliHelper(),
         bridgingHwCapabilities, IraIpRouteCliLib.noRouteTableForVrfMsg )

class SwTunnelGroupType( object ):
   singleMemberGroup = 0
   multiMemberGroup = 1

def ribBypassSupportedGuard( mode, token ):
   rhs = vxlanCliHelper().iraIpRouteCliHelper.routingHardwareStatus
   if rhs.staticVxlanRouteRibBypassSupported:
      return None
   return CliParser.guardNotThisPlatform

def isVxlan1InterfaceGuard( mode, token ):
   if mode.intf.name == "Vxlan1":
      return None
   return 'only supported for Vxlan1'

def vxlanMultiVtepMlagGuard( mode, token ):
   if bridgingHwCapabilities.vxlanMultiVtepMlagSupported:
      return isVxlan1InterfaceGuard( mode, token )
   return CliParser.guardNotThisPlatform

def vxlanSupportedGuard( mode, token ):
   if bridgingHwCapabilities.vxlanSupported:
      return None
   return CliParser.guardNotThisPlatform

def vxlanEncapIpv6SupportedGuard( mode, token ):
   if bridgingHwCapabilities.vxlanEncapIpv6ConfigSupported:
      return None
   return CliParser.guardNotThisPlatform

def vxlanV6RemoteVtepSupportedGuard( mode, token ):
   if ':' not in token or bridgingHwCapabilities.vxlanEncapIpv6ConfigSupported:
      return None
   return CliParser.guardNotThisPlatform

def vxlanVirtualVtepCmdSupportedGuard( mode, token ):
   if bridgingHwCapabilities.vxlanEncapIpv6ConfigSupported:
      return isVxlan1InterfaceGuard( mode, token )
   return CliParser.guardNotThisPlatform

def vxlanMcastGroupSupportedGuard( mode, token ):
   if bridgingHwCapabilities.vxlanMcastGroupSupported:
      return isVxlan1InterfaceGuard( mode, token )
   return CliParser.guardNotThisPlatform

def vxlanMcastDecapOnlySupportedGuard( mode, token ):
   if bridgingHwCapabilities.vxlanMcastDecapOnlySupported:
      return isVxlan1InterfaceGuard( mode, token )
   return CliParser.guardNotThisPlatform

def vxlanUnderlayMcastSupportedGuard( mode, token ):
   if bridgingHwCapabilities.vxlanUnderlayMcastSupported:
      return None
   return CliParser.guardNotThisPlatform

def vxlanRoutingSupportedGuard( mode, token ):
   if bridgingHwCapabilities.vxlanRoutingSupported:
      return None
   return CliParser.guardNotThisPlatform

def vxlanControllerClientSupportedGuard( mode, token ):
   if bridgingHwCapabilities.vxlanControllerClientSupported:
      return isVxlan1InterfaceGuard( mode, token )
   return CliParser.guardNotThisPlatform

def vxlanDecapFilterSupportedGuard( mode, token ):
   if bridgingHwCapabilities.vxlanDecapFilterSupported:
      return isVxlan1InterfaceGuard( mode, token )
   return CliParser.guardNotThisPlatform

def vxlanL3EvpnSupportedGuard( mode, token ):
   if ( bridgingHwCapabilities.vxlanRoutingSupported and
        bridgingHwCapabilities.vxlanL3EvpnSupported ):
      return None
   return CliParser.guardNotThisPlatform

def vtepCountersSupported( mode, token ):
   if bridgingHwCapabilities.vxlanVtepCtrSupported:
      return None
   return CliParser.guardNotThisPlatform

def vniCountersSupported( mode, token ):
   if bridgingHwCapabilities.vxlanVniCtrSupported:
      return None
   return CliParser.guardNotThisPlatform

def vxlanVtepMaskCliSupported( mode, token ):
   if bridgingHwCapabilities.vxlanVtepMaskCliSupported:
      return None
   return CliParser.guardNotThisPlatform

def vxlanSrcPortSupported( mode, token ):
   if bridgingHwCapabilities.vxlanSrcPortSupported != 'notSupported':
      return None
   return CliParser.guardNotThisPlatform

def vxlanEcnPropagationSupported( mode, token ):
   if bridgingHwCapabilities.vxlanEcnPropagationSupported:
      return None
   return CliParser.guardNotThisPlatform

def vxlanIpv6UnderlaySupportedGuard( mode, token ):
   if ':' not in token or bridgingHwCapabilities.vxlanIpv6UnderlaySupported:
      return None
   return CliParser.guardNotThisPlatform

def vxlan32BitVniSupportedGuard( mode, token ):
   if bridgingHwCapabilities.vxlanHeader32BitVniSupported:
      return None
   return CliParser.guardNotThisPlatform

def vxlanVtepToVtepBridgingSupportedGuard( mode, token ):
   if bridgingHwCapabilities.vxlanVtepToVtepBridgingSupported:
      return isVxlan1InterfaceGuard( mode, token )
   return CliParser.guardNotThisPlatform

def vxlanSecurityGuard( mode, token ):
   if bridgingHwCapabilities.vxlanSecuritySupported:
      return None
   return CliParser.guardNotThisPlatform

def vxlanTunnelIngressDecapAclSupported( mode, token ):
   if ( bridgingHwCapabilities.vxlanRoutingSupported and
        bridgingHwCapabilities.vxlanL3EvpnSupported and
        bridgingHwCapabilities.vxlanTunnelIngressDecapAclSupported ):
      return None
   return CliParser.guardNotThisPlatform

def vxlanMlagSharedRouterMacSupported( mode, token ):
   if ( bridgingHwCapabilities.vxlanRoutingSupported and
        bridgingHwCapabilities.vxlanL3EvpnSupported ):
      return isVxlan1InterfaceGuard( mode, token )
   return CliParser.guardNotThisPlatform

vrfExprFactory = VrfCli.VrfExprFactory(
      helpdesc='VRF which is mapped to a VNI',
      guard=vxlanL3EvpnSupportedGuard )
#
# VTI interface name to Arnet::IntfId cache.
#
# We use this to avoid building a short lived Tac object everytime we map a
# VTI name to interface Id. Mapping from a interface name string to interface
# id integer does not change.
#
class VtiNameToId( object ):
   __slots__ = [ 'nameIntfCache' ]

   def __init__( self ):
      self.nameIntfCache = {}

   def get( self, intfName ):
      if intfName not in self.nameIntfCache:
         vtiIntfId = Tac.Value( "Arnet::IntfId", intfName )
         self.nameIntfCache[ intfName ] = vtiIntfId

      return self.nameIntfCache[ intfName ]

# Global cache object for vti name intf caching
vtiNameToId = VtiNameToId()

class VniToVlanMap( object ):
   '''
   Builds and caches vni-to-vlan map for all VTIs. Cache structure is:

   { <interface> :  { expireTime : <time>, 
                      vniToVlanMap : {...} },
   ...
   }
   
   The map is refreshed on access if cached time is older than the
   configured expiry time.
   '''

   # Default cache retention duration
   RETENTION = 30.0

   def __init__( self ):
      self.cache = {}
   
   def populateCache( self ):
      if not self.cache:
         self.scan()

   # Build vni to vlan mapping from vtiStatusDir. If an entry is not found,
   # maybe due to state propagation delay, we force refresh the cache.
   def scanVti( self, intfName ):
      vlanVniMap = vtiStatusDir.vtiStatus[ intfName ].vlanToVniMap
      revMap = { vs.vni : vlan for vlan, vs in vlanVniMap.iteritems() }

      if intfName not in self.cache:
         self.cache[ intfName ] = {}

      self.cache[ intfName ][ 'expiry' ] = Tac.now() + self.RETENTION
      self.cache[ intfName ][ 'revMap' ] = revMap

   def scan( self ):
      for intfId in vtiStatusDir.vtiStatus.keys():
         self.scanVti( intfId )

   # get keys ( VNIs ) present
   def getKeys( self, intfId ):
      self.scan() # updates mapping for existing VTIs
      if intfId not in self.cache:
         self.cache[ intfId ] = { 'expiry': Tac.now() + self.RETENTION,
                                  'revMap' : {} }
      return self.cache[ intfId ][ 'revMap' ].keys()

   def getVlan( self, vni, intfId ):
      # if the Vxlan intf has gone away clear out the old cache
      if intfId not in vtiStatusDir.vtiStatus:
         if intfId in self.cache:
            del self.cache[ intfId ]
         return None

      self.populateCache()

      # Refresh the cache if retention time expired or 'vni' is missing
      if ( Tac.now() > self.cache[ intfId ][ 'expiry' ] ) or \
         ( vni not in self.cache[ intfId ][ 'revMap' ] ):
         self.scanVti( intfId )

      return self.cache[ intfId ][ 'revMap' ].get( vni )

   def setVni( self, vni, vlan, intfId ):
      self.populateCache()
      if intfId not in self.cache:
         self.cache[ intfId ] = { 'expiry': Tac.now() + self.RETENTION,
                                  'revMap' : {} }
      self.cache[ intfId ][ 'revMap' ][ vni ] = vlan

   def delVni( self, vni, intfId ):
      self.populateCache()
      if ( intfId in self.cache ) and ( vni in self.cache[ intfId ][ 'revMap' ] ):
         del self.cache[ intfId ][ 'revMap' ][ vni ]

def _vxlanSessionCommitHandler( mode, vxlanIntf ):
   if vxlanIntf in vtiConfigDir.vtiConfig:
      vxlanCounterConfigDir.vxlanCounterConfig.newMember( vxlanIntf )
   else:
      del vxlanCounterConfigDir.vxlanCounterConfig[ vxlanIntf ]

class VxlanIntf( IntfCli.VxlanVirtualIntf ):
   def __init__( self, name, mode ):
      m = re.match( r'Vxlan(\d+)$', name )
      self.vtiId = int( m.group( 1 ) )
      IntfCli.VxlanVirtualIntf.__init__( self, name, mode )
      self.vxlanStatusDir = vxlanStatusDir
      self.vxlanConfigDir = vxlanConfigDir
      self.vxlanCounterConfigDir = vxlanCounterConfigDir
      self.vtiConfigDir = vtiConfigDir
      self.vtiStatuses = vtiStatusDir.vtiStatus
      self.vtiStatus = None
      self.bumStatus = vxHwStatusDir.bumStatus
      self.learnStatus = vxHwStatusDir.learnStatus

   def createPhysical( self, startupConfig=False ):
      if self.mode_.session.inConfigSession():
         # The same callback can be used for destroyPhysical and createPhysical and
         # only one instance is needed hence the common 'VXLAN' key for both
         CliSession.registerSessionOnCommitHandler(
            self.mode_.session_.entityManager,
            'VXLAN',
            lambda m, onSessionCommit: _vxlanSessionCommitHandler( m, self.name ) )
      else:
         self.vxlanCounterConfigDir.vxlanCounterConfig.newMember( self.name )

      self.vtiConfigDir.vtiConfig.newMember( self.name )
      self.vxlanConfigDir.vxlanConfig.newMember( self.name )

   def lookupPhysical( self ):
      self.vtiStatus = self.vtiStatuses.get( self.name, None )
      return self.vtiStatus is not None

   def hardware( self ):
      return "vxlan"

   def addrStr( self ):
      return None

   def bandwidth( self ):
      return 0
   
   def getIntfStatusModel( self ):
      return VxlanIntfModel.VxlanInterfaceStatus( name=self.status().intfId )

   def showPhysical( self, mode, intfStatusModel ):
      vtiConfig = getVtiConfig( self.name )
      vxlanConfig = getVxlanConfig( self.name )
      intfStatusModel.srcIpIntf = vtiConfig.srcIpIntf
      intfStatusModel.vArpVtepSrcIpIntf = vtiConfig.varpVtepSrcIpIntf
      encapV4 = self.vtiStatus.vxlanEncap == vxlanEncapType.vxlanEncapIp4
      encapV6 = self.vtiStatus.vxlanEncap == vxlanEncapType.vxlanEncapIp6
      encapDual = self.vtiStatus.vxlanEncap == vxlanEncapType.vxlanEncapDual

      if ( encapV4 or encapDual ):
         intfStatusModel.srcIpAddr = self.vtiStatus.localVtepAddr
      if ( encapV6 or encapDual ):
         intfStatusModel.srcIpAddrV6 = self.vtiStatus.localVtepAddr6
      if ( ( encapV4 or encapDual ) and self.vtiStatus.vArpVtepAddr != '0.0.0.0' ):
         intfStatusModel.vArpVtepAddr = self.vtiStatus.vArpVtepAddr
      if ( ( encapV6 or encapDual ) and not self.vtiStatus.vArpVtepAddr6.isZero ):
         intfStatusModel.vArpVtepAddrV6 = self.vtiStatus.vArpVtepAddr6
      intfStatusModel.dataPathLearningMode = self.vtiStatus.datapathLearning
      intfStatusModel.controllerClientMode = self.vtiStatus.controllerClientMode
      intfStatusModel.floodLearnedAll = vtiConfig.floodLearnedAll
      intfStatusModel.arpReplyRelayMode = self.vtiStatus.arpReplyRelay
      intfStatusModel.arpLocalAddress = self.vtiStatus.arpLocalAddress
      intfStatusModel.controllerControlPlane = self.vtiStatus.controllerControlPlane
      intfStatusModel.floodMcastGrp = self.vtiStatus.floodMcastGrp
      intfStatusModel.vtepAddrMask = self.vtiStatus.vtepAddrMask
      intfStatusModel.vniInDottedNotation = \
            vxlanControllerConfig.vniInDottedNotation
      intfStatusModel.portDot1qVniMapping = self.vtiStatus.portDot1qVniMapping

      intfStatusModel.replicationMode = self.bumStatus.bumReplicationMode
      intfStatusModel.mlagSharedRouterMacAddr = \
             self.vtiStatus.mlagSharedRouterMacAddr
      intfStatusModel.staticFloodlists = ( bool( vxlanConfig.vlanToVtepList ) or 
                                           bool( vxlanConfig.floodVtepList ) )
      grps = ""
      if self.vtiStatus.mcastGrpDecap:
         grps = ' '.join( sorted( self.vtiStatus.mcastGrpDecap.iterkeys(),
                        key=lambda x: Arnet.IpAddress( x ).value ) )
      intfStatusModel.mcastGrpDecap = grps 

      # copy the vlanToVniMap into the Capi model from Sysdb
      def giveNextVlanVspMap( vlanToVniMap ):
         for vlan, vniSource in self.vtiStatus.vlanToVniMap.iteritems():
            vsp = VxlanIntfModel.VxlanInterfaceStatus.VniSourcePair()
            vsp.vni = vniSource.vni
            vsp.source = vniSource.source
            yield vlan, vsp

      intfStatusModel.vlanToVniMap = giveNextVlanVspMap(
         self.vtiStatus.vlanToVniMap )

      # copy the vlanToVtepList into the Capi model from Sysdb
      for mvp in self.bumStatus.bumVtepList:
         vteps = VxlanIntfModel.VxlanInterfaceStatus.VxlanRemoteVtepAddrs()
         try:
            bumVtepList = self.bumStatus.bumVtepList[ mvp ]
            vteps.remoteVtepAddr = ( v for v in bumVtepList.remoteVtepAddr )
            vteps.remoteVtepAddr6 = ( v for v in bumVtepList.remoteVtepAddr6 )
         except KeyError:
            pass
         intfStatusModel.vlanToVtepList[ mvp.vlanId ] = vteps

      # Copy the vlanToLearnRestrict into the Capi model
      # Handle different cases of the learnStatusList entries going away
      for vlan in self.learnStatus.learnStatusList:
         lr = VxlanIntfModel.VxlanInterfaceStatus.VxlanVtepPrefixes()
         try:
            lr.learnFrom = self.learnStatus.learnStatusList[ vlan ].learnFrom
            if lr.learnFrom == 'learnFromDefault':
               raise KeyError
            prefixList = self.learnStatus.learnStatusList[ vlan ].prefixList
            lr.prefixList = ( p.ipPrefix for p in prefixList )
         except KeyError:
            continue
         intfStatusModel.vlanToLearnRestrict[ vlan ] = lr

      # Copy vrfToVniMap to the Capi model
      for vrfName in self.vtiStatus.vrfToVniMap:
         vni = self.vtiStatus.vrfToVniMap[ vrfName ]
         intfStatusModel.vrfToVniMap[ vrfName ] = long( vni )

      # Copy alternateDecapVniToVrfMap to the Capi model
      for vni, vrf in self.vtiStatus.alternateDecapVniToVrfMap.iteritems():
         intfStatusModel.decapVniToVrfMap[ long( vni ) ] = vrf

      # Copy MLAG source-interface and address
      if vtiConfig.mlagSrcIpIntf:
         intfStatusModel.mlagSrcIpIntf = vtiConfig.mlagSrcIpIntf
         if ( encapV4 or encapDual ):
            intfStatusModel.mlagSrcIpAddr = self.vtiStatus.mlagVtepAddr
         if ( encapV6 or encapDual ):
            intfStatusModel.mlagSrcIpAddrV6 = self.vtiStatus.mlagVtepAddr6

      intfStatusModel.use32BitVni = self.vtiStatus.use32BitVni 
      intfStatusModel.vtepToVtepBridging = self.vtiStatus.vtepToVtepBridging
      intfStatusModel.vtepSetForSourcePruning = (
            vtep for vtep in self.vtiStatus.vtepSetForSourcePruning )
      intfStatusModel.vtepSourcePruningAll = self.vtiStatus.vtepSourcePruningAll
      intfStatusModel.decapFilterMode = vtiStatusDir.decapFilterMode
      intfStatusModel.decapFilterIntf = (
            intf for intf in vtiStatusDir.decapFilterIntf )
      intfStatusModel.mcastRouting = self.vtiStatus.mcastRouting

      # Convert vtiStatus.vccImportVlans into canonical string and 
      # copy into intfStatusModel.vccImportVlan
      intfStatusModel.vccImportVlan = \
            MultiRangeRule.multiRangeToCanonicalString( 
                  self.vtiStatus.vccImportVlans )
      if encapV4:
         intfStatusModel.vxlanEncapsulation = 'ipv4'
      elif encapV6:
         intfStatusModel.vxlanEncapsulation = 'ipv6'
      else:
         intfStatusModel.vxlanEncapsulation = 'ipv4AndIpv6'

   def destroyPhysical( self ):
      if self.mode_.session.inConfigSession():
         # The same callback can be used for destroyPhysical and createPhysical and
         # only one instance is needed hence the common 'VXLAN' key for both
         CliSession.registerSessionOnCommitHandler(
            self.mode_.session_.entityManager,
            'VXLAN',
            lambda m, onSessionCommit: _vxlanSessionCommitHandler( m, self.name ) )
      else:
         del self.vxlanCounterConfigDir.vxlanCounterConfig[ self.name ]

      del self.vxlanConfigDir.vxlanConfig[ self.name ]
      del self.vtiConfigDir.vtiConfig[ self.name ]
      adjustVxlanControllerServiceEnable()

   #----------------------------------------------------------------------------
   # Returns the vtiConfig object for this interface.
   #----------------------------------------------------------------------------
   def config( self ):
      return self.vtiConfigDir.vtiConfig.get( self.name )

   #----------------------------------------------------------------------------
   # Returns the vtiStatus object for this interface.
   #----------------------------------------------------------------------------
   def status( self ):
      return self.vtiStatus

   def getIntfCounterDir( self ):
      # we do not support counters on vti interfaces
      return None

   @staticmethod
   def getAllPhysical( sysdbRoot ):
      intfs = []
      for name in vtiStatusDir.vtiStatus:
         intf = VxlanIntf( name, sysdbRoot )
         if intf.lookupPhysical():
            intfs.append( intf )
      return intfs

   def countersSupported( self ):
      return False

   def routingSupported( self ):
      return False

   def routingCurrentlySupported( self ):
      return False

   def setDefault( self ):
      IntfCli.VxlanVirtualIntf.setDefault( self )
      vtiConfig = getVtiConfig( self.name )
      vxlanConfig = getVxlanConfig( self.name )
      vtiConfig.floodMcastGrp = Tac.newInstance( "Arnet::IpAddr", 0 )
      vtiConfig.mcastGrpDecap.clear()
      vtiConfig.vtepAddrMask = Tac.newInstance( "Arnet::IpAddr", 0xFFFFFFFF )
      vtiConfig.srcIpIntf = ''
      vtiConfig.vlanToVniMap.clear()
      vtiConfig.ttl = vtiConfig.defaultTtl
      vtiConfig.udpPort = vtiConfig.vxlanWellKnownPort
      vtiConfig.secUdpPort = vtiConfig.vxlanSecWellKnownPort
      vtiConfig.srcPortRange = Tac.Value( "Vxlan::VxlanSrcPortRange" )
      vtiConfig.controllerClientMode = False
      vtiConfig.use32BitVni = False
      vtiConfig.vtepToVtepBridging = False
      vtiConfig.vtepSetForSourcePruning.clear()
      vtiConfig.vtepSourcePruningAll = False
      vtiConfig.floodLearnedAll = False
      vtiConfig.mcastRouting = False
      vtiConfig.vxlanEncap = bridgingHwCapabilities.vxlanDefaultEncap
      vtiConfig.vxlanEncapConfigured = False
      vtiConfig.cliVccImportVlans = '1-4094'
      vtiConfig.cliVccImportVlans = '1-4094'
      adjustVxlanControllerServiceEnable()

      vxlanConfig.floodVtepList.clear()
      vxlanConfig.floodVtepList6.clear()

      vxlanConfig.vlanToVtepList.clear()
      # Get a default empty learn list
      vxlanConfig.learnFrom = 'learnFromDefault'
      vxlanConfig.vlanToLearnRestrict.clear()
      vtiConfig.vrfToVniMap.clear()
      vtiConfig.alternateDecapVniToVrfMap.clear()

      vxlanConfig.vniToIpAclMap.clear()
      vtiConfig.arpLocalAddress = False
      vtiConfig.mlagSrcIpIntf = ''
      vtiConfig.varpVtepSrcIpIntf = ''
      vtiConfig.vtepToSecProfile.clear()
      vtiConfig.bfdEnabled = False
      vtiConfig.decapFilterMode = 'filterEnabled'
      vtiConfig.decapFilterIntf.clear()

#----------------------------------------------------------------------------
# The rule for matching Vxlan Tunnel interface names. When this pattern matches,
# it returns an instance of the VxlanIntf class.
#
# This rule gets added to the Intf.rule when this class is registered with
# the Intf class by calling Intf.addPhysicalIntfType.
#----------------------------------------------------------------------------
VxlanIntf.matcher = VirtualIntfRule.VirtualIntfMatcher(
      'Vxlan', 1, 1, value=lambda mode, intf: VxlanIntf( intf, mode ),
      helpdesc="VXLAN Tunnel Interface", guard=vxlanSupportedGuard )


class VxlanIntfModelet( CliParser.Modelet ):
   def __init__( self, mode ):
      CliParser.Modelet.__init__( self )

   @staticmethod
   def shouldAddModeletRule( mode ):
      return isinstance( mode.intf, VxlanIntf )

   modeletParseTree = CliParser.ModeletParseTree()

#-------------------------------------------------------------------------------
# Register the VxlanIntf class as a type of physical interface.
#-------------------------------------------------------------------------------
IntfCli.Intf.addPhysicalIntfType( VxlanIntf, VxlanAutoIntfType )

IntfRange.registerIntfTypeGuard( VxlanAutoIntfType, vxlanSupportedGuard )

# utility functions
def getVtiConfig( intfName, create=True ):
   vtiIntfId = vtiNameToId.get( intfName )
   vtiConfig = vtiConfigDir.vtiConfig.get( vtiIntfId )
   if not vtiConfig and create:
      vtiConfig = vtiConfigDir.vtiConfig.newMember( vtiIntfId )
   return vtiConfig

def adjustVxlanControllerServiceEnable():
   """ Synchronizes the CVX controller 'enabled' config attribute with the
   per-vti equivalent, 'controllerClientMode'. This is called whenever vti
   config's controllerClientMode is changed or a vti config is deleted. """

   vxlanControllerService = serviceConfigDir.service[ "Vxlan" ]
   for vtiConfig in vtiConfigDir.vtiConfig.values():
      if vtiConfig.controllerClientMode:
         vxlanControllerService.enabled = True
         return
   vxlanControllerService.enabled = False

def getVtiStatus( intfName ):
   vtiIntfId = vtiNameToId.get( intfName )
   vtiStatus = vtiStatusDir.vtiStatus.get( vtiIntfId )
   return vtiStatus

def getVxlanConfig( intfName ):
   vtiIntfId = vtiNameToId.get( intfName )
   vxlanConfig = vxlanConfigDir.vxlanConfig.get( vtiIntfId )
   return vxlanConfig

def getVxlanCounterConfig( intfName ):
   vtiIntfId = vtiNameToId.get( intfName )
   vxlanCounterConfig = vxlanCounterConfigDir.vxlanCounterConfig.get( vtiIntfId )
   return vxlanCounterConfig

def getVxlanStatus( intfName ):
   vtiIntfId = vtiNameToId.get( intfName )
   vxlanStatus = vxlanStatusDir.vxlanStatus.get( vtiIntfId )
   return vxlanStatus

def isVtiMissing( intfName ):
   vtiIntfId = vtiNameToId.get( intfName )
   return vtiConfigDir.vtiConfig.get( vtiIntfId ) is None

vxlanNode = CliCommand.Node(
      matcher=CliMatcher.KeywordMatcher( 'vxlan',
         helpdesc='Configure VXLAN parameters' ),
      guard=vxlanSupportedGuard )
controllerClientNode = CliCommand.Node( 
      matcher=CliMatcher.KeywordMatcher( 'controller-client',
         helpdesc='Set VXLAN controller client mode parameters' ),   
      guard=vxlanControllerClientSupportedGuard )
decapFilterNode = CliCommand.Node(
      matcher=CliMatcher.KeywordMatcher( 'decapsulation',
         helpdesc='Set VXLAN decapsulation parameters' ),
      guard=vxlanDecapFilterSupportedGuard )

# Keep vlanRange so that other packages can import it
vlanMultiMatcher = MultiRangeRule.MultiRangeMatcher(
      rangeFn=lambda: ( 1, 4094 ),
      noSingletons=False,
      helpdesc='VLAN ID(s)' )
vtepMatcherForConfig = CliMatcher.KeywordMatcher( 'vtep',
                                   helpdesc='Configure VXLAN Tunnel End Points' )
addRemoveMatcher = CliMatcher.EnumMatcher( { 'add': 'Add a VTEP',
                                             'remove': 'Remove a VTEP' } )
vxlanIntfType = [ VxlanAutoIntfType ]
ipGenPrefixMatcher = IpGenAddrMatcher.IpGenPrefixMatcher(
      'IP address (or prefix) of remote VTEP',
      helpdesc4='IPv4 address (or prefix) of remote VTEP',
      helpdesc6='IPv6 address (or prefix) of remote VTEP', allowAddr=True )

def setVxlanShutdown( modeList ):
   # We can only create a single Vxlan interface at the moment
   m = re.match( r'Vxlan(\d+)$', modeList[ 0 ].intf.name )
   if m:
      vtiConfig = getVtiConfig( modeList[ 0 ].intf.name )
      serviceConfigDir.service[ "Vxlan" ].enabled = \
         vtiConfig.controllerClientMode and vtiConfig.adminEnabled

IntfCli.shutdownIfHook.addExtension( setVxlanShutdown )

#-------------------------------------------------------------------------------
# The "[no|default] vxlan multicast-group <ipAddr>" command, in "config-vxlan-if"
#
# sets the ip mcast group for vxlan flooding
#-------------------------------------------------------------------------------
def setMcastGrp( mode, args ):
   mcastGroupAddr = args[ 'MCAST_GROUP' ]
   mcastAddr = Arnet.IpAddress( mcastGroupAddr )
   validateMcastAddrResult = IpAddrMatcher.validateMulticastIpAddr(
         str( mcastAddr ), allowReserved=False )
   if validateMcastAddrResult:
      mode.addError( validateMcastAddrResult )
      return

   vtiConfig = getVtiConfig( mode.intf.name )
   vtiConfig.floodMcastGrp = mcastAddr

def noSetMcastGrp( mode, args ):
   vtiConfig = getVtiConfig( mode.intf.name )
   vtiConfig.floodMcastGrp = Tac.newInstance( "Arnet::IpAddr", 0 )

class VxlanMulticastGroupCmd( CliCommand.CliCommandClass ):
   syntax = 'vxlan multicast-group MCAST_GROUP'
   noOrDefaultSyntax = 'vxlan multicast-group [ MCAST_GROUP ]'
   data = {
      'vxlan': vxlanNode,
      'multicast-group': CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher( 'multicast-group',
            helpdesc='Flood multicast group' ),
         guard=vxlanMcastGroupSupportedGuard ),
      'MCAST_GROUP': IpAddrMatcher.IpAddrMatcher(
         helpdesc='IP multicast group address' ),
   }
   handler = setMcastGrp
   noOrDefaultHandler = noSetMcastGrp

VxlanIntfModelet.addCommandClass( VxlanMulticastGroupCmd )

#----------------------------------------------------------------------------------
# "[no|default] bfd vtep evpn interval INTERVAL min-rx MIN_RX multiplier MULTIPLIER
#
# sets the bfd interval for vxlan in non-vcs mode
#----------------------------------------------------------------------------------
def setVxlanIntfBfdParams( mode, args ):
   minTx = args[ 'INTERVAL' ]
   minRx = args[ 'MIN_RX' ]
   mult = args[ 'MULTIPLIER' ]
   vtiConfig = getVtiConfig( mode.intf.name )
   vtiConfig.bfdIntervalParams = Tac.Value( 'Bfd::BfdIntervalConfig',
                                             minTx, minRx, mult )
   vtiConfig.bfdEnabled = True

def resetVxlanIntfBfdParams( mode, args ):
   vtiConfig = getVtiConfig( mode.intf.name )
   vtiConfig.bfdIntervalParams = Tac.Value( 'Bfd::BfdIntervalConfig',
                                                            BfdInterval.defval,
                                                            BfdInterval.defval,
                                                            BfdMultiplier.defval )
   vtiConfig.bfdEnabled = False

class VxlanBfdIntfParamsCmd( CliCommand.CliCommandClass ):
   syntax = 'bfd vtep evpn interval INTERVAL min-rx MIN_RX multiplier MULTIPLIER'
   noOrDefaultSyntax = 'bfd vtep evpn ...'
   data = {
      'bfd': CliCommand.Node(
          matcher=CliMatcher.KeywordMatcher( 'bfd',
             helpdesc='Configure BFD specific configurations' ),
          guard=isVxlan1InterfaceGuard ),
      'vtep': 'BFD for remote VTEPs',
      'evpn': 'EVPN Configuration',
      'interval': BfdCli.matcherInterval,
      'INTERVAL': matcherTxRxIntervalMs,
      'min-rx': BfdCli.matcherMinRx,
      'MIN_RX': matcherTxRxIntervalMs,
      'multiplier': BfdCli.matcherMultiplier,
      'MULTIPLIER': CliMatcher.IntegerMatcher( 3, 50, helpdesc='Range is 3-50' ),
   }
   handler = setVxlanIntfBfdParams
   noOrDefaultHandler = resetVxlanIntfBfdParams

if Toggles.VxlanToggleLib.toggleRemoteTunnelPICEnabled():
   VxlanIntfModelet.addCommandClass( VxlanBfdIntfParamsCmd )

def _verifyMcastGrp( mode, mcastGrps ):
   for mcastGrp in mcastGrps:
      mcastAddr = Arnet.IpAddress( mcastGrp )
      validateMcastAddrResult = IpAddrMatcher.validateMulticastIpAddr(
                str( mcastAddr ), allowReserved=False )
      if validateMcastAddrResult:
         mode.addError( validateMcastAddrResult )
         return False
   return True

def addMcastGrpDecap( mode, args ):
   mcastGrpDecapList = args[ 'DECAP' ]
   if not _verifyMcastGrp( mode, mcastGrpDecapList ):
      return

   vtiConfig = getVtiConfig( mode.intf.name )
   for grp in mcastGrpDecapList: 
      if grp not in vtiConfig.mcastGrpDecap:
         vtiConfig.mcastGrpDecap[ grp ] = True

def removeMcastGrpDecap( mode, args ):
   mcastGrpDecapList = args.get( 'DECAP' )
   vtiConfig = getVtiConfig( mode.intf.name )

   if mcastGrpDecapList:
      for grp in mcastGrpDecapList:
         del vtiConfig.mcastGrpDecap[ grp ]
   else:
      # Remove all if list is empty 
      vtiConfig.mcastGrpDecap.clear()

class VxlanMulticastGroupDecapCmd( CliCommand.CliCommandClass ):
   syntax = 'vxlan multicast-group decap { DECAP }'
   noOrDefaultSyntax = 'vxlan multicast-group decap [ { DECAP } ]'
   data = {
      'vxlan': vxlanNode,
      'multicast-group': CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher( 'multicast-group',
            helpdesc='Flood multicast group' ),
         guard=vxlanMcastDecapOnlySupportedGuard ),
      'decap': 'Decap only',
      'DECAP': IpAddrMatcher.IpAddrMatcher( helpdesc='IP multicast group address' ),
   }
   handler = addMcastGrpDecap
   noOrDefaultHandler = removeMcastGrpDecap

VxlanIntfModelet.addCommandClass( VxlanMulticastGroupDecapCmd )

#-------------------------------------------------------------------------------
# The "[no|default] vxlan multicast routing ipv4" command, in "config-vxlan-if"
#
# enables multicast routing on the vxlan interface
# turns default bridging on only for mcast packets, allows routing on the overlay
#-------------------------------------------------------------------------------
class VxlanMcastRoutingIpCmd( CliCommand.CliCommandClass ):
   syntax = 'vxlan multicast routing ipv4'
   noOrDefaultSyntax = syntax
   data = {
      'vxlan': vxlanNode,
      'multicast': CliCommand.Node( matcher=CliMatcher.KeywordMatcher( 'multicast',
                                              helpdesc='Multicast vxlan commands' ),
                                    guard=isVxlan1InterfaceGuard ),
      'routing': 'Enable multicast routing',
      'ipv4': 'IPv4 related',
   }

   @staticmethod
   def handler( mode, args ):
      vtiConfig = getVtiConfig( mode.intf.name )
      vtiConfig.mcastRouting = True

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      vtiConfig = getVtiConfig( mode.intf.name )
      vtiConfig.mcastRouting = False

VxlanIntfModelet.addCommandClass( VxlanMcastRoutingIpCmd )

#-------------------------------------------------------------------------------
# "config-vxlan-if" vxlan decapsulation filter
#                     ( disabled | ( interface multiple-vrf disabled [ { INTF } ] ) )
#
# sets vxlan decap filter mode / parameters
#-------------------------------------------------------------------------------
def vxlanDecapFilter( mode, args ):
   # The presence of 'interface' kw determines whether filter is disabled or relaxed
   vtiConfig = getVtiConfig( mode.intf.name )
   if 'interface' not in args:
      vtiConfig.decapFilterMode = 'filterDisabled'
      vtiConfig.decapFilterIntf.clear()
   else:
      # Filter is either relaxed for all intfs or just intfs in set
      if 'INTF' not in args:
         vtiConfig.decapFilterMode = 'filterRelaxedAll'
         vtiConfig.decapFilterIntf.clear()
      else:
         vtiConfig.decapFilterMode = 'filterRelaxedIntf'
         # Reconcile set
         for intf in vtiConfig.decapFilterIntf:
            if str( intf ) not in args[ 'INTF' ]:
               vtiConfig.decapFilterIntf.remove( intf )
         for intf in args[ 'INTF' ]:
            vtiConfig.decapFilterIntf.add( str( intf ) )

def noVxlanDecapFilter( mode, args ):
   vtiConfig = getVtiConfig( mode.intf.name )
   vtiConfig.decapFilterMode = 'filterEnabled'
   vtiConfig.decapFilterIntf.clear()

class VxlanDecapFilterCmd( CliCommand.CliCommandClass ):
   syntax = ( 'vxlan decapsulation filter '
         '( disabled | ( interface multiple-vrf disabled [ { INTF } ] ) )' )
   noOrDefaultSyntax = ( 'vxlan decapsulation filter ...' )
   data = {
         'vxlan': vxlanNode,
         'decapsulation': decapFilterNode,
         'filter': 'Configure VXLAN decapsulation filter',
         'disabled': 'Disable VXLAN decapsulation filter',
         'interface': 'Interface filter parameters',
         'multiple-vrf': 'Configure interfaces in multiple VRFs',
         'INTF': EthPhyIntf.ethMatcher,
   }
   handler = vxlanDecapFilter
   noOrDefaultHandler = noVxlanDecapFilter

VxlanIntfModelet.addCommandClass( VxlanDecapFilterCmd )

#-------------------------------------------------------------------------------
# "config-vxlan-if" vxlan [ mlag ] source-interface ( INTERFACE | ( Loopback NUM ) )
#
# sets the source interface for vti
# sets the ip address of the MLAG VTEP IP for Multi-VTEP MLAG
#-------------------------------------------------------------------------------
class VxlanSourceInterfaceCmd( CliCommand.CliCommandClass ):
   syntax = 'vxlan [ mlag ] source-interface INTERFACE'
   noOrDefaultSyntax = 'vxlan [ mlag ] source-interface ...'
   data = {
      'vxlan': vxlanNode,
      'mlag': CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher( 'mlag',
            helpdesc='Configure shared VTEP IP to use when VTEP IPs are unique.' ),
         guard=vxlanMultiVtepMlagGuard ),
      'source-interface': 'VXLAN source interface',
      'INTERFACE': LoopbackIntfCli.LoopbackIntf.matcher,
   }
   
   @staticmethod
   def handler( mode, args ):
      srcIntf = args[ 'INTERFACE' ]
      vtiConfig = getVtiConfig( mode.intf.name )
      if 'mlag' not in args:
         vtiConfig.srcIpIntf = srcIntf.name
      else:
         vtiConfig.mlagSrcIpIntf = srcIntf.name

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      vtiConfig = getVtiConfig( mode.intf.name )
      if 'mlag' not in args:
         vtiConfig.srcIpIntf = ''
      else:
         vtiConfig.mlagSrcIpIntf = ''

VxlanIntfModelet.addCommandClass( VxlanSourceInterfaceCmd )

#-------------------------------------------------------------------------------
# "config-vxlan-if" vxlan virtual-vtep local-interface ( INTERFACE | (Loopback NUM) )
#
# sets the source interface for virtual vtep
#-------------------------------------------------------------------------------
class VxlanVarpVtepLocalInterfaceCmd( CliCommand.CliCommandClass ):
   syntax = 'vxlan virtual-vtep local-interface INTERFACE'
   noOrDefaultSyntax = 'vxlan virtual-vtep local-interface ...'
   data = {
      'vxlan': vxlanNode,
      'virtual-vtep': CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher( 'virtual-vtep',
            helpdesc='Configure virtual VTEP IP address.' ),
         guard=vxlanVirtualVtepCmdSupportedGuard ),
      'local-interface': 'VXLAN virtual VTEP source interface',
      'INTERFACE': LoopbackIntfCli.LoopbackIntf.matcher,
   }

   @staticmethod
   def handler( mode, args ):
      srcIntf = args[ 'INTERFACE' ]
      vtiConfig = getVtiConfig( mode.intf.name )
      vtiConfig.varpVtepSrcIpIntf = srcIntf.name

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      vtiConfig = getVtiConfig( mode.intf.name )
      vtiConfig.varpVtepSrcIpIntf = ''
if Toggles.VxlanToggleLib.toggleVxlanVarpVtepConfigEnabled():
   VxlanIntfModelet.addCommandClass( VxlanVarpVtepLocalInterfaceCmd )

#-------------------------------------------------------------------------------
# vxlan flood vtep <vteps> - replaces the entire default vtep list
# vxlan flood vtep add <vteps> - add to default vtep list
# vxlan flood vtep remove <vteps> - remove from default vtep list
# no vxlan flood vtep - remove all vtep lists
# vxlan vlan <vlan> flood vtep <vteps> - replaces the entire vtep list
# vxlan vlan <vlan> flood vtep add <vteps> - add to existing list
# vxlan vlan <vlan> flood vtep remove <vteps> - remove from existing list
# no vxlan vlan <vlan> flood vtep - remove vtep list for a vlan
# no vxlan vlan flood vtep - remove all vtep lists
# show vxlan flood vtep [vlan <vlanId>]
#-------------------------------------------------------------------------------
def _verifyVtepAddrs( mode, vteps ):
   for vtep in vteps:
      ip = Tac.Value( 'Arnet::IpGenAddr', str( vtep ) )
      if ( ip.isMulticast or ip.isReserved or
           ip.isLoopback or ip.isLinkLocal or ip.isThisNetwork ):
         mode.addError( "invalid VTEP address: %s" % vtep )
         return False
   return True

def convertVtepsToIpGenAddr( vteps ):
   if vteps and not isinstance( vteps[ 0 ], Tac.Type( "Arnet::IpGenAddr" ) ):
      vteps = [ Arnet.IpGenAddr( str( vtep ) ) for vtep in vteps ]
   return vteps

def setDefaultFloodVtep( mode, vteps ):
   '''Remove the old default vtep list and create a new one'''
   vxlanConfig = getVxlanConfig( mode.intf.name )

   if not _verifyVtepAddrs( mode, vteps ):
      return
   vteps = convertVtepsToIpGenAddr( vteps )
   v4Vteps = set( v for v in vteps if v.af == 'ipv4' )
   v6Vteps = set( v for v in vteps if v.af == 'ipv6' )
   if v4Vteps:
      for v in vxlanConfig.floodVtepList:
         if Arnet.IpGenAddr( v ) not in v4Vteps:
            del vxlanConfig.floodVtepList[ v ]
   else:
      vxlanConfig.floodVtepList.clear()

   if v6Vteps:
      for v in vxlanConfig.floodVtepList6:
         if Arnet.IpGenAddr( str( v ) ) not in v6Vteps:
            del vxlanConfig.floodVtepList6[ v ]
   else:
      vxlanConfig.floodVtepList6.clear()
   for v in vteps:
      if v.af == 'ipv4':
         if v.v4Addr not in vxlanConfig.floodVtepList:
            vxlanConfig.floodVtepList[ v.v4Addr ] = True
      elif v.v6Addr not in vxlanConfig.floodVtepList6:
         vxlanConfig.floodVtepList6[ v.v6Addr ] = True

def noSetDefaultFloodVtep( mode ):
   '''Remove the default flood vtep list'''
   setDefaultFloodVtep( mode, vteps=[] )

def addRemoveDefaultFloodVtep( mode, vteps, addOrRemove ):
   '''Add or remove vteps from the default floodVtepList'''
   vxlanConfig = getVxlanConfig( mode.intf.name )
   vteps = convertVtepsToIpGenAddr( vteps )

   s = set( Arnet.IpGenAddr( ip ) 
            for ip in chain( vxlanConfig.floodVtepList,
                             ( str( vtep6 ) for vtep6 in vxlanConfig.floodVtepList6 )
            ) )
   t = set( vteps )
   if addOrRemove == 'add':
      s = s.union( t )
   else:
      s = s.difference( t )
   vteps = list(s)
   setDefaultFloodVtep( mode, vteps )

def setFloodVtep( mode, vSet, vteps ):
   '''Remove the old list and create a new one'''
   vxlanConfig = getVxlanConfig( mode.intf.name )

   if not _verifyVtepAddrs( mode, vteps ):
      return

   vteps = list( vteps )
   vteps = convertVtepsToIpGenAddr( vteps )
   # sync the config with the provided vtep list
   for vlan in vSet.ids:
      if vteps:
         if not vlan in vxlanConfig.vlanToVtepList:
            vxlanConfig.vlanToVtepList.newMember( vlan )
         for v in vxlanConfig.vlanToVtepList[ vlan ].remoteVtepAddr:
            if Arnet.IpGenAddr( v ) not in vteps:
               del vxlanConfig.vlanToVtepList[ vlan ].remoteVtepAddr[ v ]
         for v in vxlanConfig.vlanToVtepList[ vlan ].remoteVtepAddr6:
            if Arnet.IpGenAddr( str( v ) ) not in vteps:
               del vxlanConfig.vlanToVtepList[ vlan ].remoteVtepAddr6[ v ]
         for v in vteps:
            if v.af == 'ipv4':
               if v.v4Addr not in vxlanConfig.vlanToVtepList[ vlan ].remoteVtepAddr:
                  vxlanConfig.vlanToVtepList[ vlan ].remoteVtepAddr[ v.v4Addr
                        ] = True
            elif v.v6Addr not in vxlanConfig.vlanToVtepList[ vlan ].remoteVtepAddr6:
               vxlanConfig.vlanToVtepList[ vlan ].remoteVtepAddr6[ v.v6Addr ] = True
      else:
         if vlan in vxlanConfig.vlanToVtepList:
            del vxlanConfig.vlanToVtepList[ vlan ]

def addRemoveFloodVtep( mode, vSet, vteps, addOrRemove ):
   '''Add or remove vteps from the vlanToVtepList'''
   vxlanConfig = getVxlanConfig( mode.intf.name )
   vtiConfig = getVtiConfig( mode.intf.name )
   vteps = convertVtepsToIpGenAddr( vteps )
   invalidVlans = []
   for vlan in vSet.ids:
      vs = VlanCli.VlanSet( mode=mode, vlanSetString='', vlanIds=[ vlan ] )
      # allow 'add/remove' in per vlan configuration
      if vlan in vxlanConfig.vlanToVtepList:
         s = set( [ Arnet.IpGenAddr( ip ) for ip in
                       vxlanConfig.vlanToVtepList[ vlan ].remoteVtepAddr.keys() ] +
                  [ Arnet.IpGenAddr( str( ip ) ) for ip in
                       vxlanConfig.vlanToVtepList[ vlan ].remoteVtepAddr6.keys() ] )
         t = set( vteps )
         if addOrRemove == 'add':
            s = s.union( t )
         else:
            s = s.difference( t )
         setFloodVtep( mode, vs, s )
         continue

      # allow 'add' when default flood list is empty
      if not vxlanConfig.floodVtepList:
         if addOrRemove == 'add':
            setFloodVtep( mode, vs, vteps )
         continue

      # vlan without vni is nop and will not show error message
      if vtiConfig.vlanToVniMap.has_key( vlan ):
         invalidVlans.append( vlan )

   if invalidVlans != []:
      invalidVlansStr = MultiRangeRule.multiRangeToCanonicalString( invalidVlans )
      if addOrRemove == 'add':
         mode.addError( "VLANs %s are using the default VTEP flood list. " \
               "These VLANs do not have their own configured VTEP flood lists to " \
               "add the specified VTEPs to." % invalidVlansStr )
      else:
         mode.addError( "VLANs %s are using the default VTEP flood list. " \
               "These VLANs do not have their own configured VTEP flood lists to " \
               "remove the specified VTEPs from." % invalidVlansStr )

def noSetFloodVtep( mode, vSet ):
   vxlanConfig = getVxlanConfig( mode.intf.name )
   for vlan in vSet.ids:
      if vlan in vxlanConfig.vlanToVtepList:
         del vxlanConfig.vlanToVtepList[ vlan ]

def clearFloodVtep( mode, args ):
   vxlanConfig = getVxlanConfig( mode.intf.name )
   for vlan in vxlanConfig.vlanToVtepList:
      del vxlanConfig.vlanToVtepList[ vlan ]
   
def showFloodVtep( mode, args ):
   vSet = args.get( 'VLANS' )
   vxlanConfig = getVxlanConfig( 'Vxlan1' )
   LazyMount.force( vxHwStatusDir )
   vxlanFloodVtep = Tac.newInstance( "VxlanCliCapi::VxlanFloodVtep",
                                     vxHwStatusDir, vxlanConfig )
   if vSet is not None:
      for vlan in vSet.ids:
         vxlanFloodVtep.vlanFilter[ vlan ] = True

   fd = sys.stdout.fileno()
   fmt = mode.session_.outputFormat()
   vxlanFloodVtep.render( fd, fmt )
   sys.stdout.flush()
   #VxlanFloodVtepListModel is a Deferred Model, 
   # so do not have to populate the python Cli model.
   return VxlanModel.VxlanFloodVtepListModel

class VxlanVlanFloodVtepCmd( CliCommand.CliCommandClass ):
   noOrDefaultSyntax = 'vxlan vlan flood vtep'
   data = {
      'vxlan': vxlanNode,
      'vlan': CliCommand.Node( matcher=matcherVlan,
                               guard=isVxlan1InterfaceGuard ),
      'flood': matcherFlood,
      'vtep': vtepMatcherForConfig,
   }
   noOrDefaultHandler = clearFloodVtep

VxlanIntfModelet.addCommandClass( VxlanVlanFloodVtepCmd )

class VxlanVlanVlansetFloodVtepCmd( CliCommand.CliCommandClass ):
   syntax = 'vxlan [ vlan VLANS ] flood vtep [ ACTION ] { VTEP }'
   noOrDefaultSyntax = 'vxlan [ vlan VLANS ] flood vtep ...'
   data = {
      'vxlan': vxlanNode,
      'vlan': CliCommand.Node( matcher=matcherVlan,
                               guard=isVxlan1InterfaceGuard ),
      'VLANS': VlanCli.vlanSetMatcher,
      'flood': matcherFlood,
      'vtep': vtepMatcherForConfig,
      'ACTION': addRemoveMatcher,
      'VTEP': IpAddrMatcher.IpAddrMatcher(
         helpdesc='IP address of remote VTEP' ),
   }
   if Toggles.VxlanToggleLib.toggleVxlanV6FloodSetEnabled():
      data[ 'VTEP' ] = IpGenAddrMatcher.IpGenAddrMatcher(
         helpdesc='IP address of remote VTEP' )
   @staticmethod
   def handler( mode, args ):
      if 'ACTION' in args:
         if 'vlan' in args:
            addRemoveFloodVtep( mode, args[ 'VLANS' ], args[ 'VTEP' ],
                                args[ 'ACTION' ] )
         else:
            addRemoveDefaultFloodVtep( mode, args[ 'VTEP' ], args[ 'ACTION' ] )
      elif 'vlan' in args:
         setFloodVtep( mode, args[ 'VLANS' ], args[ 'VTEP' ] )
      else:
         setDefaultFloodVtep( mode, args[ 'VTEP' ] )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      if 'vlan' in args:
         noSetFloodVtep( mode, args[ 'VLANS' ] )
      else:
         noSetDefaultFloodVtep( mode )

VxlanIntfModelet.addCommandClass( VxlanVlanVlansetFloodVtepCmd )

#-------------------------------------------------------------------------------
# vxlan flood vtep learned data-plane - include active remote VTEPs in per-VLAN 
#                                       flood-lists
#-------------------------------------------------------------------------------
class VxlanFloodVtepLearnedDataPlane( CliCommand.CliCommandClass ):
   syntax = 'vxlan flood vtep learned data-plane'
   noOrDefaultSyntax = syntax
   data = {
      'vxlan': vxlanNode,
      'flood': matcherFlood,
      'vtep': vtepMatcherForConfig,
      'learned': 'Learned remote VTEPs',
      'data-plane': 'Include learned remote VTEPs in VXLAN flood-lists',
   }

   @staticmethod
   def handler( mode, args ):
      vtiConfig = getVtiConfig( mode.intf.name )
      vtiConfig.floodLearnedAll = True

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      vtiConfig = getVtiConfig( mode.intf.name )
      vtiConfig.floodLearnedAll = False

VxlanIntfModelet.addCommandClass( VxlanFloodVtepLearnedDataPlane )

def addRemoveVtep( mode, vteps, addOrRemove, ipsecProfile=None ):
   '''Add or remove vteps from the default floodVtepList'''
   vtiConfig = getVtiConfig( mode.intf.name )

   if addOrRemove:
      for v in vteps:
         vtiConfig.vtepToSecProfile[ v ] = ipsecProfile
   else:
      invalidVteps = []
      for v in vteps:
         if v not in vtiConfig.vtepToSecProfile:
            invalidVteps.append( v )
            continue
         del vtiConfig.vtepToSecProfile[ v ]
      if invalidVteps != []:
         invalidVtepsList = ', '.join( sorted( ip.v4Addr for ip in invalidVteps ) )
         mode.addWarning( 'No security profile mapping for VTEP { %s }' % \
                           invalidVtepsList )

class VxlanVtepsetSecurityProfileCmd( CliCommand.CliCommandClass ):
   syntax = 'vxlan vtep { VTEPS } ip security profile PROFILE'
   noOrDefaultSyntax = syntax.replace( 'PROFILE', '...' )
   data = {
      'vxlan': vxlanNode,
      'vtep': vtepMatcherForConfig,
      'VTEPS': IpGenAddrMatcher.IpGenAddrMatcher(
                 helpdesc='IP address of remote VTEP' ),
      'ip': 'IP config commands',
      'security': CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher( 'security',
            helpdesc='Security configuration' ),
         guard=vxlanSecurityGuard ),
      'profile': 'Configure security profile for VTEP',
      'PROFILE': ipsecProfileNameMatcher
   }

   @staticmethod
   def handler( mode, args ):
      addRemoveVtep( mode, args[ 'VTEPS' ], True, ipsecProfile=args[ 'PROFILE' ] )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      addRemoveVtep( mode, args[ 'VTEPS' ], False )  

if Toggles.VxlanToggleLib.toggleVxlanSecEnabled():
   VxlanIntfModelet.addCommandClass( VxlanVtepsetSecurityProfileCmd )

def setVxlanSecUdpPort( mode, args ):
   udpPort = args[ 'UDP_PORT' ]
   vtiConfig = getVtiConfig( mode.intf.name )
   vtiConfig.secUdpPort = udpPort

def noSetVxlanSecUdpPort( mode, args ):
   # Default to Vxlan udp port
   vtiConfig = getVtiConfig( mode.intf.name )
   vtiConfig.secUdpPort = vtiConfig.vxlanSecWellKnownPort

class VxlanSecUdpPortUdpPortCmd( CliCommand.CliCommandClass ):
   syntax = 'vxlan security udp-port UDP_PORT'
   noOrDefaultSyntax = 'vxlan security udp-port ...'
   data = {
      'vxlan': vxlanNode,
      'security': 'Security configuration',
      'udp-port': matcherUdpPort,
      'UDP_PORT': CliMatcher.IntegerMatcher( 1024, 65535, helpdesc='UDP port' ),
   }
   handler = setVxlanSecUdpPort
   noOrDefaultHandler = noSetVxlanSecUdpPort

if Toggles.VxlanToggleLib.toggleVxlanSecEnabled():
   VxlanIntfModelet.addCommandClass( VxlanSecUdpPortUdpPortCmd )

#-------------------------------------------------------------------------------
# vxlan learn-restrict any - learn from any IP address
# vxlan learn-restrict flood - use the flood list as learn list
# vxlan learn-restrict vtep <prefixes> - replaces the entire default prefix list
# vxlan learn-restrict vtep add <prefixes> - add to default prefix list
# vxlan learn-restrict vtep remove <prefixes> - remove from default prefix list
# vxlan learn-restrict vtep - learn from no prefixes i.e., learn from nobody
# no vxlan learn-restrict - revert to default i.e. learn from any
# vxlan vlan <vlan> learn-restrict any - learn from any IP address
# vxlan vlan <vlan> learn-restrict flood - use the vlan flood list as vlan learn list
# vxlan vlan <vlan> learn-restrict vtep <prefixes> - replaces the entire prefix list
# vxlan vlan <vlan> learn-restrict vtep add <prefixes> - add to existing list
# vxlan vlan <vlan> learn-restrict vtep remove <prefixes> - remove from existing list
# vxlan vlan <vlan> learn-restrict vtep - learn from no prefixes i.e., nobody
# no vxlan vlan <vlan> learn-restrict - remove per-vlan config
# no vxlan vlan learn-restrict vtep - remove per-vlan config for all vlans
# show vxlan learn-restrict vtep [vlan <vlanId>]
# show vxlan counters learn-restrict {prefix,brief,all} [vlan <vlanId>]
# clear vxlan counters learn-restrict [vlan <vlanId>]
#-------------------------------------------------------------------------------
def _verifyVtepPrefixes( mode, prefixes ):
   for p in prefixes:
      # The predicates return false if the prefix is too short to determine
      # i.e., 0.0.0.0/0 is not isMulticast etc.
      if p.isMulticast or p.isReserved or p.isLoopback or p.isLinkLocal or \
             p.isUnspecified:
         mode.addError( "invalid VTEP address prefix: %s" % p )
         return False
      if not bridgingHwCapabilities.vxlanIpv6UnderlaySupported and p.af == 'ipv6':
         mode.addError( "IPv6 encapsulation is not supported: %s" % p )
         return False
   return True

def setDefaultLearnRestrictNone( mode ):
   setDefaultLearnRestrict( mode, prefixes=[], learnFrom='learnFromList' )

def prefixesToVtepPrefixes( prefixes ):
   def _prefixToVtepPrefix( p ):
      return Tac.Value( 'Vxlan::VtepPrefix', p )
   if not prefixes:
      return []
   return [ _prefixToVtepPrefix( p ) for p in prefixes ]

def setDefaultLearnRestrict( mode, prefixes, learnFrom='learnFromList' ):
   '''Remove the old default prefix list and create a new one'''
   vxlanConfig = getVxlanConfig( mode.intf.name )

   if not _verifyVtepPrefixes( mode, prefixes ):
      return

   if not _verifyLearnFrom( mode, vxlanConfig.learnPrefixList, prefixes, \
                               vxlanConfig.learnFrom, learnFrom ):
      return

   # Form a list of Vxlan::VtepPrefix
   vps = prefixesToVtepPrefixes( prefixes )
   # remove any prefix that don't exist in the new list
   if vxlanConfig.learnPrefixList:
      for vp in vxlanConfig.learnPrefixList:
         if not vps or vp not in vps:
            del vxlanConfig.learnPrefixList[ vp ]
   # add any vteps missing from the new list
   if vps:
      for vp in vps:
         if vp not in vxlanConfig.learnPrefixList:
            vxlanConfig.learnPrefixList[ vp ] = True
   vxlanConfig.learnFrom = learnFrom

def _verifyLearnFrom( mode, oldPrefixes, newPrefixes, oldLearnFrom, newLearnFrom ):
   if oldLearnFrom == newLearnFrom:
      # No change
      return True

   if oldLearnFrom == 'learnFromList' and oldPrefixes:
      mode.addWarning( 'Configured learn list will be ignored' )
   if newLearnFrom == 'learnFromList' and not newPrefixes:
      mode.addWarning( 'Configured learn list is empty. Will learn from nobody' )
   return True

def noSetDefaultLearnRestrict( mode, args ):
   '''Remove the default learn prefix list. Learn from any i.e. default.'''
   setDefaultLearnRestrict( mode, prefixes=[], learnFrom='learnFromDefault' )

def addRemoveDefaultLearnPrefix( mode, prefixes, addOrRemove ):
   '''Add or remove prefixes from the default learn prefix list'''
   vxlanConfig = getVxlanConfig( mode.intf.name )

   # Have to get ipPrefix attribute for each of the existing Vxlan::VtepPrefix
   k = vxlanConfig.learnPrefixList.keys()
   s = set( w.ipPrefix for w in k )
   t = set( prefixes )
   if addOrRemove == 'add':
      s = s.union( t )
   else:
      s = s.difference( t )
   prefixes = list(s)
   setDefaultLearnRestrict( mode, prefixes, 'learnFromList' )

def setDefaultLearnFlood( mode ):
   '''Remove the default learn prefix list. Use flood list for learn.'''
   setDefaultLearnRestrict( mode, prefixes=[], learnFrom='learnFromFloodList' )

def setDefaultLearnAny( mode ):
   '''Remove the default learn prefix list. Learn from any.'''
   setDefaultLearnRestrict( mode, prefixes=[], learnFrom='learnFromAny' )

def setLearnRestrictNone( mode, vSet ):
   setLearnRestrict( mode, vSet, prefixes=[], learnFrom='learnFromList' )

def setLearnRestrict( mode, vSet, prefixes, learnFrom='learnFromList' ):
   '''Update the list'''
   vxlanConfig = getVxlanConfig( mode.intf.name )

   if not _verifyVtepPrefixes( mode, prefixes ):
      return

   # Form a list of Vxlan::VtepPrefix
   vps = prefixesToVtepPrefixes( prefixes )
   for vlan in vSet.ids:
      if vxlanConfig.vlanToLearnRestrict.has_key( vlan ):
         oldLF = vxlanConfig.vlanToLearnRestrict[ vlan ].learnFrom
         oldPref = vxlanConfig.vlanToLearnRestrict[ vlan ].prefixList
      else:
         oldLF = 'learnFromAny'
         oldPref = []
      if not _verifyLearnFrom( mode, oldPref, prefixes, oldLF, learnFrom ):
         return

      if not vlan in vxlanConfig.vlanToLearnRestrict:
         vxlanConfig.vlanToLearnRestrict.newMember( vlan )
      # remove any prefix that don't exist in the new list
      if vxlanConfig.vlanToLearnRestrict[ vlan ].prefixList:
         for vp in vxlanConfig.vlanToLearnRestrict[ vlan ].prefixList:
            if not vps or vp not in vps:
               del vxlanConfig.vlanToLearnRestrict[ vlan ].prefixList[ vp ]
      # add any vteps missing from the new list
      if vps:
         for vp in vps:
            if vp not in vxlanConfig.vlanToLearnRestrict[ vlan ].prefixList:
               vxlanConfig.vlanToLearnRestrict[ vlan ].prefixList[ vp ] = True
      vxlanConfig.vlanToLearnRestrict[ vlan ].learnFrom = learnFrom
      if learnFrom == 'learnFromDefault':
         del vxlanConfig.vlanToLearnRestrict[ vlan ]

def addRemoveLearnPrefix( mode, vSet, prefixes, addOrRemove ):
   '''Add or remove prefixes from the vlanToLearnRestrict'''
   vxlanConfig = getVxlanConfig( mode.intf.name )
   for vlan in vSet.ids:
      if vlan in vxlanConfig.vlanToLearnRestrict:
         # Have to get ipPrefix attribute for each of the existing Vxlan::VtepPrefix
         k = vxlanConfig.vlanToLearnRestrict[ vlan ].prefixList.keys()
         s = set( w.ipPrefix for w in k )
         t = set( prefixes )
         if addOrRemove == 'add':
            s = s.union( t )
         else:
            s = s.difference( t )
         prefixes = list(s)
         vs = VlanCli.VlanSet( mode=mode, vlanSetString='', vlanIds=[ vlan ] )
         setLearnRestrict( mode, vs, prefixes, 'learnFromList' )

      elif addOrRemove == 'add':
         vs = VlanCli.VlanSet( mode=mode, vlanSetString='', vlanIds=[ vlan ] )
         setLearnRestrict( mode, vs, prefixes, 'learnFromList' )

def noSetLearnRestrict( mode, vSet ):
   '''Remove the per-vlan learn prefix list. Learn based on default
   (non-vlan) config.'''
   setLearnRestrict( mode, vSet, prefixes=[], learnFrom='learnFromDefault' )

def setLearnFlood( mode, vSet ):
   '''Remove the per-vlan learn prefix list. Use flood list for learn.'''
   setLearnRestrict( mode, vSet, prefixes=[], learnFrom='learnFromFloodList' )

def setLearnAny( mode, vSet ):
   '''Remove the per-vlan learn prefix list. Use flood list for learn.'''
   setLearnRestrict( mode, vSet, prefixes=[], learnFrom='learnFromAny' )

def clearLearnPrefix( mode, args ):
   '''Clear the learn prefix for all the vlans.'''
   vxlanConfig = getVxlanConfig( mode.intf.name )
   for vlan in vxlanConfig.vlanToLearnRestrict:
      vs = VlanCli.VlanSet( mode=mode, vlanSetString='', vlanIds=[ vlan ] )
      setLearnRestrict( mode, vs, prefixes=[], learnFrom='learnFromDefault' )

def showLearnRestrict( mode, args ):
   vSet = args.get( 'VLANS' )
   vxlanStatus = getVxlanStatus( 'Vxlan1' )
   learnModel = VxlanModel.VxlanLearnRestrictModel()
   if vxlanStatus:
      for vlan in vxlanStatus.vlanToLearnRestrict:
         if vSet is not None and vlan not in vSet.ids:
            continue
         pl = VxlanModel.VxlanLearnRestrictModel.VxlanPrefixListModel()
         try:
            pl.learnFrom = vxlanStatus.vlanToLearnRestrict[ vlan ].learnFrom
            if pl.learnFrom == 'learnFromDefault':
               raise KeyError
            for ip in vxlanStatus.vlanToLearnRestrict[ vlan ].prefixList:
               pl.prefixList.append( ip.ipPrefix )
         except KeyError:
            continue
         learnModel.learnMap[ vlan ] = pl
   return learnModel

def showLearnCounters( mode, args ):
   counters = args[ 'COUNTER' ]
   vSet = args.get( 'VLANS' )
   learnStatus = vxHwStatusDir.learnStatus
   statusModel = VxlanModel.VxlanLearnCountersModel()
   statusModel.counters = counters
   if learnStatus:
      for vlan in learnStatus.learnStatusList:
         if vSet is not None and vlan not in vSet.ids:
            continue
         pl = statusModel.VxlanPrefixStatusModel()
         try:
            status = learnStatus.learnStatusList[ vlan ]
            pl.learnFrom = status.learnFrom
            if pl.learnFrom == 'learnFromDefault':
               raise KeyError
            for ip in status.numMatches:
               pl.prefixList[ ip.ipPrefix ] = status.numMatches[ ip ]
            pl.numMatchAny = status.numMatchAny
            pl.numMatchFloodList = status.numMatchFloodList
            pl.numMatchList = status.numMatchList
            pl.numRejectFloodList = status.numRejectFloodList
            pl.numRejectList = status.numRejectList
         except KeyError:
            continue
         statusModel.learnCountersMap[ vlan ] = pl

   return statusModel

def clearLearnCounters( mode, args ):
   vSet = args.get( 'VLANS' )
   vxlanCounterConfig = getVxlanCounterConfig( mode.intf.name )
   learnStatus = vxHwStatusDir.learnStatus
   now = Tac.now()

   if learnStatus:
      for vlan in learnStatus.learnStatusList:
         if vSet is not None and vlan not in vSet.ids:
            continue
         try:
            del vxlanCounterConfig.vlanToClearCounterRequestTime[ vlan ]
            vxlanCounterConfig.vlanToClearCounterRequestTime[ vlan ] = now
         except KeyError:
            pass

      for vlan in learnStatus.learnStatusList:
         if vSet is not None and vlan not in vSet.ids:
            continue
         try:
            Tac.waitFor(
               lambda:
               learnStatus.learnStatusList.has_key( vlan ) and
               learnStatus.learnStatusList[ vlan ].lastClearTime >= now,
               description='learn-restrict clear counter request to complete',
               warnAfter=None, sleep=True, maxDelay=0.5, timeout=5 )
         except Tac.Timeout:
            mode.addWarning( "learn-restrict counters may not have been reset yet" )

#--------------------------------------------------------------------------------
# vxlan learn-restrict ( any | flood | ( vtep [ { VTEP } ] ) )
#--------------------------------------------------------------------------------
class VxlanLearnRestrictCmd( CliCommand.CliCommandClass ):
   syntax = 'vxlan learn-restrict ( any | flood | ( vtep [ { PREFIX_ADDR } ] ) )'
   noOrDefaultSyntax = 'vxlan learn-restrict ...'
   data = {
      'vxlan': vxlanNode,
      'learn-restrict': matcherLearnRestrict,
      'any': matcherAny,
      'flood': matcherFlood,
      'vtep': vtepMatcherForConfig,
      'PREFIX_ADDR': ipGenPrefixMatcher,
   }
   @staticmethod
   def handler( mode, args ):
      if 'any' in args:
         return setDefaultLearnAny( mode )
      elif 'flood' in args:
         return setDefaultLearnFlood( mode )
      else:
         assert 'vtep' in args
         if 'PREFIX_ADDR' in args:
            return setDefaultLearnRestrict( mode, args[ 'PREFIX_ADDR' ] )
         else:
            return setDefaultLearnRestrictNone( mode )

   noOrDefaultHandler = noSetDefaultLearnRestrict

VxlanIntfModelet.addCommandClass( VxlanLearnRestrictCmd )

# Allow setting zero prefixes
class AddRemoveLearnPrefixCmd( CliCommand.CliCommandClass ):
   syntax = 'vxlan [ vlan VLANS ] learn-restrict vtep ACTION { PREFIX_ADDR }'
   data = {
      'vxlan': vxlanNode,
      'vlan': CliCommand.Node( matcher=matcherVlan,
                               guard=isVxlan1InterfaceGuard ),
      'VLANS': VlanCli.vlanSetMatcher,
      'learn-restrict': matcherLearnRestrict,
      'ACTION': addRemoveMatcher,
      'vtep': vtepMatcherForConfig,
      'PREFIX_ADDR': ipGenPrefixMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      if 'vlan' in args:
         addRemoveLearnPrefix( mode, args[ 'VLANS' ], args[ 'PREFIX_ADDR' ],
                               args[ 'ACTION' ] )
      else:
         addRemoveDefaultLearnPrefix( mode, args[ 'PREFIX_ADDR' ], args[ 'ACTION' ] )

VxlanIntfModelet.addCommandClass( AddRemoveLearnPrefixCmd )

# Allow setting zero prefixes
class LearnRestrictNoneCmd( CliCommand.CliCommandClass ):
   syntax = 'vxlan vlan VLANS learn-restrict vtep'
   data = {
            'vxlan': vxlanNode,
            'vlan': CliCommand.Node( matcher=matcherVlan,
                                     guard=isVxlan1InterfaceGuard ),
            'VLANS': VlanCli.vlanSetMatcher,
            'learn-restrict': matcherLearnRestrict,
            'vtep': vtepMatcherForConfig,
          }
   handler = lambda mode, args: setLearnRestrictNone( mode, args[ 'VLANS' ] )

VxlanIntfModelet.addCommandClass( LearnRestrictNoneCmd )

class VxlanVlansetLearnRestrictCmd( CliCommand.CliCommandClass ):
   syntax = ( 'vxlan vlan VLANS learn-restrict '
                                       '( any | flood | ( vtep { PREFIX_ADDR } ) )' )
   noOrDefaultSyntax = 'vxlan vlan VLANS learn-restrict ...'
   data = {
      'vxlan': vxlanNode,
      'vlan': CliCommand.Node( matcher=matcherVlan,
                               guard=isVxlan1InterfaceGuard ),
      'VLANS': VlanCli.vlanSetMatcher,
      'learn-restrict': matcherLearnRestrict,
      'any': matcherAny,
      'flood': matcherFlood,
      'vtep': vtepMatcherForConfig,
      'PREFIX_ADDR': ipGenPrefixMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      if 'any' in args:
         return setLearnAny( mode, args[ 'VLANS' ] )
      elif 'flood' in args:
         return setLearnFlood( mode, args[ 'VLANS' ] )
      else:
         assert 'vtep' in args
         return setLearnRestrict( mode, args[ 'VLANS' ], args[ 'PREFIX_ADDR' ] )

   noOrDefaultHandler = lambda mode, args: noSetLearnRestrict( mode,
                                                               args[ 'VLANS' ] )

VxlanIntfModelet.addCommandClass( VxlanVlansetLearnRestrictCmd )

class ClearLearnPrefixCmd( CliCommand.CliCommandClass ):
   noOrDefaultSyntax = 'vxlan vlan learn-restrict vtep'
   data = {
      'vxlan': vxlanNode,
      'vlan': matcherVlan,
      'learn-restrict': matcherLearnRestrict,
      'vtep': vtepMatcherForConfig,
   }
   noOrDefaultHandler = clearLearnPrefix

VxlanIntfModelet.addCommandClass( ClearLearnPrefixCmd )

#--------------------------------------------------------------------------------
# clear vxlan counters learn-restrict [ vlan VLANS ]
#--------------------------------------------------------------------------------
class ClearVxlanCountersLearnRestrictCmd( CliCommand.CliCommandClass ):
   syntax = 'clear vxlan counters learn-restrict [ vlan VLANS ]'
   data = {
      'clear': CliToken.Clear.clearKwNode,
      'vxlan': vxlanNode,
      'counters': 'VXLAN counters',
      'learn-restrict': matcherLearnRestrict,
      'vlan': matcherVlan,
      'VLANS': VlanCli.vlanSetMatcher,
   }
   handler = clearLearnCounters

VxlanIntfModelet.addCommandClass( ClearVxlanCountersLearnRestrictCmd )

#-------------------------------------------------------------------------------
# vxlan vni <VNI> routed ip access-list <NAME> in
#-------------------------------------------------------------------------------
vniMatcherForConfig = CliMatcher.KeywordMatcher( 'vni',
                            helpdesc='VXLAN Network Identifier configuration' )
vniDecapsulationMatcher = CliMatcher.KeywordMatcher( 'decapsulation',
                                 helpdesc='List of additional VNIs' )

def setVniAcl( mode, args ):
   vni = args[ 'VNI' ]
   aclName = args[ 'ACL_NAME' ]
   vxlanConfig = getVxlanConfig( mode.intf.name )
   vniNum = VniFormat( vni ).toNum()
   if not isValidVniWithError( mode, vniNum, mode.intf.name ):
      return
   vxlanConfig.vniToIpAclMap[ vniNum ] = aclName

def noVniAcl( mode, args ):
   vni = args[ 'VNI' ]
   aclName = args.get( 'ACL_NAME' )
   vxlanConfig = getVxlanConfig( mode.intf.name )
   vniNum = VniFormat( vni ).toNum()
   if not isValidVniWithError( mode, vniNum, mode.intf.name ):
      return
   currentAclName = vxlanConfig.vniToIpAclMap.get( vniNum )
   if aclName and currentAclName and aclName != currentAclName:
      mode.addError( "Access list %s not configured on VNI %d" % ( aclName, 
                                                                   vniNum ) )
   else:
      del vxlanConfig.vniToIpAclMap[vniNum]

class VxlanVniRoutedIpAccessListAclnameInCmd( CliCommand.CliCommandClass ):
   syntax = 'vxlan vni VNI routed ip access-list ACL_NAME in'
   noOrDefaultSyntax = 'vxlan vni VNI routed ip access-list [ ACL_NAME ] in'
   data = {
      'vxlan': vxlanNode,
      'vni': vniMatcherForConfig,
      'VNI': vniMatcher,
      'routed': CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher( 'routed',
            helpdesc='Apply to interfaces for VXLAN routing' ),
         guard=vxlanTunnelIngressDecapAclSupported ),
      'ip': AclCli.ipKwForServiceAclMatcher,
      'access-list': AclCli.accessListKwMatcher,
      'ACL_NAME': AclCli.ipAclNameMatcher,
      'in': AclCli.inKwMatcher,
   }
   handler = setVniAcl
   noOrDefaultHandler = noVniAcl

VxlanIntfModelet.addCommandClass( VxlanVniRoutedIpAccessListAclnameInCmd )

#-------------------------------------------------------------------------------
# vxlan vlan <vlanRange> vni <vniRange>
# vxlan vlan add <vlanRange> vni <vniRange>
# vxlan vlan remove <vlanRange> vni <vniRange>
# [no|default] vxlan vlan <vlanRange> vni
#
# sets the vni to vlan mappings under interface vxlan
#-------------------------------------------------------------------------------
def validateVlanToVniAdd( mode, vlanList, vniList ):
   vtiConfig = getVtiConfig( mode.intf.name )
   # Command fails unless all mappings in range are valid
   for vlan, vni in zip( vlanList, vniList ):
      vni = VniFormat( vni ).toNum()
      if not isValidVniWithError( mode, vni, mode.intf.name ):
         return
      if not mode.session_.startupConfig():
         exisitingVlan = vtiConfig.vniInVlanCheck( vlan, vni )
         if exisitingVlan != 0:
            mode.addError( "VLAN %d is already mapped to VNI %d" %
                           ( exisitingVlan, vni ) )
            return
      # If a vlan is in use as an internal vlan by a routed port,
      # vlan to Vni creation for that vlan is not supported.
      vc = bridgingConfig.vlanConfig.get( vlan )
      if vc and vc.internal:
         mode.addError( "VLAN %d is already used as an internal VLAN." % ( vlan ) )
         return
      # If the mlag pair is configured ('primary', 'secondary', or negotiating 
      # 'inactive' ) a the peer link SVI vlan cannot be mapped
      if mlagStatus.mlagState in ( 'primary', 'secondary', 'inactive' ):
         if mlagStatus.localInterface:
            vlanIntf = mlagStatus.localInterface.intfId
            vlanId = vlanIntf[ len( 'Vlan' ): ]
            if vlanId == str( vlan ):
               mode.addError( "VLAN %d is already used as the peer-link SVI VLAN." %
                              ( vlan ) )
               return
      # If the VNI is used in a vrfToVniMap, then reject
      for vrf in vtiConfig.vrfToVniMap:
         if vtiConfig.vrfToVniMap[ vrf ] == vni:
            mode.addError( "VNI %s is already used to map VRF %s" %
                           ( str( vni ), vrf ) )
            return
      if vni in vtiConfig.alternateDecapVniToVrfMap:
         vrf = vtiConfig.alternateDecapVniToVrfMap[ vni ]
         mode.addError( "VNI %s is already used to map VRF %s" %
                         ( str( vni ), vrf ) )
   # The old syntax for setting vlan to vni mapping accepted single values only:
   # e.g. (config-if-Vx1)$ vxlan vlan <vlan> vni <vni>
   # The new syntax accepts ranges:
   # e.g. (config-if-Vx1)$ vxlan vlan <vlanList> vni <vniList>
   # If the new syntax is used, we set vlanVniRangeSyntax in VtiConfigDir so
   # that command is stored as single line in running-config. Otherwise, we
   # store mappings in individual lines.
   if len( vlanList ) > 1:
      vtiConfigDir.vlanVniRangeSyntax = True
   for vlan, vni in zip( vlanList, vniList ):
      vni = VniFormat( vni ).toNum()
      vtiConfig.vlanToVniMap[ vlan ] = vni
      vniToVlanMap.setVni( vni, vlan, mode.intf.name )

def validateVlanToVniRemove( mode, vlanList, vniList ):
   vtiConfig = getVtiConfig( mode.intf.name )
   if vniList is not None:
      # All deletions must be validated before we can commit anything
      for vlan, vni in zip( vlanList, vniList ):
         # If vlan is not mapped, ignore the delete request
         if not vtiConfig.vlanToVniMap.has_key( vlan ):
            continue
         vni = VniFormat( vni ).toNum()
         if not isValidVniWithError( mode, vni, mode.intf.name ):
            return
         if vtiConfig.vlanToVniMap[ vlan ] != vni:
            mode.addError( "vlan %d is not mapped to vni %s" %
                  ( vlan, vni ) )
            return
   for vlan in vlanList:
      # Delete vni to vlan mapping
      if vtiConfig.vlanToVniMap.has_key( vlan ):
         vni = vtiConfig.vlanToVniMap[ vlan ]
         vniToVlanMap.delVni( vni, mode.intf.name )
      # We tolerate overconstrained vlan list for usability,
      # ie. `$ vxlan vlan 1-20 vni` if only 1-10,14-20 are valid
      # If vni list is passed, explicit checking is done above
      try:
         del vtiConfig.vlanToVniMap[ vlan ]
      except KeyError:
         continue

def setVlanToVniMapping( mode, vlanList, vniList ):
   if len( vlanList ) != len( vniList ):
      mode.addError( "Number of VLANs must equal number of VNIs" )
      return
   vtiConfig = getVtiConfig( mode.intf.name )
   # If mapping with new syntax already exists, add keyword is needed
   if vtiConfigDir.vlanVniRangeSyntax and vtiConfig.vlanToVniMap:
      mode.addError( "VLAN to VNI mapping already exists. Use 'add/remove' " +
                     "keyword to modify existing map" )
      return
   # Validate before adding mappings
   validateVlanToVniAdd( mode, vlanList, vniList )

def updateVlanToVniMapping( mode, vlanList, vniList, addOrRemove ):
   if len( vlanList ) != len( vniList ):
      mode.addError( "Number of VLANs must equal number of VNIs" )
      return
   if addOrRemove == 'add':
      validateVlanToVniAdd( mode, vlanList, vniList )
   else:
      validateVlanToVniRemove( mode, vlanList, vniList )

def noSetVlanToVniMapping( mode, vlanList, vniList=None ):
   # If a vniList is passed, we must validate the mappings
   if vniList is not None and len( vlanList ) != len( vniList ):
      mode.addError( "Number of VLANs must be equal to number of VNIs" )
      return
   validateVlanToVniRemove( mode, vlanList, vniList )

def convertVlanToVniMapSyntax( mode ):
   # New syntax must persist even if Vxlan interface is removed and readded
   vtiConfigDir.vlanVniRangeSyntax = True

#-------------------------------------------------------------------------------
# Register convertVlanToVniMapSyntax via "config convert new-syntax"
#-------------------------------------------------------------------------------
ConfigConvert.registerConfigConvertCallback( convertVlanToVniMapSyntax )

class VlanVniMultiRangeCmd( CliCommand.CliCommandClass ):
   syntax = 'vxlan vlan [ ACTION ] VLAN_LIST vni VNI_LIST'
   noOrDefaultSyntax = 'vxlan vlan VLAN_LIST vni [ VNI_LIST ]'
   data = {
      'vxlan': vxlanNode,
      'vlan': CliCommand.Node( matcher=matcherVlan,
                               guard=isVxlan1InterfaceGuard ),
      'ACTION': CliMatcher.EnumMatcher( {
         'add': 'Add VLAN to VNI mapping',
         'remove': 'Remove VLAN to VNI mapping',
      } ),
      'VLAN_LIST': vlanMultiMatcher,
      'vni': vniMatcherForConfig,
      'VNI_LIST': vniMultiMatcher,
   }

   @staticmethod
   def adapter( mode, args, argsList ):
      vnis = args.get( 'VNI_LIST' )
      if vnis is None:
         return

      # Validate VNI list size before handlers
      vniList = vnis.values()
      if len(vniList) > 4094:
         mode.addError( "VNI list size cannot exceed VLAN range" )
         raise CliParserCommon.AlreadyHandledError
      else:
         args[ 'VNI_LIST' ] = vniList

   @staticmethod
   def handler( mode, args ):
      vlanList = args[ 'VLAN_LIST' ].values()
      vniList = args[ 'VNI_LIST' ]
      if 'ACTION' in args:
         updateVlanToVniMapping( mode, vlanList, vniList, args[ 'ACTION' ] )
      else:
         setVlanToVniMapping( mode, vlanList, vniList )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      vlanList = args[ 'VLAN_LIST' ].values()
      vniList = args.get( 'VNI_LIST' )
      noSetVlanToVniMapping( mode, vlanList, vniList )

VxlanIntfModelet.addCommandClass( VlanVniMultiRangeCmd )

def vniEligibleForVrfMapping( mode, vrfName, vniDottedFormat, decap=False ):
   '''
   Verify if the received VNI is valid and eligible to be vrf mapped
   '''
   vtiConfig = getVtiConfig( mode.intf.name )
   vni = VniFormat( vniDottedFormat ).toNum()
   if not isValidVniWithError( mode, vni, mode.intf.name ):
      return None

   if not mode.session_.startupConfig():
      for ( existingVrf, existingVni ) in vtiConfig.vrfToVniMap.items():
         if existingVni == vni and existingVrf != vrfName:
            mode.addError( "VRF %s is already mapped to VNI %d" %
                           ( existingVrf, vni ) )
            return None
         if decap and existingVni == vni:
            mode.addError( "VRF %s is already mapped to VNI %d" %
                           ( existingVrf, vni ) )
            return None
      if vni in vtiConfig.alternateDecapVniToVrfMap:
         mode.addError( "VRF %s is already mapped to alternate VNI %d" %
                        ( vtiConfig.alternateDecapVniToVrfMap[ vni ], vni ) )
         return None

   # If the VNI is used in a vlanToVniMap, then reject
   for vlan, mappedVni in vtiConfig.vlanToVniMap.iteritems():
      if mappedVni == vni:
         mode.addError( "VNI %s is already used to map vlan %d" %
                        ( vniDottedFormat, vlan ) )
         return None
   return vni
   
#-------------------------------------------------------------------------------
# The "[no|default] vxlan vrf <vlanId> vni <vni>, in "config-if-Vx"
#
# sets the vni to vlan mapping
#-------------------------------------------------------------------------------

def setVrfToVniMapping( mode, args ):
   vniDottedFormat = args[ 'VNI' ]
   vni = vniEligibleForVrfMapping( mode, args[ 'VRF' ], vniDottedFormat )
   if not vni:
      return
   vtiConfig = getVtiConfig( mode.intf.name )
   vrfName = args[ 'VRF' ]
   # create vrf to vni mapping
   vtiConfig.vrfToVniMap[ vrfName ] = vni

def noSetVrfToVniMapping( mode, args ):
   vrfName = args[ 'VRF' ]
   vniDottedFormat = args.get( 'VNI' )
   vtiConfig = getVtiConfig( mode.intf.name )

   # If vrf is not mapped, ignore the delete request
   if not vtiConfig.vrfToVniMap.has_key( vrfName ):
      return

   if vniDottedFormat is not None:
      vni = VniFormat( vniDottedFormat ).toNum()
      if not isValidVniWithError( mode, vni, mode.intf.name ):
         return

      if vtiConfig.vrfToVniMap[ vrfName ] != vni:
         mode.addError( "VRF %s is not mapped to VNI %s" %
                        ( vrfName, vniDottedFormat ) )
         return

   # delete vrf to vni mapping
   if vtiConfig.vrfToVniMap.has_key( vrfName ):
      vni = vtiConfig.vrfToVniMap[ vrfName ]
   del vtiConfig.vrfToVniMap[ vrfName ]

def mapDecapVni( mode, vrfName, vniSet, addOrRemove ):
   vtiConfig = getVtiConfig( mode.intf.name )
   validVnis = []
   if addOrRemove == 'add':
      for vniDottedFormat in vniSet:
         vni = vniEligibleForVrfMapping( mode, vrfName, vniDottedFormat, decap=True )
         if not vni:
            # A warning is already displayed in vniEligibleForVrfMapping()
            # No need to do it again
            return
         validVnis.append( vni )
      for vni in validVnis:
         vtiConfig.alternateDecapVniToVrfMap[ vni ] = vrfName
   else:
      for vniDottedFormat in vniSet:
         if not vniDottedFormat:
            return
         vni = VniFormat( vniDottedFormat ).toNum()
         if not isValidVniWithError( mode, vni, mode.intf.name ):
            return
         if ( vni not in vtiConfig.alternateDecapVniToVrfMap or
              vrfName != vtiConfig.alternateDecapVniToVrfMap[ vni ] ):
            mode.addError( "VRF %s is not mapped to VNI %s" %
                            ( vrfName, vniDottedFormat ) )
            return
         validVnis.append( vni )
      for vni in validVnis:
         del vtiConfig.alternateDecapVniToVrfMap[ vni ]

def getVniSet( vniList ):
   # Multiple space separated
   if not vniList:
      return set()
   return set( vniList )

def setDefaultVrfToMultipleVniMapping( mode, args ):
   vtiConfig = getVtiConfig( mode.intf.name )
   vrfName = args[ 'VRF' ]
   vnis = getVniSet( args[ 'DECAPVNI' ] )
   vrfDecapVnis = set()
   for vni, vrf in vtiConfig.alternateDecapVniToVrfMap.iteritems():
      if vrf == vrfName:
         vrfDecapVnis.add( vni )
   removeVnis = vrfDecapVnis - vnis
   addVnis = vnis - vrfDecapVnis
   mapDecapVni( mode, vrfName, addVnis, "add" )
   mapDecapVni( mode, vrfName, removeVnis, "remove" )

def setVrfToMultipleVniMapping( mode, args ):
   addOrRemove = args[ 'ACTION' ]
   vrfName = args[ 'VRF' ]
   vniSet = getVniSet( args[ 'DECAPVNI' ] )
   return mapDecapVni( mode, vrfName, vniSet, addOrRemove )

def noSetVrfToMultipleVniMapping( mode, args ):
   vtiConfig = getVtiConfig( mode.intf.name )
   vrfName = args[ 'VRF' ]
   vniSet = getVniSet( args.get( 'DECAPVNI', [] ) )
   validVnis = []
   for vniDottedFormat in vniSet:
      vni = VniFormat( vniDottedFormat ).toNum()
      if not vni:
         mode.addError( "Invalid VNI %s entered" % ( str( vni ) ) )
         return
      if vni not in vtiConfig.alternateDecapVniToVrfMap or \
         vtiConfig.alternateDecapVniToVrfMap[ vni ] != vrfName:
         mode.addError( "VRF %s is not mapped to VNI %s" %
                        ( vrfName, vni ) )
         return
      validVnis.append( vni )
   if not validVnis:
      # Delete all decap VNIs
      for vni, vrf in vtiConfig.alternateDecapVniToVrfMap.items():
         if vrf == vrfName:
            del vtiConfig.alternateDecapVniToVrfMap[ vni ]
      return
   for vni in validVnis:
      # Delete individual decap VNIs but not all
      del vtiConfig.alternateDecapVniToVrfMap[ vni ]

class VrfToMultipleVniMappingCmd( CliCommand.CliCommandClass ):
   syntax = '''vxlan VRF vni
               ( ( VNI )
               | ( decapsulation [ ACTION ] DECAPVNI ) )'''
   noOrDefaultSyntax = '''vxlan VRF vni
                          [ ( VNI )
                          | ( decapsulation [ DECAPVNI ] ) ]'''
   data = {
      'vxlan': vxlanNode,
      'VRF': vrfExprFactory,
      'vni': vniMatcherForConfig,
      'VNI': vniMatcher,
      'decapsulation': vniDecapsulationMatcher,
      'ACTION': vniAddRemoveMatcher,
      'DECAPVNI' : vniMultiMatcher,
   }

   @staticmethod
   def adapter( mode, args, argsList ):
      vnis = args.get( 'DECAPVNI' )
      if vnis is None:
         args[ 'DECAPVNI' ] = []
      else:
         vniList = vnis.values()
         args[ 'DECAPVNI' ] = vniList

   @staticmethod
   def handler( mode, args ):
      if 'VNI' in args:
         # Evpn Vrf to Vni map
         setVrfToVniMapping( mode, args )
      elif 'ACTION' in args:
         setVrfToMultipleVniMapping( mode, args )
      else:
         setDefaultVrfToMultipleVniMapping( mode, args )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      if 'decapsulation' not in args:
         noSetVrfToVniMapping( mode, args )
      else:
         noSetVrfToMultipleVniMapping( mode, args )

class VrfToVniMappingCmd( CliCommand.CliCommandClass ):
   syntax = 'vxlan VRF vni VNI'
   noOrDefaultSyntax = 'vxlan VRF vni [ VNI ]'
   data = {
      'vxlan': vxlanNode,
      'VRF': vrfExprFactory,
      'vni': vniMatcherForConfig,
      'VNI': vniMatcher,
   }
   handler = setVrfToVniMapping
   noOrDefaultHandler = noSetVrfToVniMapping

if Toggles.VxlanToggleLib.toggleVrfToMultipleVnisMappingEnabled():
   VxlanIntfModelet.addCommandClass( VrfToMultipleVniMappingCmd )
else:
   VxlanIntfModelet.addCommandClass( VrfToVniMappingCmd )

#-------------------------------------------------------------------------------
#-------------------------------------------------------------------------------
# The "[no] mac address-table static vlan <vlan> interface Vxlan<n>
#           vtep <ipAddr>
#
# configures a static host in vxlan fdb in global config mode
#-------------------------------------------------------------------------------
def setStaticVxlanHost( mode, args ):
   macAddr = args[ 'MAC_ADDR' ]
   vlanId = args[ 'VLAN_ID' ]
   intfList = args[ 'INTF' ]
   vtepAddr = args[ 'VTEP' ]

   if not Ethernet.isUnicast( macAddr ):
      mode.addError( 'Adding a static multicast address is not supported' )
      return
   
   intfNames = list( intfList.intfNames() )
   if len( intfNames ) > 1:
      mode.addError( 'Multiple interfaces are not allowed' )
      return

   intfName = intfNames.pop()
   if isVtiMissing( intfName ):
      mode.addError( 'Interface %s does not exist' % intfName )
      return

   if ( vtepAddr.isAddrZero or not vtepAddr.isUnicast ):
      mode.addError( 'Invalid unicast address for VTEP' )
      return

   macAddr = Ethernet.convertMacAddrToCanonical( macAddr )

   macVlanPair = Tac.Value( "Vxlan::MacVlanPair", macAddr, vlanId.id )
   vti = vtiNameToId.get( intfName )

   vxlanConfigDir.fdbConfig.configuredHost.addMember( Tac.Value(
         "Vxlan::ConfiguredHost", macVlanPair, vti, vtepAddr ) )

   # delete l2 static mac entry
   fdbConfig = bridgingCliConfig.fdbConfig.newMember( vlanId.id )
   table = fdbConfig.configuredHost
   if table.get( macAddr ):
      del table[ macAddr ]

def noSetStaticVxlanHost( mode, macAddr, vlanId, intfList ):
   macAddr = Ethernet.convertMacAddrToCanonical( macAddr )
   macVlanPair = Tac.Value( "Vxlan::MacVlanPair", macAddr, vlanId.id )
   configuredHost = vxlanConfigDir.fdbConfig.configuredHost

   if configuredHost.get( macVlanPair ):
      del configuredHost[ macVlanPair ]
      return True
   else:
      if mode:
         mode.addWarning( 'Address not found' )
      return False

def filterVxlanConfiguredMacAddr( host, vlanId, macAddr, vteps ):
   if vlanId:
      if vlanId.id != host.macVlanPair.vlanId:
         return False
   if macAddr:
      if Ethernet.convertMacAddrToCanonical( macAddr ) != \
             host.macVlanPair.macAddr:
         return False
   if vteps:
      if host.remoteVtepGenAddr not in vteps:
         return False
   return True

# verify that intf is not vxlan
def bridgingOkToAddHandler( mode, macAddr, vlanId, intfNamesOrDrop=None ):
   if intfNamesOrDrop is None or intfNamesOrDrop == 'drop':
      return True
   for intfName in intfNamesOrDrop:
      if 'Vxlan' in intfName:
         if len( intfNamesOrDrop ) == 1:
            mode.addError( "vtep must be specified for %s" % intfName )
         else:
            mode.addError( "%s is not allowed with other interfaces" % intfName )
         return False
   return True

# When ebra static mac is added to one of the local interfaces,
# remove corresponding vxlan static mac address.
def bridgingSetStaticHandler( macAddr, vlanId, intfsOrDrop=None ):
   # remove the vxlan host for the same macVlan pair
   noSetStaticVxlanHost( None, macAddr, vlanId, None )
   return False

# When ebra static mac is removed via 'no mac addr static vlan <vlan>', i.e.,
# without vtep token, Ebra CliPlugin will handle the command and
# notify vxlan to remove the addr
def bridgingDelStaticHandler( macAddr, vlanId, intfsOrDrop=None ):
   return noSetStaticVxlanHost( None, macAddr, vlanId, None )

# When 'show mac address-table configured' is issued, Ebra CliPlugin 
# will inform Vxlan through this function to display Vxlan configured
# entries
def bridgingGetConfiguredHandler( macAddr=None, vlanId=None, 
                                  intfsOrDrop=None, vteps=None ):
   
   # Get the configured host object for static vxlan entries  
   configuredHosts = vxlanConfigDir.fdbConfig.configuredHost

   vxlanMatchedHosts = []
   bridgingRemoteHosts = []
   if macAddr and vlanId:
      macVlanPair = Tac.Value( "Vxlan::MacVlanPair", macAddr, vlanId.id ) 
      macEntry = configuredHosts.get( macVlanPair )
      if macEntry:
         if filterVxlanConfiguredMacAddr( macEntry, vlanId,
                                          macAddr, vteps ):
            vxlanMatchedHosts = [ macEntry ]
   else:
      vxlanMatchedHosts = [ macEntry for macEntry in 
                            configuredHosts.itervalues()
                            if filterVxlanConfiguredMacAddr( macEntry, 
                                                             vlanId, 
                                                             macAddr, 
                                                             vteps ) ]
   # Create new mac address dictionary, as expected by Ebra
   # for configured mac entries
   for host in vxlanMatchedHosts:
      bridgingRemoteHosts.append( Tac.Value(
         "Bridging::ConfiguredHost", address=host.macVlanPair.macAddr, 
         intf=host.vtiIntfId, entryType='configuredStaticMac' ) )
 
   return bridgingRemoteHosts

# Register the static (configured) mac config hooks with the Ebra 
# provided hooks. 
BridgingCli.bridgingCheckStaticMacHook.addExtension( bridgingOkToAddHandler )
BridgingCli.bridgingAddStaticMacHook.addExtension( bridgingSetStaticHandler )
BridgingCli.bridgingDelStaticMacHook.addExtension( bridgingDelStaticHandler )
BridgingCli.bridgingExtraStaticMacHook.addExtension( bridgingGetConfiguredHandler )

vtepGenAddrNode = CliCommand.Node(
      IpGenAddrMatcher.IpGenAddrMatcher( 'IPv4 or IPv6 address of remote VTEP' ),
      guard=vxlanV6RemoteVtepSupportedGuard )

#--------------------------------------------------------------------------------
# [ no | default ] ( ( mac address-table ) | mac-address-table ) static MAC_ADDR 
#                                              vlan VLAN_ID interface INTF vtep VTEP
#--------------------------------------------------------------------------------
class SetStaticVxlanHostCmd( CliCommand.CliCommandClass ):
   syntax = 'MAC_ADDR_TABLE static MAC_ADDR vlan VLAN_ID interface INTF vtep VTEP'
   noOrDefaultSyntax = syntax.replace( 'VTEP', '...' )
   data = {
      'MAC_ADDR_TABLE': BridgingCli.MacAddrTableExprForConfig,
      'static': BridgingCli.staticKwMatcher,
      'MAC_ADDR': MacAddr.macAddrMatcher,
      'vlan': BridgingCli.matcherVlan,
      'VLAN_ID': VlanCli.vlanIdMatcher,
      'interface': BridgingCli.matcherInterface,
      'INTF': IntfRange.IntfRangeMatcher( explicitIntfTypes=vxlanIntfType ),
      'vtep': vtepMatcherForConfig,
      'VTEP': vtepGenAddrNode,
   }
   handler = setStaticVxlanHost
   noOrDefaultHandler = lambda mode, args: noSetStaticVxlanHost( mode,
         args[ 'MAC_ADDR' ], args[ 'VLAN_ID' ], args[ 'INTF' ] )

BasicCli.GlobalConfigMode.addCommandClass( SetStaticVxlanHostCmd )

IntfCli.IntfConfigMode.addModelet( VxlanIntfModelet )

#-------------------------------------------------------------------------------
# The "[no|default] vxlan ttl <ttl>" command, in "config-vxlan-if"
#
# sets the TTL for multicast forwarded Vxlan traffic
#-------------------------------------------------------------------------------
class TtlCmd( CliCommand.CliCommandClass ):
   syntax = 'vxlan ttl TTL'
   data = {
            'vxlan': vxlanNode,
            'ttl': 'Multicast TTL',
            'TTL': CliMatcher.IntegerMatcher( 0, 255, helpdesc='Multicast TTL' )
          }

   @staticmethod
   def handler( mode, args ):
      vtiConfig = getVtiConfig( mode.intf.name )
      vtiConfig.ttl = args[ 'TTL' ]

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      vtiConfig = getVtiConfig( mode.intf.name )
      vtiConfig.ttl = vtiConfig.defaultTtl

#BUG48350
#VxlanIntfModelet.addCommandClass( TtlCmd )

#-------------------------------------------------------------------------------
# The "[no|default] vxlan udp-port <port>" command, in "config-vxlan-if"
#
# sets the UDP port for Vxlan encapsulated traffic
#-------------------------------------------------------------------------------
def setUdpPort( mode, args ):
   udpPort = args[ 'UDP_PORT' ]
   if 0 <= udpPort <= 1023 :
      mode.addError( 'invalid udp-port ')
      return
   vtiConfig = getVtiConfig( mode.intf.name )
   vtiConfig.udpPort = udpPort

def noSetUdpPort( mode, args ):
   vtiConfig = getVtiConfig( mode.intf.name )
   vtiConfig.udpPort = vtiConfig.vxlanWellKnownPort

class VxlanUdpPortUdpportCmd( CliCommand.CliCommandClass ):
   syntax = 'vxlan udp-port UDP_PORT'
   noOrDefaultSyntax = 'vxlan udp-port [ UDP_PORT ]'
   data = {
      'vxlan': vxlanNode,
      'udp-port': matcherUdpPort,
      'UDP_PORT': CliMatcher.IntegerMatcher( 1024, 65535, helpdesc='UDP port' ),
   }
   handler = setUdpPort
   noOrDefaultHandler = noSetUdpPort

VxlanIntfModelet.addCommandClass( VxlanUdpPortUdpportCmd )

#-------------------------------------------------------------------------------
# The "[no|default] vxlan udp-port source offset <offset> range <range>" command,
# in "config-vxlan-if"
#
# sets the source UDP port range for Vxlan encapsulated traffic encapsulated by
# the switch
#-------------------------------------------------------------------------------
def setUdpSourcePortRange( mode, args ):
   offset = args[ 'OFFSET' ]
   portLength = args[ 'PORT_LENGTH' ]

   if portLength + offset > 0x10000:
      mode.addError( 'offset + length must not be greater than 0x10000')
      return
   if bridgingHwCapabilities.vxlanSrcPortSupported == \
         'offsetAllValuesRangePowerOfTwo':
      # verify that portLength is a power of 2, 0 is allowed
      if bin( portLength ).count( '1' ) > 1:
         mode.addError( 'length must be a power of 2, e.g. 0x1000' )
         return
   vtiConfig = getVtiConfig( mode.intf.name )
   vtiConfig.srcPortRange = \
         Tac.Value( "Vxlan::VxlanSrcPortRange", offset, portLength )

def noSetUdpSourcePortRange( mode, args ):
   vtiConfig = getVtiConfig( mode.intf.name )
   vtiConfig.srcPortRange = Tac.Value( "Vxlan::VxlanSrcPortRange" )

class UdpSourcePortRangeCmd( CliCommand.CliCommandClass ):
   syntax = 'vxlan udp-port source offset OFFSET length PORT_LENGTH'
   noOrDefaultSyntax = 'vxlan udp-port source ...'
   data = {
      'vxlan': vxlanNode,
      'udp-port': matcherUdpPort,
      'source': CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher( 'source',
            helpdesc='Set source port range for VXLAN encapsulated packets' ),
         guard=vxlanSrcPortSupported ),
      'offset': 'Start of source port range',
      'OFFSET': CliMatcher.IntegerMatcher( 0, 65535, helpdesc='UDP port offset' ),
      'length': 'Length of source port range',
      'PORT_LENGTH': CliMatcher.IntegerMatcher( 0, 65535,
         helpdesc='UDP port length' ),
   }
   handler = setUdpSourcePortRange
   noOrDefaultHandler = noSetUdpSourcePortRange

VxlanIntfModelet.addCommandClass( UdpSourcePortRangeCmd )

#-------------------------------------------------------------------------------
# "[no|default] vxlan vtep ipv4 address-mask <mask>" command, in "config-vxlan-if"
#-------------------------------------------------------------------------------
def setVtepIpv4AddressMask( mode, args ):
   addressMask = args[ 'MASK' ]
   try:
      Arnet.Mask( addressMask )
   except ValueError:
      mode.addError( "invalid address mask: " + addressMask )
      return
   vtiConfig = getVtiConfig( mode.intf.name )
   vtiConfig.vtepAddrMask = Arnet.IpAddress( addressMask )

def noSetVtepIpv4AddressMask( mode, args ):
   vtiConfig = getVtiConfig( mode.intf.name )
   vtiConfig.vtepAddrMask = Tac.newInstance( "Arnet::IpAddr", 0xFFFFFFFF )

class VxlanVtepIpv4AddressMaskMaskCmd( CliCommand.CliCommandClass ):
   syntax = 'vxlan vtep ipv4 address-mask MASK'
   noOrDefaultSyntax = 'vxlan vtep ipv4 address-mask ...'
   data = {
      'vxlan': vxlanNode,
      'vtep': CliCommand.Node( matcher=vtepMatcherForConfig,
         guard=vxlanVtepMaskCliSupported ),
      'ipv4': 'Configure VXLAN vtep IPv4',
      'address-mask': 'Configure vtep address-mask',
      'MASK': IpAddrMatcher.IpAddrMatcher( helpdesc='address mask' ),
   }
   handler = setVtepIpv4AddressMask
   noOrDefaultHandler = noSetVtepIpv4AddressMask

VxlanIntfModelet.addCommandClass( VxlanVtepIpv4AddressMaskMaskCmd )

def setVtepsForSourcePruning( vtiConfig, vteps ):
   for v in vtiConfig.vtepSetForSourcePruning:
      if v not in vteps:
         vtiConfig.vtepSetForSourcePruning.remove( v )
   for v in vteps:
      if v not in vtiConfig.vtepSetForSourcePruning:
         vtiConfig.vtepSetForSourcePruning.add( v )
         
#-------------------------------------------------------------------------------
# "[no|default] vxlan bridging vtep-to-vtep [ source-vtep tx disabled
# [ { VTEPS } ] ]" command, in "config-vxlan-if"
# 
# "[no|default] vxlan bridging vtep-to-vtep" -
# Enables vxlan vtep-to-vtep bridging. 
# 
# "[no|default] vxlan bridging vtep-to-vtep source-vtep tx disabled" -
# Enables source pruning for all vteps
# 
# "[no|default] vxlan bridging vtep-to-vtep source-vtep tx disabled { VTEPS }" -
# Enables source pruning for the list of vteps
#-------------------------------------------------------------------------------
class VxlanVtepToVtepBridging( CliCommand.CliCommandClass ):
   syntax = 'vxlan bridging vtep-to-vtep [ source-vtep tx disabled [ { VTEPS } ] ]'
   noOrDefaultSyntax = 'vxlan bridging vtep-to-vtep [ source-vtep tx disabled ] ...'
   data = {
         'vxlan': vxlanNode,
         'bridging': CliCommand.Node( matcher=CliMatcher.KeywordMatcher( 
                                          'bridging',
                                          helpdesc='VXLAN bridging' ),
                                      guard=vxlanVtepToVtepBridgingSupportedGuard ),
         'vtep-to-vtep': 'Enable VTEP to VTEP bridging',
         'source-vtep': 'Configure source VTEP properties',
         'tx': 'Allow bridged packets to go back to source VTEPs',
         'disabled': 'Disable bridged packets to go back to source VTEPs',
         'VTEPS': IpAddrMatcher.IpAddrMatcher( 'IP address of remote VTEP' )
   }

   @staticmethod
   def handler( mode, args ):
      vtiConfig = getVtiConfig( mode.intf.name )
      # Enable Vxlan bridging
      vtiConfig.vtepToVtepBridging = True
      if 'source-vtep' not in args:
         return

      # Source pruning for all vteps 
      vteps = args.get( 'VTEPS' )
      if not vteps:
         vtiConfig.vtepSourcePruningAll = True
         setVtepsForSourcePruning( vtiConfig, [] )
         return

      # Source pruning for vteps in the VTEPS list
      if not _verifyVtepAddrs( mode, vteps ):
         return
      vtiConfig.vtepSourcePruningAll = False
      setVtepsForSourcePruning( vtiConfig, vteps )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      vtiConfig = getVtiConfig( mode.intf.name )
      if 'source-vtep' not in args:
         vtiConfig.vtepToVtepBridging = False
      setVtepsForSourcePruning( vtiConfig, [] )
      vtiConfig.vtepSourcePruningAll = False

VxlanIntfModelet.addCommandClass( VxlanVtepToVtepBridging )


#-------------------------------------------------------------------------------
# "[no|default] vxlan encapsulation ipv6", in "config-if-Vx"
# 
# By default, VXLAN encapsulation (underlay) is assumed to be of type ipv4. This 
# CLI will set the encapsulation to be ipv6 for all VNIs. The 'no' form of the CLI 
# will set the encapsulation back to its default value (which is ipv4). 
#-------------------------------------------------------------------------------
vxlanEncapNode = CliCommand.Node(
      matcher=CliMatcher.KeywordMatcher( 'encapsulation',
         helpdesc='VXLAN encapsulation type' ),
      guard=vxlanEncapIpv6SupportedGuard )
encapOptions = { 'ipv4' : "Use Ipv4 for VXLAN Encapsulation", \
                 'ipv6' : "Use Ipv6 for VXLAN Encapsulation" }

class VxlanEncapsulation( CliCommand.CliCommandClass ):
   syntax = 'vxlan encapsulation ENCAP'
   noOrDefaultSyntax = 'vxlan encapsulation ...'
   data = {
         'vxlan': vxlanNode,
         'encapsulation': vxlanEncapNode,
         'ENCAP' : CliMatcher.EnumMatcher( encapOptions ),
   }

   @staticmethod
   def handler( mode, args ):
      vtiConfig = getVtiConfig( mode.intf.name)
      vtiConfig.vxlanEncapConfigured = True
      if args[ 'ENCAP' ] == 'ipv6':
         vtiConfig.vxlanEncap = vxlanEncapType.vxlanEncapIp6
      else:
         vtiConfig.vxlanEncap = vxlanEncapType.vxlanEncapIp4

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      vtiConfig = getVtiConfig( mode.intf.name)
      vtiConfig.vxlanEncapConfigured = False
      vtiConfig.vxlanEncap = bridgingHwCapabilities.vxlanDefaultEncap

class VxlanEncapsulationDual( CliCommand.CliCommandClass ):
   syntax = '''vxlan encapsulation 
               ( ( ipv4 [ DUAL_V6 ] ) | ( ipv6 [ DUAL_V4 ] ) ) '''
   noOrDefaultSyntax = 'vxlan encapsulation ...'
   data = {
         'vxlan': vxlanNode,
         'encapsulation': vxlanEncapNode,
         'ipv4' : 'Use IPv4 for VXLAN Encapsulation',
         'ipv6' : 'Use IPv6 for VXLAN Encapsulation',
         'DUAL_V6' : CliMatcher.KeywordMatcher( 'ipv6',
            helpdesc="Use IPv4 and IPv6 for VXLAN Encapsulation" ),
         'DUAL_V4' : CliMatcher.KeywordMatcher( 'ipv4',
            helpdesc="Use IPv4 and IPv6 for VXLAN Encapsulation" ),
   }

   @staticmethod
   def handler( mode, args ):
      vtiConfig = getVtiConfig( mode.intf.name)
      vtiConfig.vxlanEncapConfigured = True
      if 'DUAL_V6' in args or 'DUAL_V4' in args:
         vtiConfig.vxlanEncap = vxlanEncapType.vxlanEncapDual
      elif 'ipv6' in args:
         vtiConfig.vxlanEncap = vxlanEncapType.vxlanEncapIp6
      else:
         vtiConfig.vxlanEncap = vxlanEncapType.vxlanEncapIp4

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      vtiConfig = getVtiConfig( mode.intf.name)
      vtiConfig.vxlanEncapConfigured = False
      vtiConfig.vxlanEncap = bridgingHwCapabilities.vxlanDefaultEncap


if Toggles.VxlanToggleLib.toggleVxlanEncapsulationTypeEnabled():
   if Toggles.VxlanToggleLib.toggleVxlanEncapsulationTypeDualEnabled():
      VxlanIntfModelet.addCommandClass( VxlanEncapsulationDual )
   else:
      VxlanIntfModelet.addCommandClass( VxlanEncapsulation )

#-------------------------------------------------------------------------------
# The "[no|default] vxlan header vni 32-bit" command, in "config-vxlan-if"
#
# Triggers the switch to treat the reserved 8-bits in the vxlan header as an
# extension to form a 32-bit VNI
#-------------------------------------------------------------------------------
def set32BitVni( mode, args ):
   vtiConfig = getVtiConfig( mode.intf.name )
   vtiConfig.use32BitVni = True

def noSet32BitVni( mode, args ):
   vtiConfig = getVtiConfig( mode.intf.name )
   vtiConfig.use32BitVni = False

class VxlanHeaderVni32BitCmd( CliCommand.CliCommandClass ):
   syntax = 'vxlan header vni 32-bit'
   noOrDefaultSyntax = syntax
   data = {
      'vxlan': vxlanNode,
      'header': CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher( 'header',
            helpdesc='VXLAN header extensions' ),
         guard=vxlan32BitVniSupportedGuard ),
      'vni': vniMatcherForConfig,
      '32-bit': 'Allow 32-bit VXLAN Network Identifier Configuration',
   }
   handler = set32BitVni
   noOrDefaultHandler = noSet32BitVni

VxlanIntfModelet.addCommandClass( VxlanHeaderVni32BitCmd )

#-------------------------------------------------------------------------------
# [ no | default ] vxlan controller-client
# in "config-vxlan-if"
#
# sets the switch in vxlan controller client mode
#-------------------------------------------------------------------------------
def setControllerMode( mode, args ):
   vtiConfig = getVtiConfig( mode.intf.name )
   vtiConfig.vccDataPathLearning = False
   vtiConfig.controllerClientMode = True
   adjustVxlanControllerServiceEnable()

def noSetControllerMode( mode, args ):
   vtiConfig = getVtiConfig( mode.intf.name )
   vtiConfig.vccDataPathLearning = False
   vtiConfig.controllerClientMode = False
   adjustVxlanControllerServiceEnable()

class VxlanControllerClientCmd( CliCommand.CliCommandClass ):
   syntax = 'vxlan controller-client'
   noOrDefaultSyntax = syntax
   data = {
      'vxlan': vxlanNode,
      'controller-client': controllerClientNode,
   }
   handler = setControllerMode
   noOrDefaultHandler = noSetControllerMode

VxlanIntfModelet.addCommandClass( VxlanControllerClientCmd )

#-------------------------------------------------------------------------------
# The "[no|default] vxlan qos ecn propagation" command, in "enable" mode.
#-------------------------------------------------------------------------------
def setEcnPropagation( mode, args ):
   vxlanConfigDir.ecnPropagation = True

def clearEcnPropagation( mode, args ):
   vxlanConfigDir.ecnPropagation = False

class VxlanQosEcnPropagationCmd( CliCommand.CliCommandClass ):
   syntax = 'vxlan qos ecn propagation'
   noOrDefaultSyntax = syntax
   data = {
      'vxlan': vxlanNode,
      'qos': 'Qos settings',
      'ecn': CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher( 'ecn', helpdesc='Ecn settings' ),
         guard=vxlanEcnPropagationSupported ),
      'propagation': 'Set ecn propagation',
   }
   handler = setEcnPropagation
   noOrDefaultHandler = clearEcnPropagation

VxlanIntfModelet.addCommandClass( VxlanQosEcnPropagationCmd )

#-------------------------------------------------------------------------------
# The "show vxlan qos" command.
#-------------------------------------------------------------------------------
def showVxlanQos( mode, args ):
   qosModel = VxlanModel.VxlanQosModel()
   qosModel.ecnPropagation = vxlanConfigDir.ecnPropagation
   return qosModel

#-------------------------------------------------------------------------------
# The "[no|default] vxlan virtual-router encapsulation mac-addr [ <MAC> |
#                                                                 mlag-system-id ]
# command, in "config-vxlan-if"
#
# This command enables MLAG Shared Router MAC with one of 2 options:
#  (1) set MLAG shared router mac using the value of local mlag-system-id
#  (2) set MLAG shared router mac to the virtual mac provided by configure
#
# Note: This command only has effect when MLAG is configured and "active".
#-------------------------------------------------------------------------------
def setMlagSharedRouterMacAuto( mode ):
   vtiConfig = getVtiConfig( mode.intf.name )
   vtiConfig.mlagSharedRouterMacAddr = ethAddrZero
   vtiConfig.mlagSharedRouterMacConfig = 'autoGenerated'

def setMlagSharedRouterMacExplicit( mode, macAddr ):
   if Ethernet.isBroadcast( macAddr ) or \
      Ethernet.isIPMulticast( macAddr, allowIanaReserved=True ) or \
      macAddr == '00:00:00:00:00:00':
      mode.addError( 'MLAG Shared router MAC cannot be a multicast or broadcast' \
                     'MAC address' )
      return
   vtiConfig = getVtiConfig( mode.intf.name )
   vtiConfig.mlagSharedRouterMacAddr = macAddr
   vtiConfig.mlagSharedRouterMacConfig = 'explicitConfig'

def noSetMlagSharedRouterMacMode( mode, args ):
   vtiConfig = getVtiConfig( mode.intf.name )
   vtiConfig.mlagSharedRouterMacAddr = ethAddrZero
   vtiConfig.mlagSharedRouterMacConfig = 'disabled'

class MlagSharedRouterMacCmd( CliCommand.CliCommandClass ):
   syntax = '''vxlan virtual-router encapsulation mac-address
               ( MAC_ADDR | mlag-system-id )'''
   noOrDefaultSyntax = 'vxlan virtual-router encapsulation mac-address ...'
   data = {
      'vxlan': vxlanNode,
      'virtual-router': CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher( 'virtual-router',
            helpdesc='Configure a virtual router' ),
         guard=vxlanMlagSharedRouterMacSupported ),
      'encapsulation': 'VXLAN encapsulation',
      'mac-address': 'Virtual router MAC address',
      'MAC_ADDR': MacAddr.MacAddrMatcher( helpdesc='Ethernet address' ),
      'mlag-system-id': 'Configure MLAG shared router MAC with System ID',
   }

   @staticmethod
   def handler( mode, args ):
      if 'MAC_ADDR' in args:
         return setMlagSharedRouterMacExplicit( mode, args[ 'MAC_ADDR' ] )
      else:
         return setMlagSharedRouterMacAuto( mode )

   noOrDefaultHandler = noSetMlagSharedRouterMacMode

VxlanIntfModelet.addCommandClass( MlagSharedRouterMacCmd )

#-------------------------------------------------------------------------------
# The "[no|default] vxlan controller-client import vlan" command in 
# "config-vxlan-if" mode.
#
# The full syntax of this command is:
#
#   vxlan controller-client import vlan none
#   vxlan controller-client import vlan <import_enabled_vlans>
#   vxlan controller-client import vlan add <import_enabled_vlans>
#   vxlan controller-client import vlan remove <import_disabled_vlans>
#   (no|default) vxlan controller-client import vlan ...
#-------------------------------------------------------------------------------
importEnabledVlanSetMatcher = MultiRangeRule.MultiRangeMatcher(
      lambda: ( 1, 4094 ),
      False,
      'VCS import-enabled VLAN IDs',
      value=VlanCli.vlanIdListFunc )

importDisabledVlanSetMatcher = MultiRangeRule.MultiRangeMatcher(
      lambda: ( 1, 4094 ),
      False,
      'VCS import-disabled VLAN IDs',
      value=VlanCli.vlanIdListFunc )

def setVccImportVlans( mode, importVlanOp ):
   vtiConfig = getVtiConfig( mode.intf.name )
   if not vtiConfig:
      return

   importVlanSet = VlanCli.VlanSet( mode, vtiConfig.cliVccImportVlans ).ids
   ( vlanOp, vlanSet ) = importVlanOp
   
   if vlanOp == 'set': 
      importVlanSet = vlanSet
   elif vlanOp == 'add':
      importVlanSet |= vlanSet
   else:
      assert vlanOp == 'remove'
      importVlanSet -= vlanSet
   
   s = MultiRangeRule.multiRangeToCanonicalString( importVlanSet )
   vtiConfig.cliVccImportVlans = s 

def allVlanIds():
   return frozenset( xrange( 1, 4095 ) )

def noVlanIds():
   return frozenset()

class VxlanControllerClientImportVlan( CliCommand.CliCommandClass ):
   syntax = '''vxlan controller-client import vlan 
               ( ( none )
               | ( remove IMPORT_DISABLED_VLANS )
               | ( [ add ] IMPORT_ENABLED_VLANS ) )'''
   noOrDefaultSyntax = '''vxlan controller-client import vlan ...'''

   data = { 
      'vxlan' : vxlanNode,
      'controller-client' : controllerClientNode, 
      'import' : 'Set import VLAN characteristics when in controller client mode',
      'vlan' : 'Set import VLAN when in controller client mode',
      'none' : 'No VLANs',
      'add' : 'Add VLANs to the current list',
      'remove' : 'Remove VLANs from the current list',
      'IMPORT_DISABLED_VLANS' : importDisabledVlanSetMatcher,
      'IMPORT_ENABLED_VLANS' : importEnabledVlanSetMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      if args.get( 'none' ):
         setVccImportVlans( mode, ( 'set', noVlanIds() ) )
      elif args.get( 'add' ):
         vlanSet = args.get( 'IMPORT_ENABLED_VLANS' ).ids
         setVccImportVlans( mode, ( 'add', vlanSet ) )
      elif args.get( 'remove' ):
         vlanSet = args.get( 'IMPORT_DISABLED_VLANS' ).ids
         setVccImportVlans( mode, ( 'remove', vlanSet ) )
      else:
         vlanSet = args.get( 'IMPORT_ENABLED_VLANS' ).ids
         assert vlanSet
         setVccImportVlans( mode, ( 'set', vlanSet ) )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      setVccImportVlans( mode, ( 'set', allVlanIds() ) )

VxlanIntfModelet.addCommandClass( VxlanControllerClientImportVlan )

#-------------------------------------------------------------------------------
# [no|default] arp reply relay
#-------------------------------------------------------------------------------
def enableArpReplyRelay( mode, args ):
   vtiConfig = getVtiConfig( mode.intf.name )
   vtiConfig.arpReplyRelay = True

def disableArpReplyRelay( mode, args ):
   vtiConfig = getVtiConfig( mode.intf.name )
   vtiConfig.arpReplyRelay = False

class ArpReplyRelayCmd( CliCommand.CliCommandClass ):
   syntax = 'arp reply relay'
   noOrDefaultSyntax = syntax
   data = {
      'arp': 'Configure ARP feature',
      'reply': 'Configure ARP Reply',
      'relay': 'ARP Relay',
   }
   handler = enableArpReplyRelay
   noOrDefaultHandler = disableArpReplyRelay

VxlanIntfModelet.addCommandClass( ArpReplyRelayCmd )

#--------------------------------------------------------------------------------
# [ no | default ] arp source ip address local
#--------------------------------------------------------------------------------
def enableArpLocalAddress( mode, args ):
   vtiConfig = getVtiConfig( mode.intf.name )
   vtiConfig.arpLocalAddress = True

def disableArpLocalAddress( mode, args ):
   vtiConfig = getVtiConfig( mode.intf.name )
   vtiConfig.arpLocalAddress = False

class ArpSourceIpAddressLocalCmd( CliCommand.CliCommandClass ):
   syntax = 'arp source ip address local'
   noOrDefaultSyntax = syntax
   data = {
      'arp': 'Configure ARP feature',
      'source': 'Modify sender address in VXLAN encapsulated ARP requests',
      'ip': 'Modify sender IP in VXLAN encapsulated ARP requests',
      'address': 'Modify sender IP address in VXLAN encapsulated ARP requests',
      'local': ( 'Enable rewrite of virtual IP in VXLAN encapsulated ARP requests '
                 'with local IP before local bridging' ),
   }
   handler = enableArpLocalAddress
   noOrDefaultHandler = disableArpLocalAddress

VxlanIntfModelet.addCommandClass( ArpSourceIpAddressLocalCmd )

#-------------------------------------------------------------------------------
# [no|default] ip route vrf <vrfId> <prefix> vtep <vtepAddr> vni <vni>
# router-mac-address <router-mac>
#-------------------------------------------------------------------------------

def isValidVniWithError( mode, vni, vti ):
   if not vti:
      vti = 'Vxlan1'
   vtiConfig = getVtiConfig( vti, create=False )
   if not vtiConfig:
      errorStr = "Interface %s doesn't exist." % vti
      mode.addError( errorStr )
      return False
   minVni = 1
   maxVni = 0xFFFFFF if not vtiConfig.use32BitVni else 0xFFFFFFFE
   if vni < minVni or vni > maxVni:
      errorStr = "Invalid VXLAN Network Identifier. "
      if vxlanCtrlCfgBox[ 0 ].vniInDottedNotation:
         errorStr += "The VNI should be of the form A.B.C where A, B and C are " \
                     "integers in the range 0-255. "
      else:
         errorStr += "Allowed range is: %d-%d. " % ( minVni, maxVni )
      mode.addError( errorStr )
      return False
   return True

def getVxlanIntf( localIntf ):
   vxlanIntfStr = localIntf.strWithLongTag() if localIntf else ''
   if vxlanIntfStr:
      intf = vtiNameToId.get( vxlanIntfStr )
   else:
      intf = Tac.Value( "Arnet::IntfId", "" )
   return intf

def manageIpv4StaticEvpnRoute( mode, args ):
   prefix = args[ 'PREFIX' ]
   evpnNextHop = args[ 'EVPN_NEXTHOP' ]
   routeOptions = args[ 'OPTION' ]
   vrfName = args.get( 'VRF', DEFAULT_VRF )
   assert isinstance( routeOptions, set )

   roDict = dict( routeOptions )
   preference = roDict.get( 'preference',
                            RoutingConsts.defaultStaticRoutePreference )
   tag = roDict.get( 'tag', RoutingConsts.defaultStaticRouteTag )
   nextHopName = roDict.get( 'nextHopName', "" )

   vxlanIntf = getVxlanIntf( evpnNextHop.get( 'vxlanSrcIntf', "" ) )
   routerMac = evpnNextHop[ 'routerMac' ]
   vni = int( evpnNextHop[ 'vni' ] )
   vtepAddr = evpnNextHop[ 'vtepAddr' ]
   dynamic = evpnNextHop[ 'dynamic' ]
   ribBypass = evpnNextHop[ 'ribBypass' ]
   if hasattr( vtepAddr, 'stringValue' ):
      # this can be a v4 or v6 address, v4 is a string, v6 is an
      # Arnet::Ip6Addr, we normalize to a string
      vtepAddr = vtepAddr.stringValue

   error = not vxlanCliHelper().manageStaticEvpnRoute( vrfName,
         prefix, vni, preference, tag, nextHopName, vtepAddr,
         routerMac, vxlanIntf, dynamic, ribBypass, mode.session_.startupConfig() )

   if error:
      mode.addError( vxlanCliHelper().errorMsg( 
                     vxlanCtrlCfgBox[ 0 ].vniInDottedNotation ) )

vtepAddrNode = CliCommand.Node(
      IpGenAddrMatcher.IpGenAddrMatcher( 'IPv4 or IPv6 address of remote VTEP' ),
      guard=vxlanIpv6UnderlaySupportedGuard )
routerMacMatcherForConfig = CliMatcher.KeywordMatcher( 'router-mac-address',
                                        helpdesc='Remote router Mac address' )

class IpRouteEvpnCmd( CliCommand.CliCommandClass ):
   syntax = '''ip route [ VRF ] PREFIX vtep VTEP vni VNI
               router-mac-address MAC_ADDR
               [ { ( PREFERENCE )
                 | ( tag TAGNUM )
                 | ( name NEXTHOP_NAME )
                 | ( dynamic )
                 | ( rib-bypass ) } ]'''
   noOrDefaultSyntax = '''ip route [ VRF ] PREFIX vtep
                          [ VTEP vni VNI router-mac-address MAC_ADDR ]
                          [ PREFERENCE ] ...'''
   data = { 'ip': CliToken.Ip.ipMatcherForConfig,
            'route': IraIpRouteCliLib.routeMatcherForConfig,
            'VRF': IraIpCli.vrfExprFactoryForConfig,
            'PREFIX': IraIpRouteCliLib.prefixMatcher, 
            'vtep': vtepMatcherForConfig,
            'VTEP': vtepAddrNode,
            'vni': vniMatcherForConfig,
            'VNI': vniMatcher,
            'router-mac-address': routerMacMatcherForConfig,
            'MAC_ADDR': MacAddr.MacAddrMatcher(),
            'PREFERENCE': CliCommand.Node( IraIpRouteCliLib.preferenceRangeMatcher,
                                           maxMatches=1 ),
            'tag': CliCommand.Node( IraIpRouteCliLib.tagMatcherForConfig,
                                    maxMatches=1 ),
            'TAGNUM': CliCommand.Node( IraIpRouteCliLib.tagNumberMatcher,
                                        maxMatches=1 ),
            'name': CliCommand.Node( IraCommonCli.nameMatcherForConfig,
                                     maxMatches=1 ),
            'NEXTHOP_NAME': CliCommand.Node( IraCommonCli.nexthopNameMatcher,
                                             maxMatches=1 ),
            'dynamic' : CliCommand.Node( dynamicMatcherForConfig, maxMatches=1 ),
            'rib-bypass' : CliCommand.Node( ribBypassMatcherForConfig, maxMatches=1,
               guard=ribBypassSupportedGuard )
            }
   rootNodesCacheEnabled = True

   @staticmethod
   def adapter( mode, args, argsList ):
      args[ 'EVPN_NEXTHOP' ] = {}
      if 'VTEP' in args:
         args[ 'EVPN_NEXTHOP' ][ 'vtepAddr' ] = args[ 'VTEP' ]
      if 'VNI' in args:
         args[ 'EVPN_NEXTHOP' ][ 'vni' ] = args[ 'VNI' ]
      if 'MAC_ADDR' in args:
         args[ 'EVPN_NEXTHOP' ][ 'routerMac' ] = args[ 'MAC_ADDR' ]
      args[ 'EVPN_NEXTHOP' ][ 'dynamic' ] = 'dynamic' in args
      args[ 'EVPN_NEXTHOP' ][ 'ribBypass' ] = 'rib-bypass' in args

      args[ 'OPTION' ] = set()
      if 'PREFERENCE' in args:
         args[ 'OPTION' ].add( ( 'preference', args.pop( 'PREFERENCE' ) ) )
      if 'tag' in args:
         args[ 'OPTION' ].add( ( 'tag', args.pop( 'TAGNUM' ) ) )
      if 'name' in args:
         args[ 'OPTION' ].add( ( 'nextHopName', args.pop( 'NEXTHOP_NAME' ) ) )

   @staticmethod
   def handler( mode, args ):
      manageIpv4StaticEvpnRoute( mode, args )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      preference = args.get( 'PREFERENCE', None )
      vrfName = args.get( 'VRF', DEFAULT_VRF )
      evpnNextHop = args[ 'EVPN_NEXTHOP' ]
      nexthop = {}
      if 'vtepAddr' in evpnNextHop and 'vni' in evpnNextHop and \
            'routerMac' in evpnNextHop:
         nexthop[ 'evpn' ] = {}
         nexthop[ 'evpn' ][ 'vtepAddr' ] = evpnNextHop[ 'vtepAddr' ]
         nexthop[ 'evpn' ][ 'vni' ] = evpnNextHop[ 'vni' ]
         nexthop[ 'evpn' ][ 'routerMac' ] = evpnNextHop[ 'routerMac' ]
      IraIpCli.noIpRoute( mode, args[ 'PREFIX' ], nexthop, preference, vrfName,
                          None ) 

class IpRouteEvpnCmdMultiVtep( CliCommand.CliCommandClass ):
   syntax = '''ip route [ VRF ] PREFIX vtep VTEP vni VNI
               router-mac-address MAC_ADDR
               [ { PREFERENCE | ( tag TAGNUM ) | ( name NEXTHOP_NAME ) |
                 ( local-interface INTF ) | dynamic | rib-bypass } ]'''
   noOrDefaultSyntax = '''ip route [ VRF ] PREFIX vtep [ VTEP vni VNI
                          router-mac-address MAC_ADDR ] [ PREFERENCE ] ...'''
   data = { 'ip': CliToken.Ip.ipMatcherForConfig,
            'route': IraIpRouteCliLib.routeMatcherForConfig,
            'VRF': IraIpCli.vrfExprFactoryForConfig,
            'PREFIX': IraIpRouteCliLib.prefixMatcher, 
            'vtep': vtepMatcherForConfig,
            'VTEP': vtepAddrNode,
            'vni': vniMatcherForConfig,
            'VNI': vniMatcher,
            'router-mac-address': routerMacMatcherForConfig,
            'MAC_ADDR': MacAddr.MacAddrMatcher(),
            'local-interface': CliCommand.Node( BridgingCli.matcherInterface,
                                          maxMatches=1 ),
            'INTF': CliCommand.Node( IntfRange.IntfRangeMatcher(
                                     explicitIntfTypes=vxlanIntfType ),
                                     maxMatches=1 ),
            'PREFERENCE': CliCommand.Node( IraIpRouteCliLib.preferenceRangeMatcher,
                                           maxMatches=1 ),
            'tag': CliCommand.Node( IraIpRouteCliLib.tagMatcherForConfig,
                                    maxMatches=1 ),
            'TAGNUM': CliCommand.Node( IraIpRouteCliLib.tagNumberMatcher,
                                        maxMatches=1 ),
            'name': CliCommand.Node( IraCommonCli.nameMatcherForConfig,
                                     maxMatches=1 ),
            'NEXTHOP_NAME': CliCommand.Node( IraCommonCli.nexthopNameMatcher,
                                             maxMatches=1 ),
            'dynamic' : CliCommand.Node( dynamicMatcherForConfig, maxMatches=1 ),
            'rib-bypass' : CliCommand.Node(
               ribBypassMatcherForConfig, maxMatches=1 ),
            }
   rootNodesCacheEnabled = True

   @staticmethod
   def adapter( mode, args, argsList ):
      args[ 'EVPN_NEXTHOP' ] = {}
      if 'VTEP' in args:
         args[ 'EVPN_NEXTHOP' ][ 'vtepAddr' ] = args[ 'VTEP' ]
      if 'VNI' in args:
         args[ 'EVPN_NEXTHOP' ][ 'vni' ] = args[ 'VNI' ]
      if 'MAC_ADDR' in args:
         args[ 'EVPN_NEXTHOP' ][ 'routerMac' ] = args[ 'MAC_ADDR' ]
      if 'local-interface' in args:
         args[ 'EVPN_NEXTHOP' ][ 'vxlanSrcIntf' ] = args[ 'INTF' ]
      args[ 'EVPN_NEXTHOP' ][ 'dynamic' ] = 'dynamic' in args
      args[ 'EVPN_NEXTHOP' ][ 'ribBypass' ] = 'rib-bypass' in args

      args[ 'OPTION' ] = set()
      if 'PREFERENCE' in args:
         args[ 'OPTION' ].add( ( 'preference', args.pop( 'PREFERENCE' ) ) )
      if 'tag' in args:
         args[ 'OPTION' ].add( ( 'tag', args.pop( 'TAGNUM' ) ) )
      if 'name' in args:
         args[ 'OPTION' ].add( ( 'nextHopName', args.pop( 'NEXTHOP_NAME' ) ) )

   @staticmethod
   def handler( mode, args ):
      manageIpv4StaticEvpnRoute( mode, args )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      preference = args.get( 'PREFERENCE', None )
      vrfName = args.get( 'VRF', DEFAULT_VRF )
      evpnNextHop = args[ 'EVPN_NEXTHOP' ]
      nexthop = {}
      if 'vtepAddr' in evpnNextHop and 'vni' in evpnNextHop and \
            'routerMac' in evpnNextHop:
         nexthop[ 'evpn' ] = {}
         nexthop[ 'evpn' ][ 'vtepAddr' ] = evpnNextHop[ 'vtepAddr' ]
         nexthop[ 'evpn' ][ 'vni' ] = evpnNextHop[ 'vni' ]
         nexthop[ 'evpn' ][ 'routerMac' ] = evpnNextHop[ 'routerMac' ]

      IraIpCli.noIpRoute( mode, args[ 'PREFIX' ], nexthop, preference, vrfName,
                          None ) 

if Toggles.VxlanToggleLib.toggleMultiVxlanTunnelInterfacesEnabled():
   BasicCli.GlobalConfigMode.addCommandClass( IpRouteEvpnCmdMultiVtep )
else:
   BasicCli.GlobalConfigMode.addCommandClass( IpRouteEvpnCmd )

#-------------------------------------------------------------------------------
# [no|default] ipv6 route vrf <vrfId> <prefix> vtep <vtepAddr> vni <vni>
# router-mac-address <router-mac>
#-------------------------------------------------------------------------------
def manageIpv6StaticEvpnRoute( mode, args ):
   prefix = args[ 'PREFIX' ]
   evpnNextHop = args[ 'EVPN_NEXTHOP' ]
   routeOptions = args[ 'OPTION' ]
   vrfName = args.get( 'VRF', DEFAULT_VRF )
   assert isinstance( routeOptions, set )
   vxlanIntf = getVxlanIntf( evpnNextHop.get( 'vxlanSrcIntf', "" ) )

   if not mode.session_.startupConfig(): 
      if not VrfCli.vrfExists( vrfName ):
         mode.addError( "%s Create first." % noIpv6RouteTableForVrfMsg % vrfName )
         return
      vni = int( evpnNextHop[ 'vni' ] )
      if not isValidVniWithError( mode, vni, vxlanIntf.stringValue ):
         return

   evpnNextHop[ 'vxlanSrcIntf' ] = vxlanIntf

   nexthop = { 'evpn': evpnNextHop }
   roDict = dict( routeOptions )
   manageIpv6Route( mode, prefix, nexthop, roDict, vrfName )

class Ipv6RouteEvpnCmd( CliCommand.CliCommandClass ):
   syntax = '''ipv6 route [ VRF ] PREFIX vtep VTEP vni VNI
               router-mac-address MAC_ADDR
               [ { ( PREFERENCE )
                 | ( tag TAGNUM )
                 | ( name NEXTHOP_NAME )
                 | ( dynamic )
                 | ( rib-bypass ) } ]'''
   noOrDefaultSyntax = '''ipv6 route [ VRF ] PREFIX vtep
                          [ VTEP vni VNI router-mac-address MAC_ADDR ]
                          [ PREFERENCE ] ...'''
   data = { 'ipv6': CliToken.Ipv6.ipv6MatcherForConfig,
            'route': routeMatcherForIpv6Config,
            'VRF': IraIp6Cli.vrfExprFactoryForConfig,
            'PREFIX': ipv6PrefixMatcher,
            'vtep': vtepMatcherForConfig,
            'VTEP': vtepAddrNode,
            'vni': vniMatcherForConfig,
            'VNI': vniMatcher,
            'router-mac-address': routerMacMatcherForConfig,
            'MAC_ADDR': MacAddr.MacAddrMatcher(),
            'PREFERENCE': CliCommand.Node( IraIpRouteCliLib.preferenceRangeMatcher,
                                           maxMatches=1 ),
            'tag': CliCommand.Node( IraIpRouteCliLib.tagMatcherForConfig,
                                    maxMatches=1 ),
            'TAGNUM': CliCommand.Node( IraIpRouteCliLib.tagNumberMatcher,
                                       maxMatches=1 ),
            'name': CliCommand.Node( IraCommonCli.nameMatcherForConfig,
                                     maxMatches=1 ),
            'NEXTHOP_NAME': CliCommand.Node( IraCommonCli.nexthopNameMatcher,
                                             maxMatches=1 ),
            'dynamic' : CliCommand.Node( dynamicMatcherForConfig, maxMatches=1 ),
            'rib-bypass' : CliCommand.Node( ribBypassMatcherForConfig, maxMatches=1,
               guard=ribBypassSupportedGuard )
            }

   @staticmethod
   def adapter( mode, args, argsList ):
      args[ 'EVPN_NEXTHOP' ] = {}
      if 'VTEP' in args:
         args[ 'EVPN_NEXTHOP' ][ 'vtepAddr' ] = args[ 'VTEP' ].stringValue
      if 'VNI' in args:
         args[ 'EVPN_NEXTHOP' ][ 'vni' ] = args[ 'VNI' ]
      if 'MAC_ADDR' in args:
         args[ 'EVPN_NEXTHOP' ][ 'routerMac' ] = args[ 'MAC_ADDR' ]
      args[ 'EVPN_NEXTHOP' ][ 'dynamic' ] = 'dynamic' in args
      args[ 'EVPN_NEXTHOP' ][ 'ribBypass' ] = 'rib-bypass' in args

      args[ 'OPTION' ] = set()
      if 'PREFERENCE' in args:
         args[ 'OPTION' ].add( ( 'preference', args.pop( 'PREFERENCE' ) ) )
      if 'tag' in args:
         args[ 'OPTION' ].add( ( 'tag', args.pop( 'TAGNUM' ) ) )
      if 'name' in args:
         args[ 'OPTION' ].add( ( 'nextHopName', args.pop( 'NEXTHOP_NAME' ) ) )

   @staticmethod
   def handler( mode, args ):
      manageIpv6StaticEvpnRoute( mode, args )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      preference = args.get( 'PREFERENCE', None )
      vrfName = args.get( 'VRF', DEFAULT_VRF )
      evpnNextHop = args[ 'EVPN_NEXTHOP' ]
      nexthop = {}
      if 'vtepAddr' in evpnNextHop and 'vni' in evpnNextHop and \
            'routerMac' in evpnNextHop:
         nexthop[ 'evpn' ] = {}
         nexthop[ 'evpn' ][ 'vtepAddr' ] = evpnNextHop[ 'vtepAddr' ]
         nexthop[ 'evpn' ][ 'vni' ] = evpnNextHop[ 'vni' ]
         nexthop[ 'evpn' ][ 'routerMac' ] = evpnNextHop[ 'routerMac' ]

      IraIp6Cli.noIp6Route( mode, args[ 'PREFIX' ], nexthop, preference, vrfName,
                          None ) 

class Ipv6RouteEvpnCmdMultiVtep( CliCommand.CliCommandClass ):
   syntax = '''ipv6 route [ VRF ] PREFIX vtep VTEP vni VNI
               router-mac-address MAC_ADDR
               [ { PREFERENCE | ( tag TAGNUM ) | ( name NEXTHOP_NAME ) |
               ( local-interface INTF ) | dynamic | rib-bypass } ]'''
   noOrDefaultSyntax = '''ipv6 route [ VRF ] PREFIX vtep [ VTEP vni VNI
                          router-mac-address MAC_ADDR ] [ PREFERENCE ] ...'''
   data = { 'ipv6': CliToken.Ipv6.ipv6MatcherForConfig,
            'route': routeMatcherForIpv6Config,
            'VRF': IraIp6Cli.vrfExprFactoryForConfig,
            'PREFIX': ipv6PrefixMatcher,
            'vtep': vtepMatcherForConfig,
            'VTEP': vtepAddrNode,
            'vni': vniMatcherForConfig,
            'VNI': vniMatcher,
            'router-mac-address': routerMacMatcherForConfig,
            'MAC_ADDR': MacAddr.MacAddrMatcher(),
            'local-interface': CliCommand.Node( BridgingCli.matcherInterface,
                                          maxMatches=1 ),
            'INTF': CliCommand.Node( IntfRange.IntfRangeMatcher(
                                     explicitIntfTypes=vxlanIntfType ),
                                     maxMatches=1 ),
            'PREFERENCE': CliCommand.Node( IraIpRouteCliLib.preferenceRangeMatcher,
                                           maxMatches=1 ),
            'tag': CliCommand.Node( IraIpRouteCliLib.tagMatcherForConfig,
                                    maxMatches=1 ),
            'TAGNUM': CliCommand.Node( IraIpRouteCliLib.tagNumberMatcher,
                                       maxMatches=1 ),
            'name': CliCommand.Node( IraCommonCli.nameMatcherForConfig,
                                     maxMatches=1 ),
            'NEXTHOP_NAME': CliCommand.Node( IraCommonCli.nexthopNameMatcher,
                                             maxMatches=1 ),
            'dynamic' : CliCommand.Node( dynamicMatcherForConfig, maxMatches=1 ),
            'rib-bypass' : CliCommand.Node(
               ribBypassMatcherForConfig, maxMatches=1 ),
            }

   @staticmethod
   def adapter( mode, args, argsList ):
      args[ 'EVPN_NEXTHOP' ] = {}
      args[ 'EVPN_NEXTHOP' ][ 'vtepAddr' ] = args[ 'VTEP' ].stringValue
      args[ 'EVPN_NEXTHOP' ][ 'vni' ] = args[ 'VNI' ]
      args[ 'EVPN_NEXTHOP' ][ 'routerMac' ] = args[ 'MAC_ADDR' ]
      if 'local-interface' in args:
         args[ 'EVPN_NEXTHOP' ][ 'vxlanSrcIntf' ] = args[ 'INTF' ]
      args[ 'EVPN_NEXTHOP' ][ 'dynamic' ] = 'dynamic' in args
      args[ 'EVPN_NEXTHOP' ][ 'ribBypass' ] = 'rib-bypass' in args

      args[ 'OPTION' ] = set()
      if 'PREFERENCE' in args:
         args[ 'OPTION' ].add( ( 'preference', args.pop( 'PREFERENCE' ) ) )
      if 'tag' in args:
         args[ 'OPTION' ].add( ( 'tag', args.pop( 'TAGNUM' ) ) )
      if 'name' in args:
         args[ 'OPTION' ].add( ( 'nextHopName', args.pop( 'NEXTHOP_NAME' ) ) )

   @staticmethod
   def handler( mode, args ):
      manageIpv6StaticEvpnRoute( mode, args )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      preference = args.get( 'PREFERENCE', None )
      vrfName = args.get( 'VRF', DEFAULT_VRF )
      evpnNextHop = args[ 'EVPN_NEXTHOP' ]
      nexthop = {}
      if 'vtepAddr' in evpnNextHop and 'vni' in evpnNextHop and \
            'routerMac' in evpnNextHop:
         nexthop[ 'evpn' ] = {}
         nexthop[ 'evpn' ][ 'vtepAddr' ] = evpnNextHop[ 'vtepAddr' ]
         nexthop[ 'evpn' ][ 'vni' ] = evpnNextHop[ 'vni' ]
         nexthop[ 'evpn' ][ 'routerMac' ] = evpnNextHop[ 'routerMac' ]

      IraIp6Cli.noIp6Route( mode, args[ 'PREFIX' ], nexthop, preference, vrfName,
                          None ) 

if Toggles.VxlanToggleLib.toggleMultiVxlanTunnelInterfacesEnabled():
   BasicCli.GlobalConfigMode.addCommandClass( Ipv6RouteEvpnCmdMultiVtep )
else:
   BasicCli.GlobalConfigMode.addCommandClass( Ipv6RouteEvpnCmd )

#-------------------------------------------------------------------------------
# The "show vxlan counters software" command, in "enable" mode.
#-------------------------------------------------------------------------------
def createMapSwCounterIdToStr( countersToStr, countersToKey, countersToName,
                               cntrIdMap, strKey=False ):
   # countersToStr stores the description for the counter
   # countersToName stores the name of the counter
   def getSwKeyRepresentation( key ):
      return key if strKey else counterId.enumVal( key )

   def addSwCountersToKey( key, name=None ):
      countKey = getSwKeyRepresentation( key )
      countersToKey[ countKey ] = key
      if name:
         countersToName[ countKey ] = name
      else:
         countersToName[ countKey ] = key
      return countKey

   counterId = Tac.newInstance("Vxlan::VxlanCounterId") 

   for cntr, value in cntrIdMap.iteritems():
      key = value[ 0 ] if value[ 0 ] else cntr
      countKey = addSwCountersToKey( key, cntr )
      countersToStr[ countKey ] = value[ 1 ]

def showSwCtrs( mode, args ):
   counterIdToStr = {}
   counterIdToKey = {}
   counterIdToName = {}
   counterModel = VxlanModel.VxlanSwCounters()
   cntrIdMap = counterModel.getCntrIdMap()
   createMapSwCounterIdToStr( counterIdToStr, counterIdToKey,
                              counterIdToName, cntrIdMap )

   if not vxAgentCounter.vxlanCounter.keys():
      idx = 0
      for cntr in counterIdToName.iteritems():
         counterModel.__setattr__( counterIdToName[ idx ], 0 )
         idx += 1
   else:
      for cntr in vxAgentCounter.vxlanCounter.iteritems():
         cntr = cntr[ 1 ]
         if counterIdToName.has_key( cntr.key ):
            counterModel.__setattr__( counterIdToName[ cntr.key ],
                                           cntr.counterValue )

   return counterModel

#-------------------------------------------------------------------------------
# The "clear vxlan counters software" command, in "enable" mode.
#-------------------------------------------------------------------------------
def clearSwCounters( mode, args ):
   vxlanConfigDir.clearSwCounterRequestTime = Tac.now()

#-------------------------------------------------------------------------------
# The "show vxlan counters vtep [<vtep>] [encap|decap]" command,
# in "enable" mode.
#-------------------------------------------------------------------------------
# This is a hook for the platform to implement its own function to collect vtep
# counters. The hook will receive VxlanVtepCountersModel object and zero or
# more filled VxlanVtepCounters . If the hooks finds a counter to be already
# populated it should add its own values for that counter to it.

showVtepCountersHook = CliHook()

def showVtepCounters( mode, args ):
   ''' Create an object of the container and pass to platform hooks to
       update it. Platform may use more than one hook to collect the counters
       and update the model using addToAttr() calls.'''
   direction = args.get( 'DIRECTION' )
   vtep = None
   if 'unlearnt' in args:
      vtep = 'unlearnt'
   elif 'IP' in args:
      vtep = args[ 'IP' ]
      vtep = str( vtep )

   vtepCounters = VxlanModel.VxlanVtepCountersModel()
   showVtepCountersHook.notifyExtensions( mode=mode, inVtep=vtep,
         direction=direction, vtepCounters=vtepCounters )
   return vtepCounters

#-------------------------------------------------------------------------------
# The "clear vxlan counters vtep [<vtep>]" command,
# in "enable" mode.
#-------------------------------------------------------------------------------
# Clear Command for Vxlan VTEP counters. 
# The command clears vxlan hardware counters by publishing state into 
# vxlan/hardware/counter vtepClearRequest collection.
# The feature agents react to changes in the collection and clear the vteps

def clearVtepCounters( mode, args ):
   ''' clearVtepCounters populates the entity in vxlan/hardware/counter
       with either the vtep's Ip address, keyword 'unlearnt' or 
       255.255.255.255 to clear all counters '''
   
   # if there is no vtep specified we use the special address 255.255.255.255 to
   # signal that all vteps should be cleared
   vtep = args.get( 'VTEP' )
   if 'unlearnt' in args:
      # if vtep is 'unlearnt' then we normalize to 0.0.0.0
      vtep = '0.0.0.0'

   if vtep is None:
      vtep = '255.255.255.255'
      vxlanHwCounter.vtepClearRequest.clear()
   else:
      vtep = str( vtep )
      del vxlanHwCounter.vtepClearRequest[ Arnet.IpGenAddr( vtep ) ]

   vxlanHwCounter.vtepClearRequest[ Arnet.IpGenAddr( vtep ) ] = True

#-------------------------------------------------------------------------------
# The "show vxlan counters vni [<vni>] [encap|decap]" command,
# in "enable" mode.
#-------------------------------------------------------------------------------
# This is a hook for the platform to implement its own function to collect vni
# counters. The hook will receive VxlanVniCountersModel object and zero or
# more filled VxlanVniCounters . If the hooks finds a counter to be already
# populated it should add its own values for that counter to it.
showVniCountersHook = CliHook()

def showVniCounters( mode, args ):
   ''' Create an object of the container and pass to platform hooks to
       update it. Platform may use more than one hook to collect the counters
       and update the model without stepping on each other.'''
   vniDottedFormat = args.get( 'VNI' )
   direction = args.get( 'DIRECTION' )
   vni = None
   if vniDottedFormat is not None:
      # vni passes to hooks is always a number
      vni = VniFormat( vniDottedFormat ).toNum()
      if not isValidVniWithError( mode, vni, 'Vxlan1' ):
         return None

   vniCounters = VxlanModel.VxlanVniCountersModel()
   vniCounters.vniInDottedNotation = \
            vxlanControllerConfig.vniInDottedNotation
   showVniCountersHook.notifyExtensions( mode=mode, inVni=vni,
         direction=direction, vniCounters=vniCounters )
   return vniCounters

#-------------------------------------------------------------------------------
# The "clear vxlan counters vni [<vni>]" command,
# in "enable" mode.
#-------------------------------------------------------------------------------
# Clear Command for Vxlan VNI counters. 
# The command clears vxlan hardware counters by publishing state into 
# vxlan/hardware/counter vniClearRequest collection.
# The feature agents react to changes in the collection and clear the vteps

def clearVniCounters( mode, args ):
   ''' clearVniCounters populates the entity in vxlan/hardware/counter
       with either the VNI's Id  or VNI's invalid Id to clear all counters '''
   vniDottedFormat = args.get( 'VNI' )
   
   # if there is no VNI specified we use the special value 'invalid'  to
   # signal that all VNIs should be cleared
   if vniDottedFormat is None:
      vniDottedFormat = 'invalid'

   vni = VniFormat( vniDottedFormat ).toNum()
   if vniDottedFormat != 'invalid' and \
         not isValidVniWithError( mode, vni, 'Vxlan1' ):
      return

   if vni == Tac.Value( "Vxlan::VniExtOrNone" ).invalidVni:
      vxlanHwCounter.vniClearRequest.clear()
   else:
      del vxlanHwCounter.vniClearRequest[ vni ]

   vxlanHwCounter.vniClearRequest[ vni ] = True

#-------------------------------------------------------------------------------
# The "show vxlan counters varp" command, in "enable" mode.
#-------------------------------------------------------------------------------
def createMapVarpCounterIdToStr( countersToStr, countersToKey,
                                 cntrIdMap, strKey=False ):
   # countersToStr maps a counter ID to its counter description.
   # varpCountersToName maps a counter ID to its counter name.

   def getVarpKeyRepresentation( key ):
      return key if strKey else counterId.enumVal( key )

   def addVarpCountersToKey( key ):
      countKey = getVarpKeyRepresentation( key )
      countersToKey[ countKey ] = key
      return countKey

   counterId = Tac.newInstance("Vxlan::VarpCounterId") 
   for cntr, value in cntrIdMap.iteritems():
      key = value[ 0 ] if value[ 0 ] else cntr
      countKey = addVarpCountersToKey( key )
      countersToStr[ countKey ] = value[ 1 ]

def showVarpCounters( mode, args ):
   counterIdToStr = {}
   counterIdToName = {}
   counterModel = VxlanModel.VxlanVarpCounters()
   cntrIdMap = counterModel.getCntrIdMap()
   createMapVarpCounterIdToStr( counterIdToStr, counterIdToName,
                                cntrIdMap)

   if not vxAgentCounter.varpCounter.keys():
      idx = 0
      for cntr in counterIdToName.iteritems():
         counterModel.__setattr__( counterIdToName[ idx ], 0 )
         idx += 1
   else:
      for cntr in vxAgentCounter.varpCounter.iteritems():
         cntr = cntr[ 1 ]
         if counterIdToName.has_key( cntr.key ):
            counterModel.__setattr__( counterIdToName[ cntr.key ],
                                          cntr.counterValue )

   return counterModel

#-------------------------------------------------------------------------------
# The "clear vxlan counters varp" command, in "enable" mode.
#-------------------------------------------------------------------------------
def clearVarpCounters( mode, args ):
   vxlanConfigDir.clearVarpCounterRequestTime = Tac.now()

#-------------------------------------------------------------------------------
# Build and return a VxlanVniStatus CAPI object filtered by macAddr and
# vlanId with entries from the passed in vxlanvni status object.
# -------------------------------------------------------------------------------
def buildVxlanVniStatusV2( vni, macAddr, vlanId, vxlanVniStatusV2,
                           intfName, vtepIpList=None ):
   vlan = vniToVlanMap.getVlan( int( vni ), intfName )
   
   # Skip this vni if there's no vlan mapping for it or if a vlanId
   # was specified and it doesn't match.
   if vlan is None or (vlanId and vlanId.id != vlan):
      return None

   vs = VxlanVniStatus()
   vs.vlan = vlan
   vs.vni = int(vni)
   
   # scan the macVtepTable 
   for lhmacAddr, lhmvp in vxlanVniStatusV2.macVtepTable.iteritems():
   
      # if no mac is specified print them all, else print the requested mac
      if macAddr and not MacAddr.compareMacs( macAddr, lhmacAddr ):
         continue

      # if no vtep is specified print them all, else print the requested vtep 
      if vtepIpList and lhmvp.vtepIp not in vtepIpList:
         continue

      mvp = VxlanMacVtepPair()
      mvp.macAddr = lhmvp.macAddr
      mvp.vtepIp = lhmvp.vtepIp
      mvp.moveCount = lhmvp.moveCount
      vs.unicastHostTable.append( mvp )

   # scan the macVtepMultiPathTable
   for bhmacAddr, bhmvlp in vxlanVniStatusV2.macVtepMultiPathTable.iteritems():
      # if no mac is specified print them all, else print the requested mac
      if macAddr and not MacAddr.compareMacs( macAddr, bhmacAddr ):
         continue

      # if no vtep is specified print them all, else print the requested vtep 
      if vtepIpList:
         filteredVtepIpList = [ vtepIp for vtepIp in vtepIpList \
                                          if vtepIp in bhmvlp.vtepIpList ]
         if not filteredVtepIpList:
            continue

      mvlp = VxlanMacVtepListPair()
      mvlp.macAddr = bhmacAddr
      mvlp.vtepIpListType = 'sendToAny'
   
      # append all VTEP IPs for this mac
      vteps = bhmvlp.vtepIpList.keys()
      mvlp.vtepIpList.extend(vteps)
   
      vs.bumVtepListTable.append( mvlp )

   # scan the macVtepFloodListTable
   for bhmacAddr, bhmvlp in vxlanVniStatusV2.macVtepFloodListTable.iteritems():
      # if mac is also present in macVtepMultiPathTable, skip it
      if bhmacAddr in vxlanVniStatusV2.macVtepMultiPathTable:
         continue

      # if no mac is specified print them all, else print the requested mac
      if macAddr and not MacAddr.compareMacs( macAddr, bhmacAddr ):
         continue

      # if no vtep is specified print them all, else print the requested vtep 
      if vtepIpList:
         filteredVtepIpList = [ vtepIp for vtepIp in vtepIpList \
                                          if vtepIp in bhmvlp.vtepIpFloodList ]
         if not filteredVtepIpList:
            continue

      mvlp = VxlanMacVtepListPair()
      mvlp.macAddr = bhmacAddr
      mvlp.vtepIpListType = 'sendToAll'
   
      # append all VTEP IPs for this mac
      vteps = bhmvlp.vtepIpFloodList.keys()
      mvlp.vtepIpList.extend(vteps)
   
      vs.bumVtepListTable.append( mvlp )
   
   return vs

#-------------------------------------------------------------------------------
# show vxlan controller address-table advertised [ vlan <vlan> | mac <mac> ]
#-------------------------------------------------------------------------------
def showVxlanVniStatusAdvertised( mode, args ):
   vsd = VxlanModel.VxlanVniStatusDirModel()
   vsd.tableType = 'Advertised'
   vlanId = args.get( 'VLAN_ID' )
   macAddr = args.get( 'MAC_ADDR' )

   if vlanId:
      vni = None
      for intfId in vtiStatusDir.vtiStatus:
         vlanVniMap = vtiStatusDir.vtiStatus[ intfId ].vlanToVniMap
         if vlanId.id in vlanVniMap:
            vni = vlanVniMap[ vlanId.id ].vni
            break
      if vni and vni in vxlanVniStatusDir.vniStatusV2:
         vs = buildVxlanVniStatusV2( vni, macAddr, vlanId,
                                     vxlanVniStatusDir.vniStatusV2[ vni ],
                                     'Vxlan1' )
         vsd.vniStatus.append( vs )
      return vsd

   # scan each vni
   for vni, vniStatus in vxlanVniStatusDir.vniStatusV2.iteritems():

      vsV1 = buildVxlanVniStatusV2( vni, macAddr, vlanId, vniStatus,
                                    'Vxlan1' )
      if vsV1:
         vsd.vniStatus.append( vsV1 )

   return vsd

#-------------------------------------------------------------------------------
# show vxlan controller address-table received
#                                      [ vlan <vlan> | mac <mac> | vtep <vtep> ]
#-------------------------------------------------------------------------------
def showVxlanVniStatusReceived( mode, args ):
   vsd = VxlanModel.VxlanVniStatusDirModel()
   vsd.tableType = 'Received'
   vlanId = args.get( 'VLAN_ID' )
   macAddr = args.get( 'MAC_ADDR' )
   vteps = args.get( 'VTEP' )

   if not vxlanVniFdbStatusDir:
      return vsd

   vfsd = vxlanVniFdbStatusDir

   # scan each vni
   for vni in vfsd:
      vsV1 = buildVxlanVniStatusV2( vni, macAddr, vlanId, vfsd[ vni ],
                                    'Vxlan1', vtepIpList=vteps )
      if vsV1:
         vsd.vniStatus.append( vsV1 )

   return vsd

def showVxlanControlServiceStatus( mode, args ):
   statuses = { 'oobStateConnecting': 'connecting',
                'oobStateNegotiating': 'negotiating',
                'oobStateEstablished': 'established',
                'oobStateDisconnecting': 'disconnecting',
                'oobStateShuttingDown': 'disconnecting',
                'oobStateShutdown': 'disconnecting' }

   vcsStatus = VxlanModel.VxlanControlServiceStatus()
   controllerStatus = None
   for controller in cvxClientStatus.controllerStatus:
      if cvxClientStatus.controllerStatus[ controller ].leader:
         controllerStatus = cvxClientStatus.controllerStatus[ controller ]
         break
   if not controllerStatus:
      vcsStatus.status = 'disconnecting'
   else:
      status = controllerStatus.serviceStatusDir.service.get( 'Vxlan' )
      vcsStatus.status = statuses[ controllerStatus.connectionStatus.state ] if \
                            status and status.enabled else 'disconnecting'
   vcsStatus.resyncInProgress = vcsStateClientView.convergenceInProgress
   vcsStatus.purgeInProgress = (
         vxlanClientConvergenceStatus.unicastDeferredDeletionInProgress |
         vxlanClientConvergenceStatus.bumDeferredDeletionInProgress )
   return vcsStatus

#-------------------------------------------------------------------------------
# Build and return a VxlanArpTable CAPI object filtered by ip and vlanId with
# entries from the passed in vxlanVniStatus object.
#-------------------------------------------------------------------------------
def buildVxlanArpTable( vrf, vlanId, ip, macAddr, ipToMacByVni, arpType, intfName ):
   def makeArpTable( vni, ipToMac ):
      entries = ( ( k, v ) for ( k, v ) in ipToMac.iteritems()
                  if not ip or str( ip ) == str( k ) )
      entries = ( ( k, v ) for ( k, v ) in entries
                  if not macAddr or MacAddr.compareMacs( macAddr, v.macAddr ) )

      arpTable = VxlanArpTable()
      arpTable.addresses = { k: VxlanArpEntry( macAddress=v.macAddr,
                                               preference=v.preference,
                                               changes=v.changeCount )
                             for ( k, v ) in entries }
      return arpTable

   def accept( vlan ):
      if not( vlan ) or ( vlanId and vlanId.id != vlan ):
         return False

      if vrf and vrf not in ipStatus.vrfIpIntfStatus:
         return False

      if vrf:
         return 'Vlan%d' % vlan in ipStatus.vrfIpIntfStatus[ vrf ].ipIntfStatus

      return True

   col = ( ( vni, vniToVlanMap.getVlan( vni, intfName ), ipToMac )
          for ( vni, ipToMac ) in ipToMacByVni )

   model = VxlanModel.VxlanArpTableModel( tableType=arpType )
   model.vlanToArpTable = { vlan: makeArpTable( vni, ipToMac )
                            for ( vni, vlan, ipToMac ) in col if accept( vlan ) }
   return model

#-------------------------------------------------------------------------------
# show vxlan controller arp advertised [ vlan <vlan> | ip <A.B.C.D> ]
#-------------------------------------------------------------------------------
def showVxlanIpToMacAdvertised( mode, args ):
   vrf = args.get( 'VRF' )
   vlanId = args.get( 'VLAN_ID' )
   ip = args.get( 'IP' )
   macAddr = args.get( 'MAC_ADDR' )
   ipToMacByVni = ( ( k, v.ipToMacTable )
                    for ( k, v ) in vxlanVniStatusDir.vniStatusV2.iteritems() )
   return buildVxlanArpTable( vrf, vlanId, ip, macAddr, ipToMacByVni,
                              'advertised', 'Vxlan1' )

#-------------------------------------------------------------------------------
# show vxlan controller arp received [ vlan <vlan> | ip <A.B.C.D> ]
#-------------------------------------------------------------------------------
def showVxlanIpToMacReceived( mode, args ):
   vrf = args.get( 'VRF' )
   vlanId = args.get( 'VLAN_ID' )
   ip = args.get( 'IP' )
   macAddr = args.get( 'MAC_ADDR' )
   ipToMacByVni = ( ( int( k ), v.ipToMacTable )
                    for ( k, v ) in vxlanVniFdbStatusDir.iteritems() )
   return buildVxlanArpTable( vrf, vlanId, ip, macAddr, ipToMacByVni,
                              'received', 'Vxlan1' )

#-------------------------------------------------------------------------------
# show vxlan controller logical-router [ name <lr-name> ] [ vni <vni> ]
# show vxlan controller logical-router uplink [ name <lr-name> ] 
# show vxlan controller logical-router routes [ name <lr-name> ] 
#-------------------------------------------------------------------------------
def isLocalVtepAddr( vtepIp ):
   for vtiStatus in vtiStatusDir.vtiStatus.values():
      if vtiStatus.localVtepAddr == vtepIp:
         return True
   return False   

def addLogicalRouter( logicalRouter, lRModel, lRName, vniFilter, processIpPort,
                      processUplinkPort, processRoute ):
   lRModel.routingTable[ lRName ] = VxlanModel.VxlanRoutingTable()
   lRModel.routingTable[ lRName ].lRCreateLocalIpPortsOnly = \
       logicalRouter.createLocalIpPortsOnly
   if processIpPort:
      for ipPort in logicalRouter.ipPort.values():
         if ipPort.vtep.stringValue != '127.0.0.1' and \
            not isLocalVtepAddr( ipPort.vtep.stringValue ):
            continue
         if vniFilter and vniFilter != ipPort.vni:
            continue
         modelIpPort = VxlanModel.LRIpPort( vni = ipPort.vni,
                                            macAddr = ipPort.mac )
         lRModel.routingTable[ lRName ].lRPort[ ipPort.ip.stringValue ] = modelIpPort

   if processUplinkPort:
      for uplinkPort in logicalRouter.ipUplinkPort.values():
         if uplinkPort.vtep.stringValue != '127.0.0.1' and \
            not isLocalVtepAddr( uplinkPort.vtep.stringValue ):
            continue
         if uplinkPort.portVlan:
            # Assuming vlan id is same in all keys
            vlanId = uplinkPort.portVlan.keys()[ 0 ].vlanId
            interfaceList = [ x.port for x in uplinkPort.portVlan ]
            modelIpUplinkPort = VxlanModel.LRIpUplinkPort(
               vlan = vlanId,
               interfaces = interfaceList,
               macAddr = uplinkPort.mac )
            lRModel.routingTable[ lRName ].lRUplinkPort[
               uplinkPort.ip.stringValue ] = modelIpUplinkPort

   if processRoute:
      for route in logicalRouter.route.values():
         modelIpRoute = VxlanModel.LRIpRoute()
         for nexthop in route.nexthop:
            modelIpRoute.nexthop.append( nexthop )
         lRModel.routingTable[ lRName ].lRRoute[ route.prefix.stringValue ] = \
            modelIpRoute

def processLRStatus( lRNameFilter, vniFilter, processIpPort, 
                     processUplinkPort, processRoute ):
   lRModel = VxlanModel.VxlanLogicalRouterModel()
   # pylint: disable-msg=W0212
   lRModel._vniDotted = vxlanControllerConfig.vniInDottedNotation
   if lRNameFilter:
      if lRNameFilter in lRStatus.entityPtr:
         logicalRouter = lRStatus[ lRNameFilter ]
         addLogicalRouter( logicalRouter, lRModel, lRNameFilter, vniFilter,
                           processIpPort, processUplinkPort, processRoute )
      return lRModel

   for lRName in lRStatus.entityPtr:
      logicalRouter = lRStatus[ lRName ]
      addLogicalRouter( logicalRouter, lRModel, lRName, vniFilter,
                        processIpPort, processUplinkPort, processRoute )
   return lRModel

def showVxlanLogicalRouter( mode, args ):
   vniFilter = None
   lRNameFilter = None
   if 'LOGICAL_ROUTER' in args:
      lRNameFilter = args[ 'LOGICAL_ROUTER' ][ 0 ]
   if 'VNI' in args:
      vniFilter = int( args[ 'VNI' ][ 0 ] )
      if not isValidVniWithError( mode, vniFilter, 'Vxlan1' ):
         return None
 
   if vniFilter:
      return processLRStatus( lRNameFilter=lRNameFilter, 
                              vniFilter=vniFilter,
                              processIpPort=True,
                              processUplinkPort=False,
                              processRoute=False )
   else:
      return processLRStatus( lRNameFilter=lRNameFilter, 
                              vniFilter=None,
                              processIpPort=True,
                              processUplinkPort=True,
                              processRoute=True )
   
def showVxlanUplinkPorts( mode, args ):
   return processLRStatus( lRNameFilter=args.get( 'LOGICAL_ROUTER' ),
                           vniFilter=None,
                           processIpPort=False,
                           processUplinkPort=True,
                           processRoute=False )

def showVxlanRoutes( mode, args ):
   return processLRStatus( lRNameFilter=args.get( 'LOGICAL_ROUTER' ),
                           vniFilter=None,
                           processIpPort=False,
                           processUplinkPort=False,
                           processRoute=True )

#-------------------------------------------------------------------------------
# The "show vxlan address-table" command, in "enable" mode.
#
# The full syntax of this command is:
#
#   show vxlan address-table [ mlag-peer ] [static|dynamic|received|unicast|evpn]
#                            [address <mac_addr>] [vtep <ip_addr>] [vlan <vlan_id>]
#-------------------------------------------------------------------------------
def getMovesAndLastMoveTime( vlanId, macAddr, host ):
   fid = bridgingConfig.vidToFidMap.get( vlanId )
   if not fid:
      fid = vlanId
   smashKey = Tac.Value( 'Bridging::HostKey', fid, macAddr )
   macEntry = smashBridgingStatus.smashFdbStatus.get( smashKey )
   if macEntry is None:
      return ( 0, 0 )
   else:
      return ( macEntry.moves, macEntry.lastMoveTime )

def filterVxlanMacAddr( host, vlanId, macAddr, hostType, vteps,
                        vtepType=SwTunnelGroupType.singleMemberGroup ):
   if vlanId:
      if vlanId.id != host.macVlanPair.vlanId:
         return False
   if macAddr:
      if Ethernet.convertMacAddrToCanonical( macAddr ) != \
            host.macVlanPair.macAddr:
         return False
   if hostType:
      if hostType != 'evpn' and vtepType == SwTunnelGroupType.multiMemberGroup:
         return False
      if hostType == 'dynamic' and host.entryType != 'learnedMac':
         return False
      if hostType == 'static' and host.entryType != 'configuredMac':
         return False
      if hostType == 'received' and host.entryType != 'receivedMac':
         return False
      if hostType == 'unicast' and not Ethernet.isUnicast(host.macVlanPair.macAddr):
         return False
      if hostType == 'evpn' and host.entryType != 'evpnMac':
         return False
   if vteps:
      if vtepType is SwTunnelGroupType.singleMemberGroup:
         if str( host.remoteVtepAddr ) not in vteps:
            return False
      else:
         vteps = set( vteps )
         return any( str( ip ) in vteps for ip in host.remoteVtepList )

   return True

def vxlanMacAddressGenerator( vlanId, macAddr, hostType, vteps, vti ):
   if hostType == 'configured':
      # For 'configured', get the mac addresses from configDir.
      hosts = vxlanConfigDir.fdbConfig.configuredHost
   else:
      hosts = vxHwStatusDir.fdbStatus.learnedHost
   multiVtepHosts = vxHwStatusDir.fdbStatus.multiVtepLearnedHost

   sortedHosts = {}
   if macAddr and vlanId:
      macVlanPair = Tac.Value( "Vxlan::MacVlanPair", macAddr, vlanId.id )
      vtepType = SwTunnelGroupType.multiMemberGroup if multiVtepHosts.has_key(
            macVlanPair ) else SwTunnelGroupType.singleMemberGroup
      macEntry = multiVtepHosts.get( macVlanPair ) if \
            vtepType is SwTunnelGroupType.multiMemberGroup else \
            hosts.get( macVlanPair )
      if macEntry:
         if filterVxlanMacAddr( macEntry, vlanId, macAddr, hostType, vteps,
                                vtepType ):
            sortedHosts[ macEntry.macVlanPair ] = [ macEntry, vtepType ]
   else:
      if any( [ vlanId, macAddr, hostType, vteps ] ):
         for macEntry in multiVtepHosts.itervalues():
            if filterVxlanMacAddr( macEntry, vlanId, macAddr, hostType, vteps,
                                   SwTunnelGroupType.multiMemberGroup ):
               sortedHosts[ macEntry.macVlanPair ] = [ macEntry,
                     SwTunnelGroupType.multiMemberGroup ]
         for macEntry in hosts.itervalues():
            if not sortedHosts.has_key( macEntry.macVlanPair ) and \
                  filterVxlanMacAddr( macEntry, vlanId, macAddr, hostType, vteps ):
               sortedHosts[ macEntry.macVlanPair ] = [ macEntry,
                     SwTunnelGroupType.singleMemberGroup ]

      else:
         for macEntry in multiVtepHosts.itervalues():
            sortedHosts[ macEntry.macVlanPair ] = [ macEntry,
                  SwTunnelGroupType.multiMemberGroup ]
         for macEntry in hosts.itervalues():
            if not sortedHosts.has_key( macEntry.macVlanPair ):
               sortedHosts[ macEntry.macVlanPair ] = [ macEntry,
                     SwTunnelGroupType.singleMemberGroup ]

   sortedHostsList = [ [ vtepType, macEntry.macVlanPair.vlanId, 
                         macEntry.macVlanPair.macAddr, macEntry ]
                     for [ macEntry, vtepType ] in sortedHosts.itervalues() ]

   sortedHostsList = sorted( sortedHostsList, key=itemgetter( 1, 2 ) )

   for sVtepType, sVlanId, sMacAddr, host in sortedHostsList:
      if hostType == 'configured':
         ( moves, lastMoveTime ) = ( 0, 0 )
         entType = 'static'
      elif host.entryType == 'evpnMac':
         ( moves, lastMoveTime) = getMovesAndLastMoveTime( sVlanId, sMacAddr, host )
         entType = 'evpn'
      else:
         ( moves, lastMoveTime) = getMovesAndLastMoveTime( sVlanId, sMacAddr, host )
         if host.entryType == 'learnedMac':
            entType = 'dynamic'
         elif host.entryType == 'configuredMac':
            entType = 'static'
            ( moves, lastMoveTime ) = ( 0, 0 )
         elif host.entryType == 'receivedMac':
            entType = 'received'
         else:
            entType = 'unknown'
      if sVtepType is SwTunnelGroupType.multiMemberGroup:
         vtepAddrs = sorted( host.remoteVtepList.keys() )
      else:
         vtepAddrs = [ host.remoteVtepAddr ]

      yield VxlanModel.VxlanMacAddresses.VxlanMacAddrListing(
         vlanId=sVlanId,
         macAddress=sMacAddr,
         addrType=entType,
         interface=IntfCli.Intf.getShortname( vti ),
         vteps=vtepAddrs,
         moves=moves,
         lastMove=float( lastMoveTime ) )

def showVxlanMacAddr( mode, args ):
   vlanId = args.get( 'VLAN_ID' )
   macAddr = args.get( 'MAC_ADDR' )
   vteps = args.get( 'VTEP' )
   hostType = args.get( 'HOST_TYPE' )
   addrsModel = VxlanModel.VxlanMacAddresses()

   if vteps is not None:
      vteps = set( str( v ) for v in vteps )
   if 'mlag-peer' in args:
      addresses = mlagPeerVxlanMacAddrGenerator( hostType, macAddr, vlanId, vteps )
   else:
      addresses = vxlanMacAddressGenerator( vlanId, macAddr, hostType, vteps,
                                            'Vxlan1' )

   addrsModel.addresses = addresses
   return addrsModel

def showVxlanMacAddrCount( mode, args ):
   vtepInputs = args.get( 'VTEP' )
   if vtepInputs is not None:
      vtepInputs = set( str( v ) for v in vtepInputs )
   addrsModel = VxlanModel.VxlanAddressTableCount()
   learnedHosts = vxHwStatusDir.fdbStatus.learnedHost
   multiVtepHosts = vxHwStatusDir.fdbStatus.multiVtepLearnedHost

   counter = Counter()
   for macVlanPair, learnedHost in learnedHosts.iteritems():
      host = multiVtepHosts.get( macVlanPair )
      if host is None:
         vtep = str( learnedHost.remoteVtepAddr )
         if vtepInputs is None or vtep in vtepInputs:
            counter.update( [ vtep ] )
      else:
         vteps = set( str( ip ) for ip in host.remoteVtepList )
         if vtepInputs is not None:
            vteps = vteps.intersection( set( vtepInputs ) )
         counter.update( list( vteps ) )

   addrsModel.vtepCounts = dict( counter )
   return addrsModel

def mlagPeerVxlanMacAddrGenerator( hostType, macAddr, vlanId, vteps ):
   hostEntryToType = {
      'peerLearnedRemoteMac' : 'dynamic',
      'peerConfiguredRemoteMac' : 'static',
      'peerReceivedRemoteMac' : 'received',
      'peerEvpnRemoteMac' : 'evpn',
   }

   keys = []
   # hostTables are indexed by MacAddr:VlanId. Optimize if both are specified
   if macAddr and vlanId:
      key = Tac.Value( "Mlag::HostEntryKey", vlanId.id, macAddr )
      if key in mlagHostTable.hostEntry:
         entry = mlagHostTable.hostEntry[ key ]
         if entry.entryType in hostEntryToType:
            keys.append( key )
   else:
      keys = [ key for key in mlagHostTable.hostEntry \
                  if mlagHostTable.hostEntry[ key ].entryType in hostEntryToType ]

   for key in sorted( keys, key=lambda k: k.vlanId ):
      entry = mlagHostTable.hostEntry.get( key )
      if not entry:
         continue
      # Apply the filters
      if hostType:
         if hostType == 'unicast':
            if not Ethernet.isUnicast( entry.address ):
               continue
         elif hostEntryToType[ entry.entryType ] != hostType:
            continue
      if ( ( macAddr and
             Ethernet.convertMacAddrToCanonical( macAddr ) != entry.address ) or
           ( vlanId and entry.vlanId != vlanId.id ) ):
         continue
      tokens = entry.intf.split( ':' )
      vtepAddr = tokens[ 1 ]
      if vteps and vtepAddr not in vteps:
         continue

      yield VxlanModel.VxlanMacAddresses.VxlanMacAddrListing(
         vlanId=entry.vlanId,
         macAddress=entry.address,
         addrType=hostEntryToType[ entry.entryType ],
         interface=IntfCli.Intf.getShortname( tokens[ 0 ] ),
         vteps=[ vtepAddr ],
         moves=entry.moves,
         lastMove=float( entry.lastMoveTime ) )

#-------------------------------------------------------------------------------
# The "show vxlan vtep" command, in "enable" mode.
#
# The full syntax for this command is:
#   show vxlan vtep [ type TUNNEL_TYPE ] [ detail ]
#-------------------------------------------------------------------------------
def showVxlanVteps( mode, args ):
   vtepModel = VxlanModel.VxlanVtepsModel()
   vtepModel.detailIs( 'detail' in args )
   tunnelTypeFilter = args.get( 'TUNNEL_TYPE' )
   vtiIntfIds = vxHwStatusDir.vxlanHwStatus.keys()
   tunnelTypes = defaultdict( VxlanModel.TunnelTypeList )
   allVteps = set()
   for tunnelType, vtepStatus in vtepHwStatus.vtepStatus.items():
      if ( tunnelTypeFilter and
           VxlanCliLib.tunnelTypeToString( tunnelType ) != tunnelTypeFilter ):
         continue
      allVteps |= set( vtepStatus.vtepList )
      tunnelTypeStr = VxlanCliLib.tunnelTypeToString( tunnelType )
      for vtep in vtepStatus.vtepList:
         tunnelType = VxlanModel.TunnelType( tunnelType=tunnelTypeStr )
         tunnelTypes[ vtep ].tunnelTypes.append( tunnelType )
   vtepModel.vtepTunnelTypes = dict( tunnelTypes )
   sortedVteps = sorted( allVteps, key=lambda x: x.sortKey )
   for vtiIntfId in vtiIntfIds:
      status = vxHwStatusDir.vxlanHwStatus.get( vtiIntfId, None )
      if not status:
         continue
      vi = VxlanModel.VxlanVtepsModel.VxlanInterface( vteps=sortedVteps )
      vtepModel.interfaces[ vtiIntfId ] = vi
   if vtepModel.detail():
      for vtep, config in remoteVtepHwConfig.vtepConfig.items():
         # It's possible for a VTEP to have been classified before a tunnel is
         # established, and in some cases, a tunnel will never be established
         # (eg. local VTEP) - don't include detail info for such VTEPs
         if vtep not in allVteps:
            continue
         learnedVia = ( 'dataPlane' if config.vtepType == VtepType.dataPlaneVtep
                        else 'controlPlane' )
         macAddressLearning = ( 'datapath' if config.learningEnabled
                                else 'controlPlane' )
         remoteVtepConfig = VxlanModel.VxlanVtepsModel.RemoteVtepConfig(
            learnedVia=learnedVia, macAddressLearning=macAddressLearning )
         vtepModel.vteps[ vtep.stringValue ] = remoteVtepConfig
   return vtepModel

def showVxlanVniSummary( mode, args ):
   model = VxlanModel.VxlanVniSummaryModel()
   if not vtiStatusDir or not vtiStatusDir.vtiStatus:
      # early exit if vtiStatus is empty
      return model
   for vtiStatus in vtiStatusDir.vtiStatus.itervalues():
      for vniSourcePair in vtiStatus.vlanToVniMap.itervalues():
         source = vniSourcePair.source or 'static'
         model.vxlanVniSources[ source ] = \
            model.vxlanVniSources.get( source, 0 ) + 1
   return model

def showVxlanVtepsSummary( mode, args ):
   model = VxlanModel.VxlanVtepsSummaryModel()
   allVteps = set()
   for vtepStatus in vtepHwStatus.vtepStatus.values():
      allVteps |= set( vtepStatus.vtepList )
   vtiIntfIds = vxHwStatusDir.vxlanHwStatus.keys()
   for vtiIntfId in vtiIntfIds:
      status = vxHwStatusDir.vxlanHwStatus.get( vtiIntfId, None )
      if not status:
         continue
      model.vxlanInterfaces[ vtiIntfId ] = len( allVteps )
   return model

#-------------------------------------------------------------------------------
# The "show vxlan vni" command, in "enable" mode.
#
# The full syntax for this command is:
#   show vxlan vni [<vni>] [dot1q <vlan_id>] [interface <interface>]
#                      [source <source>]
#-------------------------------------------------------------------------------
def showVxlanVni( mode, args ):
   filters = {}
   vniFilter = args.get( 'VNI' )
   if 'INTF' in args:
      filters[ 'intfs' ] = args[ 'INTF' ][ 0 ]
   if 'SOURCE' in args:
      filters[ 'source' ] = args[ 'SOURCE' ][ 0 ]
   if 'dot1q' in args:
      if 'VLANS' in args:
         filters[ 'dot1q' ] = args[ 'VLANS' ][ 0 ]
      else:
         filters[ 'dot1q' ] = 'untagged'

   if vniFilter is not None:
      vni = int( vniFilter )
      if not isValidVniWithError( mode, vni, 'Vxlan1' ):
         return None

   def filterIntfs( intfName, intfs ):
      # filter out MLAG peerlink
      if ( mlagStatus.mlagState != 'disabled' and mlagStatus.peerLinkIntf and
           mlagStatus.peerLinkIntf.intfId == intfName ):
         return False
      if intfs:
         return intfName in intfs
      else:
         # default, match Ethernet and Port-Channel interfaces
         return re.match( '[Ethernet|Port|Vxlan]', intfName )

   def filterVlanTag( vlan, filters ):
      vlanTagFilter = filters.get( 'dot1q' )
      if vlanTagFilter:
         if isinstance( vlanTagFilter, VlanCli.VlanSet ):
            return vlan in vlanTagFilter.ids
         else:
            # check if vlan is untagged
            return vlan == 0
      else:
         return True

   def filterVni( vni ):
      if vniFilter:
         vniValue = VniFormat( vniFilter ).toNum()
         return vni == vniValue
      else:
         return True

   def filterSource( source, filters ):
      # Can this source be allowed to render?
      sourceFilter = filters.get( 'source' )
      if sourceFilter:
         if sourceFilter == 'static':
            # CLI configured VLAN to VNI binding will not have a source.
            # calling CLI configured bindings 'static'
            return not source
         return source == sourceFilter
      else:
         return True

   def checkXlateConsistency( vxStatus, ingressVlan, internalVlan, model ):
      # error check to make sure hwIngressVlanXlate matches
      key = Tac.Value( "Bridging::VlanXlateKey", ingressVlan, 0 )
      ingressXlate = vxStatus.hwIngressVlanXlate.get( key )
      if not ingressXlate or ingressXlate.vlanId != internalVlan:
         model.error = 'ingress and egress VLAN translation do not match'

   def buildVniBindingsToVlan( vtiStatus, mapping ):
      # For a VtiStatus, walk over all VLAN <-> VNI map and add it to
      # result provided the VLAN is not an internal VLAN with an SVI.
      # mapping is VxlanVniVlanCollection CLI model which is the output.
      dynVlanSet = VlanCli.Vlan.getDynVlanSet( mode )
      for vlan, vs in vtiStatus.vlanToVniMap.iteritems():
         # filter out vni and source. If source is evpn, ignore it as
         # it gets handled as part of building VNIs bound to VRF.
         if ( not filterVni( vs.vni ) or
              not filterSource( vs.source, filters ) or
              vs.source == "evpn" ):
            continue
         vId = VlanCli.Vlan( vlan )
         vlanModel = VxlanModel.VxlanVniVlanMappingModel()
         vlanModel.vlan = vlan
         vlanModel.dynamicVlan = vlan in dynVlanSet
         # if source is not defined, assume it is configured by CLI
         vlanModel.source = vs.source or 'static'
         ports = vId.activePorts( mode )
         for intf in ports:
            # filter out interface
            if not filterIntfs( intf.name, intfNames ):
               continue

            intfConfig = bridgingConfig.switchIntfConfig.get( intf.name )
            if intfConfig:
               xlateData = None
               intfModel = VxlanModel.VxlanIntfDot1qModel()
               vxStatus = vlanXlateStatusDir.vlanXlateStatus.get( intf.name )
               if vxStatus:
                  key = Tac.Value( "Bridging::VlanXlateKey", vlan, 0 )
                  xlateData = vxStatus.hwEgressVlanXlate.get( key )
                  # BUG298228, mss uses a different system for vlan xlate.
                  # ignore mss source when checking Xlate Consistency
                  if xlateData and vs.source != 'mss' :
                     checkXlateConsistency( vxStatus, xlateData.vlanId, vlan,
                                            intfModel )
               if 'Vxlan' in intf.name:
                  intfModel.dot1q = vlan
               elif intfConfig.switchportMode == 'trunk':
                  intfModel.dot1q = xlateData.vlanId if xlateData else vlan
               elif intfConfig.switchportMode == 'access':
                  intfModel.dot1q = xlateData.vlanId if xlateData else 0
               elif intfConfig.switchportMode == 'dot1qTunnel':
                  intfModel.dot1q = 0
                  intfModel.dot1qTunnel = True
               else:
                  # not sure what to do with other switchport modes
                  # excluding them for now
                  continue

               # filter out dot1q tag
               if filterVlanTag( intfModel.dot1q, filters ):
                  vlanModel.interfaces[ intf.name ] = intfModel

         if ( ( ports and not vlanModel.interfaces ) or
              ( not ports and ( filters.has_key( 'intfs' ) or
                                filters.has_key( 'dot1q' ) ) ) ):
            # no interfaces found after filter. do not add this entry
            continue

         mapping.vniBindings[ long( vs.vni ) ] = vlanModel

   def buildVniBindingsToVrf( vtiStatus, mapping ):
      # For the VtiStatus, provide the mapping's vniBindingsToVrf
      # collection which contains both local and remote VNIs.
      if not filterSource( "evpn", filters ):
         t0( "evpn source didn't pass the filters", filters )
         return
      if filters.has_key( 'intfs' ) or filters.has_key( 'dot1q' ):
         t0( "intfs and dot1q filters render empty vniBindingsToVrf" )
         return
      # Cache the reverse map of SBD-VLAN -> VRF Name.
      sbdVlanToVrfMap = defaultdict( list )
      for ( vrfName, sbdVlan )  in vtiStatus.vrfToSBDVlan.iteritems():
         sbdVlanToVrfMap[ sbdVlan ].append( vrfName )

      def getVrfForSbdVlan( sbdVlan ):
         vrfList = sbdVlanToVrfMap.get( vlan, [] )
         return vrfList[ 0 ] if len( vrfList ) == 1 else None

      for vlan, vs in vtiStatus.vlanToVniMap.iteritems():
         # Only "evpn" source is allowed for these VNI bindings.
         if vs.source != "evpn":
            continue
         # Apply VNI and source filter if available.
         if not filterVni( vs.vni ):
            continue
         vrfModel = VxlanModel.VxlanVniVrfMappingModel()
         vrfModel.source = vs.source
         vrfModel.vlan = vlan
         vrfName = getVrfForSbdVlan( vlan )
         if vrfName is None:
            t0( "vniSource", vs, "vlan", vlan, "contains invalid mapping to VRFs",
                sbdVlanToVrfMap.get( vlan ) )
         else:
            vrfModel.vrfName = vrfName
         mapping.vniBindingsToVrf[ long( vs.vni ) ] = vrfModel
      # Now build the map using RemoteVniToVlanStatus. Note that the
      # source is still "evpn" for filter purpose. We list it as evpn
      # multicast for clarity of the provisioning model.
      source = "evpn multicast"
      for vniSbdVlan in remoteVniToVlanStatus.remoteVniToVlan.values():
         # Apply VNI and source filter if available.
         if not filterVni( vniSbdVlan.remoteVni ):
            continue
         vrfModel = VxlanModel.VxlanVniVrfMappingModel()
         vrfModel.source = source
         vlan = vniSbdVlan.vlan
         vrfModel.vlan = vlan
         vrfName = getVrfForSbdVlan( vlan )
         if vrfName is None:
            t0( "Remote VNI, SBD VLAN", vniSbdVlan,
                "contains invalid mapping to VRFs", sbdVlanToVrfMap.get( vlan ) )
         else:
            vrfModel.vrfName = vrfName
         if long( vniSbdVlan.remoteVni ) not in mapping.vniBindingsToVrf:
            mapping.vniBindingsToVrf[ long( vniSbdVlan.remoteVni ) ] = vrfModel
         else:
            t0( "There appears to be a local configuration for", vniSbdVlan )

   intfNames = []
   if isinstance( filters.get( 'intfs' ), IntfRange.IntfList ):
      intfNames = list( filters[ 'intfs' ].intfNames() )

   # build vlan to vni mapping
   model = VxlanModel.VxlanVniModel()
   # pylint: disable-msg=W0212
   model._vniInDottedNotation = vxlanControllerConfig.vniInDottedNotation
   if not vtiStatusDir or not vtiStatusDir.vtiStatus:
      # early exit if vtiStatus is empty
      return model

   for vxlanIntfName, vtiStatus in vtiStatusDir.vtiStatus.iteritems():
      mapping = VxlanModel.VxlanVniVlanCollection()
      buildVniBindingsToVlan( vtiStatus, mapping )
      buildVniBindingsToVrf( vtiStatus, mapping )
      model.vxlanIntfs[ vxlanIntfName ] = mapping

   return model

#-------------------------------------------------------------------------------
# VxlanStaticMacCleaner is used to remove the static macs associated with vti
# when the vti is deleted
#-------------------------------------------------------------------------------
class VxlanStaticMacCleaner( IntfCli.IntfDependentBase ):
   #----------------------------------------------------------------------------
   # Destroys the vxlan static mac 
   #----------------------------------------------------------------------------
   def setDefault( self ):
      intfName = self.intf_.name
      configuredHost = vxlanConfigDir.fdbConfig.configuredHost
      hostsInVti = [ host for host in configuredHost.itervalues()
                     if host.vtiIntfId == intfName ]
      for host in hostsInVti:
         del configuredHost[ host.macVlanPair ]

def _supportedCommands( cmds ):
   if bridgingHwCapabilities.vxlanSupported and vxlanConfigDir.vxlanConfig:
      return cmds
   else:
      return []

#-------------------------------------------------------------------------------
# The "show vxlan config-sanity" command, in "enable" mode.
#
# The full syntax of this command is:
#   show vxlan config-sanity [ brief | detail ]
#
# Refer to ConfigCheckItem, ConfigCheckCategory and ConfigSanityModel CAPI
# models. A VxlanConfigSanity check function should perform a single,
# focused check on a certain config or function related to VXLAN, and return 
# a single, or a list of ConfigCheckItem (i.e. an example of a list of items
# would be VLAN-VNI mappings, where each VLAN-VNI map takes an entry). A new 
# check function should be added to "vxlanConfigChecks" dictionary defined
# below.
#
# The actual output rendered on the CLI or returned via CAPI is built using
# this "vxlanConfigChecks" collection, together with logic found under
# buildCommonConfigSanityModel(). CLI rendering is performed by
# ConfigSanityModel.render() method
#
# Please use decorators found in VxlanConfigSanity.py to wrap around your
# check functions. This is mainly to prevent code duplication.
#-------------------------------------------------------------------------------

def getVxlanVlans( excludedSource='evpn' ):
   vxlanVlans = set()
   vtiStatus = getVtiStatus( 'Vxlan1' )
   vnis = vniToVlanMap.getKeys( 'Vxlan1' )
   for vni in vnis:
      if vni:
         vlan = vniToVlanMap.getVlan( vni, 'Vxlan1' )
         if vlan not in vtiStatus.vlanToVniMap:
            continue
         if vtiStatus.vlanToVniMap[ vlan ].source == excludedSource:
            continue
         vxlanVlans.add( vniToVlanMap.getVlan( vni, 'Vxlan1' ) )
   return vxlanVlans

def isL2EvpnConfigured():
   if 'evpn' in vxlanClientDir:
      return vxlanClientDir[ 'evpn' ].enabled
   return False

def isL3EvpnConfigured():
   return bool( vxlanEvpnDynamicVlans.vlans )

def isEvpnConfigured():
   return isL2EvpnConfigured() or isL3EvpnConfigured()

@ConfigCheckItem( 'Loopback IP Address', 1 )
def checkLocalVtepConfigured( item ):
   vtiConfig = getVtiConfig( 'Vxlan1', create=False )
   vtiStatus = getVtiStatus( 'Vxlan1' )

   if not vtiConfig:
      return item # no VTI configured

   if not vtiStatus:
      # probably will never hit this, but need to probe vtiStatus to get
      # localVtepAddr
      item.detail = 'VtiStatus does not exist!'
      item.checkPass = False
   elif not vtiConfig.srcIpIntf:
      item.detail = 'Source interface not configured'
      item.checkPass = False
   elif vtiStatus.localVtepAddr in [ '0.0.0.0', '255.255.255.255' ] and \
        vtiStatus.localVtepAddr6.isUnspecified:
      item.detail = 'Invalid IP %s %s' % (
            vtiStatus.localVtepAddr, vtiStatus.localVtepAddr6 )
      item.checkPass = False

   return item

# get errors from resolving VLAN-VNI Mappings
def getDynVlanVniErrors( vccMode ):
   vlanErrors = set()
   vniErrors = set()

   if not vccMode:
      return vlanErrors, vniErrors

   for vtepSrc in controllerErrorStatus.iterkeys():
      errorStatus = controllerErrorStatus.get( vtepSrc )
      if not errorStatus:
         return None
      for port in errorStatus.portError.iterkeys():
         portError = errorStatus.portError.get( port )
         if not portError:
            continue
         for vlan in portError.vlanVniError:
            vniOrNone = portError.vlanVniError.get( vlan )
            vlanErrors.add( vlan )
            if vniOrNone:
               vniErrors.add( vniOrNone )

   return vlanErrors, vniErrors

@GenConfigCheckItems( 'VLAN-VNI Map', 2 )
def checkVlanVniMap( itemNameBase ):
   items = []
   vtiStatus = getVtiStatus( 'Vxlan1' )

   if not vtiStatus:
      return items

   vnis = vniToVlanMap.getKeys( 'Vxlan1' )

   # display an error if no VLAN-VNI map is found
   if not vnis:
      item = VxlanModel.ConfigCheckItem( name=itemNameBase,
               checkPass=False, detail='No VLAN-VNI mapping' )
      items.append( item )
   else: # otherwise check each mappings
      vlanErrors, vniErrors = getDynVlanVniErrors(
                                       vtiStatus.controllerClientMode )
      vlanVniMap = {}
      for vni in vnis:
         vlan = vniToVlanMap.getVlan( vni, 'Vxlan1' )
         vlanVniMap[ vlan ] = vni

      for vlan, vni in sorted( vlanVniMap.iteritems() ):
         if vlan and vlan not in bridgingConfig.vlanConfig:
            items.append( VxlanModel.ConfigCheckItem(
                              name=itemNameBase,
                              hasWarning=True,
                              detail=vlanNotCreatedFmt % vlan ) )

         item = VxlanModel.ConfigCheckItem()
         item.name = itemNameBase
         item.checkPass = True

         # check if dynamic errors exist
         if vlan in vlanErrors:
            item.detail = dynVlanVniConflictFmt % ( 'VLAN', vlan )
            item.hasWarning = True
         elif vni in vniErrors:
            item.detail = dynVlanVniConflictFmt % ( 'VNI', vni )
            item.hasWarning = True

         # do not show all VLAN-VNI mappings as this information can be retrieved
         # using other commands, only show the ones with dynamic vlan-vni conflicts
         if item.hasWarning:
            items.append( item )

   # if no warnings has been added here, everything is fine
   if not items:
      item = VxlanModel.ConfigCheckItem( name=itemNameBase, checkPass=True )
      items.append( item )

   return items

@GenConfigCheckItems( 'Flood List', 3 )
def checkFloodListConfigured( itemNameBase ):
   vxlanConfig = getVxlanConfig( 'Vxlan1' )
   vtiStatus = getVtiStatus( 'Vxlan1' )

   items = []
   if not vxlanConfig or not vtiStatus:
      return items

   vxlanVlans = getVxlanVlans()

   # mvp stands for MacVlanPair
   validFloodlistVlans = set()
   floodlistConfigured = False
   for mvp in vxHwStatusDir.bumStatus.bumVtepList:
      floodlistConfigured = True
      if mvp.vlanId not in vxlanVlans:
         # don't care
         continue
      bumVtepList = vxHwStatusDir.bumStatus.bumVtepList[ mvp ]
      for addr in bumVtepList.remoteVtepAddr:
         if addr not in ( vtiStatus.localVtepAddr,
                          vtiStatus.vArpVtepAddr,
                          vtiStatus.mlagVtepAddr ):
            validFloodlistVlans.add( mvp.vlanId )
            break
      for addr in bumVtepList.remoteVtepAddr6:
         if addr not in ( vtiStatus.localVtepAddr6,
                          vtiStatus.vArpVtepAddr6,
                          vtiStatus.mlagVtepAddr6 ):
            validFloodlistVlans.add( mvp.vlanId )
            break

   itemPass = vtiStatus.controllerClientMode
   fromCvx = ' from CVX' if vtiStatus.controllerClientMode else ''
   # checks before checking each VXLAN VLANs
   if not vxlanVlans:
      if not isEvpnConfigured():
         item = VxlanModel.ConfigCheckItem(
                  name=itemNameBase,
                  checkPass=itemPass, hasWarning=True,
                  detail='No VXLAN VLANs' + fromCvx )
         items.append( item )
      return items
   elif not floodlistConfigured:
      if not isEvpnConfigured():
         item = VxlanModel.ConfigCheckItem(
                  name=itemNameBase,
                  checkPass=itemPass, hasWarning=True,
                  detail='No flood list configured' + fromCvx )
         items.append( item )
      return items

   if isL2EvpnConfigured():
      return items

   # check each VXLAN-VLANs has proper remote VTEP in flood list
   for vlan in vxlanVlans:
      if vlan not in validFloodlistVlans:
         item = VxlanModel.ConfigCheckItem(
                  name=itemNameBase,
                  checkPass=itemPass, hasWarning=True,
                  detail=noRemoteVtepVlanFloodlistFmt % vlan )
         items.append( item )

   if not items:
      item = VxlanModel.ConfigCheckItem(
               name=itemNameBase,
               checkPass=True, hasWarning=False,
               detail='' )
      items.append( item )

   return items

def isSviVirtual( vlanId ):
   vlanStr = 'Vlan%d' % vlanId

   # status updated with 'ip address virtual'
   ipIntfStatus = ipStatus.ipIntfStatus.get( vlanStr )
   if ipIntfStatus and ipIntfStatus.useVirtualAddr:
      return True

   # status updated with 'ip virtual-router address'
   if vlanStr in fhrpStatus.vrIntfStatus:
      return True

   return False

# check if any Vxlan SVIs are using virtual IP
# 1. SVI configured with ip addr and "ip virtual-router"
# 2. SVI configured with "ip address virtual"
def virtualIpInSvis():
   vxlanVlans = getVxlanVlans()

   for vlan in vxlanVlans:
      if isSviVirtual( vlan ):
         return True

   return False

@GenConfigCheckItems( 'Routing', 4 )
def checkVirtualRouting( itemNameBase ):
   vxlanConfig = getVxlanConfig( 'Vxlan1' )
   vtiStatus = getVtiStatus( 'Vxlan1' )

   items = []
   if not vxlanConfig or not vtiStatus:
      return items
   vccMode = vtiStatus.controllerClientMode
   if virtualIpInSvis():
      # check for VARP MAC
      if vrMacStatus.varpVirtualMac == ethAddrZero:
         items.append( VxlanModel.ConfigCheckItem(
                           name=itemNameBase, checkPass=vccMode, hasWarning=True,
                           detail='Virtual MAC is not configured' ) )
      # check for VTEP IP
      if vtiStatus.vArpVtepAddr == ipAddrZero and vtiStatus.vArpVtepAddr6.isZero:
         if not isEvpnConfigured():
            items.append( VxlanModel.ConfigCheckItem(
                          name=itemNameBase, checkPass=vccMode, hasWarning=True,
                          detail='Virtual VTEP IP is not configured' ) )

   # if no warnings up to this point, all checks passed
   if not items:
      item = VxlanModel.ConfigCheckItem(
               name=itemNameBase,
               checkPass=True, hasWarning=False,
               detail='' )
      items.append( item )
   return items

@GenConfigCheckItems( 'VNI VRF ACL', 5 )
def checkVniVrfAcl( itemNameBase ):
   items = []
   vxlanConfig = getVxlanConfig( 'Vxlan1' )
   vtiStatus = getVtiStatus( 'Vxlan1' )

   if not vxlanConfig or not vtiStatus:
      return items

   vnis = vxlanConfig.vniToIpAclMap.keys()
   vrfToVniMap = vtiStatus.vrfToVniMap

   if not vnis:
      # if no ingress VNI ACLs configured, everything is fine
      pass
   elif not vrfToVniMap:
      # display an error if no VRF-VNI map is found
      item = VxlanModel.ConfigCheckItem( name=itemNameBase,
               checkPass=False, detail='No VRF-VNI mapping' )
      items.append( item )
   else:
      # otherwise check each mappings
      for vni in vnis:
         if vni in vrfToVniMap.values():
            continue
         items.append( VxlanModel.ConfigCheckItem(
                           name=itemNameBase,
                           checkPass=False,
                           detail=noVniInVrfToVniFormat % vni ) )

   # if no warnings has been added here, everything is fine
   if not items:
      item = VxlanModel.ConfigCheckItem( name=itemNameBase, checkPass=True )
      items.append( item )

   return items

@GenConfigCheckItems( 'Decap VRF-VNI Map', 6 )
def checkDecapVniToVrfMap( itemNameBase ):
   items = []
   vtiConfig = getVtiConfig( 'Vxlan1' )
   vtiStatus = getVtiStatus( 'Vxlan1' )

   if not vtiConfig or not vtiStatus:
      return items

   decapVrfNames = set( vtiConfig.alternateDecapVniToVrfMap.values() )
   vrfToVniMap = vtiStatus.vrfToVniMap

   if not decapVrfNames:
      # if no alternate decap VNI-to-VRF map configured, everything is fine
      pass
   elif not vrfToVniMap:
      # display an error if no VRF-VNI map is found
      item = VxlanModel.ConfigCheckItem( name=itemNameBase,
               checkPass=False, detail='No VRF-VNI mapping' )
      items.append( item )
   else:
      # otherwise check each mappings
      for vrfName in decapVrfNames:
         if vrfName in vrfToVniMap:
            continue
         items.append( VxlanModel.ConfigCheckItem(
                           name=itemNameBase,
                           checkPass=False,
                           detail=noVrfInVrfToVniFormat % vrfName ) )

   # if no warnings has been added here, everything is fine
   if not items:
      item = VxlanModel.ConfigCheckItem( name=itemNameBase, checkPass=True )
      items.append( item )

   return items

def arpResolved( vrfId, intfId, destIp ):
   ipGenAddr = Arnet.IpGenAddr( str( destIp ) )
   arpKey = Tac.newInstance( 'Arp::Table::ArpKey', vrfId, ipGenAddr, intfId )
   if ipGenAddr.af == 'ipv4':
      return arpKey in arpTableStatus.arpEntry
   else:
      return arpKey in arpTableStatus.neighborEntry

def blackholeVia( via ):
   return via.hop == ipAddrZero and via.intfId == ''

# returns an ConfigCheckItem containing results and descriptions from checking
# route and arp to ipAddr
def ipRouteCheckHelper( vrf, ipAddr, itemName ):
   ipAddr = str( ipAddr )
   item = VxlanModel.ConfigCheckItem( name=itemName )
   ipGenPrefix = Arnet.IpGenPrefix( ipAddr )
   route, vias = routeTrie.getRoute( ipGenPrefix )

   if not vias or not route:
      item.detail = 'No route to ' + ipAddr
      item.checkPass = False
      return item

   nexthopArpResolved = False
   # handle ECMP case
   for via in vias:
      lookupIp = ipAddr if Arnet.IpGenAddr( str( via.hop ) ).isAddrZero else via.hop
      if blackholeVia( via ):
         continue
      elif arpResolved( vrf, via.intfId, lookupIp ):
         nexthopArpResolved = True
         break

   if not nexthopArpResolved:
      item.detail = 'Unresolved ARPs to ' + ipAddr
      item.checkPass = False

   return item

@GenConfigCheckItems( 'Remote VTEP', 1 )
def checkRemoteVtepRoute( itemNameBase ):
   vtiStatus = getVtiStatus( 'Vxlan1' )
   if not vtiStatus:
      return []

   floodVtepsToCheck = []
   bumVtepList = vxHwStatusDir.bumStatus.bumVtepList
   for mvp in bumVtepList:
      floodVtepsToCheck += bumVtepList[ mvp ].remoteVtepAddr
      floodVtepsToCheck += bumVtepList[ mvp ].remoteVtepAddr6

   items = []
   # let's skip localVtep and vArpVtep
   for vtep in floodVtepsToCheck:
      # we don't have to check configured VTEP addresses of this switch
      # including the local VTEP address and VARP VTEP address
      if vtep in ( vtiStatus.localVtepAddr,
                   vtiStatus.vArpVtepAddr,
                   vtiStatus.mlagVtepAddr,
                   vtiStatus.localVtepAddr6,
                   vtiStatus.vArpVtepAddr6,
                   vtiStatus.mlagVtepAddr6 ):
         continue

      # underlay reachability is currently only in vrf 0 ( default vrf )
      item = ipRouteCheckHelper( underlayDefaultVrfId, vtep, itemNameBase )
      if not item.checkPass:
         items.append( item )

   if not items:
      items.append( VxlanModel.ConfigCheckItem( name=itemNameBase ) )

   return items

def vlanIpAddrSet( vlanId ):
   vlanStr = 'Vlan%d' % vlanId

   # this is how Ira shows interface-level IP addresses
   # first go through status, then config
   if vlanStr in ipStatus.ipIntfStatus:
      if ipStatus.ipIntfStatus[ vlanStr ].activeAddrWithMask != zeroAddrWithMask:
         return True

   if vlanStr in ipConfig.ipIntfConfig:
      if ipConfig.ipIntfConfig[ vlanStr ].addrWithMask != zeroAddrWithMask:
         return True

   return False

def vxlanRoutingEnabled():
   vxlanVlan = getVxlanVlans()

   if not vxlanVlan:
      return False

   for vlan in vxlanVlan:
      if vlanIpAddrSet( vlan ) or isSviVirtual( vlan ):
         return True

   return False

def registerVxlanBridgingConfigCheckCallback( callback ):
   # platform-specific code should register a callback here
   if callback:
      vxlanBridgingConfigCheckCallback.append( callback )

def registerVxlanRoutingConfigCheckCallback( callback ):
   # platform-specific code should register a callback here
   if callback:
      vxlanRoutingConfigCheckCallback.append( callback )

@GenConfigCheckItems( 'VXLAN Bridging', 1 )
def checkVxlanBridging( itemNameBase ):
   vtiStatus = getVtiStatus( 'Vxlan1' )

   items = []
   if not vtiStatus:
      return [ VxlanModel.ConfigCheckItem( name=itemNameBase ) ]

   if not bridgingHwCapabilities.vxlanSupported:
      items.append( VxlanModel.ConfigCheckItem( name=itemNameBase,
                              detail='VXLAN Bridging not supported' ) )
      return items

   # call callback functions registered by platform-specific Cli plugins
   for checkCallback in vxlanBridgingConfigCheckCallback:
      checkCallback( itemNameBase, items )

   if not items:
      items.append( VxlanModel.ConfigCheckItem( name=itemNameBase ) )

   return items

@GenConfigCheckItems( 'VXLAN Routing', 1 )
def checkVxlanRouting( itemNameBase ):
   vtiStatus = getVtiStatus( 'Vxlan1' )

   items = []
   if not vtiStatus:
      return [ VxlanModel.ConfigCheckItem( name=itemNameBase ) ]

   if not bridgingHwCapabilities.vxlanRoutingSupported:
      items.append( VxlanModel.ConfigCheckItem( name=itemNameBase,
                              detail='VXLAN Routing not supported' ) )
      return items

   if not vxlanRoutingEnabled():
      items.append( VxlanModel.ConfigCheckItem( name=itemNameBase,
                              detail='VXLAN Routing not enabled' ) )
      return items

   # call callback functions registered by platform-specific Cli plugins
   for checkCallback in vxlanRoutingConfigCheckCallback:
      checkCallback( itemNameBase, items )

   if not items:
      items.append( VxlanModel.ConfigCheckItem( name=itemNameBase ) )

   return items

@ConfigCheckItem( 'CVX Server', 1 )
def cvxServerHostCheck( item ):
   vtiStatus = getVtiStatus( 'Vxlan1' )
   if not vtiStatus:
      return item

   if not vtiStatus.controllerClientMode:
      item.detail = 'Not in controller client mode'
      return item

   leaderController = None
   for controller in cvxClientStatus.controllerStatus:
      if cvxClientStatus.controllerStatus[ controller ].leader:
         leaderController = controller
         break

   if not leaderController:
      item.checkPass = False
      item.detail = 'No leader Found'
      return item

   # underlay reachability is currently only in vrf 0 ( default vrf )
   return ipRouteCheckHelper( underlayDefaultVrfId, leaderController, item.name )

#-------------------------------------------------------------------------------
#
# VXLAN MLAG check ( including checking for multi-VTEP MLAG configs )
#
# We check for default VTEP address and MLAG VTEP address for Vxlan1 interface
# on local switch and peer switch. We have the following 4 scenarios, depending
# on what's configured locally and on the peer
#
# 1. Local ( default ), peer ( default )
#   * check that local_default == peer_default
#
# 2. Local ( default, mlag ), peer ( default, mlag )
#   * check that local_mlag == peer_mlag
#   * check that local_default != peer_default
#
# 3. Local ( default ), peer ( default, mlag )
#   * check that local_default in [ peer_default, peer_mlag ]
#
# 4. Local ( default, mlag ), peer ( default )
#   * this is inverse of case 3
#
# Note that we can keep config-sanity logic simple by show errors on only one of
# the switches invovled in MLAG for case 3&4 ( the one with MLAG VTEP IP, since
# the one without MLAG VTEP IP can simply skip checking for MLAG VTEP IP )
#
#-------------------------------------------------------------------------------

def mlagConnected():
   return mlagStatus.mlagState in [ 'primary', 'secondary' ] \
          and mlagStatus.negotiationStatus == 'Connected'

def localMlagVtepIpConfigured():
   vtiStatus = getVtiStatus( 'Vxlan1' )
   if not vtiStatus:
      return False
   return vtiStatus.mlagVtepAddr not in [ '0.0.0.0', '255.255.255.255' ] or \
      not vtiStatus.mlagVtepAddr6.isZero

def getPeerVtepAddrByType( vtepType ):
   return mlagVxlanStatus.vtepTypeAddr.get( vtepType, None )

def getPeerVtepAddr6ByType( vtepType ):
   return mlagVxlanStatus.vtepTypeAddr6.get( vtepType, None )

def isNullAddr( addr ):
   return addr is None or Arnet.IpGenAddr( str( addr ) ).isAddrZero

def validateAddr( addr, validAddrList ):
   'given a list of valid'
   if all( isNullAddr( a ) for a in validAddrList ) :
      return isNullAddr( addr )
   return addr in validAddrList

@ConfigCheckItem( 'Peer VTEP IP', 1 )
def peerVtepIpCheck( item ):
   vtiStatus = getVtiStatus( 'Vxlan1' )
   if not vtiStatus:
      return item

   if not mlagConnected():
      item.detail = 'MLAG peer is not connected'
      return item

   peerVtepAddr = getPeerVtepAddrByType( 'defaultVtep' )
   peerMlagVtepAddr = getPeerVtepAddrByType( 'mlagVtep' )
   peerVtepAddr6 = getPeerVtepAddr6ByType( 'defaultVtep' )
   peerMlagVtepAddr6 = getPeerVtepAddr6ByType( 'mlagVtep' )
   if not peerVtepAddr and not peerVtepAddr6:
      item.detail = 'No VTEP IP from peer'
      item.checkPass = False
      return item

   if localMlagVtepIpConfigured():
      if peerMlagVtepAddr or peerMlagVtepAddr6:
         # when MLAG VTEP IP is configured, there needs to be three distinct VTEP
         # IPs on the MLAG: the MLAG VTEP IP and one VTEP IP from each VTEP. 
         if vtiStatus.localVtepAddr == peerVtepAddr or \
            vtiStatus.localVtepAddr6 == peerVtepAddr6:
            item.detail = 'Peer has same VTEP IP'
            item.checkPass = False
      else:
         # peer has no MLAG VTEP IP configured
         if not ( validateAddr( peerVtepAddr,
                                [ vtiStatus.localVtepAddr,
                                  vtiStatus.mlagVtepAddr ] ) and
                  validateAddr( peerVtepAddr6,
                                [ vtiStatus.localVtepAddr6,
                                  vtiStatus.mlagVtepAddr6 ] ) ):
            item.detail = 'Invalid peer VTEP IP'
            item.checkPass = False
   else:
      # when no MLAG VTEP IP is configured, local VTEP IP need to be the same
      # as peer VTEP IP or peer MLAG IP
      if not ( validateAddr( vtiStatus.localVtepAddr,
                             [ peerVtepAddr, peerMlagVtepAddr ] ) and
               validateAddr( vtiStatus.localVtepAddr6,
                             [ peerVtepAddr6, peerMlagVtepAddr6 ] ) ):
         item.detail = 'Invalid local VTEP IP'
         item.checkPass = False

   return item

@ConfigCheckItem( 'MLAG VTEP IP', 2 )
def mlagVtepIpCheck( item ):
   vtiStatus = getVtiStatus( 'Vxlan1' )
   if not vtiStatus:
      return item
   if not mlagConnected():
      return item
   if not localMlagVtepIpConfigured():
      return item

   peerVtepAddr = getPeerVtepAddrByType( 'defaultVtep' )
   peerMlagVtepAddr = getPeerVtepAddrByType( 'mlagVtep' )
   peerVtepAddr6 = getPeerVtepAddr6ByType( 'defaultVtep' )
   peerMlagVtepAddr6 = getPeerVtepAddr6ByType( 'mlagVtep' )
   if not peerVtepAddr and not peerVtepAddr6:
      item.detail = 'No VTEP IP from peer'
      item.checkPass = False
      return item

   if not peerMlagVtepAddr and not peerMlagVtepAddr6:
      if not ( validateAddr( peerVtepAddr,
                             [ vtiStatus.localVtepAddr,
                               vtiStatus.mlagVtepAddr ] ) and
               validateAddr( peerVtepAddr6,
                             [ vtiStatus.localVtepAddr6,
                               vtiStatus.mlagVtepAddr6 ] ) ):
         item.detail = 'Invalid peer VTEP IP'
         item.checkPass = False
   elif not ( validateAddr( vtiStatus.mlagVtepAddr, [ peerMlagVtepAddr ] ) and
              validateAddr( vtiStatus.mlagVtepAddr6, [ peerMlagVtepAddr6 ] ) ):
      item.detail = 'Peer has different MLAG VTEP IP'
      item.checkPass = False

   return item

@ConfigCheckItem( 'Peer VLAN-VNI', 3 )
def peerVlanVniCheck( item ):
   vtiConfig = getVtiConfig( 'Vxlan1' )
   vtiStatus = getVtiStatus( 'Vxlan1' )
   if not vtiConfig or not vtiStatus:
      return item
   if not mlagConnected():
      return item

   # vlan-vni mapping gets sent from primary to secondary, if we are primary
   # prompt the user to check output of this command from the secondary
   if mlagStatus.mlagState == 'primary':
      item.detail = 'Check this command from the peer'
      return item

   errorItem = VxlanModel.ConfigCheckItem(
                  name=item.name,
                  checkPass=False, hasWarning=False,
                  detail='VLAN-VNI Mapping not identical' )

   # VtiStatus contains configured and dynamic mapping
   if len( vtiStatus.vlanToVniMap ) != len( mlagVxlanStatus.vniToDynVlanMap ):
      return errorItem

   for vlan, vsp in sorted( vtiStatus.vlanToVniMap.iteritems() ):
      if vsp.vni not in mlagVxlanStatus.vniToDynVlanMap:
         return errorItem
      if vlan != mlagVxlanStatus.vniToDynVlanMap[ vsp.vni ]:
         return errorItem

   return item

@ConfigCheckItem( 'Peer Flood List', 4 )
def peerFloodListCheck( item ):
   vtiStatus = getVtiStatus( 'Vxlan1' )
   if not vtiStatus:
      return item
   if not mlagConnected():
      return item

   # flood-list gets sent from primary to secondary, if we are primary
   # prompt the user to check output of this command from the secondary
   if mlagStatus.mlagState == 'primary':
      item.detail = 'Check this command from the peer'
      return item

   errorItem = VxlanModel.ConfigCheckItem(
                  name=item.name,
                  checkPass=False, hasWarning=False,
                  detail='Flood List not identical' )

   vxlanVlans = getVxlanVlans()

   # mvp stands for MacVlanPair
   for mvp in vxHwStatusDir.bumStatus.bumVtepList:
      if mvp.vlanId not in vxlanVlans:
         # don't care
         continue
      if mvp not in mlagVxlanStatus.bumStatus.bumVtepList:
         return errorItem
      vxList = vxHwStatusDir.bumStatus.bumVtepList[ mvp ]
      mlagList = mlagVxlanStatus.bumStatus.bumVtepList[ mvp ]
      if ( set( vxList.remoteVtepAddr ) != set( mlagList.remoteVtepAddr ) or
           set( vxList.remoteVtepAddr6 ) != set( mlagList.remoteVtepAddr6 ) ):
         return errorItem

   return item

@ConfigCheckItem( 'Virtual VTEP IP', 5 )
def virtualVtepCheck( item ):
   vtiStatus = getVtiStatus( 'Vxlan1' )
   if not vtiStatus:
      return item
   if not mlagConnected():
      return item

   varpVtepAddr = getPeerVtepAddrByType( 'varpVtep' )
   varpVtepAddr6 = getPeerVtepAddr6ByType( 'varpVtep' )
   if not ( validateAddr( vtiStatus.vArpVtepAddr, [ varpVtepAddr ] ) and
            validateAddr( vtiStatus.vArpVtepAddr6, [ varpVtepAddr6 ] ) ):
      return VxlanModel.ConfigCheckItem( name=item.name,
                                         checkPass=False, hasWarning=False,
                                         detail='Peer vVTEP IP does not match' )

   return item

# collection to hold all config-sanity checks
vxlanConfigChecks = {
   'localVtep' : {
      'priority' : 1,
      "description" : "Local VTEP Configuration Check",
      "checks" : [
         checkLocalVtepConfigured,
         checkVlanVniMap,
         checkFloodListConfigured,
         checkVirtualRouting,
         checkVniVrfAcl,
      ] 
   },
   'remoteVtep' : {
      'priority' : 2,
      "description" : "Remote VTEP Configuration Check",
      "checks" : [
         checkRemoteVtepRoute,
      ] 
   },
   'pd' : {
      'priority' : 3,
      "description" : "Platform Dependent Check",
      "checks" : [
         checkVxlanBridging,
         checkVxlanRouting,
      ] 
   },
   'cvx' : {
      'priority' : 4,
      "description" : "CVX Configuration Check",
      "checks" : [
         cvxServerHostCheck,
      ] 
   },
   'mlag' : {
      'priority' : 5,
      "description" : "MLAG Configuration Check",
      "checks" : [
         peerVtepIpCheck,
         mlagVtepIpCheck,
         virtualVtepCheck,
         peerVlanVniCheck,
         peerFloodListCheck,
      ] 
   }
}

if Toggles.VxlanToggleLib.toggleVrfToMultipleVnisMappingEnabled():
   vxlanConfigChecks[ 'localVtep' ][ 'checks' ].append( checkDecapVniToVrfMap )

def checkDynVlanVniWarning( configSanityModel ):
   category = configSanityModel.categories.get( 'localVtep', None )
   if category:
      for item in category.items:
         if item.name.startswith( 'VLAN-VNI Map' ) and item.hasWarning:
            return True
   return False

def checkWarnings( configSanityModel, cliMode ):
   # check presence of any warning
   hasWarning = any( c.hasWarning for \
                     c in configSanityModel.categories.itervalues() )
   if hasWarning:
      cliMode.addWarning( warnExplainedMsg )

   hasDynVlanVniWarning = checkDynVlanVniWarning( configSanityModel )
   if hasDynVlanVniWarning:
      cliMode.addWarning( dynVlanVniWarnMsg )

def saveConfigSanityModelToSysdb( model, mode ):
   # Save config-sanity output to Sysdb so that it can be streamed via TerminAttr
   vxlanFeature = configSanityFeatureDir.newMember( 'vxlan' )
   newSeqNum = 1 if vxlanFeature.seqNum == sys.maxint else vxlanFeature.seqNum + 1

   def capiToConfigSanityResult( checkPass, hasWarning ):
      if not checkPass:
         return 'Fail'
      elif hasWarning:
         return 'Warn'
      return 'Pass'

   itemCache = {}

   # Delete stale entries
   for tag, category in vxlanFeature.categories.iteritems():
      if tag not in model.categories:
         del vxlanFeature.categories[ tag ]
         continue
      categoryItemNames = set( i.name for i in model.categories[ tag ].items )
      for itemName in category.items:
         if itemName not in categoryItemNames:
            del vxlanFeature.categories[ tag ].items[ itemName ]

   for tag, category in model.categories.iteritems():
      vxlanFeature.categories.newMember( tag )
      vxlanFeature.categories[ tag ].description = category.description
      vxlanFeature.categories[ tag ].result = capiToConfigSanityResult(
            category.allCheckPass, category.hasWarning )
      vxlanFeature.categories[ tag ].priority = category.priority()

      for item in category.items:
         result = capiToConfigSanityResult( item.checkPass, item.hasWarning )
         i = Tac.Value( 'ConfigSanity::Item', item.name, item.detail, result )
         itemTag = item.name
         if item.name in itemCache:
            itemCache[ item.name ] += 1
            itemTag += '_' + str( itemCache[ item.name ] )
         else:
            itemCache[ item.name ] = 1
         vxlanFeature.categories[ tag ].items[ itemTag ] = i

   for warning in mode.session.warnings_:
      vxlanFeature.warnings[ warning ] = True

   vxlanFeature.seqNum = newSeqNum

   return model

def showConfigSanity( mode, args ):
   # pylint: disable-msg=W0612
   model = VxlanModel.ConfigSanityModel()
   if 'brief' in args:
      model.detailLevelIs( 'brief' )
   elif 'detail' in args:
      model.detailLevelIs( 'detailed' )

   vtiIntfId = vtiNameToId.get( 'Vxlan1' )
   if not vtiConfigDir.vtiConfig.get( vtiIntfId ):
      mode.addMessage( 'No VXLAN interface' )
      # Write ConfigSanityModel to Sysdb so that CVP can stream state via
      # TerminAttr Prevent write to Sysdb when supervisor is not active
      if em.redundancyStatus().mode == 'active':
         saveConfigSanityModelToSysdb( model, mode )
      return model

   routeTrie.refreshRoutes()
   # perform all config-sanity checks and build model
   for checkTag, check in vxlanConfigChecks.iteritems():
      category = VxlanModel.ConfigCheckCategory()
      category.description = check[ 'description' ]
      for doCheck in check[ 'checks' ]:
         item = doCheck()
         # there are 2 types of sanity check functions, one type returns a list
         # while the other type returns a single CheckItem
         if isinstance( item, list ):
            category.items.extend( item )
         else:
            category.items.append( item )
      category.allCheckPass = all( i.checkPass for i in category.items )
      category.hasWarning = any( i.hasWarning for i in category.items )
      category.priorityIs( check.get( 'priority', 200 ) )
      model.categories[ checkTag ] = category

   checkWarnings( model, mode )

   # Write ConfigSanityModel to Sysdb so that CVP can stream state via TerminAttr
   # Prevent write to Sysdb when supervisor is not active
   if em.redundancyStatus().mode == 'active':
      saveConfigSanityModelToSysdb( model, mode )
   return model

#-------------------------------------------------------------------------------
# end of "show vxlan config-sanity" command section
#-------------------------------------------------------------------------------

def Plugin( entityManager ):
   global configSanityFeatureDir
   global vxlanConfigDir, vxHwStatusDir, vtiConfigDir, vtiStatusDir
   global vxlanCounterConfigDir
   global bridgingHwCapabilities, bridgingConfig, smashBridgingStatus
   global arpTableStatus
   global fhrpStatus, vrMacStatus
   global bridgingCliConfig, vxAgentCounter
   global vxlanVniStatusDir
   global vxlanVniFdbStatusDir
   global serviceConfigDir
   global vxlanControllerConfig
   global controllerErrorStatus
   global vniToVlanMap
   global mlagStatus
   global mlagHostTable
   global mlagVxlanStatus
   global em
   global vxlanStatusDir
   global vcsStateClientView
   global lRStatus
   global cvxClientStatus
   global hwCounterFeatureStatusDir
   global vlanXlateStatusDir
   global vxlanHwCounter
   global ipConfig, ipStatus
   global ipStatus
   global vxlanClientConvergenceStatus
   global vxlanClientDir
   global vxlanEvpnDynamicVlans
   global remoteVniToVlanStatus
   global remoteVtepHwConfig
   global ipsecConfig
   global vtepHwStatus

   readerInfo = SmashLazyMount.mountInfo( 'reader' )

   configSanityFeatureDir = LazyMount.mount( entityManager, 'configsanity/feature',
                                             'ConfigSanity::FeatureDir', 'w' )
   vxAgentCounter = SmashLazyMount.mount( entityManager, "vxlan/counter",
                                          "Vxlan::VxlanAgentCounter",
                                          readerInfo )
   smashBridgingStatus = SmashLazyMount.mount( entityManager, "bridging/status",
                                               "Smash::Bridging::Status",
                                               readerInfo )
   def createRouteTrie():
      global routingStatus, routing6Status, forwardingStatus, forwarding6Status
      global routeTrie
      fecModeStatus = Tac.newInstance( 'Smash::Fib::FecModeStatus', 'fms' )
      _ = Tac.newInstance( 'Ira::FecModeSm', l3Config, fecModeStatus )
      routingStatus = SmashLazyMount.mount( entityManager, "routing/status",
                                            'Smash::Fib::RouteStatus',
                                            readerInfo )
      routing6Status = SmashLazyMount.mount( entityManager, "routing6/status",
                                            'Smash::Fib6::RouteStatus',
                                            readerInfo )
      if fecModeStatus.fecMode == 'fecModeUnified':
         forwardingStatus = SmashLazyMount.mount( entityManager,
                                                  "forwarding/unifiedStatus",
                                                  'Smash::Fib::ForwardingStatus',
                                                  readerInfo )
         forwarding6Status = SmashLazyMount.mount( entityManager,
                                                   "forwarding6/unifiedStatus",
                                                   'Smash::Fib6::ForwardingStatus',
                                                   readerInfo )
      else:
         forwardingStatus = SmashLazyMount.mount( entityManager, "forwarding/status",
                                                  'Smash::Fib::ForwardingStatus',
                                                  readerInfo )
         forwarding6Status = SmashLazyMount.mount( entityManager,
                                                   "forwarding6/status",
                                                   'Smash::Fib6::ForwardingStatus',
                                                   readerInfo )
      routeTrie = RouteTrie( routingStatus, forwardingStatus,
                             routing6Status, forwarding6Status )
   mg = entityManager.mountGroup()
   l3Config = mg.mount( 'l3/config', 'L3::Config', 'ri' )
   mg.close( createRouteTrie )
   arpTableStatus = SmashLazyMount.mount( entityManager, "arp/status",
                                          'Arp::Table::Status',
                                           readerInfo )
   fhrpStatus = LazyMount.mount( entityManager,
                                 'routing/fhrp/status',
                                 'Routing::Fhrp::StatusV4',
                                 'r' )
   vrMacStatus = LazyMount.mount( entityManager,
                                  'routing/fhrp/vrMacStatus',
                                  'Routing::Fhrp::VirtualRouterMacStatus',
                                  'r' )
   vxlanStatusDir = LazyMount.mount( entityManager,
                                     "vxlan/status",
                                     "Vxlan::VxlanStatusDir", "r" )
   vxlanConfigDir = ConfigMount.mount( entityManager,
                                       "vxlan/config",
                                       "Vxlan::VxlanConfigDir", "w" )
   vxlanCounterConfigDir = LazyMount.mount( entityManager,
                                            "vxlan/counterconfig",
                                            "Vxlan::VxlanCounterConfigDir", "w" )
   vxHwStatusDir = LazyMount.mount( entityManager,
                                     "vxlan/hardware/status",
                                     "Vxlan::VxlanHwStatusDir", "r" )
   vtiConfigDir = ConfigMount.mount( entityManager,
                                     "interface/config/eth/vxlan",
                                     "Vxlan::VtiConfigDir", "w" )
   vtiStatusDir = LazyMount.mount( entityManager,
                                   "interface/status/eth/vxlan",
                                   "Vxlan::VtiStatusDir", "r" )
   bridgingHwCapabilities = LazyMount.mount( entityManager,
                                             "bridging/hwcapabilities",
                                             "Bridging::HwCapabilities", "r" )
   bridgingConfig = LazyMount.mount( entityManager, "bridging/config",
                                     "Bridging::Config", "r" )
   bridgingCliConfig = ConfigMount.mount( entityManager,
                                        "bridging/input/config/cli",
                                        "Bridging::Input::CliConfig", "w" )
   vlanXlateStatusDir = LazyMount.mount( entityManager, "bridging/vlanxlate/status",
                                         "Bridging::VlanXlateStatusDir", "r" )
   vxlanVniStatusDir = LazyMount.mount( entityManager,
                                     "vxlan/version2/vniStatusDir",
                                     "VxlanController::VniStatusDirV2", "r" )

   vxlanClientDir = LazyMount.mount( entityManager, "vxlan/clientDir",
                                     "Tac::Dir", "ri" )
   vxlanEvpnDynamicVlans = LazyMount.mount( entityManager,
                                            "bridging/input/dynvlan/vlan/evpn",
                                            "Bridging::Input::VlanIdSet", "r" )
   # LazyMount mlag/status, Mlag::Status and its dependent paths
   mlagStatus = MlagStatusLazyMounter( entityManager )
   mlagHostTable = LazyMount.mount( entityManager,
                                    "mlag/hostTable",
                                    "Mlag::HostTable", "r" )
   mlagVxlanStatus = LazyMount.mount( entityManager,
                                      'mlag/vxlan/status',
                                      'Mlag::VxlanStatus', 'r' )
   # To let the CVX infrastructure to know that Vxlan controller service is
   # enabled/disabled on the switch
   serviceConfigDir = ConfigMount.mount( entityManager,
                                           "mgmt/controller/service/config",
                                           "Controller::ServiceConfigDir", "w" )

   vxlanVniFdbStatusDir = entityManager.mount(
      "vxlancontroller/version2/vni", "Tac::Dir", "ri" )
   lRStatus = LazyMount.mount( entityManager,
                               "vxlancontroller/version2/logicalRouter",
                               "Tac::Dir", "ri" )
   vxlanControllerConfig = LazyMount.mount( entityManager,
                                     "vxlancontroller/config",
                                     "VxlanController::Config", "r" )
   controllerErrorStatus = LazyMount.mount( entityManager,
                                      "interface/status/vxlan/errorStatus/version2",
                                      "Tac::Dir", "ri" )
   vcsStateClientView = LazyMount.mount( entityManager,
                                         "vxlan/version2/vcsStateClientView",
                                         "VxlanController::VcsStateClientViewV2",
                                         "r" )
   cvxClientStatus = LazyMount.mount( entityManager, "mgmt/controller/status",
                                           "ControllerClient::Status", "r" )
   hwCounterFeatureStatusDir = LazyMount.mount( entityManager, 
                                      "flexCounter/featureStatusDir",
                                      'Tac::Dir', "ri" )
   vxlanHwCounter = LazyMount.mount( entityManager, "vxlan/hardware/counter",
                                        "Vxlan::VxlanHwCounter", "w" )

   ipConfig = LazyMount.mount( entityManager, "ip/config", "Ip::Config", "r" )
   ipStatus = LazyMount.mount( entityManager, "ip/status", "Ip::Status", "r" )

   vxlanClientConvergenceStatus = LazyMount.mount(
      entityManager,
      "vxlan/clientConvergenceStatus",
      "Vxlan::VxlanClientConvergenceStatus", "r" )

   # Boxing vxlanCtrlConfig since VniMatcher needs it at mod-load
   vxlanCtrlCfgBox.append( vxlanControllerConfig )

   vniToVlanMap = VniToVlanMap()

   mountPath = Tac.Type( 'Vxlan::RemoteVniToVlanStatus' ).mountPath( "irb",
                                                                     "Vxlan1" )
   remoteVniToVlanStatus = SmashLazyMount.mount( entityManager, mountPath,
                                                 "Vxlan::RemoteVniToVlanStatus",
                                                 readerInfo )
   ipsecConfig = LazyMount.mount( entityManager, 'ipsec/ike/config',
                                  'Ipsec::Ike::Config', 'r' )
   remoteVtepHwConfig = LazyMount.mount( entityManager, "vxlan/remoteVtepHwConfig",
                                         "Vxlan::RemoteVtepHwConfigDir", "r" )
   vtepHwStatus = LazyMount.mount( entityManager, "vxlan/vtepHwStatus",
                                   "Vxlan::VtepHwStatusDir", "r" )

   em = entityManager

   # Register vxlan cli commands with "show tech-support"
   TechSupportCli.registerShowTechSupportCmdCallback(
      '2013-05-20 12:20:55',
      lambda: _supportedCommands( [ 'show interface vxlan 1-$',
                                    'show vxlan counter software',
                                    'show vxlan counter varp',
                                    'show vxlan vtep',
                                    'show vxlan address-table',
                                    'show vxlan config-sanity detail',
                                  ] ) )

   # Time stamp added to maintain historical ordering of commands in show tech
   TechSupportCli.registerShowTechSupportCmdCallback(
         "2016-12-21 06:20:00",
         lambda: _supportedCommands( [ "show vxlan flood vtep" ] ) )

   # Register commands in show tech-support extended evpn
   TechSupportCli.registerShowTechSupportCmdCallback(
         '2017-11-03 12:06:10',
         lambda: _supportedCommands( [ 'show interface vxlan $',
                                       'show vxlan address-table',
                                     ] ),
         extended = 'evpn' )

   # Register summary commands for show tech-support summary
   TechSupportCli.registerShowTechSupportCmdCallback(
         '2020-06-15 13:31:06',
         summaryCmdCallback=lambda: _supportedCommands(
                                       [ 'show vxlan vni summary' ] ) )

   TechSupportCli.registerShowTechSupportCmdCallback(
         '2020-07-02 17:12:23',
         summaryCmdCallback=lambda: _supportedCommands(
                                       [ 'show vxlan vtep summary' ] ) )
